#!/usr/bin/env python

from __future__ import division, print_function 
from gimpfu import *                            
from math import *                              
from copy import copy                           
from functools import partial                   

# Gimp plugin.
# Draw parametric curves as approximate Bezier curves.

# (c) Markku Koppinen 2021

# History:
# 
# v1.08 2020-03-18: Initial published version
# v1.09 2020-04-14: Unpublished
# v2.0  2021-07-15: Overhauling. Simplified. New approximation algorithm. Unpublished.
# v2.1  2021-07-29: Published.
# v2.2  2021-08-20: Bug fixes.

#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'



#||<>||=======================================================================
#||<>||    Section 1
#||<>||=======================================================================
#||<>||    Data structures for working with Bezier arcs
#||<>||=======================================================================

# A Bezier arc is the segment 0<=t<=1 of a Bezier curve B(t) determined by four
# control points.
# Note: The plane is treated as the complex number plane.

#==============================================================
# Section 1.1:  class BCurve       
#==============================================================

class BCurve(object):
    """Data structures to handle Gimp vectors objects by means of individual
    Bezier arcs.
    By a "Bezier arc" we mean here a Bezier curve determined by 4 control points
    p0,p1,p2,p3.
    This class contains data structures needed to do computations on
    a Bezier curve which are done arc by arc.
    To this end, Gimp data structures
    - gimp.Vectors
    - gimp.VectorsBezierStroke
    are here represented as data structures (classes)
    - GimpVectors
    - GimpStroke
    - BezierCurve
    - BezierArc
    with the aim that
    - each control point [x,y] is represented as a complex number x+iy;
    - each Bezier arc (four control points, see above) is represented as an instance
      of BezierArc.
    """
    class GimpVectors(object):
        """Essentially same data as in a gimp.Vectors object except that
        strokes are instances of GimpStroke rather than
        gimp.VectorsBezierStroke.
        Attributes:
        - stroke_list: [GimpStroke]
        - name:        string
        """
        def __init__(self,
                     stroke_list,            # [GimpStroke]
                     name='GimpVectors'      # string
                     ):
            self.stroke_list = stroke_list
            self.name = name
        def __str__(self):
            s = 'GimpVectors '+self.name
            count = 0
            for stroke in self.stroke_list:
                s += '\n  Stroke '+str(count)+': GimpStroke'
                s += '\n    '+str(stroke)
                count += 1
            return s
        def gv2vectors_object(self, image, name=None):
            """Conversion GimpVectors -> gimp.Vectors
            (The converse: vectors_object2gv)
            """
            def cp_list2xy_list(cp_list):
                xy = []
                for cp in cp_list:
                    xy += [cp.real, cp.imag]
                return xy
            if name is None:
                name = self.name
            vectors = pdb.gimp_vectors_new(image, name)
            for stroke in self.stroke_list:
                closed = stroke.closed
                xy_list = cp_list2xy_list(stroke.cp_list)
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                             vectors, 0, len(xy_list), xy_list, closed)
            return vectors
    class GimpStroke(object):
        """Essentially same data as in a gimp.VectorsBezierStroke.points,
        except that control points list [x,y,x,y,...] is arranged as
        a list [complex] by joining successive x,y.
        Attributes:
        - cp_list:     [complex]
        - closed:      boolean
        - stroke_name: string
        """
        def __init__(self,
                     cp_list,                   # [complex]
                     closed,                    # boolean
                     stroke_name = 'GimpStroke' # string
                     ):
            self.stroke_name = stroke_name
            self.cp_list = cp_list
            self.closed = closed
        def __str__(self):
            s = self.stroke_name + '; Control points:'
            for cp in self.cp_list:
                s += '\n    '+str(cp)
            s += '\n    closed: '+str(self.closed)
            return s
        def gs2bc(self):
            """Conversion GimpStroke -> BezierCurve
            """
            head = self.cp_list[0]
            tail = self.cp_list[-1]
            inner = self.cp_list[1:-1]
            if len(inner) > 1: # Must be divisible by 4
                ba_list = []
                count = 0
                for i in range(0,len(inner)-1,3):
                    ba_list.append(BCurve.BezierArc(inner[i:i+4], 'arc '+str(count)))
                    count += 1
            elif len(inner) == 1: # Stroke has only one anchor
                ba_list = [BCurve.BezierArc([inner[0]], 'arc 0')]
            else:
                raise Exception("BCurve.GimpStroke.gs2bc: No anchors in stroke?")
            return BCurve.BezierCurve(
                               bezier_arcs = ba_list,
                               head_handle = head,
                               tail_handle = tail,
                               closed = self.closed,
                               curve_name = self.stroke_name
                               )
    class BezierCurve(object):
        """BezierCurve is a list of butting Bezier arcs with some extra data.
        Attributes:
        - curve_name:  string
        - bezier_arcs: [BezierArc];
        - head_handle: complex;
        - tail_handle: complex;
        - closed:      boolean.
        Note: "Butting" means that for any pair of successive arcs, the last
              control point of the former equals the first control point of
              the latter.
              In initialization no checks are done about this condition.
        """
        def __init__(self,
                     bezier_arcs=[],           # [BezierArc]
                     head_handle=None,         # complex
                     tail_handle=None,         # complex
                     closed=False,             # boolean
                     curve_name='BezierCurve', # string
                     ):
            self.curve_name = curve_name
            self.bezier_arcs = bezier_arcs
            if head_handle is None:
                self.head_handle = bezier_arcs[0].cp4[0]
            else:
                self.head_handle = head_handle
            if tail_handle is None:
                self.tail_handle = bezier_arcs[-1].cp4[-1]
            else:
                self.tail_handle = tail_handle
            self.closed = closed
        def __str__(self):
            s = self.curve_name + ': Bezier curve' + '; Bezier arcs:'
            count = 0
            for arc in self.bezier_arcs:
                s += '\n  arc '+ str(count)+ ': ' +str(arc)
                count += 1
            s += '\n  Head handle:    '+str(self.head_handle)
            s += '\n  Tail handle:    '+str(self.tail_handle)
            s += '\n  Closed:'    +str(self.closed)
            return s
        def bc2gs(self):
            """Conversion BezierCurve -> GimpStroke
            """
            cp_list = [self.head_handle]
            if len(self.bezier_arcs[0].cp4) > 1:
                for ba in self.bezier_arcs:
                    cp_list += ba.cp4[:3]
                cp_list.append(self.bezier_arcs[-1].cp4[-1])
            else:
                # Only one BezierArc and it has cp4 = [p0]
                # (may rise from Gimp stroke with only one anchor).
                cp_list.append(self.bezier_arcs[0].cp4[0])
            cp_list.append(self.tail_handle)
            return BCurve.GimpStroke(cp_list, self.closed, self.curve_name)
        def bc2vectors_object(self, image, name=None):
            """Conversion BezierCurve -> gimp.Vectors
            Makes vectors_object with 1 stroke.
            """
            if name is None:
                name = self.curve_name
            gs = self.bc2gs()             # BCurve.GimpStroke
            gv = BCurve.GimpVectors([gs]) # BCurve.GimpVectors
            return gv.gv2vectors_object(image, name)
    class BezierArc(object):
        """Data structure for one Bezier arc: arc determined by 4 control
           points.
        Attributes:
        - cp4:      [p0,p1,p2,p3] where each pi:complex, or
                    [p0] if only one control point (may rise from
                    a stroke with only one anchor);
        - arc_name: string
        """
        def __init__(self,
                     cp4,                 # cp4=[p0,p1,p2,p3] with pi:complex
                                          # or [p0]
                     arc_name='BezierArc' # string
                     ):
            self.cp4 = cp4
            self.arc_name = arc_name
        def reverse(self):
            self.cp4.reverse()
        def __str__(self):
            s = 'BezierArc, control points:'
            for cp in self.cp4:
                s += '\n  '+ str(cp)
            return s

# -----------------
# vectors_object2gv
# -----------------
# Conversion gimp.Vectors -> GimpVectors
# Args:
# - vectors_object: gimp.Vectors (Gimp vectors object)
# Returns:
# - GimpVectors
#
# Note: The inverse conversion is GimpVectors.gv2vectors_object(...).
#       See also BezierCurve.bc2vectors_object(...).
def vectors_object2gv(vectors_object):
    def xy_list2cp_list(xy_list): # xy_list = [x,y,x,y,...,]
        cp = []
        for i in range(0,len(xy_list),2):
            cp.append(complex(xy_list[i], xy_list[i+1]))
        return cp
    stroke_list = []
    count = 0
    for stroke in vectors_object.strokes:
        xy_list = stroke.points[0]
        closed = stroke.points[1]
        cp_list = xy_list2cp_list(xy_list)
        stroke_name = vectors_object.name + ' stroke '+str(count)
        stroke_list.append(BCurve.GimpStroke(cp_list, closed, stroke_name))
        count += 1
    return BCurve.GimpVectors(stroke_list, vectors_object.name)


#==============================================================
# Section 1.2:  Bezier curve: functions and routines
#==============================================================

# ---------
# bezier_rv
# ---------
# The usual Bezier curve from control points.
# Version where all points are complex numbers
# (the plane is the complex number plane).
# Args:
# - t:  float
# - cp4:[complex,complex,complex,complex]
# Returns:
# - complex
def bezier_rv(t, cp4):
    P0,P1,P2,P3 = cp4
    if t > 0.5:
        u = (1-t)/t
        t3 = t**3
        return t3*(u*(u*(u*P0+3*P1)+3*P2)+P3) # complex
    else:
        u = t/(1-t)
        t3 = (1-t)**3
        return t3*(u*(u*(u*P3+3*P2)+3*P1)+P0) # complex

# Derivative
# Args:
# - t:  float
# - cp4:[complex,complex,complex,complex]
# Returns:
# - complex
def bezier_dot_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    P01 = -P0+P1
    P12 = -P1+P2
    P23 = -P2+P3
    if t > 0.5:
        u = (1-t)/t
        t2 = t**2
        return 3*t2*(u*(u*P01+2*P12)+P23) # complex
    else:
        u = t/(1-t)
        t2 = (1-t)**2
        return 3*t2*(u*(u*P23+2*P12)+P01) # complex

# Second derivative
# Args:
# - t:  float
# - cp4:[complex,complex,complex,complex]
# Returns:
# - complex
def bezier_dot_dot_rv(t, control_points): # Could be made faster as above.
    p0,p1,p2,p3 = control_points
    return -6*(1-t)*(-p0+p1) + 6*(1-2*t)*(-p1+p2) + 6*t*(-p2+p3)

# --------------------
# bezier_closest_point
# --------------------
# Find closest point (parameter value t and the point B(t)) to a given v on
# the Bezier curve arc B(t), 0<=t<=1. Control points are cp4.
# Args:
# - cp4: [complex,complex,complex,complex] (control points)
# - v:   complex
# - level: integer (working level: higher number, harder work)
# Returns:
# - [float,complex] (=[t,B(t)])
def bezier_closest_point(cp4, v, level=5):
    accuracy = 1e-8
    linear_approx = True # Linear approximation
    if level <= 2:
        guesses = [1/3, 2/3]
    elif level <= 5:
        guesses = [1/4, 2/4, 3/4]
    elif level <= 8:
        guesses = [1/6, 2/6, 3/6, 4/6, 5/6]
    else:
        guesses = [1/8, 2/8, 3/8, 4/8, 5/8, 6/8, 7/8]
        linear_approx = False # Quadratic approximation
    candidates = [[0,cp4[0]], [1,cp4[-1]]] # [B(t),t]: First, store end points.
    for guess in guesses: # Second, points in open interval (tangent condition)
        t = guess
        Bv = bezier_rv(t,cp4) - v
        for tries in range(20):
            Bd = bezier_dot_rv(t,cp4)
            Bdd = bezier_dot_dot_rv(t,cp4)
            b = (Bv.real * Bdd.real + Bv.imag * Bdd.imag     # b = (B-v).B'' + B'.B'
                 + Bd.real * Bd.real + Bd.imag * Bd.imag)
            c = Bv.real * Bd.real + Bv.imag * Bd.imag        # c = (B-v).B'
            if not linear_approx: # Quadratic: solve a(dt)^2 + b(dt) + c = 0
                try:
                    a = Bd.real * Bdd.real + Bd.imag * Bdd.imag  # a = B'.B''
                    inv_2a = 1/(2*a)
                    dt = inv_2a * (-b + sqrt(b*b - 4*a*c))
                except ZeroDivisionError:
                    linear_approx = True # Try linear
                except ValueError:
                    linear_approx = True # Try linear
            if linear_approx:     # Linear: solve b(dt) + c = 0
                try:
                    dt = -c / b
                except ZeroDivisionError:
                    break
            tdt = t+dt
            if tdt < 0: tdt = 0 # Restrict the parameter to [0,1]
            elif tdt > 1: tdt = 1
            if abs(t - tdt) < accuracy:
                t = (t + tdt)/2
                candidates.append([t, bezier_rv(t,cp4)])
                break
            else:
                t = tdt
                Bv = bezier_rv(t,cp4) - v
        else:
            candidates.append([t, Bv])
    return min(candidates, key=(lambda x: abs(x[1]-v)))


#||<>||=======================================================================
#||<>||    Section 2
#||<>||=======================================================================
#||<>||    Polynomial zeroes
#||<>||=======================================================================

# --------------------
# zeroes_of_polynomial
# --------------------
# Given a real polynomial, find the real zeroes, the x values of extrema,
# and the x values of the inflection points. May restrict to an interval.
# Written just for fun. Other algorithms exist in abundance.
#
# Args:
# - coefficients: list [c0,c1,...,cn] of the coefficients of the
#                     polynomial c0 + c1*x + ... + cn*(x**n);
#                     the ci are integer or float.
# - interval:     None or [float,float]
# Returns:
# 1. None in the case of a zero polynomial.
# 2. None if interval is given (not None) with interval=[a,b] where a>b.
# 3. If interval=[a,a] is given, if a is a root then [a],[],[] is returned,
#    otherwise [],[],[]. So, no effort is made to give any extrema or inflection
#    points.
# 3. Otherwise:
#    - the list of real zeroes of the polynomial, in increasing order
#      (possibly []) with multiple roots repeated;
#    - the extrema (the x values);
#    - the inflection points (the x values).
# 4. Because of the chosen accuracy and tolerance in the code, choosing different
#    interval may change the returned values slightly.
def zeroes_of_polynomial(coefficients, interval=None):
    from math import sqrt
    from functools import partial
    def f(x, coeff): # Horner
        s = 0.
        for ci in coeff[::-1]:
            s = ci + x*s
        return s
    if not (interval is None):
        LO,HI = interval
        if LO > HI:
            return None
        if LO == HI:
            if f(LO) == 0:
                return [LO],[],[]
            else:
                return [],[],[]
    coefficients = [float(ci) for ci in coefficients]
    # Find largest n with coefficients[n] non-zero and
    # discard all higher terms:
    n = -1
    for i in range(len(coefficients)):
        if coefficients[i] != 0.:
            n = i
    if n < 0: # Polynomial is identically zero.
        return None
    c = coefficients[:n+1] 
    # Make c[n] positive:
    if c[n] < 0:
        c = [-ci for ci in c]
    if n <= 2:
        if n == 0: # Polynomial is a non-zero constant.
            results = [[],[],[]]
        if n == 1: # Polynomial is c[0] + c[1]*x
            results = [[-c[0] / c[1]], [], []]
        if n == 2: # Polynomial is c[0] + c[1]*x + c[2]*(x**2).
            discr = c[1]**2 - 4*c[0]*c[2]
            try:
                root = sqrt(discr)
                x1 = (-c[1] - root) / (2*c[2])
                x2 = (-c[1] + root) / (2*c[2])
                #return [x1,x2], [-c[1]/(2*c[2])], []
                results = [[x1,x2], [-c[1]/(2*c[2])], []]
            except ValueError:
                #return [], [-c[1]/(2*c[2])], []
                results = [[], [-c[1]/(2*c[2])], []]
        if interval is None:
            return results
        else:
            results0 = [t for t in results[0] if LO <= t <= HI]
            results1 = [t for t in results[1] if LO <= t <= HI]
            results2 = [t for t in results[2] if LO <= t <= HI]
            return results0, results1, results2
    # Build subdivision such that in each subinterval
    # the polynomial is monotonous:
    derivative = [e*i for e,i in enumerate(c)][1:]
    extrema, inflection, _ = zeroes_of_polynomial(derivative, interval)
    if interval is None:
        x_far = sum([abs(ci/c[-1]) for ci in c])
        subdiv = [-x_far] + extrema + [x_far]
    else:
        subdiv = [LO] + [t for t in extrema if LO < t < HI] + [HI]
    # Solve and collect the zeroes of each monotonous segment:
    fc = partial(f, coeff=c)
    zeroes = []
    for i in range(len(subdiv)-1):
        lo, hi = subdiv[i], subdiv[i+1]
        x = zero_of_monotonous_function(fc, lo, hi,
                                    accuracy=1e-10, tolerance=1e-10)
        if not(x is None):
            zeroes.append(x)
    return zeroes, extrema, inflection

# ---------------------------
# zero_of_monotonous_function
# ---------------------------
# Find the zero, if any, of a monotonous function f(x)
# in the interval [lo,hi]. If none is found, None is returned.
# Args:
# - f, lo, hi: the monotonous function with start and end value;
# - accuracy:  accuracy for x.
# - tolerance: float>=0; if no zeroes exist in the interval, then
#              if f(lo) or f(hi) is less than 'tolerance' from 0,
#              then lo or hi is returned, respectively.
# Returns: the zero or None.
def zero_of_monotonous_function(f, lo, hi, accuracy=1e-10, tolerance=1e-10):
    MAX_ROUNDS = 50
    lo, hi = float(lo), float(hi)
    flo, fhi = float(f(lo)), float(f(hi))
    if (flo > 0.) == (fhi > 0.):
        if abs(flo) <= tolerance:
            return lo
        elif abs(fhi) <= tolerance:
            return hi
        else:
            return None
    if flo == 0.:
        return lo
    if fhi == 0.:
        return hi
    count = 0
    while hi-lo > accuracy:
        mid = (lo+hi)/2.
        fmid = float(f(mid))
        if fmid == 0.:
            return mid
        if (flo > 0.) == (fmid > 0.):
            lo, flo = mid, fmid
        else:
            hi, fhi = mid, fmid
        count += 1
        if count > MAX_ROUNDS:
            break
    x = (fhi*lo - flo*hi) / (fhi-flo)
    return x

# ------
# solve2
# ------
# Solve ax^2 + 2bx + c = 0 for real roots. Return [x1,x2] or [x] or [].
# Return None if the polynomial is identically zero.
def solve2(a,b,c):
    from math import sqrt
    ZERO = 1e-12
    try:
        inva = 1./a
    except ZeroDivisionError:
        try:
            invb = 1./b
        except ZeroDivisionError:
            if abs(c) < ZERO:
                return None
            else:
                return []
        return [-c*invb/2]
    try:
        root = sqrt(b*b - a*c)
    except ValueError:
        return []
    return [(-b+root)*inva, (-b-root)*inva]

# -------------------------
# solve_bernstein2_equation
# -------------------------
# Find real zeroes of the equation c0*b0(t) + c1*b1(t) + c2*b2(t) = 0 where
# b0,b1,b2 are the Bernstein polynomials of degree 2 and
# c0,c1,c2 are real coefficients.
# Return the zeroes as a list (0, 1, or 2 items).
# Exception: If the function is constant 0, return None.
# Args:
# - c0,c1,c2: float
# Returns:
# - None or list of floats (possibly empty)
def solve_bernstein2_equation(c0,c1,c2):
    from math import sqrt
    ZERO = 1e-14
    if abs(c0 - 2.*c1 + c2) < ZERO: # equation of degree <= 1
        try:
            return [c0 / (2.*(c0 - c1))]
        except ZeroDivisionError: # equation of degree 0
            if abs(c0) < ZERO: # constant 0
                return None
            else: # constant not 0
                return []
    try:
        root = sqrt(c1**2 - c0*c2)
    except ValueError: # No real roots
        return []
    return [(c0 - c1 + root) / (c0 - 2.*c1 + c2),
            (c0 - c1 - root) / (c0 - 2.*c1 + c2)]


#||<>||=======================================================================
#||<>||    Section 3
#||<>||=======================================================================
#||<>||    Determination of a Bezier arc from three points and tangents
#||<>||=======================================================================

#==============================================================
# Section 3.1:  Affine map
#==============================================================
# Note: The plane is treated as the complex number plane

class MakeAffineMapError(Exception):
    def __init__(self,message):
        self.message = message
    def __str__(self):
        return self.message

# ---------------
# make_affine_map
# ---------------
# Affine map sending three given points u0,u1,u2 to another
# three given points v0,v1,v2. 
# Args:
# - base:   [complex,complex,complex] (at least 3 points)
# - target: [complex,complex,complex] (at least 3 points)
# Returns:
# - callable (complex -> complex)
def make_affine_map(base, target):
    def A_inv_B(A,B): # Two 2x2 matrices. Compute A^(-1)*B
        a,b,c,d = A
        x,y,z,u = B
        try:
            idet = 1./(a*d-b*c)
        except ZeroDivisionError:
            raise Exception("make_affine_map: Zero determinant")
        C = [d*x-b*z, d*y-b*u,
            -c*x+a*z, -c*y+a*u]
        return [idet*e for e in C]
    u0,u1,u2 = base[:3] # The first three anchors: complex
    v0,v1,v2 = target[:3] # The first three anchors: complex
    if collinear([u0,u1,u2]):
        raise MakeAffineMapError("make_affine_map: Anchors of Base are collinear.")
    u10 = u1-u0
    u20 = u2-u0
    v10 = v1-v0
    v20 = v2-v0
    A = [u10.real, u10.imag, # 2x2 matrix 
         u20.real, u20.imag]
    B = [v10.real, v10.imag, # 2x2 matrix 
         v20.real, v20.imag]
    try:
        AB = A_inv_B(A,B)
    except Exception as e:
        raise MakeAffineMapError("make_affine_map: "+str(e))
    def pm(x):
        return v0 + complex((x-u0).real*AB[0] + (x-u0).imag*AB[2],
                     (x-u0).real*AB[1] + (x-u0).imag*AB[3])
    return pm

#==============================================================
# Section 3.2:  Bezier arc from tangent data
#==============================================================

class NoSolutionError(Exception):
    def __init__(self,m):
        self.message = m

# ------------------------
# bezier_from_tangent_data
# ------------------------
# Given two points p0,p3, a point K, and three directions (0,1,2),
# find control points cp4=[p0,p1,p2,p3] for all Bezier curves:
# - running through K and having there tangent with direction 0,
# - having at p0,p3 tangents with directions 1 and 2.
# Return the solutions as a list, each Bezier curve as cp4, list of 4 control points.
# Return also the parameter values t at which K is reached.
# There are two extra conditions to reject too wild solutions:
# respect_directions and wild_handle_limit.
# Args:
# - p0:         complex # Start point
# - p3:         complex # End point
# - K:          complex # Mean point
# - direction1: complex # Tangent at the start point p0
# - direction2: complex # Tangent at the end point p3
# - direction0: complex # Tangent at the mean point K
# - restrict01: boolean # Accept only 0<=t<=1
# - respect_directions: boolean # The arc should point (about) in the directions 1,2
#                               # at the end points,
#                               # and the derivative should point (about) in
#                               # the direction0 at point K
# - wild_handle_limit: None or float # to reject solutions with too long handles
#                                    # (None means that this condition is not used)
# - print_info: boolean
# Returns:
# - [[float, [complex,complex,complex,complex]] (=[[t,cp4]])
def bezier_from_tangent_data(p0, p3, K,
                             direction1, direction2, direction0,
                             restrict01=True,
                             respect_directions=True,
                             wild_handle_limit=2, # Ad hoc limit
                             print_info=False):
    ZERO = 1e-8
    D1,D2 = p0+direction1, p3+direction2
    # Cases:
    if p0 == p3:
        case = 'VI'      # stroke is one loop or one point
    elif abs(direction1.real*direction2.imag - direction1.imag*direction2.real) < ZERO:
        if collinear([p0,p3,p0+direction1]):
            case = 'V'   # all collinear
        else:
            case = 'IV'  # parallel lines
    elif collinear([p0,p3,D2]):
        case = 'II'      # anchors on one of the lines
    elif collinear([p0,p3,D1]):
        case = 'III'     # anchors on one of the lines
    else:
        case = 'I'       # proper triangle
    # Different procedures for different cases:
    if print_info:
        print("\nCase = "+case)
        print("p0 = "+str(p0))
        print("p3 = "+str(p3))
        print("K = "+str(K))
        print("direction1 = "+str(direction1))
        print("direction2 = "+str(direction2))
        print("direction0 = "+str(direction0))
    if case == 'I':
        C = intersection_of_lines([p0,p0+direction1], [p3, p3+direction2]) # apex
        if print_info:
            print("C = "+str(C))
        t_cp4_list = case_I_triangle(p0,p3,C,K, direction0,
                                     restrict01=restrict01,
                                     print_info=print_info)
    elif case == 'II':
        D = p0 + direction1
        t_cp4_list = case_II_flat(p0,p3,D,K, direction0,
                                  restrict01=restrict01,
                                  print_info=print_info)
    elif case == 'III':
        D = p3 + direction2
        t_cp4_list = case_II_flat(p3,p0,D,K, direction0,
                                  restrict01=restrict01,
                                  print_info=print_info)
        t_cp4_list = [[1-t,cp4[::-1]] for t,cp4 in t_cp4_list]
    elif case == 'IV':
        D = p0 + direction1
        t_cp4_list = case_IV_parallel(p0,p3,D,K, direction0,
                                      restrict01=restrict01,
                                      print_info=print_info)
    elif case == 'V':
        result = t_cp4_list = case_V_collinear(p0,p3,K,
                                                print_info=print_info)
    elif case == 'VI':
        D1 = p0+direction1
        D2 = p0+direction2
        t_cp4_list = case_VI_zero_chord(p0,D1,D2,K, direction0,
                                        restrict01=restrict01,
                                        print_info=print_info)
    if len(t_cp4_list) == 0:
        raise NoSolutionError("bezier_from_tangent_data: empty t_cp4_list")
    if case != 'V': # Case V: zero handles
        # Remove solutions violating the extra requirements:
        if respect_directions:
            accept = []
            for t,cp4 in t_cp4_list:
                q01 = -cp4[0]+cp4[1]
                q32 = -cp4[3]+cp4[2]
                derivative_at_K = bezier_dot_rv(t, cp4)
                chord = -p0+p3
                if (
                    (vdot(q01,direction1) > 0)
                    and (vdot(q32,direction2) > 0)
                    and (vdot(chord,derivative_at_K) > 0)
                    ):
                    accept.append([t,cp4])
            t_cp4_list = accept
        if not (wild_handle_limit is None):
            accept = []
            for t,cp4 in t_cp4_list:
                q0,q1,q2,q3 = cp4
                q01 = -q0+q1
                q02 = -q0+q2
                q03 = -q0+q3
                q12 = -q1+q2
                q13 = -q1+q3
                q23 = -q2+q3
                if (wild_handle_limit*(abs(q01)+abs(q23))
                                   <= (abs(q02)+abs(q03)+abs(q12)+abs(q13))
                    ):
                    accept.append([t,cp4])
            t_cp4_list = accept
    if len(t_cp4_list) == 0:
        raise NoSolutionError("bezier_from_tangent_data: empty t_cp4_list after removals")
    return t_cp4_list

# ---------------
# case_I_triangle
# ---------------
# One case of bezier_from_tangent_data: The given data forms a proper triangle
# with vertices A,B,C.
# Args:
# - A,B,C,K:    complex
# - direction0: complex
# - restrict01: boolean
# - print_info: boolean
# Returns:
# - [[float, [complex,complex,complex,complex]]] (=[[t,cp4]])
def case_I_triangle(A,B,C,K, direction0, restrict01=True, print_info=False):
    # Transform to a standard position:
    # A -> (0,1)
    # B -> (1,0)
    # C -> (0,0)
    sA, sB, sC = 1j, 1, 0
    try:
        aff_map = make_affine_map([A,B,C], [sA,sB,sC])
        inv_map = make_affine_map([sA,sB,sC], [A,B,C])
    except MakeAffineMapError as e:
        if print_info:
            print("case_I_triangle: make_affine_map raised: "+str(e))
        return []
    sK = aff_map(K)
    sd0 = aff_map(K + direction0) - sK
    u,v = sd0.real, sd0.imag
    a,b = sK.real, sK.imag
    # Solve the parameter values t.
    # Coefficients:
    c3 = -u+v
    c2 = 3*u
    c1 = -3*u -3*a*v + 3*b*u
    c0 = u + 2*a*v - b*u
    #ts = solve_pol3_R(c3,c2,c1,c0)
    ts = zeroes_of_polynomial([c0,c1,c2,c3])[0]
    if len(ts) == 0:
        raise NoSolutionError("case_I_triangle: empty ts")
    # Compute the handle lengths:
    result = []
    for t in ts:
        if restrict01 and not(0<t<1):
            continue
        xi  = (t*t*(3-2*t) - a) / (3*(1-t)*t*t)
        eta = ((1-t)*(1-t)*(1+2*t) - b) / (3*(1-t)*(1-t)*t)
        scp4 = [1j, complex(0,1-eta), complex(1-xi,0), 1] # cp4 in standard position
        cp4 = [inv_map(z) for z in scp4]
        result.append([t,cp4])
    return result

# ------------
# case_II_flat
# ------------
# Another case of bezier_from_tangent_data: The tangent at p0 runs through p3,
# or vice versa.
# Args:
# - A,B,D,K:    complex
# - direction0: complex
# - restrict01: boolean
# - print_info: boolean
# Returns:
# - [[float, [complex,complex,complex,complex]]] (=[[t,cp4]])
def case_II_flat(A,B,D,K, direction0, restrict01=True, print_info=False):
    # Transform the control points to standard position:
    # p0 -> (0,0)
    # p3 -> (1,0)
    # D -> (0,1)
    try:
        aff_map = make_affine_map([A,B,D], [0,1,1j])
        inv_map = make_affine_map([0,1,1j], [A,B,D])
    except MakeAffineMapError as e:
        if print_info:
            print("case_II_flat: make_affine_map raised: "+str(e))
        return []
    sK = aff_map(K)
    sd0 = aff_map(K + direction0) - sK
    u,v = sd0.real, sd0.imag
    a,b = sK.real, sK.imag
    # Solve the parameter values t.
    # Coefficients:
    c3 = v
    c2 = 0
    c1 = -3*a*v + 3*b*u
    c0 = 2*a*v - b*u
    #ts = solve_pol3_R(c3,c2,c1,c0)
    ts = zeroes_of_polynomial([c0,c1,c2,c3])[0]
    # Compute the handle lengths:
    result = []
    for t in ts:
        if restrict01 and not(0<=t<=1):
            continue
        xi  = (t*t*(3-2*t) - a) / (3*(1-t)*t*t)
        eta = b / (3*(1-t)*(1-t)*t)
        scp4 = [0, complex(0,eta), complex(1-xi,0), 1] # cp4 in standard position
        cp4 = [inv_map(z) for z in scp4]
        result.append([t,cp4])
    return result

# ----------------
# case_IV_parallel
# ----------------
# Another case of bezier_from_tangent_data: parallel tangents at end points.
# Args:
# - A,B,D,K:    complex
# - direction0: complex
# - restrict01: boolean
# - print_info: boolean
# Returns:
# - [[float, [complex,complex,complex,complex]]] (=[[t,cp4]])
def case_IV_parallel(A,B,D,K, direction0, restrict01=True, print_info=False):
    # Transform the control points to standard position:
    # p0 -> (0,0)
    # p3 -> (1,0)
    # D -> (0,1)
    try:
        aff_map = make_affine_map([A,B,D], [0,1,1j])
        inv_map = make_affine_map([0,1,1j], [A,B,D])
    except MakeAffineMapError as e:
        if print_info:
            print("case_IV_parallel: make_affine_map raised: "+str(e))
        return []
    sK = aff_map(K)
    sd0 = aff_map(K + direction0) - sK
    u,v = sd0.real, sd0.imag
    a,b = sK.real, sK.imag
    # Solve the parameter values t.
    # Coefficients:
    c3 = 2
    c2 = -3
    c1 = 0
    c0 = a
    #ts = solve_pol3_R(c3,c2,c1,c0)
    ts = zeroes_of_polynomial([c0,c1,c2,c3])[0]
    # Compute the handle lengths:
    try:
        v_per_u = v/u
    except ZeroDivisionError:
        if a in (0,1) and b == 0:
            raise NoSolutionError("Infinitely many solutions, IV")
        else:
            raise NoSolutionError("No solutions, IV, 1")
    result = []
    for t in ts:
        if restrict01 and not(0<=t<=1):
            continue
        try:
            upper = 2*v_per_u*(1-t)*t
            lower = b / (3*(1-t)*t)
        except ZeroDivisionError:
            if b == 0:
                raise NoSolutionError("Infinitely many solutions, IV")
            else:
                raise NoSolutionError("No solutions, IV, 2")
        xi  = (upper - (1-3*t)*lower) / t
        eta = (-upper + (2-3*t)*lower) / (1-t)
        scp4 = [0, complex(0,eta), complex(1,xi), 1]
        cp4 = [inv_map(z) for z in scp4]
        result.append([t,cp4])
    return result

# ----------------
# case_V_collinear
# ----------------
# Another case of bezier_from_tangent_data: all collinear
# Notes:
# 1. Returns t = 0.5.
# 2. Fails regularly. Must have K on the same line.
# Args:
# - p0,p3,K:    complex
# - restrict01: boolean
# - print_info: boolean
# Returns:
# - [[float, [complex,complex,complex,complex]]] (=[[t,cp4]])
def case_V_collinear(p0,p3,K, print_info=False):
    if not collinear([p0,p3,K]):
        if print_info:
            print("p0 = "+str(p0))
            print("p3 = "+str(p3))
            print("K = "+str(K))
        raise NoSolutionError("No solutions, V, 1")
    return [[.5, [p0,p0,p3,p3]]]

# ------------------
# case_VI_zero_chord
# ------------------
# Another case of bezier_from_tangent_data: Zero chord (p0=p3).
# Args:
# - A,D1,D2,K:    complex
# - direction0: complex
# - restrict01: boolean
# - print_info: boolean
# Returns:
# - [[float, [complex,complex,complex,complex]]] (=[[t,cp4]])
def case_VI_zero_chord(A,D1,D2,K, direction0, restrict01=True, print_info=False):
    # Transform the control points to standard position:
    # p0 = p3 -> (0,0)
    # D1 -> (0,1)
    # D2 -> (1,0)
    try:
        aff_map = make_affine_map([A,D1,D2], [0,1j,1])
        inv_map = make_affine_map([0,1j,1], [A,D1,D2])
    except MakeAffineMapError as e:
        if print_info:
            print("case_VI_zero_chord: make_affine_map raised: "+str(e))
        return []
    sK = aff_map(K)
    sd0 = aff_map(K + direction0) - sK
    u,v = sd0.real, sd0.imag
    a,b = sK.real, sK.imag
    # Solve the parameter values t.
    try:
        t = (2*a*v - b*u) / (3*(a*v - b*u))
    except ZeroDivisionError:
        raise NoSolutionError("Should not happen: ZeroDivisionError when computing t (K=p0?)")
    # Compute the handle lengths:
    if restrict01 and not(0<=t<=1):
        return []
    xi  = a / (3*(1-t)*t*t)
    eta = b / (3*(1-t)*(1-t)*t)
    scp4 = [0, complex(0,eta), complex(xi,0), 0] # cp4 in standard position
    cp4 = [inv_map(z) for z in scp4]
    return [[t,cp4]]


#||<>||=======================================================================
#||<>||    Section 4
#||<>||=======================================================================
#||<>||    Routines for plane vectors (=complex numbers)
#||<>||=======================================================================


# v dot (w perpendicular)
def cross(v,w):
    return (v*w.conjugate()).imag

def vdot(v,w):
    return v.real*w.real + v.imag*w.imag

def collinear(points): # points: [complex] (The plane is the complex number plane.)
    ZERO = 1e-8
    n = len(points)
    if n < 3:
        return True
    for i in range(n):
        for j in range(i+1,n):
            for k in range(j+1,n):
                a,b,c = points[i], points[j], points[k]
                ab = -a+b
                ac = -a+c
                bc = -b+c
                if abs(cross(ab,ac)) > ZERO:
                    return False
                if abs(cross(ab,bc)) > ZERO:
                    return False
                if abs(cross(ac,bc)) > ZERO:
                    return False
    return True

# ---------------------
# intersection_of_lines
# ---------------------
# Intersection of two lines.
# The plane is viewed as the complex number plane.
# The lines are given as pairs of points: [complex,complex]
# Return None if
# - the lines are parallel (coincident or not), or
# - the points in the pair for line0 or line1 are identical.
# Args:
# - line0: [complex,complex]
# - line1: [complex,complex]
# Returns:
# - None or complex
def intersection_of_lines(line0,line1):
    a,b = line0
    c,d = line1
    try:
        t = cross(c-a,d-c) / cross(b-a,d-c)
    except ZeroDivisionError: # parallel lines or a=b or c=d
        return None
    return a + t*(b-a)

# --------------------
# intersection_of_rays
# --------------------
# Intersection of two rays.
# The plane is viewed as the complex number plane.
# The lines are given as [complex,complex] meaning
# [starting point, direction vector].
# Return None if the rays do not intersect or if they are on top of each other
# (parallel).
# Args:
# - ray0: [complex,complex]
# - ray1: [complex,complex]
# Returns:
# - None or complex
def intersection_of_rays(ray0,ray1):
    a0,s0 = ray0
    a1,s1 = ray1
    try:
        t0 = -cross(a0-a1,s1) / cross(s0,s1)
        t1 = -cross(a0-a1,s0) / cross(s0,s1)
    except ZeroDivisionError: # parallel rays
        return None
    if t0 <= 0 or t1 <= 0:
        return None
    apex0 = a0 + t0*s0
    apex1 = a1 + t1*s1
    return (apex0+apex1)/2

# -------------------------------------
# circular_like_arc_in_tangent_triangle
# -------------------------------------
# Assume ABC is a triangle. Find a Bezier arc running through A,B and
# having ABC as a tangent triangle.
# If the triangle is isosceles (|AC|=|BC|) the arc is a very good approximation
# of a circular arc.
# Return as a control point quadruple.
# The plane is viewed as the complex number plane.
# Args:
# - A,B,C: complex
# Returns:
# - [complex,complex,complex,complex] (control points A,p1,p2,B)
def circular_like_arc_in_tangent_triangle(A,B,C):
    a = abs(-A+C)
    b = abs(-B+C)
    c = abs(-A+B)
    h = (4/3.)*(c/(2*b+c))
    k = (4/3.)*(c/(2*a+c))
    return [A, (1-h)*A + h*C, (1-k)*B + k*C, B]


#||<>||=======================================================================
#||<>||    Section 5
#||<>||=======================================================================
#||<>||    Routines for drawing
#||<>||=======================================================================

# ------------------------
# gimp_draw_vectors_object
# ------------------------
# The Gimp version of the routine 'drawing_routine_for_bezier':
# Insert into Gimp the input vectors object, and make it visible if
# visible=True.
# Args:
# - image: Gimp's image (obtained usually from GUI);
# - vectors_object: a vectors object in Gimp;
# - visible: Boolean.
# Returns the vectors object in Gimp, and makes it visible if visible=True.
def gimp_draw_vectors_object(image, vectors_object, visible=True):
    # Insert a Gimp vectors object in Gimp.
    # Adapted from http://registry.gimp.org/files/lqr_wpset-0.14.py.txt
    # by Mike Kazantsev.
    def gimp_insert_vectors(image, vectors_object, position=0):
        if gimp.version >= (2, 8, 0):
            # 2.7.0 has gimp-image-insert-vectors, but fails with
            # parent=None
            parent = None
            return pdb.gimp_image_insert_vectors(
                     image, vectors_object, parent, position)
        else:
            return pdb.gimp_image_add_vectors(
                     image, vectors_object, position)# Deprecated!
    gimp_insert_vectors(image, vectors_object)
    vectors_object.visible = visible
    return vectors_object

# For debugging and experimenting only
def draw_cp4list(image, cp4list, name='cp4list'):
    balist = [BCurve.BezierArc(cp4=cp4) for cp4 in cp4list]
    bc = BCurve.BezierCurve(bezier_arcs = balist)
    vectors = bc.bc2vectors_object(image, name=name)
    return gimp_draw_vectors_object(image, vectors, visible=True)

# Args:
# - image
# - plot_origo: complex
# Returns:
# - gimp.Vectors
def make_axes(image, plot_origo):
    x_cp = [complex(0,               plot_origo.imag),
            complex(plot_origo.real, plot_origo.imag), 
            complex(image.width,     plot_origo.imag)]
    y_cp = [complex(plot_origo.real, 0),
            complex(plot_origo.real, plot_origo.imag), 
            complex(plot_origo.real, image.height)]
    x_cpx3 = []
    for cp in x_cp:
        x_cpx3 += [cp,cp,cp]
    y_cpx3 = []
    for cp in y_cp:
        y_cpx3 += [cp,cp,cp]
    x_gs = BCurve.GimpStroke(cp_list = x_cpx3, closed = False)
    y_gs = BCurve.GimpStroke(cp_list = y_cpx3, closed = False)
    axes_gv = BCurve.GimpVectors(stroke_list = [x_gs,y_gs], name = 'axes')
    return axes_gv.gv2vectors_object(image)


#||<>||=======================================================================
#||<>||    Section 6
#||<>||=======================================================================
#||<>||    Data structure and routines for splitting parametric arcs
#||<>||=======================================================================

#==============================================================
# Section 6.1:  class SplittableFunction
#==============================================================

class NoSplitError(Exception):
    def __init__(self, message):
        self.message = message
    def __str__(self):
        return self.message

class BoxSize(object):
    """To store size of the bounding box of a curve.
    Attributes:
    - f_size: float (size of the bounding box (rough estimate))
    - t_size: the same but for the parameter t
    """
    def __init__(self,t_size, f_size):
        self.t_size = t_size
        self.f_size = f_size
    def __str__(self):
        s = "BoxSize:"
        s += "\n  t_size = "+str(self.t_size)
        s += "\n  f_size = "+str(self.f_size)
        return s

class SplittableFunction(object):
    """Data structure for storing and splitting a parametric arc
    (a complex valued function and an interval).
    Attributes:
    - function: callable (R->C)
    - interval: [float,float]
    - name: string
    - extreme_parameter
    - extreme_point
    - start_direction   (unit vector)
    - end_direction     (unit vector)
    - extreme_direction (unit vector)
    - sample_points     ([t,f(t)])
    - length (a very rough estimate)
    Methods:
    - split (splits self at an extremal point)
    - box_size (a very rough estimate of the bounding box, (t_size, f_size))
    - __str__
    Init arguments:
    - function: callable (R->C)
    - interval: [float,float]
    - name: string
    - start_direction: None or complex
    - end_direction: None or complex
    - sample_size: integer
    Notes:
    1. The interval will be turned so that interval[0] < interval[1].
    2. All three tangent directions will point (hopefully) in the direction
       where the point moves when the arc is traversed interval[0]->interval[1].
    """
    def __init__(self,
                 function,
                 interval,
                 name,
                 start_direction=None,
                 end_direction=None,
                 sample_size=10
                 ):
        #ZERO = 1e-3
        self.function = function
        self.name = name
        t0,t1 = interval
        #if t0 > t1:
        #    print("Interval "+str([t0,t1]))
        t0,t1 = [min(t0,t1), max(t0,t1)]
        self.interval = [t0,t1]
        delta = min((t1-t0)/200, 1e-3) # ?
        if start_direction is None:
            self.start_direction = tangent_direction(function,
                  t0, delta, algorithm=3)
        else:
            self.start_direction = start_direction
        if end_direction is None:
            self.end_direction = tangent_direction(function,
                  t1, delta, algorithm=3)
        else:
            self.end_direction = end_direction
        # Correct the tangent directions:
        if vdot(self.start_direction, -function(t0) + function(t0+delta)) < 0:
            self.start_direction = -self.start_direction
        if vdot(self.end_direction, -function(t1-delta) + function(t1)) < 0:
            self.end_direction = -self.end_direction
        # Sample points:
        self.sample_points = sample_points(function, self.interval, sample_size=sample_size)
        f_sample = [ft for t,ft in self.sample_points]
        self.length = sum([abs(f_sample[i-1]-f_sample[i])
                          for i in range(1,len(self.sample_points))])
        # Extreme point:
        t_sample = [t for t,ft in self.sample_points]
        ext_t, ext_p = extreme_point(function, self.interval, t_sample)
        ext_direction = extreme_direction(function, ext_t, self.interval)
        
        self.extreme_parameter = ext_t
        self.extreme_point = ext_p
        self.extreme_direction = ext_direction
    def __str__(self):
        s = "SplittableFunction:"
        s += "\nfunction: "+self.name
        s += "\ninterval: "+str(self.interval)
        s += "\nextreme_parameter: "+str(self.extreme_parameter)
        s += "\nextreme_point:     "+str(self.extreme_point)
        s += "\nstart_direction:   "+str(self.start_direction)
        s += "\nend_direction:     "+str(self.end_direction)
        s += "\nextreme_direction: "+str(self.extreme_direction)
        return s
    def split(self):
        left = SplittableFunction(function = self.function,
                                  name = self.name,
                                  interval = [self.interval[0], self.extreme_parameter],
                                  start_direction = self.start_direction,
                                  end_direction = self.extreme_direction
                                  )
        right = SplittableFunction(function = self.function,
                                  name = self.name,
                                  interval = [self.extreme_parameter, self.interval[1]],
                                  start_direction = self.extreme_direction,
                                  end_direction = self.end_direction
                                  )
        return left,right
    # box_size: returns box_size:BoxSize
    # where box_size.f_size is a rough estimate of the bounding box,
    # and box_size.t_size is the same but for the parameter t.
    def box_size(self, samples=50): # Ad hoc: samples = 5*sample_size
        t_f_sample = sample_points(self.function, self.interval, sample_size=samples)
        t_xmin = min(t_f_sample, key=(lambda tf:tf[1].real))
        t_xmax = max(t_f_sample, key=(lambda tf:tf[1].real))
        t_ymin = min(t_f_sample, key=(lambda tf:tf[1].imag))
        t_ymax = max(t_f_sample, key=(lambda tf:tf[1].imag))
        t_size = max(abs(t_xmax[0]-t_xmin[0]), abs(t_ymax[0]-t_ymin[0]))
        f_size = max(abs(t_xmax[0]-t_xmin[1]), abs(t_ymax[1]-t_ymin[1]))
        #return t_size, f_size
        return BoxSize(t_size, f_size)

#==============================================================
# Section 6.2:  Routines needed to initialize SplittableFunction
#==============================================================

# -----------------
# extreme_direction
# -----------------
# Given function f and parameter value t. Assume that t corresponds to an extremal
# point of f, so the tangent at t is parallel to the chord.
# Find the unit direction vector of f at t.
# Args:
# Args:
# - f: ComplexValuedFunction
# - t: float
# - interval: [float,float]
# Returns:
# - complex
def extreme_direction(f, t, interval):
    # At t compute the tangent direction in two ways:
    # 1. from the function
    # 2. from the chord.
    ZERO = 1e-3
    int_len = abs(-interval[0]+ interval[1])
    d = min(int_len/1000, 0.001)
    utangent1 = tangent_direction(f, t, delta=d, algorithm=3)
    utangent2 = tangent_direction(f, t, delta=-d, algorithm=3)
    if vdot(utangent1, utangent2) < 0:
        utangent2 = -utangent2
    utangent = utangent1 + utangent2
    utangent = utangent/abs(utangent)
    try:
        chord = -f(interval[0])+ f(interval[1])
        utangent_chord = chord/abs(chord)
    except ZeroDivisionError: # Abandon the tangent computed from the chord
        utangent_chord = utangent
    # If the two tangent directions agree, use the one computed from the chord,
    # otherwise the one computed from the function:
    if ((abs(utangent_chord - utangent) < ZERO)
             or (abs(utangent_chord + utangent) < ZERO)):
        direction = utangent_chord
    else:
        direction = utangent
    # Correct the tangent direction:
    if vdot(direction, -f(t) + f(t+d)) < 0:
        direction = -direction
    return direction

# -------------
# extreme_point
# -------------
# Given a function f:R->C and an interval [a,b], find in the interval
# the extremal point, meaning the point f(t) with a<=t<=b (or b<=t<=a)
# which is the most distant from the line through f(a), f(b).
# Args:
#    f: ComplexValuedFunction
#    interval: [float,float]
# Returns:
#    float   (the parameter value t for the extreme point)
#    complex (the extreme point)
def extreme_point(f, interval, sample_ts=None):
    ACCURACY = 1e-8
    MAX_ROUNDS = 50
    LOOP_CASE = 1e-3
    a,b = interval
    if a > b:
        a,b = b,a
    if sample_ts is None:
        N = 10
        ts = [(N-i)*a/N + i*b/N for i in range(N+1)] # sample parameter values
    else: ##########  KOE
        ts = sample_ts
        N = len(sample_ts) + 1
    fs = [f(t) for t in ts]                       # sample points
    A,B = fs[0],fs[-1]
    chord = -A+B
    loop_case = abs(chord) < LOOP_CASE
    if loop_case:
        farthest_i = max(range(len(fs)),
                     key=(lambda j: abs(fs[j]-(A+B)/2)))
    else:
        farthest_i = max(range(len(fs)),
                     key=(lambda j: abs(cross(fs[j]-A, chord))))
    if farthest_i in (0,len(fs)-1):
        # Straight line?
        return (a+b)/2, f((a+b)/2)
    t0,t1 = ts[farthest_i-1], ts[farthest_i+1] # Search interval
    count = 0
    while t1-t0 > ACCURACY:
        if count > MAX_ROUNDS:
            raise Exception("extreme_point: MAX_ROUNDS reached!")
        count += 1
        sample_t = [(N-i)*t0/N + i*t1/N for i in range(N+1)]
        if loop_case:
            farthest_t = max([t for t in sample_t],
                         key=(lambda x: abs(f(x)-(A+B)/2)))
        else:
            farthest_t = max([t for t in sample_t],
                         key=(lambda x: abs(cross(f(x)-A, chord))))
        t0,t1 = max(farthest_t - 2*(t1-t0)/N, t0), min(farthest_t + 2*(t1-t0)/N, t1)
    t = (t0+t1)/2
    return t, f(t)

class NudgeError(Exception):
    def __init__(self,t0,t1,t2, d0,d2, message=''):
        self.t0=t0
        self.t1=t1
        self.t2=t2
        self.d0=d0
        self.d2=d2
        self.message = message

# -------------
# sample_points
# -------------
# Given a function f(t):R->C (a parametric curve), and a parameter interval [a,b],
# find sample points on the arc r=f(t), a<=t<=b.
# Try to find sample points that are roughly equally spaced in the plane.
# Return the sample points as a list [(t,f(t))], sorted with t running
# from min(a,b) to max(a,b).
# The plane is viewed as the complex number plane.
# Args:
# - f:             callable (R->C)
# - interval:      [float,float]
# - sample_size:   integer (size of the required sample, including end points)
# - custom_sample: [float]
# Returns:
# - [(float,complex)] ( =[(t,f(t))] )
# Idea:
# First iteration:
# Make initial set of sample points (smaller than the final). Then find
# (1) the pair of neighbouring sample points farthest apart from each other;
# (2) the sample point closest to its two neighbours.
# Remove the sample point (2).
# Between the pair of sample points (1) add one or two new sample points
# (one if the required sample size is already reached, two otherwise).
# Iterate.
# Second iteration:
# Do another iteration where each parameter value is nudged
# a little in order to achieve that the sample point is equally distant from its
# two neighbours.
def sample_points(f, interval, sample_size=6):
    # nudge: Imagine a parabola F with vertical axis such that
    # F(t0)=d0, F(t1)=0, F(t2)=d2. Find t such that F(t)=(d0+d2)/2.
    # Returns [complex].
    def nudge(t0,t1,t2, d0,d2):
        try:
            idet = 1 / ((t0-t1)*(t1-t2)*(t2-t0))
        except ZeroDivisionError:
            raise NudgeError(t0,t1,t2, d0,d2,"sample_points: nudge: some of t0,t1,t2 are equal")
        A = idet * ((t2-t1)*d0 -(t0-t1)*d2)
        B = (1/2) * idet * (-((t2-t1)**2)*d0 + ((t0-t1)**2)*d2)
        return solve2(A,B,-(d0+d2)/2)
    # Consider ratio = min(d)/max(d) where d is the distance of any two
    # neighbouring sample points on the curve. This is the measure of
    # how good (=equally-spaced) the set of sample points is. Ideal is ratio=1.
    # The following constants determine what will be accepted (experimental values):
    ACCEPT_RATIO_LO = 1/3 # (first iteration)
    ACCEPT_RATIO_HI = 3/4 # (second iteration)
    #
    N = max(2,sample_size)
    MAX_ROUNDS = N//2 + 100 # Ad hoc
    a,b = interval
    if a > b:
        a,b = b,a
    fa,fb = f(a),f(b)
    M = 1+N//2 # Size of initial sample is M+1 (including a,b)
    sample_t = [(M-i)*a/M + i*b/M for i in range(M+1)] # [t]
    sample_tf = [(t,f(t)) for t in sample_t]           # [(t,f(t))]
    reached_N = False
    #best_ratio, best_sample_tf = 0, None
    best_ratio, best_sample_tf = 0, sample_tf  ##############  KOE
    for count in range(MAX_ROUNDS):
        # (1) Find the pair of neighbouring sample points f(i),f(i+1)
        # which maximizes |f(i)-f(i+1)|:
        long_i, long_d = 0, 0.
        d_list = []
        for i in range(len(sample_tf)-1):
            d = abs(sample_tf[i][1] - sample_tf[i+1][1])
            d_list.append(d)
            if d > long_d:
                long_i, long_d = i, d
        # (2) Find the sample point f(i)
        # which minimizes |f(i-1)-f(i)| + |f(i)-f(i+1)|:
        short_i, short_dd = 0, 2*long_d + 1
        dd_list = []
        for i in range(1,len(sample_tf)-1):
            dd = (abs(sample_tf[i-1][1] - sample_tf[i][1])
                + abs(sample_tf[i][1] - sample_tf[i+1][1]))
            dd_list.append(dd)
            if dd < short_dd:
                short_i, short_dd = i, dd
        # Check if we have acceptable result:
        min_d, max_d = min(d_list), max(d_list)
        if (not reached_N) and (len(sample_tf) == N):
            reached_N = True
            #print("Reached N="+str(N)+" at round "+str(count))
        try:
            if min_d/max_d > best_ratio:
                best_ratio, best_sample_tf = min_d/max_d, sample_tf
        except ZeroDivisionError:
            raise Exception("sample_points: ZeroDivisionError when computing min_d/max_d")
        if reached_N and best_ratio >= ACCEPT_RATIO_LO:
            #print("First iteration: Accepted result at round "+str(count))
            break
        # Not acceptable:
        # Create new sample points ("sub_sample") between pair (1)
        aa,bb = sample_tf[long_i][0], sample_tf[long_i + 1][0]
        if len(sample_tf) < N: # Not yet constructed N sample points
            # Number of points in sub_sample = K-1 = 2.
            # Adding these and removing the one point (2) grows
            # the set of sample points by 1 at each iteration round.
            K = 3
        else: # N reached
            # Number of points in sub_sample = K-1 = 1.
            # Adding this and removing the one point (2) keeps
            # the size of the set of sample points fixed.
            K = 2
        sub_sample_t = [(K-i)*aa/K + i*bb/K for i in range(1,K)] # Excludes aa,bb
        sub_sample_tf = [(p,f(p)) for p in sub_sample_t]         # [(t,f(t))]
        # Update sample_tf
        # First, remove short_i:
        sample_tf.pop(short_i)
        if short_i <= long_i:
            long_i -= 1
        # Second, insert the sub_sample:
        sample_tf[long_i+1: long_i+1] = sub_sample_tf
    # Now best_sample_tf contains the best result so far.
    # Go to nudging iteration
    sample_tf = best_sample_tf
    for nudge_count in range(MAX_ROUNDS):
        nudged_t = []
        for i in range(1,len(sample_tf)-1):
            t0,f0 = sample_tf[i-1]
            t1,f1 = sample_tf[i]
            t2,f2 = sample_tf[i+1]
            d0, d2 = -abs(f0-f1), abs(f1-f2)
            try:
                t1_nudges = nudge(t0,t1,t2, d0, d2)
            except NudgeError as e:
                t1_nudges = [] ################  KOE
            new_t1s = [t1 + delta/2 for delta in t1_nudges]
            new_t1s = [tt for tt in new_t1s if t0 < tt < t2]
            if len(new_t1s) == 1:
                nudged_t.append(new_t1s[0])
            else:
                #print("Wrong number of nudges for t = "+str(t1))
                nudged_t.append(t1) # No nudge
        sample_tf = ([(a,fa)]
                    + [(t,f(t)) for t in nudged_t]
                    +[(b,fb)]
                    )
        sample_tf.sort(key=(lambda tf:tf[0])) # Just to make certain
        d_list = []
        for i in range(len(sample_tf)-1):
            d = abs(sample_tf[i][1] - sample_tf[i+1][1])
            d_list.append(d)
        max_d = max(d_list)
        min_d = min(d_list)
        if min_d / max_d > ACCEPT_RATIO_HI:
            #print("Second iteration (nudge): Accepted result at round "+str(nudge_count))
            break
    return sample_tf

# -----------------
# tangent_direction
# -----------------
# Compute, for the input arc, a direction vector of the tangent at
# the input parameter value.
#
# Args:
# - f:     callable (R->C)
# - t:     float;
# - delta: a small increment for t, to be used in the calculations,
#          positive or negative:
#          the tangent is calculated from points
#          f(t), f(t+delta), f(t+2*delta), ..., so the sign of delta
#          determines which of the two directions along the curve is
#          taken into account;
# - algorithm: one of 0,1,2,3 (default=3): the choice of the algorithm.
#              If a value greater than 3 is inserted, the value 3 is used.
#              If a value less than 0 is inserted, the value 0 is used.
#
# Returns: A unit direction vector [dx,dy] of the tangent at f(t); the orientation
#          of the vector is arbitrary. 
#
# Note: The tangent is calculated from points
#       f(t), f(t+delta), f(t+2*delta), ...
# Algorithms:
# 0: uses two points:   f(t) and f(t+delta)
# 1: uses three points: f(t), f(t+delta), f(t+2*delta) (fits a circle)
# 2: uses four points:  f(t), f(t+delta), ... (fits a curve of degree 2)
# 3: uses eight points: f(t), f(t+delta), ... (fits a curve of degree 3)
def tangent_direction(f, t, delta, algorithm=3):
    n = algorithm
    n = max(n,0) # If n < 0, value 0 is used.
    n = min(n,3) # If n > 3, value 3 is used.
    if n == 0: # line through f(t) and f(t+delta)
        return (f(t+delta) - f(t)) / abs(f(t+delta) - f(t))
    if n == 1: # fit a circle to the curve
        f0, f1, f2 = f(t), f(t+delta), f(t+2*delta)
        x0,y0 = f0.real, f0.imag
        x1,y1 = f1.real, f1.imag
        x2,y2 = f2.real, f2.imag
        c1 = (-y2+y1)*(x2-x0) + (x2-x1)*(y2-y0)
        c2 = (x2-x1)*(x2-x0) + (y2-y1)*(y2-y0)
        dx = c1*(-y0+y1) + c2*(x0-x1)
        dy = c1*(x0-x1) + c2*(y0-y1)
        return complex(dx,dy) / abs(complex(dx,dy))
    if 2 <= n:
        # fit a curve of degree 2: ax^2+bxy+cy^2+hx+ky=m
        # or of degree 3: ax^3+bx^2y+cxy^2+dy^3+ex^2+fxy+gy^2+hx+ky=m
        zs = []
        z0 = f(t)
        # Make z0 origo. Then m=0 and (dx,dy)=(h,k).
        for i in range(((n+1)*(n+2))//2 - 2):
            zs.append(f(t+(i+1)*delta) - z0)
        scaling = 0.
        for z in zs:
            scaling += abs(z)
        scaling /= (2*len(zs))
        try:
            zs = [z/scaling for z in zs]
        except ZeroDivisionError:
            mess = "tangent_direction"
            mess += "\n    Error: perhaps the parametric curve is locally constant?"
            raise ParametricCurveError(mess)
        matrix = []
        for z in zs:
            x,y = z.real, z.imag
            if n == 2:
                matrix.append( [x**2, x*y, y**2, x, y] )
            if n == 3:
                matrix.append( [ x**3, (x**2)*y, x*(y**2), y**3,
                                 x**2, x*y, y**2, 
                                 x, y ] )
        reduced = reducematrix(matrix)
        dx,dy = reduced[0][0],reduced[0][1]
        return complex(dx,dy) / abs(complex(dx,dy))

# ------------
# reducematrix
# ------------
# Currently needed only to handle cusps!
#
# Given an n x (n+1) matrix (n>=2), reduce it to a 1x2 matrix by means
# of row operations, always discarding one row and one column,
# and keeping the two right-most columns.
def reducematrix(a):
    if len(a) <= 1:
        return a
    imax = jmax = 0
    amax = a[0][0]
    for i in range(len(a)): # search for largest element
        for j in range(len(a[i])-2):
            if abs(a[i][j]) > amax:
                imax,jmax,amax = i,j, float(a[i][j])
    rowmax = []
    try:
        for j in range(len(a[imax])): # Divide row a[imax] by amax
            rowmax.append( a[imax][j] / amax )
    except ZeroDivisionError:
        for j in range(len(a[imax])): # Divide by 1
            rowmax.append( a[imax][j])
    newmatrix = []
    for i in range(len(a)): # 
        if i == imax:
            continue
        newrow = []
        for j in range(len(a[i])):
            if j == jmax:
                continue
            c = a[i][jmax]
            newrow.append( a[i][j] - c*rowmax[j] )
        newmatrix.append( newrow )
    if len(newmatrix)==1:
        return newmatrix
    else:
        return reducematrix(newmatrix)


#||<>||=======================================================================
#||<>||    Section 7
#||<>||=======================================================================
#||<>||    Approximate a parametric curve by a Bezier arc (algorithm)
#||<>||=======================================================================

class ParametricCurveError(Exception):
    def __init__(self,message):
        self.message = message
    def __str__(self):
        return self.message

#==============================================================
# Section 7.1:  Error measure
#==============================================================

# --------------
# half_hausdorff
# --------------
# This is the error measure: how much does the Bezier arc (represented by cp4)
# deviate from the complex valued function (represented by sf)?
# Return (1) absolute error, and (2) error relative to approximate length of the arc.
# Args:
# - cp4:   [complex,complex,complex,complex] (control points)
# - sf:    SplittableFunction
# - level: integer 0..10 (working level)
# Returns:
# - float (absolute error)
# - float (relative error)
def half_hausdorff(cp4, sf, level=5):
    f = sf.function
    sample_f = [tf[1] for tf in sf.sample_points]
    max_error = 0
    for v in sample_f:
        _,w = bezier_closest_point(cp4, v, level)
        max_error = max(abs(v-w), max_error)
    return max_error, max_error/sf.length

#==============================================================
# Section 7.2:  High level routines
#==============================================================

# ----------
# simple_arc
# ----------
# Given sf:SplittableFunction (a very short arc), find an approximate(?)
# Bezier arc (4 control points), by a simple rule.
# Args:
# - sf: SplittableFunction
# Returns:
# - [complex,complex,complex,complex] (= cp4, control points)
# - float (error)
# except if the arc couldn't be created because of non-intersection or coincidence
# of the two rays, returns:
# - None,None
def simple_arc(sf):
    A = sf.function(sf.interval[0])
    B = sf.function(sf.interval[1])
    direction1 = sf.start_direction
    direction2 = sf.end_direction
    C = intersection_of_rays([A,direction1], [B, -direction2]) # apex
    if C is None: # 
        return None,None
    cp4 = circular_like_arc_in_tangent_triangle(A,B,C)
    return cp4, half_hausdorff(cp4, sf, level=5)[0]

# ---------------
# very_simple_arc
# ---------------
# Given sf:SplittableFunction (a very very short arc), find an approximate(?)
# Bezier arc (4 control points), by a very very simple rule.
# Args:
# - sf: SplittableFunction
# Returns:
# - [complex,complex,complex,complex] (= cp4, control points)
# - float (error)
def very_simple_arc(sf):
    p0 = sf.function(sf.interval[0])
    p3 = sf.function(sf.interval[1])
    direction1 = sf.start_direction
    direction2 = -sf.end_direction
    #if vdot(direction1, -p0+p3) < 0:
    #    print("very_simple_arc: correcting start direction")
    #    direction1 = -direction1
    #if vdot(direction2, -p3+p0) < 0:
    #    print("very_simple_arc: correcting end direction")
    #    direction2 = -direction2
    chord = abs(p0-p3)
    cp4 = [p0,
           p0 + direction1 * chord/3,
           p3 + direction2 * chord/3,
           p3]
    return cp4, half_hausdorff(cp4, sf, level=5)[0]

# ----------------
# approx_sf_by_cp4
# ----------------
# Given sf:SplittableFunction, find an approximate Bezier arc (4 control points).
# Args:
# - sf:                   SplittableFunction
# - approx_params:        ApproxParameters
# - total_box_size:       BoxSize
# Returns:
# - [complex,complex,complex,complex] (= cp4, control points)
# - float (absolute error)
# - boolean (True: simple solution; False: not simple solution)
# except occasionally (when creating simple_arc failed), returns:
# - None,None,False
def approx_sf_by_cp4(sf,
                     approx_params,
                     total_box_size
                     ):
    # TO DO: jos funktio tekee silmukan? (Vai luotetaanko splittiin?)
    short_interval_limit = total_box_size.t_size * approx_params.short_interval
    # /100 is experimental:
    short_arc_limit      = total_box_size.f_size * approx_params.short_arc / 100
    if sf.length <= short_arc_limit / 10:
        #print("\napprox_sf_by_cp4: Doing VERY simple arc because:")
        #print("sf.length = "+str(sf.length)+" <= short_arc_limit/10 = "+str(short_arc_limit/10))
        cp4, abs_error = very_simple_arc(sf)
        return cp4, abs_error, True # (cp4, error, simple)
    elif sf.length <= short_arc_limit:
        # Use simple rule to create the arc.
        #print("\napprox_sf_by_cp4: Doing simple arc because:")
        #print("sf.length = "+str(sf.length)+" <= short_arc_limit = "+str(short_arc_limit))
        cp4, abs_error = simple_arc(sf)
        if cp4 is None: # simple_arc failed
            return None,None,False
        return cp4, abs_error, True # (cp4, error, simple)
    elif abs(sf.interval[0]-sf.interval[1]) <= short_interval_limit:
        # Use simple rule to create the arc.
        #print("\napprox_sf_by_cp4: Doing simple arc because:")
        #print("interval length = "+str(abs(sf.interval[0]-sf.interval[1]))+" <= short_interval_limit = "+str(short_interval_limit))
        cp4, abs_error = simple_arc(sf)
        if cp4 is None: # simple_arc failed
            return None,None,False
        return cp4, abs_error, True # (cp4, error, simple)
    # Tangent data from sf:
    p0 = sf.function(sf.interval[0])
    p3 = sf.function(sf.interval[1])
    K = sf.extreme_point
    direction1 = sf.start_direction
    direction2 = -sf.end_direction
    direction0 = sf.extreme_direction
    try:
        t_cp4s = bezier_from_tangent_data(p0, p3, K,
                                      direction1, direction2, direction0,
    #                             restrict01=True,
    #                             respect_directions=True,
    #                             wild_handle_limit=2, # Ad hoc limit
                                 print_info=False
                                 )
    except NoSolutionError as e:
        raise NoSolutionError(e.message)
    if len (t_cp4s) == 0:
        raise NoSolutionError(".. approx_sf_by_cp4: t_cp4s empty")
    # Compute errors and choose best:
    error_cp4s = [[half_hausdorff(cp4, sf, level=5)[0], cp4]
                      for t,cp4 in t_cp4s]
    best_error, best_cp4 = min(error_cp4s, key=(lambda x:x[0])) # [error,cp4]
    # Ad hoc: Make new tries by shortening the handles in the same proportion:
    M = 20
    count = 0
    for x in [i/M for i in range(1,M)]:
        k = -2*x*x*x + 3*x*x # floats in (0,1), densest close to 0 and 1
        try_cp4 = [cp4[0],
                  k*cp4[0] + (1-k)*cp4[1],
                  k*cp4[3] + (1-k)*cp4[2],
                  cp4[3]]
        try_error = half_hausdorff(try_cp4, sf, level=5)[0]
        if try_error >= best_error - 0.001:
            break # Break at first point of no improvement
        else:
            relative = (best_error - try_error)# /  best_error
            best_error, best_cp4 = try_error, try_cp4
            count += 1
    return best_cp4, best_error, False   # (cp4, error, not simple)

# --------------------
# approx_sf_by_cp4list
# --------------------
# Given sf:SplittableFunction, find an approximate succession of butting Bezier arcs
# (each with 4 control points).
# Args:
# - sf:             SplittableFunction
# - custom_params:  [float]
# - subdiv_options: SubdivisionOptions
# - approx_params:  ApproxParameters
# - total_box_size: BoxSize
# Returns:
# - [[complex,complex,complex,complex]] (= cp4list = [cp4] )
# - float (absolute error)
# Note: Non-recursive version.
def approx_sf_by_cp4list(sf,
                         custom_params,
                         subdiv_options,
                         approx_params,
                         total_box_size
                         ):
    # Compute MAX_ABS_ERROR (maximum allowed absolute error during computations).
    # This is something we really cannot compute.
    # We need the maximum error in the (1) coordinates of calculations (coordinates
    # for the curve coming from the definition of sf.function).
    # We have approx_params.allowed_error which is the desired maximum error in the
    # (2) coordinates of the final plot (Gimp's window).
    # At the final phase of the program there happens scaling from coordinates (1)
    # to coordinates (2). But the scaling factor will be known only at the time of
    # scaling. So now we must do guessing and hope for the best.
    # Guesses:
    # 1. The "size" fsize of the curve (sf.function in sf.interval) is computed from
    #    sf.sample_points very roughly.
    # 2. The final window (Gimp) is 1000x1000.
    # 3. The scaling is done so that the plot will fill the window.
    # Then the scaling factor is: scaling_factor = 1000/fsize.
    # So: MAX_ABS_ERROR = approx_params.allowed_error / scaling_factor
    f_sample = [ft for t,ft in sf.sample_points]
    max_x = max(z.real for z in f_sample)
    min_x = min(z.real for z in f_sample)
    max_y = max(z.imag for z in f_sample)
    min_y = min(z.imag for z in f_sample)
    fsize = max(max_x-min_x, max_y-min_y) * 1.1
    scaling_factor = 1000 / fsize
    MAX_ABS_ERROR = approx_params.allowed_error / scaling_factor
    MAX_ROUNDS = approx_params.max_rounds
    result_cp4list = []
    
    subdivision = subdivision_points(sf.function, # [SubDivPoint]
                                     sf.interval,
                                     custom_params,
                                     subdiv_options,
                                     approx_params,
                                     )
    subdiv_points = [sd.parameter for sd in subdivision]
    
    for subdiv_count in range(len(subdiv_points)-1):
        sd_start = subdiv_points[subdiv_count]
        sd_end = subdiv_points[subdiv_count+1]
        sd_name = 'subdivision '+str(subdiv_count)+' '+str([sd_start,sd_end])
        
        work_sf = SplittableFunction(sf.function, # work piece: SplittableFunction
                                     [sd_start, sd_end],
                                     name = sd_name,
                                     sample_size = len(sf.sample_points)
                                     )
        
        pending_sf_list = []  # pending: [SplittableFunction]
        total_error = 0.
        for work in range(MAX_ROUNDS):
            try:
                cp4, abs_err, simple = approx_sf_by_cp4(work_sf,  # Main call
                                                approx_params,
                                                total_box_size
                                                )
                if cp4 is None: # Tried to make simple_arc and failed.
                    #print("approx_sf_by_cp4list: Failed to make simple_arc. Going to split.")
                    pass
                elif (abs_err <= MAX_ABS_ERROR):# or simple:
                    # Good.
                    # Store cp4. Update the work piece and the pending list:
                    result_cp4list.append(cp4)
                    total_error = max(total_error, abs_err)
                    try:
                        work_sf = pending_sf_list[0]
                        pending_sf_list = pending_sf_list[1:]
                        continue # Next round
                    except IndexError: # pending_sf_list empty. Work done.
                        break
                else:
                    #print(". Error is "+str(abs_err)+". No good. Going to split.")
                    pass
            except NoSolutionError as e:
                #print(". No solution. Going to split.")
                pass
            # Not good. Split. Update the work piece and the pending list:
            sf1, sf2 = work_sf.split()
            work_sf = sf1
            pending_sf_list = [sf2] + pending_sf_list
        else: # If come here, MAX_ROUND was not enough to get the work done.
            raise Exception("MAX_ROUND "+str(MAX_ROUNDS)+" was not enough to get the work done.")
    return result_cp4list, total_error

# -----------
# arc2cp4list
# -----------
# Given a parametric arc (a function and an interval), find an approximate
# succession of butting Bezier arcs (each with 4 control points),
# already fitted in the desired plot region
# (return also the origo on the screen and the scaling factor).
# Args:
# - image
# - function:       callable (R->C)
# - interval:       [float,float]
# - custom_params:  [float]
# - name:           string
# - subdiv_options: SubdivisionOptions;
# - approx_params:  ApproxParameters
# - plotsettings:   CoordinateSettings;
# Returns:
# - [[complex,complex,complex,complex]] (= cp4list = [cp4] )
# - float (absolute error)
# - complex (the origo of the plot on the screen)
# - float   (the scaling factor of the function in the plot)
def arc2cp4list(image,
             function,
             interval,
             custom_params,
             name,
             subdiv_options,
             approx_params,
             plotsettings,
             ):
    sf = SplittableFunction(function = function,
                            interval = interval,
                            name = name,
                            sample_size = approx_params.sample_size)
    cp4list, abs_error = approx_sf_by_cp4list(sf, # Main call
                                              custom_params,
                                              subdiv_options,
                                              approx_params,
                                              total_box_size = sf.box_size()
                                              )
    # In Gimp: Turn upside down:
    cp4list = [[z.conjugate() for z in cp4] for cp4 in cp4list]
    # Scale and translate the plot:
    if plotsettings.fit_window:
        padding = plotsettings.padding
        padding = min(image.height/3, image.width/3, padding) # !
        w,h = image.width, image.height
        NW,SE = cp4list_BB(cp4list, Gimp=True) # BB
        Width, Height = abs(NW.real - SE.real), abs(NW.imag - SE.imag)
        try:
            scale = min((w-2*padding)/Width, (h-2*padding)/Height)
        except ZeroDivisionError:
            try:
                scale = (w-2*padding)/Width
            except ZeroDivisionError:
                try:
                    scale = (h-2*padding)/Height
                except ZeroDivisionError:
                    raise Exception("Plot is one point?")
        origo = (-scale*(NW+SE) + complex(w,h)) / 2
    else:
        origo = complex(plotsettings.origo[0], plotsettings.origo[1]) # complex
        scale = plotsettings.scale
    cp4list = [[origo + scale*z for z in cp4] for cp4 in cp4list]
    return cp4list, abs_error, origo, scale


#||<>||=======================================================================
#||<>||    Section 8
#||<>||=======================================================================
#||<>||    Preprocessing: Building subdivision (cusp points, inflection points,...)
#||<>||=======================================================================

#==============================================================
# Section 8.1:  Auxiliary geometric routines
#==============================================================

def angle_between_vectors(v,w):
    vw = v.real*w.real + v.imag*w.imag
    vv = v.real*v.real + v.imag*v.imag
    ww = w.real*w.real + w.imag*w.imag
    try:
        return acos( vw / sqrt(vv*ww) )
    except ZeroDivisionError as e:
        mess = "angle_between_vectors: "+str(e)
        raise ParametricCurveError(mess)
    except ValueError as e:
        arg = vw / sqrt(vv*ww)
        if arg >= 1.:
            return 0.
        elif arg <= -1.:
            return pi
        else:
            mess = "angle_between_vectors: "+str(e)
            raise ParametricCurveError(mess)

# --------------
# circle_radius2
# --------------
# Return the squared radius of the circle through the three points.
def circle_radius2(a,b,c):
    ab = a-b
    ac = a-c
    bc = b-c
    acbc  = ac.real*bc.real + ac.imag*bc.imag
    abxbc = cross(bc,ab)
    try:
        z = acbc/abxbc
    except:
        return None
    ab2 = abs(ab)**2
    R2 = .25 * ab2 * ( 1 + z**2 )
    return R2

#==============================================================
# Section 8.2:  Other auxiliary routines
#==============================================================

# -----------
# extrapolate
# -----------
# Given a vector-valued function g and two points (t1,g(t1)) and
#(t2,g(t2)), and a third value for the argument t,
# compute (t, g(t)) by linear extrapolation.
def extrapolate(t, t1, t2, gt1, gt2):
    return ((t2-t)*gt1 + (t-t1)*gt2) / (t2-t1)

# ------------------
# quadratic_extremum
# ------------------
# For a function f with f(x0-c)=left, f(x0)=middle, f(x0+c)=right, 
# return the minimum or maximum point by approximating with a parabola.
def quadratic_extremum(x0, c, left, middle, right):
    nomin = float(left - right)
    denom = float(left - 2*middle + right)
    try:
        x = nomin/(2*denom)
    except ZeroDivisionError: # straight line
        return None
    y = ( denom*x*x - nomin*x )/2 + middle
    return [x0+c*x, y]

# ------------
# find_minimum
# ------------
# Assume that the real function g(x) has a single minimum in [lo,hi].
# Returns:
# - the minimum point as (x,g(x)),
# - a short interval containing x for the minimum.
def find_minimum(g, lo, hi, accuracy=1e-5, max_accuracy=1e-10, tolerance=1e-10):
    class xgx:
        def __init__(self, g, x):
            self.x = x
            self.gx = g(x)
    def find_smallest_i(a): # a is an array of elements of type xgx
        smallest_i, smallest_gx = [0], a[0].gx
        for i in range(1,len(a)):
            if a[i].gx < smallest_gx:
                smallest_i, smallest_gx = [i], a[i].gx
            elif a[i].gx == smallest_gx:
                smallest_i.append(i)
        return smallest_i
    def middle_point(g,interval):
        x1,x2 = interval
        x = (x1+x2)/2
        return x,g(x)
    accuracy = min(accuracy, max_accuracy)
    a = [xgx(g, lo), 
         xgx(g, 0.75*lo + 0.25*hi),
         xgx(g, 0.5*lo + 0.5*hi),
         xgx(g, 0.25*lo + 0.75*hi),
         xgx(g, hi)]
    z = 4 # highest index in a
    while abs(a[0].x - a[z].x) > accuracy:
        smallest_i = find_smallest_i(a)
        if len(smallest_i) == 1:
            si = smallest_i[0]
            if si == 0:
                lx, mx, hx = a[0].x, a[1].x, a[2].x
                a = [a[0],
                     xgx(g, 0.5*lx + 0.5*mx),
                     a[1],
                     xgx(g, 0.5*mx + 0.5*hx),
                     a[2]]
            elif si == z:
                lx, mx, hx = a[z-2].x, a[z-1].x, a[z].x
                a = [a[z-2],
                     xgx(g, 0.5*lx + 0.5*mx),
                     a[z-1],
                     xgx(g, 0.5*mx + 0.5*hx),
                     a[z]]
            else:
                lx, mx, hx = a[si-1].x, a[si].x, a[si+1].x
                a = [a[si-1],
                     xgx(g, 0.5*lx + 0.5*mx),
                     a[si],
                     xgx(g, 0.5*mx + 0.5*hx),
                     a[si+1]]
        elif len(smallest_i) == 2:
            s1, s2 = smallest_i
            if s2-s1 != 1: # Should not happen but happens (computing accuracy?)
                s1,s2 = min(s1,s2), max(s1,s2)
                gs2 = a[s2].gx
                for ai in a[s1:s2]:
                    if abs(ai.gx - gs2) > tolerance:
                        mess = "find_minimum: impossible: not in tolerance"
                        raise ParametricCurveError(mess)  # SLOPPY?
                interval = [a[s1].x, a[s2].x]
                return middle_point(g, interval), interval
            lx, hx = a[s1].x, a[s2].x
            a = [a[s1],
                 xgx(g, 0.75*lx + 0.25*hx),
                 xgx(g, 0.5*lx + 0.5*hx),
                 xgx(g, 0.25*lx + 0.75*hx),
                 a[s2]]
        else:
            list_smallest = [[a[sm_i].x, g(a[sm_i].x)] for sm_i in smallest_i]
            interval = [list_smallest[0][0], list_smallest[-1][0]]
            return middle_point(g, interval), interval
    smallest_i = find_smallest_i(a)
    if len(smallest_i) == 1:
        si = smallest_i[0]
        interval = [a[si].x, a[si].x]
        return middle_point(g, interval), interval
    elif len(smallest_i) == 2:
        s1, s2 = smallest_i
        interval = [a[s1].x, a[s2].x]
        return middle_point(g, interval), interval
    else:
        list_smallest = [[a[sm_i].x, g(a[sm_i].x)] for sm_i in smallest_i]
        interval = [list_smallest[0][0], list_smallest[-1][0]]
        return middle_point(g, interval), interval

# --------------------------
# directionvectors_for_curve
# --------------------------
# Return a list [..., [t,tangent(t)], ...], where tangent(t) is, roughly,
# a direction vector of the tangent of the input function f at t.
#
# Args:
# - f:       a parametric curve (f:R->C) having no cusp
#            points in the open interval (lo,hi);
# - lo, hi:  the lowest and highest parameter values of the curve;
# - fvalues: a list [t,f(t)] of points densely along the curve.
# Returns: a list [..., [t,tangent(t)], ...], where tangent(t) is,
#          roughly, a direction vector of the input function f at t.
#
# In fact, generally, tangent(t) is approximately the derivative f'(t).
# It is assumed that there are no cusp points except possibly at the
# end points. If there is a cusp at either end point, then the computed
# tangent(t) gives the correct tangent in that end point and points in
# the general direction of f'(t) close to that end point but has
# arbitrary length. (In a cusp point we usually have f'(t) = 0.)
def directionvectors_for_curve(f,lo,hi, fvalues):
    if len(fvalues) < 30:
        return []
    ftangents = [] # list of [t,f'(t)]
    ## Find tangents at points not first or last.
    t0,f0 = fvalues[0]
    t1,f1 = fvalues[1]
    for t2,f2 in fvalues[2:]:
        diffleft  = f1-f0
        diffright = f2-f1
        tangleft  = diffleft/(t1-t0)
        tangright = diffright/(t2-t1)
        tang = extrapolate(t1, t0,t2, tangleft, tangright )
        ftangents.append( [ t1, tang ] )
        t0,f0 = t1,f1
        t1,f1 = t2,f2
    ## Find tangent at first point:
    t0,f0 = fvalues[0]
    t1,f1 = fvalues[1]
    delta = t1-t0
    tangent_at_start = tangent_direction(f,t0, delta)
    # Turn the vector to point in the direction of f'(lo).
    f01 = f1-f0
    if tangent_at_start.real*f01.real + tangent_at_start.imag*f01.imag < 0:
        tangent_at_start = -tangent_at_start
    len01 = abs(f01)
    lentang = abs(tangent_at_start)
    try:
        coeff = len01 / ( (t1-t0)*lentang )
        tangent_at_start = coeff*tangent_at_start
    except ZeroDivisionError:
        pass
    ftangents = [[ t0,tangent_at_start]] + ftangents
    ## Find tangent at last point:
    t0,f0 = fvalues[-2]
    t1,f1 = fvalues[-1]
    delta = t1-t0
    tangent_at_end = tangent_direction(f,t1, -delta)
    # Turn the vector to point in the direction of f'(hi).
    f01 = f1-f0
    if tangent_at_end.real*f01.real + tangent_at_end.imag*f01.imag < 0:
        tangent_at_end = -tangent_at_end
    len01 = abs(f01)
    lentang = abs(tangent_at_end)
    try:
        coeff = len01 / ( (t1-t0)*lentang )
        tangent_at_end = coeff*tangent_at_end
    except ZeroDivisionError:
        #print("directionvectors_for_curve: ZeroDivisionError at last tangent")
        pass
    ftangents.append( [ t1,tangent_at_end] )
    return ftangents

# ----------------
# is_straight_line
# ----------------
# Decide if the input arc is a straight line segment.
# If the arc is a loop, return False, disregarding if it is practically one point.
#
# Args:
# - f: a parametric arc (f:R->C);
# - start, end: the starting and ending values of the parameter;
# - rel_accuracy: determines how dense subdivision the algorithm uses
#                 (default=0,00001);
# - tolerance: determines how large deviations from a straight line is
#              tolerated (default=1e-8)).
# Returns: boolean
def is_straight_line(f, start, end, rel_accuracy=1e-5, tolerance = 1e-8):
    f0,f3 = f(start), f(end)
    f0f3 = f3-f0
    lenf0f3 = abs(f0f3)
    if lenf0f3 > tolerance:
        u = f0f3/lenf0f3
    else: # A loop
        return False
    def is_on_line(point):
        return abs(cross(point-f0,u)) < tolerance
    accuracy = float(end-start)*rel_accuracy
    product = 2
    nsteps = 3
    stept = (end-start)/nsteps
    while stept > accuracy:
        param = start + stept
        point = f(param)
        stop = end - stept/2.
        while param < stop:
            if not is_on_line(point):
                return False
            param += stept
        product *= nsteps
        nsteps = product+1
        stept = (end-start)/nsteps
    return True

# -----------
# arc_size_ge
# -----------
# Tell if a certain rough measure of the
# size of the bounding block of the arc (f,lot,hit) is greater than
# 'test_element'.
# Args:
# - f:            a parametric curve (f:R->C);
# - lot, hit:     the lowest and highest parameter values of the curve;
# - test_element: float
# Returns True if the arc is, roughly of greater size than 'test_element'.
# Note: This routine is applied to check if an arc is very short.
def arc_size_ge(f, lot, hit, test_element):
    at = [1., 1./2, 1./4, 3./4, 1./8., 3./8., 5./8., 7./8.]
    f0 = f(lot)
    xmax = xmin = f0.real
    ymax = ymin = f0.imag
    for t in at:
        ft = f((1-t)*lot + t*hit)
        xmax = max(xmax,ft.real)
        xmin = min(xmin,ft.real)
        ymax = max(ymax,ft.imag)
        ymin = min(ymin,ft.imag)
        if xmax-xmin+ymax-ymin > test_element:
            return True
    return False

# ------------------
# curve_step_by_step
# ------------------
# Return a subdivision of a parametric curve, positioning the
# points more densely where the curvature is high.
#
# Args:
# - f:             a parametric arc (f:R->C);
# - start, end:    the starting and ending values of the parameter;
# - steps:         the number of subdivision intervals to be used in the
#                  initial phase of the algorithm;
# - minrefinement: a float number which limits how short subdivision
#                  intervals are allowed (default=0.001);
# - maxbend:       a float number to control how sharp bend is allowed
#                  in each subdivision interval (default=10).
#
# Returns: a list [ ..., [t,f(t)], ... ] where each t is a parameter
#          value and f(t) is the point on the curve (f(t)=[x,y]).
def curve_step_by_step(f,start, end, steps, minrefinement=0.001, maxbend=10):
    mincos2 = cos(maxbend * pi/180)**2
    lo,hi = start, end
    reversed = (lo > hi) 
    if reversed:
        lo,hi = hi, lo
    stept = (hi-lo)/float(steps)
    minstep = stept*minrefinement
    fvalues = []
    t = lo
    while t <= hi-stept:
        fvalues.append([t,f(t)])
        t += stept
    fvalues.append([hi,f(hi)])
    first = 0 # first to study in refining the steps array
    while True:
        try:
            t1,f1 = fvalues[first]
            t2,f2 = fvalues[first+1]
            t3,f3 = fvalues[first+2]
        except IndexError:
            break
        if t2-t1 < minstep:
            first += 1
            continue
        f12 = f2-f1
        f23 = f3-f2
        dot = f12.real*f23.real + f12.imag*f23.imag
        f12sq = abs(f12)**2
        f23sq = abs(f23)**2
        if dot**2 > f12sq * f23sq * mincos2: # bend not too sharp
            first += 1
            continue
        t12, t23 = (t1+t2)/2, (t2+t3)/2
        added = [ [t12,f(t12)] , [t2,f2], [t23,f(t23)] ]
        fvalues = fvalues[:first+1] + added + fvalues[first+2:]
    if not reversed:
        return fvalues
    else:
        return fvalues[::-1]

# --------------------------
# exists_an_inflection_point
# --------------------------
# Check existence of at least one inflection point
# in interval lot < t < hit.
# 
# Args:
# - f:        a parametric arc (f:R->C);
# - lot, hit: the end points of the interval;
# - fvalues:  a list [t,f(t)] of points densely along the curve.
# Returns: Boolean.
#
# Notes:
# 1. Returns True if arc deemed to be straight line.
# 2. Code extracted from routine inflection_points.
def exists_an_inflection_point(f,lot,hit, fvalues):
    ZERO = 1e-12
    class PointCrossData:
        def __init__(self, t0, t1, t2, cross0, cross1, cross2):
            self.t012 = [t0, t1, t2]
            self.cross012 = [cross0, cross1, cross2]
    if is_straight_line(f,lot,hit):
        #print("exists_an_inflection_point: straight_line")
        return True
    fvalues_turns = []
    f0, f1 = fvalues[0][1], fvalues[1][1]
    f01 = f1-f0
    cross_array = [0.]
    for i in range(1, len(fvalues)-1):
        f2 = fvalues[i+1][1]
        f12 = f2-f1
        cro = cross(f12,f01)
        cross_array.append(cro)
        f1 = f2
        f01 = f12
    cross_array.append(0.)
    # Discard nodes with cross = 0 both from fvalues and cross_array:
    reduced_fvalues = []
    reduced_cross_array = []
    for i in range(len(fvalues)):
        if abs(cross_array[i]) > ZERO: ######  KOE
            reduced_fvalues.append(fvalues[i])
            reduced_cross_array.append(cross_array[i])
    if len(reduced_fvalues) < 2: # Effort to rectify an error in experiment
        return False
    point_cross_data_array = []
    for i in range(len(reduced_fvalues)):
        if i == 0:
            t0 = reduced_fvalues[0][0]
            t1 = reduced_fvalues[0][0]
            t2 = reduced_fvalues[1][0]
            cross0 = reduced_cross_array[0]
            cross1 = reduced_cross_array[0]
            cross2 = reduced_cross_array[1]
        elif i == len(reduced_fvalues)-1:
            t0 = reduced_fvalues[-2][0]
            t1 = reduced_fvalues[-1][0]
            t2 = reduced_fvalues[-1][0]
            cross0 = reduced_cross_array[-2]
            cross1 = reduced_cross_array[-1]
            cross2 = reduced_cross_array[-1]
        else:
            t0 = reduced_fvalues[i-1][0]
            t1 = reduced_fvalues[i][0]
            t2 = reduced_fvalues[i+1][0]
            cross0 = reduced_cross_array[i-1]
            cross1 = reduced_cross_array[i]
            cross2 = reduced_cross_array[i+1]
        pc = PointCrossData(t0,t1,t2, cross0, cross1, cross2)
        point_cross_data_array.append(pc)
    interval_cross_data_array = []
    try:
        prev = point_cross_data_array[0]
    except IndexError: # straight line
        #print("exists_an_inflection_point: IndexError")
        return True
    for curr in point_cross_data_array[1:]:
        if (curr.cross012[1] > 0.) != (prev.cross012[1] > 0.):
            #print("Found inflection point")
            return True
    return False

#==============================================================
# Section 8.3:  Creating subdivision
#==============================================================

class SubDivPoint:
    """Contains data about a parametric curve at one subdivision point.
    
    Attributes:
        type:             the type of the subdivision point (string);
        parameter:        the value of the parameter at the point;
        functionvalue:    the value of the parametric function
                          (i, e., a point on the curve);
    """
    subdivtypes = ['custom',
                   'low end',
                   'high end',
                   'cusp',
                   'high curvature',
                   'low curvature',
                   'straight end',
                   'inflection',
                   'angle turn'
                   ]
    def __init__(self,
            type,
            parameter,
            functionvalue,
            ):
        if type not in SubDivPoint.subdivtypes:
            mess = "Invalid type for SubDivPoint: "+type
            raise ParametricCurveError(mess)
        self.type          = type
        self.parameter     = parameter
        self.functionvalue = functionvalue
    def __str__(self):
        s = "SubDivPoint:"
        s += "\n  type:             " + self.type
        s += "\n  parameter:        " + str(self.parameter)
        s += "\n  functionvalue:    " + str(self.functionvalue)
        return s

# ------------
# basic_points
# ------------
# The first step in creating the subdivision on the parametric curve:
# Build a list consisting of the end points and custom points.
#
# Args:
# - f: callable    (f:R->C)
# - interval:      [float,float]
# - custom_params: [float]
# Returns: [SubDivPoint], sorted according to the field 'parameter'.
#
# Note: Thus, the returned list contains the subdivision points of the
# following types:
#    - start point  (type = 'low end'),
#    - custom point (type = 'custom'),
#    - end point    (type = 'high end').
def basic_points(f, interval, custom_params):
    ZERO = 1e-6
    lot,hit = interval
    custom = sorted(custom_params)
    custom = [t for t in custom if lot+ZERO < t < hit-ZERO]
    # Remove duplicates:
    if len(custom) > 0:
        custom1 = [custom[0]]
        for c in custom[1:]:
            if abs(c - custom1[-1]) > ZERO:
                custom1.append(c)
        custom = custom1
    bp = [SubDivPoint('low end', lot, f(lot))]      # start point
    for t in custom:
        bp.append(SubDivPoint('custom', t, f(t)))
    bp.append(SubDivPoint('high end', hit, f(hit))) # end point
    return bp

# -----------
# cusp_points
# -----------
# Create a sorted list of elements of class SubDivPoint for
# cusp points in interval lot < t < hit.
# 
# Args:
# - f:        parametric arc (f:R->C);
# - lot, hit: the end points of the interval;
# - steps:    the number of subdivision intervals used in the initial
#             phase of the algorithm;
# - rel_accuracy: a float number.
#
# Returns: a list of elements of class SubDivPoint for the cusp points
#          (type = 'cusp') in interval lot < t < hit.
#          The list is sorted according to the field 'parameter'.
#
# Note: In the program, this routine is not applied to the whole
#       user-input parametric curve. Instead, it is applied to some
#       smaller arc limited by previously created subdivision points.
def cusp_points(f,lot,hit,steps, rel_accuracy=1e-10):
    def potential_cusp(f0,f1,f2): # A cusp point ?
        f01 = f0-f1
        f12 = f1-f2
        lf01 = abs(f01)**2
        lf12 = abs(f12)**2
        dotprod = f01.real*f12.real + f01.imag*f12.imag
        if dotprod < 0:
            return True
        if dotprod**2 < 0.9 * lf01 * lf12:
            return True
        return False
    def is_flat(f0,f1,f2,f3, tolerance=1e-4):
        f03 = f3-f0
        f01 = f1-f0
        f23 = f3-f2
        f01xf03 = cross(f01,f03)
        f23xf03 = cross(f23,f03)
        f01len2 = abs(f01)**2
        f23len2 = abs(f23)**2
        f03len2 = abs(f03)**2
        c = (tolerance**2) * f03len2
        return (f01xf03**2 < c*f01len2) and (f23xf03**2 < c*f23len2)
    def measure_general(x, f0, f03):
        fx = f(x)
        f0x = fx-f0
        return -cross(f0x,f03)**2
    stept = (hit-lot)/float(steps)
    potential = []
    t0,t1,t2 = lot, lot+stept, lot+2*stept
    f0,f1,f2 = f(t0),f(t1),f(t2)
    prev = None
    while t2 < hit:
        if potential_cusp(f0,f1,f2):
            if prev is not None: # combine two intervals
                prev = [prev[0],t2]
                potential[-1] = prev
            else:
                prev = [t0,t2]
                potential.append(prev)
        else:
            prev = None
        t0,t1,t2 = t1,t2,t2+stept
        f0,f1 = f1,f2
        if t2<hit:
            f2 = f(t2)
    accuracy = rel_accuracy*(hit-lot)
    cusp_intervals = []
    for pot_lo, pot_hi in potential:
        t0, t3 = pot_lo, pot_hi
        flat = False
        round = 0
        #max_rounds = 3 # This puts also limit to the achieved accuracy.
        max_rounds = 6 # This puts also limit to the achieved accuracy.
        while (t3-t0 > accuracy) and (round < max_rounds):
            round += 1
            f0, f3 = f(t0), f(t3)
            f03 = f3-f0
            measure = partial(measure_general, f0=f0, f03=f03)
            try:
                _, interval = find_minimum(measure, t0, t3, 1e-10)
            except ParametricCurveError as e:
                mess = "cusp_points\n    "+str(e)
                gimp_message(mess)      # Give message
                #raise ParametricCurveError(mess) # but do not raise error.
                break
            t1, t2 = interval
            f1, f2 = f(t1), f(t2)
            if is_flat(f0,f1,f2,f3):
                flat = True
                #print("FLAT")
                break
            else:
                t0,t3 = t1,t2
        if not flat:
            cusp_intervals.append([t0,t3])
    vertices = []
    for t0,t3 in cusp_intervals:
        ave = (t0+t3)/2
        vertices.append(SubDivPoint( 'cusp', ave, f(ave)))
    vertices.sort(key=(lambda x: x.parameter))
    return vertices

# -------------------------
# high_low_curvature_points
# -------------------------
# Create a sorted list elements of class SubDivPoint in interval
# lot < t < hit for all or only some of the following:
# - points of local maxima of curvature;
# - points of local minima of curvature;
# - start and end points of straight line segments,
# according to the input Boolean arguments.
# 
# Args:
# - f:             a parametric arc (f:R->C);
# - lot, hit:      the end points of the interval;
# - find_high:     Boolean;
# - find_low:      Boolean;
# - find_straight: Boolean;
# - steps:         the number of subdivision intervals used in the algorithm.
#
# Returns: a list of elements of class SubDivPoint for the
#            - points of local maxima of curvature
#             (type = 'high curvature') if find_high = True;
#            - points of local minima of curvature if find_high=True;
#             (type = 'low curvature') if find_low = True;
#            - start and end points of straight line segments if
#             (type = 'straight end') if find_straight = True;
#          in interval lot < t < hit.
#          The list is sorted according to the field 'parameter'
#
# Note: In the program, this routine is not applied to the whole
#       user-input parametric curve. Instead, it is applied to some
#       smaller arc limited by previously created subdivision points.
def high_low_curvature_points(f, lot, hit, steps, short_interval_limit,
                              find_high, find_low, find_straight):
    
    class PointRadiusData:
        def __init__(self, t, ft, radius2):
            self.t = t
            self.ft = ft
            self.radius2 = radius2
    limit = 1e-10 # because of computation accuracy
    high_limit = 1.+limit
    low_limit = 1.-limit
    circle_limit = 1e-6 # to test circle cases
    if is_straight_line(f,lot,hit):
        return []
    fvalues = curve_step_by_step(f,lot,hit, steps) # list of [t,f(t)]
    pr = [PointRadiusData(fvalues[0][0], fvalues[0][1], None)]
    for i in range(1,len(fvalues)-1):
        a = fvalues[i-1][1]
        b = fvalues[i][1]
        c = fvalues[i+1][1]
        r2 = circle_radius2(a,b,c)
        pr.append(PointRadiusData(fvalues[i][0], fvalues[i][1], r2))
    pr.append(PointRadiusData(fvalues[-1][0], fvalues[-1][1], None))
    hcps = []
    if len(pr)<=2:
        circle_case = True
    else:
        pr_not_None = [prd for prd in pr if not(prd.radius2 is None)]
        max_radius2 = max(pr_not_None, key=lambda x:x.radius2).radius2
        min_radius2 = min(pr_not_None, key=lambda x:x.radius2).radius2
        circle_case = (max_radius2 - min_radius2 < circle_limit)
    if circle_case:
        return hcps
    i = 1
    prev_added = lot # lot is not added but this is used as a reference point
    while 1 <= i < len(pr)-3:
        i += 1                  # Now 2 <= i < len(pr)-2
        r2 = pr[i].radius2
        if pr[i].radius2 is None: # radius2 = infinite
            j = i
            while (pr[j+1].radius2 is None) and (j < len(pr)-3):
                j += 1
            if j > i: # Found consecutive cases with radius2 = infinite
                if find_straight:
                    t1, ft1 = pr[i].t, pr[i].ft
                    t2, ft2 = pr[j].t, pr[j].ft
                    if (arc_size_ge(f, prev_added, t2, short_interval_limit)
                      and arc_size_ge(f, t2, hit, short_interval_limit)):
                        hcps.append(SubDivPoint('straight end', t1, ft1))
                        hcps.append(SubDivPoint('straight end', t2, ft2))
                        #if not (prev_added < t1 < t2):
                        #    print("1: "+str((prev_added,t1,t2)))
                        prev_added = t2
                i = j+1
                continue
        if pr[i].radius2 is None:
            if find_low:
                t = pr[i].t
                ft = pr[i].ft
                if (arc_size_ge(f, prev_added, t, short_interval_limit)
                  and arc_size_ge(f, t, hit, short_interval_limit)):
                    hcps.append( SubDivPoint('low curvature', t, ft))
                    #if not (prev_added < t):
                    #    print("2: "+str((prev_added,t)))
                    prev_added = t
            continue
        t0, t1, t2 = pr[i-1].t, pr[i].t, pr[i+1].t
        delta = min(abs(t1-t0), abs(t2-t1))
        r20, r21, r22 = pr[i-1].radius2, pr[i].radius2, pr[i+1].radius2
        if (r20 is None) or (r22 is None):
            continue
        if r20 < r21 * low_limit > r22:
            t0, t2 = t1 - delta, t1 + delta
            f0, f2 = f(t0), f(t2)
            r20 = circle_radius2(pr[i-2].ft, f0, pr[i].ft)
            r22 = circle_radius2(pr[i].ft, f2, pr[i+2].ft)
            if (r20 is None) or (r22 is None):
                continue
            t_ext, r2_ext = quadratic_extremum(t1, delta, r20, r21, r22)
            if find_low:
                if (arc_size_ge(f, prev_added, t_ext, short_interval_limit)
                  and arc_size_ge(f, t_ext, hit, short_interval_limit)):
                    hcps.append(SubDivPoint('low curvature', t_ext, f(t_ext)))
                    #if not (prev_added < t_ext):
                    #    print("3: "+str((prev_added,t_ext)))
                    prev_added = t_ext
        elif (r20 > r21 * high_limit < r22):
            t0, t2 = t1 - delta, t1 + delta
            f0, f2 = f(t0), f(t2)
            r20 = circle_radius2(pr[i-2].ft, f0, pr[i].ft)
            r22 = circle_radius2(pr[i].ft, f2, pr[i+2].ft)
            if (r20 is None) or (r22 is None):
                continue
            t_ext, r2_ext = quadratic_extremum(t1, delta, r20, r21, r22)
            if find_high:
                if (arc_size_ge(f, prev_added, t_ext, short_interval_limit)
                  and arc_size_ge(f, t_ext, hit, short_interval_limit)):
                    hcps.append(SubDivPoint('high curvature', t_ext, f(t_ext)))
                    #if not (prev_added < t_ext):
                    #    print("4: "+str((prev_added,t_ext)))
                    prev_added = t_ext
        elif (-limit < r21-r22 < limit):
            t_ext = (t1+t2)/2
            if find_high and (r20 > r21*high_limit):
                if (arc_size_ge(f, prev_added, t_ext, short_interval_limit)
                  and arc_size_ge(f, t_ext, hit, short_interval_limit)):
                    hcps.append(SubDivPoint('high curvature', t_ext, f(t_ext)))
                    #if not (prev_added < t_ext):
                    #    print("5: "+str((prev_added,t_ext)))
                    prev_added = t_ext
            elif find_low and (r20 < r21*low_limit):
                if (arc_size_ge(f, prev_added, t_ext, short_interval_limit)
                  and arc_size_ge(f, t_ext, hit, short_interval_limit)):
                    hcps.append(SubDivPoint('low curvature', t_ext, f(t_ext)))
                    
                    ##############  KOE
                    if not (prev_added < t_ext):
                        print("6: "+str((prev_added,t_ext)))
                    
                    prev_added = t_ext
        else:
            pass
    hcps.sort(key=(lambda x: x.parameter))
    return hcps

# -----------------
# inflection_points
# -----------------
# Create a sorted list of elements of class SubDivPoint for inflection
# points in interval lot < t < hit.
# 
# Args:
# - f:        a parametric arc (f:R->C);
# - lot, hit: the end points of the interval;
# - steps:    the number of subdivision intervals used in the algorithm.
#
# Returns: A list of elements of class SubDivPoint for the inflection
#          points (type = 'inflection') in interval lot < t < hit. The list
#          is sorted according to the field 'parameter'.
#
# Note: In the program, this routine is not applied to the whole
#       user-input parametric curve. Instead, it is applied to some
#       smaller arc limited by previously created subdivision points.
def inflection_points(f,lot,hit,steps, short_interval_limit):
    class PointCrossData:
        def __init__(self, t0, t1, t2, cross0, cross1, cross2):
            self.t012 = [t0, t1, t2]
            self.cross012 = [cross0, cross1, cross2]
    class IntervalCrossData:
        def __init__(self, t0 ,t1, t2, t3, cross0, cross1, cross2, cross3):
            self.t0123 = [t0 ,t1, t2, t3]
            self.cross0123 = [cross0, cross1, cross2, cross3]
    if is_straight_line(f,lot,hit):
        return []
    fvalues = curve_step_by_step(f,lot,hit, steps) # list of [t,f(t)]
    fvalues_turns = []
    f0, f1 = fvalues[0][1], fvalues[1][1]
    f01 = f1-f0
    cross_array = [0.]
    for i in range(1, len(fvalues)-1):
        f2 = fvalues[i+1][1]
        f12 = f2-f1
        cross_array.append(cross(f12,f01))
        f1 = f2
        f01 = f12
    cross_array.append(0.)
    # Discard nodes with cross = 0 both from fvalues and cross_array:
    reduced_fvalues = []
    reduced_cross_array = []
    for i in range(len(fvalues)):
        if cross_array[i] != 0.:
            reduced_fvalues.append(fvalues[i])
            reduced_cross_array.append(cross_array[i])
    point_cross_data_array = []
    for i in range(len(reduced_fvalues)):
        if i == 0:
            t0 = reduced_fvalues[0][0]
            t1 = reduced_fvalues[0][0]
            t2 = reduced_fvalues[1][0]
            cross0 = reduced_cross_array[0]
            cross1 = reduced_cross_array[0]
            cross2 = reduced_cross_array[1]
        elif i == len(reduced_fvalues)-1:
            t0 = reduced_fvalues[-2][0]
            t1 = reduced_fvalues[-1][0]
            t2 = reduced_fvalues[-1][0]
            cross0 = reduced_cross_array[-2]
            cross1 = reduced_cross_array[-1]
            cross2 = reduced_cross_array[-1]
        else:
            t0 = reduced_fvalues[i-1][0]
            t1 = reduced_fvalues[i][0]
            t2 = reduced_fvalues[i+1][0]
            cross0 = reduced_cross_array[i-1]
            cross1 = reduced_cross_array[i]
            cross2 = reduced_cross_array[i+1]
        pc = PointCrossData(t0,t1,t2, cross0, cross1, cross2)
        point_cross_data_array.append(pc)
    interval_cross_data_array = []
    try:
        prev = point_cross_data_array[0]
    except IndexError: # straight line
        return []
    for curr in point_cross_data_array[1:]:
        if (curr.cross012[1] > 0.) == (prev.cross012[1] > 0.):
            prev = curr
            continue
        t0, t1, t2 = prev.t012
        cross0, cross1, cross2 = prev.cross012
        t3 = curr.t012[-1]
        cross3 = curr.cross012[-1]
        ic = IntervalCrossData(t0,t1,t2,t3, cross0, cross1, cross2, cross3)
        interval_cross_data_array.append(ic)
        prev = curr
    infl_points = []
    prev_added = lot # lot is not added but this is used as a reference point
    for ic in interval_cross_data_array:
        t0, t1, t2, t3 = ic.t0123
        cross0, cross1, cross2, cross3 = ic.cross0123
        # Interpolate to get the inflection point:
        try:
            t12 = (t2*cross1 - t1*cross2) / (cross1 - cross2)
            t03 = (t3*cross0 - t0*cross3) / (cross0 - cross3)
            t = 0.5*(3*t12 - t03) # Interpolated
        except ZeroDivisionError:
            try:
                t = (t2*cross1 - t1*cross2) / (cross1 - cross2)
            except ZeroDivisionError:
                t = (t1+t2)/2
        if (arc_size_ge(f, prev_added, t, short_interval_limit)
          and arc_size_ge(f, t, hit, short_interval_limit)):
            infl_points.append( SubDivPoint('inflection', t, f(t)))
            prev_added = t
    infl_points.sort(key=(lambda x: x.parameter))
    return infl_points

# --------------
# turns_by_angle
# --------------
# Create a sorted list of elements of class SubDivPoint for points such
# that they divide the arc into smaller arcs, each of which turns
# (roughly) the same amount <= input angle.
#
# Args:
# - f:      a parametric arc (f:R->C);
# - lo, hi: the end points of the interval;
# - steps:  the number of subdivision intervals used in the algorithm;
# - angle:  the target angle; must be in (0, 360 degrees) (default=pi/2).
#
# Returns: A list of elements of class SubDivPoint for the angle turn
#          points (type = 'angle turn') in interval lot < t < hit.
#          The list is sorted according to the field 'parameter'.
#
# Note: Existence of inflection points is checked, and if any are present,
#       the work is abandoned and [] is returned.
#       Namely, the algorithm fails unless the arc turns all the way in the
#       same direction, i.e., constantly to the left or constantly to
#       the right.
#
# Note: In the program, this routine is not applied to the whole
#       user-input parametric curve. Instead, it is applied to some
#       smaller arc limited by previously created subdivision points.
def turns_by_angle(f,lo,hi, steps, angle=pi/2): 
    # Must have angle in (0,360 degrees). 
    if not (0 < angle < 2*pi):
        mess = "turns_by_angle: angle not valid."
        raise ParametricCurveError(mess)
    # first_turn_point_by_angle:
    # Find the first point where the curve has turned by angle.
    # - Always angle >= 0, in radians.
    # - turnright tells the direction in which the arc turns when moving
    #   from start to end, all the way.
    # - If forwards=False, the search is done backwards; in that case
    #   the caller must change turnright to the opposite value!
    # - ftangentsin is the output of directionvectors_for_curve,
    #   i.e., a list [..., [t,tangent(t)], ...], where tangent(t) is, roughly,
    #   a direction vector of f at t.
    def first_turn_point_by_angle(f,ftangentsin, angle, turnright, forwards=True):
        from cmath import exp as cexp
        c = cos(angle)
        s = sin(angle)
        def rotate(v, turnright):
            if turnright:
                return v * cexp(-1j*angle) # ccw
            else:
                return v * cexp(1j*angle) # ccw
        def check(comp,tang):
            if comp.real*tang.real + comp.imag*tang.imag < 0:
                return False
            if turnright:
                return cross(comp, tang) >= 0.
            else:
                return cross(comp, tang) <= 0.
        if forwards:
            ftangents = ftangentsin
        else:
            ftangents = ftangentsin[::-1] # reversed
        start, starttang = ftangents[0] # start of the next arc to be examined
        compare = rotate(starttang, turnright)
        prev,prevtang = start,starttang
        for cur, curtang in ftangents[1:]:
            if not check(compare,curtang):
                prev,prevtang = cur,curtang
                continue # not yet found a turn by angle
            # found!
            # Interpolation
            x1,tang1 = prev,prevtang
            x2,tang2 = cur, curtang
            cross1 = cross(compare, tang1)
            cross2 = cross(compare, tang2)
            midx = (cross2*x1-cross1*x2) / (cross2-cross1)
            midtang = extrapolate(midx, x1,x2, tang1, tang2 )
            return midx,midtang
        return None
    # find_turn_points:
    # Find the sequence of parameters to divide the arc lo<t<hi into
    # smaller arcs each turning the amount of angle (roughly).
    # Return the list and the remaining angle.
    def find_turn_points(f,ftangents, angle, turnright):
        result = []
        start, starttang = ftangents[0] # start of the next arc to be examined
        next_ftangents = ftangents
        while True:
            findnext = first_turn_point_by_angle(
                            f,
                            next_ftangents,
                            angle,
                            turnright,
                            forwards = True)
            if findnext is None:
                break
            next,nexttang = findnext
            next_ftangents = [ [next,nexttang] ]
            for mt,mftang in ftangents:
                if mt <= next:
                    continue
                next_ftangents.append([ mt,mftang ])
            result.append(next)
            start, starttang = next,nexttang
        remainder = angle_between_vectors( starttang, ftangents[-1][1] )
        return result, remainder
    tolerance = 0.015
    #tolerance = 0.5
    limit = 0.0000001
    fvalues = curve_step_by_step(f,lo,hi, steps=1000) # list of [t,f(t)]
    # The algorithm fails if there are inflection points.
    # Normally there should be none but it may happen:
    # Even if inflection points are searched for by inflection_points or
    # high_low_curvature_points, some may have been rejected because of
    # short_interval_limit.
    # Therefore check now.
    if exists_an_inflection_point(f,lo,hi, fvalues):
        #print("exists_an_inflection_point: True")
        return []
    ftangents = directionvectors_for_curve(f,lo,hi, fvalues)
    left_or_right = []
    t0,d0 = ftangents[0]
    for t1,d1 in ftangents[1:]:
        d0xd1 = cross(d0,d1)
        if d0xd1 >= limit:
            left_or_right.append(1)
        elif d0xd1 <= -limit:
            left_or_right.append(-1)
        else:
            left_or_right.append(0)
        t0,d0 = t1,d1
    vote = 0
    for to_right in left_or_right:
        vote += to_right
    turn_right = ( vote > 0 )
    not_all_same = ( abs(vote) < len(left_or_right) )
    if not_all_same:
        if is_straight_line(f,lo,hi):
            return [] # Aiheuttanee virheen joskus.
    turn_points, remainder = find_turn_points(f,ftangents, angle, turn_right)
    n = len(turn_points)
    total_angle = n*angle + remainder
    if remainder/angle < tolerance:
        in_tolerance = True
        turn_points = turn_points[:-1] # Drop the last one
    elif abs(angle-remainder)/angle < tolerance:
        in_tolerance = True # Now don't drop the last one!
    else: # Make a new search using the more exact new_angle
        in_tolerance = False
        new_angle = total_angle / (n+1)
        turn_points, new_remainder = find_turn_points(f,
                                        ftangents, new_angle, turn_right)
        new_n = len(turn_points)
        new_total_angle = new_n*new_angle + new_remainder
        if new_remainder/new_angle < tolerance:
            #print("new_remainder/new_angle = "+str(new_remainder/new_angle))
            in_tolerance = True
            #turn_points = turn_points[:-1]  # Drop the last one
        #elif abs(new_angle-new_remainder)/new_angle < tolerance:
        #    in_tolerance = True # Now don't drop the last one!
        #else:
        #    print("new_remainder/new_angle = "+str(new_remainder/new_angle))
        #    print("abs(new_angle-new_remainder)/new_angle = "+str(abs(new_angle-new_remainder)/new_angle))
        #    print("tolerance = "+str(tolerance))
        else:
            #print("new_remainder/new_angle = "+str(new_remainder/new_angle))
            in_tolerance = True # Now don't drop the last one! Don't check tolerance!
    #if not in_tolerance:
    #    mess = "turns_by_angle: Not in tolerance"
    #    raise ParametricCurveError(mess)
    typed_turn_points = []
    for t in turn_points:
        typed_turn_points.append(SubDivPoint('angle turn', t, f(t)))
    typed_turn_points.sort(key=(lambda x: x.parameter))
    return typed_turn_points

# ------------------
# subdivision_points
# ------------------
# Create a sorted list of the various types of subdivision points on
# the whole input curve.
#
# Args:
# - f:                 callable (f:R->C)
# - interval:          [float,float]
# - custom_parameters: [float]
# - subdiv_options:    SubdivisionOptions;
# - approx_parameters: ApproxParameters
# Returns:
# - 'rivps':       [SubDivPoint], sorted according to the field 'parameter'. 
def subdivision_points(f,
                       interval,
                       custom_params,
                       subdiv_options,
                       approx_parameters
                       ):
    #def row(type, t, ft): # For printing only
    #    str_ft = '[' + ', '.join('{: .3f}'.format(x) for x in ft) + ']'
    #    r ='{:<14}  {:>6.3f}  {}'.format(type,t, str_ft)
    #    r += '    t = '+repr(t)
    #    return r
    ## Initial phase: Collect basic points (start, end, and custom points)
    ## into a list.
    steps = approx_parameters.steps
    short_interval_limit = approx_parameters.short_interval
    # cbps will be basic points (= end points + custom points)
    bps = basic_points(f, interval, custom_params)
    ## Next phase: Find and add cusp points.
    if subdiv_options.cusp_points:
        # cbps will be basic points + cusp points
        cbps = [bps[0]]
        for i in range(len(bps)-1):
            arc_start, arc_end = bps[i].parameter, bps[i+1].parameter
            cps = cusp_points(f, arc_start, arc_end, steps)
            cbps = cbps + cps + [bps[i+1]]
    else:
        cbps = bps
    ## Next phase: Find and add inflection points.
    if subdiv_options.inflection_points:
        # icbps will be basic + cusp + inflection points
        icbps = [cbps[0]]
        for i in range(len(cbps)-1):
            arc_start, arc_end = cbps[i].parameter, cbps[i+1].parameter
            ips = inflection_points(f,
                        arc_start, arc_end,
                        steps, short_interval_limit
                        )
            # Remove those too close to arc_start or arc_end
            ok_lo = arc_start + approx_parameters.short_interval
            ok_hi = arc_end - approx_parameters.short_interval
            ips = [ip for ip in ips if ok_lo < ip.parameter < ok_hi]
            icbps = icbps + ips + [cbps[i+1]]
    else:
        icbps = cbps
    ## Next phase: Find and add high and/or low curvature points,
    ## together with end points of straight line segments.
    add_high = subdiv_options.high_curvature_points
    add_low = subdiv_options.low_curvature_points
    add_straight = subdiv_options.straight_end_points
    if add_high or add_low or add_straight:
        # hicbps will be
        # basic + cusp + inflection + high/low curvature + straight points
        hicbps = [icbps[0]]
        for i in range(len(icbps)-1):
            arc_start, arc_end = icbps[i].parameter, icbps[i+1].parameter
            hcps = high_low_curvature_points(f,
                        arc_start, arc_end,
                        steps, short_interval_limit,
                        add_high, add_low, add_straight
                        )
            # Remove those too close to arc_start or arc_end
            ok_lo = arc_start + approx_parameters.short_interval
            ok_hi = arc_end - approx_parameters.short_interval
            hcps = [hp for hp in hcps if ok_lo < hp.parameter < ok_hi]
            hicbps = hicbps + hcps + [icbps[i+1]]
    else:
        hicbps = icbps
    ## Next phase: Find and add turn points by angle.
    # (That is, find points splitting each arc into segments such that
    # in each segment the curve turns at most by the amount of angle.)
    angle = subdiv_options.angle
    if subdiv_options.angle_turn_points:
        # rivps will be
        # basic + cusp + inflection + high/low curvature + straight + angle turn points
        rivps = [hicbps[0]]
        for i in range(len(hicbps)-1):
            arc_start, arc_end = hicbps[i].parameter, hicbps[i+1].parameter
            # For angle turn points, check shortness of arc only about the whole arc
            # to be subdivided, and not about new subdivision arcs to be created.
            # (Otherwise the algorithm in 'turns_by_angle' might fail.)
            if arc_size_ge(f, arc_start, arc_end, short_interval_limit):
                rtps = turns_by_angle(f, arc_start, arc_end, steps, angle)
            else:
                rtps = []
            # Remove those too close to arc_start or arc_end
            ok_lo = arc_start + approx_parameters.short_interval
            ok_hi = arc_end - approx_parameters.short_interval
            rtps = [rtp for rtp in rtps if ok_lo < rtp.parameter < ok_hi]
            rivps = rivps + rtps + [hicbps[i+1]]
    else:
        rivps = hicbps
    rivps.sort(key=(lambda x: x.parameter))
    return rivps


#||<>||=======================================================================
#||<>||    Section 9
#||<>||=======================================================================
#||<>||    Data structures for controlling the approximation and plot
#||<>||=======================================================================

def gimp_message(txt, h=2):
    # 0 = message box; 1 = console; 2 = error console
    pdb.gimp_message_set_handler(h)
    pdb.gimp_message(txt)

#==============================================================
# Section 9.1:  Approximation
#==============================================================

class ApproxParameters(object):
    """Numerical parameters to control the approximation work.
    
    The program works by subdividing the input curve into a succession of
    arcs; the endpoints of the arcs will then be anchor points of the
    resulting Bezier curve. (The program may create also additional
    anchor points.) The rules for how the subdivision is built, is
    governed by the following attributes:
    
    Attributes (the same as initializatin arguments except for the last two):
        allowed_error:  maximum allowed error (obscure meaning, but is something
                        like absolute error in the final plot in pixels)
                        (default=0.3);
        max_rounds:     maximum number of rounds in some loop
                        (default=3);
        steps:          the number of steps to be used when searching
                        for special points
                        (default=500);
        short_interval: the upper limit of intervals (parameter interval!)
                        allowed in the subdivision, and also shorter arcs
                        are done with a simple rule;
                        this is supposed to be taken relative to the whole interval
                        (default=0.001);
        short_arc:      shorter arcs are done with a simple rule;
                        this is supposed to be taken relative to the length of the
                        whole curve
                        (default=0.005);
        sample_size:    integer, the number of sample points on each arc used
                        for error measure (default=10).
    """
    def __init__(self,
                 allowed_error  = 0.3,
                 max_rounds     = 200,
                 steps          = 500,
                 short_interval = 0.001,
                 short_arc      = 0.005,
                 sample_size    = 10
                 ):
        MIN_max_error, MAX_max_error = 0.045, 10 # arbitrary
        if not MIN_max_error <= allowed_error <= MAX_max_error:
            m1 = "Only max_errors in [{:1.6f}, {:1.6f}] are allowed.".format(MIN_max_error, MAX_max_error)
            m2 = "\n    Instead of {:1.6f}, the value {:1.6f} is used".format(allowed_error,self.allowed_error)
            m = m1+m2
            gimp_message(m) # Tell the forced change of allowed_error.
        self.allowed_error  = max(MIN_max_error, min(MAX_max_error, allowed_error))
        self.max_rounds     = max(0, max_rounds)
        self.steps          = steps
        self.short_interval = short_interval
        self.short_arc      = short_arc
        self.sample_size    = sample_size
        self.allowed_error = allowed_error
        self.max_rounds = max_rounds

class SubdivisionOptions:
    """Boolean parameters to manage which special points are searched for
    and included among subdivision points (in addition to the
    user-defined custom points if any).
    
    The default values should be safe. Other choices may cause the
    algorithm to fail but often may give interesting results.
    
    Attributes (all Boolean, the same as initialization arguments):
        cusp_points:           include cusp points
                               (default=True);
        high_curvature_points: include points of locally highest
                               curvature
                               (default=True);
        low_curvature_points:  include points of locally lowest
                               curvature
                               (default=True);
        straight_end_points:   include both start and end points
                               of straight line segments
                               (default=True);
        inflection_points:     include inflection points
                               (the program forces this to be
                               - False if 'low_curvature_points'=True,
                               - True if 'low_curvature_points'=False
                                        and 'angle_turn_points'=True)
                               (default=False);
        angle_turn_points:     include subdivision points to ensure
                               that each arc turns at most by the
                               amount of 'angle' (below)
                               (default=True).
        angle:                 the maximum allowed turn in each subdivision
                               arc (forced to between 30 and 180 degrees with
                               tolerance of 1e-8)
                               (default=90 degrees);
    """
    def __init__(self, 
            cusp_points           = True,
            high_curvature_points = True,
            low_curvature_points  = True,
            straight_end_points   = True,
            inflection_points     = False,
            angle_turn_points     = True,
            angle                 = pi/2
            ):
        MIN_angle, MAX_angle = pi/6-1e-8, pi+1e-8
        if not MIN_angle <= angle <= MAX_angle:
            m1 = "Only angles in [pi/6, pi] are allowed."
            m2 = "\n    Instead of {:1.6f}, the angle {:1.6f} radians is used".format(angle,self.angle)
            m = m1+m2
            gimp_message(m) # Tell the forced change of angle.
        self.angle                 = max(MIN_angle, min(angle, MAX_angle))
        self.cusp_points           = cusp_points
        self.high_curvature_points = high_curvature_points
        self.low_curvature_points  = low_curvature_points
        self.straight_end_points   = straight_end_points
        self.angle_turn_points     = angle_turn_points
        if low_curvature_points:
            self.inflection_points = False
        elif angle_turn_points:
            self.inflection_points = True
        else:
            self.inflection_points = inflection_points

#==============================================================
# Section 9.2:  Bounding box
#==============================================================
# ----------
# cp4list_BB
# ----------
# Bounding box of the stroke (Bezier curve segment) represented by cp4list.
# Return as list [NW, SE] of two corner points,
# where NW is top left, SE is bottom right. (The directions are indeed as
# on the screen: N is up, S is down, provided that the argument 'Gimp' is correct!)
# Args:
# - [[complex,complex,complex,complex]] (= cp4list = [cp4] )
# - Gimp: boolean
# Returns:
# - [NW, SE]: [complex,complex]
# Note: In this program it does not matter if the stroke is closed or not.
def cp4list_BB(cp4list, Gimp=True):
    hor = cp4list_tangent_points(cp4list, 1+0j)
    ver = cp4list_tangent_points(cp4list, 0+1j)
    anchors = [cp4[0] for cp4 in cp4list] + [cp4list[-1][-1]]
    xmax = max([v.real for v in ver+anchors])
    xmin = min([v.real for v in ver+anchors])
    ymax = max([v.imag for v in hor+anchors])
    ymin = min([v.imag for v in hor+anchors])
    if Gimp:
        return [complex(xmin,ymin), complex(xmax,ymax)]
    else:
        return [complex(xmin,ymax), complex(xmax,ymin)]

# ----------------------
# cp4list_tangent_points
# ----------------------
# Given a stroke (Bezier curve segment) represented by cp4list and a non-zero
# direction vector d,
# find the points where some arc has tangent parallel to d.
# Include cusps.
# Exception: For arcs which are all the way parallel, the point t=0.5 is chosen.
# Args:
# - path_vectors: gimp.Vectors
# - direction: complex (assumed non-zero)
# Returns:
# - [complex] (touching points)
def cp4list_tangent_points(cp4list, direction):
    balist = [BCurve.BezierArc(cp4=cp4) for cp4 in cp4list]
    bc = BCurve.BezierCurve(bezier_arcs = balist)
    return BezierCurve_tangent_points(bc, direction)

# --------------------------
# BezierCurve_tangent_points
# --------------------------
# Given a Bezier curve:Bcurve.BezierCurve (essentially list of butting
# Bezier arcs) and a non-zero direction vector d,
# find the points on B where the curve is parallel to d.
# Includes cusps and isolated anchors.
# For arcs which are all the way parallel to d, the point
# with t=0.5 is returned
# Args:
# - bcurve: BCurve.BezierCurve
# - direction: complex (assumed non-zero)
# Returns:
# - [complex]
def BezierCurve_tangent_points(bcurve, direction):
    if not bcurve.closed:
        balist = bcurve.bezier_arcs
    else: # closed
        # In Gimp a closed stroke is only marked to be closed.
        # We must append one more arc to close the gap:
        gap_ba = BCurve.BezierArc(cp4 = [bcurve.bezier_arcs[-1].cp4[-1],
                                         bcurve.tail_handle,
                                         bcurve.head_handle,
                                         bcurve.bezier_arcs[0].cp4[0]])
        balist = bcurve.bezier_arcs + [gap_ba]
    cp4s = [ba.cp4 for ba in balist]
    points = []
    for cp4 in cp4s:
        if len(cp4) == 1: # A one-anchore stroke
            points.append(cp4[0])
            continue
        tBs = bezier_curve_tangent_points(cp4, direction, restrict01=True)
        if tBs is None: # arc is straight line segment parallel to direction
            t = 0.5
            p = bezier_rv(t, cp4)
            points.append(p)
            #t = 1.
            #p = bezier_rv(t, cp4)
            #points.append(p)
        else:
            for t,p in tBs:
                points.append(p)
    return points

# ---------------------------
# bezier_curve_tangent_points
# ---------------------------
# Given an infinite Bezier curve B(t) (4 control points)
# and a non-zero direction vector d,
# find the points on B with B'(t) parallel to d.
# If restrict01=True, accept only those with 0<=t<=1.
# Return as a list of pairs [t,p] where t is the parameter value (float)
# and p is the touching point B(t) (complex).
# Exception: If d=0, or if the curve is contained on a line parallel to d,
# or if len)control_points) is not 4 (1-anchor stroke!),
# return None.
# Args:
# - control_points: [complex,complex,complex,complex]
# - direction:      complex (assumed non-zero)
# - restrict01:     boolean
# Returns: None or
# - list [[t,p]]: [[float,complex]]   (=[[t,B(t)],...])
def bezier_curve_tangent_points(control_points, direction, restrict01=True):
    if len(control_points) < 4:
        return None
    p0,p1,p2,p3 = control_points
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    # pij dot (direction perpendicular):
    p01d = -p01.real * direction.imag + p01.imag * direction.real
    p12d = -p12.real * direction.imag + p12.imag * direction.real
    p23d = -p23.real * direction.imag + p23.imag * direction.real
    #
    ts = solve_bernstein2_equation(p01d,p12d,p23d)
    if ts is None:
        return None
    if restrict01:
        ts = [t for t in ts if 0<=t<=1]
    tBs = [[t, bezier_rv(t, control_points)] for t in ts]
    return tBs


#==============================================================
# Section 9.3:  Plot
#==============================================================

class PlotSettings:
    """Manage how the final plot is positioned on the screen and whether to
    draw the axes.
    
    Attributes (the same as initalization arguments): 
        fit_window:   Boolean: determines if the plot is to be fit in
                      the window minus the paddings
                      (when 'fit_window' is True, the settings for
                      'origo', and 'scale' are ignored)
                      (default=True);
        padding:      float: the padding to be left empty on the screen
                      as a fraction of the window size when 'fit_window'
                      is True (it is forced that 0 <= padding <= 0.45)
                      (default=0.);
        origo:        a pair of floats: where to set the origo on the
                      screen in screen coordinates
                      (ignored if 'fit_window' is True)
                      (default=[500,500]);
        scale:        scaling factor from raw coordinates to screen
                      coordinates
                      (ignored if 'fit_window' is True)
                      (default=100.).
        draw_axes     boolean
    """
    def __init__(self, 
            fit_window = True,
            padding    = 0.,
            origo      = [500.,500.],
            scale      = 100.,
            draw_axes  = False
            ):
        self.fit_window = fit_window
        self.padding = float(max(padding, 0.))
        self.origo  = [float(origo[0]), float(origo[1])]
        self.scale = float(scale)
        self.draw_axes = draw_axes


#||<>||=======================================================================
#||<>||    Section 10
#||<>||=======================================================================
#||<>||    Main procedures
#||<>||=======================================================================

#==============================================================
# Section 10.1:  Top procedure called by various main procedures
#==============================================================
# Note: This is the correct procedure to call when the approximation code is
# embedded in some other program.

# ---------------
# paramcurve2path
# ---------------
# Top procedure called by various main procedures.
# Given a parametric function and an interval (a parametric arc),
# find an approximate path (Bezier curve).
# Draw the path if do_drawing=True.
# Args:
# - image:            gimp.Image
# - curve_name:       string
# - function:         callable (float->complex) (the parametric function)
# - interval:         [float,float]
# - closed:           boolean
# - fit_window:       boolean
# - padding:          float         (has effect if fit_window = True)
# - origo:            [float,float] (has effect if fit_window = False)
# - scale:            float         (has effect if fit_window = False)
# - draw_axes:        boolean
# - custom_params:    [float]
# - display_messages: boolean
# - extra:            None or dict (extra tweaking parameters)
# - do_drawing:       boolean
# Returns:
# - gimp.Vectors
def paramcurve2path(
             image,             # gimp.Image
             curve_name,        # string
             function,          # callable (float->complex)
             interval,          # [float,float]
             closed,            # boolean
             fit_window,        # boolean
             padding,           # float         (has effect if fit_window = True)
             origo,             # [float,float] (has effect if fit_window = False)
             scale,             # float         (has effect if fit_window = False)
             draw_axes,         # boolean
             custom_params,     # [float]
             display_messages,  # boolean
             extra = None,      # dict (extra tweaking parameters)
             do_drawing = False # boolean
            ):
    # Check potential extra tweaking parameters
    # (currently only one, 'boost_accuracy'):
    if not extra is None:
        try:
            boost_accuracy = extra['boost_accuracy']
            boost_accuracy = max(0, min(10, boost_accuracy))
        except KeyError:
            boost_accuracy = 0
        allowed_error = 0.5 /(1+boost_accuracy)
    else:
        allowed_error = 0.5 # Default, ad hoc
    subdiv_options = SubdivisionOptions(  # All default values: No options for the user!
                                        #cusp_points           = True,
                                        #high_curvature_points = True,
                                        #low_curvature_points  = True,
                                        #straight_end_points   = True,
                                        #inflection_points     = False,
                                        #angle_turn_points     = True,
                                        #angle                 = pi/2
                                        )
    approx_params = ApproxParameters( # No options for the user except those in extra:
                                     allowed_error = allowed_error,
                                     #max_rounds     = 200,   # Default
                                     #steps          = 500,   # Default
                                     #short_interval = 0.001, # Default
                                     #short_arc      = 0.005, # Default
                                     #sample_size    = 10     # Default
                                     )
    plotsettings = PlotSettings(fit_window = fit_window,
                                padding = padding,
                                origo = origo,
                                scale = scale,
                                draw_axes = draw_axes
                                )
    cp4list, abs_error, plot_origo, plot_scale = arc2cp4list( ######  Main call
            image,
            function,
            interval,
            custom_params,
            'koe',
            subdiv_options,
            approx_params,
            plotsettings,
            )
    final_error = abs_error * plot_scale
    balist = [BCurve.BezierArc(cp4=cp4) for cp4 in cp4list]
    if not closed:
        bc = BCurve.BezierCurve(bezier_arcs = balist,
                                curve_name = curve_name
                                )
    else:
        first_anchor, last_anchor = cp4list[0][0], cp4list[-1][-1]
        move = -last_anchor + first_anchor
        if abs(move) <= approx_params.allowed_error: # Close properly
            _,p1,p2,_ = cp4list[-1]
            new_head, new_tail = p2 + move, p1
            balist = balist[:-1]
        else: # Close stroke by a straight edge
            new_head, new_tail = first_anchor, last_anchor
        bc = BCurve.BezierCurve(bezier_arcs = balist,
                                head_handle = new_head,
                                tail_handle = new_tail,
                                closed      = True,
                                curve_name  = curve_name
                                )
    path = bc.bc2vectors_object(image)
    if do_drawing:
        pdb.gimp_image_undo_group_start(image)
        if draw_axes:
            axes_path = make_axes(image, plot_origo)
            axes_path.name = curve_name+'|axes'
            gimp_draw_vectors_object(image, axes_path, visible=True)
        gimp_draw_vectors_object(image, path, visible=True)
        pdb.gimp_image_undo_group_end(image)
    if display_messages:
        if final_error > 1e-2:
            str_error = "{:1.4f}".format(final_error)
        elif final_error > 1e-3:
            str_error = "{:1.5f}".format(final_error)
        elif final_error > 1e-4:
            str_error = "{:1.6f}".format(final_error)
        else:
            str_error = "{:1.3e}".format(final_error)
        m = "The plugin finished succesfully"
        m += "\n"+curve_name # Addition to 2.1
        m += "\nwith error measure (absolute error)\n"+str_error
        gimp_message(m)
    return path


#==============================================================
# Section 10.2:  Main procedures
#==============================================================

# -------------------------
# cartesian_curve2path_main
# -------------------------
# Main procedure for parametric curves x=x(t), y=y(t) in cartesian form,
# with x(t),y(t) read from the GUI.
# Returns:
# - gimp.Vectors
def cartesian_curve2path_main(
             image,             # gimp.Image
             #
             curve_name,        # string
             x_function,        # string, in Python syntax
             y_function,        # string, in Python syntax
             start_as_string,   # string, in Python syntax
             end_as_string,     # string, in Python syntax
             closed,            # boolean
             #
             fit_window,        # boolean
             padding,           # float (has effect if fit_window = True)
             origo_x,           # float (has effect if fit_window = False)
             origo_y,           # float (has effect if fit_window = False)
             scale,             # float (has effect if fit_window = False)
             draw_axes,         # boolean
             #
             param_string,      # string
             display_messages   # boolean
            ):
    curve_name = curve_name.strip()
    if curve_name == '': # blank
        curve_name = 'Curve' # Use this
    sx = x_function.strip()
    sy = y_function.strip()
    if (sx == '') or (sy == ''):
        raise Exception("Empty x(t) or y(t) in the GUI")
    else: # From the strings sx, sy construct function f(t): R -> R2
        source = 'def curve_function(t): return complex('+sx+','+sy+')'
        code = compile(source, 'koe', 'exec')
        exec(code, globals())
    interval = [float(eval(start_as_string)), float(eval(end_as_string))]
    if param_string.strip() == '': # blank
        custom_params = []
    else:
        try:
            list_or_tuple_params = eval(param_string)
            if type(list_or_tuple_params) in (int,float):
                list_or_tuple_params = [float(list_or_tuple_params)]
        except:
            if messages:
                gimp_message(("Something wrong in custom parameter values."
                              +"\nPlease check syntax and so on."))
            raise Exception(("Something wrong in custom parameter values."
                              +"\nPlease check syntax and so on."))
        custom_params = [float(p) for p in list_or_tuple_params]
    origo = [origo_x,origo_y]
    return paramcurve2path(
             image,             # gimp.Image
             curve_name,        # string
             curve_function,    # callable (float->complex)
             interval,          # [float,float]
             closed,            # boolean
             fit_window,        # boolean
             padding,           # float (has effect if fit_window = True)
             origo,             # [float,float] (has effect if fit_window = False)
             scale,             # float (has effect if fit_window = False)
             draw_axes,         # boolean
             custom_params,     # [float]
             display_messages,  # boolean
             do_drawing = True
             )

# ---------------------
# polar_curve2path_main
# ---------------------
# Main procedure for parametric curves r=r(t) in polar form, with r(t) read from
# the GUI.
# Returns:
# - gimp.Vectors
def polar_curve2path_main(
             image,             # gimp.Image
             #
             curve_name,        # string
             r_function,        # string, in Python syntax
             start_as_string,   # string, in Python syntax
             end_as_string,     # string, in Python syntax
             closed,            # boolean
             #
             fit_window,        # boolean
             padding,           # float (has effect if fit_window = True)
             origo_x,           # float (has effect if fit_window = False)
             origo_y,           # float (has effect if fit_window = False)
             scale,             # float (has effect if fit_window = False)
             draw_axes,         # boolean
             #
             param_string,      # string
             display_messages   # boolean
            ):
    curve_name = curve_name.strip()
    if curve_name == '': # blank
        curve_name = 'Curve' # Use this
    sr = r_function.strip()
    if (sr== ''):
        raise Exception("Empty r(t) in the GUI")
    else: # From the string sr construct function f(t): R -> R2
        sr = '('+sr+')'
        source = 'def curve_function(t): return complex('+sr+'*cos(t),'+sr+'*sin(t))'
        code = compile(source, 'koe', 'exec')
        exec(code, globals())
    interval = [float(eval(start_as_string)), float(eval(end_as_string))]
    if param_string.strip() == '': # blank
        custom_params = []
    else:
        try:
            list_or_tuple_params = eval(param_string)
            if type(list_or_tuple_params) in (int,float):
                list_or_tuple_params = [float(list_or_tuple_params)]
        except:
            if messages:
                gimp_message(("Something wrong in custom parameter values."
                              +"\nPlease check syntax and so on."))
            raise Exception(("Something wrong in custom parameter values."
                              +"\nPlease check syntax and so on."))
        custom_params = [float(p) for p in list_or_tuple_params]
    origo = [origo_x,origo_y]
    return paramcurve2path(
             image,             # gimp.Image
             curve_name,        # string
             curve_function,    # callable (float->complex)
             interval,          # [float,float]
             closed,            # boolean
             fit_window,        # boolean
             padding,           # float (has effect if fit_window = True)
             origo,             # [float,float] (has effect if fit_window = False)
             scale,             # float (has effect if fit_window = False)
             draw_axes,         # boolean
             custom_params,     # [float]
             display_messages,  # boolean
             do_drawing = True
             )

allowed_extra_parameters = ['boost_accuracy'] # Currently only one

# ------------------------------------
# cartesian_curve_file2path_main
# ------------------------------------
# Main procedure for parametric curves x=x(t), y=y(t) in cartesian form,
# with function read from a file.
# The contents of the input file:
# - function:      callable (float->complex) (mandatory)
# - curve_name:    string                    (optional)
# - interval:      [float,float]             (optional)
# - closed:        boolean                   (optional)
# - custom_params: [float,...]               (optional)
# Returns:
# - gimp.Vectors
def cartesian_curve_file2path_main(
             image,             # gimp.Image
             #
             curve_name,        # string
             input_file,        # file containing the function and possibly others
             start_as_string,   # string, in Python syntax
             end_as_string,     # string, in Python syntax
             closed,            # boolean
             #
             fit_window,        # boolean
             padding,           # float (has effect if fit_window = True)
             origo_x,           # float (has effect if fit_window = False)
             origo_y,           # float (has effect if fit_window = False)
             scale,             # float (has effect if fit_window = False)
             draw_axes,         # boolean
             #
             param_string,      # string
             display_messages   # boolean
            ):
    glob_d = globals()
    exec_d = glob_d.copy() # read and exec'd dict
    try:
        with open(input_file) as f:
            try:
                code = compile(f.read(), f.name, 'exec')
            except Exception as e:
                m = "Something wrong with your input file."
                m += "\nGot Error message:"
                m += "\n"+type(e).__name__ +': ' +str(e)
                raise Exception(m)
                #raise Exception(m+str(e))
            exec(code, exec_d)
    except IOError as e:
        m = "Forgot to choose the input File or gave the wrong file name?"
        m += "\nIOError:"+str(e)
        raise Exception(m)
    try:
        curve_function = exec_d['function']
    except KeyError:
        raise Exception("The input file does not contain definition of 'function")
    missing_in_file = []
    # Get curve name from file, or if not there, from GUI:
    try:
        curve_name = exec_d['curve_name']
    except KeyError: # failed
        missing_in_file.append(['curve_name', curve_name])
    # Get interval from file, or if not there, from GUI:
    try:
        interval = exec_d['interval']
    except KeyError: # failed
        interval = [float(eval(start_as_string)), float(eval(end_as_string))]
        missing_in_file.append(['interval', interval])
    # Get 'closed' from file, or if not there, from GUI:
    try:
        closed = exec_d['closed']
    except KeyError: # failed
        missing_in_file.append(['closed', closed])
    if display_messages:
        if len(missing_in_file) > 0:
            m = "The following were not found in the file,"
            m += "\nso they were taken from the GUI:"
            for name,value in missing_in_file:
                m += "\n    "+name +" = "+str(value)
            gimp_message(m)
    # See if there are extra parameters in the file:
    extra = dict()
    for parameter_name in allowed_extra_parameters:
        try:
            extra[parameter_name] = exec_d[parameter_name]
        except KeyError:
            pass
    # Test the function:
    try:
        test = (curve_function(interval[0]), curve_function(interval[0]),
                  curve_function((interval[0]+interval[1])/2))
    except Exception as e:
        m = "Something wrong with your function definition in the file."
        m += "\nGot error message:"
        m += "\n"+type(e).__name__ +': ' +str(e)
        raise Exception(m)
    for t in test:
        if type(t) != complex:
            raise Exception(("The return type of the function must be complex."
                             +"\nGot "+str(type(t))))
            #break
    # Get custom parameter lists:
    # First, form GUI:
    if param_string.strip() == '': # blank
        custom_params_GUI = []
    else:
        try:
            list_or_tuple_params = eval(param_string)
            if type(list_or_tuple_params) in (int,float):
                list_or_tuple_params = [float(list_or_tuple_params)]
        except Exception as e:
            m = ("Something wrong in custom parameter values in the GUI."
                +"\nPlease check syntax and so on.")
            if display_messages:
                gimp_message(m)
            m += "\nGot error message:"
            m += "\n"+type(e).__name__ +': ' +str(e)
            raise Exception(m)
            #raise Exception(m + '\nGot error message:\n' + str(e))
        custom_params_GUI = [float(p) for p in list_or_tuple_params]
    # Second, from file:
    try:
        custom_params_file = exec_d['custom_params']
    except KeyError:
        custom_params_file = []
    # Merge the lists:
    custom_params = custom_params_GUI + custom_params_file
    #
    origo = [origo_x,origo_y]
    # Main call
    return paramcurve2path(
             image,             # gimp.Image
             curve_name,        # string
             curve_function,    # callable (float->complex)
             interval,          # [float,float]
             closed,            # boolean
             fit_window,        # boolean
             padding,           # float (has effect if fit_window = True)
             origo,             # [float,float] (has effect if fit_window = False)
             scale,             # float (has effect if fit_window = False)
             draw_axes,         # boolean
             custom_params,     # [float]
             display_messages,  # boolean
             extra,             # dict (extra tweaking parameters)
             do_drawing = True
             )


#||<>||=======================================================================
#||<>||=======================================================================
#||<>||                         Registrations
#||<>||=======================================================================
#||<>||=======================================================================

versionnumber = "2.2"

procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2021"
image_types = "*"

menupath = "<Image>/Filters/Render/Parametric curves"

####  Parametric curves in cartesian form  ####

procedure_name  = "parametric_cartesian_curve"
procedure_blurb = ("Approximate a parametric curve,"
                   +"\ngiven in cartesian form x=x(t), y=y(t), by a Bezier curve."
                   "\n(Version "+versionnumber+")"
                   )
procedure_label = "Parametric curve (cartesian)"
procedure_help  = ("Draw a parametric curve approximately as a Bezier curve."
                  +"\nThe plugin does its best to find special points on the curve,"
                  +"\nsuch as cusps or inflection points."
                  +"\nAt these points the plugin will place anchors."
                  +"\nThe user can force some custom points to be included"
                  +"\namong those by giving them as custom parameter values."
                  )
procedure_function = cartesian_curve2path_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
        (PF_IMAGE, "image", "Input image", None),
        (PF_STRING, "curve_name", "Name for the curve", "Half circle"),
        (PF_STRING, "x_function", "Function x(t) -- in Python syntax", "cos(t)"),
        (PF_STRING, "y_function", "Function y(t) -- in Python syntax", "sin(t)"),
        (PF_STRING, "start_as_string", "Start value for t -- in Python syntax", "0."),
        (PF_STRING, "end_as_string", "End value for t -- in Python syntax", "pi"),
        (PF_BOOL, "closed", "Is the curve closed?", False),
        #
        (PF_BOOL, "fit_window", "Fit the plot to fill the window?", True),
        (PF_FLOAT, "padding", "Padding (if fit plot in window)", 0.),
        (PF_FLOAT, "origo_x", "x of the origo for the plot (ignored if fit plot in window)", 500.),
        (PF_FLOAT, "origo_y", "y of the origo for the plot (ignored if fit plot in window)", 500.),
        (PF_FLOAT, "scale",   "Scaling factor for the plot (ignored if fit plot in window)", 100.),
        (PF_BOOL, "draw_axes", "Draw the coordinate axes?", False),
        #
        (PF_STRING, "param_string",
                    ("Optional: custom values for the parameter t"
                     +"\nto force certain anchors on the curve"
                     +"\n -- a Python list or values separated by commas"), ""),
        (PF_BOOL, "display_messages", "Display messages in the error console?", False),
    ],
    [
        (PF_VECTORS, "paramcurve_bezier", "Approximate Bezier curve"),
    ],
    procedure_function,
    menu=menupath)


####  Parametric curves in polar form  ####

procedure_name  = "parametric_polar_curve"
procedure_blurb = ("Approximate a parametric curve,"
                   +"\ngiven in polar form r=r(t), by a Bezier curve."
                   "\n(Version "+versionnumber+")"
                   )
procedure_label = "Parametric curve (polar)"
procedure_help  = ("Draw a parametric curve, given in polar form,"
                  +"\napproximately as a Bezier curve."
                  +"\nThe plugin does its best to find special points on the curve,"
                  +"\nsuch as cusps or inflection points."
                  +"\nAt these points the plugin will place anchors."
                  +"\nThe user can force some custom points to be included"
                  +"\namong those by giving them as custom parameter values."
                  )


procedure_function = polar_curve2path_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
        (PF_IMAGE, "image", "Input image", None),
        (PF_STRING, "curve_name", "Name for the curve", "Logarithmic spiral"),
        (PF_STRING, "r_function", "Function r(t) -- in Python syntax", "exp(t/10)"),
        (PF_STRING, "start_as_string", "Start value for t -- in Python syntax", "0."),
        (PF_STRING, "end_as_string", "End value for t -- in Python syntax", "4*pi"),
        (PF_BOOL, "closed", "Is the curve closed?", False),
        #
        (PF_BOOL, "fit_window", "Fit the plot to fill the window?", True),
        (PF_FLOAT, "padding", "Padding (if fit in window)", 0.),
        (PF_FLOAT, "origo_x", "x of the origo for the plot (ignored if fit plot in window)", 500.),
        (PF_FLOAT, "origo_y", "y of the origo for the plot (ignored if fit plot in window)", 500.),
        (PF_FLOAT, "scale",   "Scaling factor for the plot (ignored if fit plot in window)", 100.),
        (PF_BOOL, "draw_axes", "Draw the coordinate axes?", False),
        #
        (PF_STRING, "param_string",
                    ("Optional: custom values for the parameter t"
                     +"\nto force certain anchors on the curve"
                     +"\n -- a Python list or values separated by commas"), ""),
        (PF_BOOL, "display_messages", "Display messages in the error console?", False),
    ],
    [
        (PF_VECTORS, "paramcurve_bezier", "Approximate Bezier curve"),
    ],
    procedure_function,
    menu=menupath)

####  Parametric curves in cartesian form, the function read from a file ####

procedure_name  = "parametric_curve_with_function_from_file"
procedure_blurb = ("Approximate a parametric curve by a Bezier curve."
                   +"\nThe function for the curve is read from an input Python file."
                   +"\nFor what the file should contain,"
                   +"\nread doc.pdf (should have been zipped with this plugin file),"
                   +"\nor see Help > Plug-in Browser (type 'parametric' and choose)."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_label = "Parametric curve (read function from file)"
procedure_help  = ("Draw a parametric curve approximately as a Bezier curve."
                  +"\nThe defining function of the curve is read from a Python file"
                  +"\nwhich can contain optionally some other data too."
                  +"\nMandatory in the input file: the function (float->complex)"
                  +"\nin Python syntax:"
                  +"\n      def function(t):"
                  +"\n            ... ... ..."
                  +"\n            return complex(x,y)"
                  +"\nOptionally, in the same file:"
                  +"\n      curve_name = 'name';"
                  +"\n      interval = [t0,t1] (the interval for t);"
                  +"\n      closed = boolean;"
                  +"\n      custom_params = ... a Python list ..."
                  +"\nThe plugin does its best to find special points on the curve,"
                  +"\nsuch as cusps or inflection points or others."
                  +"\nAt these points the plugin will place anchors."
                  +"\nThe user can force some custom points to be included"
                  +"\namong those by giving them as custom parameter values,"
                  +"\neither in the GUI, or in the input file, or both."
                  )
procedure_function = cartesian_curve_file2path_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
        (PF_IMAGE, "image", "Input image", None),
        (PF_STRING, "curve_name", "Name for the curve", "Curve"),
        (PF_FILE, "input_file", "Choose the input file containing 'function' and possibly others:", None),
        (PF_STRING, "start_as_string", "Start value for t -- in Python syntax", "0."),
        (PF_STRING, "end_as_string", "End value for t -- in Python syntax", "pi"),
        (PF_BOOL, "closed", "Is the curve closed?", False),
        #
        (PF_BOOL, "fit_window", "Fit the plot to fill the window?", True),
        (PF_FLOAT, "padding", "Padding (if fit in window)", 0.),
        (PF_FLOAT, "origo_x", "x of the origo for the plot (ignored if fit plot in window)", 500.),
        (PF_FLOAT, "origo_y", "y of the origo for the plot (ignored if fit plot in window)", 500.),
        (PF_FLOAT, "scale",   "Scaling factor for the plot (ignored if fit plot in window)", 100.),
        (PF_BOOL, "draw_axes", "Draw the coordinate axes?", False),
        #
        (PF_STRING, "param_string",
                    ("Optional: custom values for the parameter t"
                     +"\nto force certain anchors on the curve"
                     +"\n -- a Python list or values separated by commas"), ""),
        (PF_BOOL, "display_messages", "Display messages in the error console?", False),
    ],
    [
        (PF_VECTORS, "paramcurve_bezier", "Approximate Bezier curve"),
    ],
    procedure_function,
    menu=menupath)


main()

