#!/usr/bin/env python
#
# History:
# v0.1: 2021-27-09: Plugin "Chop path at its anchors"
# v0.2: 2021-06-10: Plugin "Chop path by another path"
# v0.3: 2021-19-10: Plugin "Chop path by lengths"

# (c) Markku Koppinen 2021
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'

from __future__ import division, print_function
from gimpfu import *
from copy import deepcopy
from functools import partial 
from itertools import product
from math import *

#==============================================================
#             class BCurve       
#==============================================================

# Note: The plane is treated as the complex plane.

class BCurve(object):
    """Data structures to handle Gimp vectors objects by means of individual
    Bezier arcs.
    By a "Bezier arc" we mean here a Bezier curve determined by 4 control points
    p0,p1,p2,p3.
    This class contains data structures needed to do computations on
    a Bezier curve which are done arc by arc.
    To this end, Gimp data structures
    - gimp.Vectors
    - gimp.VectorsBezierStroke
    are here represented as data structures (classes)
    - GimpVectors
    - GimpStroke
    - BezierCurve
    - BezierArc
    - ControlPoint = complex
    with the aim that
    - each control point [x,y] is represented as an instance of ControlPoint;
    - each Bezier arc (four control points, see above) is represented as an instance
      of BezierArc.
    """
    
    ControlPoint = complex
    
    class GimpVectors(object):
        """Essentially same data as in a gimp.Vectors object except that
        strokes are instances of GimpStroke rather than
        gimp.VectorsBezierStroke.
        Attributes:
        - stroke_list: [GimpStroke]
        - name:        string
        """
        def __init__(self,
                     stroke_list,            # [GimpStroke]
                     name='GimpVectors'      # string
                     ):
            self.stroke_list = stroke_list
            self.name = name
        def __str__(self):
            s = 'GimpVectors '+self.name
            count = 0
            for stroke in self.stroke_list:
                s += '\n  Stroke '+str(count)+': GimpStroke'
                s += '\n    '+str(stroke)
                count += 1
            return s
        def gv2vectors_object(self, image, name=None):
            """Conversion GimpVectors -> gimp.Vectors
            (The converse: vectors_object2gv)
            """
            def cp_list2xy_list(cp_list):
                xy = []
                for cp in cp_list:
                    xy += [cp.real, cp.imag]
                return xy
            if name is None:
                name = self.name
            vectors = pdb.gimp_vectors_new(image, name)
            for stroke in self.stroke_list:
                closed = stroke.closed
                xy_list = cp_list2xy_list(stroke.cp_list)
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                             vectors, 0, len(xy_list), xy_list, closed)
            return vectors
    
    class GimpStroke(object):
        """Essentially same data as in a gimp.VectorsBezierStroke.points,
        except that control points list [x,y,x,y,...] is arranged as
        a list [ControlPoint] by joining successive x,y.
        Attributes:
        - cp_list:     [ControlPoint]
        - closed:      boolean
        - stroke_name: string
        """
        def __init__(self,
                     cp_list,                   # [ControlPoint]
                     closed,                    # boolean
                     stroke_name = 'GimpStroke' # string
                     ):
            self.stroke_name = stroke_name
            self.cp_list = cp_list
            self.closed = closed
        def __str__(self):
            s = self.stroke_name + '; Control points:'
            for cp in self.cp_list:
                s += '\n    '+str(cp)
            s += '\n    closed: '+str(self.closed)
            return s
        def gs2bc(self):
            """Conversion GimpStroke -> BezierCurve
            """
            head = self.cp_list[0]
            tail = self.cp_list[-1]
            inner = self.cp_list[1:-1]
            if len(inner) > 1: # Must be divisible by 4
                ba_list = []
                count = 0
                for i in range(0,len(inner)-1,3):
                    ba_list.append(BCurve.BezierArc(inner[i:i+4], 'arc '+str(count)))
                    count += 1
            elif len(inner) == 1: # Stroke has only one anchor
                ba_list = [BCurve.BezierArc([inner[0]], 'arc 0')]
            else:
                raise Exception("BCurve.GimpStroke.gs2bc: No anchors in stroke?")
            return BCurve.BezierCurve(
                               bezier_arcs = ba_list,
                               head_handle = head,
                               tail_handle = tail,
                               closed = self.closed,
                               curve_name = self.stroke_name
                               )
    
    class BezierCurve(object):
        """BezierCurve is a list of butting Bezier arcs with some extra data.
        Attributes:
        - curve_name:  string
        - bezier_arcs: [BezierArc];
        - head_handle: ControlPoint;
        - tail_handle: ControlPoint;
        - closed:      boolean.
        Note: "Butting" means that for any pair of successive arcs, the last
              control point of the former equals the first control point of
              the latter.
              In initialization no checks are done about this condition.
        """
        def __init__(self,
                     bezier_arcs=[],           # [BezierArc]
                     head_handle=None,         # ControlPoint
                     tail_handle=None,         # ControlPoint
                     closed=False,             # boolean
                     curve_name='BezierCurve', # string
                     ):
            self.curve_name = curve_name
            self.bezier_arcs = bezier_arcs
            if head_handle is None:
                self.head_handle = bezier_arcs[0].cp4[0]
            else:
                self.head_handle = head_handle
            if tail_handle is None:
                self.tail_handle = bezier_arcs[-1].cp4[-1]
            else:
                self.tail_handle = tail_handle
            self.closed = closed
        def __str__(self):
            s = self.curve_name + ': Bezier curve' + '; Bezier arcs:'
            count = 0
            for arc in self.bezier_arcs:
                s += '\n  arc '+ str(count)+ ': ' +str(arc)
                count += 1
            s += '\n  Head handle:    '+str(self.head_handle)
            s += '\n  Tail handle:    '+str(self.tail_handle)
            s += '\n  Closed:'    +str(self.closed)
            return s
        def bc2gs(self):
            """Conversion BezierCurve -> GimpStroke
            """
            cp_list = [self.head_handle]
            if len(self.bezier_arcs[0].cp4) > 1:
                for ba in self.bezier_arcs:
                    cp_list += ba.cp4[:3]
                cp_list.append(self.bezier_arcs[-1].cp4[-1])
            else:
                # Only one BezierArc and it has cp4 = [p0]
                # (may rise from Gimp stroke with only one anchor).
                cp_list.append(self.bezier_arcs[0].cp4[0])
            cp_list.append(self.tail_handle)
            return BCurve.GimpStroke(cp_list, self.closed, self.curve_name)
        def bc2vectors_object(self, image, name=None):
            """Conversion BezierCurve -> gimp.Vectors
            Makes vectors_object with 1 stroke.
            """
            if name is None:
                name = self.curve_name
            gs = self.bc2gs()             # BCurve.GimpStroke
            gv = BCurve.GimpVectors([gs]) # BCurve.GimpVectors
            return gv.gv2vectors_object(image, name)
    
    class BezierArc(object):
        """Data structure for one Bezier arc: arc determined by 4 control
           points.
        Attributes:
        - cp4:      [p0,p1,p2,p3] where each pi:ControlPoint, or
                    [p0] if only one control point (may rise from
                    a stroke with only one anchor);
        - arc_name: string
        """
        def __init__(self,
                     cp4,                 # cp4=[p0,p1,p2,p3] with pi:ControlPoint
                                          # or [p0]
                     arc_name='BezierArc' # string
                     ):
            self.cp4 = cp4
            self.arc_name = arc_name
        def reverse(self):
            self.cp4.reverse()
        def __str__(self):
            s = 'BezierArc, control points:'
            for cp in self.cp4:
                s += '\n  '+ str(cp)
            return s

# -----------------
# vectors_object2gv
# -----------------
# Conversion gimp.Vectors -> GimpVectors
# Args:
# - vectors_object: gimp.Vectors (Gimp vectors object)
# Returns:
# - GimpVectors
#
# Note: The inverse conversion is GimpVectors.gv2vectors_object(...).
#       See also BezierCurve.bc2vectors_object(...).
def vectors_object2gv(vectors_object):
    def xy_list2cp_list(xy_list): # xy_list = [x,y,x,y,...,]
        cp = []
        for i in range(0,len(xy_list),2):
            cp.append(BCurve.ControlPoint(xy_list[i], xy_list[i+1]))
        return cp
    stroke_list = []
    count = 0
    for stroke in vectors_object.strokes:
        xy_list = stroke.points[0]
        closed = stroke.points[1]
        cp_list = xy_list2cp_list(xy_list)
        stroke_name = vectors_object.name + ' stroke '+str(count)
        stroke_list.append(BCurve.GimpStroke(cp_list, closed, stroke_name))
        count += 1
    return BCurve.GimpVectors(stroke_list, vectors_object.name)

# ---------------------
# completed_bezier_arcs
# ---------------------
# Give bc:BCurve.BezierCurve, get the list of bezier arcs where, in the case of
# closed bc, a gap-closing arc is added at the end, with necessary modification
# for the case of closed 1-anchor stroke.
# Only copies of the original arcs are returned.
# Args:
# - bc:BCurve.BezierCurve
# Returns:
# - [BCurve.BezierArc]
def completed_bezier_arcs(bc):
    bas = deepcopy(bc.bezier_arcs)
    if not bc.closed:
        return bas
    gap_ba=BCurve.BezierArc(cp4=[bas[-1].cp4[-1],
                            bc.tail_handle,
                            bc.head_handle,
                            bas[0].cp4[0]])
    if len(bas[0].cp4) == 1: # 1-anchor stroke
        return [gap_ba]
    else:
        return bas + [gap_ba]

# -------------------
# reversed_GimpStroke
# -------------------
# Reverse GimpStroke.
# Note: In Gimp, for a closed stroke the question of
# starting and ending anchors is problematic.
# Args:
# - gs: BCurve.GimpStroke
# Returns:
# - BCurve.GimpStroke
def reversed_GimpStroke(gs):
    return BCurve.GimpStroke(cp_list = gs.cp_list[::-1],
                             closed = gs.closed,
                             stroke_name = gs.stroke_name+' reversed')


#==============================================================
#                     Drawing
#==============================================================

# ------------------------
# gimp_draw_vectors_object
# ------------------------
# Args:
# - image: Gimp's image (obtained usually from GUI);
# - vectors_object: a vectors object in Gimp;
# - visible: Boolean.
# Returns the vectors object in Gimp, and makes it visible if visible=True.
def gimp_draw_vectors_object(image, vectors_object, visible=True):
    # Insert a Gimp vectors object in Gimp.
    # Adapted from http://registry.gimp.org/files/lqr_wpset-0.14.py.txt
    # by Mike Kazantsev.
    def gimp_insert_vectors(image, vectors_object, position=0):
        if gimp.version >= (2, 8, 0):
            # 2.7.0 has gimp-image-insert-vectors, but fails with
            # parent=None
            parent = None
            return pdb.gimp_image_insert_vectors(
                     image, vectors_object, parent, position)
        else:
            return pdb.gimp_image_add_vectors(
                     image, vectors_object, position)# Deprecated!
    gimp_insert_vectors(image, vectors_object)
    vectors_object.visible = visible
    return vectors_object

#======================================================
#                 Zeroes of a polynomial
#======================================================

# A function to find zeroes of a polynomial, and as by-products the
# x-values of the extrema and of the inflection points.
# At the final phase the zeroes are found by binary search from monotonous
# segments.
# 
# Written just for fun. Other algorithms exist in abundance.
#
# In addition, as a simple wrap-up, a function to do the same job when the
# polynomial is of degree 3 and given in the Bernstein basis.

# --------------------
# zeroes_of_polynomial
# --------------------
# Given a real polynomial, find the real zeroes, the x values of extrema,
# and the x values of the inflection points. May restrict to an interval.
#
# Args:
# - coefficients: list [c0,c1,...,cn] of the coefficients of the
#                     polynomial c0 + c1*x + ... + cn*(x**n);
#                     the ci are integer or float.
# - interval:     None or [float,float]
# Returns:
# 1. None in the case of a zero polynomial.
# 2. None if interval is given (not None) with interval=[a,b] where a>b.
# 3. If interval=[a,a] is given, if a is a root then [a],[],[] is returned,
#    otherwise [],[],[]. So, no effort is made to give any extrema or inflection
#    points.
# 3. Otherwise:
#    - the list of real zeroes of the polynomial, in increasing order
#      (possibly []) with multiple roots repeated;
#    - the extrema (the x values);
#    - the inflection points (the x values).
# 4. Because of the chosen accuracy and tolerance in the code, choosing different
#    interval may change the returned values slightly.
def zeroes_of_polynomial(coefficients, interval=None):
    from math import sqrt
    from functools import partial
    def f(x, coeff): # Horner
        s = 0.
        for ci in coeff[::-1]:
            s = ci + x*s
        return s
    if not (interval is None):
        LO,HI = interval
        if LO > HI:
            return None
        if LO == HI:
            if f(LO) == 0:
                return [LO],[],[]
            else:
                return [],[],[]
    coefficients = [float(ci) for ci in coefficients]
    # Find largest n with coefficients[n] non-zero and
    # discard all higher terms:
    n = -1
    for i in range(len(coefficients)):
        if coefficients[i] != 0.:
            n = i
    if n < 0: # Polynomial is identically zero.
        return None
    c = coefficients[:n+1] 
    # Make c[n] positive:
    if c[n] < 0:
        c = [-ci for ci in c]
    if n <= 2:
        if n == 0: # Polynomial is a non-zero constant.
            results = [[],[],[]]
        if n == 1: # Polynomial is c[0] + c[1]*x
            results = [[-c[0] / c[1]], [], []]
        if n == 2: # Polynomial is c[0] + c[1]*x + c[2]*(x**2).
            discr = c[1]**2 - 4*c[0]*c[2]
            try:
                root = sqrt(discr)
                x1 = (-c[1] - root) / (2*c[2])
                x2 = (-c[1] + root) / (2*c[2])
                #return [x1,x2], [-c[1]/(2*c[2])], []
                results = [[x1,x2], [-c[1]/(2*c[2])], []]
            except ValueError:
                #return [], [-c[1]/(2*c[2])], []
                results = [[], [-c[1]/(2*c[2])], []]
        if interval is None:
            return results
        else:
            results0 = [t for t in results[0] if LO <= t <= HI]
            results1 = [t for t in results[1] if LO <= t <= HI]
            results2 = [t for t in results[2] if LO <= t <= HI]
            return results0, results1, results2
    # Build subdivision such that in each subinterval
    # the polynomial is monotonous:
    derivative = [e*i for e,i in enumerate(c)][1:]
    extrema, inflection, _ = zeroes_of_polynomial(derivative, interval)
    if interval is None:
        x_far = sum([abs(ci/c[-1]) for ci in c])
        subdiv = [-x_far] + extrema + [x_far]
    else:
        subdiv = [LO] + [t for t in extrema if LO < t < HI] + [HI]
    # Solve and collect the zeroes of each monotonous segment:
    fc = partial(f, coeff=c)
    zeroes = []
    for i in range(len(subdiv)-1):
        lo, hi = subdiv[i], subdiv[i+1]
        x = zero_of_monotonous_function(fc, lo, hi,
                                    accuracy=1e-10, tolerance=1e-10)
        if not(x is None):
            zeroes.append(x)
    return zeroes, extrema, inflection

# ---------------------------
# zero_of_monotonous_function
# ---------------------------
# Find the zero, if any, of a monotonous function f(x)
# in the interval [lo,hi]. If none is found, None is returned.
# Args:
# - f, lo, hi: the monotonous function with start and end value;
# - accuracy:  accuracy for x.
# - tolerance: float>=0; if no zeroes exist in the interval, then
#              if f(lo) or f(hi) is less than 'tolerance' from 0,
#              then lo or hi is returned, respectively.
# Returns: the zero or None.
def zero_of_monotonous_function(f, lo, hi, accuracy=1e-10, tolerance=1e-10):
    MAX_ROUNDS = 50
    lo, hi = float(lo), float(hi)
    flo, fhi = float(f(lo)), float(f(hi))
    if (flo > 0.) == (fhi > 0.):
        if abs(flo) <= tolerance:
            return lo
        elif abs(fhi) <= tolerance:
            return hi
        else:
            return None
    if flo == 0.:
        return lo
    if fhi == 0.:
        return hi
    count = 0
    while hi-lo > accuracy:
        mid = (lo+hi)/2.
        fmid = float(f(mid))
        if fmid == 0.:
            return mid
        if (flo > 0.) == (fmid > 0.):
            lo, flo = mid, fmid
        else:
            hi, fhi = mid, fmid
        count += 1
        if count > MAX_ROUNDS:
            break
    x = (fhi*lo - flo*hi) / (fhi-flo)
    return x

# -------------------------
# solve_bernstein2_equation
# -------------------------
# Find real zeroes of the equation c0*b0(t) + c1*b1(t) + c2*b2(t) = 0 where
# b0,b1,b2 are the Bernstein polynomials of degree 2 and
# c0,c1,c2 are real coefficients.
# Return the zeroes as a list (0, 1, or 2 items).
# If the function is constant 0, return None.
# Args:
# - c0,c1,c2: float
# Returns:
# - None or list of floats (possibly empty)
def solve_bernstein2_equation(c0,c1,c2):
    from math import sqrt
    ZERO = 1e-14
    if abs(c0 - 2.*c1 + c2) < ZERO: # equation of degree <= 1
        try:
            return [c0 / (2.*(c0 - c1))]
        except ZeroDivisionError: # equation of degree 0
            if abs(c0) < ZERO: # constant 0
                return None
            else: # constant not 0
                return []
    try:
        root = sqrt(c1**2 - c0*c2)
    except ValueError: # No real roots
        return []
    return [(c0 - c1 + root) / (c0 - 2.*c1 + c2),
            (c0 - c1 - root) / (c0 - 2.*c1 + c2)]


#==============================================================
#                 Plane vectors (=complex numbers)
#==============================================================

# v dot (w perpendicular)
def cross(v,w):
    return (v*w.conjugate()).imag

def vdot(v,w):
    return v.real*w.real + v.imag*w.imag

def collinear(points): # points: [complex] (The plane is the complex plane.)
    ZERO = 1e-8
    n = len(points)
    if n < 3:
        return True
    for i in range(n):
        for j in range(i+1,n):
            for k in range(j+1,n):
                a,b,c = points[i], points[j], points[k]
                ab = -a+b
                ac = -a+c
                bc = -b+c
                if abs(cross(ab,ac)) > ZERO:
                    return False
                if abs(cross(ab,bc)) > ZERO:
                    return False
                if abs(cross(ac,bc)) > ZERO:
                    return False
    return True


#==============================================================
#                       Bezier routines
#==============================================================

# ---------
# bezier_rv
# ---------
# The usual Bezier curve from control points.
# Version where all points are complex numbers
# (the plane is the complex number plane).
# Args:
# - t:  float
# - cp4:[complex,complex,complex,complex]
# Returns:
# - complex
def bezier_rv(t, cp4):
    P0,P1,P2,P3 = cp4
    if t > 0.5:
        u = (1-t)/t
        t3 = t**3
        return t3*(u*(u*(u*P0+3*P1)+3*P2)+P3) # complex
    else:
        u = t/(1-t)
        t3 = (1-t)**3
        return t3*(u*(u*(u*P3+3*P2)+3*P1)+P0) # complex

# Derivative: Returns complex.
def bezier_dot_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    P01 = -P0+P1
    P12 = -P1+P2
    P23 = -P2+P3
    if t > 0.5:
        u = (1-t)/t
        t2 = t**2
        return 3*t2*(u*(u*P01+2*P12)+P23) # complex
    else:
        u = t/(1-t)
        t2 = (1-t)**2
        return 3*t2*(u*(u*P23+2*P12)+P01) # complex

# -------------------------
# bezier_new_control_points
# -------------------------
# Given a Bezier arc B(t) and two distinct parameter values a and b,
# find a new parametrization C(t) of the same curve such that
# B(a) = C(0) and B(b) = C(1).
# Args:
# - a:  float;
# - b:  float;
# - cp4:[complex,complex,complex,complex] (control points [p0,p1,p2,p3]).
# Returns:
# - [complex,complex,complex,complex] (new control points [q0,q1,q2,q3])
# Notes:
# - There is an exception: If a=b, the returned control points
#   are equal, so the new curve is a single point, not the original
#   curve. The function gives no signal about such occurence.
def bezier_new_control_points(a, b, cp4):
    def bezier_2_rv(t, r0, r1, r2): # Bezier curve of degree two (complex)
        b20 = (1-t)**2
        b21 = 2*(1-t)*t
        b22 = t**2
        return b20*r0 + b21*r1 + b22*r2
    p0,p1,p2,p3 = cp4
    q0 = bezier_rv(a, cp4)
    q1 = (1-b)*bezier_2_rv(a, p0,p1,p2) + b*bezier_2_rv(a, p1,p2,p3)
    q2 = (1-a)*bezier_2_rv(b, p0,p1,p2) + a*bezier_2_rv(b, p1,p2,p3)
    q3 = bezier_rv(b, cp4)
    return [q0,q1,q2,q3]

# ------------------------------
# bezier_subdivide_at_parameters
# ------------------------------
# Given a Bezier arc B(t) and a list of parameter values t1,t2,...,
# find control points for all subarcs when B(t), 0<=t<=1, is subdivided
# at parameter values t1,t2,...
# The input Bezier arc is given as its 4 control points [p0,p1,p2,p3],
# and the output is a list of new sets of 4 control points [[q0,q1,q2,q3],...].
# The plane is viewed as the complex number plane.
# Args:
# - parameter_list: [float]                           ([t1,t2,...])
# - control_points: [complex,complex,complex,complex] ([p0,p1,p2,p3])
# Returns:
# - [[complex,complex,complex,complex]]               ([[q0,q1,q2,q3],...])
def bezier_subdivide_at_parameters(parameter_list, control_points):
    ts01 = [t for t in parameter_list if 0 < t < 1] # Drop t's outside (0,1)
    ts01.sort()
    try:
        ts = [ts01[0]]
    except IndexError: # Empty parameter list
        return [[p for p in control_points]] # copy
    for t in ts01[1:]: # Drop duplicates.
        if t != ts[-1]:
            ts.append(t)
    ts = [0]+ts+[1] # Add t=0,1
    # Collect pairs [t(i),t(i+1)]
    abs = []
    for i in range(1,len(ts)):
        abs.append([ts[i-1],ts[i]])
    arc_cps = []
    for a,b in abs:
        arc_cps.append(bezier_new_control_points(a, b, control_points))
    return arc_cps

# --------------------------------
# bezier_critical_point_candidates
# --------------------------------
# Candidates for closest or farthest points:
# Solutions to (B(t)-v).B'(t) = 0.
# Args:
# - cp4: [complex,complex,complex,complex] (control points)
# - v:   complex
# Returns:
# - [[float,complex]] (=[[t,B(t)]])
def bezier_critical_point_candidates(cp4, v, interval=None):
    p0,p1,p2,p3 = cp4
    # Control points w0,w1,w2,w3 when the basis {1, t, t^2/2, t^3/6}
    # is used instead of the Bernstein polynomials:
    w0 = p0
    w1 = 3*(-p0 + p1)
    w2 = 6*(p0 - 2*p1 + p2)
    w3 = 6*(-p0 + 3*p1 - 3*p2 + p3)
    # Dot products:
    w0w1 = w0.real * w1.real + w0.imag * w1.imag
    w0w2 = w0.real * w2.real + w0.imag * w2.imag
    w0w3 = w0.real * w3.real + w0.imag * w3.imag
    w1w1 = w1.real * w1.real + w1.imag * w1.imag
    w1w2 = w1.real * w2.real + w1.imag * w2.imag
    w1w3 = w1.real * w3.real + w1.imag * w3.imag
    w2w2 = w2.real * w2.real + w2.imag * w2.imag
    w2w3 = w2.real * w3.real + w2.imag * w3.imag
    w3w3 = w3.real * w3.real + w3.imag * w3.imag
    #
    vw1 = v.real * w1.real + v.imag * w1.imag
    vw2 = v.real * w2.real + v.imag * w2.imag
    vw3 = v.real * w3.real + v.imag * w3.imag
    # Polynomial coefficients:
    c0 = w0w1 - vw1
    c1 = w0w2 + w1w1 - vw2
    c2 = w0w3/2 + 3*w1w2/2 - vw3/2
    c3 = 2*w1w3/3 + w2w2/2
    c4 = 5*w2w3/12
    c5 = w3w3/12
    # Solve polynomial zeroes:
    zeroes = zeroes_of_polynomial([c0,c1,c2,c3,c4,c5], interval)
    if zeroes is None:
        return []
    ts = zeroes[0]
    # Compute B(t)'s:
    t_Bs = [[t, w0 + t*w1 + t*t*w2/2 + t*t*t*w3/6] for t in ts]
    return t_Bs

# ------------------------
# bezier_tangential_points
# ------------------------
# Given an infinite Bezier curve B(t) (4 control points), find the points on
# the curve where B'(t) is parallel with the given direction.
# Notes:
# 1. Cusps are included (B'(t)=0).
# 2. The plane is viewed as the complex number plane.
# 3. If the curve is a straight line parallel to the direction None is returned.
# Args:
# - cp4: [complex,complex,complex,complex]
# - direction: complex
# Returns: None or
# - [[float,complex]] (=[[t,B(t)]])
def bezier_tangential_points(cp4, direction):
    p0,p1,p2,p3 = cp4
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    c0 = -p01.real * direction.imag + p01.imag * direction.real
    c1 = -p12.real * direction.imag + p12.imag * direction.real
    c2 = -p23.real * direction.imag + p23.imag * direction.real
    ts = solve_bernstein2_equation(c0,c1,c2)
    if ts is None:
        return None
    return [[t,bezier_rv(t,cp4)] for t in ts]

# ---------------------
# bezier_special_points
# ---------------------
# Find the special points of a cubic Bezier curve:
# inflection points, cusp points, self-intersection points.
#
# Args:
# - control_points: [p0,p1,p2,p3]
#
# Returns: (case, param_list), which is one of:
# (-1, [t1,t2]) when there is a loop or a cusp point, with
#               t1,t2 the parameter values with B(t1)=B(t2)
#               (ideally, t1=t2 in the case of a cusp);
# (0, [])       when the curve is degenerate: a parabola or
#               contained on a straight line;
# (1, [t])      when there is a single inflection point, with
#               t the parameter of the inflection point.
# (2, [t1,t2])  when there are two inflection points, with
#               t1,t2 the parameters for the inflection points.
#
# Note: The algorithm works as if infinite computation precision were
#   possible. The caller had better check the following:
# - In the case of a loop or of two inflection points (cases -1 or 2):
#   If the returned parameter values t1,t2 are very close, it may be
#   a sign of a cusp point, instead. (Or maybe the caller should take
#   it as a cusp point.)
# - In the case of inflection points (cases 1 or 2):
#   If one or both of the returned parameter values has very large
#   absolute value, it may mean that the inflection point is at infinity
#   (so, non-existent).
def bezier_special_points(control_points):
    from math import sqrt
    p0,p1,p2,p3 = control_points
    w1 = -p0 - p1 + p2 + p3
    w2 =  p0 - p1 - p2 + p3
    w3 = -p0 + 3*p1 - 3*p2 + p3
    C1 = 18.0 * (-w2.real * w3.imag + w2.imag * w3.real)
    C2 = 4.50 * (-w3.real * w1.imag + w3.imag * w1.real)
    C3 = 2.25 * (-w1.real * w2.imag + w1.imag * w2.real)
    try:
        X = C2/C1
        Y = C3/C1
        Z = -X**2 + 2*Y
        try:
            root = sqrt(3*Z)
            t1 = .5 + X + root
            t2 = .5 + X - root
            return -1, [t1,t2] # loop or cusp
        except ValueError: # Z < 0
            root = sqrt(-Z)
            t1 = .5 + X + root
            t2 = .5 + X - root
            return 2, [t1,t2] # two inflection points
    except ZeroDivisionError: # C1 = 0
        try:
            t = .5 + C3/C2
            return 1, [t] # one inflection point
        except ZeroDivisionError: # C1 = C2 = 0
            return 0,[] # parabola or everything on straight line.

# --------------------
# bezier_equals_bezier
# --------------------
# Given two infinite Bezier curves B(t) and C(u) (t,u in R), are they
# the same curve? More precisely, are there A,B in R with B!=0 such that
# B(A+Bu) = C(u) identically? (Written otherwise: B(t)=C(u) when t = A+Bu.)
# The Bezier curves are given as 4 control points each (control points as
# complex numbers since the plane is viewed as the complex number plane).
# Optional argument: typical_scale. If the Bezier curves' coordinates are typically
# around 1000, input  typical_scale=1000.
# Return None if the answer is 'No', otherwise [A,B].
# Args:
# - control_points_1: [complex,complex,complex,complex]
# - control_points_2: [complex,complex,complex,complex]
# - typical_scale:    None or float
# Returns:
# - None, or [float,float].
# Note: Not very well tested.
def bezier_equals_bezier(control_points_1, control_points_2, typical_scale=None):
    if typical_scale is None:
        ave1 = sum(abs(z) for z in control_points_1) / len(control_points_1)
        ave2 = sum(abs(z) for z in control_points_1) / len(control_points_2)
        typical_scale = 2*max(ave1,ave2)
    ZERO = 1e-8*typical_scale
    TOLERANCE = 1e-4*typical_scale # How large error in points is allowed
    try:
        p0,p1,p2,p3 = control_points_1
        q0,q1,q2,q3 = control_points_2
    except ValueError: # A 1-anchor stroke
        if len(control_points_1) != len(control_points_2):
            return None
        p0, q0 = control_points_1[0], control_points_2[0]
        if abs(q0-p0) < ZERO:
            return [0,1]
        else:
            return None
    # Expand B(t) and C(u) in the power basis:
    # B(t) = b0 + b1*t + b2*t*t + b3*t*t*t
    # C(u) = c0 + c1*u + c2*u*u + c3*u*u*u
    coeff1 = [p0, 3*(-p0 + p1), 3*(p0 - 2*p1 + p2), -p0 + 3*p1 - 3*p2 + p3]
    coeff2 = [q0, 3*(-q0 + q1), 3*(q0 - 2*q1 + q2), -q0 + 3*q1 - 3*q2 + q3]
    interchange = (abs(coeff2[-1]) > abs(coeff1[-1]))
    if interchange:
        coeff1, coeff2 = coeff2, coeff1 # Now |coeff1[-1]| >= |coeff2[-1]|
    b0,b1,b2,b3 = coeff1
    c0,c1,c2,c3 = coeff2
    if abs(b3) >= ZERO: # b3 != 0
        Q = c3/b3
        if abs(Q.imag) >= ZERO:
            return None # No solution
        if Q.real >= 0:
            B = Q.real**(1./3)
        else:
            B = -((-Q.real)**(1./3))
        if B == 0:
            return None # No solution
        R = c2/(B*B*b3) - b2/b3
        if abs(R.imag) >= ZERO:
            return None # No solution
        A = R.real / 3
        if abs(B*(b1 + 2*b2*A + 3*b3*A*A) - c1) >= TOLERANCE:
            return None # No solution
        if abs(b0 + b1*A + b2*A*A + b3*A*A*A - c0) >= TOLERANCE:
            return None # No solution
        if not interchange:
            return [A,B]
        else:
            return [-A/B, 1/B] # inverse transform
    elif abs(b2) >= ZERO: # b3 = 0, b2 != 0: parabola
        if abs(c3) >= ZERO:
            return None # No solution
        interchange_again = (abs(coeff2[2]) > abs(coeff1[2]))
        if interchange_again:
            coeff1, coeff2 = coeff2, coeff1 # Now |coeff1[2]| >= |coeff2[2]|
            interchange = not interchange
        b0,b1,b2,b3 = coeff1
        c0,c1,c2,c3 = coeff2
        Q = c2/b2
        if abs(Q.imag) >= ZERO:
            return None # No solution
        try:
            B0 = sqrt(Q.real)
            Bs = [B0, -B0]
        except ValueError:
            return None # No solution
        if B0 == 0:
            return None # No solution
        for B_try in Bs:
            R = (c1/B_try - b1)/b2
            if abs(R.imag) < ZERO:
                B = B_try
                A = R.real / 2
                break
        else: # No break: neither B_try good
            return None # No solution
        if abs(b0 + b1*A + b2*A*A - c0) >= TOLERANCE:
            return None # No solution
        if not interchange:
            return [A,B]
        else:
            return [-A/B, 1/B] # inverse transform
    elif abs(b1) >= ZERO: # b3 = b2 = 0, b1 != 0: straight line
        if max(abs(c3),abs(c2)) >= ZERO:
            return None # No solution
        Q = c1/b1
        if abs(Q.imag) >= ZERO:
            return None # No solution
        B = Q.real
        R = (c0 - b0) / b1
        if abs(R.imag) >= ZERO:
            return None # No solution
        A = R.real
        if not interchange:
            return [A,B]
        else:
            return [-A/B, 1/B] # inverse transform
    else: # b3 = b2 = b1 = 0: B(t) is constant
        if max(abs(c3),abs(c2),abs(c1)) >= ZERO:
            return None # No solution
        if abs(b0-c0) >= ZERO: # Each curve constant but different constants
            return None # No solution
        return [0,1] # B(t) = C(u) = constant: pick one solution


#==============================================================
#                   class BezierSubArc
#==============================================================
# Required by "Chop path by another path"

class BezierSubArc(object):
    """Data structure for storing and splitting a subarc of a longer Bezier arc.
    The long Bezier arc is given by its control points (cp4) which is stored;
    the parameter interval for it is [0,1] and is not stored.
    Let B(t) be the function for the Bezier curve.
    The subarc is given by its parameter subinterval and some other data:
    - interval: [t0,t1] where 0 <= t0 <= t1 <= 1
    - end_points: [q0,q3] where q0=B(t0) and q3=B(t1).
    - xmin etc. (see below)
    If the subarc is ascending or descending, the xmin etc. give the box
    with horizontal and vertical sides where the subarc resides.
    The plane is viewed as the complex number plane.
    Attributes:
    - cp4: [complex,complex,complex,complex] (list of control points [p0,lp1,p2,p3])
    - interval
    - end_points
    - xmin (min of x coordinates of q0,q3)
    - xmax (max of x coordinates of q0,q3)
    - ymin (min of y coordinates of q0,q3)
    - ymax (max of y coordinates of q0,q3)
    Methods:
    - split (splits self at an extremal point)
    - __str__
    Init arguments:
    - cp4
    - interval
    """
    def __init__(self, cp4, interval, end_points=None):
        self.cp4 = cp4
        
        self.interval = interval
        if end_points is None:
            q0,q3 = [bezier_rv(interval[0],cp4), bezier_rv(interval[1],cp4)]
        else:
            q0,q3 = end_points
        self.end_points = [q0,q3]
        self.xmin = min(q0.real, q3.real)
        self.xmax = max(q0.real, q3.real)
        self.ymin = min(q0.imag, q3.imag)
        self.ymax = max(q0.imag, q3.imag)
    def __str__(self):
        s = "BezierSubArc:"
        s += "\n  cp4: "+str(self.cp4)
        s += "\n  interval: "+str(self.interval)
        s += "\n  end_points: "+str(self.end_points)
        s += "\n  (xmin,ymin): "+str((self.xmin,self.ymin))
        s += "\n  (xmax,ymax): "+str((self.xmax,self.ymax))
        return s
    def split(self):
        ZERO = 1e-8
        #t,Bt = extreme_for_cp4(self.cp4) # splitting point
        try:
            ext = extreme_for_Bezier_sub_arc(self) # splitting point
        except NoExtremeError:
            t = (self.interval[0] + self.interval[1]) / 2
            Bt = bezier_rv(t, self.cp4)
            ext = [t,Bt]
        if ext is None: #straight line segment:
            t = (self.interval[0] + self.interval[1]) / 2
            Bt = bezier_rv(t, self.cp4)
        else:
            t,Bt = ext
        left = BezierSubArc(cp4 = self.cp4,
                            interval = [self.interval[0], t],
                            end_points = [self.end_points[0], Bt]
                           )
        right = BezierSubArc(cp4 = self.cp4,
                            interval = [t, self.interval[1]],
                            end_points = [Bt, self.end_points[1]]
                           )
        return left,right

class NoExtremeError(Exception):
    def __init__(self,m):
        self.message = m

# --------------------------
# extreme_for_Bezier_sub_arc
# --------------------------
# Given a Bezier arc, find the point B(t) which is the
# most distant from the chord.
# Assume there is only one!
# Return
# - parameter value t
# - point B(t).
# Args:
# - bsa: BezierSubArc
# Returns:
# - float    (=t)
# - complex (=B(t))
# Note: So far, a loop not allowed.
# Note: If the curve is a line segment, None is returned.
def extreme_for_Bezier_sub_arc(bsa):
    cp4 = bsa.cp4
    t0,t1 = bsa.interval
    a,b = bsa.end_points
    direction = -a + b
    loop_case = (direction == 0)
    if loop_case:
        #print("loop_case: a,b = "+str((a,b)))
        #print("and: cp4 = "+str(cp4))
        #print("and: interval = "+str(bsa.interval))
        #print("and: end_points = "+str(bsa.end_points))
        raise Exception("loop_case: ulos") ########  KOE
        t_Bs = bezier_critical_point_candidates(cp4, a, interval=[0,1]) # [[t,B(t)]]
    else:
        t_Bs = bezier_tangential_points(cp4, direction) # [[t,B(t)]]
    if not (t_Bs is None):
        t_Bs1 = [[t,B] for t,B in t_Bs if t0 < t < t1]
        if len(t_Bs1) != 1:
            #print("\n\n>>>> t_Bs ="+str(t_Bs))
            #print(str(bsa))
            raise NoExtremeError("len(t_Bs1) = "+ str(len(t_Bs1)))
        #if t_Bs1[0][0] == 1.0:
        #    print("oho")
        #    print("t_Bs1:")
        #    print(str([t for t,B in t_Bs1]))
        #    print("and: cp4 = "+str(cp4))
        #    print("and: interval = "+str(bsa.interval))
        #    print("and: end_points = "+str(bsa.end_points))
        return t_Bs1[0]
    else:
        #print("extreme_for_Bezier_sub_arc: Returning None")
        return None

#==============================================================
#                     class TotalParameters
#==============================================================
# Required by "Chop path by another path"

class TotalParameter(object):
    """From a point P on a composite Bezier curve, store: index of the arc
    and the parameter value. The special cases of end points of the arc
    are stored in attribute 'end_point' (cases parameter = 0 or 1).
    Attributes:
    - index: float (the index of the arc in the list of the Bezier arcs)
    - parameter: float
    - end_point: string (one of 'start', 'end', 'not end point'
    """
    def __init__(self, index, parameter, end_point='not end point'):
        self.index = index
        self.parameter = parameter
        self.end_point = end_point
    def __str__(self):
        s = 'TotalParameter'
        s += '\n(index, parameter) = '+str((self.index,self.parameter))
        s += '\nend_point = '+str(self.end_point)
        s += '\ntotal parameter = '+str(self.index + self.parameter)
        return s


def str_total_parameter(tp): # short
    return str((tp.index, tp.parameter))

# -----------------------------
# remove_close_total_parameters
# -----------------------------
# Given a composite Bezier curve (as a succession of butting Bezier arcs)
# and a list of total parameters on it:
# - find parameters giving points too close to some arc end point,
#   and when found replace then with exact end points, remocing duplicates;
# - find parameters giving too close (or duplicate) points and when found,
#   replace them with their averages. 
# Assume that tps is sorted according to T.index+T.parameter.
# Args:
# - tps: [TotalParameter]
# - bas: [BCurve.BezierArc]
# Returns:
# - [TotalParameter]
def remove_close_total_parameters(tps, bas):
    ZERO_point = 1e-3 # To checl too close points
    ZERO_param = 1e-3 # To checl too close parameters
    if len(tps) == 0:
        return tps
    # 1: If some T is too close to an end point of an arc, make it
    #    exactly end point.
    red_tps = []
    for T in tps:
        t = T.parameter
        if t < ZERO_param:
            red_tps.append(TotalParameter(index = T.index,
                                          parameter = 0,
                                          end_point = 'start'))
        elif t > 1-ZERO_param:
            if T.index < len(bas)-1:
                red_tps.append(TotalParameter(index = T.index + 1,
                                              parameter = 0,
                                              end_point = 'start'))
            else:
                red_tps.append(TotalParameter(index = T.index,
                                              parameter = 1,
                                              end_point = 'end'))
        else: # (ZERO_param < t < 1-ZERO_param)
            red_tps.append(T)
    tps = red_tps
    # 2: This may have caused duplicates at points 'start' or 'end'. Remove those:
    red_tps = [tps[0]]
    current = red_tps[0]
    for T in tps[1:]:
        if not ((current.index == T.index)
                  and (current.end_point == T.end_point in ('start','end'))):
            red_tps.append(T)
            current = T
    tps = red_tps
    # 3: Remove duplicates or close points:
    reduced_tps = []
    tps_iter = iter(tps)
    stop = False
    T0 = next(tps_iter) # first Total parameter
    while not stop:
        collect = [T0] # Collect T's close to T0
        if T0.end_point not in ('start','end'):
            while True:
                t0 = T0.parameter
                point = bezier_rv(t0, bas[T0.index].cp4)
                try:
                    T = next(tps_iter)
                    t1 = T.parameter
                    point1 = bezier_rv(t1, bas[T.index].cp4)
                except StopIteration:
                    stop = True
                    break # collecting
                if T.end_point in ('start','end'):
                    T0_next = T
                    break # collecting
                if ((abs(t0 - t1) < ZERO_param)
                      and (abs(point - point1) < ZERO_point)):
                    collect.append(T)
                else:
                    T0_next = T
                    break # collecting
        else:
            try:
                T0_next = next(tps_iter)
            except StopIteration:
                stop = True
        if len(collect) > 1:
            ave_tp = sum([T.index+T.parameter for T in collect]) / len(collect)
            ave_T = TotalParameter(index = int(floor(ave_tp)),
                                    parameter = ave_tp - int(floor(ave_tp)),
                                    end_point = 'not end point')
            reduced_tps.append(ave_T)
        else:
            reduced_tps.append(T0)
        if not stop:
            T0 = T0_next
    return reduced_tps


#==============================================================
#                     Point on arc
#==============================================================

# ------------------------------
# point_on_barc_total_parameters
# ------------------------------
# Wrapper for point_on_barc: Return list of TotalParameters
# where all points very close to an end point of the arc are changed to
# be exactly end points.
#  Args:
# - point: complex
# - cp4:   [complex,complex,complex,complex] (control points)
# - index: integer
# Returns:
# - [TotalParameter]
def point_on_barc_total_parameters(point, cp4, index):
    ZERO = 1e-6
    ts = point_on_barc(point, cp4)
    total_params = []
    for t in ts:
        if t < ZERO:
            total_params.append(TotalParameter(index=index,
                                               parameter=0,
                                               end_point='start'))
        elif t > 1-ZERO:
            total_params.append(TotalParameter(index=index,
                                               parameter=1,
                                               end_point='end'))
        else:
            total_params.append(TotalParameter(index=index,
                                               parameter=t,
                                               end_point='not end point'))
    #print("\npoint_on_barc_total_parameters:")
    #print("Input point = "+str(point))
    #print("Output total_params:")
    #for tp in total_params: print(str(tp))
    return total_params

# -------------
# potential_hit
# -------------
# Given a point and a Bezier arc (ascending or descending)
# find out if the point is possibly on the arc.
# Args:
# - bsa: BezierSubArc
# Returns:
# - boolean
def potential_hit(point, bsa, tolerance=None):
    if tolerance is None:
        ZERO = 1e-8
    else:
        ZERO = tolerance
    return ((bsa.xmin - ZERO <= point.real <= bsa.xmax + ZERO)
              and (bsa.ymin - ZERO <= point.imag <= bsa.ymax + ZERO))

# -------------
# point_on_barc
# -------------
# Given a point and a Bezier arc B(t) (control points cp4, 0 <= t <= 1),
# find out if the point is on the barc, and if it is, the parameter value.
# Return as a sorted list of parameter values t. Empty list means 'No'.
# Accept only those with 0<=t<=1.
#  Args:
# - point: complex
# - cp4: [complex,complex,complex,complex] (control points)
# Returns:
# - [float]
# Notes:
# 1. Phase 1 is the interval subdivision algorithm modified,
#    Phase 2 is is improvement of the result by search.
# 2. Accepts also points a little off the arc.
def point_on_barc(point, cp4, final_accuracy=1e-6, initial_accuracy=None):
    if initial_accuracy is None:
        initial_accuracy = final_accuracy*100 # Accuracy for the first phase.
    MAX_ROUNDS = 100
    if len(cp4) == 1: # 1-anchor stroke
        return []
    # Phase 1
    bsa_list = initial_subdivision(cp4)       # [BezierSubArc]
    # Pairs with possibly hits:
    bsa_candidates = [b1 for b1 in bsa_list if potential_hit(point,b1,initial_accuracy)]
    finished_bsa_candidates = []
    if len(bsa_candidates) == 0:
        #print("point_on_barc: empty bsa_pairs")
        return []
    side_point_parameters = [] # Collect parameters for points not quite on the curve.
    for i in range(MAX_ROUNDS):
        #print("\n----------  Round " + str(i)+"  --------------")
        split_candidates = []
        for b in bsa_candidates:
            b1,b2 = b.split()
            split_candidates += [x for x in [b1,b2]
                             if potential_hit(point,x, initial_accuracy)]
            if len(split_candidates) == 0: # Accept points a little off the curve
                t1 = b1.interval[1]
                t2 = b2.interval[0]
                z = bezier_rv((t1+t2)/2,b1.cp4)
                if abs(point - z) < initial_accuracy:
                    side_point_parameters.append((t1+t2)/2)
                    #print("point_on_barc: Adding side point: "+str(z))
        bsa_candidates = split_candidates
        # Check accuracy
        next_round_bsa_candidates = []
        for bsa in bsa_candidates:
            size = (bsa.xmax-bsa.xmin) + (bsa.ymax-bsa.ymin)
            if (size > initial_accuracy):
                #print("Round "+str(i)+": appending to next:")
                #print(str(bsa))
                next_round_bsa_candidates.append(bsa)
            else:
                #print("Round "+str(i)+": appending to finished:")
                #print(str(bsa))
                finished_bsa_candidates.append(bsa)
        if len(next_round_bsa_candidates) == 0:
            #print("Stop loop at round "+str(i))
            break
        else:
            bsa_candidates = next_round_bsa_candidates
        #print("point_on_barc: Ending round "+str(i)+" with bsa_candidates: "+str(len(bsa_candidates)))
    #else:
    #    print("No break in loop MAX_ROUNDS = "+str(MAX_ROUNDS))
    parameter_candidates = [(bsa.interval[0] + bsa.interval[1]) / 2
                               for bsa in finished_bsa_candidates]
    #
    parameter_candidates += side_point_parameters
    parameter_candidates.sort()
    # Remove duplicates or close values:
    tBs = [(t, bezier_rv(t,cp4)) for t in parameter_candidates]
    reduced_tBs = []
    tB_iter = iter(tBs)
    stop = False
    try:
        tB = next(tB_iter) # first 
    except StopIteration:
        stop = True
    while not stop:
        collect = [tB] # Collect (t,B)'s where B is close to tB[1]
        while True:
            try:
                tB1 = next(tB_iter)
            except StopIteration:
                stop = True
                break # collecting
            if (abs(tB[1]-tB1[1]) < initial_accuracy):
                collect.append(tB1)
            else:
                tB = tB1
                break # collecting (leaving stop=False)
        # Store the best of the collected items: closest B(t) and C(u):
        best_tB = min(collect, key = (lambda x: abs(point-x[1])))
        reduced_tBs.append(best_tB)    
    # Phase 2
    improved_tBs = []
    for tB in reduced_tBs:
        step = final_accuracy
        tB0 = tB
        for i in range(100):
            #print(str(i)+": "+str(abs(tB0[1]-point))+": t = "+str(tB0[0]))
            t0,B0 = tB0
            t1 = t0-step
            t2 = t0+step
            B1 = bezier_rv(t1,cp4)
            B2 = bezier_rv(t2,cp4)
            
            tt1 = t1-t0
            tt2 = t2-t0
            y0 = abs(B0-point)
            y1 = abs(B1-point)
            y2 = abs(B2-point)
            
            yy1 = y1-y0
            yy2 = y2-y0
            
            tB1 = (t1, bezier_rv(t1,cp4))
            tB2 = (t2, bezier_rv(t2,cp4))
            tB_tries = [tB0, tB1, tB2]
            try:
                tmin = t0 + 0.5*(tt2*tt2*yy1 - tt1*tt1*yy2)/(tt2*yy1-tt1*yy2)
                tmin = min(1, max(0,tmin))
                tBmin = (tmin, bezier_rv(tmin,cp4))
                tB_tries.append(tBmin)
            except ZeroDivisionError:
                pass
            tB0 = min(tB_tries, key=(lambda tb: abs(tb[1]-point)))
            if abs(tB0[1]-point) < final_accuracy:
                break
            step /= 2
        improved_tBs.append(tB0)
    return [t for t,_ in improved_tBs]

#==============================================================
#              Chop by path: Initial subdivisions
#==============================================================

# -------------------
# initial_subdivision
# -------------------
# Given a Bezier arc (control points cp4, 0 <= t <= 1), subdivide it at
# the following points contained in the arc:
# (1) self-intersection points (including cusps)
# (2) inflection points
# (3) points with horizontal or vertical tangent.
# Return as a list of objects of class BezierSubArc.
# Note: Because of (1)+(2), each section arc will be curving in the same direction
# the whole way. (Actually, inflection points and cusps would suffice for this, but
# self-intersection points are added just since it is easier to do this way.)
# Because of (3), each section will be ascending or descending the whole way.
# Args:
# - cp4: [complex,complex,complex,complex] (control points)
# Returns:
# - [BezierSubArc]
def initial_subdivision(cp4):
    ZERO = 1e-6 # To remove duplicates or too close values
    if len(cp4) == 1: # 1-anchor stroke
        return []
    # First find special points: self-intersection points or cusps, inflection points
    # (these are the points where the direction of curvature may change):
    case, special_ts = bezier_special_points(cp4)
    #print("Special points:")
    #print(str(case)+': '+str(special_ts))
    ts = [] # Collect subdivision parameter values
    if case == -1: # loop or cusp
        t1,t2 = special_ts
        # Both points must be in the arc for either to be included:
        if (0<t1<1) and (0<t2<1):
            ts += special_ts
    elif case in (1,2): # inflection points: include those in the arc
        ts += [t for t in special_ts if 0<t<1]
    # Then find the horizontal and vertical tangents:
    hor_t_Bs = bezier_tangential_points(cp4, direction=1)
    ver_t_Bs = bezier_tangential_points(cp4, direction=1j)
    if not (hor_t_Bs is None):
        ts += [t for t,_ in hor_t_Bs]
    if not (ver_t_Bs is None):
        ts += [t for t,_ in ver_t_Bs]
    ts = [0] + [t for t in ts if 0 < t < 1] + [1]
    ts.sort()
    # Remove duplicates or close values:
    reduced_ts = []
    ts_iter = iter(ts)
    stop = False
    t = next(ts_iter) # first t (=0)
    while not stop:
        collect = [t] # Collect t's close to t
        while True:
            try:
                t1 = next(ts_iter)
            except StopIteration:
                stop = True
                break # collecting
            if abs(t-t1) < ZERO:
                collect.append(t1)
            else:
                t = t1
                break # collecting
        reduced_ts.append(sum(collect)/len(collect))    
    ts = reduced_ts
    return [BezierSubArc(cp4=cp4, interval=ts[i:i+2]) for i in range(len(ts)-1)]

#==============================================================
#              Chop by path: Basic intersection routines
#==============================================================

# --------------------
# anchor_intersections
# --------------------
# Given two composite Bezier curves as lists of Bezier arcs,
# find the points where an anchor of one curve is on the other curve.
# Return as a list of total parameter pairs (T1,T2)
# where T1 refers to the first Bezier arc and T2 to the second Bezier arc,
# and the list is sorted according to T1.index+T.parameter,
# without any duplicates.
#  Args:
# - bas1: [BCurve.BezierArc]
# - bas2: [BCurve.BezierArc]
# Returns:
# - [(TotalParameter, TotalParameter)] (list of total parameter pairs (T1,T2))
def anchor_intersections(bas1,bas2):
    ZERO = 1e-6
    # First check anchor-anchor equalities:
    tpp1 = [] # Collect total parameter pairs (T1,T2)
    for i1,ba1 in enumerate(bas1):
        for i2,ba2 in enumerate(bas2):
            if i1 == i2 == 0:
                continue # Exclude cases where both are stroke starts.
            if abs(ba1.cp4[0] - ba2.cp4[0]) < ZERO:
                T1 = TotalParameter(index=i1, parameter=0, end_point='start')
                T2 = TotalParameter(index=i2, parameter=0, end_point='start')
                tpp1.append((T1,T2))
    # Abandon those where the arcs both to the left and to the right are equal,
    # in either order:
    tpp1_corr = []
    for T1,T2 in tpp1:
        left1_cp4, right1_cp4 = bas1[T1.index - 1].cp4, bas1[T1.index].cp4
        left2_cp4, right2_cp4 = bas2[T2.index - 1].cp4, bas2[T2.index].cp4
        same_LL = not (bezier_equals_bezier(left1_cp4, left2_cp4) is None)
        same_RR = not (bezier_equals_bezier(right1_cp4, right2_cp4) is None)
        if same_LL and same_RR: # Abandon
            continue
        same_LR = not (bezier_equals_bezier(left1_cp4, right2_cp4) is None)
        same_RL = not (bezier_equals_bezier(right1_cp4, left2_cp4) is None)
        if same_LL and same_RR: # Abandon
            continue
        tpp1_corr.append((T1,T2)) # Accept
    tpp1 = tpp1_corr
    last_anchor1 = bas1[-1].cp4[-1]
    last_anchor2 = bas2[-1].cp4[-1]
    for i2,ba2 in enumerate(bas2):
        if abs(last_anchor1 - ba2.cp4[0]) < ZERO:
            T1 = TotalParameter(index=len(bas1)-1, parameter=1, end_point='end')
            T2 = TotalParameter(index=i2, parameter=0, end_point='start')
            tpp1.append((T1,T2))
    for i1,ba1 in enumerate(bas1):
        if abs(last_anchor2 - ba1.cp4[0]) < ZERO:
            T1 = TotalParameter(index=i1, parameter=0, end_point='start')
            T2 = TotalParameter(index=len(bas2)-1, parameter=1, end_point='end')
            tpp1.append((T1,T2))
    if abs(last_anchor1 - last_anchor2) < ZERO:
        T1 = TotalParameter(index=len(bas1)-1, parameter=1, end_point='end')
        T2 = TotalParameter(index=len(bas2)-1, parameter=1, end_point='end')
        tpp1.append((T1,T2))
    # Second, where anchor of bc1 is on bc2, but exclude anchor-anchor pairs:
    tpp2 = []
    for i1,ba1 in enumerate(bas1):
        for i2,ba2 in enumerate(bas2):
            T2s = point_on_barc_total_parameters(ba1.cp4[0], ba2.cp4, index=i2)
            for T2 in T2s:
                if T2.end_point in ('start','end'): # reject anchor-anchor pairs
                    continue
                T1 = TotalParameter(index=i1, parameter=0, end_point='start')
                tpp2.append((T1,T2))
    for i2,ba2 in enumerate(bas2):
        T2s = point_on_barc_total_parameters(last_anchor1, ba2.cp4, index=i2)
        for T2 in T2s:
            T1 = TotalParameter(index=len(bas1)-1, parameter=1, end_point='end')
            tpp2.append((T1,T2))
    # Third, where anchor of bc2 is on bc1:
    tpp3 = []
    for i2,ba2 in enumerate(bas2):
        for i1,ba1 in enumerate(bas1):
            T1s = point_on_barc_total_parameters(ba2.cp4[0], ba1.cp4, index=i1)
            for T1 in T1s:
                if T1.end_point in ('start','end'): # reject anchor-anchor pairs
                    continue
                T2 = TotalParameter(index=i2, parameter=0, end_point='start')
                tpp3.append((T1,T2))
    for i1,ba1 in enumerate(bas1):
        T1s = point_on_barc_total_parameters(last_anchor2, ba1.cp4, index=i1)
        for T1 in T1s:
            T2 = TotalParameter(index=len(bas2)-1, parameter=1, end_point='end')
            tpp3.append((T1,T2))
    tpp = tpp1+tpp2+tpp3
    # Remove duplicates:
    ddd = dict()
    for TT in tpp:
        T1,T2 = TT
        ddd[(T1.index, T1.parameter, T2.index, T2.parameter)] = TT
    tpp = list(ddd.values())
    #
    tpp.sort(key=(lambda TT: TT[0].index + TT[0].parameter))
    return tpp

# ----------------------
# open_arc_intersections
# ----------------------
# Given two composite Bezier curves as lists of Bezier arcs,
# find the intersections, rejecting those where an anchor is involved.
# Return as a list of total parameter pairs (T1,T2)
# where T1 refers to the first Bezier arc and T2 to the second Bezier arc,
# and the list is sorted according to T1..index+T.parameter.
# Note: There may be very close points which should be removed later.
#  Args:
# - bas1: [BCurve.BezierArc]
# - bas2: [BCurve.BezierArc]
# Returns:
# - [(TotalParameter, TotalParameter)] (list of total parameter pairs (T1,T2))
def open_arc_intersections(bas1,bas2):
    ZERO = 1e-3 # To decide which intersections are anchors
    ttp = [] # [(T1,T2)] list of total parameter pairs
    reject = 0
    for count_arc1, arc1 in enumerate(bas1):
        for count_arc2, arc2 in enumerate(bas2):
            for t1,t2 in barc_intersections(arc1.cp4, arc2.cp4): #[(t1,t2)]
                if t1 == 0 or t2 == 0:
                    reject += 1
                    continue
                T1 = TotalParameter(index=count_arc1, parameter=t1) # 'not end point'
                T2 = TotalParameter(index=count_arc2, parameter=t2) # 'not end point'
                ttp.append((T1,T2))
    ttp.sort(key=(lambda TT: TT[0].index + TT[0].parameter))
    # Remove those too close to anchors:
    anchors1 = [ba.cp4[0] for ba in bas1] + [bas1[-1].cp4[-1]]
    anchors2 = [ba.cp4[0] for ba in bas2] + [bas2[-1].cp4[-1]]
    reduced_ttp = []
    for TT in ttp:
        T1,T2 = TT
        B1 = bezier_rv(T1.parameter, bas1[T1.index].cp4)
        B2 = bezier_rv(T2.parameter, bas2[T2.index].cp4)
        mindist1 = min(abs(B1-a1) for a1 in anchors1)
        mindist2 = min(abs(B2-a2) for a2 in anchors2)
        if (mindist1 > ZERO) and (mindist2 > ZERO):
            reduced_ttp.append(TT)
    return reduced_ttp

#==============================================================
#                   Chop by path: Intersections
#==============================================================

# ----------------------
# potential_intersection
# ----------------------
# Given to Bezier arcs, each ascending or descending,
# find out if they possibly intersect.
# Args:
# - bsa1: BezierSubArc
# - bsa2: BezierSubArc
# Returns:
# - boolean
def potential_intersection(bsa1, bsa2):
    return not((bsa1.xmin > bsa2.xmax) or (bsa1.xmax < bsa2.xmin)
               or (bsa1.ymin > bsa2.ymax) or (bsa1.ymax < bsa2.ymin))

# ------------------
# barc_intersections
# ------------------
# Given two Bezier arcs B(t), C(u) (control points cp4, 0 <= t <= 1),
# find intersections. Return as a list of parameter value pairs (t1,t2)
# where t1 refers to the first Bezier arc and t2 to the second Bezier arc.
# Sort the list according to t1.
# Accept only those pairs (t1,t2) where 0<=t1<=1 and  0<=t2<=1.
#  Args:
# - cp4: [complex,complex,complex,complex] (control points)
# - cp4_other: [complex,complex,complex,complex] (control points)
# Returns:
# - [(float,float)] (list of parameter value pairs (t1,t2))
def barc_intersections(cp4, cp4_other):
    # Assuming B(t)=C(u), is this a cut point or are the curves coincident?
    def proper_cut(t, u, cp4, cp4_other):
        COS_ZERO = 0.9999619230641713 # cos of 0.5 degrees
        Bdot = bezier_dot_rv(t, cp4)
        Cdot = bezier_dot_rv(u, cp4_other)
        return vdot(Bdot,Cdot) < COS_ZERO * abs(Bdot*Cdot)
    ACCURACY = 1e-6 # Goal maximum distance of points accuracy in the plane
    CUSP_ZERO = 1e-7     # To recognize cusps
    MAX_ROUNDS = 100
    if len(cp4) == 1: # 1-anchor stroke
        return []
    if len(cp4_other) == 1: # 1-anchor stroke
        return []
    ############# Phase 1 ##############
    # Check if the Bezier curves are the same up to linear transform of parameter:
    check_equal = bezier_equals_bezier(cp4_other, cp4)
    same_curve = not (check_equal is None)
    # If the arcs are parts of the same curve, find self-intersection points if any
    # and return the result:
    if same_curve:
        A,B = check_equal
        # The Bezier curves B(t) (=cp4) and C(u) (=cp4_other) are related by
        # B(t)=C(u) when u=A+Bt.
        special = bezier_special_points(cp4)
        if special[0] == -1: # Loop or cusp
            # Self-intersection point, or cusp if tt1=tt2:
            tt1,tt2 = special[1]           # Parameters for cp4
            if abs(tt1-tt2) < CUSP_ZERO: # cusp: no self-intersection
                pass
            else:                        # self-intersection
                uu1, uu2 = A+B*tt1, A+B*tt2
                tus = [(tt1,uu1), (tt2,uu1), (tt1,uu2), (tt2,uu2)]
                parameter_pairs = [(t,u) for t,u in tus if (0<=t<=1) and (0<=u<=1)]
                # Discard cases where the two arcs are locally coincident,
                # hence no true cut:
                parameter_pairs = [(t,u) for t,u in parameter_pairs
                                    if proper_cut(t, u, cp4, cp4_other)]
                Bt_Cus = [(bezier_rv(t,cp4),bezier_rv(u,cp4_other))
                            for t,u in parameter_pairs]
                return parameter_pairs
        # No self-intersection point => no intersections
        return []
    ############# Phase 2 ##############
    # Now the arcs are not parts of the same curve.
    # Use the interval subdivision algorithm (modified).
    bsa1_list = initial_subdivision(cp4)       # [BezierSubArc]
    bsa2_list = initial_subdivision(cp4_other) # [BezierSubArc]
    # Pairs with possibly intersections:
    bsa_pairs = [(b1,b2) for b1,b2 in product(bsa1_list, bsa2_list)
                             if potential_intersection(b1, b2)]
    finished_bsa_pairs = []
    if len(bsa_pairs) == 0:
        return []
    for i in range(MAX_ROUNDS):
        split_pairs = []
        for b,c in bsa_pairs:
            b1,b2 = b.split()
            c1,c2 = c.split()
            split_pairs += [(x,y) for x,y in product([b1,b2],[c1,c2]) 
                             if potential_intersection(x,y)]
        bsa_pairs = split_pairs
        # Check accuracy
        next_round_bsa_pairs = []
        for bsa1,bsa2 in bsa_pairs:
            size1 = (bsa1.xmax-bsa1.xmin) + (bsa1.ymax-bsa1.ymin)
            size2 = (bsa2.xmax-bsa2.xmin) + (bsa2.ymax-bsa2.ymin)
            if (size1 > ACCURACY) or (size2 > ACCURACY):
                next_round_bsa_pairs.append((bsa1,bsa2))
            else:
                finished_bsa_pairs.append((bsa1,bsa2))
        if len(next_round_bsa_pairs) == 0:
            #print("Stop loop at round "+str(i))
            break
        else:
            bsa_pairs = next_round_bsa_pairs
    #else:
    #    print("No break in loop MAX_ROUNDS = "+str(MAX_ROUNDS))
    t1_list = []
    t2_list = []
    for bsa1,bsa2 in finished_bsa_pairs:
        t1_list.append((bsa1.interval[0] + bsa1.interval[1]) / 2)
        t2_list.append((bsa2.interval[0] + bsa2.interval[1]) / 2)
    parameter_pairs = zip(t1_list,t2_list)
    ############# Phase 3 ##############
    # Remove duplicates or close values:
    ttBBs = [(t, u, bezier_rv(t,cp4), bezier_rv(u,cp4_other))
                    for t,u in parameter_pairs]
    reduced_ttBBs = []
    ttBB_iter = iter(ttBBs)
    stop = False
    try:
        ttBB = next(ttBB_iter) # first 
    except StopIteration:
        stop = True
    while not stop:
        collect = [ttBB] # Collect tt's close to tt
        while True:
            try:
                ttBB1 = next(ttBB_iter)
            except StopIteration:
                stop = True
                break # collecting
            if ((abs(ttBB[2]-ttBB1[2]) < ACCURACY)
                        and (abs(ttBB[3]-ttBB1[3]) < ACCURACY)):
                collect.append(ttBB1)
            else:
                ttBB = ttBB1
                break # collecting (leaving stop=False)
        # Store the best of the collected items: closest B(t) and C(u):
        best_ttBB = min(collect, key = (lambda x: abs(x[2]-x[3])))
        reduced_ttBBs.append(best_ttBB)    
    return [(t,u) for t,u,_,_ in reduced_ttBBs]

# --------------------
# bcurve_intersections
# --------------------
# Given two composite Bezier curves
# (that is, successions of butting Bezier arcs (BCurve.BezierArc)),
# find intersections. Return as a list of total parameter pairs (T1,T2)
# where T1 refers to the first Bezier arc and T2 to the second Bezier arcrve.
# Discard cases where the point is stroke start or end of either curve.
# Sort the list according to (T1.index + T1.parameter).
#  Args:
# - bas1: [BCurve.BezierArc]
# - bas2: [BCurve.BezierArc]
# Returns:
# - [(TotalParameter,TotalParameter)] (list of total parameter value pairs (T1,T2))
def bcurve_intersections(bas1, bas2):
    # Intersections with anchors involved:
    tpp_anchors = anchor_intersections(bas1,bas2) # No duplicates
    # Intersections in open Bezier arcs:
    tpp_arcs = open_arc_intersections(bas1,bas2)
    # Combined:
    bc_intersect = tpp_anchors + tpp_arcs
    # Remove those (T1,T2) where T1 or T2 is stroke start or end:
    return sorted(bc_intersect, key=(lambda TT:TT[0].index + TT[0].parameter))

#==============================================================
#                     Chop path by lengths
#==============================================================

class GimpStrokeData(object):
    """
    To store one stroke of Gimp.
    Attributes:
    - image
    - path:      gimp.Vectors
    - stroke_id: integer (=stroke.ID)
    - bas:       [BCurve.BezierArc] (as returned by completed_bezier_arcs)
    """
    def __init__(self, image, path, stroke):
        self.image = image
        self.path = path
        self.stroke_id = stroke.ID
        for i, gimp_stroke in enumerate(path.strokes):
            if stroke.ID == gimp_stroke.ID:
                stroke_number = i
                break
        else:
            raise Exception("GimpStrokeData: stroke_number not found")
        gv = vectors_object2gv(path)         # BCurve.GimpVectors
        gs = gv.stroke_list[stroke_number]   # BCurve.GimpStroke
        bc = gs.gs2bc()                      # BCurve.BezierCurve
        self.bas = completed_bezier_arcs(bc) # [BCurve.BezierArc]

# ----------------------
# path2strokes_with_data
# ----------------------
# From path construct [GimpStrokeData].
# Args:
# - image
# - path:   gimp.Vectors
# - stroke: gimp.VectorsBezierStroke (a stroke of the path)
# Returns:
# - [GimpStrokeData] (strokes in the same order as in the path)
def path2strokes_with_data(image, path):
    return [GimpStrokeData(image=image,
                           path=path,
                           stroke=stroke,
                           ) for stroke in path.strokes]

# ---------------------
# path_tps_at_distances
# ---------------------
# For each stroke in path.strokes, compute the TotalParameters of the points at
# the given distances from the start. Return a list of lists where, for each stroke,
# items are a list of TotalParameters for that stroke (possibly []).
# Args:
# - image
# - path:        gimp.Vectors
# - chop_case:        string ('from start' or 'whole strokes')
# - chop_length: float
# - chop_number: integer
# Returns:
# - [[TotalParameter]] # List number i corresponds to stroke number i.
def path_tps_at_distances(image, path, chop_case, chop_length, chop_number):
    precision = 1e-20
    stroke_list = path2strokes_with_data(image,path) # [GimpStrokeData]
    #distances_in_general = [i*chop_length for i in range(1, chop_number + 1)]
    stroke_tps_list = []
    for stroke in stroke_list:
        if chop_case == 'whole strokes':
            stroke_length = pdb.gimp_vectors_stroke_get_length(path,
                                                               stroke.stroke_id,
                                                               precision)
            if stroke_length == -1.0: # 1-anchor stroke?
                stroke_tps_list.append([])
                continue
            stroke_chop_length = stroke_length / chop_number
            distances = [i * stroke_chop_length
                                      for i in range(1, chop_number + 1)]
        elif chop_case == 'from start':
            distances = [i*chop_length for i in range(1, chop_number + 1)]
        else:
            raise Exception("Unknown chop_case: "+chop_case)
        stroke_tps_list.append(stroke_tps_at_distances(stroke, distances))
    return stroke_tps_list

# -----------------------
# stroke_tps_at_distances
# -----------------------
# For a stroke:GimpStrokeData, compute the TotalParameters of the points at
# the distances in the given distance list. Return as sorted list of TotalParameters.
# The sequence of the distances is assumed to be growing, hence the work is stopped
# as soon one case where the point was not found is encountered, meaning that
# the end of the stroke was reached.
# Args:
# - stroke: GimpStrokeData
# - distances: [float]
# Returns:
# - [TotalParameter]
def stroke_tps_at_distances(stroke, distances):
    tp_list = []
    for distance in distances:
        tp,_ = stroke_tp_at_distance(stroke, distance)
        if not (tp is None):
            tp_list.append(tp)
        else:
            # End of stroke: stop work
            break
    tp_list.sort(key=(lambda tp:tp.index + tp.parameter))
    # Reduce the list: Remove too close ones and set those very close to an anchor
    # to be exactly the anchor.
    bas = stroke.bas
    tp_list_reduced = remove_close_total_parameters(tp_list, bas)
    return tp_list_reduced

# ---------------------
# stroke_tp_at_distance
# ---------------------
# For a stroke:GimpStrokeData, compute the TotalParameter of the point at
# the given distance from start. Return TotalParameter or None,
# the latter meaning that the distance is not valid for the stroke.
# Return also error (the distance between the point which 
# pdb.gimp_vectors_stroke_get_point_at_dist gives and the point computed 
# from the found TotalParameter.
# Args:
# - stroke: GimpStrokeData
# - distance: float (pixels)
# Returns:
# - TotalParameter or None
# - float or None 
def stroke_tp_at_distance(stroke, distance):
    precision = 1e-12 # pdb
    final_accuracy   = 1e-6
    initial_accuracy = 1e-4
    #
    point, index = bas_get_point_at_dist(stroke.image,
                                         stroke.bas,
                                         distance,
                                         precision)
    if point is None:
        return None,None
    ba = stroke.bas[index]
    ts = point_on_barc(point, ba.cp4, final_accuracy, initial_accuracy)
    #print("stroke_tp_at_distance: ts = "+str(ts))
    if len(ts) == 0:
        raise Exception("Valid point but no total parameter found, point = "+str(point), None)
    t = min(ts, key=(lambda x: abs(point - bezier_rv(x, ba.cp4))))
    #print("stroke_tp_at_distance: choose t = "+str(t))
    error = abs(point - bezier_rv(t, ba.cp4))
    return TotalParameter(index=index, parameter=t), error # not end point

# ---------------------
# bas_get_point_at_dist
# ---------------------
# Given a stroke as bas (list of Bezier arcs) and a distance, find coordinates of
# the point at that distance from the start. Return also the index of the Bezier
# arc in bas where the point is found. Return point=None if the point was not found.
# Args:
# - image
# - bas:       [BCurve.BezierArc]
# - distance:  float
# - precision: float
# Returns:
# - complex or None (the point, or None if not found)
# - integer (the index of the Bezier Arc)
def bas_get_point_at_dist(image, bas, distance, precision):
    for ba_count in range(len(bas)):
        sub_bc = BCurve.BezierCurve(bezier_arcs = bas[:ba_count+1]) # Cumulate
        sub_path = sub_bc.bc2vectors_object(image, name=None)       # One stroke
        sub_stroke = sub_path.strokes[0] 
        x_point, y_point, slope, valid = pdb.gimp_vectors_stroke_get_point_at_dist(
                    sub_path,
                    sub_stroke.ID,
                    distance,
                    precision)
        if not valid:
            continue
        else:
            return complex(x_point, y_point), ba_count
    # If come here, point was not found
    return None,None

#==============================================================
#             Chop by path. Main procedure
#==============================================================

# -------------------------------
# bcurve_chop_by_total_parameters
# -------------------------------
# Given a composite Bezier curve as a list of total parameters along the curve,
# chop the curve at those points.
# Args:
# - bc: BCurve.BezierCurve
# - total_params: [TotalParameter]
# Returns:
# - [BCurve.BezierCurve]
def bcurve_chop_by_total_parameters(bc, total_params):
    bas = completed_bezier_arcs(bc) # Takes care of closed case
    # Remove potential start and end anchors of bc:
    tps = []
    for T in total_params:
        if (T.index == 0) and (T.end_point == 'start'):
            continue
        if (T.index == len(bas)-1) and (T.end_point == 'end'):
            continue
        tps.append(T)
    # Change all 'end' cases to 'start' cases:
    for T in tps:
        if T.end_point == 'end':
            T.index += 1
            T.parameter = 0
            T.end_point = 'start'
    if len(total_params) == 0:
        return [deepcopy(bc)]
    # Cut arcs and collect in list:
    chop_bas = []
    for index,ba in enumerate(bas):
        Ts = [T for T in total_params if T.index == index]
        ts = [T.parameter for T in Ts]
        if len(Ts) == 0:
            chop_bas += [ba]
            continue
        chop_cp4s = bezier_subdivide_at_parameters(ts, ba.cp4)
        chop_bas += [BCurve.BezierArc(cp4=cut_cp4) for cut_cp4 in chop_cp4s]
    # To cut the list chop_bas to strokes:
    # cut_indexes = list of indexes in total_params;
    # cut_points = list of those indexes in the list chop_bas where new
    # stroke should start (except for the very first start).
    cut_indexes = [T.index for T in total_params]
    corrections = [0 for count in range(len(cut_indexes))]
    for count,T in enumerate(total_params):
        if T.end_point == 'start':
            for j in range(count,len(cut_indexes)):
                corrections[j] -= 1
    cut_points = [1 + count + i + corrections[count]
                  for count,i in enumerate(cut_indexes)] # cut_points of new strokes
    # Collect arcs according to new strokes:
    chop_ba_lists = []
    lower = 0
    for upper in cut_points:
        chop_ba_lists.append(chop_bas[lower:upper])
        lower = upper
    chop_ba_lists.append(chop_bas[cut_points[-1]:])
    if bc.closed: # Concatenate last and first
        chop_ba_lists[0] = chop_ba_lists[-1] + chop_ba_lists[0]
        chop_ba_lists = chop_ba_lists[:-1]
    # Make strokes:
    return [BCurve.BezierCurve(bezier_arcs=ba_list)
               for ba_list in chop_ba_lists if len(ba_list) != 0]

# -----------------
# path_chop_by_path
# -----------------
# Chop path1 by path2: Cut the strokes at crossing points with path2
# to new strokes. Return as new path.
# Args:
# - image
# - path1: gimp.Vectors (path to be chopped)
# - path2: gimp.Vectors (slicing path)
# - result_case:      string ('one path' or 'separate paths')
# Returns:
# - gimp.Vectors
def path_chop_by_path(image, path1, path2, result_case):
    gv1 = vectors_object2gv(path1)  # BCurve.GimpVectors
    gv2 = vectors_object2gv(path2)  # BCurve.GimpVectors
    # Make lists of Bezier curves:
    bc1_list = [gs.gs2bc() for gs in gv1.stroke_list]
    bc2_list = [gs.gs2bc() for gs in gv2.stroke_list]
    #
    chopped_bc_list = []
    for bc1 in bc1_list:
        bas1 = completed_bezier_arcs(bc1) # Takes care of closed case
        total_params = []
        for bc2 in bc2_list: # BCurve.GimpStroke
            bas2 = completed_bezier_arcs(bc2) # Takes care of closed case
            total_parameter_pairs = bcurve_intersections(bas1, bas2)
            total_params += [T1 for T1,T2 in total_parameter_pairs]
        total_params.sort(key=(lambda T:T.index + T.parameter))
        total_params = remove_close_total_parameters(total_params, bas1)
        chop_bcs = bcurve_chop_by_total_parameters(bc1, total_params) # [BCurve.BezierCurve]
        chopped_bc_list += chop_bcs
    chopped_gs_list = [chop_bc.bc2gs()                 # [BCurve.GimpStroke]
                     for chop_bc in chopped_bc_list]
    if result_case == 'one path':
        new_gv = BCurve.GimpVectors(stroke_list = chopped_gs_list)
        chopped_path = new_gv.gv2vectors_object(image, name=path1.name+'|chop')
        return [chopped_path]
    elif result_case == 'separate paths':
        chopped_paths = []
        count = 0
        for chopped_gs in chopped_gs_list:
            new_gv = BCurve.GimpVectors(stroke_list = [chopped_gs])
            name = path1.name+'|chop '+str(count)
            chopped_paths.append(new_gv.gv2vectors_object(image, name=name))
            count += 1
        return chopped_paths
    else:
        raise Exception("Unknown result_case: "+result_case)

# ----------------------
# path_chop_by_path_main
# ----------------------
# Main procedure.
# Chop path1 by path2: The strokes of path1 are cut at the crossing points
# with path2 to new strokes. From these a new path is built. Its strokes are
# faithfully in the same ordering as in which they appear in the visual image.
# Draw.
# Args:
# - image
# - path1: gimp.Vectors (path to be chopped)
# - path2: gimp.Vectors (chopping path)
# - chop_result_option: integer
# Returns:
# - gimp.Vectors (only the first path if there are more than one)
def path_chop_by_path_main(image, path1, path2, chop_result_option):
    result_case = chop_result_options[chop_result_option][1]
    chopped_paths = path_chop_by_path(image, path1, path2, result_case)
    pdb.gimp_image_undo_group_start(image)
    for p in chopped_paths:
        gimp_draw_vectors_object(image, p, visible=True) # Draw
    pdb.gimp_image_undo_group_end(image)
    try:
        return chopped_paths[0]
    except IndexError:
        return

#======================================================
#          Chop path at anchors. Main procedure
#======================================================

# --------------------
# chop_path_at_anchors
# --------------------
# Given a path, chop it into small paths with two anchors each (end points).
# Exception: Open one-anchor strokes are made to a one-anchor paths.
# Args:
# - image
# - path:  gimp.Vectors
# - result_case: string
# Returns:
# - [gimp.Vectors]
def chop_path_at_anchors(image, path, result_case):
    gv = vectors_object2gv(path)             # BCurve.GimpVectors
    if result_case == 'one path':
        chopped_gss = []                     # collect here [BCurve.GimpStroke]
        for gs in gv.stroke_list:                # gs:BCurve.GimpStroke
            bc = gs.gs2bc()                      # BCurve.BezierCurve
            for ba in completed_bezier_arcs(bc): # ba:BCurve.BezierArc
                new_bc = BCurve.BezierCurve(bezier_arcs=[ba])
                chopped_gss.append(new_bc.bc2gs())
        new_gv = BCurve.GimpVectors(stroke_list=chopped_gss)
        chopped_path = new_gv.gv2vectors_object(image, name=path.name+'|chop')
        return [chopped_path]
    elif result_case == 'separate paths':
        chopped_paths = []                             # result to here:[gimp.Vectors]
        count = 0
        for gs in gv.stroke_list:                # gs:BCurve.GimpStroke
            bc = gs.gs2bc()                      # BCurve.BezierCurve
            for ba in completed_bezier_arcs(bc): # ba:BCurve.BezierArc
                new_bc = BCurve.BezierCurve(bezier_arcs=[ba])
                name = path.name+'|chop '+str(count)
                chopped_paths.append(new_bc.bc2vectors_object(image, name=name))
                count += 1
        return chopped_paths
    else:
        raise Exception("Unknown result_case: "+result_case)

# -------------------------
# chop_path_at_anchors_main
# -------------------------
# Main procedure. Draw.
# Args:
# - image
# - path:               gimp.Vectors
# - chop_result_option: integer
# Returns:
# - gimp.Vectors (only the first path if there are more than one)
def chop_path_at_anchors_main(image, path, chop_result_option):
    result_case = chop_result_options[chop_result_option][1]
    chopped = chop_path_at_anchors(image, path, result_case)
    pdb.gimp_image_undo_group_start(image)
    for p in chopped:
        gimp_draw_vectors_object(image, p, visible=True)
    pdb.gimp_image_undo_group_end(image)
    try:
        return chopped[0]
    except IndexError:
        return

#==============================================================
#               Chop path by lengths. Main procedure
#==============================================================

# --------------------
# path_chop_by_lengths
# --------------------
# Chop each stroke of the path by lengths, either
# - by cutting from the start 'chop_number' pieces of length 'chop_length',
# - or, if chop_length = 0, cutting the whole stroke to 'chop_number' pieces
#   of equal length.
# Depending on 'result_case', return one path with the pieces as strokes, or
# several paths, each consisting of one piece.
# Args:
# - image
# - path:             gimp.Vectors (path to be chopped)
# - chop_case:        string ('from start' or 'whole strokes')
# - chop_length:      float
# - chop_number:      integer
# - result_case:      string ('one path' or 'separate paths')
# - reversed_strokes: boolean
# Returns:
# - [gimp.Vectors]
def path_chop_by_lengths(image,
                         path,
                         chop_case,
                         chop_length,
                         chop_number,
                         result_case,
                         reversed_strokes):
    gv = vectors_object2gv(path)  # BCurve.GimpVectors
    if reversed_strokes:
        gs_list = [reversed_GimpStroke(gs) for gs in gv.stroke_list]
        gv = BCurve.GimpVectors(stroke_list = gs_list, name = gv.name+' reversed')
        work_path = gv.gv2vectors_object(image, name=path.name+' reversed')
    else:
        gs_list = gv.stroke_list
        work_path = path
    # Make lists of Bezier curves:
    bc_list = [gs.gs2bc() for gs in gs_list]
    bc_tps_list = path_tps_at_distances(image, # [[TotalParameter]]
                                        work_path, chop_case, chop_length, chop_number)
    chopped_bc_list = []
    for stroke_number,bc in enumerate(bc_list):
        total_params = bc_tps_list[stroke_number]
        if (chop_case != 'whole strokes') and bc.closed: # Not whole closed bc chopped
            bas = completed_bezier_arcs(bc)
            # Not whole bc chopped: want a cut at the end of the stroke:
            total_params.append(TotalParameter(index = len(bas)-1,
                                               parameter = 1,
                                               end_point = 'end'))
        chop_bcs = bcurve_chop_by_total_parameters(bc, total_params) # [BCurve.BezierCurve]
        chopped_bc_list += chop_bcs
    chopped_gs_list = [chop_bc.bc2gs()                 # [BCurve.GimpStroke]
                     for chop_bc in chopped_bc_list]
    if result_case == 'one path':
        new_gv = BCurve.GimpVectors(stroke_list = chopped_gs_list)
        chopped_path = new_gv.gv2vectors_object(image, name=path.name+'|chop')
        return [chopped_path]
    elif result_case == 'separate paths':
        chopped_paths = []
        count = 0
        for chopped_gs in chopped_gs_list:
            new_gv = BCurve.GimpVectors(stroke_list = [chopped_gs])
            name = path.name+'|chop '+str(count)
            chopped_paths.append(new_gv.gv2vectors_object(image, name=name))
            count += 1
        return chopped_paths
    else:
        raise Exception("Unknown result_case: "+result_case)

# -------------------------
# chop_path_at_lengths_main
# -------------------------
# Main procedure. Draw.
# Args:
# - image
# - path:               gimp.Vectors (path to be chopped)
# - chop_length_option: integer
# - chop_number:        integer
# - chop_length:        float
# - chop_result_option: integer
# - reversed_strokes:   boolean
# Returns:
# - gimp.Vectors (only the first path if there are more than one)
def chop_path_at_lengths_main(image,
                              path,
                              chop_length_option,
                              chop_number,
                              chop_length,
                              chop_result_option,
                              reversed_strokes
                              ):
    chop_case = chop_length_options[chop_length_option][1]
    result_case = chop_result_options[chop_result_option][1]
    chop_length = max(1.,chop_length)
    chop_number = max(1,chop_number)
    chopped_paths = path_chop_by_lengths(image,
                                           path,
                                           chop_case,
                                           chop_length,
                                           chop_number,
                                           result_case,
                                           #merge,
                                           reversed_strokes)
    pdb.gimp_image_undo_group_start(image)
    for p in chopped_paths:
        gimp_draw_vectors_object(image, p, visible=True) # Draw
    pdb.gimp_image_undo_group_end(image)
    try:
        return chopped_paths[0]
    except IndexError:
        return

chop_length_options = [ # (description, identifier)
        (('Chop pieces from each stroke from the start'
       +'\nuntil the given number of pieces is reached'
       +'\nor the stroke end is met'),      'from start'),
        (('Chop each stroke to equal pieces from start to end'),
                                            'whole strokes'),
        ]

chop_result_options = [ # (description, identifier)
        ('Make one path from the chopped pieces', 'one path'),
        ('Make separate paths from the chopped pieces', 'separate paths')
        ]


#======================================================
#                    Registrations
#======================================================

versionnumber = "0.3"
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2021"
image_types = "*"
menupath = '<Vectors>/Tools/Chopping'

#####################  Chop path at its anchors  #####################

procedure_name  = "chop_path_at_anchors"
procedure_blurb = ("Chop the path at its anchors."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = ("Chop the strokes of the path into 2-anchor strokes"
                   +"\nby cutting them at the anchors."
                   +"\nOptionally each stroke can be made into a separate path."
                   +"\nOpen one-anchor strokes become open one-anchor paths."
                   +"\nClosed one-anchor strokes are sliced open."
                   +"\nLeading and trailing handles are not preserved."
                   )

procedure_label = "Chop path at its anchors"

procedure_function = chop_path_at_anchors_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path", None),
      (PF_OPTION, 'chop_result_option','Result',
                   0,
                  [case[0] for case in chop_result_options]),
    ],
    [
        (PF_VECTORS, "chopped", "First chopped path"),
    ],
    procedure_function,
    menu=menupath)

#####################  Chop path by another path  #####################

procedure_name  = "chop_path_by_path"
procedure_blurb = ("Chop the strokes of the path by cutting them"
                   +"\nat the crossing points with the other path."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = ("Chop the strokes of the path into shorter strokes"
                   +"\nby cutting them at the crossing points with the other path."
                   +"\nOptionally each stroke can be made into a separate path."
                   )

procedure_label = "Chop path by another path"

procedure_function = path_chop_by_path_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path1", "The path to be chopped", None),
      (PF_VECTORS, "path2", "The chopping path", None),
      (PF_OPTION, 'chop_result_option','Result',
                   0,
                  [case[0] for case in chop_result_options]),
    ],
    [
        (PF_VECTORS, "chopped", "First chopped path"),
    ],
    procedure_function,
    menu=menupath)

#####################  Chop path by lengths  #####################

procedure_name  = "chop_path_by_lengths"
procedure_blurb = ("Chop pieces from each stroke, according to"
                   +"\ngiven number and length of the pieces."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = ("Chop each stroke of the path by cutting"
                   +"\npieces of certain length from the start or"
                   +"\nthe whole stroke to pieces of equal length."
                   +"\nOptionally each piece can be made into a separate path."
                   +"\nNotes: 1. The starting anchor of each stroke is meaningful"
                   +"\nbut not obvious to the user."
                   +"\n2. The lengths are measured along the path as given by"
                   +"\npdb.gimp_vectors_stroke_get_point_at_dist."
                   )

procedure_label = "Chop path by lengths"

procedure_function = chop_path_at_lengths_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path to be chopped", None),
      (PF_OPTION, 'chop_length_option','Action',
                   0,
                  [case[0] for case in chop_length_options]),
      (PF_INT, "chop_number", "Number of chopped pieces in each stroke", 4),
      (PF_FLOAT, "chop_length", ("Length of chopped pieces in pixels"
                              +"\n(ignored if strokes are chopped to equal pieces)"),
                                 100),
      (PF_OPTION, 'chop_result_option','Result',
                   0,
                  [case[0] for case in chop_result_options]),
      (PF_BOOL, "reversed_strokes", "Use all strokes reversed?", False),
    ],
    [
        (PF_VECTORS, "chopped", "First chopped path"),
    ],
    procedure_function,
    menu=menupath)


main()
