#!/usr/bin/env python

# Gimp plugins to crop or slice paths
#
# History:
# v0.1: 2020-10-04: Four plugins:
#                   crop path by rectangle,
#                   crop path by convex polygon,
#                   slice path by guides
#                   slice path by lines
# v0.2: 2020-10-08: First published version.
#                   New algorithm for polynomial roots. Bug fixes.
# v0.3: 2021-01-01: Bug fix: tail handle when splitting closed stroke.
# v0.4: 2021-01-17: Developing
# v0.5: 2021-01-17: New plugin:
#                   crop path by circle
# v0.6: 2021-03-29: New plugin:
#                   crop path by general selection
# v0.7: 2021-03-31: Making faster. Bug fix.
# v0.8: 2021-04-01: Bug fix.

# (c) Markku Koppinen 2020, 2021
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'


from __future__ import division, print_function
from gimpfu import *
from math import *
from functools import partial

#======================================================
#                       BCurve
#======================================================

# Class: BCurve. Designed to facilitate manipulating paths arc-by-arc.
# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

RowVector = complex

class BCurve(object):
    """Data structures to handle Gimp vectors objects by means of individual
    Bezier arcs.
    By a "Bezier arc" we mean here a Bezier curve determined by 4 control points
    p0,p1,p2,p3.
    This class contains data structures needed to do computations on
    a Bezier curve which are done arc by arc.
    To this end, Gimp data structures
    - gimp.Vectors
    - gimp.VectorsBezierStroke
    are here represented as data structures (classes)
    - GimpVectors
    - GimpStroke
    - BezierCurve
    - BezierArc
    - ControlPoint = complex
    with the aim that
    - each control point [x,y] is represented as an instance of ControlPoint;
    - each Bezier arc (four control points, see above) is represented as an instance
      of BezierArc.
    """
    
    ControlPoint = RowVector # =complex
    
    class GimpVectors(object):
        """Essentially same data as in a gimp.Vectors object except that
        strokes are instances of GimpStroke rather than
        gimp.VectorsBezierStroke.
        Attributes:
        - stroke_list: [GimpStroke]
        - name:        string
        """
        def __init__(self,
                     stroke_list,            # [GimpStroke]
                     name='GimpVectors'      # string
                     ):
            self.stroke_list = stroke_list
            self.name = name
        def __str__(self):
            s = 'GimpVectors '+self.name
            count = 0
            for stroke in self.stroke_list:
                s += '\n  Stroke '+str(count)+': GimpStroke'
                s += '\n    '+str(stroke)
                count += 1
            return s
        def gv2vectors_object(self, image, name=None):
            """Conversion GimpVectors -> gimp.Vectors
            (The converse: vectors_object2gv)
            """
            def cp_list2xy_list(cp_list):
                xy = []
                for cp in cp_list:
                    xy += [cp.real, cp.imag]
                return xy
            if name is None:
                name = self.name
            vectors = pdb.gimp_vectors_new(image, name)
            for stroke in self.stroke_list:
                closed = stroke.closed
                xy_list = cp_list2xy_list(stroke.cp_list)
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                             vectors, 0, len(xy_list), xy_list, closed)
            return vectors
    
    class GimpStroke(object):
        """Essentially same data as in a gimp.VectorsBezierStroke.points,
        except that control points list [x,y,x,y,...] is arranged as
        a list [ControlPoint] by joining successive x,y.
        Attributes:
        - cp_list:     [ControlPoint]
        - closed:      boolean
        - stroke_name: string
        """
        def __init__(self,
                     cp_list,                   # [ControlPoint]
                     closed,                    # boolean
                     stroke_name = 'GimpStroke' # string
                     ):
            self.stroke_name = stroke_name
            self.cp_list = cp_list
            self.closed = closed
        def __str__(self):
            s = self.stroke_name + '; Control points:'
            for cp in self.cp_list:
                s += '\n    '+str(cp)
            s += '\n    closed: '+str(self.closed)
            return s
        def gs2bc(self):
            """Conversion GimpStroke -> BezierCurve
            """
            head = self.cp_list[0]
            tail = self.cp_list[-1]
            inner = self.cp_list[1:-1]
            if len(inner) > 1: # Must be divisible by 4
                ba_list = []
                count = 0
                for i in range(0,len(inner)-1,3):
                    ba_list.append(BCurve.BezierArc(inner[i:i+4], 'arc '+str(count)))
                    count += 1
            elif len(inner) == 1: # Stroke has only one anchor
                ba_list = [BCurve.BezierArc([inner[0]], 'arc 0')]
            else:
                raise Exception("BCurve.GimpStroke.gs2bc: No anchors in stroke?")
            return BCurve.BezierCurve(
                               bezier_arcs = ba_list,
                               head_handle = head,
                               tail_handle = tail,
                               closed = self.closed,
                               curve_name = self.stroke_name
                               )
    
    class BezierCurve(object):
        """BezierCurve is a list of butting Bezier arcs with some extra data.
        Attributes:
        - curve_name:  string
        - bezier_arcs: [BezierArc];
        - head_handle: ControlPoint;
        - tail_handle: ControlPoint;
        - closed:      boolean.
        Note: "Butting" means that for any pair of successive arcs, the last
              control point of the former equals the first control point of
              the latter.
              In initialization no checks are done about this condition.
        """
        def __init__(self,
                     bezier_arcs=[],           # [BezierArc]
                     head_handle=None,         # ControlPoint
                     tail_handle=None,         # ControlPoint
                     closed=False,             # boolean
                     curve_name='BezierCurve', # string
                     ):
            self.curve_name = curve_name
            self.bezier_arcs = bezier_arcs
            if head_handle is None:
                self.head_handle = bezier_arcs[0].cp4[0]
            else:
                self.head_handle = head_handle
            if tail_handle is None:
                self.tail_handle = bezier_arcs[-1].cp4[-1]
            else:
                self.tail_handle = tail_handle
            self.closed = closed
        def __str__(self):
            s = self.curve_name + ': Bezier curve' + '; Bezier arcs:'
            count = 0
            for arc in self.bezier_arcs:
                s += '\n  arc '+ str(count)+ ': ' +str(arc)
                count += 1
            s += '\n  Head handle:    '+str(self.head_handle)
            s += '\n  Tail handle:    '+str(self.tail_handle)
            s += '\n  Closed:'    +str(self.closed)
            return s
        def bc2gs(self):
            """Conversion BezierCurve -> GimpStroke
            """
            cp_list = [self.head_handle]
            if len(self.bezier_arcs[0].cp4) > 1:
                for ba in self.bezier_arcs:
                    cp_list += ba.cp4[:3]
                cp_list.append(self.bezier_arcs[-1].cp4[-1])
            else:
                # Only one BezierArc and it has cp4 = [p0]
                # (may rise from Gimp stroke with only one anchor).
                cp_list.append(self.bezier_arcs[0].cp4[0])
            cp_list.append(self.tail_handle)
            return BCurve.GimpStroke(cp_list, self.closed, self.curve_name)
        def bc2vectors_object(self, image, name=None):
            """Conversion BezierCurve -> gimp.Vectors
            Makes vectors_object with 1 stroke.
            """
            if name is None:
                name = self.curve_name
            gs = self.bc2gs()             # BCurve.GimpStroke
            gv = BCurve.GimpVectors([gs]) # BCurve.GimpVectors
            return gv.gv2vectors_object(image, name)
    
    class BezierArc(object):
        """Data structure for one Bezier arc: arc determined by 4 control
           points.
        Attributes:
        - cp4:      [p0,p1,p2,p3] where each pi:ControlPoint, or
                    [p0] if only one control point (may rise from
                    a stroke with only one anchor);
        - arc_name: string
        """
        def __init__(self,
                     cp4,                 # cp4=[p0,p1,p2,p3] with pi:ControlPoint
                                          # or [p0]
                     arc_name='BezierArc' # string
                     ):
            self.cp4 = cp4
            self.arc_name = arc_name
        def reverse(self):
            self.cp4.reverse()
        def __str__(self):
            s = 'BezierArc, control points:'
            for cp in self.cp4:
                s += '\n  '+ str(cp)
            return s

# -----------------
# vectors_object2gv
# -----------------
# Conversion gimp.Vectors -> GimpVectors
# Args:
# - vectors_object: gimp.Vectors (Gimp vectors object)
# Returns:
# - GimpVectors
#
# Note: The inverse conversion is GimpVectors.gv2vectors_object(...).
#       See also BezierCurve.bc2vectors_object(...).
def vectors_object2gv(vectors_object):
    def xy_list2cp_list(xy_list): # xy_list = [x,y,x,y,...,]
        cp = []
        for i in range(0,len(xy_list),2):
            cp.append(BCurve.ControlPoint(xy_list[i], xy_list[i+1]))
        return cp
    stroke_list = []
    count = 0
    for stroke in vectors_object.strokes:
        xy_list = stroke.points[0]
        closed = stroke.points[1]
        cp_list = xy_list2cp_list(xy_list)
        stroke_name = vectors_object.name + ' stroke '+str(count)
        stroke_list.append(BCurve.GimpStroke(cp_list, closed, stroke_name))
        count += 1
    return BCurve.GimpVectors(stroke_list, vectors_object.name)


#======================================================
#           Miscellaneous auxiliary routines
#======================================================

def gimp_message(txt, h=2):
    # 0 = message box; 1 = console; 2 = error console
    pdb.gimp_message_set_handler(h)
    pdb.gimp_message(txt)

def print_list_split_points(bc, lsp): # lsp:[[n,t]] For debugging only
        p0 = bezier_rv(0, bc.bezier_arcs[0].cp4)
        p3 = bezier_rv(1, bc.bezier_arcs[-1].cp4)
        print("bcurve end points: "+str(p0) +', '+str(p3))
        for n,t in lsp:
            p = bezier_rv(t, bc.bezier_arcs[n].cp4)
            print('n='+str(n) +', t='+str(t)+', B(t)='+str(p))

# ---------
# bezier_rv
# ---------
# The usual Bezier curve and derivatives from control points
# Version where plane points are complex numbers.
# Args:
# - t:float
# - control_points = [P0,P1,P2,P3]:[complex]
# Returns:
# - complex
def bezier_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    if t > 0.5:
        u = (1-t)/t
        t3 = t**3
        return t3*(u*(u*(u*P0+3*P1)+3*P2)+P3) # complex
    else:
        u = t/(1-t)
        t3 = (1-t)**3
        return t3*(u*(u*(u*P3+3*P2)+3*P1)+P0) # complex

# Derivative: Returns complex.
def bezier_dot_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    P01 = -P0+P1
    P12 = -P1+P2
    P23 = -P2+P3
    if t > 0.5:
        u = (1-t)/t
        t2 = t**2
        return 3*t2*(u*(u*P01+2*P12)+P23) # complex
    else:
        u = t/(1-t)
        t2 = (1-t)**2
        return 3*t2*(u*(u*P23+2*P12)+P01) # complex

# Second derivative
def bezier_dot_dot_rv(t, control_points): # Could be made faster as above.
    p0,p1,p2,p3 = control_points
    return -6*(1-t)*(-p0+p1) + 6*(1-2*t)*(-p1+p2) + 6*t*(-p2+p3)

# -------------------------
# bezier_new_control_points
# -------------------------
# Given a Bezier arc B(t) and two distinct parameter values a and b,
# find a new parametrization C(t) of the same curve such that
# B(a) = C(0) and B(b) = C(1).
# Args:
# - a: float;
# - b: float;
# - control_points: the list [p0,p1,p2,p3] of control points.
# Returns:
# - new control points [q0,q1,q2,q3]:[RowVector] such that the corresponding
#   Bezier curve C(t) is precisely the same as B as a curve
#   but B(a) = C(0) and B(b) = C(1).
# Notes:
# - This enables one to extend or to shorten the given Bezier arc.
# - There is an exception: If a=b, the returned control points
#   are equal, so the new curve is a single point, not the original
#   curve. The function gives no signal about such occurence.
def bezier_new_control_points(a, b, control_points):
    def bezier_2_rv(t, r0, r1, r2): # Bezier curve of degree two (complex)
        b20 = (1-t)**2
        b21 = 2*(1-t)*t
        b22 = t**2
        return b20*r0 + b21*r1 + b22*r2
    p0,p1,p2,p3 = control_points
    q0 = bezier_rv(a, control_points)
    q1 = (1-b)*bezier_2_rv(a, p0,p1,p2) + b*bezier_2_rv(a, p1,p2,p3)
    q2 = (1-a)*bezier_2_rv(b, p0,p1,p2) + a*bezier_2_rv(b, p1,p2,p3)
    q3 = bezier_rv(b, control_points)
    return [q0,q1,q2,q3]

# -------------------------
# bezier_arc_end_directions
# -------------------------
# Given a Bezier arc, find, at each end, the direction vector of the arc.
# Note: this does not mean the derivatives of the parametric representation,
# but non-zero vectors which are tangential to the arc and point in the
# direction where the arc is located when viewed from that end point.
# Return the result as two RowVectors (=complex),
# or as None if the arc is one point.
# Args:
########### - ba: BCurve.BezierArc
# - cp4: [complex,complex,complex,complex]
# Returns: either None (when the arc is one point), or
# - RowVector (direction vector at starting point)
# - RowVector (direction vector at ending point)
#def bezier_arc_end_directions(ba):
def bezier_arc_end_directions(cp4):
    ZERO = 1e-12
    #if len(ba.cp4) == 1:
    #    return None
    #p0,p1,p2,p3 = ba.cp4
    if len(cp4) == 1:
        return None
    p0,p1,p2,p3 = cp4
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    if vdot(p01,p01) > ZERO:
        head_dir = p01
    elif vdot(p12,p12) > ZERO:
        head_dir = p12
    elif vdot(p23,p23) > ZERO:
        head_dir = p23
    else:
        return None
    if vdot(p23,p23) > ZERO:
        tail_dir = -p23
    elif vdot(p12,p12) > ZERO:
        tail_dir = -p12
    elif vdot(p01,p01) > ZERO:
        tail_dir = -p01
    else:
        raise Exception("bezier_arc_end_directions: Should never happen!")
    return head_dir, tail_dir


# ------------------------
# gimp_draw_vectors_object
# ------------------------
# The Gimp version of the routine 'drawing_routine_for_bezier':
# Insert into Gimp the input vectors object, and make it visible if
# visible=True.
# Args:
# - image: Gimp's image (obtained usually from GUI);
# - vectors_object: a vectors object in Gimp;
# - visible: Boolean.
# Returns the vectors object in Gimp, and makes it visible if visible=True.
def gimp_draw_vectors_object(image, vectors_object, visible=True):
    # Insert a Gimp vectors object in Gimp.
    # Adapted from http://registry.gimp.org/files/lqr_wpset-0.14.py.txt
    # by Mike Kazantsev.
    def gimp_insert_vectors(image, vectors_object, position=0):
        if gimp.version >= (2, 8, 0):
            # 2.7.0 has gimp-image-insert-vectors, but fails with
            # parent=None
            parent = None
            return pdb.gimp_image_insert_vectors(
                     image, vectors_object, parent, position)
        else:
            return pdb.gimp_image_add_vectors(
                     image, vectors_object, position)# Deprecated!
    gimp_insert_vectors(image, vectors_object)
    vectors_object.visible = visible
    return vectors_object

# ----------------
# gimp_draw_circle
# ----------------
# center:[float,float]
# radius: float
def gimp_draw_circle(image, center, radius, name='circle'):
    kappa = 4.*(sqrt(2)-1)/3
    xy = [1, -kappa, 1, 0, 1, kappa,
          kappa, 1, 0, 1, -kappa, 1,
          -1, kappa, -1, 0, -1, -kappa,
          -kappa, -1, 0, -1, kappa, -1]
    for i in range(len(xy)):
        if i%2 == 0:
            xy[i] = center[0] + radius*xy[i]
        else:
            xy[i] = center[1] + radius*xy[i]
    circle = pdb.gimp_vectors_new(image, name)
    pdb.gimp_vectors_stroke_new_from_points(
                        circle, 0, len(xy), xy, True)
    gimp_draw_vectors_object(image, circle, visible=True)
    return circle

# ------------
# path_anchors
# ------------
# Get anchors of the path (optionally the first stroke only).
# Arg:
# - vectors_object:    gimp.Vectors
# - only_first_stroke: boolean
# Returns:
# - [BCurve.ControlPoint]
def path_anchors(vectors_object, only_first_stroke=False):
    gv = vectors_object2gv(vectors_object) # BCurve.GimpVectors
    if len(gv.stroke_list) == 0:
        raise Exception("Path without any strokes: "+vectors_object.name)
    anchors = []
    for gs in gv.stroke_list:
        bc = gs.gs2bc()                        # BCurve.BezierCurve
        bas = bc.bezier_arcs                   # [BCurve.BezierArc]
        #anchors = [ba.cp4[0] for ba in bas]
        anchors += [ba.cp4[0] for ba in bas]
        if len(bas[0].cp4) > 1: # Case len=1 may rise from stroke with one anchor.
            anchors.append(bas[-1].cp4[-1])
        if only_first_stroke:
            break
    return anchors

# ----------
# gv_anchors
# ----------
# Get anchors of gv:BCurve.GimpVectors
# (optionally the first stroke only).
# Arg:
# - gv:    BCurve.GimpVectors
# - only_first_stroke: boolean
# Returns:
# - [BCurve.ControlPoint]
def gv_anchors(gv, only_first_stroke=False):
    #gv = vectors_object2gv(vectors_object) # BCurve.GimpVectors
    if len(gv.stroke_list) == 0:
        raise Exception("Path without any strokes: "+vectors_object.name)
    anchors = []
    for gs in gv.stroke_list:
        bc = gs.gs2bc()                        # BCurve.BezierCurve
        bas = bc.bezier_arcs                   # [BCurve.BezierArc]
        #anchors = [ba.cp4[0] for ba in bas]
        anchors += [ba.cp4[0] for ba in bas]
        if len(bas[0].cp4) > 1: # Case len=1 may rise from stroke with one anchor.
            anchors.append(bas[-1].cp4[-1])
        if only_first_stroke:
            break
    return anchors

# ---------
# is_convex
# ---------
# Is a polygon convex? The polygon is inputted as
# a list of corner points (complex).
# Args:
# - corners: [complex]
# Returns:
# - boolean
# Notes:
# 1. Returns True if all corner points are collinear.
def is_convex(corners):
    if len(corners) <= 3:
        return True
    if collinear(corners): # TEE
        return True
    c = corners + corners[:2]
    edges = [c[i]-c[i+1] for i in range(len(c)-1)]
    edges = [e for e in edges if e != 0] # remove zero edges
    pairs = [[edges[i],edges[i+1]] for i in range(len(edges)-1)]
    # Cross and dot products:
    cr_dots = [[-a.real*b.imag + a.imag*b.real,
                 a.real*b.real + a.imag*b.imag] for a,b in pairs]
    crs = []
    for crd in cr_dots:
        if (crd[0]==0):
            if (crd[1]<0):
                return False # spike of angle 0
        else:
            crs.append(crd[0]) # collect non-zero cross products
    try:
        pos = (crs[0] > 0)
    except IndexError: # All points collinear
        return True
    for cr in crs:
        if pos != (cr > 0):
            return False
    return True

## v dot (w perpendicular)
def cross(v,w):
    return (v*w.conjugate()).imag

def vdot(v,w):
    return v.real*w.real + v.imag*w.imag

def collinear(points): # points: [complex] (The plane R2 is the complex plane.)
    ZERO = 1e-14
    n = len(points)
    if n < 3:
        return True
    for i in range(n):
        for j in range(i+1,n):
            for k in range(j+1,n):
                a,b,c = points[i], points[j], points[k]
                ab = -a+b
                ac = -a+c
                bc = -b+c
                if abs(cross(ab,ac)) > ZERO:
                    return False
                if abs(cross(ab,bc)) > ZERO:
                    return False
                if abs(cross(ac,bc)) > ZERO:
                    return False
    return True

# -------
# get_box
# -------
# Get box in the form [SW, SE, NE, NW] where items:complex and the directions
# are as on the screen (so, top is N, bottom is S).
# If Gimp=True, y coordinate runs downwards.
# Four cases: see crop_box_options.
# Args:
# - path_vectors: gimp.Vectors
# - base_vectors: gimp.Vectors
# - box_case:     string, one of the identifiers in crop_box_options
# - Gimp: boolean
# Returns:
# - [complex,complex,complex,complex]
def get_box(image, box_case, Gimp=True):
    if box_case == 'bb_selection':
        bounds = pdb.gimp_selection_bounds(image)
        if bounds[0] == 0:
            raise Exception("No active selection")
        W,N,E,S = bounds[1:]
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        return [sw,se,ne,nw]
    elif box_case == 'guides':
        hor,ver = get_guide_positions(image)
        if (len(hor) > 2) or (len(ver) > 2):
            raise Exception("Too many guides: must have two horizontal and two vertical.")
        if (len(hor) < 2) or (len(ver) < 2):
            raise Exception("Not enough guides: must have two horizontal and two vertical.")
        W,E = sorted(ver)
        if Gimp:
            N,S = sorted(hor)
        else:
            S,N = sorted(hor)
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        return [sw,se,ne,nw]
    elif box_case == 'image':
        h,w = image.height, image.width
        return [complex(0,h), complex(w,h), complex(w,0), complex(0,0)]
    elif box_case == 'layer':
        layer = pdb.gimp_image_get_active_layer(image) # Active layer
        h,w = pdb.gimp_drawable_height(layer), pdb.gimp_drawable_width(layer)
        offx, offy = pdb.gimp_drawable_offsets(layer)
        return [complex(offx,offy+h), complex(offx+w,offy+h),
                complex(offx+w,offy), complex(offx,offy)]
    else:
        raise Exception("get_box: invalid choice: "+str(box_case))

# ---------------------------
# bezier_intersect_horizontal
# ---------------------------
# Intersect Bezier arc with a horizontal line.
# Args:
# - control_points: [complex,complex,complex,complex]
# - p: complex (one point on the line)
# - interval: [float,float] (interval [t0,t1] for the parameter)
# Returns:
# - [float] (parameter values of intersection points)
def bezier_intersect_horizontal(control_points, p, interval):
    if not (interval is None):
        t0,t1 = interval
        if t0 > t1:
            t0,t1 = t1,t0
    p0,p1,p2,p3 = control_points # complex
    R0 = -p+p0
    R1 = -p0+p1
    R2 = p0-2*p1+p2
    R3 = -p0+3*p1-3*p2+p3
    coeff = [R0.imag, 3*R1.imag, 3*R2.imag, R3.imag]
    zeroes = zeroes_of_polynomial(coeff, interval=interval) # None or [float,float,float]
    if zeroes is None: # Zero polynomial!
        return []
    return zeroes[0]
    #coeff = [R3.imag, 3*R2.imag, 3*R1.imag, R0.imag]
    #return [t for t in solve_pol3_R(coeff) if t0 <= t <= t1]

# -------------------------
# bezier_intersect_vertical
# -------------------------
# Intersect Bezier arc with a  a vertical line.
# Args:
# - control_points: [complex,complex,complex,complex]
# - p: complex (one point on the line)
# - interval: [float,float] (interval [t0,t1] for the parameter)
# Returns:
# - [float] (parameter values of intersection points)
def bezier_intersect_vertical(control_points, p, interval):
    if not (interval is None):
        t0,t1 = interval
        if t0 > t1:
            t0,t1 = t1,t0
    p0,p1,p2,p3 = control_points # complex
    R0 = -p+p0
    R1 = -p0+p1
    R2 = p0-2*p1+p2
    R3 = -p0+3*p1-3*p2+p3
    coeff = [R0.real, 3*R1.real, 3*R2.real, R3.real]
    zeroes = zeroes_of_polynomial(coeff, interval=interval) # None or [float,float,float]
    if zeroes is None: # Zero polynomial!
        return []
    return zeroes[0]
    #coeff = [R3.real, 3*R2.real, 3*R1.real, R0.real]
    #return [t for t in solve_pol3_R(coeff) if t0 <= t <= t1]


# -------------------
# get_guide_positions
# -------------------
# Guide positions in image, horizontal and vertical in separate lists.
# Not sorted.
def get_guide_positions(image):
    horizontal = []
    vertical = []
    guide = 0
    while True:
        guide = pdb.gimp_image_find_next_guide(image,guide)
        if guide == 0:
            break
        position = pdb.gimp_image_get_guide_position(image, guide)
        orientation = pdb.gimp_image_get_guide_orientation(image, guide)
        if orientation == ORIENTATION_HORIZONTAL:
            horizontal.append(position)
        elif orientation == ORIENTATION_VERTICAL:
            vertical.append(position)
        else:
            raise Exception("get_guide_positions: ???")
    return horizontal, vertical

# --------------------------------
# intersect_horizontal_or_vertical
# --------------------------------
# Intersect a Bezier arc with a horizontal line or a vertical line.
# Args:
# - control_points: [complex,complex,complex,complex]
# - p: complex (one point on the line)
# - interval: [float,float] (interval [t0,t1] for the parameter)
# - horizontal: boolean
# Returns:
# - [float] (parameter values of intersection points)
def intersect_horizontal_or_vertical(control_points, p, interval, horizontal):
    t0,t1 = interval
    if t0 > t1:
        t0,t1 = t1,t0
    p0,p1,p2,p3 = control_points # complex
    R0 = -p+p0
    R1 = -p0+p1
    R2 = p0-2*p1+p2
    R3 = -p0+3*p1-3*p2+p3
    if horizontal:
        #coeff = [R3.imag, 3*R2.imag, 3*R1.imag, R0.imag]
        coeff = [R0.imag, 3*R1.imag, 3*R2.imag, R3.imag]
    else:
        #coeff = [R3.real, 3*R2.real, 3*R1.real, R0.real]
        coeff = [R0.real, 3*R1.real, 3*R2.real, R3.real]
    #return [t for t in solve_pol3_R(coeff) if t0 <= t <= t1]
    return zeroes_of_polynomial(coeff, interval=interval)[0]


#======================================================
#                   Bounding box
#======================================================

# -------------
# bezier_arc_BB
# -------------
# Bounding box of one Bezier arc, given as list [p0,p1,p2,p3]:[complex] of
# control points.
# Return as list [NW, SE] of two corner points,
# where NW is top left, SE is bottom right. (The directions are indeed as
# on the screen: N is up, S is down, provided that the argument 'Gimp' is correct!)
# Args:
# - cp4: [complex,complex,complex,complex]
# - Gimp: boolean
# Returns:
# - [NW, SE]: [complex,complex]
def bezier_arc_BB(cp4, Gimp=True):
    if len(cp4) == 1: # 1-point stroke
        return [cp4[0],cp4[0]]
    p0,p1,p2,p3 = cp4
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    p01x, p12x, p23x = p01.real, p12.real, p23.real
    p01y, p12y, p23y = p01.imag, p12.imag, p23.imag
    xts = solve_bernstein2_equation(p01x,p12x,p23x)
    yts = solve_bernstein2_equation(p01y,p12y,p23y)
    if not (xts is None):
        xts = [t for t in xts if 0<=t<=1]
        xts += [0.,1.]
        Bxs = [bezier_rv(t,cp4) for t in xts]
        Bxs.sort(key = lambda b:b.real)
        xmin, xmax = Bxs[0].real, Bxs[-1].real
    else: # Curve on vertical line
        xmin = xmax = p0.real
    if not (yts is None):
        yts = [t for t in yts if 0<=t<=1]
        yts += [0.,1.]
        Bys = [bezier_rv(t,cp4) for t in yts]
        Bys.sort(key = lambda b:b.imag)
        ymin, ymax = Bys[0].imag, Bys[-1].imag
    else: # Curve on horizontal line
        ymin = ymax = p0.imag
    if Gimp:
        return [complex(xmin,ymin), complex(xmax,ymax)]
    else:
        return [complex(xmin,ymax), complex(xmax,ymin)]

# -------------------------
# solve_bernstein2_equation
# -------------------------
# Find real zeroes of the equation c0*b0(t) + c1*b1(t) + c2*b2(t) = 0 where
# b0,b1,b2 are the Bernstein polynomials of degree 2 and
# c0,c1,c2 are real coefficients.
# Return the zeroes as a list (0, 1, or 2 items).
# Exception: If the function is constant 0, return None.
# Args:
# - c0,c1,c2: float
# Returns:
# - None or list of floats (possibly empty)
def solve_bernstein2_equation(c0,c1,c2):
    from math import sqrt
    ZERO = 1e-14
    if abs(c0 - 2.*c1 + c2) < ZERO: # equation of degree <= 1
        try:
            return [c0 / (2.*(c0 - c1))]
        except ZeroDivisionError: # equation of degree 0
            if abs(c0) < ZERO: # constant 0
                return None
            else: # constant not 0
                return []
    try:
        root = sqrt(c1**2 - c0*c2)
    except ValueError: # No real roots
        return []
    return [(c0 - c1 + root) / (c0 - 2.*c1 + c2),
            (c0 - c1 - root) / (c0 - 2.*c1 + c2)]


#======================================================
#           Intersections with integer lattice
#======================================================

# -------------------
# arc_Z_intersections
# -------------------
# Given a Bezier arc B(t), 0<=t<=1, given as control points cp4,
# find intersection points with the integer lattice ZxZ.
# Restrict the solutions to box given as (x1,y1,x2,y2) where (x1,y1) is the
# top left corner and (x2,y2) the bottom right corner.
# Return as sorted list of parameter values t, without duplicates, together with
# values B(t).
# Args:
# - cp4: [complex,complex,complex,complex]
# - box: (int,int,int,int)
# Returns:
# - [[float, complex]] ([[t,B(t)]]
def arc_Z_intersections(cp4, box, padding=1):
    from math import floor
    ZERO = 1e-10
    # Work only in the bounding box of the arc:
    nw,se = bezier_arc_BB(cp4)
    xmin = int(floor(nw.real))
    xmax = int(floor(se.real)) + 1
    ymin = int(floor(nw.imag))
    ymax = int(floor(se.imag)) + 1
    # Preliminary restriction to the box:
    x1,y1,x2,y2 = box
    xmin = max(xmin,x1) - padding
    xmax = min(xmax,x2) + padding
    ymin = max(ymin,y1) - padding
    ymax = min(ymax,y2) + padding
    tx = []
    for x in range(xmin, xmax+1):
        tx += bezier_intersect_vertical(cp4, complex(x,0), interval=[0.,1.])
    ty = []
    for y in range(ymin, ymax+1):
        ty += bezier_intersect_horizontal(cp4, complex(0,y), interval=[0.,1.])
    s = sorted(tx+ty)
    # Remove duplicates:
    try:
        ts = [s[0]]
    except IndexError:
        return []
    prev = s[0]
    for t in s[1:]:
        if t != prev:
            ts.append(t)
            prev = t
    # Final restriction to the box, computing the B(t)'s:
    t_Bs = []
    for t in ts:
        Bt = bezier_rv(t,cp4)
        if (xmin-ZERO < Bt.real < xmax+ZERO) and (ymin-ZERO < Bt.imag < ymax+ZERO):
            t_Bs.append([t,Bt])
    return t_Bs

# ------------------
# pixel_in_selection
# ------------------
# Args:
# - x:         float
# - y:         float
# - rgn:       gimp.PixelRgn
# - threshold: float
# Returns:
# - boolean
def pixel_in_selection(x, y, rgn, threshold):
    try:
        return ord(rgn[(int(floor(x)), int(floor(y)))]) >= threshold
    except IndexError:
        return False


# -----------------------
# arc_intersect_selection
# -----------------------
# Given a Bezier arc B(t), 0<=t<=1, given as control points cp4,
# find intersection points of the arc with the boundary of the selection.
# Return as a sorted list of parameter values t with 0<t<1.
# Args:
# - cp4:         [complex,complex,complex,complex]
# - crop_inputs: CroppingInputData
# Returns:
# - [float]
def arc_intersect_selection(cp4, crop_inputs):
    from math import floor
    ZERO = 1e-10
    #non_empty, box_x1, box_y1, box_x2, box_y2 = pdb.gimp_selection_bounds(image)
    box_x1, box_y1, box_x2, box_y2 = crop_inputs.selection_bounds
    Z_intersections = arc_Z_intersections(cp4, (box_x1,box_y1,box_x2,box_y2))
    if len(Z_intersections) == 0:
        return []
    ts = []
    Z_intersections = [[0, cp4[0]]] + Z_intersections + [[1, cp4[-1]]]
    tB0, tB1 = Z_intersections[:2]
    t0, B0 = tB0
    t1, B1 = tB1
    t01 = (t0+t1)/2
    B01 = bezier_rv(t01, cp4)
    x01, y01 = int(floor(B01.real)), int(floor(B01.imag))
    test01 = pixel_in_selection(x01, y01, crop_inputs.rgn, crop_inputs.threshold)
    for i in range(1, len(Z_intersections)-1):
        t2, B2 = Z_intersections[i+1]
        t12 = (t1+t2)/2
        B12 = bezier_rv(t12, cp4)
        x12, y12 = int(floor(B12.real)), int(floor(B12.imag))
        test12 = pixel_in_selection(x12, y12, crop_inputs.rgn, crop_inputs.threshold)
        if test01 != test12:
            ts.append(t1)
        # update for next round
        t1 = t2
        test01 = test12
    ts = [t for t in ts if 0 < t < 1]
    return ts



#======================================================
#              Polynomial zeroes: degree 3
#======================================================
# ------------
# solve_pol3_C
# ------------
# Solving complex roots of a real cubic polynomial.
# The polynomial is ax^3 + bx^2 + cx + d.
# Option to return only real roots (better to call solve_pol3_R).
# Args:
# - coefficients: [float or integer] (=[a,b,c,d])
# - only_real:    boolean
# Returns: None if the polynomial is identically zero, otherwise:
# - [float] (the roots [x1,x2,x3]).
# Note:
# The main code copied and heavily adapted from:
# https://github.com/shril/CubicEquationSolver
#
def solve_pol3_C(coefficients, only_real=False):
    from math import sqrt,sin,cos, acos
    a,b,c,d = [float(t) for t in coefficients]
    if (a == 0 and b == 0):                     # Linear Equation
        try:
            return [-d / c]
        except ZeroDivisionError:
            if d == 0:
                return None                     # zero polynomial
            else:
                return []                       # non-zero constant polynomial
    elif (a == 0):                              # Quadratic Equation
        D = c * c - 4.0 * b * d
        if D >= 0:                              # two real roots
            D = sqrt(D)
            x1 = (-c + D) / (2.0 * b)
            x2 = (-c - D) / (2.0 * b)
            return [x1, x2]
        elif not only_real:                     # return two complex roots
            D = sqrt(-D)
            x1 = (-c + D * 1j) / (2.0 * b)
            x2 = (-c - D * 1j) / (2.0 * b)
            return [x1, x2]
        else: # D<0, only_real                  # no real roots, return nothing
            return []
    f = ((3.0 * c / a) - ((b ** 2.0) / (a ** 2.0))) / 3.0
    g = ((((2.0 * (b ** 3.0)) / (a ** 3.0)) - ((9.0 * b * c) / (a **2.0))
          + (27.0 * d / a)) / 27.0)
    h = (g ** 2.0) / 4.0 + (f ** 3.0) / 27.0
    #
    if f == 0 and g == 0 and h == 0:            # All 3 Roots are Real and Equal
        if (d / a) >= 0: # Is this trick necessary?
            x = -((d / a) ** (1 / 3.0))
        else:
            x = (-d / a) ** (1 / 3.0)
        return [x, x, x]
    elif h <= 0:                                # All 3 roots are Real
        i = sqrt(((g ** 2.0) / 4.0) - h)
        j = i ** (1 / 3.0)
        k = acos(-(g / (2 * i)))
        L = -j
        M = cos(k / 3.0)
        N = sqrt(3) * sin(k / 3.0)
        P = -(b / (3.0 * a))
        x1 = 2 * j * cos(k / 3.0) - (b / (3.0 * a))
        x2 = L * (M + N) + P
        x3 = L * (M - N) + P
        return [x1, x2, x3]
    elif h > 0:                                 # One Real Root and two Complex Roots
        R = -(g / 2.0) + sqrt(h)
        if R >= 0: # Is this trick necessary?
            S = R ** (1 / 3.0)
        else:
            S = -((-R) ** (1 / 3.0))
        T = -(g / 2.0) - sqrt(h)
        if T >= 0: # Is this trick necessary?
            U = T ** (1 / 3.0)
        else:
            U = -((-T) ** (1 / 3.0))
        x1 = (S + U) - (b / (3.0 * a))
        x2 = -(S + U) / 2 - (b / (3.0 * a)) + (S - U) * sqrt(3) * 0.5j
        x3 = -(S + U) / 2 - (b / (3.0 * a)) - (S - U) * sqrt(3) * 0.5j
        if only_real:
            return [x1]
        else:
            return [x1, x2, x3]

# ------------
# solve_pol3_R
# ------------
# Solving real roots of a real cubic polynomial.
# The polynomial is ax^3 + bx^2 + cx + d.
# Args:
# - coefficients: [float or integer] (=[a,b,c,d])
# Returns: None if the polynomial is identically zero, otherwise:
# - [float] (the roots [], or [x1], or [x1,x2], or [x1,x2,x3], in increasing order,
#            multiple roots repeated).
def solve_pol3_R(coefficients):
    def improve(coefficients,x0):       # Try to improve a real root x0
        accuracy = 1e-20
        a,b,c,d = [float(t) for t in coefficients]
        for i in range(20):         # max 20 rounds
            f0 = a*(x0**3) + b*(x0**2) + c*x0 + d     # f(x0)
            f0dot = 3*a*(x0**2) + 2*b*x0 + c          # f'(x0)
            try:
                x1 = x0 - f0 / f0dot                  # Newton
                f1 = a*(x1**3) + b*(x1**2) + c*x1 + d # f(x1)
                xm = (f1*x0 - f0*x1) / (f1-f0)        # interpolate
            except ZeroDivisionError:
                return x0
            fm = a*(xm**3) + b*(xm**2) + c*xm + d     # f(xm)
            x = min([[x0,f0],[x1,f1],[xm,fm]], key = (lambda p:abs(p[1])))[0]
            if -accuracy < x-x0 < accuracy:
                break
            x0 = x
        return x
    s = solve_pol3_C(coefficients, only_real=True)
    if s is None:
        return None
    else:
        return sorted([improve(coefficients,x0) for x0 in s])

# ------------------------------
# zeroes_of_bernstein_polynomial
# ------------------------------
# Given a polynomial of degree 3, in Bernstein basis,
# find the real zeroes, the x values of extrema, and the x values of the
# inflection points.
# The polynomial is c0*b0(t) + c1*b1(t) + c2*b2(t) + c3*b3(t)
# where b0,...,b3 are the Bernstein polynomials of degree 3;
# it is inputted as list of coefficients [c0,c1,c2,c3].
# Args:
# - coefficients: [float or integer] (must be 4 coefficients).
# Returns: Either None (in case of zero input polynomial), or
# - [float] (the list of real zeroes of the polynomial, in increasing order,
#            multiple roots repeated);
def zeroes_of_bernstein_polynomial(coefficients):
    # Convert a polynomial given in the Bernstein basis {b0,b1,b2,b3}:
    # f(t) = c0*b0(t) + c1*b1(t) + c2*b2(t) + c3*b3(t)
    # to the power basis: f(t) = a0 + a1*t + a2*(t**2) + a3*(t**3).
    # Args:
    # - coefficients: [c0,c1,c2,c3] (ci:float)
    # Returns:
    # - [a0,a1,a2,a3] (ai:float)
    def bernstein2power(coefficients):
        c0,c1,c2,c3 = [float(c) for c in coefficients]
        cb0 = [c0*b for b in [1,-3, 3,-1]]
        cb1 = [c1*b for b in [0, 3,-6, 3]]
        cb2 = [c2*b for b in [0, 0, 3,-3]]
        cb3 = [c3*b for b in [0, 0, 0, 1]]
        result = []
        for i in range(4):
            result.append(cb0[i] + cb1[i] + cb2[i] + cb3[i])
        return result
    if len(coefficients) != 4:
        s = "zeroes_of_bernstein_polynomial: "
        s += "wrong number of coefficients"
        raise Exception(s)
    power_basis_coefficients = bernstein2power(coefficients)
    power_basis_coefficients.reverse() # To get correct order
    return solve_pol3_R(power_basis_coefficients)


#======================================================
#             Polynomial zeroes: any degree
#======================================================
# My own algorithm, written for fun.
# Certainly could find faster elsewhere.
# --------------------
# zeroes_of_polynomial
# --------------------
# Given a real polynomial, find the real zeroes, the x values of extrema,
# and the x values of the inflection points. May restrict to an interval.
#
# Args:
# - coefficients: list [c0,c1,...,cn] of the coefficients of the
#                     polynomial c0 + c1*x + ... + cn*(x**n);
#                     the ci are integer or float.
# - interval:     None or [float,float]
# Returns: Either None (in case of zero input polynomial), or
# - [float] (the list of real zeroes of the polynomial, in increasing order,
#            multiple roots repeated);
# - [float] (the extrema (the x values));
# - [float] (the inflection points (the x values)).
# Notes:
# 1. Return None in the case of a zero polynomial.
# 2. Return None if interval is given (not None) with interval=[a,b] where a>b.
# 3. If interval=[a,a] is given, if a is a root then [a],[],[] is returned,
#    otherwise [],[],[]. So, no effort is made to give any extrema or inflection
#    points.
# 3. Otherwise:
#    - the list of real zeroes of the polynomial, in increasing order
#      (possibly []) with multiple roots repeated;
#    - the extrema (the x values);
#    - the inflection points (the x values).
# 4. Because of the chosen accuracy and tolerance in the code, choosing different
#    interval may change the returned values slightly.
def zeroes_of_polynomial(coefficients, interval=None):
    from math import sqrt
    from functools import partial
    def f(x, coeff): # Horner
        s = 0.
        for ci in coeff[::-1]:
            s = ci + x*s
        return s
    if not (interval is None):
        LO,HI = interval
        if LO > HI:
            return None
        if LO == HI:
            if f(LO) == 0:
                return [LO],[],[]
            else:
                return [],[],[]
    coefficients = [float(ci) for ci in coefficients]
    # Find largest n with coefficients[n] non-zero and
    # discard all higher terms:
    n = -1
    for i in range(len(coefficients)):
        if coefficients[i] != 0.:
            n = i
    if n < 0: # Polynomial is identically zero.
        return None
    c = coefficients[:n+1] 
    # Make c[n] positive:
    if c[n] < 0:
        c = [-ci for ci in c]
    if n <= 2:
        if n == 0: # Polynomial is a non-zero constant.
            #return [],[],[]
            results = [[],[],[]]
        if n == 1: # Polynomial is c[0] + c[1]*x
            #return [-c[0] / c[1]], [], []
            results = [[-c[0] / c[1]], [], []]
        if n == 2: # Polynomial is c[0] + c[1]*x + c[2]*(x**2).
            discr = c[1]**2 - 4*c[0]*c[2]
            try:
                root = sqrt(discr)
                x1 = (-c[1] - root) / (2*c[2])
                x2 = (-c[1] + root) / (2*c[2])
                #return [x1,x2], [-c[1]/(2*c[2])], []
                results = [[x1,x2], [-c[1]/(2*c[2])], []]
            except ValueError:
                #return [], [-c[1]/(2*c[2])], []
                results = [[], [-c[1]/(2*c[2])], []]
        if interval is None:
            return results
        else:
            results0 = [t for t in results[0] if LO <= t <= HI]
            results1 = [t for t in results[1] if LO <= t <= HI]
            results2 = [t for t in results[2] if LO <= t <= HI]
            return results0, results1, results2
    # Build subdivision such that in each subinterval
    # the polynomial is monotonous:
    derivative = [e*i for e,i in enumerate(c)][1:]
    extrema, inflection, _ = zeroes_of_polynomial(derivative, interval)
    if interval is None:
        x_far = sum([abs(ci/c[-1]) for ci in c])
        subdiv = [-x_far] + extrema + [x_far]
    else:
        subdiv = [LO] + [t for t in extrema if LO < t < HI] + [HI]
    # Solve and collect the zeroes of each monotonous segment:
    fc = partial(f, coeff=c)
    zeroes = []
    for i in range(len(subdiv)-1):
        lo, hi = subdiv[i], subdiv[i+1]
        x = zero_of_monotonous_function(fc, lo, hi,
                                    accuracy=1e-10, tolerance=1e-10)
        if not(x is None):
            zeroes.append(x)
    return zeroes, extrema, inflection

# ---------------------------
# zero_of_monotonous_function
# ---------------------------
# Find the zero, if any, of a monotonous function f(x)
# in the interval [lo,hi]. If none is found, None is returned.
# Args:
# - f:         callable (real function, assumed to be monotonous in [lo,hi];
# - lo, hi:    float; (the interval)
# - accuracy:  accuracy for x.
# - tolerance: float>=0; if no zeroes exist in the interval, then
#              if f(lo) or f(hi) is less than 'tolerance' from 0,
#              then lo or hi is returned, respectively.
# Returns: None or the unique zero.
def zero_of_monotonous_function(f, lo, hi, accuracy=1e-10, tolerance=1e-10):
    MAX_ROUNDS = 100
    lo, hi = float(lo), float(hi)
    flo, fhi = float(f(lo)), float(f(hi))
    if (flo > 0.) == (fhi > 0.):
        if abs(flo) <= tolerance:
            return lo
        elif abs(fhi) <= tolerance:
            return hi
        else:
            return None
    if flo == 0.:
        return lo
    if fhi == 0.:
        return hi
    count = 0
    while hi-lo > accuracy:
        mid = (lo+hi)/2.
        fmid = float(f(mid))
        if fmid == 0.:
            return mid
        if (flo > 0.) == (fmid > 0.):
            lo, flo = mid, fmid
        else:
            hi, fhi = mid, fmid
        count += 1
        if count > MAX_ROUNDS:
            break
    x = (fhi*lo - flo*hi) / (fhi-flo)
    return x


#======================================================
#               Splitting a Bezier curve
#======================================================
# Routines for splitting a Bezier curve, given list of
# splitting points in the form [[n,t]] where
# n = number of a Bezier arc along the curve,
# t = parameter value (where splitting should be done).

class SplitBezierArc(object):
    """
    Attributes:
    - cp4:    [complex]
    - splits: [float] (sorted)
    """
    def __init__(self, cp4, splits):
        self.cp4 = cp4
        self.splits = splits
    def __str__(self):
        s = 'cp4:    '+str(self.cp4)
        s += '\nsplits: '+str(self.splits)
        return s

# ----------------------
# bcurve_with_split_data
# ----------------------
# Incorporate list of splitting points into a Bezier curve:
# Given a Bezier curve and a list of splits points [[n,t]]
# (n=arc number, t=parameter value),
# construct list [SplitBezierArc] - list of butting Bezier arcs,
# each of them equipped with a sorted list of split points for that arc.
# Args:
# - bcurve: BCurve.BezierCurve
# - split_list: [[integer, float]]
# Returns:
# - [SplitBezierArc]
def bcurve_with_split_data(bcurve, split_list=[]):
    lsba = [] # Resulting [SplitBezierArc] is build to this
    arc_no = -1
    for ba in bcurve.bezier_arcs:
        arc_no += 1
        splits = [t for n,t in split_list if n == arc_no]
        splits.sort()
        lsba.append(SplitBezierArc(cp4=ba.cp4, splits=splits))
    return lsba

# -----------
# split_at_01
# -----------
# Given a list [SplitBezierArc] (butting Bezier arcs with split data),
# do splittings at split points t=0 and t=1 if any.
# Return as list [[SplitBezierArc]], with all split points t=0 and t=1 removed.
# Args:
# - list_sba:[SplitBezierArc]
# Returns:
# - [[SplitBezierArc]]
def split_at_01(list_sba):
    result_llsba = [] # Final result: [[SplitBezierArc]]
    cum_lsba = []     # Cumulate current under work: [SplitBezierArc]
    for sba in list_sba:
        if len(sba.splits) == 0:
            cum_lsba.append(sba)
            continue
        split0 = (sba.splits[0] == 0) # split at t=0?
        split1 = (sba.splits[-1] == 1) # split at t=1?
        if (not split0) and (not split1):
            cum_lsba.append(sba)
            continue
        if split0 and (not split1):
            result_llsba.append(cum_lsba) # end cumulation
            sba.splits = sba.splits[1:]   # remove t=0
            cum_lsba = [sba]              # start new cumulation
        if (not split0) and split1:
            sba.splits = sba.splits[:-1]  # remove t=1
            cum_lsba.append(sba)
            result_llsba.append(cum_lsba) # end cumulation
            cum_lsba = []                 # start new cumulation
        if split0 and split1:
            result_llsba.append(cum_lsba) # end cumulation
            sba.splits = sba.splits[1:-1] # remove t=0, t=1
            result_llsba.append([sba])
            cum_lsba = []                 # start new cumulation
    result_llsba.append(cum_lsba) # The last section
    # Remove empty lists from results:
    return [lsba for lsba in result_llsba if len(lsba)>0]

# ---------------
# split_at_others
# ---------------
# Given a list [SplitBezierArc] (butting Bezier arcs with split data),
# assuming no split points are t=0 or t=1, do splittings and convert the result
# to list of Bezier curves:[BCurve.BezierCurve].
# Args:
# - list_sba: [SplitBezierArc]
# Returns:
# - [BCurve.BezierCurve]
def split_at_others(list_sba):
    result_lbc = [] # Final result: [BCurve.BezierCurve]
    cum_lba = []    # Cumulate current under work: [BCurve.BezierArc]
    for sba in list_sba:
        if len(sba.splits) == 0:
            cum_lba.append(BCurve.BezierArc(cp4 = sba.cp4))
            continue
        ori_cp4 = sba.cp4
        splits  = sba.splits
        cp4_first = bezier_new_control_points(0, splits[0], ori_cp4)
        cp4_last  = bezier_new_control_points(splits[-1], 1, ori_cp4)
        cp4_others = [bezier_new_control_points(splits[i], splits[i+1], ori_cp4)
                        for i in range(len(splits)-1)]
        ba_first = BCurve.BezierArc(cp4 = cp4_first)
        ba_last  = BCurve.BezierArc(cp4 = cp4_last)
        ba_others = [BCurve.BezierArc(cp4 = cp4) for cp4 in cp4_others]
        bc_first = BCurve.BezierCurve(bezier_arcs = cum_lba + [ba_first])
        bc_others = [BCurve.BezierCurve(bezier_arcs = [ba]) for ba in ba_others]
        result_lbc.append(bc_first)
        result_lbc += bc_others
        cum_lba = [ba_last]
    bc_last = BCurve.BezierCurve(bezier_arcs = cum_lba)
    result_lbc.append(bc_last)
    return result_lbc

# ------------------
# split_bezier_curve
# ------------------
# Given Bezier curve and list of splitting points, do the splitting.
# The Bezier arcs in the curve are thought numbered 0,1,2,....
# The splittings are in the form [[n,t]]: [[integer, float]] where
# n is the number of an arc, and t is a splitting point (parameter value)
# for that arc.
# Return as list of Bezier curves.
# Args:
# - bcurve: BCurve.BezierCurve
# - split_list: [[integer, float]]
# Returns:
# - [BCurve.BezierCurve]
def split_bezier_curve(bcurve, split_list):
    list_sba = bcurve_with_split_data(bcurve, split_list) # [SplitBezierArc]
    done_01 = split_at_01(list_sba)                       # [[SplitBezierArc]]
    result = []                                           # [BCurve.BezierCurve]
    for lsba in done_01:
        result += split_at_others(lsba)                   # [BCurve.BezierCurve]
    return result

#======================================================
#        Splitting points for splits by a line
#======================================================
# Given a Bezier curve and a line, create a list of splitting points.

# -------------------
# barc_intersect_line
# -------------------
# Compute the intersection points (parameter values) of a Bezier arc with
# a straight line. Accept values outside [0,1], so actually we are finding
# intersections for the whole infinite Bezier curve.
# The plane is viewed as the complex plane.
# Input is the control points cp4=[p0,p1,p2,p3] and two complex numbers z0,z1.
# The line is the line through z0 and z1.
# Args:
# - cp4:    [p0,p1,p2,p3]: [complex]
# - z0, z1: complex
# Returns: None if the curve coincides with the line, otherwise:
# - [float]
def barc_intersect_line(cp4, z0, z1):
    # Find the coefficients a,b,c of the line ax+by+c=0 through z0 and z1:
    A,B = z0.real, z0.imag
    C,D = z1.real, z1.imag
    if A==C and B==D:
        raise Exception("barc_intersect_line: input two equal points.")
    a,b,c = -B+D, A-C, -A*D + B*C
    # Real coordinates [x,y] of p0,...,p3:
    x = [pk.real for pk in cp4]
    y = [pk.imag for pk in cp4]
    # Find the coefficients c0,...,c3 in the intersection equation
    #     c0*b0(t) + c1*b1(t) + c2*b2(t) + c3*b3(t) = 0,
    # expressed in terms of the Bernstein polynomials b0,...,b3.
    try:
        coefficients = [a*x[k] + b*y[k] + c for k in [0,1,2,3]]
    except IndexError: # cp4 is one point
        p = cp4[0]
        x,y = p.real, p.imag
        if a*x+b*y+c == 0:
            return None
        else:
            return []
    zs = zeroes_of_bernstein_polynomial(coefficients)
    # Mark intersections (later split points):
    # - single root => intersection
    # - double root => no intersection
    # - triple root => one intersection
    if zs is None:
        return None
    elif len(zs) <= 1: # 0 or 1
        return zs
    elif len(zs) == 2:
        if zs[0] == zs[1]:
            return [] # double root => no intersection
        else:
            return zs # two intersections
    else: # len(zs)=3
        zs = sorted(zs) # Should be sorted anyhow
        if zs[0] == zs[1] == zs[2]:
            return [zs[0]] # triple root => one intersection
        elif zs[0] == zs[1]:
            return [zs[2]] # no intersection from double root, one from single root
        elif zs[1] == zs[2]:
            return [zs[0]] # no intersection from double root, one from single root
        else: # three distinct roots => three intersections
            return zs
        return zs

# ---------------------
# bcurve_intersect_line
# ---------------------
# Given a Bezier curve and a straight line (through points z0,z1),
# find intersections.
# Return results as list of split points:[[n,t]]
# where n is arc number and the t is a parameter value.
# The path may contain "flats": one or more successive straight line segments
# contained in the intersecting line.
# 1. The default behaviour (isolated_flats=True) is to put split points
#    at the ends of each "flat", so that "flats" become isolated.
# 2. But if isolated_flats=False and zref (reference point) is given,
#    the "flats" are viewed as parts of the path on the same side
#    as the reference point, so some split points are not put.
#    More precisely:
#    - If a "flat" is connected at one end with an arc heading to the
#      same side of the line as zref, then the "flat" is connected also in
#      the result at that particular end (no split point generated there).
#    - But if a "flat" is connected at one end with an arc heading to the
#      opposite side of the line from zref, then the "flat" is not connected
#      in the result at that particular end (a split point is generated).
#    - Consequently, the "flat" may become isolated: split point or curve end
#      at each end.
# 3. But if zref is not given, isolated_flats=True is forced.
# Args:
# - bcurve:         BCurve.BezierCurve
# - z0,z1, zref:    complex
# - isolated_flats: boolean
# Returns:
# - [[integer, float]]
def bcurve_intersect_line(bcurve, z0, z1, zref=None, isolated_flats=True):
    ZERO = 1e-8
    AT_END = 1e-3
    def check_side(p, a,b,c): # On which side of the line point p:complex is?
        d = a*p.real + b*p.imag + c
        if   d > ZERO:  return +1 # First point determines
        elif d < -ZERO: return -1
        return 0
    if zref is None: # Cannot separate flats without a reference point, so force:
        isolated_flats = False
    elif not isolated_flats:
        # Find the coefficients a,b,c of the line ax+by+c=0 through z0 and z1:
        A,B = z0.real, z0.imag
        C,D = z1.real, z1.imag
        if A==C and B==D:
            raise Exception("bcurve_intersect_line: input two equal points for the line.")
        a,b,c = -B+D, A-C, -A*D + B*C
        norm = max(abs(a),abs(b),abs(c))
        a,b,c = a/norm, b/norm, c/norm
        # On which side of the line the reference point is?
        ref_value = a*zref.real + b*zref.imag + c
        if ref_value > ZERO:
            ref = +1
        elif ref_value < -ZERO:
            ref = -1
        else:
            raise Exception("bcurve_intersect_line: reference point too close to the line")
    result = [] # [[n,t]]
    arc_count = 0
    arcs_on_line = [] # collect numbers of arcs on the intersecting line
    for ba in bcurve.bezier_arcs:
        bil = barc_intersect_line(ba.cp4, z0, z1) # sorted [float] or None
        if bil is None: # ba lies on the intersecting line: endpoints to result
            result.append([arc_count, 0.])
            result.append([arc_count, 1.])
            arcs_on_line.append(arc_count)
            arc_count += 1
            continue
        bil = [t for t in bil if 0<=t<=1]
        for t in bil:
            result.append([arc_count, t])
        arc_count += 1
    result.sort(key=(lambda x:x[0]))
    # Check that there are no split points too close (AT_END) to the start or end
    # of the whole bcurve.
    # For closed bcurve allow the last point as a splitting point, however:
    if result != []:
        first = result[0]
        if first[0] == 0 and first[1] < AT_END:
            result = result[1:]
    if result != []:
        if not bcurve.closed:
            last = result[-1]
            if ((last[0] == len(bcurve.bezier_arcs)-1) # index of the last arc
              and (last[1] > 1 - AT_END)):
                result = result[:-1]
    # Follows some prunings.
    # To get the prunings go correctly, in closed case
    # append the first item of the result to the tail. This is again removed
    # at the end. (Hopefully this is right!)
    if result != []:
        if bcurve.closed:
            result.append(result[0])
    # Pruning 1
    # If an anchor is precisely on the line, the intersection point will be in
    # two arcs. Remove such duplicates:
    pruned1 = []
    check_next = False
    for n,t in result:
        if check_next and (t < ZERO): # Skip without saving this t...
            pruned1[-1][1] = 1. # ...but change the latest saved t to 1
            check_next = False
            continue
        pruned1.append([n,t]) # save
        check_next = (t > 1-ZERO)
    # Pruning 2
    # If two or more successive arcs are on the intersecting line,
    # remove the split points in between: only the last end point remains.
    pruned2 = []
    for n,t in pruned1:
        if (n in arcs_on_line) and (n+1 in arcs_on_line):
            continue                # skip without saving
        pruned2.append([n,t])       # save
    # If not isolated_flats, remove split points between flats and arcs which are
    # heading to the same side of the line where the reference point is:
    if isolated_flats:
        final = pruned2
    else:
        # Pruning 3
        pruned3 = []
        for n,t in pruned2:
            if n+1 in arcs_on_line:
                ba = bcurve.bezier_arcs[n]
                end_point_dirs = bezier_arc_end_directions(ba)
                if end_point_dirs is None:
                    pruned3.append([n,t])   # save
                    continue
                end_dir = end_point_dirs[1]
                p3 = ba.cp4[-1]
                if check_side(p3 + end_dir, a,b,c) == ref:
                    continue                # same side: skip without saving
            elif n in arcs_on_line:
                try:
                    ba = bcurve.bezier_arcs[n+1]
                except IndexError:
                    pruned3.append([n,t])   # save
                    continue
                end_point_dirs = bezier_arc_end_directions(ba)
                if end_point_dirs is None:
                    pruned3.append([n,t])   # save
                    continue
                start_dir = end_point_dirs[0]
                p0 = ba.cp4[0]
                if check_side(p3 + start_dir, a,b,c) == ref:
                    continue                # same side: skip without saving
            pruned3.append([n,t])           # save
        final = pruned3
    if final != []:
        if bcurve.closed: # Remove the last item from the tail.
            final = final[:-1]
    return final

#======================================================
#                 Split path by a line
#======================================================

# ---------------------
# group_bcurves_by_line
# ---------------------
# Given a list of Bezier curves, a line L, and a reference point P,
# where it is assumed that none of the Bezier curves crosses L
# (for instance, splitting step is done),
# group the Bezier curves according to if they are on the same side of L as P,
# on the opposite side, or flat on L.
# The line L is given by two points z0,z1.
# Args:
# - lbcurve:         [BCurve.BezierCurve]
# - z0, z1:          complex
# - reference_point: complex
# Returns:
# - near_side:    [BCurve.BezierCurve]
# - far_side:     [BCurve.BezierCurve]
# - flat_on_line: [BCurve.BezierCurve]
def group_bcurves_by_line(lbcurve, z0, z1, reference_point):
    ZERO = 1e-10
    # Find the coefficients a,b,c of the line ax+by+c=0 through z0 and z1:
    A,B = z0.real, z0.imag
    C,D = z1.real, z1.imag
    if A==C and B==D:
        raise Exception("group_bcurves_by_line: input two equal points for the line.")
    a,b,c = -B+D, A-C, -A*D + B*C
    norm = max(abs(a),abs(b),abs(c))
    a,b,c = a/norm, b/norm, c/norm
    # On which side of the line the reference point is?
    ref_value = a*reference_point.real + b*reference_point.imag + c
    if ref_value > ZERO:
        ref = +1
    elif ref_value < -ZERO:
        ref = -1
    else:
        raise Exception("group_bcurves_by_line: reference point too close to the line")
    # On which side of the line bcurve is?
    def check_bcurve(bcurve, a,b,c):
        for barc in bcurve.bezier_arcs:
            if len(barc.cp4) == 1:
                ps = barc.cp4
            else:
                p0,p1,p2,p3 = barc.cp4
                ps = [p0,p3,p1,p2]
            for p in ps:
                d = a*p.real + b*p.imag + c
                if   d > ZERO:  return +1 # First point determines
                elif d < -ZERO: return -1
        return 0
    near_side = []
    far_side = []
    flat_on_line = []
    for bcurve in lbcurve:
        check = check_bcurve(bcurve, a,b,c)
        if check == ref:
            near_side.append(bcurve)
        elif check == -ref:
            far_side.append(bcurve)
        else:
            flat_on_line.append(bcurve)
    return near_side, far_side, flat_on_line

# ----------------
# split_gv_by_line
# ----------------
# Given gv:BCurve.GimpVectors and a line (as a pair of points),
# split gv to 3 such objects (or None) by intersecting with the line.
# 1. Behaviour when isolated_flats=True:
#    If reference_point is given, the 3 returned paths are ordered as
#    - path on the same side of the line as the reference point;
#    - path on the opposite side of the line from the reference point;
#    - path lying flat on the line;
#    otherwise some default reference point is used and the ordering is ambigue.
# 2. Behaviour when isolated_flats=False:
#    Any part which lies flat on the line is treated as follows:
#    - if it is connected with some part lying on the same side of the line
#      as the reference point, it is connected so also in the result (belongs
#      to the same stroke) and will therefore be a part of the first
#      GimpVectors returned;
#    - otherwise it is isolated (no such connections) and will be a part of the
#      third GimpVectors returned.
# Args:
# - gv:              BCurve.GimpVectors
# - line_start:      [float,float] (=[x,y], point in plane)
# - line_end:        [float,float] (=[x,y], point in plane)
# - reference_point: None or [float,float] (=[x,y], point in plane)
# Returns:
# - BCurve.GimpVectors (the same side as the reference_point)
# - BCurve.GimpVectors (the opposite side from the reference_point)
# - BCurve.GimpVectors (flat on the splitting line)
# Notes:
# 1. Some of the returned objects may be empty (no strokes).
def split_gv_by_line(gv,
                     line_start,
                     line_end,
                     reference_point=None,
                     isolated_flats=True):
    # fix_breaks: Fix (*)
    # Args:
    # - lbc:[BCurve.BezierCurve]
    # - point: complex
    # Returns:
    # - [BCurve.BezierCurve] (fixed)
    # Note: instead of '==' in (!), should we allow some tolerance?
    def fix_breaks(lbc, point):
        #ZERO = 1e-12
        start_index = []
        end_index = []
        count = 0
        for bc in lbc:
            #if abs(bc.bezier_arcs[0].cp4[0] - point) < ZERO: # bc has point as start anchor (!)
            if bc.bezier_arcs[0].cp4[0] == point: # bc has point as start anchor (!)
                start_index.append(count)
            #if abs(bc.bezier_arcs[-1].cp4[-1] - point) < ZERO: # bc has point as end anchor (!)
            if bc.bezier_arcs[-1].cp4[-1] == point: # bc has point as end anchor (!)
                end_index.append(count)
            count += 1
        if len(start_index) == len(end_index) == 0: # Nothing to fix.
            return lbc
        elif len(start_index) != len(end_index): # Hopefully never happens!
            return lbc # Error: skip fixing and return lbc as it is.
            #raise Exception("split_gv_by_line:fix_breaks: error 0")
        elif len(start_index) != 1:              # Hopefully never happens!
            return lbc # Error: skip fixing and return lbc as it is.
            #raise Exception("split_gv_by_line:fix_breaks: error 1")
        s,e = start_index[0], end_index[0]
        if s == e: # Double anchor is start and end of same bc
            # Make it into new closed Bezier curve in the style of Gimp.
            # Remove the arc made previously to fill the gap.
            ba = lbc[s].bezier_arcs[-1] # To be removed
            new_bc = BCurve.BezierCurve(bezier_arcs = lbc[s].bezier_arcs[:-1],
                                        head_handle = ba.cp4[2],
                                        tail_handle = ba.cp4[1],
                                        closed = True
                                        )
        else: # Double anchor is start and end of two different bc's
            # Join the two bc's into one.
            bc1 = lbc[e]
            bc2 = lbc[s]
            new_bc = BCurve.BezierCurve(bezier_arcs = bc1.bezier_arcs + bc2.bezier_arcs,
                                        head_handle = bc1.head_handle,
                                        #tail_handle = bc2.head_handle, # fix 2021-01-01
                                        tail_handle = bc2.tail_handle,
                                        closed = False
                                        )
        return [lbc[i] for i in range(len(lbc)) if (i!=s and i!=e)] + [new_bc]
    z0 = complex(line_start[0], line_start[1])
    z1 = complex(line_end[0], line_end[1])
    if reference_point is None:
        zref = z0 + complex(-line_start[1]+line_end[1], line_start[0]-line_end[0])
    else:
        zref = complex(reference_point[0], reference_point[1])
    all_bc_near = []
    all_bc_far = []
    all_bc_flat = []
    for gs in gv.stroke_list: # gs: BCurve.GimpStroke
        bc = gs.gs2bc()      # BCurve.BezierCurve
        if bc.closed:
            #(*) In Gimp a closed stroke is just marked closed.
            #(*) We must add one Bezier arc to close the gap:
            new_cp4 = [bc.bezier_arcs[-1].cp4[-1],
                       bc.tail_handle,
                       bc.head_handle,
                       bc.bezier_arcs[0].cp4[0]
                       ]
            bc.bezier_arcs.append(BCurve.BezierArc(cp4=new_cp4))
            double_cp = new_cp4[-1]
            #(*) This creates a break and a double anchor at
            #(*) bc.bezier_arcs[0].cp4[0] (= now bc.bezier_arcs[-1].cp4[-1]).
            #(*) This is fixed later.
        split_list = bcurve_intersect_line(bc, z0, z1,
                                           zref,
                                           isolated_flats) # [[integer,float]]
        splitted = split_bezier_curve(bc, split_list) # [BCurve.BezierCurve]
        near, far, flat = group_bcurves_by_line(splitted, z0, z1, zref)
        #(*) Fix the previously made breaks and double anchors:
        if bc.closed:
            near = fix_breaks(near, double_cp)
            far = fix_breaks(far, double_cp)
            flat = fix_breaks(flat, double_cp)
        # Discard possible empty Bezier curves:
        near = [bc for bc in near if len(bc.bezier_arcs)>0]
        far = [bc for bc in far if len(bc.bezier_arcs)>0]
        flat = [bc for bc in flat if len(bc.bezier_arcs)>0]
        #
        all_bc_near += near
        all_bc_far += far
        all_bc_flat += flat
    # Now gathered all_bc_near, all_bc_far, all_bc_flat: [BCurve.BezierCurve]
    # From these build three objects: BCurve.GimpVectors.
    # Note: Some of the three stroke lists may be empty
    all_gs_near = [bc.bc2gs() for bc in all_bc_near] # [BCurve.GimpStroke]
    gv_near = BCurve.GimpVectors(stroke_list=all_gs_near,
                                     name=gv.name + '|near')
    all_gs_far = [bc.bc2gs() for bc in all_bc_far] # [BCurve.GimpStroke]
    gv_far = BCurve.GimpVectors(stroke_list=all_gs_far,
                                     name=gv.name + '|far')
    all_gs_flat = [bc.bc2gs() for bc in all_bc_flat] # [BCurve.GimpStroke]
    gv_flat = BCurve.GimpVectors(stroke_list=all_gs_flat,
                                     name=gv.name + '|flat')
    return gv_near, gv_far, gv_flat


# ------------------
# split_path_by_line
# ------------------
# Given a paths path_vectors:gimp.Vectors and a line (two points on the line),
# split the path path_vectors to 3 paths (or None) by intersecting with the line:
# - path on one side of the line;
# - path on the other side of the line;
# - path lying flat on the line.
# If isolated_flats=False, the "path on one side of the line" contains attached
# "flats" (parts on the line). But "path lying flat on the line" contains all
# parts not attached so. See comments for split_gv_by_line.
# Nothing is drawn.
# Args:
# - image
# - path_vectors: gimp.Vectors
# - line_anchors: [complex,complex] 
# Returns:
# - gimp.Vectors (the same side as the reference_point)
# - gimp.Vectors (the opposite side from the reference_point)
# - gimp.Vectors (flat on the splitting line)
# Note: Some of the returned paths may be empty (no strokes).
def split_path_by_line(image,
                       path_vectors,
                       line_anchors,
                       reference_point=None,
                       isolated_flats=True):
    line_start = [line_anchors[0].real, line_anchors[0].imag]
    line_end = [line_anchors[1].real, line_anchors[1].imag]
    gv = vectors_object2gv(path_vectors)
    gv_splitted = split_gv_by_line(gv,
                                   line_start,
                                   line_end,
                                   reference_point,
                                   isolated_flats)
    gv_1, gv_2, gv_on_line = gv_splitted
    #
    path_1 = gv_1.gv2vectors_object(image)
    path_1.name = path_vectors.name+'|split'
    path_2 = gv_2.gv2vectors_object(image)
    path_2.name = path_vectors.name+'|split'
    path_on_line = gv_on_line.gv2vectors_object(image)
    path_on_line.name = path_vectors.name+'|split'
    return path_1, path_2, path_on_line

#======================================================
#          Split path by a circle or selection
#======================================================

class PathPointData(object):
    """Record from point on a path:
    - one_anchor_stroke: boolean (True if the stroke consist of one anchor only)
    - stroke_number (the number of the stroke in the path)
    - arc_number (the number of the Bezier arc in the stroke)
    - t (parameter value of the point on the infinite Bezier curve B(t))
    - Bt (the point B(t))
    - Bdot (the derivative B'(t))
    - Bdotdot (the second derivative B''(t))
    Note: It may happen that arc_num is 1 larger than actually exists
    in the stroke. This means that the stroke is closed and this arc is the
    gap-closing (imaginary) arc.
    """
    def __init__(self,
                 one_anchor_stroke = False,
                 stroke_num = 0,
                 arc_num    = 0,
                 parameter  = 0,
                 Bt         = None,
                 Bdot       = None,
                 Bdotdot    = None
                 ):
        self.one_anchor_stroke = one_anchor_stroke
        self.stroke_number = stroke_num
        self.arc_number = arc_num
        self.parameter = parameter
        self.Bt = Bt
        self.Bdot = Bdot
        self.Bdotdot = Bdotdot
    def __str__(self):
        s  = 'stroke_number = '+str(self.stroke_number)
        s += '\narc_number    = '+str(self.arc_number)
        s += '\nparameter     = '+str(self.parameter)
        s += '\nB(t)          = '+str(self.Bt)
        s += '\nB\'(t)         = '+str(self.Bdot)
        s += '\nB\'\'(t)        = '+str(self.Bdotdot)
        return s


# -------------------
# gv_intersect_circle
# -------------------
# Find intersection points (as PathPointData) between a gv:BCurve.GimpVectors
# with the circle.
# Assume no intersection points exist at anchors.
# Args:
# - gv:     BCurve.GimpVectors
# - center: complex
# - radius: float
# Returns:
# - [PathPointData] (parameters t)
# Notes:
# 1. Each closed stroke is extended with a gap-closing arc, and the field
#    arc_number in PathPointData shows it.
# 2. Multiple roots are reduced to one.
def gv_intersect_circle(gv, center, radius):
    ZERO = 1e-6
    EQUAL_ROOT = 1e-10
    results = []
    gs_num = 0
    for gs in gv.stroke_list:     # BCurve.GimpStroke
        bc = gs.gs2bc()           # BCurve.BezierCurve
        if not bc.closed:
            bas = bc.bezier_arcs
        else: # Closed stroke. To close the gap, add one arc
            bas = bc.bezier_arcs + [(BCurve.BezierArc(cp4 = [bc.bezier_arcs[-1].cp4[-1],
                                                  bc.tail_handle,
                                                  bc.head_handle,
                                                  bc.bezier_arcs[0].cp4[0]
                                                  ]
                         ))]
        ba_num = 0
        for ba in bas: # BCurve.BezierArc
            cp4 = ba.cp4
            if len(cp4) == 1: # stroke with one anchor
                if abs(cp4[0]-center) < ZERO:
                    results.append(PathPointData(one_anchor_stroke=True,
                                                 stroke_num = gs_num,
                                                 arc_num = ba_num,
                                                 Bt = cp4[0]))
                ba_num += 1
                continue
            ts = bezier_intersect_circle(cp4, center, radius, restrict01=True)
            # Reduce multiple roots:
            ts.sort()
            try:
                reduced = [ts[0]]
            except IndexError: # ts = []
                ba_num += 1
                continue
            for t in ts[1:]:
                if t-reduced[-1] > EQUAL_ROOT:
                    reduced.append(t)
                #else:
                #    print("Reject t="+str(t)+" with t-reduced[-1]="+str(t-reduced[-1]))
            for t in reduced:
                results.append(PathPointData(one_anchor_stroke=False,
                                             stroke_num = gs_num,
                                             arc_num = ba_num,
                                             parameter = t,
                                             Bt = bezier_rv(t, cp4)
                                             ))
            ba_num += 1
        gs_num += 1
    return results

# ----------------------
# gv_intersect_selection
# ----------------------
# Find intersection points (as PathPointData) between a gv:BCurve.GimpVectors
# with the selection boundary.
# Assume no intersection points exist at anchors.
# Args:
# - gv:          BCurve.GimpVectors
# - crop_inputs: CroppingInputData
# Returns:
# - [PathPointData] (parameters t)
# Notes:
# 1. Each closed stroke is extended with a gap-closing arc, and the field
#    arc_number in PathPointData shows it.
def gv_intersect_selection(gv, crop_inputs):
    results = []
    gs_num = 0
    for gs in gv.stroke_list:     # BCurve.GimpStroke
        bc = gs.gs2bc()           # BCurve.BezierCurve
        if not bc.closed:
            bas = bc.bezier_arcs
        else: # Closed stroke. To close the gap, add one arc
            cp4 = [bc.bezier_arcs[-1].cp4[-1],
                   bc.tail_handle,
                   bc.head_handle,
                   bc.bezier_arcs[0].cp4[0]
                   ]
            bas = bc.bezier_arcs + [(BCurve.BezierArc(cp4=cp4))]
        ba_num = 0
        for ba in bas: # BCurve.BezierArc
            cp4 = ba.cp4
            if len(cp4) == 1: # stroke with one anchor
                x,y = cp4[0].real, cp4[0].imag
                if pixel_in_selection(x, y, crop_inputs.rgn, crop_inputs.threshold):
                    results.append(PathPointData(one_anchor_stroke=True,
                                                 stroke_num = gs_num,
                                                 arc_num = ba_num,
                                                 Bt = cp4[0]))
                ba_num += 1
                continue
            cut_parameters = arc_intersect_selection(cp4, crop_inputs)
            for t in cut_parameters:
                results.append(PathPointData(one_anchor_stroke=False,
                                             stroke_num = gs_num,
                                             arc_num = ba_num,
                                             parameter = t,
                                             Bt = bezier_rv(t,cp4)
                                             ))
            ba_num += 1
        gs_num += 1
    return results

# ----------------
# gv_split_by_CorS
# ----------------
# Given gv:BCurve.GimpVectors and either
# a circle (center, radius) or a selection input as 'crop_inputs'),
# split the gv at the intersection points with the circle or the selection boundary.
# Return two list of new Bezier curves, one for the inside curves, and the
# other for the outside curves.
# Args:
# - gv:          BCurve.GimpVectors
# - crop_inputs: CroppingInputData
# Returns:
# - [BCurve.BezierCurve] (inside)
# - [BCurve.BezierCurve] (outside)
def gv_split_by_CorS(gv, crop_inputs):
    ZERO = 1e-8
    # 1. Find intersection points at anchors and split gv at those points:
    #    Then no intersection points at anchors should exist.
    gv_split_anchors = gv_split_by_CorS_at_anchors(gv, crop_inputs)
    # 2. Find others as [PathPointData]:
    if crop_inputs.case == 'circle':
        center = crop_inputs.center
        radius = crop_inputs.radius
        pps_all = gv_intersect_circle(gv_split_anchors, center, radius) # [PathPointData]
    elif crop_inputs.case == 'selection':
        pps_all = gv_intersect_selection(gv_split_anchors, crop_inputs) # [PathPointData]
    else:
        raise Exception("Unknown case: "+crop_inputs.case)
    gs_num = 0
    split_bc_list = []
    for gs in gv_split_anchors.stroke_list: # BCurve.GimpStroke
        pps = [pp for pp in pps_all if pp.stroke_number == gs_num]
        split_bc_list += gs_split_by_pps(gs, pps)
        gs_num += 1
    inside_bcs = []
    outside_bcs = []
    if crop_inputs.case == 'circle':
        for bc in split_bc_list:
            center = crop_inputs.center
            radius = crop_inputs.radius
            if bcurve_inside_circle(bc, center, radius):
                inside_bcs.append(bc)
            else:
                outside_bcs.append(bc)
    elif crop_inputs.case == 'selection':
        for bc in split_bc_list:
            if bcurve_inside_selection(bc, crop_inputs):
                inside_bcs.append(bc)
            else:
                outside_bcs.append(bc)
    return inside_bcs, outside_bcs

# ---------------------------
# gv_split_by_CorS_at_anchors
# ---------------------------
# Preliminary step for splitting by circle: split at anchors.
# Split each stroke of gv where an anchor is on the circle or selectin boundary.
# Return new gv (copy of the old one with some strokes split).
# Args:
# - gv:    BCurve.GimpVectors
# - crop_inputs: CroppingInputData
# Returns:
# - BCurve.GimpVectors
def gv_split_by_CorS_at_anchors(gv, crop_inputs):
    from copy import deepcopy
    if crop_inputs.case == 'circle':
        center = crop_inputs.center
        radius = crop_inputs.radius
    elif crop_inputs.case == 'selection':
        R = 1 # Test radius
        threshold = crop_inputs.threshold
        image = crop_inputs.image
        #samples = crop_inputs.samples
        width,height = image.width, image.height
    else:
        raise Exception("Unknown case:" +str(crop_inputs.case))
    ZERO = 1e-8
    gv = deepcopy(gv)
    total_gs_list = [] # Total list of new gs's
    for gs in gv.stroke_list:
        new_gs_list = [] # List of new gs's created from this gs
        split_indexes = []
        for i in range(1,len(gs.cp_list),3): # anchors
            anchor = gs.cp_list[i]
            # Check if anchor is on the curve (either circle or selection boundary)
            if crop_inputs.case == 'circle':
                if radius-ZERO < abs(anchor-center) < radius+ZERO:
                    split_indexes.append(i)
            elif crop_inputs.case == 'selection':
                x,y = anchor.real, anchor.imag
                test = pixel_in_selection(x, y, crop_inputs.rgn, crop_inputs.threshold)
                for x1,y1 in [[x-R,y],[x-R,y-R],[x,y-R]]:
                    if test != (pixel_in_selection(x1, y1, crop_inputs.rgn, crop_inputs.threshold)):
                        split_indexes.append(i)
                        break
        if len(split_indexes) == 0: # No splitting of this gs
            total_gs_list += [gs]
            continue
        first_i = split_indexes[0]
        last_i = split_indexes[-1]
        split_at_first_anchor = (first_i == 1)
        split_at_last_anchor = (last_i == len(gs.cp_list)-2)
        if len(split_indexes) == 1 and (split_at_first_anchor or split_at_last_anchor):
            if not gs.closed:           # No splitting: save as such
                pass
            elif split_at_first_anchor: # Closed: must make open at first anchor
                gs.cp_list += [gs.cp_list[0], gs.cp_list[1], gs.cp_list[1]]
                gs.cp_list[0] = gs.cp_list[1]
                gs.closed = False
            elif split_at_last_anchor:  # Closed: must make open at last anchor
                add_at_start = [gs.cp_list[-2],gs.cp_list[-2],gs.cp_list[-1]]
                gs.cp_list[-1] = gs.cp_list[-2]
                gs.cp_list = add_at_start + gs.cp_list
                gs.closed = False
            total_gs_list += [gs] # Done this gs
            continue
        if not split_at_first_anchor:
            first_new_gs = BCurve.GimpStroke(cp_list=gs.cp_list[:first_i+1]+[gs.cp_list[first_i]],
                                         closed=False)
            new_gs_list.append(first_new_gs)
        for k in range(len(split_indexes)):
            try:
                i,j = split_indexes[k], split_indexes[k+1]
            except IndexError:
                #print("break")
                break
            new_cp_list = gs.cp_list[i:j+1]
            new_cp_list = [new_cp_list[0]] + new_cp_list + [new_cp_list[-1]]
            new_gs = BCurve.GimpStroke(cp_list=new_cp_list,
                                         closed=False)
            new_gs_list.append(new_gs)
        if not split_at_last_anchor:
            last_new_gs = BCurve.GimpStroke(cp_list=[gs.cp_list[last_i]]+gs.cp_list[last_i:],
                                         closed=False)
            new_gs_list.append(last_new_gs)
        if gs.closed:
            if split_at_first_anchor and split_at_last_anchor:
                new_cp_list = [gs.cp_list[-2],gs.cp_list[-2],gs.cp_list[-1],
                               gs.cp_list[0],gs.cp_list[1],gs.cp_list[1]]
                new_gs_list.append(BCurve.GimpStroke(cp_list=new_cp_list,
                                                     closed=False))
            elif split_at_first_anchor: # but not at last
                new_gs_list[-1].cp_list += [gs.cp_list[0],gs.cp_list[1],gs.cp_list[1]]
                new_gs_list[-1].closed = False
            elif split_at_last_anchor: # but not at first
                add_at_start = [gs.cp_list[-2],gs.cp_list[-2],gs.cp_list[-1]]
                new_gs_list[0].cp_list = add_at_start + new_gs_list[0].cp_list
            elif len(split_indexes) > 0: # Must join two strokes with gap-closing arc
                gap_cp_list = [gs.cp_list[-1], gs.cp_list[0]]
                first_new_gs.cp_list = (last_new_gs.cp_list[:-1] + # replace
                                        gap_cp_list +
                                        first_new_gs.cp_list[1:])
                new_gs_list = new_gs_list[:-1]                     # remove
        total_gs_list += new_gs_list
    new_gv = BCurve.GimpVectors(stroke_list=total_gs_list)
    #################  KOE
    #new_koe = new_gv.gv2vectors_object(image)
    #new_koe.name = 'NEW KOE'
    #gimp_draw_vectors_object(image, new_koe, visible=True)
    return new_gv

# ---------------
# gs_split_by_pps
# ---------------
# Given gs:BCurve.GimpStroke and a list pps:[PathPointData]
# split the gs at the path points.
# Reject possible splittings at anchors
# (handled already by gv_split_by_CorS_at_anchors).
# Return list of Bezier curves [BCurve.BezierCurve].
# Args:
# - gv:  BCurve.GimpStroke
# - pps: [PathPointData]
# Returns:
# - [BCurve.BezierCurve]
def gs_split_by_pps(gs, pps):
    ZERO = 1e-8
    if len(gs.cp_list) == 3: # 1-anchor stroke, no splitting
        return [gs.gs2bc()]
    bc_list_for_gs = []   # List bezier curves resulting from splitting this gs
    bc = gs.gs2bc()       # BCurve.BezierCurve
    pps.sort(key=(lambda p: p.arc_number))
    cp4_list = []
    len_list = []
    current = 0
    bas = bc.bezier_arcs
    if bc.closed:
        #(*) In Gimp a closed stroke is just marked closed.
        #(*) We must add one Bezier arc to close the gap:
        bas.append(BCurve.BezierArc(cp4=[bc.bezier_arcs[-1].cp4[-1],
                                         bc.tail_handle,
                                         bc.head_handle,
                                         bc.bezier_arcs[0].cp4[0]]
                                    )
                   )
    ba_num = 0
    splits_in_bc = 0
    for ba in bas:
        # splitting t's for this arc:
        ts = [pp.parameter for pp in pps if pp.arc_number == ba_num]
        ts = [t for t in ts if ZERO < t < 1-ZERO] # Reject splittings at anchors
        splits_in_bc += len(ts)
        sub_arcs = bezier_subdivide_at_parameters(ts, ba.cp4)
        cp4_list += sub_arcs
        n = len(ts)
        if n == 0:
            current += 1
            ba_num += 1
            continue
        else:
            len_list.append(current+1)
            len_list += [1]*(n-1)
            current = 1
        ba_num += 1
    len_list.append(current)
    cp4_sublists = []
    cum = 0
    for slen in len_list:
        cp4_sublists.append(cp4_list[cum : cum+slen])
        cum += slen
    #for ccc in cp4_sublists:
    #    print(str(ccc))
    for sublist in cp4_sublists:
        new_ba_list = [BCurve.BezierArc(cp4=cp4) for cp4 in sublist]
        if len(new_ba_list) == 0:
            raise Exception("ULOS")
        bc_list_for_gs.append(BCurve.BezierCurve(bezier_arcs=new_ba_list))
    if bc.closed and (splits_in_bc > 0):
        last = bc_list_for_gs[-1] # Must combine these two: last+first
        first = bc_list_for_gs[0]
        combined_bc = BCurve.BezierCurve(
            bezier_arcs = last.bezier_arcs + first.bezier_arcs,
            head_handle = last.head_handle,
            tail_handle = first.tail_handle,
            )
        bc_list_for_gs = bc_list_for_gs[1:-1] + [combined_bc]
    if bc.closed and (splits_in_bc == 0):
        bc1 = bc_list_for_gs[-1] # Last appended: must close properly
        last_arc = bc1.bezier_arcs[-1]
        bc1.bezier_arcs = bc1.bezier_arcs[:-1] # Drop last arc
        bc1.head_handle = last_arc.cp4[2]
        bc1.tail_handle = last_arc.cp4[1]
        bc1.closed = True
    return bc_list_for_gs

# --------------------
# bcurve_inside_circle
# --------------------
# Given a Bezier curve and a circle, assuming that the curve is either inside
# or outside of the circle (or on the circle), tell if it is inside or outside.
# (If the curve lies on the circle sufficiently closely, it is taken to be inside.)
# Arg:
# - bc:     BCurve.BezierCurve
# - center: complex
# - radius: float
# Returns:
# - boolean
def bcurve_inside_circle(bc, center, radius):
    ZERO = 1e-10
    # First try anchors:
    anchors = [ba.cp4[0] for ba in bc.bezier_arcs] + [bc.bezier_arcs[-1].cp4[-1]]
    # Find the anchor farthest from the circle (inside or outside):
    farthest = max(anchors, key=(lambda a: abs(abs(a-center)-radius)))
    distance = abs(farthest-center)
    if distance > radius+ZERO:
        return False
    elif distance < radius-ZERO:
        return True
    # If not found, try middle points of arcs:
    for ba in bc.bezier_arcs:
        try:
            p0,p1,p2,p3 = ba.cp4
            middle = (p0 + 3*p1 + 3*p2 + p3) / 8
        except ValueError: # one-point stroke: handled above
            #middle = ba.cp4[0]
            continue
        r = abs(middle-center)
        if r < radius-ZERO:
            return True
        elif r > radius+ZERO:
            return False
    # If not found, try some other points:
    for ba in bc.bezier_arcs:
        if len(ba.cp4) == 1:
            continue
        for t in [.15, .3, .45, .55, .7, .85]:
            r = abs(bezier_rv(t, ba.cp4) - center)
        if r < radius-ZERO:
            return True
        elif r > radius+ZERO:
            return False
    # Nothing found: the curve is so exactly(?) on the circle
    # that return that it is inside:
    return True

# -----------------------
# bcurve_inside_selection
# -----------------------
# Given a Bezier curve and a selection, assuming that the curve is either inside
# or outside of the selection (or on the selection boundary), tell if it is inside
# or outside.
# (If the curve lies on the selection sufficiently closely, it is taken to be inside.)
# Arg:
# - bc:          BCurve.BezierCurve
# - crop_inputs: CroppingInputData
# Returns:
# - boolean
def bcurve_inside_selection(bc, crop_inputs):
    SAMPLES = 4 # Experimental: not much effect in running time
    if len(bc.bezier_arcs) == 1:
        if len(bc.bezier_arcs[0].cp4) == 1: # Curve is one point P
            P = bc.bezier_arcs[0].cp4[0]
            x,y = P.real, P.imag
            return pixel_in_selection(x, y, crop_inputs.rgn, crop_inputs.threshold)
    inside_vote = 0
    outside_vote = 0
    anchors = [ba.cp4[0] for ba in bc.bezier_arcs] + [bc.bezier_arcs[-1].cp4[-1]]
    count = 0
    for p0 in anchors:
        x,y = p0.real, p0.imag
        if pixel_in_selection(x, y, crop_inputs.rgn, crop_inputs.threshold):
            inside_vote += 1
        else:
            outside_vote += 1
        count += 1
    count = 0
    if abs(inside_vote - outside_vote) < 3: # Try midpoints of arcs
        for ba in bc.bezier_arcs:
            try:
                p0,p1,p2,p3 = ba.cp4
            except ValueError: # 1-anchor stroke
                continue
            point = (p0 + 3*p1 + 3*p2 + p3)/8
            x,y = point.real, point.imag
            if pixel_in_selection(x, y, crop_inputs.rgn, crop_inputs.threshold):
                inside_vote += 1
            else:
                outside_vote += 1
            count += 1
    count = 0
    if abs(inside_vote - outside_vote) < 3: # Try sample points
        for ba in bc.bezier_arcs:
            cp4 = ba.cp4
            for i in range(1,SAMPLES):
                point = bezier_rv(i/SAMPLES, cp4)
                x, y = point.real, point.imag
                if pixel_in_selection(x, y, crop_inputs.rgn, crop_inputs.threshold):
                    inside_vote += 1
                else:
                    outside_vote += 1
                count += 1
        #print("Samples: "+str(count))
    count = 0
    if abs(inside_vote - outside_vote) < 3: # Try further sample points
        for ba in bc.bezier_arcs:
            cp4 = ba.cp4
            for i in range(1, 2*SAMPLES, 2):
                point = bezier_rv(i/(2*SAMPLES), cp4)
                x, y = point.real, point.imag
                if pixel_in_selection(x, y, crop_inputs.rgn, crop_inputs.threshold):
                    inside_vote += 1
                else:
                    outside_vote += 1
                count += 1
        #print("Further samples: "+str(count))
    #if abs(inside_vote - outside_vote) < 3:
    #    print("                                   Voting failed")
    return (inside_vote >= outside_vote)

## This is not used anywhere.
## It is here only to record how the coefficients array in
## bezier_intersect_circle was created.
#def initialize_coefficients_for_bezier_intersect_circle():
#    coeff = []
#    for i in range(7):
#        coeff.append([[None,None,None,None], # a[i][k][l]
#                      [None,None,None,None],
#                      [None,None,None,None],
#                      [None,None,None,None]
#                     ])
#    c = [[  1,   0,   0,   0,   0,   0,   0], # c0
#         [ -6,   1,   0,   0,   0,   0,   0], # c1
#         [ 15,  -5,   1,   0,   0,   0,   0], # c2
#         [-20,  10,  -4,   1,   0,   0,   0], # c3
#         [ 15, -10,   6,  -3,   1,   0,   0], # c4
#         [ -6,   5,  -4,   3,  -2,   1,   0], # c5
#         [  1,  -1,   1,  -1,   1,  -1,   1], # c6
#         ]
#    for i in range(7):
#        for k in range(4):
#            for l in range(4):
#                coeff[i][k][l] = (3**(min(k,3-k)+min(l,3-l))) * c[i][k+l]
#    return coeff

# -----------------------
# bezier_intersect_circle
# -----------------------
# Given a Bezier curve (4 control points) and a circle (center and radius),
# find the parameter values of all intersection points.
# The plane is viewed as the complex number plane.
# Args:
# - cp:         [complex,complex,complex,complex]
# - center:     complex
# - radius:     float
# - restrict01: boolean (accept only 0 <= t <= 1?)
# Returns:
# - [float] (parameters t)
def bezier_intersect_circle(cp, center, radius, restrict01=True):
    coefficients = [
    [[  1,   0,   0,  0], [  0,   0,   0,  0], [  0,   0,   0,  0], [ 0,  0,  0, 0]],
    [[ -6,   3,   0,  0], [  3,   0,   0,  0], [  0,   0,   0,  0], [ 0,  0,  0, 0]],
    [[ 15, -15,   3,  0], [-15,   9,   0,  0], [  3,   0,   0,  0], [ 0,  0,  0, 0]],
    [[-20,  30, -12,  1], [ 30, -36,   9,  0], [-12,   9,   0,  0], [ 1,  0,  0, 0]],
    [[ 15, -30,  18, -3], [-30,  54, -27,  3], [ 18, -27,   9,  0], [-3,  3,  0, 0]],
    [[ -6,  15, -12,  3], [ 15, -36,  27, -6], [-12,  27, -18,  3], [ 3, -6,  3, 0]],
    [[  1,  -3,   3, -1], [ -3,   9,  -9,  3], [  3,  -9,   9, -3], [-1,  3, -3, 1]]
    ]
    ## Test:
    #coeff = initialize_coefficients_for_bezier_intersect_circle()
    #print("Test: "+str(coeff == coefficients))
    A = [0,0,0,0,0,0,0] # This will be the coefficients of the polynomial (deg=6)
    for i in range(7):
        s = 0
        for k in range(4):
            for l in range(4):
                s += coefficients[i][k][l] * vdot(cp[k]-center, cp[l]-center)
        A[i] = s
    A[0] -= radius**2
    m = max([abs(x) for x in A])
    A = [x/m for x in A]
    if restrict01:
        zeroes = zeroes_of_polynomial(A, [0,1])[0] # Solve
    else:
        zeroes = zeroes_of_polynomial(A)[0] # Solve
    #if restrict01:
    #    zeroes = [t for t in zeroes if 0 <= t <= 1]
    return zeroes

# ------------------------------
# bezier_subdivide_at_parameters
# ------------------------------
# Given a Bezier arc B(t) and a list of parameter values t1,t2,...,
# find control points for all subarcs when B(t), 0<=t<=1, is subdivided
# at parameter values t1,...
# The input Bezier arc is given as its 4 control points [p0,p1,p2,p3],
# and the output is a list of new sets of 4 control points [[q0,q1,q2,q3],...].
# The plane is viewed as the complex number plane.
# Args:
# - parameter_list: [float]                           ([t1,t2,...])
# - control_points: [complex,complex,complex,complex] ([p0,p1,p2,p3])
# Returns:
# - [[complex,complex,complex,complex]]               ([[q0,q1,q2,q3],...])
def bezier_subdivide_at_parameters(parameter_list, control_points):
    ts01 = [t for t in parameter_list if 0 < t < 1] # Drop t's outside (0,1)
    ts01.sort()
    try:
        ts = [ts01[0]]
    except IndexError: # Empty parameter list
        return [[p for p in control_points]] # copy
    for t in ts01[1:]: # Drop duplicates.
        if t != ts[-1]:
            ts.append(t)
    ts = [0]+ts+[1] # Add t=0,1
    # Collect pairs [t(i),t(i+1)]
    abs = []
    for i in range(1,len(ts)):
        abs.append([ts[i-1],ts[i]])
    arc_cps = []
    for a,b in abs:
        arc_cps.append(bezier_new_control_points(a, b, control_points))
    return arc_cps



###======================================================
###            Splitting by line: Main function
###======================================================
### -----------------------
### split_path_by_line_main
### -----------------------
### Given two paths
### - path:gimp.Vectors
### - line:gimp.Vectors (a 2-anchor path),
### split the path to 3 paths (or None) by intersecting with the line.
### - path on one side of the line;
### - path on the other side of the;
### - path lying flat on the line.
### The non-empty of these are drawn.
### Args:
### - image
### - path_vectors: gimp.Vectors
### - line_vectors: gimp.Vectors
### Returns nothing
##def split_path_by_line_main(image, path_vectors, line_vectors):
##    line_anchors = path_anchors(line_vectors)
##    if len(line_anchors) != 2:
##        raise Exception("The dividing line must have 2 anchors. Got "+str(len(line_anchors)))
##    path_1, path_2, path_3 = split_path_by_line(image, path_vectors, line_anchors)
##    pdb.gimp_image_undo_group_start(image)           # Draw
##    if path_1.strokes != []:
##        gimp_draw_vectors_object(image, path_1, visible=True)
##    if path_2.strokes != []:
##        gimp_draw_vectors_object(image, path_2, visible=True)
##    if path_3.strokes != []:
##        gimp_draw_vectors_object(image, path_3, visible=True)
##    pdb.gimp_image_undo_group_end(image)
##

#======================================================
#            Slicing by lines: Main function
#======================================================

MAX_SLICE_LINES = 8 # Maximum number of slicing lines allowed

# ------------------------
# slice_path_by_lines_main
# ------------------------
# Given two paths
# - path_vectors: gimp.Vectors
# - line_vectors: gimp.Vectors,
# slice the first path to several paths by intersecting with the lines.
# The lines are taken from the second path by taking the first and second
# anchor of each stroke. Strokes with 0 or 1 anchors are ignored.
# Draw non-empty slices.
# Args:
# - image
# - path_vectors: gimp.Vectors
# - line_vectors: gimp.Vectors
# Returns nothing
def slice_path_by_lines_main(image, path_vectors, line_vectors):
    path_copy = pdb.gimp_vectors_copy(path_vectors)
    list_paths = [path_copy]             # [gimp.Vectors]     (paths under work)
    gv = vectors_object2gv(line_vectors) # BCurve.GimpVectors (slicing lines)
    if len(gv.stroke_list) > MAX_SLICE_LINES:
        m = "At most "+str(MAX_SLICE_LINES)+" slicing lines allowed, "
        m += "got "+str(len(gv.stroke_list))
        raise Exception(m)
    for gs in gv.stroke_list:            # BCurve.GimpStroke  (gs = one slicing line)
        if len(gs.cp_list) < 2:
            continue # ignore this stroke
        bc = gs.gs2bc()                  # BCurve.BezierCurve
        try:
            ba = bc.bezier_arcs[0]       # BCurve.BezierCurve (take first arc)
            p0,p3 = ba.cp4[0], ba.cp4[3] # complex (take first two anchors)
        except IndexError:
            continue
        splitted_by_gs = []              # [gimp.Vectors]
        for path in list_paths:
            splitted = split_path_by_line(image, # [gimp.Vectors]
                                          path,
                                          line_anchors=[p0,p3])
            splitted_by_gs += splitted
        # Update list, and go to slicing with the next line:
        list_paths = splitted_by_gs
    pdb.gimp_image_undo_group_start(image)           # Draw
    for path in list_paths:
        if path.strokes != []:           # ignore empty
            path.name = path_vectors.name + '|slice'
            gimp_draw_vectors_object(image, path, visible=True)
    pdb.gimp_image_undo_group_end(image)


#======================================================
#            Slicing by guides: Main function
#======================================================

slice_guide_options = [ # (description, identifier)
        ('Use all guides',           'all_guides'),
        ('Use horizontal guides',    'hor_guides'),
        ('Use vertical guides',      'ver_guides'),
        ]

# -------------------------
# slice_path_by_guides_main
# -------------------------
# Slice a path to several paths by intersecting with guides.
# Choosing the guides is controlled by guide_option (see slice_guide_options).
# Draw non-empty slices.
# Args:
# - image
# - path_vectors: gimp.Vectors
# - guide_option: integer
# Returns nothing
def slice_path_by_guides_main(image, path_vectors, guide_option):
    path_copy = pdb.gimp_vectors_copy(path_vectors)
    list_paths = [path_copy]          # [gimp.Vectors]     (paths under work)
    # Get slicing lines. Enough to get a pair of points on each line.
    slicing_pairs = [] # [[complex,complex]]
    hor,ver = get_guide_positions(image)
    guide_case = slice_guide_options[guide_option][1]
    if guide_case == 'all_guides':
        nguides = len(hor)+len(ver)
    elif guide_case == 'hor_guides':
        nguides = len(hor)
    elif guide_case == 'ver_guides':
        nguides = len(ver)
    if nguides > MAX_SLICE_LINES:
        m = "At most "+str(MAX_SLICE_LINES)+" slicing guides allowed, "
        m += "got "+str(nguides)
        raise Exception(m)
    h,w = image.height, image.width
    if guide_case in ('all_guides', 'hor_guides'):
        for y in hor:
            slicing_pairs.append([complex(0,y),complex(w,y)])
    if guide_case in ('all_guides', 'ver_guides'):
        for x in ver:
            slicing_pairs.append([complex(x,0),complex(x,h)])
    for split_pair in slicing_pairs:
        splitted_by_pair = []              # [gimp.Vectors]
        for path in list_paths:
            splitted = split_path_by_line(image, # [gimp.Vectors]
                                          path,
                                          line_anchors=split_pair)
            splitted_by_pair += splitted
        # Update list, and go to slicing with the next line:
        list_paths = splitted_by_pair
    pdb.gimp_image_undo_group_start(image)           # Draw
    for path in list_paths:
        if path.strokes != []:           # ignore empty
            path.name = path_vectors.name + '|slice'
            gimp_draw_vectors_object(image, path, visible=True)
    pdb.gimp_image_undo_group_end(image)


#======================================================
#      Cropping by convex polygon: Main function
#======================================================

# ---------------------------
# crop_path_by_convex_polygon
# ---------------------------
# Given a path and polygon corners (assumed convex polygon),
# crop the path path_vectors by the polygon.
# Nothing is drawn.
# Args:
# - image
# - path_vectors:    gimp.Vectors
# - polygon_corners: [complex]
# Returns:
# - gimp.Vectors
# Note: The returned path may be empty (no strokes).
def crop_path_by_convex_polygon(image, path_vectors, polygon_corners):
    if collinear(polygon_corners) or (not is_convex(polygon_corners)):
        raise Exception("Not a convex polygon")
    v_polygon = [[p.real,p.imag] for p in polygon_corners]
    pairs_polygon = [[v_polygon[i],v_polygon[i+1]] for i in range(len(v_polygon)-1)]
    pairs_polygon.append([v_polygon[-1], v_polygon[0]])
    zref = sum(polygon_corners) / len(polygon_corners)
    v_ref = [zref.real, zref.imag]
    #
    gv = vectors_object2gv(path_vectors)
    #count = 0
    for s,e in pairs_polygon:
        splitted = split_gv_by_line(gv,
                              line_start=s,
                              line_end=e,
                              reference_point=v_ref,
                              isolated_flats=False) # discard two others
        gv0 = splitted[0]
        gv2 = splitted[2] # Possible isolated segments lying on the dividing line
        gv = BCurve.GimpVectors(stroke_list = gv0.stroke_list + gv2.stroke_list)
        #ve_temp = gv.gv2vectors_object(image)
        #ve_temp.name = str(count)
        #gimp_draw_vectors_object(image, ve_temp, visible=True)
        #count += 1
    gv.name = path_vectors.name+'|crop'
    return gv.gv2vectors_object(image)

# --------------------------------
# crop_path_by_convex_polygon_main
# --------------------------------
# Given two paths
# - path_vectors:gimp.Vectors
# - polygon_vectors:gimp.Vectors (straight edges, convex),
# crop the path path_vectors by the polygon. Draw.
# Args:
# - image
# - path_vectors:    gimp.Vectors
# - polygon_vectors: gimp.Vectors
# Returns:
# - gimp.Vectors
# Note: The resulting path may be empty (no strokes). Even then the vectors
# object is created and inserted in Gimp and drawn - but nothing appears
# on the screen.
def crop_path_by_convex_polygon_main(image, path_vectors, polygon_vectors):
    polygon_corners = path_anchors(polygon_vectors) # [complex]
    cropped = crop_path_by_convex_polygon(image, path_vectors, polygon_corners)
    pdb.gimp_image_undo_group_start(image)           # Draw
    gimp_draw_vectors_object(image, cropped, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return cropped


#======================================================
#        Cropping by rectangle: Main function
#======================================================

crop_box_options = [ # (description, identifier)
        ('Rectangular selection',                  'bb_selection'),
        ('Guides - two horizontal, two vertical',  'guides'),
        ('Image (canvas)',                         'image'),
        ('Active layer',                           'layer'),
        ]

# ---------------------------
# crop_path_by_rectangle_main
# ---------------------------
# Crop a path by a rectangular. The choice of the rectangular is controlled by
# box_option (see crop_box_options).
# Draw.
# Args:
# - image
# - path_vectors: gimp.Vectors
# - box_option:   integer
# Returns:
# - gimp.Vectors
# Note: The resulting path may be empty (no strokes). Even then the vectors
# object is created and inserted in Gimp and drawn - but nothing appears
# on the screen.
def crop_path_by_rectangle_main(image, path_vectors, box_option):
    def check_box(nw,se): # Check that the box is not flat.
        ZERO = 1e-8
        N,W = nw.imag, nw.real
        S,E = se.imag, se.real
        hor_ok = (abs(W-E) > ZERO)
        vert_ok = (abs(N-S) > ZERO)
        if not (hor_ok and vert_ok):
            raise Exception("The cropping rectangle is too close to flat.")
    box_case = crop_box_options[box_option][1]
    box_sw, box_se, box_ne, box_nw = get_box(image, box_case)
    check_box(box_nw, box_se)
    polygon_corners = [box_sw, box_se, box_ne, box_nw]
    cropped = crop_path_by_convex_polygon(image, path_vectors, polygon_corners)
    pdb.gimp_image_undo_group_start(image)           # Draw
    gimp_draw_vectors_object(image, cropped, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return cropped


#======================================================
#        Cropping by rectangle: Main function
#======================================================

crop_box_options = [ # (description, identifier)
        ('Rectangular selection',                  'bb_selection'),
        ('Guides - two horizontal, two vertical',  'guides'),
        ('Image (canvas)',                         'image'),
        ('Active layer',                           'layer'),
        ]

# ---------------------------
# crop_path_by_rectangle_main
# ---------------------------
# Crop a path by a rectangular. The choice of the rectangular is controlled by
# box_option (see crop_box_options).
# Draw.
# Args:
# - image
# - path_vectors: gimp.Vectors
# - box_option:   integer
# Returns:
# - gimp.Vectors
# Note: The resulting path may be empty (no strokes). Even then the vectors
# object is created and inserted in Gimp and drawn - but nothing appears
# on the screen.
def crop_path_by_rectangle_main(image, path_vectors, box_option):
    def check_box(nw,se): # Check that the box is not flat.
        ZERO = 1e-8
        N,W = nw.imag, nw.real
        S,E = se.imag, se.real
        hor_ok = (abs(W-E) > ZERO)
        vert_ok = (abs(N-S) > ZERO)
        if not (hor_ok and vert_ok):
            raise Exception("The cropping rectangle is too close to flat.")
    box_case = crop_box_options[box_option][1]
    box_sw, box_se, box_ne, box_nw = get_box(image, box_case)
    check_box(box_nw, box_se)
    polygon_corners = [box_sw, box_se, box_ne, box_nw]
    cropped = crop_path_by_convex_polygon(image, path_vectors, polygon_corners)
    pdb.gimp_image_undo_group_start(image)           # Draw
    gimp_draw_vectors_object(image, cropped, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return cropped


#======================================================
#        Cropping by rectangle: Main function
#======================================================

crop_box_options = [ # (description, identifier)
        ('Rectangular selection',                  'bb_selection'),
        ('Guides - two horizontal, two vertical',  'guides'),
        ('Image (canvas)',                         'image'),
        ('Active layer',                           'layer'),
        ]

# ---------------------------
# crop_path_by_rectangle_main
# ---------------------------
# Crop a path by a rectangular. The choice of the rectangular is controlled by
# box_option (see crop_box_options).
# Draw.
# Args:
# - image
# - path_vectors: gimp.Vectors
# - box_option:   integer
# Returns:
# - gimp.Vectors
# Note: The resulting path may be empty (no strokes). Even then the vectors
# object is created and inserted in Gimp and drawn - but nothing appears
# on the screen.
def crop_path_by_rectangle_main(image, path_vectors, box_option):
    def check_box(nw,se): # Check that the box is not flat.
        ZERO = 1e-8
        N,W = nw.imag, nw.real
        S,E = se.imag, se.real
        hor_ok = (abs(W-E) > ZERO)
        vert_ok = (abs(N-S) > ZERO)
        if not (hor_ok and vert_ok):
            raise Exception("The cropping rectangle is too close to flat.")
    box_case = crop_box_options[box_option][1]
    box_sw, box_se, box_ne, box_nw = get_box(image, box_case)
    check_box(box_nw, box_se)
    polygon_corners = [box_sw, box_se, box_ne, box_nw]
    cropped = crop_path_by_convex_polygon(image, path_vectors, polygon_corners)
    pdb.gimp_image_undo_group_start(image)           # Draw
    gimp_draw_vectors_object(image, cropped, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return cropped


#======================================================
#    Cropping by circle or selection: Main functions
#======================================================

circle_options = [ # (description, identifier)
        ('a diameter of the cropping circle',                 'diameter'),
        ('the center and one point on the cropping circle',   'center_point'),
        ('the same as above but used reversed',    'center_point_reversed'),
        ]

circle_action_options = [ # (description, identifier)
        ('Draw only the path inside the circle',              'inside'),
        ('Draw only the path outside the circle',             'outside'),
        ('Draw both the inside and the outside paths',        'both sides'),
        ]

selection_action_options = [ # (description, identifier)
        ('Draw only the path inside the selection',              'inside'),
        ('Draw only the path outside the selection',             'outside'),
        ('Draw both the inside and the outside paths',        'both sides'),
        ]

class CroppingInputData(object):
    """To store input data for cropping. Attributes:
    - case: string. Currently one of: 'circle', 'selection'.
    Additional data: For 'circle':
    - image
    - center: complex
    - radius: float
    For 'selection':
    - rgn: gimp.PixelRgn
    - selection_bounds: (int,int,int,int)
    - threshold: integer (0..255)
    """
    def __init__(self,
                 image,
                 case,
                 center=None,
                 radius=None,
                 rgn=None,
                 selection_bounds=None,
                 threshold=None):
        self.image = image
        self.case = case
        self.center = center
        self.radius = radius
        self.rgn = rgn
        self.selection_bounds = selection_bounds
        self.threshold = threshold
    def __str__(self):
        s = "CroppingInputData:"
        s += "\n    image:   "+str(self.image)
        s += "\n    case:    "+str(self.case)
        s += "\n    center:  "+str(self.center)
        s += "\n    radius:  "+str(self.radius)
        return s

# ------------------------
# crop_path_by_circle_main
# ------------------------
# Crop a path by a circle. The choice of the circle is controlled by
# circle_option (see circle_options).
# Draw.
# Args:
# - image
# - path_vectors:  gimp.Vectors
# - circle_path:   gimp.Vectors (diameter or ...)
# - circle_option: integer
# - action_option: integer
# - draw_circle:   boolean
# Returns:
# - gimp.Vectors
# Note: The resulting path may be empty (no strokes). Even then the vectors
# object is created and inserted in Gimp and drawn - but nothing appears
# on the screen.
def crop_path_by_circle_main(image,
                             path_vectors,
                             circle_path,
                             circle_option,
                             action_option,
                             draw_circle
                             ):
    circle_case = circle_options[circle_option][1]
    circle_anchors = path_anchors(circle_path, only_first_stroke=True)
    if len(circle_anchors) != 2:
        raise Exception("Please give a 2-anchors path for the circle")
    a,b = circle_anchors
    if circle_case == 'diameter':
        center = (a+b)/2
        radius = abs(a-b)/2
    elif circle_case == 'center_point':
        center = a
        radius = abs(a-b)
    elif circle_case == 'center_point_reversed':
        center = b
        radius = abs(a-b)
    else:
        raise Exception("crop_path_by_circle_main: Unknown case: "+circle_case)
    crop_input_data = CroppingInputData(image=image,
                                        case='circle',
                                        center=center,
                                        radius=radius)
    path_gv = vectors_object2gv(path_vectors)
    inside_gs_list, outside_gs_list = crop_path_by_CorS(path_gv, # BCurve.GimpVectors
                        crop_input_data)
    inside_gv = BCurve.GimpVectors(inside_gs_list)
    outside_gv = BCurve.GimpVectors(outside_gs_list)
    inside_path = inside_gv.gv2vectors_object(image, name=path_vectors.name+'|inside')
    outside_path = outside_gv.gv2vectors_object(image, name=path_vectors.name+'|outside')
    
    action_case = circle_action_options[action_option][1]
    pdb.gimp_image_undo_group_start(image)           # Draw
    if draw_circle:
        gimp_draw_circle(image, [center.real,center.imag], radius, name='cropping circle')
    if action_case in ('outside', 'both sides'):
        gimp_draw_vectors_object(image, outside_path, visible=True)
    if action_case in ('inside', 'both sides'):
        gimp_draw_vectors_object(image, inside_path, visible=True)
    pdb.gimp_image_undo_group_end(image)           # Draw
    if action_case == 'outside':
        return outside_path
    else:
        return inside_path

# ---------------------------
# crop_path_by_selection_main
# ---------------------------
# Crop a path by a selection.
# Draw.
# Args:
# - image
# - path_vectors:  gimp.Vectors
# - action_option: integer
# - threshold:     integer
# Returns:
# - gimp.Vectors
# Note: The resulting path may be empty (no strokes). Even then the vectors
# object is created and inserted in Gimp and drawn - but nothing appears
# on the screen.
def crop_path_by_selection_main(image,
                                path_vectors,
                                action_option,
                                threshold,
                                ):
    threshold = max(0, min(255, int(threshold)))
    non_empty, x1, y1, x2, y2 = pdb.gimp_selection_bounds(image)
    if not non_empty:
        raise Exception("No active selection")
    non_empty, x1, y1, x2, y2 = pdb.gimp_selection_bounds(image)
    rgn = image.selection.get_pixel_rgn(x1, y1, x2, y2)
    crop_input_data = CroppingInputData(image=image,
                                        case='selection',
                                        rgn=rgn,
                                        selection_bounds = (x1, y1, x2, y2),
                                        threshold=threshold,
                                        )
    path_gv = vectors_object2gv(path_vectors)
    inside_gs_list, outside_gs_list = crop_path_by_CorS(path_gv,
                                                        crop_input_data)
    inside_gv = BCurve.GimpVectors(inside_gs_list)
    outside_gv = BCurve.GimpVectors(outside_gs_list)
    inside_path = inside_gv.gv2vectors_object(image, name=path_vectors.name+'|inside')
    outside_path = outside_gv.gv2vectors_object(image, name=path_vectors.name+'|outside')
    
    action_case = selection_action_options[action_option][1]
    pdb.gimp_image_undo_group_start(image)           # Draw
    if action_case in ('outside', 'both sides'):
        gimp_draw_vectors_object(image, outside_path, visible=True)
    if action_case in ('inside', 'both sides'):
        gimp_draw_vectors_object(image, inside_path, visible=True)
    pdb.gimp_image_undo_group_end(image)           # Draw
    if action_case == 'outside':
        return outside_path
    else:
        return inside_path

# -----------------
# crop_path_by_CorS
# -----------------
# Do the main work for crop_path_by_circle_main or crop_path_by_selection_main
# except for the drawing.
# Args:
# - gv:          BCurve.GimpVectors
# - crop_inputs: CroppingInputData
# Returns:
# - [BCurve.GimpStroke] (inside)
# - [BCurve.GimpStroke] (outside)
def crop_path_by_CorS(gv, crop_inputs):
    inside, outside = gv_split_by_CorS(gv, crop_inputs)
    return [bc.bc2gs() for bc in inside], [bc.bc2gs() for bc in outside]


#======================================================
#                    Registrations
#======================================================

versionnumber = "0.8"
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2021"
image_types = "*"
menupath = '<Vectors>/Tools/Cropping and slicing'


#####################  Crop path by rectangle  #####################

procedure_name  = "crop_path_by_rectangle"
procedure_blurb = ("Crop a path by a rectangle."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Crop a path by a rectangle, given either as a selection or four guides."
procedure_label = "Crop path by rectangle"

procedure_function = crop_path_by_rectangle_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path_vectors", "The path to be cropped", None),
      (PF_OPTION, 'box_option','Cropping rectangle',
                   0,
                  [case[0] for case in crop_box_options]), # Descriptions of cases
    ],
    [
        (PF_VECTORS, "cropped", "cropped path"),
    ],
    procedure_function,
    menu=menupath)

###################  Crop path by convex polygon  #####################

procedure_name  = "crop_path_by_convex_polygon"
procedure_blurb = ("Crop a path by a convex polygon."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Crop a path by a convex polygon, given as a path."
procedure_label = "Crop path by convex polygon"

procedure_function = crop_path_by_convex_polygon_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path_vectors", "The path to be cropped", None),
      (PF_VECTORS, "polygon_vectors",
                   "The polygon (closed path, straight edges, encircles a convex region)",
                   None),
    ],
    [
        (PF_VECTORS, "cropped", "cropped path"),
    ],
    procedure_function,
    menu=menupath)


#####################  Split path by line  #####################
##
##procedure_name  = "split_path_by_line"
##procedure_blurb = ("Split a path by a straight line."
##                   +"\n(Version "+versionnumber+")"
##                   )
##procedure_help  = "Split a path by a line, given as a 2-anchor path."
##procedure_label = "Split path by one line"
##
##procedure_function = split_path_by_line_main
##
##menupath = '<Vectors>/Tools/Crop'
##
##register(
##    procedure_name,
##    procedure_blurb,
##    procedure_help,
##    procedure_author,
##    procedure_copyright,
##    procedure_date,
##    procedure_label,
##    image_types,
##    [
##      (PF_IMAGE, "image", "Input image", None),
##      (PF_VECTORS, "path", "The path to be split", None),
##      (PF_VECTORS, "line", "The line (a 2-anchor path)", None),
##    ],
##    [
##    ],
##    procedure_function,
##    menu=menupath)


###################  Slice path by lines  #####################

procedure_name  = "slice_path_by_lines"
procedure_blurb = ("Slice a path by lines."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Slice a path by lines, given as strokes of a path."
procedure_label = "Slice path by lines"

procedure_function = slice_path_by_lines_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path to be sliced", None),
      (PF_VECTORS, "line", "The slicing lines: strokes of a path", None),
    ],
    [
    ],
    procedure_function,
    menu=menupath)


###################  Slice path by guides  #####################

procedure_name  = "slice_path_by_guides"
procedure_blurb = ("Slice a path by guides."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Slice a path by guides"
procedure_label = "Slice path by guides"

procedure_function = slice_path_by_guides_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path_vectors", "The path to be sliced", None),
      (PF_OPTION, 'guide_option','Cropping box',
                   0,
                  [case[0] for case in slice_guide_options]), # Descriptions of cases
    ],
    [
    ],
    procedure_function,
    menu=menupath)

###################  Crop path by circle  #####################

procedure_name  = "crop_path_by_circle"
procedure_blurb = ("Crop a path by a circle."
                   +"\nThe cropping circle is input as a line segment giving either"
                   +"\n(1)   a diameter of the circle, or"
                   +"\n(2)   the center and one point on the circle."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Crop a path by a circle."
procedure_label = "Crop path by circle"

procedure_function = crop_path_by_circle_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path_vectors", "The path to be cropped", None),
      (PF_VECTORS, "circle_path", "A 2-anchors path to determine the cropping circle", None),
      (PF_OPTION, 'circle_option','The 2-anchors path is:',
                   0,
                  [case[0] for case in circle_options]), # Descriptions of cases
      (PF_OPTION, 'action_option','Action',
                   0,
                  [case[0] for case in circle_action_options]), # Descriptions of cases
      (PF_BOOL, "draw_circle", "Draw the cropping circle?", False),
    ],
    [
        (PF_VECTORS, "cropped", "cropped path"),
    ],
    procedure_function,
    menu=menupath)

###################  Crop path by general selection  #####################

procedure_name  = "crop_path_by_general_selection"
procedure_blurb = ("Crop a path by a general selection."
                   +"\nMay be slow in complicated cases."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = ""
procedure_label = "Crop path by general selection"

procedure_function = crop_path_by_selection_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path_vectors", "The path to be cropped", None),
      (PF_OPTION, 'action_option','Action',
                   0,
                  [case[0] for case in selection_action_options]), # Descriptions of cases
      (PF_FLOAT, "threshold", "Threshold (0..255)", 127),
    ],
    [
        (PF_VECTORS, "cropped", "cropped path"),
    ],
    procedure_function,
    menu=menupath)


main()

