#!/usr/bin/env python
#
# History:
# v0.1: 2021-24-06: Plugin "Join strokes"
# v0.2: 2021-24-06: Developing
# v0.3: 2021-03-07: Options for how to close gaps

# (c) Markku Koppinen 2021
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'

from __future__ import division, print_function
from gimpfu import *
#from math import *
from itertools import product
from copy import deepcopy

#==============================================================
#             class BCurve       
#==============================================================

# Note: The plane is treated as the complex plane.

class BCurve(object):
    """Data structures to handle Gimp vectors objects by means of individual
    Bezier arcs.
    By a "Bezier arc" we mean here a Bezier curve determined by 4 control points
    p0,p1,p2,p3.
    This class contains data structures needed to do computations on
    a Bezier curve which are done arc by arc.
    To this end, Gimp data structures
    - gimp.Vectors
    - gimp.VectorsBezierStroke
    are here represented as data structures (classes)
    - GimpVectors
    - GimpStroke
    - BezierCurve
    - BezierArc
    - ControlPoint = complex
    with the aim that
    - each control point [x,y] is represented as an instance of ControlPoint;
    - each Bezier arc (four control points, see above) is represented as an instance
      of BezierArc.
    """
    
    ControlPoint = complex
    
    class GimpVectors(object):
        """Essentially same data as in a gimp.Vectors object except that
        strokes are instances of GimpStroke rather than
        gimp.VectorsBezierStroke.
        Attributes:
        - stroke_list: [GimpStroke]
        - name:        string
        """
        def __init__(self,
                     stroke_list,            # [GimpStroke]
                     name='GimpVectors'      # string
                     ):
            self.stroke_list = stroke_list
            self.name = name
        def __str__(self):
            s = 'GimpVectors '+self.name
            count = 0
            for stroke in self.stroke_list:
                s += '\n  Stroke '+str(count)+': GimpStroke'
                s += '\n    '+str(stroke)
                count += 1
            return s
        def gv2vectors_object(self, image, name=None):
            """Conversion GimpVectors -> gimp.Vectors
            (The converse: vectors_object2gv)
            """
            def cp_list2xy_list(cp_list):
                xy = []
                for cp in cp_list:
                    xy += [cp.real, cp.imag]
                return xy
            if name is None:
                name = self.name
            vectors = pdb.gimp_vectors_new(image, name)
            for stroke in self.stroke_list:
                closed = stroke.closed
                xy_list = cp_list2xy_list(stroke.cp_list)
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                             vectors, 0, len(xy_list), xy_list, closed)
            return vectors
    
    class GimpStroke(object):
        """Essentially same data as in a gimp.VectorsBezierStroke.points,
        except that control points list [x,y,x,y,...] is arranged as
        a list [ControlPoint] by joining successive x,y.
        Attributes:
        - cp_list:     [ControlPoint]
        - closed:      boolean
        - stroke_name: string
        """
        def __init__(self,
                     cp_list,                   # [ControlPoint]
                     closed,                    # boolean
                     stroke_name = 'GimpStroke' # string
                     ):
            self.stroke_name = stroke_name
            self.cp_list = cp_list
            self.closed = closed
        def __str__(self):
            s = self.stroke_name + '; Control points:'
            for cp in self.cp_list:
                s += '\n    '+str(cp)
            s += '\n    closed: '+str(self.closed)
            return s
        def gs2bc(self):
            """Conversion GimpStroke -> BezierCurve
            """
            head = self.cp_list[0]
            tail = self.cp_list[-1]
            inner = self.cp_list[1:-1]
            if len(inner) > 1: # Must be divisible by 4
                ba_list = []
                count = 0
                for i in range(0,len(inner)-1,3):
                    ba_list.append(BCurve.BezierArc(inner[i:i+4], 'arc '+str(count)))
                    count += 1
            elif len(inner) == 1: # Stroke has only one anchor
                ba_list = [BCurve.BezierArc([inner[0]], 'arc 0')]
            else:
                raise Exception("BCurve.GimpStroke.gs2bc: No anchors in stroke?")
            return BCurve.BezierCurve(
                               bezier_arcs = ba_list,
                               head_handle = head,
                               tail_handle = tail,
                               closed = self.closed,
                               curve_name = self.stroke_name
                               )
    
    class BezierCurve(object):
        """BezierCurve is a list of butting Bezier arcs with some extra data.
        Attributes:
        - curve_name:  string
        - bezier_arcs: [BezierArc];
        - head_handle: ControlPoint;
        - tail_handle: ControlPoint;
        - closed:      boolean.
        Note: "Butting" means that for any pair of successive arcs, the last
              control point of the former equals the first control point of
              the latter.
              In initialization no checks are done about this condition.
        """
        def __init__(self,
                     bezier_arcs=[],           # [BezierArc]
                     head_handle=None,         # ControlPoint
                     tail_handle=None,         # ControlPoint
                     closed=False,             # boolean
                     curve_name='BezierCurve', # string
                     ):
            self.curve_name = curve_name
            self.bezier_arcs = bezier_arcs
            if head_handle is None:
                self.head_handle = bezier_arcs[0].cp4[0]
            else:
                self.head_handle = head_handle
            if tail_handle is None:
                self.tail_handle = bezier_arcs[-1].cp4[-1]
            else:
                self.tail_handle = tail_handle
            self.closed = closed
        def __str__(self):
            s = self.curve_name + ': Bezier curve' + '; Bezier arcs:'
            count = 0
            for arc in self.bezier_arcs:
                s += '\n  arc '+ str(count)+ ': ' +str(arc)
                count += 1
            s += '\n  Head handle:    '+str(self.head_handle)
            s += '\n  Tail handle:    '+str(self.tail_handle)
            s += '\n  Closed:'    +str(self.closed)
            return s
        def bc2gs(self):
            """Conversion BezierCurve -> GimpStroke
            """
            cp_list = [self.head_handle]
            if len(self.bezier_arcs[0].cp4) > 1:
                for ba in self.bezier_arcs:
                    cp_list += ba.cp4[:3]
                cp_list.append(self.bezier_arcs[-1].cp4[-1])
            else:
                # Only one BezierArc and it has cp4 = [p0]
                # (may rise from Gimp stroke with only one anchor).
                cp_list.append(self.bezier_arcs[0].cp4[0])
            cp_list.append(self.tail_handle)
            return BCurve.GimpStroke(cp_list, self.closed, self.curve_name)
        def bc2vectors_object(self, image, name=None):
            """Conversion BezierCurve -> gimp.Vectors
            Makes vectors_object with 1 stroke.
            """
            if name is None:
                name = self.curve_name
            gs = self.bc2gs()             # BCurve.GimpStroke
            gv = BCurve.GimpVectors([gs]) # BCurve.GimpVectors
            return gv.gv2vectors_object(image, name)
    
    class BezierArc(object):
        """Data structure for one Bezier arc: arc determined by 4 control
           points.
        Attributes:
        - cp4:      [p0,p1,p2,p3] where each pi:ControlPoint, or
                    [p0] if only one control point (may rise from
                    a stroke with only one anchor);
        - arc_name: string
        """
        def __init__(self,
                     cp4,                 # cp4=[p0,p1,p2,p3] with pi:ControlPoint
                                          # or [p0]
                     arc_name='BezierArc' # string
                     ):
            self.cp4 = cp4
            self.arc_name = arc_name
        def reverse(self):
            self.cp4.reverse()
        def __str__(self):
            s = 'BezierArc, control points:'
            for cp in self.cp4:
                s += '\n  '+ str(cp)
            return s

# -----------------
# vectors_object2gv
# -----------------
# Conversion gimp.Vectors -> GimpVectors
# Args:
# - vectors_object: gimp.Vectors (Gimp vectors object)
# Returns:
# - GimpVectors
#
# Note: The inverse conversion is GimpVectors.gv2vectors_object(...).
#       See also BezierCurve.bc2vectors_object(...).
def vectors_object2gv(vectors_object):
    def xy_list2cp_list(xy_list): # xy_list = [x,y,x,y,...,]
        cp = []
        for i in range(0,len(xy_list),2):
            cp.append(BCurve.ControlPoint(xy_list[i], xy_list[i+1]))
        return cp
    stroke_list = []
    count = 0
    for stroke in vectors_object.strokes:
        xy_list = stroke.points[0]
        closed = stroke.points[1]
        cp_list = xy_list2cp_list(xy_list)
        stroke_name = vectors_object.name + ' stroke '+str(count)
        stroke_list.append(BCurve.GimpStroke(cp_list, closed, stroke_name))
        count += 1
    return BCurve.GimpVectors(stroke_list, vectors_object.name)


#==============================================================
#             Auxiliary for BCurve
#==============================================================

# Reverse the list of Bezier arcs and inside it each cp4.
# Args:
# - balist: [BCurve.BezierArc]
# Returns:
# - [BCurve.BezierArc]
def reversed_balist(balist):
    return [BCurve.BezierArc(cp4 = ba.cp4[::-1]) for ba in balist][::-1]

#==============================================================
#                           Drawing
#==============================================================
# ------------------------
# gimp_draw_vectors_object
# ------------------------
# The Gimp version of the routine 'drawing_routine_for_bezier':
# Insert into Gimp the input vectors object, and make it visible if
# visible=True.
# Args:
# - image: Gimp's image (obtained usually from GUI);
# - vectors_object: a vectors object in Gimp;
# - visible: Boolean.
# Returns the vectors object in Gimp, and makes it visible if visible=True.
def gimp_draw_vectors_object(image, vectors_object, visible=True):
    # Insert a Gimp vectors object in Gimp.
    # Adapted from http://registry.gimp.org/files/lqr_wpset-0.14.py.txt
    # by Mike Kazantsev.
    def gimp_insert_vectors(image, vectors_object, position=0):
        if gimp.version >= (2, 8, 0):
            # 2.7.0 has gimp-image-insert-vectors, but fails with
            # parent=None
            parent = None
            return pdb.gimp_image_insert_vectors(
                     image, vectors_object, parent, position)
        else:
            return pdb.gimp_image_add_vectors(
                     image, vectors_object, position)# Deprecated!
    gimp_insert_vectors(image, vectors_object)
    vectors_object.visible = visible
    return vectors_object


#==============================================================
#                           Bezier
#==============================================================

def vdot(v,w):
    return v.real*w.real + v.imag*w.imag

# -------------------------
# bezier_arc_end_directions
# -------------------------
# Given a Bezier arc (control points, 0<=t<=1), find, at each end,
# the unit direction vector of the arc.
# Note: this does not mean the derivatives of the parametric representation,
# but non-zero vectors which are tangential to the arc and point in the
# direction where the arc is located when viewed from that end point.
# Return the result as two complex numbers (= two vectors),
# or as None if the arc is one point.
# The plane is viewed as the complex number plane, so points are complex numbers.
# Args:
# - cp4: [complex,complex,complex,complex] (control points)
# Returns: either None (when the arc is one point), or
# - complex (unit direction vector at starting point)
# - complex (unit direction vector at ending point)
def bezier_arc_end_directions(cp4):
    ZERO = 1e-12
    if len(cp4) == 1: # May happen in a 1-anchor stroke
        return None
    try:
        p0,p1,p2,p3 = cp4
    except ValueError:
        raise Exception("bezier_arc_end_directions: Should not happen 1!")
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    if vdot(p01,p01) > ZERO:
        head_dir = p01
    elif vdot(p12,p12) > ZERO:
        head_dir = p12
    elif vdot(p23,p23) > ZERO:
        head_dir = p23
    else: # Arc is one point
        return None
    if vdot(p23,p23) > ZERO:
        tail_dir = -p23
    elif vdot(p12,p12) > ZERO:
        tail_dir = -p12
    elif vdot(p01,p01) > ZERO:
        tail_dir = -p01
    else:
        raise Exception("bezier_arc_end_directions: Should never happen 2!")
    return head_dir/abs(head_dir), tail_dir/abs(tail_dir)


#==============================================================
#                           Join
#==============================================================

def point_in_selection(x,y,image):
    if (0 <= x < image.width) and (0 <= y < image.height):
        _,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
        return (pixel[0] > 127)
    else:
        return False

class BC_work(object):
    """For a stroke:BCurve.BezierCurve record which of its end points are free
    for joining. Here a "free end point" means that the end point is not excluded
    from joining by the following restrictins:
    - end points of a closed stroke are not free;
    - an end point of n open stroke is not free if
      - selection_case='inside' and the end point is not in the selection;
      - selection_case='outside' and the end point is in the selection.
    Attributes:
    - bc: BCurve.BezierCurve
    - condition: string (one of 'none_free',
                                'start_free',
                                'end_free',
                                'both_free',
                                'one_free' (for 1-anchor strokes)
    - free_ends: [] or [(complex,boolean)] or [(comple,boolean), (complex,boolean)]
                 where the boolean value is True if this is the start anchor of bc,
                 False otherwise.
    Methods:
    - __init__
    - __str__
    - join_BC_BC
    - join_BC_ends
    """
    def __init__(self, bc, selection_case, image):
        self.bc = bc
        if bc.closed:
            self.condition = 'none_free'
            self.free_ends = []
        elif len(bc.bezier_arcs[0].cp4) == 1: # 1-anchor stroke:
            anchor = bc.bezier_arcs[0].cp4[0]
            x,y = anchor.real, anchor.imag
            anchor_in_selection = point_in_selection(x,y,image)
            if selection_case == 'ignore':
                self.condition = 'one_free'
                self.free_ends = [(anchor, True)]
            elif selection_case == 'inside':
                if anchor_in_selection:
                    self.condition = 'one_free'
                    self.free_ends = [(anchor, True)]
                else:
                    self.condition = 'none_free'
                    self.free_ends = []
            elif selection_case == 'outside':
                if not anchor_in_selection:
                    self.condition = 'one_free'
                    self.free_ends = [(anchor, True)]
                else:
                    self.condition = 'none_free'
                    self.free_ends = []
            else:
                raise Exception("Unknown selection_case: "+selection_case)
        else:
            if selection_case == 'ignore':
                self.condition = 'both_free'
                self.free_ends = [(bc.bezier_arcs[0].cp4[0], True),
                                  (bc.bezier_arcs[-1].cp4[-1], False)]
            else:
                start, end = bc.bezier_arcs[0].cp4[0], bc.bezier_arcs[-1].cp4[-1]
                x,y = start.real, start.imag
                start_in_selection = point_in_selection(x,y,image)
                x,y = end.real, end.imag
                end_in_selection = point_in_selection(x,y,image)
                if selection_case == 'inside': 
                    if start_in_selection and end_in_selection:
                        self.condition = 'both_free'
                        self.free_ends = [(bc.bezier_arcs[0].cp4[0], True),
                                          (bc.bezier_arcs[-1].cp4[-1], False)]
                    elif start_in_selection:
                        self.condition = 'start_free'
                        self.free_ends = [(bc.bezier_arcs[0].cp4[0], True)]
                    elif end_in_selection:
                        self.condition = 'end_free'
                        self.free_ends = [(bc.bezier_arcs[-1].cp4[-1], False)]
                    else:
                        self.condition = 'none_free'
                        self.free_ends = []
                elif selection_case == 'outside': 
                    if not(start_in_selection or end_in_selection):
                        self.condition = 'both_free'
                        self.free_ends = [(bc.bezier_arcs[0].cp4[0], True),
                                          (bc.bezier_arcs[-1].cp4[-1], False)]
                    elif not start_in_selection:
                        self.condition = 'start_free'
                        self.free_ends = [(bc.bezier_arcs[0].cp4[0], True)]
                    elif not end_in_selection:
                        self.condition = 'end_free'
                        self.free_ends = [(bc.bezier_arcs[-1].cp4[-1], False)]
                    else:
                        self.condition = 'none_free'
                        self.free_ends = []
                else:
                    raise Exception("Unknown selection_case: "+selection_case)
    # ----------
    # join_BC_BC
    # ----------
    # Join two objects:BC_work. They are assumed to be butting up to allowed tolerance
    # (not checked here).
    # Ordering: First 'self', then 'other' (either one possibly reverted).
    # Where the two are butting, average of the two anchors will be used as the new
    # anchor, and the adjacent handles are moved accordingly.
    # Args:
    # - self: BC_work
    # - other: BC_work
    # - close_gap_case: string
    # - selection_case: string
    # - image
    # Returns:
    # - BC_work
    def join_BC_BC(self, other, close_gap_case, selection_case, image):
        self_ends = self.free_ends
        other_ends = other.free_ends
        end_pairs = list(product(self_ends,other_ends))
        closest = min(end_pairs, key=(lambda p:abs(p[0][0]-p[1][0])))
        # First check if self or other is a 1-anchor stroke:
        if len(self.bc.bezier_arcs[0].cp4) == len(other.bc.bezier_arcs[0].cp4) == 1:
            # Both are 1-anchor strokes
            left_anchor = self.bc.bezier_arcs[0].cp4[0]
            right_anchor = other.bc.bezier_arcs[0].cp4[0]
            if close_gap_case == 'average':
                move = (-left_anchor + right_anchor)/2
                new_anchor = left_anchor + move # = (left_anchor + right_anchor)/2
                new_bezier_arc = BCurve.BezierArc(cp4=[new_anchor])
                joined_bc = BCurve.BezierCurve(bezier_arcs = [new_bezier_arc],
                                               head_handle = new_anchor,
                                               tail_handle = new_anchor,
                                               closed = False
                                               )
            elif close_gap_case in ('smooth','straight'): # Make straight edge
                new_bezier_arc = BCurve.BezierArc(cp4=[left_anchor, left_anchor, right_anchor, right_anchor])
                joined_bc = BCurve.BezierCurve(bezier_arcs = [new_bezier_arc],
                                               head_handle = left_anchor,
                                               tail_handle = right_anchor,
                                               closed = False
                                               )
            else:
                raise Exception("BC_work: join_BC_BC: Unknown close_gap_case: "+close_gap_case)
            joined_BC = BC_work(joined_bc, selection_case, image)
            return joined_BC
        elif len(self.bc.bezier_arcs[0].cp4) == 1:
            # Self is 1-anchor stroke, other is not
            other_start = closest[1][1] # True: start; False: end
            if other_start:
                right_balist = deepcopy(other.bc.bezier_arcs)
                tail_handle = other.bc.tail_handle
            else:
                right_balist = reversed_balist(other.bc.bezier_arcs)
                tail_handle = other.bc.head_handle
            left_anchor = self.bc.bezier_arcs[0].cp4[0]
            right_anchor = right_balist[0].cp4[0]
            if close_gap_case == 'average':
                move = (-left_anchor + right_anchor)/2
                new_anchor = left_anchor + move # = (left_anchor + right_anchor)/2
                right_balist[0].cp4[0:2] = [new_anchor, right_balist[0].cp4[1] - move]
                joined_bc = BCurve.BezierCurve(bezier_arcs = right_balist,
                                               head_handle = new_anchor,
                                               tail_handle = tail_handle,
                                               closed = False
                                               )
            elif close_gap_case == 'smooth':
                right_start_direction,_ = bezier_arc_end_directions(right_balist[0].cp4)
                gap_length = abs(left_anchor-right_anchor)
                new_ba = BCurve.BezierArc(cp4=[left_anchor,
                                          left_anchor,
                                          right_anchor - right_start_direction * gap_length / 3,
                                          right_anchor
                                          ])
                joined_bc = BCurve.BezierCurve(bezier_arcs = [new_ba] + right_balist,
                                               head_handle = left_anchor,
                                               tail_handle = tail_handle,
                                               closed = False
                                               )
            elif close_gap_case == 'straight':
                new_ba = BCurve.BezierArc(cp4=[left_anchor,
                                               left_anchor,
                                               right_anchor,
                                               right_anchor
                                               ])
                joined_bc = BCurve.BezierCurve(bezier_arcs = [new_ba] + right_balist,
                                               head_handle = left_anchor,
                                               tail_handle = tail_handle,
                                               closed = False
                                               )
            joined_BC = BC_work(joined_bc, selection_case, image)
            return joined_BC
        elif len(other.bc.bezier_arcs[0].cp4) == 1:
            # Other is 1-anchor stroke, self is not
            self_start = closest[0][1] # True: start; False: end
            if self_start:
                left_balist = reversed_balist(self.bc.bezier_arcs)
                head_handle = self.bc.tail_handle
            else:
                left_balist = deepcopy(self.bc.bezier_arcs)
                head_handle = self.bc.head_handle
            left_anchor = left_balist[-1].cp4[-1]
            right_anchor = other.bc.bezier_arcs[0].cp4[0]
            if close_gap_case == 'average':
                move = (-left_anchor + right_anchor)/2
                new_anchor = left_anchor + move # = (left_anchor + right_anchor)/2
                left_balist[-1].cp4[-2:] = [left_balist[-1].cp4[2] + move, new_anchor]
                joined_bc = BCurve.BezierCurve(bezier_arcs = left_balist,
                                               head_handle = head_handle,
                                               tail_handle = new_anchor,
                                               closed = False
                                               )
            elif close_gap_case == 'smooth':
                _,left_end_direction = bezier_arc_end_directions(left_balist[-1].cp4)
                gap_length = abs(left_anchor-right_anchor)
                new_ba = BCurve.BezierArc(cp4=[left_anchor,
                                               left_anchor - left_end_direction * gap_length / 3,
                                               right_anchor,
                                               right_anchor
                                               ])
                joined_bc = BCurve.BezierCurve(bezier_arcs = left_balist + [new_ba],
                                               head_handle = head_handle,
                                               tail_handle = right_anchor,
                                               closed = False
                                               )
            elif close_gap_case == 'straight':
                new_ba = BCurve.BezierArc(cp4=[left_anchor,
                                               left_anchor,
                                               right_anchor,
                                               right_anchor
                                               ])
                joined_bc = BCurve.BezierCurve(bezier_arcs = left_balist + [new_ba],
                                               head_handle = head_handle,
                                               tail_handle = right_anchor,
                                               closed = False
                                               )
            joined_BC = BC_work(joined_bc, selection_case, image)
            return joined_BC
        # Now neither is 1-anchor stroke.
        self_start = closest[0][1] # True: start; False: end
        other_start = closest[1][1] # True: start; False: end
        if self_start:
            left_balist = reversed_balist(self.bc.bezier_arcs)
            head_handle = self.bc.tail_handle
        else:
            left_balist = deepcopy(self.bc.bezier_arcs)
            head_handle = self.bc.head_handle
        if other_start:
            right_balist = deepcopy(other.bc.bezier_arcs)
            tail_handle = other.bc.tail_handle
        else:
            right_balist = reversed_balist(other.bc.bezier_arcs)
            tail_handle = other.bc.head_handle
        left_anchor = left_balist[-1].cp4[-1]
        right_anchor = right_balist[0].cp4[0]
        
        if close_gap_case == 'average':
            move = (-left_anchor + right_anchor)/2
            new_anchor = left_anchor + move # = (left_anchor + right_anchor)/2
            left_balist[-1].cp4[2:4] = [left_balist[-1].cp4[2] + move, new_anchor]
            right_balist[0].cp4[0:2] = [new_anchor, right_balist[0].cp4[1] - move]
            joined_bc = BCurve.BezierCurve(bezier_arcs = left_balist + right_balist,
                                           head_handle = head_handle,
                                           tail_handle = tail_handle,
                                           closed = False
                                           )
        
        elif close_gap_case == 'smooth':
            _,left_end_direction = bezier_arc_end_directions(left_balist[-1].cp4)
            right_start_direction,_ = bezier_arc_end_directions(right_balist[0].cp4)
            gap_length = abs(left_anchor-right_anchor)
            new_ba = BCurve.BezierArc(cp4=[left_anchor,
                                           left_anchor - left_end_direction * gap_length / 3,
                                           right_anchor - right_start_direction * gap_length / 3,
                                           right_anchor
                                           ])
            joined_bc = BCurve.BezierCurve(bezier_arcs = left_balist + [new_ba] + right_balist,
                                           head_handle = head_handle,
                                           tail_handle = tail_handle,
                                           closed = False
                                           )
        elif close_gap_case == 'straight':
            new_ba = BCurve.BezierArc(cp4=[left_anchor,
                                           left_anchor,
                                           right_anchor,
                                           right_anchor
                                           ])
            joined_bc = BCurve.BezierCurve(bezier_arcs = left_balist + [new_ba] + right_balist,
                                           head_handle = head_handle,
                                           tail_handle = tail_handle,
                                           closed = False
                                           )
        else:
            raise Exception("BC_work: join_BC_ends: Unknown close_gap_case: "+close_gap_case)
        
        joined_BC = BC_work(joined_bc, selection_case, image)
        return joined_BC
    # ------------
    # join_BC_ends
    # ------------
    # Join ends of one object:BC_work. The ends are assumed to be close enough
    # up to allowed tolerance (not checked here).
    # The average of the two end anchors will be used as the new anchor,
    # and the head and tail handles are moved accordingly.
    # - self: BC_work
    # - close_gap_case: string
    # - selection_case: string
    # - image
    def join_BC_ends(self, close_gap_case, selection_case, image):
        balist = self.bc.bezier_arcs
        first_anchor = balist[0].cp4[0]
        last_anchor = balist[-1].cp4[-1]
        
        if close_gap_case == 'average':
            move = (-first_anchor + last_anchor)/2
            new_first_anchor = first_anchor + move # = (first_anchor + last_anchor)/2
            if len(balist) == 1:
                new_ba = BCurve.BezierArc(cp4=[new_first_anchor])
                handle1, handle2 = balist[0].cp4[1:3]
                new_head_handle = handle1 + move
                new_tail_handle = handle2 - move
                joined_bc = BCurve.BezierCurve(bezier_arcs = [new_ba],
                                               head_handle = new_head_handle,
                                               tail_handle = new_tail_handle,
                                               closed = True
                                               )
            else:
                balist[0].cp4[0:2] = [new_first_anchor, balist[0].cp4[1] + move]
                new_head_handle = balist[-1].cp4[2] - move
                new_tail_handle = balist[-1].cp4[1]
                new_balist = balist[:-1] # Drop last arc
                joined_bc = BCurve.BezierCurve(bezier_arcs = deepcopy(new_balist),
                                               head_handle = new_head_handle,
                                               tail_handle = new_tail_handle,
                                               closed = True
                                               )
        elif close_gap_case == 'smooth':
            head_direction,_ = bezier_arc_end_directions(balist[0].cp4)
            _,tail_direction = bezier_arc_end_directions(balist[-1].cp4)
            gap_length = abs(first_anchor-last_anchor)
            new_head_handle = first_anchor - head_direction * gap_length / 3
            new_tail_handle = last_anchor - tail_direction * gap_length / 3
            joined_bc = BCurve.BezierCurve(bezier_arcs = deepcopy(balist),
                                           head_handle = new_head_handle,
                                           tail_handle = new_tail_handle,
                                           closed = True
                                           )
        elif close_gap_case == 'straight':
            joined_bc = BCurve.BezierCurve(bezier_arcs = deepcopy(balist),
                                           head_handle = first_anchor,
                                           tail_handle = last_anchor,
                                           closed = True
                                           )
        else:
            raise Exception("BC_work: join_BC_ends: Unknown close_gap_case: "+close_gap_case)
        joined_BC = BC_work(joined_bc, selection_case, image)
        return joined_BC
    def __str__(self):
        s = 'BC_work: '+self.condition
        bc = self.bc
        s += '\n'+str(bc.bezier_arcs[0].cp4[0]) + '-->'+str(bc.bezier_arcs[-1].cp4[-1])
        s += '\ncondition: '+self.condition
        s += '\nfree_ends: '+str(self.free_ends)
        return s


# ------------
# join_BC_list
# ------------
# Join strokes in [BC_work] where the ends are close enough.
# Note: There may be clashes: if more than two stroke ends are close, the
# joining is done so that the closest pair is joined first, then the second
# closest, and so on.
# Args:
# - BC_list: [BC_work]
# - distance: float # pixels: join ends closer than this limit
# - close_gap_case: string
# - selection_case: string
# Returns:
# - [BC_work]
def join_BC_list(BC_list, distance, close_gap_case, selection_case, image):
    BC_finished_list = []
    BC_unfinished_list = []
    for BC in BC_list:
        if BC.condition == 'none_free':
            BC_finished_list.append(BC)
        else:
            BC_unfinished_list.append(BC)
    while len(BC_unfinished_list) != 0: # Main loop
        # Among free ends, find the closest pair:
        all_free_ends = []              # all_free_ends: [(complex,int)]
        for BC_index in range(len(BC_unfinished_list)):
            BC = BC_unfinished_list[BC_index]
            for free,_ in BC.free_ends: # free: complex
                all_free_ends.append((free, BC_index))
        closest = None                  # closest: (complex,int,int)
        for k in range(len(all_free_ends)):
            free1, BC_index1 = all_free_ends[k]
            for kk in range(k+1, len(all_free_ends)):
                free2, BC_index2 = all_free_ends[kk]
                dist = abs(free1 - free2)
                if dist > distance: # reject
                    continue
                if closest is None:
                    closest = (dist, BC_index1, BC_index2)
                elif dist < closest[0]:
                    closest = (dist, BC_index1, BC_index2)
        found_closest = not(closest is None)
        if found_closest:
            index1, index2 = closest[1:]
            BC1 = BC_unfinished_list[index1]
            BC2 = BC_unfinished_list[index2]
        if not found_closest: # No acceptable: move all unfinished to finished and break
            BC_finished_list += BC_unfinished_list
            break # while
        # Now closest was found: join:
        if index1 == index2: # Main call: same stroke, join its ends.
            BC_joined = BC1.join_BC_ends(close_gap_case, selection_case, image)
            drop_indices = [index1]
        else: # Main call: different strokes, join them together
            BC_joined = BC1.join_BC_BC(BC2, close_gap_case, selection_case, image)
            drop_indices = [index1, index2]
        if BC_joined.condition == 'none_free':
            BC_finished_list.append(BC_joined)
        else:
            BC_unfinished_list.append(BC_joined)
        # Drop items in 'drop_indices' from unfinished list:
        BC_unfinished_list = [BC_unfinished_list[k]
                              for k in range(len(BC_unfinished_list))
                              if not (k in drop_indices)]
    return BC_finished_list

#==============================================================
#                      Main procedure
#==============================================================
# -----------------
# join_strokes_main
# -----------------
# Args:
# - image
# - path:             gimp.Vectors
# - distance:         float
# - close_gap_option: integer
# - selection_option: integer
# Returns:
# - gimp.Vectors
def join_strokes_main(image, path, distance, close_gap_option, selection_option):
    close_gap_case = close_gap_options[close_gap_option][1]
    selection_case = selection_options[selection_option][1]
    #
    gv = vectors_object2gv(path)    # BCurve.GimpVectors
    gs_list = gv.stroke_list        # [BCurve.GimpStroke]
    bc_list = [gs.gs2bc() for gs in gs_list] # [BCurve.BezierCurve]
    BC_list = []
    for bc in bc_list:
        BC_list.append(BC_work(bc, selection_case, image))
    new_BC_list = join_BC_list(BC_list,   # Main call
                               distance,
                               close_gap_case,
                               selection_case,
                               image
                               )
    new_gs_list = [BC.bc.bc2gs() for BC in new_BC_list]
    new_gv = BCurve.GimpVectors(stroke_list = new_gs_list)
    joined_path = new_gv.gv2vectors_object(image, name = path.name+'|join')
    pdb.gimp_image_undo_group_start(image)
    gimp_draw_vectors_object(image, joined_path, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return joined_path


selection_options = [ # (description, identifier)
        ('Ignore any selection',                         'ignore'),
        ('Join only those ends which are inside the selection',      'inside'),
        ('Join only those ends which are outside of the selection',  'outside'),
        ]

close_gap_options = [ # (description, identifier)
        ('Replace the two end anchors with their average',  'average'),
        ('Join the two ends with a smooth arc',                 'smooth'),
        ('Join the two ends with a straight edge',              'straight'),
        ]


#======================================================
#                    Registration
#======================================================

versionnumber = "0.3"
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2021"
image_types = "*"
menupath = '<Vectors>/Tools/Modify path'

procedure_name  = "join_strokes"
procedure_blurb = ("Join strokes having ends close to each other."
                   +"\nClose open strokes having ends close to each other."
                   +"\n\nNote: If you use a selection, out-of-canvas anchors"
                   +"\nare taken to be outside of the selection."
                   +"\nIf you need to include out-of-canvas anchors in the selection,"
                   +"\nyou have to enlarge the canvas temporarily first."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = ("Join strokes whose ends are close to each other."
                   +"\nClose open strokes whose ends are close to each other."
                   +"\nThree ways to do the joining: make an average anchor;"
                   +"\nmake a smooth arc; make a straight edge."
                   +"\nThe effect can be restricted by means of a selection."
                   )

procedure_label = "Join strokes"

procedure_function = join_strokes_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path", None),
      (PF_FLOAT, "distance",
                 ("Join stroke ends closer than ... (pixels)"
                  #+"\n..."
                  ),
                  0.5),
      (PF_OPTION, 'close_gap_option','How to close the gaps when joining?',
                   0,
                  [case[0] for case in close_gap_options]), # Descriptions of cases
      (PF_OPTION, 'selection_option','How to use the selection?',
                   0,
                  [case[0] for case in selection_options]), # Descriptions of cases
    ],
    [
        (PF_VECTORS, "joined_path", "Path with strokes joined"),
    ],
    procedure_function,
    menu=menupath)


main()
