#!/usr/bin/env python

# Gimp plugin to smooth paths consisting of straight line segments.
# Implemented is the simple idea presented in:

# https://web.archive.org/web/20190327001540/http://antigrain.com/research/bezier_interpolation/index.html#PAGE_BEZIER_INTERPOLATION
#
# History:
# v0.1: 2021-05-06: Plugin "Simple smooth"
# v0.2: 2021-27-06: Bug fix (using selection + anchor outside of canvas)
# v0.3: 2021-28-06: Better wording in the GUI.

# (c) Markku Koppinen 2020, 2021
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'

from __future__ import division, print_function
from gimpfu import *
from math import *
from copy import deepcopy

#==============================================================
#             class BCurve       
#==============================================================

# Note: The plane is treated as the complex plane.

class BCurve(object):
    """Data structures to handle Gimp vectors objects by means of individual
    Bezier arcs.
    By a "Bezier arc" we mean here a Bezier curve determined by 4 control points
    p0,p1,p2,p3.
    This class contains data structures needed to do computations on
    a Bezier curve which are done arc by arc.
    To this end, Gimp data structures
    - gimp.Vectors
    - gimp.VectorsBezierStroke
    are here represented as data structures (classes)
    - GimpVectors
    - GimpStroke
    - BezierCurve
    - BezierArc
    - ControlPoint = complex
    with the aim that
    - each control point [x,y] is represented as an instance of ControlPoint;
    - each Bezier arc (four control points, see above) is represented as an instance
      of BezierArc.
    """
    
    ControlPoint = complex
    
    class GimpVectors(object):
        """Essentially same data as in a gimp.Vectors object except that
        strokes are instances of GimpStroke rather than
        gimp.VectorsBezierStroke.
        Attributes:
        - stroke_list: [GimpStroke]
        - name:        string
        """
        def __init__(self,
                     stroke_list,            # [GimpStroke]
                     name='GimpVectors'      # string
                     ):
            self.stroke_list = stroke_list
            self.name = name
        def __str__(self):
            s = 'GimpVectors '+self.name
            count = 0
            for stroke in self.stroke_list:
                s += '\n  Stroke '+str(count)+': GimpStroke'
                s += '\n    '+str(stroke)
                count += 1
            return s
        def gv2vectors_object(self, image, name=None):
            """Conversion GimpVectors -> gimp.Vectors
            (The converse: vectors_object2gv)
            """
            def cp_list2xy_list(cp_list):
                xy = []
                for cp in cp_list:
                    xy += [cp.real, cp.imag]
                return xy
            if name is None:
                name = self.name
            vectors = pdb.gimp_vectors_new(image, name)
            for stroke in self.stroke_list:
                closed = stroke.closed
                xy_list = cp_list2xy_list(stroke.cp_list)
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                             vectors, 0, len(xy_list), xy_list, closed)
            return vectors
    
    class GimpStroke(object):
        """Essentially same data as in a gimp.VectorsBezierStroke.points,
        except that control points list [x,y,x,y,...] is arranged as
        a list [ControlPoint] by joining successive x,y.
        Attributes:
        - cp_list:     [ControlPoint]
        - closed:      boolean
        - stroke_name: string
        """
        def __init__(self,
                     cp_list,                   # [ControlPoint]
                     closed,                    # boolean
                     stroke_name = 'GimpStroke' # string
                     ):
            self.stroke_name = stroke_name
            self.cp_list = cp_list
            self.closed = closed
        def __str__(self):
            s = self.stroke_name + '; Control points:'
            for cp in self.cp_list:
                s += '\n    '+str(cp)
            s += '\n    closed: '+str(self.closed)
            return s
        def gs2bc(self):
            """Conversion GimpStroke -> BezierCurve
            """
            head = self.cp_list[0]
            tail = self.cp_list[-1]
            inner = self.cp_list[1:-1]
            if len(inner) > 1: # Must be divisible by 4
                ba_list = []
                count = 0
                for i in range(0,len(inner)-1,3):
                    ba_list.append(BCurve.BezierArc(inner[i:i+4], 'arc '+str(count)))
                    count += 1
            elif len(inner) == 1: # Stroke has only one anchor
                ba_list = [BCurve.BezierArc([inner[0]], 'arc 0')]
            else:
                raise Exception("BCurve.GimpStroke.gs2bc: No anchors in stroke?")
            return BCurve.BezierCurve(
                               bezier_arcs = ba_list,
                               head_handle = head,
                               tail_handle = tail,
                               closed = self.closed,
                               curve_name = self.stroke_name
                               )
    
    class BezierCurve(object):
        """BezierCurve is a list of butting Bezier arcs with some extra data.
        Attributes:
        - curve_name:  string
        - bezier_arcs: [BezierArc];
        - head_handle: ControlPoint;
        - tail_handle: ControlPoint;
        - closed:      boolean.
        Note: "Butting" means that for any pair of successive arcs, the last
              control point of the former equals the first control point of
              the latter.
              In initialization no checks are done about this condition.
        """
        def __init__(self,
                     bezier_arcs=[],           # [BezierArc]
                     head_handle=None,         # ControlPoint
                     tail_handle=None,         # ControlPoint
                     closed=False,             # boolean
                     curve_name='BezierCurve', # string
                     ):
            self.curve_name = curve_name
            self.bezier_arcs = bezier_arcs
            if head_handle is None:
                self.head_handle = bezier_arcs[0].cp4[0]
            else:
                self.head_handle = head_handle
            if tail_handle is None:
                self.tail_handle = bezier_arcs[-1].cp4[-1]
            else:
                self.tail_handle = tail_handle
            self.closed = closed
        def __str__(self):
            s = self.curve_name + ': Bezier curve' + '; Bezier arcs:'
            count = 0
            for arc in self.bezier_arcs:
                s += '\n  arc '+ str(count)+ ': ' +str(arc)
                count += 1
            s += '\n  Head handle:    '+str(self.head_handle)
            s += '\n  Tail handle:    '+str(self.tail_handle)
            s += '\n  Closed:'    +str(self.closed)
            return s
        def bc2gs(self):
            """Conversion BezierCurve -> GimpStroke
            """
            cp_list = [self.head_handle]
            if len(self.bezier_arcs[0].cp4) > 1:
                for ba in self.bezier_arcs:
                    cp_list += ba.cp4[:3]
                cp_list.append(self.bezier_arcs[-1].cp4[-1])
            else:
                # Only one BezierArc and it has cp4 = [p0]
                # (may rise from Gimp stroke with only one anchor).
                cp_list.append(self.bezier_arcs[0].cp4[0])
            cp_list.append(self.tail_handle)
            return BCurve.GimpStroke(cp_list, self.closed, self.curve_name)
        def bc2vectors_object(self, image, name=None):
            """Conversion BezierCurve -> gimp.Vectors
            Makes vectors_object with 1 stroke.
            """
            if name is None:
                name = self.curve_name
            gs = self.bc2gs()             # BCurve.GimpStroke
            gv = BCurve.GimpVectors([gs]) # BCurve.GimpVectors
            return gv.gv2vectors_object(image, name)
    
    class BezierArc(object):
        """Data structure for one Bezier arc: arc determined by 4 control
           points.
        Attributes:
        - cp4:      [p0,p1,p2,p3] where each pi:ControlPoint, or
                    [p0] if only one control point (may rise from
                    a stroke with only one anchor);
        - arc_name: string
        """
        def __init__(self,
                     cp4,                 # cp4=[p0,p1,p2,p3] with pi:ControlPoint
                                          # or [p0]
                     arc_name='BezierArc' # string
                     ):
            self.cp4 = cp4
            self.arc_name = arc_name
        def reverse(self):
            self.cp4.reverse()
        def __str__(self):
            s = 'BezierArc, control points:'
            for cp in self.cp4:
                s += '\n  '+ str(cp)
            return s

# -----------------
# vectors_object2gv
# -----------------
# Conversion gimp.Vectors -> GimpVectors
# Args:
# - vectors_object: gimp.Vectors (Gimp vectors object)
# Returns:
# - GimpVectors
#
# Note: The inverse conversion is GimpVectors.gv2vectors_object(...).
#       See also BezierCurve.bc2vectors_object(...).
def vectors_object2gv(vectors_object):
    def xy_list2cp_list(xy_list): # xy_list = [x,y,x,y,...,]
        cp = []
        for i in range(0,len(xy_list),2):
            cp.append(BCurve.ControlPoint(xy_list[i], xy_list[i+1]))
        return cp
    stroke_list = []
    count = 0
    for stroke in vectors_object.strokes:
        xy_list = stroke.points[0]
        closed = stroke.points[1]
        cp_list = xy_list2cp_list(xy_list)
        stroke_name = vectors_object.name + ' stroke '+str(count)
        stroke_list.append(BCurve.GimpStroke(cp_list, closed, stroke_name))
        count += 1
    return BCurve.GimpVectors(stroke_list, vectors_object.name)


#==============================================================
#             Auxiliary
#==============================================================

# v dot (w perpendicular)
def cross(v,w):
    return (v*w.conjugate()).imag


# Given a triplet a,b,c of points, is b on the line segment from a to c?
# Args:
# - triplet: [complex,complex,complex]
# - allowed_deviation: float (pixels)
# Returns:
# - boolean
def straight_triplet(triplet, allowed_deviation=1):
    a,b,c = triplet
    if abs(cross(-a+b, -a+c)) > abs(-a+c) * allowed_deviation:
        return False
    return (abs(-a+b) <= abs(-a+c)) and (abs(-c+b) <= abs(-a+c))

#==============================================================
#                       Pre- and postprocessing
#==============================================================

def point_in_selection(x,y,image):
    if (0 <= x < image.width) and (0 <= y < image.height):
        _,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
        return (pixel[0] > 127)
    else:
        return False
# ---------------------
# remove_excess_anchors
# ---------------------
# From a line figure (all arcs are assumed to be straight lines) remove
# excess anchors: Combine successive arcs into one if they are on a straight line
# and going in the same direction.
# Args:
# - cp4list: [[complex,complex,complex,complex]]
# - closed:  boolean
# - allowed_deviation: float (pixels)
# Returns:
# - new_cp4list: [[complex,complex,complex,complex]]
def remove_excess_anchors(cp4list, closed=False, allowed_deviation=1):
    if len(cp4list) <= 1:
        return cp4list
    anchors = [cp4[0] for cp4 in cp4list] + [cp4list[-1][-1]]
    new_anchors = anchors[:2]
    for c in anchors[2:]:
        a,b = new_anchors[-2:]
        if straight_triplet([a,b,c], allowed_deviation):
            new_anchors[-1] = c
        else:
            new_anchors.append(c)
    new_cp4list = []
    for i in range(1,len(new_anchors)):
        p0,p3 = new_anchors[i-1:i+1]
        new_cp4list.append([p0,p0,p3,p3])
    if closed: # Fix the gap-closing arcs:
        p0,p1,p2,p3 = new_cp4list[0] # first cp4
        q0,q1,q2,q3 = new_cp4list[-1] # last cp4
        close_gap_cp4 = [q0,q0,p3,p3]
        new_cp4list[0] = close_gap_cp4
        new_cp4list[-1] = close_gap_cp4
    return new_cp4list

# ---------
# subdivide
# ---------
# A preliminary  step in the rounding algorithm: Subdivide the given
# cp4list at certain anchors.
# Later, each part will be processed separately. This causes that
# the subdivision points (anchors) will not be altered in the process.
# The subdivision points are:
# - None if rounding_parameters.selection_case = 'ignore'
# - anchors not in the selection if rounding_parameters.selection_case = 'inside'
# - anchors in the selection if rounding_parameters.selection_case = 'outside'
# Args:
# - image
# - cp4list:             [[complex,complex,complex,complex]] (butting Bezier arcs)
# - simplify_parameters: SimplifyParameters
# - print_info:          boolean
# Returns:
# - [[[complex,complex,complex,complex]]] (butting list of butting Bezier arcs)
def subdivide(image, cp4list, selection_case):
    ZERO = 1e-8
    # Make a list to record for each cp4 whether its start and end points
    # will be subdivision points
    # (This brings some redundancy since almost every anchor appears in two cp4's.)
    subdivision_anchors = [[False,False] for i in range(len(cp4list))] # Initially all False
    # Subdivision points: Points in the selection or outside of it.
    if selection_case == 'ignore':
        #return [cp4list]
        pass
    elif selection_case == 'inside':
        # Points not in the selection are made subdivision points.
        for i in range(len(cp4list)):
            cp4 = cp4list[i]
            x,y = cp4[0].real, cp4[0].imag
            #_,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
            #if pixel[0] <= 127: # Not in selection
            if not point_in_selection(x,y,image):
                subdivision_anchors[i][0] = True
            x,y = cp4[-1].real, cp4[-1].imag
            #_,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
            #if pixel[0] <= 127: # Not in selection
            if not point_in_selection(x,y,image):
                subdivision_anchors[i][1] = True
    elif selection_case == 'outside':
        # Points in the selection are made subdivision points.
        for i in range(len(cp4list)):
            cp4 = cp4list[i]
            x,y = cp4[0].real, cp4[0].imag
            #_,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
            #if pixel[0] > 127: # In selection
            if point_in_selection(x,y,image):
                subdivision_anchors[i][0] = True
            x,y = cp4[-1].real, cp4[-1].imag
            #_,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
            #if pixel[0] > 127: # In selection
            if point_in_selection(x,y,image):
                subdivision_anchors[i][1] = True
    else:
        raise Exception("subdivide: Unknown selection_case: "+selection_case)
    # Now the subdivision points are marked. Do the subdivision:
    split_happens = False
    cp4list_list = []
    cumulate_cp4list = []
    for i in range(len(cp4list)):
        cp4 = cp4list[i]
        split_at_start, split_at_end = subdivision_anchors[i]
        if split_at_start and split_at_end:
            split_happens = True
            cp4list_list.append(cumulate_cp4list)
            cp4list_list.append([cp4])
            cumulate_cp4list = []
        elif split_at_start:
            split_happens = True
            cp4list_list.append(cumulate_cp4list)
            cumulate_cp4list = [cp4]
        elif split_at_end:
            split_happens = True
            cp4list_list.append(cumulate_cp4list + [cp4])
            cumulate_cp4list = []
        else:
            cumulate_cp4list.append(cp4)
    if subdivision_anchors[-1][1] == False:
        cp4list_list.append(cumulate_cp4list)
    cp4list_list = [x for x in cp4list_list if len(x) > 0]
    return cp4list_list


#==============================================================
#                           Drawing
#==============================================================
# ------------------------
# gimp_draw_vectors_object
# ------------------------
# The Gimp version of the routine 'drawing_routine_for_bezier':
# Insert into Gimp the input vectors object, and make it visible if
# visible=True.
# Args:
# - image: Gimp's image (obtained usually from GUI);
# - vectors_object: a vectors object in Gimp;
# - visible: Boolean.
# Returns the vectors object in Gimp, and makes it visible if visible=True.
def gimp_draw_vectors_object(image, vectors_object, visible=True):
    # Insert a Gimp vectors object in Gimp.
    # Adapted from http://registry.gimp.org/files/lqr_wpset-0.14.py.txt
    # by Mike Kazantsev.
    def gimp_insert_vectors(image, vectors_object, position=0):
        if gimp.version >= (2, 8, 0):
            # 2.7.0 has gimp-image-insert-vectors, but fails with
            # parent=None
            parent = None
            return pdb.gimp_image_insert_vectors(
                     image, vectors_object, parent, position)
        else:
            return pdb.gimp_image_add_vectors(
                     image, vectors_object, position)# Deprecated!
    gimp_insert_vectors(image, vectors_object)
    vectors_object.visible = visible
    return vectors_object

# For debugging and experimenting only
def draw_cp4list(image, cp4list, name='cp4list'):
    balist = [BCurve.BezierArc(cp4=cp4) for cp4 in cp4list]
    bc = BCurve.BezierCurve(bezier_arcs = balist)
    vectors = bc.bc2vectors_object(image, name=name)
    gimp_draw_vectors_object(image, vectors, visible=True)

#==============================================================
#                         Simple smooth
#==============================================================

# ---------------------
# simple_smooth_cp4list
# ---------------------
# From a cp4list make new cp4list with corners rounded.
# Args:
# - cp4list:  [[complex,complex,complex,complex]]
# - strength: float
# Returns:
# - new_cp4list: [[complex,complex,complex,complex]]
def simple_smooth_cp4list(cp4list, strength):
    k = strength
    if len(cp4list) <= 1:
        return deepcopy(cp4list)
    A,p1,_,C = cp4list[0]
    _,_,_,B = cp4list[1]
    p2 = C + (k/2)*(A-B) * abs(-A+C) / (abs(-A+C) + abs(-B+C))
    new_cp4list = [[A,p1,p2,C]]
    for i in range(1, len(cp4list)-1):
        prev, curr, next = cp4list[i-1:i+2]
        A = prev[0]
        C = curr[0]
        B = curr[-1]
        D = next[-1]
        p1 = C + (k/2)*(-A+B) * abs(-B+C) / (abs(-A+C) + abs(-B+C))
        p2 = B + (k/2)*(-D+C) * abs(-C+B) / (abs(-B+C) + abs(-B+D))
        new_cp4 = [C,p1,p2,B]
        new_cp4list.append(new_cp4)
    B,_,p2,D = cp4list[-1]
    C,_,_,_ = cp4list[-2]
    p1 = B + (k/2)*(D-C) * abs(-D+B) / (abs(-B+C) + abs(-B+D))
    new_cp4list.append([B,p1,p2,D])
    return new_cp4list

# --------------------
# simple_smooth_bcurve
# --------------------
# Assumes that bc consists of straight line segments.
# Args:
# - bc:             BCurve.BezierCurve
# - strength:       float
# - selection_case: integer
# - remove_excess: boolean
# Returns:
# - BCurve.BezierCurve
def simple_smooth_bcurve(image, bc, strength, selection_case, remove_excess):
    cp4list = [ba.cp4 for ba in bc.bezier_arcs]
    if bc.closed: # Close gap twice: add one arc at the head and at the tail:
        start = cp4list[0][0]
        end = cp4list[-1][-1]
        head_cp4 = tail_cp4 = [end, end, start, start]
        cp4list = [head_cp4] + cp4list + [tail_cp4]
    # --- Preprocessing ---
    list_cp4lists = subdivide(image, cp4list, selection_case)
    # Are the curves represented by the cp4lists in list_cp4lists closed or not?
    # They are closed if and only if bc is closed and no subdivision occurred
    # (in which case there is only list):
    closed = bc.closed and (len(list_cp4lists) == 1)
    if remove_excess:
        rem_list_cp4lists = []
        for cp4l in list_cp4lists:
            ##############  KOE
            #rem_list_cp4lists.append(remove_excess_anchors_old(cp4l, closed))
            rem_list_cp4lists.append(remove_excess_anchors(cp4l, closed))
        list_cp4lists = rem_list_cp4lists
    # --- Smoothing ---
    new_cp4list = []
    for one_cp4list in list_cp4lists:
        new_cp4list += simple_smooth_cp4list(one_cp4list, strength) # Main call
    if bc.closed: # Remove added head and tail
        new_head_handle = new_cp4list[0][2]
        new_tail_handle = new_cp4list[-1][1]
        new_cp4list = new_cp4list[1:-1]
    else:
        new_head_handle = new_cp4list[0][0]
        new_tail_handle = new_cp4list[-1][-1]
    new_balist = [BCurve.BezierArc(cp4=cp4) for cp4 in new_cp4list]
    return BCurve.BezierCurve(bezier_arcs = new_balist,
                              head_handle=new_head_handle,
                              tail_handle=new_tail_handle,
                              closed=bc.closed,
                              curve_name='smoothed bc',
                              )

# ------------------
# simple_smooth_path
# ------------------
# Args:
# - image
# - path:           gimp.Vectors
# - strength:       float
# - selection_case: integer
# - remove_excess: boolean
# Returns:
# - gimp.Vectors
def simple_smooth_path(image, path, strength, selection_case, remove_excess):
    gv = vectors_object2gv(path) # BCurve.GimpVectors
    new_stroke_list = []
    for gs in gv.stroke_list: # BCurve.GimpStroke
        bc = gs.gs2bc()       # BCurve.BezierCurve
        new_bc = simple_smooth_bcurve(image, bc, strength, selection_case, remove_excess) # Main call
        new_gs = new_bc.bc2gs()
        new_stroke_list.append(new_gs)
    new_gv = BCurve.GimpVectors(stroke_list = new_stroke_list)
    name = path.name+'|smooth('+str(strength)+')'
    return new_gv.gv2vectors_object(image, name=name)

selection_options = [ # (description, identifier)
        ('Ignore any selection',                                      'ignore'),
        ('Smooth the corners only inside the selection', 'inside'),
        ('Smooth the corners only outside of the selection', 'outside'),
        ]

#==============================================================
#             Main procedure
#==============================================================

def simple_smooth_path_main(image,
                            path, 
                            strength,
                            remove_excess,
                            selection_option, # integer
                            ):
    selection_case = selection_options[selection_option][1]
    path_smoothed = simple_smooth_path(image, path, strength, selection_case, remove_excess) # Main call
    pdb.gimp_image_undo_group_start(image)
    gimp_draw_vectors_object(image, path_smoothed, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return path_smoothed


#======================================================
#                    Registrations
#======================================================

versionnumber = "0.3"
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2021"
image_types = "*"
menupath = '<Vectors>/Tools/Modify path'


#####################  Simple smooth path  #####################

procedure_name  = "simple_smooth_path"
procedure_blurb = ("Smooth a path consisting of straight line segments."
                   +"\nNote: If you use a selection, out-of-canvas anchors"
                   +"\nare taken to be outside of the selection."
                   +"\nIf you need to include out-of-canvas anchors in the selection,"
                   +"\nyou have to enlarge the canvas temporarily first."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = ("Smooth a path consisting of straight line segments."
                   +"\nA very simple algorithm."
                   )

procedure_label = "Simple smooth"

procedure_function = simple_smooth_path_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path", None),
      (PF_FLOAT, "strength",
                 ("Strength of the effect (float)"
                  #+"\n..."
                  ),
                  1),
      (PF_BOOL, "remove_excess",
                  ("Remove superfluous anchors from straight edges?"
                  #+"\n..."
                  ),
                  True),
      (PF_OPTION, 'selection_option','How to use the selection?',
                   0,
                  [case[0] for case in selection_options]), # Descriptions of cases
    ],
    [
        (PF_VECTORS, "path_smoothed", "Smoothed path"),
    ],
    procedure_function,
    menu=menupath)


main()
