#!/usr/bin/env python

# Gimp plugin to repeat a pattern (a path) along a circle.
#
# History:
# v0.1: 2021-11-28: Plugin "Replicate along circle"
# v0.2: 2021-12-05: Better approximation algorithm. Some support for curved paths.
#                   First published version.
# v0.3: 2021-12-10: Improvements on approximation, curved paths.

# (c) Markku Koppinen 2021
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'


from __future__ import division, print_function
from gimpfu import *
from math import *
from cmath import exp as cexp
from copy import deepcopy

#==============================================================
#             class BCurve       
#==============================================================

# Note: The plane is treated as the complex plane.

class BCurve(object):
    """Data structures to handle Gimp vectors objects by means of individual
    Bezier arcs.
    By a "Bezier arc" we mean here a Bezier curve determined by 4 control points
    p0,p1,p2,p3.
    This class contains data structures needed to do computations on
    a Bezier curve which are done arc by arc.
    To this end, Gimp data structures
    - gimp.Vectors
    - gimp.VectorsBezierStroke
    are here represented as data structures (classes)
    - GimpVectors
    - GimpStroke
    - BezierCurve
    - BezierArc
    - ControlPoint = complex
    with the aim that
    - each control point [x,y] is represented as an instance of ControlPoint;
    - each Bezier arc (four control points, see above) is represented as an instance
      of BezierArc.
    """
    
    ControlPoint = complex
    
    class GimpVectors(object):
        """Essentially same data as in a gimp.Vectors object except that
        strokes are instances of GimpStroke rather than
        gimp.VectorsBezierStroke.
        Attributes:
        - stroke_list: [GimpStroke]
        - name:        string
        """
        def __init__(self,
                     stroke_list,            # [GimpStroke]
                     name='GimpVectors'      # string
                     ):
            self.stroke_list = stroke_list
            self.name = name
        def __str__(self):
            s = 'GimpVectors '+self.name
            count = 0
            for stroke in self.stroke_list:
                s += '\n  Stroke '+str(count)+': GimpStroke'
                s += '\n    '+str(stroke)
                count += 1
            return s
        def gv2vectors_object(self, image, name=None):
            """Conversion GimpVectors -> gimp.Vectors
            (The converse: vectors_object2gv)
            """
            def cp_list2xy_list(cp_list):
                xy = []
                for cp in cp_list:
                    xy += [cp.real, cp.imag]
                return xy
            if name is None:
                name = self.name
            vectors = pdb.gimp_vectors_new(image, name)
            for stroke in self.stroke_list:
                closed = stroke.closed
                xy_list = cp_list2xy_list(stroke.cp_list)
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                             vectors, 0, len(xy_list), xy_list, closed)
            return vectors
    
    class GimpStroke(object):
        """Essentially same data as in a gimp.VectorsBezierStroke.points,
        except that control points list [x,y,x,y,...] is arranged as
        a list [ControlPoint] by joining successive x,y.
        Attributes:
        - cp_list:     [ControlPoint]
        - closed:      boolean
        - stroke_name: string
        """
        def __init__(self,
                     cp_list,                   # [ControlPoint]
                     closed,                    # boolean
                     stroke_name = 'GimpStroke' # string
                     ):
            self.stroke_name = stroke_name
            self.cp_list = cp_list
            self.closed = closed
        def __str__(self):
            s = self.stroke_name + '; Control points:'
            for cp in self.cp_list:
                s += '\n    '+str(cp)
            s += '\n    closed: '+str(self.closed)
            return s
        def gs2bc(self):
            """Conversion GimpStroke -> BezierCurve
            """
            head = self.cp_list[0]
            tail = self.cp_list[-1]
            inner = self.cp_list[1:-1]
            if len(inner) > 1: # Must be divisible by 4
                ba_list = []
                count = 0
                for i in range(0,len(inner)-1,3):
                    ba_list.append(BCurve.BezierArc(inner[i:i+4], 'arc '+str(count)))
                    count += 1
            elif len(inner) == 1: # Stroke has only one anchor
                ba_list = [BCurve.BezierArc([inner[0]], 'arc 0')]
            else:
                raise Exception("BCurve.GimpStroke.gs2bc: No anchors in stroke?")
            return BCurve.BezierCurve(
                               bezier_arcs = ba_list,
                               head_handle = head,
                               tail_handle = tail,
                               closed = self.closed,
                               curve_name = self.stroke_name
                               )
    
    class BezierCurve(object):
        """BezierCurve is a list of butting Bezier arcs with some extra data.
        Attributes:
        - curve_name:  string
        - bezier_arcs: [BezierArc];
        - head_handle: ControlPoint;
        - tail_handle: ControlPoint;
        - closed:      boolean.
        Note: "Butting" means that for any pair of successive arcs, the last
              control point of the former equals the first control point of
              the latter.
              In initialization no checks are done about this condition.
        """
        def __init__(self,
                     bezier_arcs=[],           # [BezierArc]
                     head_handle=None,         # ControlPoint
                     tail_handle=None,         # ControlPoint
                     closed=False,             # boolean
                     curve_name='BezierCurve', # string
                     ):
            self.curve_name = curve_name
            self.bezier_arcs = bezier_arcs
            if head_handle is None:
                self.head_handle = bezier_arcs[0].cp4[0]
            else:
                self.head_handle = head_handle
            if tail_handle is None:
                self.tail_handle = bezier_arcs[-1].cp4[-1]
            else:
                self.tail_handle = tail_handle
            self.closed = closed
        def __str__(self):
            s = self.curve_name + ': Bezier curve' + '; Bezier arcs:'
            count = 0
            for arc in self.bezier_arcs:
                s += '\n  arc '+ str(count)+ ': ' +str(arc)
                count += 1
            s += '\n  Head handle:    '+str(self.head_handle)
            s += '\n  Tail handle:    '+str(self.tail_handle)
            s += '\n  Closed:'    +str(self.closed)
            return s
        def bc2gs(self):
            """Conversion BezierCurve -> GimpStroke
            """
            cp_list = [self.head_handle]
            if len(self.bezier_arcs[0].cp4) > 1:
                for ba in self.bezier_arcs:
                    cp_list += ba.cp4[:3]
                cp_list.append(self.bezier_arcs[-1].cp4[-1])
            else:
                # Only one BezierArc and it has cp4 = [p0]
                # (may rise from Gimp stroke with only one anchor).
                cp_list.append(self.bezier_arcs[0].cp4[0])
            cp_list.append(self.tail_handle)
            return BCurve.GimpStroke(cp_list, self.closed, self.curve_name)
        def bc2vectors_object(self, image, name=None):
            """Conversion BezierCurve -> gimp.Vectors
            Makes vectors_object with 1 stroke.
            """
            if name is None:
                name = self.curve_name
            gs = self.bc2gs()             # BCurve.GimpStroke
            gv = BCurve.GimpVectors([gs]) # BCurve.GimpVectors
            return gv.gv2vectors_object(image, name)
    
    class BezierArc(object):
        """Data structure for one Bezier arc: arc determined by 4 control
           points.
        Attributes:
        - cp4:      [p0,p1,p2,p3] where each pi:ControlPoint, or
                    [p0] if only one control point (may rise from
                    a stroke with only one anchor);
        - arc_name: string
        """
        def __init__(self,
                     cp4,                 # cp4=[p0,p1,p2,p3] with pi:ControlPoint
                                          # or [p0]
                     arc_name='BezierArc' # string
                     ):
            self.cp4 = cp4
            self.arc_name = arc_name
        def reverse(self):
            self.cp4.reverse()
        def __str__(self):
            s = 'BezierArc, control points:'
            for cp in self.cp4:
                s += '\n  '+ str(cp)
            return s

# -----------------
# vectors_object2gv
# -----------------
# Conversion gimp.Vectors -> GimpVectors
# Args:
# - vectors_object: gimp.Vectors (Gimp vectors object)
# Returns:
# - GimpVectors
#
# Note: The inverse conversion is GimpVectors.gv2vectors_object(...).
#       See also BezierCurve.bc2vectors_object(...).
def vectors_object2gv(vectors_object):
    def xy_list2cp_list(xy_list): # xy_list = [x,y,x,y,...,]
        cp = []
        for i in range(0,len(xy_list),2):
            cp.append(BCurve.ControlPoint(xy_list[i], xy_list[i+1]))
        return cp
    stroke_list = []
    count = 0
    for stroke in vectors_object.strokes:
        xy_list = stroke.points[0]
        closed = stroke.points[1]
        cp_list = xy_list2cp_list(xy_list)
        stroke_name = vectors_object.name + ' stroke '+str(count)
        stroke_list.append(BCurve.GimpStroke(cp_list, closed, stroke_name))
        count += 1
    return BCurve.GimpVectors(stroke_list, vectors_object.name)

# ---------------------
# completed_bezier_arcs
# ---------------------
# Give bc:BCurve.BezierCurve, get the list of bezier arcs where, in the case of
# closed bc, a gap-closing arc is added at the end, with necessary modification
# for the case of closed 1-anchor stroke.
# Only copies of the original arcs are returned.
# Args:
# - bc:BCurve.BezierCurve
# Returns:
# - [BCurve.BezierArc]
def completed_bezier_arcs(bc):
    bas = deepcopy(bc.bezier_arcs)
    if not bc.closed:
        return bas
    gap_ba=BCurve.BezierArc(cp4=[bas[-1].cp4[-1],
                            bc.tail_handle,
                            bc.head_handle,
                            bas[0].cp4[0]])
    if len(bas[0].cp4) == 1: # 1-anchor stroke
        return [gap_ba]
    else:
        return bas + [gap_ba]

#==============================================================
#                         Draw
#==============================================================
# ------------------------
# gimp_draw_vectors_object
# ------------------------
# The Gimp version of the routine 'drawing_routine_for_bezier':
# Insert into Gimp the input vectors object, and make it visible if
# visible=True.
# Args:
# - image: Gimp's image (obtained usually from GUI);
# - vectors_object: a vectors object in Gimp;
# - visible: Boolean.
# Returns the vectors object in Gimp, and makes it visible if visible=True.
def gimp_draw_vectors_object(image, vectors_object, visible=True):
    # Insert a Gimp vectors object in Gimp.
    # Adapted from http://registry.gimp.org/files/lqr_wpset-0.14.py.txt
    # by Mike Kazantsev.
    def gimp_insert_vectors(image, vectors_object, position=0):
        if gimp.version >= (2, 8, 0):
            # 2.7.0 has gimp-image-insert-vectors, but fails with
            # parent=None
            parent = None
            return pdb.gimp_image_insert_vectors(
                     image, vectors_object, parent, position)
        else:
            return pdb.gimp_image_add_vectors(
                     image, vectors_object, position)# Deprecated!
    gimp_insert_vectors(image, vectors_object)
    vectors_object.visible = visible
    return vectors_object

# ----------------
# gimp_draw_circle
# ----------------
# center:complex
# radius: float
def gimp_draw_circle(image, center, radius, name='circle'):
    kappa = 4.*(sqrt(2)-1)/3
    xy = [1, -kappa, 1, 0, 1, kappa,
          kappa, 1, 0, 1, -kappa, 1,
          -1, kappa, -1, 0, -1, -kappa,
          -kappa, -1, 0, -1, kappa, -1]
    for i in range(len(xy)):
        if i%2 == 0:
            xy[i] = center.real + radius*xy[i]
        else:
            xy[i] = center.imag + radius*xy[i]
    circle = pdb.gimp_vectors_new(image, name)
    pdb.gimp_vectors_stroke_new_from_points(
                        circle, 0, len(xy), xy, True)
    gimp_draw_vectors_object(image, circle, visible=True)
    return circle


#==============================================================
#                     Plane vectors
#==============================================================

class NoIntersection(Exception):
    def __init__(self, message):
        self.message = message
    def __str__(self):
        return str(self.message)

# -------------------
# line_intersection_C
# -------------------
# Given in the complex plane two straight lines:
# - through z, direction sz,
# - through w, direction sw,
# compute the intersection point. If positive=True, accept only those where the
# coefficients of sz and sw are non-negative.
# Args:
# - z,w,sz,sw: complex
# - positive: boolean
# Returns:
# - complex
def line_intersection_C(z,w,sz,sw, positive=False):
    ZERO = 1e-12
    if abs((sz*sw.conjugate()).imag) < ZERO:
       raise NoIntersection("line_intersection_C: parallel lines or a zero direction vector")
    try:
       t = -((z-w)*sw.conjugate()).imag / (sz*sw.conjugate()).imag
    except ZeroDivisionError:
       raise NoIntersection("line_intersection_C: parallel lines or a zero direction vector")
    if positive:
       u = -((w-z)*sz.conjugate()).imag / (sw*sz.conjugate()).imag
       if t < 0 or u < 0:
           raise NoIntersection("line_intersection_C: coefficients not non-negative")
    return z + t*sz

# Make similitude where z0,z1 are mapped onto w0,w1.
def make_similitude(z0,z1,w0,w1):
    def similitude(z):
        return (w1*(z-z0) - w0*(z-z1)) / (-z0+z1)
    return similitude

def vdot(v,w):
    return v.real*w.real + v.imag*w.imag

# v dot (w perpendicular)
def cross(v,w):
    return (v*w.conjugate()).imag

#==============================================================
#                        Bezier
#==============================================================
# ---------
# bezier_rv
# ---------
# The usual Bezier curve from control points.
# Version where all points are complex numbers
# (the plane is the complex number plane).
# Args:
# - t:  float
# - cp4:[complex,complex,complex,complex]
# Returns:
# - complex
def bezier_rv(t, cp4):
    P0,P1,P2,P3 = cp4
    if t > 0.5:
        u = (1-t)/t
        t3 = t**3
        return t3*(u*(u*(u*P0+3*P1)+3*P2)+P3) # complex
    else:
        u = t/(1-t)
        t3 = (1-t)**3
        return t3*(u*(u*(u*P3+3*P2)+3*P1)+P0) # complex

# Derivative: Returns complex.
def bezier_dot_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    P01 = -P0+P1
    P12 = -P1+P2
    P23 = -P2+P3
    if t > 0.5:
        u = (1-t)/t
        t2 = t**2
        return 3*t2*(u*(u*P01+2*P12)+P23) # complex
    else:
        u = t/(1-t)
        t2 = (1-t)**2
        return 3*t2*(u*(u*P23+2*P12)+P01) # complex

# Second derivative
# Args:
# - t:  float
# - cp4:[complex,complex,complex,complex]
# Returns:
# - complex
def bezier_dot_dot_rv(t, control_points): # Could be made faster as above.
    p0,p1,p2,p3 = control_points
    return -6*(1-t)*(-p0+p1) + 6*(1-2*t)*(-p1+p2) + 6*t*(-p2+p3)

# -------------------------
# bezier_new_control_points
# -------------------------
# Given a Bezier arc B(t) and two distinct parameter values a and b,
# find a new parametrization C(t) of the same curve such that
# B(a) = C(0) and B(b) = C(1).
# Args:
# - a: float;
# - b: float;
# - control_points: the list [p0,p1,p2,p3] of control points (pi:complex).
# Returns:
# - new control points [q0,q1,q2,q3]:[RowVector] such that the corresponding
#   Bezier curve C(t) is precisely the same as B as a curve
#   but B(a) = C(0) and B(b) = C(1).
# Notes:
# - This enables one to extend or to shorten the given Bezier arc.
# - There is an exception: If a=b, the returned control points
#   are equal, so the new curve is a single point, not the original
#   curve. The function gives no signal about such occurence.
def bezier_new_control_points(a, b, control_points):
    def bezier_2_rv(t, r0, r1, r2): # Bezier curve of degree two (complex)
        b20 = (1-t)**2
        b21 = 2*(1-t)*t
        b22 = t**2
        return b20*r0 + b21*r1 + b22*r2
    p0,p1,p2,p3 = control_points
    q0 = bezier_rv(a, control_points)
    q1 = (1-b)*bezier_2_rv(a, p0,p1,p2) + b*bezier_2_rv(a, p1,p2,p3)
    q2 = (1-a)*bezier_2_rv(b, p0,p1,p2) + a*bezier_2_rv(b, p1,p2,p3)
    q3 = bezier_rv(b, control_points)
    return [q0,q1,q2,q3]

# ------------------------------
# bezier_subdivide_at_parameters
# ------------------------------
# Given a Bezier arc B(t) and a list of parameter values t1,t2,...,
# find control points for all subarcs when B(t), 0<=t<=1, is subdivided
# at parameter values t1,...
# The input Bezier arc is given as its 4 control points [p0,p1,p2,p3],
# and the output is a list of new sets of 4 control points [[q0,q1,q2,q3],...].
# The plane is viewed as the complex number plane.
# Args:
# - parameter_list: [float]                           ([t1,t2,...])
# - control_points: [complex,complex,complex,complex] ([p0,p1,p2,p3])
# Returns:
# - [[complex,complex,complex,complex]]               ([[q0,q1,q2,q3],...])
def bezier_subdivide_at_parameters(parameter_list, control_points):
    ts01 = [t for t in parameter_list if 0 < t < 1] # Drop t's outside (0,1)
    ts01.sort()
    try:
        ts = [ts01[0]]
    except IndexError: # Empty parameter list
        return [[p for p in control_points]] # copy
    for t in ts01[1:]: # Drop duplicates.
        if t != ts[-1]:
            ts.append(t)
    ts = [0]+ts+[1] # Add t=0,1
    # Collect pairs [t(i),t(i+1)]
    abs = []
    for i in range(1,len(ts)):
        abs.append([ts[i-1],ts[i]])
    arc_cps = []
    for a,b in abs:
        arc_cps.append(bezier_new_control_points(a, b, control_points))
    return arc_cps

# -------------------------
# bezier_arc_end_directions
# -------------------------
# Given a Bezier arc (control points, 0<=t<=1), find, at each end,
# the unit direction vector of the arc.
# Note: this does not mean the derivatives of the parametric representation,
# but non-zero vectors which are tangential to the arc and point in the
# direction where the arc is located when viewed from that end point.
# Return the result as two complex numbers (= two vectors),
# or as None if the arc is one point.
# The plane is viewed as the complex number plane, so points are complex numbers.
# Args:
# - cp4: [complex,complex,complex,complex] (control points)
# Returns: either None (when the arc is one point), or
# - complex (unit direction vector at starting point)
# - complex (unit direction vector at ending point)
# Note: If len(cp4)=1, None is returned. That case comes from a 1-anchor stroke.
def bezier_arc_end_directions(cp4):
    ZERO = 1e-12
    if len(cp4) == 1: # May happen in a 1-anchor stroke
        return None
    try:
        p0,p1,p2,p3 = cp4
    except ValueError:
        raise Exception("bezier_arc_end_directions: Should not happen 1!")
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    if vdot(p01,p01) > ZERO:
        head_dir = p01
    elif vdot(p12,p12) > ZERO:
        head_dir = p12
    elif vdot(p23,p23) > ZERO:
        head_dir = p23
    else: # Arc is one point
        return None
    if vdot(p23,p23) > ZERO:
        tail_dir = -p23
    elif vdot(p12,p12) > ZERO:
        tail_dir = -p12
    elif vdot(p01,p01) > ZERO:
        tail_dir = -p01
    else:
        raise Exception("bezier_arc_end_directions: Should never happen 2!")
    return head_dir/abs(head_dir), tail_dir/abs(tail_dir)

# ---------------------
# bezier_extremal_points
# ---------------------
# Given a Bezier curve (4 control points p0,p1,p2,p3:complex), find its
# extremal points.
# Return list [[t,B(t)]] where B(t) is an extremal point and t is the
# parameter value.
# By an extremal point we understand roughly a point on the curve which is
# locally most distant from the chord. The exact condition is B'(t)||p0p3.
# Args:
# - cp4: [complex,complex,complex,complex]
# Returns:
# - [[float,complex]]
def bezier_extremal_points(cp4, restrict01=True):
    p0,p1,p2,p3 = cp4
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    p30 = -p3+p0 # chord
    p01xp03 = cross(p01,p30)
    p12xp03 = cross(p12,p30)
    p23xp03 = cross(p23,p30)
    ts = solve_bernstein2_equation(p01xp03, p12xp03, p23xp03)
    if ts is None:
        return []
    if restrict01:
        ts = [t for t in ts if 0<=t<=1]
    return [[t,bezier_rv(t,cp4)] for t in ts]

# ---------------------
# bezier_side_extremals
# ---------------------
# Extension of 'bezier_extremal_points' except that assumes that
# - the arc is not a straight line; (TO DO: need better handling)
# - there are no inflection points (direction of curvature does not change);
# - there is a new argument 'level' (default=0).
# If level=0 works just as 'bezier_extremal_points' except that instead of
# a list returns now the extremal point [(t,B(t))] (1-item list).
# It is unique since there are no inflection points.
# If level=1 returns [(t0,B(t0)), (t,B(t)), (t1,B(t1))] where
# - 0 < t0 < t < t1 < 1;
# - (t0,B(t0)) is the extremal point;
# - (t,B(t))   is the extremal point for the arc from B(0) to B(t);
# - (t1,B(t1)) is the extremal point for the arc from B(t) to B(1).
# Currently this is how far it goes.
# In the future, maybe the scheme is extended recursively, giving a nice
# subdivision for the whole Bezier arc cp4.
# Failures:
# - In case of failure to find (t,B(t)) raises Exception.
# - In case of failure to find (t0,B(t0)), None is returned in place of (t0,B(t0)).
# - In case of failure to find (t1,B(t1)), None is returned in place of (t1,B(t1)).
# 
# Args:
# - cp4:[complex,complex,complex,complex]
# Returns:
# - [None or (float,complex)] ( = [None or (t,B(t))] )
def bezier_side_extremals(cp4, level=0):
    p0,p1,p2,p3 = cp4
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    p30 = -p3+p0 # chord
    ts = solve_bernstein2_equation(cross(p01,p30), cross(p12,p30), cross(p23,p30))
    if ts is None:
        raise Exception("bezier_side_extremals: len(ts) = 0. Straight line?")
    ts = [t for t in ts if 0<t<1]
    if len(ts) != 1:
        raise Exception("bezier_side_extremals: len(ts) = "+str(len(ts))+". Inflection point?")
    t_ext = ts[0]
    B_ext = bezier_rv(t_ext,cp4)
    if level == 0:
        return [(t_ext,B_ext)]
    elif level == 1:
        p0Bext = -p0 + B_ext
        ts = solve_bernstein2_equation(cross(p01,p0Bext), cross(p12,p0Bext), cross(p23,p0Bext))
        if ts is None:
            t0B0 = None
        ts = [t for t in ts if 0<t<1]
        if len(ts) != 1:
            t0B0 = None
        else:
            t0 = ts[0]
            B0 = bezier_rv(t0,cp4)
            t0B0 = (t0,B0)
        Bextp3 = -B_ext + p3
        ts = solve_bernstein2_equation(cross(p01,Bextp3), cross(p12,Bextp3), cross(p23,Bextp3))
        if ts is None:
            t1B1 = None
        ts = [t for t in ts if 0<t<1]
        if len(ts) != 1:
            t1B1 = None
        else:
            t1 = ts[0]
            B1 = bezier_rv(t1,cp4)
            t1B1 = (t1,B1)
        return [t0B0, (t_ext,B_ext), t1B1]
    else:
        raise Exception("bezier_side_extremals: levle > 1 not implemented yet.")

# --------------------
# bezier_closest_point
# --------------------
# Find closest point (parameter value t and the point B(t)) to a given v on
# the Bezier curve arc B(t), 0<=t<=1. Control points are cp4.
# Args:
# - cp4: [complex,complex,complex,complex] (control points)
# - v:   complex
# - level: integer (working level: higher number, harder work)
# Returns:
# - [float,complex] (=[t,B(t)])
def bezier_closest_point(cp4, v, level=5):
    accuracy = 1e-8
    linear_approx = True # Linear approximation
    if level <= 2:
        guesses = [1/3, 2/3]
    elif level <= 5:
        guesses = [1/4, 2/4, 3/4]
    elif level <= 8:
        guesses = [1/6, 2/6, 3/6, 4/6, 5/6]
    else:
        guesses = [1/8, 2/8, 3/8, 4/8, 5/8, 6/8, 7/8]
        linear_approx = False # Quadratic approximation
    candidates = [[0,cp4[0]], [1,cp4[-1]]] # [B(t),t]: First, store end points.
    for guess in guesses: # Second, points in open interval (tangent condition)
        t = guess
        Bv = bezier_rv(t,cp4) - v
        for tries in range(20):
            Bd = bezier_dot_rv(t,cp4)
            Bdd = bezier_dot_dot_rv(t,cp4)
            b = (Bv.real * Bdd.real + Bv.imag * Bdd.imag     # b = (B-v).B'' + B'.B'
                 + Bd.real * Bd.real + Bd.imag * Bd.imag)
            c = Bv.real * Bd.real + Bv.imag * Bd.imag        # c = (B-v).B'
            if not linear_approx: # Quadratic: solve a(dt)^2 + b(dt) + c = 0
                try:
                    a = Bd.real * Bdd.real + Bd.imag * Bdd.imag  # a = B'.B''
                    inv_2a = 1/(2*a)
                    dt = inv_2a * (-b + sqrt(b*b - 4*a*c))
                except ZeroDivisionError:
                    linear_approx = True # Try linear
                except ValueError:
                    linear_approx = True # Try linear
            if linear_approx:     # Linear: solve b(dt) + c = 0
                try:
                    dt = -c / b
                except ZeroDivisionError:
                    break
            tdt = t+dt
            if tdt < 0: tdt = 0 # Restrict the parameter to [0,1]
            elif tdt > 1: tdt = 1
            if abs(t - tdt) < accuracy:
                t = (t + tdt)/2
                candidates.append([t, bezier_rv(t,cp4)])
                break
            else:
                t = tdt
                Bv = bezier_rv(t,cp4) - v
        else:
            candidates.append([t, Bv])
    return min(candidates, key=(lambda x: abs(x[1]-v)))

# ---------------------
# bezier_special_points
# ---------------------
# Find the special points of a cubic Bezier curve:
# inflection points, cusp points, self-intersection points.
#
# Args:
# - control_points: [p0,p1,p2,p3]
#
# Returns: (case, param_list), which is one of:
# (-1, [t1,t2]) when there is a loop or a cusp point, with
#               t1,t2 the parameter values with B(t1)=B(t2)
#               (ideally, t1=t2 in the case of a cusp);
# (0, [])       when the curve is degenerate: a parabola or
#               contained on a straight line;
# (1, [t])      when there is a single inflection point, with
#               t the parameter of the inflection point.
# (2, [t1,t2])  when there are two inflection points, with
#               t1,t2 the parameters for the inflection points.
#
# Note: The algorithm works as if infinite computation precision were
#   possible. The caller had better check the following:
# - In the case of a loop or of two inflection points (cases -1 or 2):
#   If the returned parameter values t1,t2 are very close, it may be
#   a sign of a cusp point, instead. (Or maybe the caller should take
#   it as a cusp point.)
# - In the case of inflection points (cases 1 or 2):
#   If one or both of the returned parameter values has very large
#   absolute value, it may mean that the inflection point is at infinity
#   (so, non-existent).
def bezier_special_points(control_points):
    from math import sqrt
    p0,p1,p2,p3 = control_points
    #w1 = [-p0[0] - p1[0] + p2[0] + p3[0],
    #      -p0[1] - p1[1] + p2[1] + p3[1]]
    #w2 = [p0[0] - p1[0] - p2[0] + p3[0],
    #      p0[1] - p1[1] - p2[1] + p3[1]]
    #w3 = [-p0[0] + 3*p1[0] - 3*p2[0] + p3[0],
    #      -p0[1] + 3*p1[1] - 3*p2[1] + p3[1]]
    ## The following also make C1,C2,C3 float.
    #C1 = 18.0 * (-w2[0]*w3[1] + w2[1]*w3[0])
    #C2 = 4.50 * (-w3[0]*w1[1] + w3[1]*w1[0])
    #C3 = 2.25 * (-w1[0]*w2[1] + w1[1]*w2[0])
    w1 = -p0 - p1 + p2 + p3
    w2 = p0 - p1 - p2 + p3
    w3 = -p0 + 3*p1 - 3*p2 + p3
    C1 = 18.0 * (-w2.real*w3.imag + w2.imag*w3.real)
    C2 = 4.50 * (-w3.real*w1.imag + w3.imag*w1.real)
    C3 = 2.25 * (-w1.real*w2.imag + w1.imag*w2.real)
    try:
        X = C2/C1
        Y = C3/C1
        Z = -X**2 + 2*Y
        try:
            root = sqrt(3*Z)
            t1 = .5 + X + root
            t2 = .5 + X - root
            return -1, [t1,t2] # loop or cusp
        except ValueError: # Z < 0
            root = sqrt(-Z)
            t1 = .5 + X + root
            t2 = .5 + X - root
            return 2, [t1,t2] # two inflection points
    except ZeroDivisionError: # C1 = 0
        try:
            t = .5 + C3/C2
            return 1, [t] # one inflection point
        except ZeroDivisionError: # C1 = C2 = 0
            return 0,[] # parabola or everything on straight line.


#==============================================================
#                Error measure
#==============================================================
# From parametric_curves.py

# --------------
# half_hausdorff
# --------------
# This is the error measure: how much does the Bezier arc (represented by cp4)
# deviate from sample points?
# Return (1) absolute error, and (2) error relative to approximate length of the arc.
# Args:
# - cp4:     [complex,complex,complex,complex] (control points)
# - samples: [complex]
# - level:   integer 0..10 (working level)
# Returns:
# - float (absolute error)
##### - float (relative error)
def half_hausdorff(cp4, samples, level=5):
    #f = sf.function
    #sample_f = [tf[1] for tf in sf.sample_points]
    max_error = 0
    for v in samples:
        _,w = bezier_closest_point(cp4, v, level)
        max_error = max(abs(v-w), max_error)
    return max_error  #, max_error/sf.length

#==============================================================
#                Polynomial zeroes
#==============================================================

# A function to find zeroes of a polynomial, and as by-products the
# x-values of the extrema and of the inflection points.
# At the final phase the zeroes are found by binary search from monotonous
# segments.
# 
# Written just for fun. Other algorithms exist in abundance.
#
# In addition, as a simple wrapper, a function to do the same job when the
# polynomial is of degree 3 and given in the Bernstein basis.

# --------------------
# zeroes_of_polynomial
# --------------------
# Given a real polynomial, find the real zeroes, the x values of extrema,
# and the x values of the inflection points.
#
# Args: coefficients: list [c0,c1,...,cn] of the coefficients of the
#                     polynomial c0 + c1*x + ... + cn*(x**n);
#                     the ci are integer or float.
# Returns:
# 1. None in the case of a zero polynomial.
# 2. Otherwise:
#    - the list of real zeroes of the polynomial, in increasing order
#      (possibly []) with multiple roots repeated;
#    - the extrema (the x values);
#    - the inflection points (the x values).
def zeroes_of_polynomial(coefficients):
    from math import sqrt
    from functools import partial
    def f(x, coeff): # Horner
        s = 0.
        for ci in coeff[::-1]:
            s = ci + x*s
        return s
    coefficients = [float(ci) for ci in coefficients]
    # Find largest n with coefficients[n] non-zero and
    # discard all higher terms:
    n = -1
    for i in range(len(coefficients)):
        if coefficients[i] != 0.:
            n = i
    if n < 0: # Polynomial is identically zero.
        return None
    c = coefficients[:n+1] 
    # Make c[n] positive:
    if c[n] < 0:
        c = [-ci for ci in c]
    if n == 0: # Polynomial is a non-zero constant.
        return [],[],[]
    if n == 1: # Polynomial is c[0] + c[1]*x
        return [-c[0] / c[1]], [], []
    if n == 2: # Polynomial is c[0] + c[1]*x + c[2]*(x**2).
        discr = c[1]**2 - 4*c[0]*c[2]
        try:
            root = sqrt(discr)
            x1 = (-c[1] - root) / (2*c[2])
            x2 = (-c[1] + root) / (2*c[2])
            return [x1,x2], [-c[1]/(2*c[2])], []
        except ValueError:
            return [], [-c[1]/(2*c[2])], []
    # Build subdivision such that in each subinterval
    # the polynomial is monotonous:
    derivative = [e*i for e,i in enumerate(c)][1:]
    extrema, inflection, _ = zeroes_of_polynomial(derivative)
    x_far = sum([abs(ci/c[-1]) for ci in c])
    subdiv = [-x_far] + extrema + [x_far]
    # Solve and collect the zeroes of each monotonous segment:
    fc = partial(f, coeff=c)
    zeroes = []
    for i in range(len(subdiv)-1):
        lo, hi = subdiv[i], subdiv[i+1]
        x = zero_of_monotonous_function(fc, lo, hi,
                                    accuracy=1e-10, tolerance=1e-10)
        if not(x is None):
            zeroes.append(x)
    return zeroes, extrema, inflection

# ------------------------------
# zeroes_of_bernstein_polynomial
# ------------------------------
# Given a polynomial of degree 3, in Bernstein basis,
# find the real zeroes, the x values of extrema, and the x values of the
# inflection points.
#
# Args: coefficients: list [c0,c1,c2,c3] of the coefficients of the
#                     polynomial c0*b0(t) + c1*b1(t) + c2*b2(t) + c3*b3(t)
#                     where b0,b1,b2,b3 are the bernstein polynomial of degree 3,
#                     and the ci are integer or float.
#                     Must be 4 coefficient.
# Returns:
# 1. None in the case of a zero polynomial.
# 2. Otherwise:
#    - the list of real zeroes of the polynomial, in increasing order
#      (possibly []) with multiple roots repeated;
#    - the extrema (the x values);
#    - the inflection points (the x values).
def zeroes_of_bernstein_polynomial(coefficients):
    # Convert a polynomial f(t) from the Bernstein basis {b0,b1,b2,b3}
    # to the power basis {1,t,t**2,t**3}.
    # Args:
    # - coefficients: list [c0,c1,c2,c3] = coefficients of a polynomial in
    #   Bernstein basis: so, the polynomial is
    #   f(t) = c0*b0(t) + c1*b1(t) + c2*b2(t) + c3*b3(t).
    # Returns:
    # - a list [a0,a1,a2,a3]: coefficients of f(t) in power basis:
    #   f(t) = a0 + a1*t + a2*(t**2) + a3*(t**3).
    def bernstein2power(coefficients):
        c0,c1,c2,c3 = [float(c) for c in coefficients]
        cb0 = [c0*b for b in [1,-3, 3,-1]]
        cb1 = [c1*b for b in [0, 3,-6, 3]]
        cb2 = [c2*b for b in [0, 0, 3,-3]]
        cb3 = [c3*b for b in [0, 0, 0, 1]]
        result = []
        for i in range(4):
            result.append(cb0[i] + cb1[i] + cb2[i] + cb3[i])
        return result
    if len(coefficients) != 4:
        s = "zeroes_of_bernstein_polynomial: "
        s += "wrong number of coefficients"
        raise Exception(s)
    power_basis_coefficients = bernstein2power(coefficients)
    return zeroes_of_polynomial(power_basis_coefficients)

# ---------------------------
# zero_of_monotonous_function
# ---------------------------
# Find the zero, if any, of a monotonous function f(x)
# in the interval [lo,hi]. If none is found, None is returned.
# Args:
# - f, lo, hi: the monotonous function with start and end value;
# - accuracy:  accuracy for x.
# - tolerance: float>=0; if no zeroes exist in the interval, then
#              if f(lo) or f(hi) is less than 'tolerance' from 0,
#              then lo or hi is returned, respectively.
# Returns: the zero or None.
def zero_of_monotonous_function(f, lo, hi, accuracy=1e-10, tolerance=1e-10):
    MAX_ROUNDS = 100
    lo, hi = float(lo), float(hi)
    flo, fhi = float(f(lo)), float(f(hi))
    if (flo > 0.) == (fhi > 0.):
        if abs(flo) <= tolerance:
            return lo
        elif abs(fhi) <= tolerance:
            return hi
        else:
            return None
    if flo == 0.:
        return lo
    if fhi == 0.:
        return hi
    count = 0
    while hi-lo > accuracy:
        mid = (lo+hi)/2.
        fmid = float(f(mid))
        if fmid == 0.:
            return mid
        if (flo > 0.) == (fmid > 0.):
            lo, flo = mid, fmid
        else:
            hi, fhi = mid, fmid
        count += 1
        if count > MAX_ROUNDS:
            break
    x = (fhi*lo - flo*hi) / (fhi-flo)
    return x

# -------------------------
# solve_bernstein2_equation
# -------------------------
# Find real zeroes of the equation c0*b0(t) + c1*b1(t) + c2*b2(t) = 0 where
# b0,b1,b2 are the Bernstein polynomials of degree 2 and
# c0,c1,c2 are real coefficients.
# Return the zeroes as a list (0, 1, or 2 items).
# Exception: If the function is constant 0, return None.
# Args:
# - c0,c1,c2: float
# Returns:
# - None or list of floats (possibly empty)
def solve_bernstein2_equation(c0,c1,c2):
    from math import sqrt
    ZERO = 1e-14
    if abs(c0 - 2.*c1 + c2) < ZERO: # equation of degree <= 1
        try:
            return [c0 / (2.*(c0 - c1))]
        except ZeroDivisionError: # equation of degree 0
            if abs(c0) < ZERO: # constant 0
                return None
            else: # constant not 0
                return []
    try:
        root = sqrt(c1**2 - c0*c2)
    except ValueError: # No real roots
        return []
    return [(c0 - c1 + root) / (c0 - 2.*c1 + c2),
            (c0 - c1 - root) / (c0 - 2.*c1 + c2)]


#==============================================================
#                Bezier arc from three tangents
#==============================================================
# --------------------------
# bezier_from_three_tangents
# --------------------------
# Find control points for a Bezier curve B(t), given
# - end points (= B(0) and B(1));
# - one third point;
# - tangent directions in those three points (assuming that none of the three
#   points will turn out to be a cusp point).
# Args:
# - p0, p3, p:        complex
# - u0, u3, u:        complex
# Returns:
# - list [ [[q0,q1,q2,q3],t,check],  [[q0,q1,q2,q3],t,check],  ... ]
#   containing zero to three items, where
#   - [q0,q1,q2,q3] is the list of control points of a Bezier curve B(t) such that
#     - the points q0,q3 are the given p0,p3;
#     - B(t)=p for some t;
#     - B'(0), B'(1), B'(t) are scalar multiples of u0,u3,u, respectively
#       (so that if they are not cusp points, then the curve B(t) has tangents
#       parallel to the prescribed u0,u3,u0, respectively;
#   - t is the parameter value where the curve reaches the point p, i.e., B(t)=p;
#   - check is [item,item,item] where each 'item' corresponds to
#     p0, p3, or p, and 'item' is None or True or False, meaning:
#     - None:  cusp;
#     - True:  B' at that point points in the same direction as the
#              prescribed u0,u3, or u;
#     - False: B' at that point points in the opposite direction from the
#              prescribed u0,u3, or u
# Exceptions:
# - If inputs are not ok, raises Bezier3TangentsInputError.
# Notes:
# 1. If the returned Bezier arc has a cusp in one of the points p0,p3,p, then
#    its tangent at that point has NOT the required direction but is something
#    one cannot even guess!
# 2. Solutions with t=0 or t=1 are rejected.
# 3. The plane is viewed as the complex number plane,
#    so all points are complex numbers.
def bezier_from_three_tangents(p0, p3, p, u0, u3, u):
    from math import sqrt
    bernstein = [lambda x: (1-x)**3,
                 lambda x: 3*(1-x)*(1-x)*x,
                 lambda x: 3*(1-x)*x*x,
                 lambda x: x**3]
    def Q(t,z0,z3):
        coeff0 = -(bernstein[0](t) + bernstein[1](t))
        coeff3 =   bernstein[2](t) + bernstein[3](t)
        return coeff0 * z0 + coeff3 * z3
    def cross(v,w):
        return -v.real*w.imag + v.imag*w.real
    def check_directions(v0,v3,v, control_points, t0,t3,t):
        epsilon = 1e-10
        nv0 = bezier_unit_direction_vector(t0, control_points)
        nv3 = bezier_unit_direction_vector(t3, control_points)
        nv  = bezier_unit_direction_vector(t,  control_points)
        directions = []
        num = 0
        for x,y in [[v0,nv0], [v3,nv3], [v,nv]]:
            num += 1
            if -epsilon < x.real*y.real + x.imag*y.imag < epsilon: # cusp
                directions.append(None)
            elif x.real*y.real + x.imag*y.imag < 0:  # opposite direction
                directions.append(False)
            else:
                directions.append(True)
        return directions
    def non_zero(v):
        epsilon = 1e-14
        return abs(v)**2 > epsilon
    def parallel(v,w):
        epsilon = 1e-12
        len_v = abs(v)
        len_w = abs(w)
        s = cross(v,w)
        return -epsilon < s / (len_v*len_w) < epsilon
    C0 = cross(u3,u)
    C3 = cross(u,u0)
    C  = cross(u0,u3)
    z0 = -p0+p     # -p0 + p
    z3 = -p+p3     # -p  + p3
    z  = -p3+p0   # -p3 + p0
    # Check input
    if not (non_zero(z0) and non_zero(z3)): # p=p0 or p=p3
        m = 'bezier_from_three_tangents: two points too close (identical?)'
        #gimp_error_message(m, print_message=False)
        raise Bezier3TangentsInputError(m)
    if not (non_zero(u0) and non_zero(u3) and non_zero(u)): # u0,u3, or u is 0
        m = 'bezier_from_three_tangents: a tangent vector is 0 (or too close)'
        #gimp_error_message(m, print_message=False)
        raise Bezier3TangentsInputError(m)
    if parallel(u0,u3) and parallel(u,u3) and parallel(u,u0): # u0,u3,u parallel
        m = 'bezier_from_three_tangents: all tangents parallel (or too close to be)'
        #gimp_error_message(m, print_message=False)
        raise Bezier3TangentsInputError(m)
    #
    z0xC0u0 = C0*cross(z0,u0)
    z0xCu   = C *cross(z0,u)
    z3xC3u3 = C3*cross(z3,u3)
    z3xCu   = C *cross(z3,u)
    #
    # Coefficient of H(t) in Bernstein basis:
    c0 = -z0xC0u0 + z0xCu
    c1 = -z0xC0u0
    c2 = -z3xC3u3
    c3 = -z3xC3u3 + z3xCu
    #
    polynomial_zeroes = zeroes_of_bernstein_polynomial([c0,c1,c2,c3])
    if polynomial_zeroes is None: # Polynomial identically zero: raise error
        m = 'bezier_from_three_tangents: polynomial identically 0'
        #gimp_error_message(m, print_message=False)
        raise Bezier3TangentsInputError(m)
    zeroes, extrema, inflection = polynomial_zeroes
    results = []
    for t in zeroes:
        if t in (0,1): # Reject
            continue
        Qt = Q(t,z0,z3)
        q0t = -(t**3)*z + (2-3*t)*z0
        q3t = ((1-t)**3)*z + (1-3*t)*z3
        if abs(C) >= abs(C3):
            alpha = -cross(Qt,u3) / (bernstein[1](t) * C)
        else:
            alpha = -cross(q0t,u) / (bernstein[1](t) * C3)
        if abs(C) >= abs(C0):
            beta = -cross(Qt,u0) / (bernstein[2](t) * C)
        else:
            beta = -cross(q3t,u) / (bernstein[2](t) * C0)
        q0 = p0
        q1 = p0 + alpha * u0
        q2 = p3 - beta * u3
        q3 = p3
        check = check_directions(u0,u3,u, [q0,q1,q2,q3], 0.,1.,t)
        results.append([[q0,q1,q2,q3], t, check])
    return results

# ------------------------------
# bezier_arc_from_three_tangents
# ------------------------------
# Version of bezier_from_three_tangents where accepted are only answers that
# serve building a Bezier arc from the data:
# - The directions of B' at the end points must be u0 and u3
#   (opposite directions not accepted);
# - the paremeter t (giving p) must satisfy 0<t<1 so that p will be on the arc
#   (not on its extension).
# Args:
# - p0, p3, p: complex
# - u0, u3, u: complex
# Returns:
# - [[cp4, t]] where cp4 is the control point quadruple and t is the parameter
#   value with which p is reached.
# Note: The returned list will be empty if no accepted arc was found.
# Generally there are 0-3 answers.
def bezier_arc_from_three_tangents(p0, p3, p, u0, u3, u):
    results = bezier_from_three_tangents(p0, p3, p, u0, u3, u)
    # Accept only those with 0<t<1:
    results = [r for r in results if 0 < r[1] < 1]
    # Accept only those where the directions of B' at the end points are as required:
    results = [r for r in results if (r[2][0] is True) and (r[2][1] is True)]
    # Drop the direction checkings:
    return [r[:-1] for r in results]

class Bezier3TangentsInputError(Exception):
    def __init__(self,message): # 'message' is a string
        self.message = message
    def __str__(self):
        return self.message

# Gimp error message function.
# The input 'message' is a string.
def gimp_error_message(message, print_message=False):
    if print_message:
        print(message)
    # 0 = message box; 1 = console; 2 = error console
    pdb.gimp_message_set_handler(2) # Laita 0 lopulliseen!
    pdb.gimp_message(message)

# ----------------------------
# bezier_unit_direction_vector
# ----------------------------
# Find the unit direction vector of a Bezier curve B(t) at parameter value t.
# Args:
# - t: float
# - control_points: [complex]
# Returns:
# - complex or None (None is returned when B(t) is constant).
def bezier_unit_direction_vector(t, control_points):
    p0,p1,p2,p3 = control_points
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    b2 = [(1-t)**2, 2*(1-t)*t, t**2] # Bernstein 2
    dot = b2[0]*p01 + b2[1]*p12 + b2[2]*p23 # B'(t)/3
    try:
        return dot/abs(dot)
    except ZeroDivisionError:
        dotdot = -(1-t)*p01 + (1-2*t)*p12 + t*p23 # B''(t)/6
        try:
            return dotdot/abs(dotdot)
        except ZeroDivisionError:
            dotdotdot = p01 - 2*p12 + p23 # B'''(t)/6
            try:
                return dotdotdot/abs(dotdotdot)
            except ZeroDivisionError:
                return None


#==============================================================
#             Copy a pattern along a circle
#==============================================================

# -----------------
# preprocess_bclist
# -----------------
# Preprocess bclist:[BCurve.BezierCurve]. Currently this means only
# finding inflection points and adding anchors at those points.
# Args:
# - bclist:[BCurve.BezierCurve]
# Returns:
# - [BCurve.BezierCurve]
def preprocess_bclist(bclist):
    new_bclist = []
    for bc in bclist:
        bas = completed_bezier_arcs(bc) # Handles closed cases ok: closes the gap.
        if (len(bc.bezier_arcs) == 1) and (len(bc.bezier_arcs[0].cp4) == 1):
            # 1-anchor stroke
            new_bclist.append(deepcopy(bc))
            continue
        new_balist = []
        for ba in bas:
            special = bezier_special_points(ba.cp4)
            if special[0] in (1,2): # Case: inflection points
                ts = special[1]     # Parameter values
                ts = [t for t in ts if 0 < t < 1]
                if len(ts) == 0: # No inflection points in the arc: copy ba as such
                    new_balist.append(deepcopy(ba))
                else: # Subdivide
                    subdiv_cp4 = bezier_subdivide_at_parameters(ts, ba.cp4) # [cp4]
                    new_balist += [BCurve.BezierArc(cp4=cp4) for cp4 in subdiv_cp4]
            else: # No inflection points: copy ba as such
                new_balist.append(deepcopy(ba))
        if bc.closed: # Remove the gap-closing arc
            gap_ba = new_balist[-1]
            new_balist = new_balist[:-1]
            new_bc = BCurve.BezierCurve(bezier_arcs=new_balist,
                                        head_handle=gap_ba.cp4[2],
                                        tail_handle=gap_ba.cp4[1],
                                        closed=True
                                        )
        else:
            new_bc = BCurve.BezierCurve(bezier_arcs=new_balist)
        new_bclist.append(new_bc)
    return new_bclist

# ------------------
# bc2cp4list_as_such
# ------------------
# Copies the bc onto the circle between start_point and end_point
# suitably scaled and rotated and without any distortion.
# Args:
# - base_ends:           [complex,complex]
# - bc:             BCurve.BezierCurve
# - start_point:    complex
# - end_point:      complex
# - radial_scaling: float
# Returns:
# - [[complex,complex,complex,complex]] or [[complex]] (=[cp4])
def bc2cp4list_as_such(base_ends, bc, start_point, end_point, radial_scaling):
    A,B = base_ends
    P,Q = start_point, end_point
    similitude01j = make_similitude(A,B,0,1j)
    similitudePQ = make_similitude(0,1j,P,Q)
    if (len(bc.bezier_arcs) == 1) and (len(bc.bezier_arcs[0].cp4) == 1):
        # 1-anchor stroke
        z0 = similitude01j(bc.bezier_arcs[0].cp4[0])
        z0_scaled = complex(radial_scaling * z0.real, z0.imag)
        z1 = similitudePQ(z0_scaled)
        return [[z1]]
    new_cp4list = []
    bas = completed_bezier_arcs(bc)
    for ba in bas:
        cp4_0 = [similitude01j(z) for z in ba.cp4]
        cp4_0_scaled = [complex(radial_scaling * z.real, z.imag) for z in cp4_0]
        cp4_1 = [similitudePQ(z) for z in cp4_0_scaled]
        new_cp4list.append(cp4_1)
    return new_cp4list

# -------------------
# cp42cp4list_distort
# -------------------
# Copies cp4 onto the circle between start_point and end_point
# suitably scaled and rotated and distorted with the designated method
# (polar coordinates or exponential map).
# Returns list of cp4's and a relative error measure.
# Args:
# - base_ends:      [complex,complex]
# - cp4:            [complex,complex,complex,complex] (control point quadruple)
# - similitude_standard: callable (similitude to standard position)
# - similitude_final:    callable (similitude to final position)
# - theta:          float
# - radial_scaling: float
# - distort_case:   string
# Returns:
# - [[complex,complex,complex,complex]]
# - float (error)
def cp42cp4list_distort(base_ends,
                        cp4,
                        similitude_standard,
                        similitude_final,
                        theta,
                        radial_scaling,
                        distort_case):
    from cmath import phase, polar
    ZERO = 1e-8
    MAX_ROUNDS = 100
    MAX_LEVEL_ALLOWED = 4 # level of subdivision
    MAX_ERROR_ALLOWED = 0.005
    #MAX_ERROR_ALLOWED = 0.01
    unfinished_cp4list = [(cp4,0)] # [(cp4,subdiv_depth)]
    finished_cp4list = []          # [(cp4,error)]
    # Take from unfinished_cp4list one cp4 at a time,
    # try to transform it, and if successful, put the result in finished_cp4list;
    # otherwise subdivide the cp4 and put the pieces back in unfinished_cp4list.
    #
    # Transformation phases:
    # 1. Map with similitude_standard to standard position:
    #    polar:       base_ends -> (1, 1+i);
    #    exponential: base_ends -> (0, i).
    # 2. Do scaling in x direction (this will imply radial scaling).
    # 3. Do the transformation:
    #    polar:       (x,y) -> x * exp(i * theta * y);
    #    exponential: z -> exp(theta*z).
    # 4. Construct an approximate Bezier arc.
    # 5. Check the success of the approximation. If failure, do subdivision and
    #    jump to next round in the loop.
    # 6. Otherwise (success), map with similitude_final to final position.
    subdiv_depth_max = 0 # Just to record deepest subdivision for development
    if distort_case == 'polar':
        scale_radial = theta * radial_scaling
    elif distort_case == 'exponential':
        scale_radial = (log(theta+1)/theta) * radial_scaling
    else:
        raise Exception("Unknown distort_case: "+distort_case)
    #print("\n================  Starting new cp4  ====================")
    for round_count in range(MAX_ROUNDS):
        #print("\n----------------  Round "+str(round_count)+"  -----------------")
        #print("len(unfinished_cp4list) = "+str(len(unfinished_cp4list)))
        #for uuu in unfinished_cp4list: print(str(uuu))
        try:
            cp4_under_work, subdiv_depth = unfinished_cp4list.pop(0) # First unfinished arc
        except IndexError:
            #print("All finished for this cp4. Stop.")
            break # unfinished_cp4list is empty
        cp4_0 = [similitude_standard(z) for z in cp4_under_work]
        # Both polar and exponential transformation maps horizontal lines to
        # straight lines through origo: no need for subdivision or approximation!
        # Check first this.
        y0,y1,y2,y3 = [z.imag for z in cp4_0]
        if (abs(-y0+y1) < ZERO) and (abs(-y0+y2) < ZERO) and (abs(-y0+y3) < ZERO):
            #print("Horizontal")
            p0,p1,p2,p3 = cp4_0
            if distort_case == 'polar':
                p0_scaled, p3_scaled = [complex(1 + scale_radial*(-1+z.real), z.imag)
                                             for z in [p0,p3]]
                p0_1, p3_1 = [z.real * cexp(1j*theta*z.imag)
                                             for z in [p0_scaled,p3_scaled]]
                p0_final, p3_final = [similitude_final(z)
                                             for z in [p0_1,p3_1]]
                cp4_final = [p0_final, p0_final, p3_final, p3_final]
                finished_cp4list.append((cp4_final,0)) # set error = 0
            elif distort_case == 'exponential':
                p0_scaled, p3_scaled = [complex(scale_radial * z.real, z.imag)
                                             for z in [p0,p3]]
                p0_1, p3_1 = [cexp(theta*z)  for z in [p0_scaled,p3_scaled]]
                p0_final, p3_final = [similitude_final(z)
                                             for z in [p0_1,p3_1]]
                cp4_final = [p0_final, p0_final, p3_final, p3_final]
                finished_cp4list.append((cp4_final,0)) # set error = 0
            else:
                raise Exception("Unknown distort_case: "+distort_case)
            continue
        # Phase 2: Scaling
        if distort_case == 'polar':
            cp4_0_scaled = [complex(1 + scale_radial*(-1+z.real), z.imag) for z in cp4_0]
        elif distort_case == 'exponential':
            cp4_0_scaled = [complex(scale_radial * z.real, z.imag) for z in cp4_0]
        else:
            raise Exception("Unknown distort_case: "+distort_case)
        # Phase 3: Transformation
        if distort_case == 'polar':
            cp4_1 = [z.real * cexp(1j*theta*z.imag) for z in cp4_0_scaled]
            C0,D0 = cp4_0_scaled[0], cp4_0_scaled[-1]
            C1,D1 = cp4_1[0], cp4_1[-1]
            C0x,C0y = C0.real, C0.imag
            D0x,D0y = D0.real, D0.imag
            # C0direction, D0direction point towards the arc:
            C0direction, D0direction = bezier_arc_end_directions(cp4_0_scaled)
            C1direction = ((C0direction.real + 1j * theta * C0x * (C0direction.imag))
                            * cexp(1j*theta*C0y))
            D1direction = ((D0direction.real + 1j * theta * D0x * (D0direction.imag))
                            * cexp(1j*theta*D0y))
        elif distort_case == 'exponential':
            cp4_1 = [cexp(theta*z) for z in cp4_0_scaled]
            C0,D0 = cp4_0_scaled[0], cp4_0_scaled[-1]
            C1,D1 = cp4_1[0], cp4_1[-1]
            # C0direction, D0direction point towards the arc:
            C0direction, D0direction = bezier_arc_end_directions(cp4_0_scaled)
            C1direction, D1direction = C1*C0direction, D1*D0direction
        else:
            raise Exception("Unknown distort_case: "+distort_case)
        # Get sample points from the mathematically correct curve
        # for later checking of approximation error.
        # First: Take sample points from cp4_0_scaled:
        try:
            t_point_side_extremals = bezier_side_extremals(cp4_0_scaled, level=1)
            t_point_left, t_point_middle, t_point_right = t_point_side_extremals
            t_mean,P0 = t_point_middle
            t_left, P0_left = t_point_left
            t_right, P0_right = t_point_right
        except Exception as e:
            #print("cp42cp4list_distort: bezier_side_extremals raised:")
            #print(str(e))
            t_mean,P0 = 0.5, bezier_rv(0.5, cp4_0_scaled)
            t_left, P0_left = 0.25, bezier_rv(0.25, cp4_0_scaled)
            t_right, P0_right = 0.75, bezier_rv(0.75, cp4_0_scaled)
        # Second: Transform the sample points:
        if distort_case == 'polar':
            P0x, P0y = P0.real, P0.imag
            P1 = P0x * cexp(1j*theta*P0y)
            P0x_left, P0y_left = P0_left.real, P0_left.imag
            P1_left = P0x_left * cexp(1j*theta*P0y_left)
            P0x_right, P0y_right = P0_right.real, P0_right.imag
            P1_right = P0x_right * cexp(1j*theta*P0y_right)
            P0direction = bezier_dot_rv(t_mean, cp4_0_scaled)
            P1direction = ((P0direction.real + 1j * theta * P0x * (P0direction.imag))
                            * cexp(1j*theta*P0y))
        elif distort_case == 'exponential':
            P0 = bezier_rv(t_mean, cp4_0_scaled)
            P1 = cexp(theta*P0)
            P1_left = cexp(theta*P0_left)
            P1_right = cexp(theta*P0_right)
            P0direction = bezier_dot_rv(t_mean, cp4_0_scaled)
            P1direction = P1*P0direction
        else:
            raise Exception("Unknown distort_case: "+distort_case)
        # Phase 4: Construct Bezier approximation:
        try:
            arc_data = bezier_arc_from_three_tangents(C1, D1, P1,
                                           C1direction, -D1direction, P1direction)
        except Bezier3TangentsInputError as e: # Failure in approximation
            if subdiv_depth < MAX_LEVEL_ALLOWED:
                subdiv = bezier_subdivide_at_parameters([t_mean], cp4_under_work)
                subdiv_with_level = [(sd, subdiv_depth+1) for sd in subdiv]
                unfinished_cp4list = subdiv_with_level + unfinished_cp4list
                subdiv_depth_max = max(subdiv_depth_max, subdiv_depth+1)
            else:
                # Put to finished though not good.
                finished_cp4list.append(([similitude_final(z) for z in cp4_approx],
                                         error_approx))
            continue
        if len(arc_data) == 0:
            # Make a simple workaround
            #print("len(arc_data) = 0")
            chord = abs(C1-D1)
            try:
                q1 = C1 + (C1direction/abs(C1direction)) * chord/4
            except ZeroDivisionError:
                q1 = C1
            try:
                q2 = D1 + (D1direction/abs(D1direction)) * chord/4
            except ZeroDivisionError:
                q2 = D1
            cp4_approx = [C1,q1,q2,D1]
            t_approx = 0.5
        elif len(arc_data) == 1:
            #print("len(arc_data) = 1")
            cp4_approx, t_approx = arc_data[0]
        else: # len(arc_data) = 2 or 3: got 2 or 3 acceptable Bezier arcs
            #print("len(arc_data) = "+str(len(arc_data)))
            # A very simple ad-hoc way to choose one:
            # Choose the one where t is closest to 1/2.
            # For later development: Should choose the one which runs closest
            # to the desired curve. Should compute some sample points etc.
            # (But it seems to make no big differense.)
            cp4_approx, t_approx = min(arc_data, key=(lambda ad:abs(ad[1]-0.5)))
        # Phase 5: Check the approximation
        samples = [P1_left, P1_right]
        error_approx = half_hausdorff(cp4_approx, samples, level=5)
        if error_approx <= MAX_ERROR_ALLOWED:
            #print("OK, append to finished")
            finished_cp4list.append(([similitude_final(z) for z in cp4_approx],
                                      error_approx))
        elif subdiv_depth < MAX_LEVEL_ALLOWED:
            #print(">>> error_approx > MAX_ERROR_ALLOWED: "+str((error_approx, MAX_ERROR_ALLOWED)))
            subdiv = bezier_subdivide_at_parameters([t_mean], cp4_under_work)
            subdiv_with_level = [(sd, subdiv_depth+1) for sd in subdiv]
            unfinished_cp4list = subdiv_with_level + unfinished_cp4list
            subdiv_depth_max = max(subdiv_depth_max, subdiv_depth+1)
        else:
            # Put to finished though not good.
            #print("Reached MAX_LEVEL_ALLOWED = "+str(MAX_LEVEL_ALLOWED))
            # Phase 6: Apply similitude_final
            finished_cp4list.append(([similitude_final(z) for z in cp4_approx],
                                     error_approx))
    cp4list = [cp4 for cp4,_ in finished_cp4list]
    total_error = max(err for _,err in finished_cp4list)
    #print("Greatest subdivision depth used = "+str(subdiv_depth))
    return cp4list, total_error


# ------------------
# bc2cp4list_distort
# ------------------
# Copies bc onto the circle between start_point and end_point
# suitably scaled and rotated and distorted with polar coordinates mapping.
# Args:
# - base_ends:     [complex,complex]
# - bc:             BCurve.BezierCurve
# - start_point:    complex
# - end_point:      complex
# - center:         complex
# - theta:          float
# - radial_scaling: float
# - distort_case:   string
# Returns:
# - [[complex,complex,complex,complex]] or [[complex]] (=[cp4])
# - float (an error measure)
def bc2cp4list_distort(base_ends,
                       bc,
                       start_point,
                       end_point,
                       center,
                       theta,
                       radial_scaling,
                       distort_case):
    from cmath import phase, polar
    ZERO = 1e-8
    A,B = base_ends
    P,Q,K = start_point, end_point, center
    # Similitude to standard position:
    if distort_case == 'polar':
        similitude_standard = make_similitude(A,B, 1, 1+1j)
    elif distort_case == 'exponential':
        similitude_standard = make_similitude(A,B,0,1j)
    else:
        raise Exception("Unknown distort_case: "+distort_case)
    # Similitude to final position:
    similitude_final = make_similitude(0, 1, K, P)
    if (len(bc.bezier_arcs) == 1) and (len(bc.bezier_arcs[0].cp4) == 1):
        # 1-anchor stroke
        z0 = similitude_standard(bc.bezier_arcs[0].cp4[0])
        if distort_case == 'polar':
            scale_radial = theta * radial_scaling
            z0_scaled = complex(1 + scale_radial*(-1+z0.real), z0.imag)
            z1 = z0_scaled.real * cexp(1j*theta*z0_scaled.imag)
        elif distort_case == 'exponential':
            scale_radial = (log(theta+1)/theta) * radial_scaling
            z0_scaled = complex(scale_radial * z0.real, z0.imag)
            z1 = cexp(theta*z0_scaled)
        else:
            raise Exception("Unknown distort_case: "+distort_case)
        return ([[similitude_final(z1)]], 0) # set error=0
    new_cp4list = []
    bas = completed_bezier_arcs(bc)
    total_error = 0
    for ba in bas:
        cp4list, ba_error = cp42cp4list_distort(base_ends,
                                                ba.cp4,
                                                similitude_standard,
                                                similitude_final,
                                                theta,
                                                radial_scaling,
                                                distort_case)
        new_cp4list += cp4list
        total_error = max(total_error, ba_error)
    return new_cp4list, total_error

# ------------
# bc_to_sector
# ------------
# Copies the bc:BCurve.BezierCurve onto the circle sector between start_angle
# and end_angle suitably scaled and rotated and possibly distorted.
# Args:
# - bc:             BCurve.BezierCurve
# - base_ends:      [complex,complex]
# - center:         complex
# - theta:          float (sector angle for one copy of the pattern)
# - radius:         float
# - start_angle:    float (start the sector, radians)
# - end_angle:      float (end the sector, radians)
# - radial_scaling: float
# - distort_case:   string
# Returns:
# - [[complex,complex,complex,complex]] (= [cp4], list of control point quadruples or singlets)
# - float (an error measure)
def bc_to_sector(bc,
                 base_ends,
                 center,
                 theta,
                 radius,
                 start_angle,
                 end_angle,
                 radial_scaling,
                 distort_case):
    P = center + radius * cexp(1j*start_angle)
    Q = center + radius * cexp(1j*end_angle)
    if distort_case == 'no distortion':
        return (bc2cp4list_as_such(base_ends, bc, P,Q, radial_scaling), 0) # error=0
    elif distort_case in ('polar','exponential'):
        return bc2cp4list_distort(base_ends, bc, P,Q, center, theta, radial_scaling, distort_case)
    else:
        raise Exception("Unknown distort_case: "+distort_case)

# -----------------
# base_bc_to_sector
# -----------------
# Copy the bc of the base path onto the circle sector ncopies times
# suitably scaled and rotated and possibly distorted.
# Differs from the action of pattern_bclist_to_sector in that
# - the input is one stroke only and it is always open;
# - the output path has one stroke only (the replicate strokes are joined to one);
# - the output stroke is closed if it is a full circle.
# Args:
# - image
# - base_ends:      [complex,complex]
# - bc:             BCurve.BezierCurve
# - center:         complex
# - radius:         float
# - sector_start:   float (start the sector, radians)
# - sector_angle:   float (central angle of the sector, radians)
# - ncopies:        integer (number of copies of pattern on the sector)
# - radial_scaling: float
# - distort_case:   string
# Returns: 
# - gimp.Vectors
# - float (an error measure)
def base_bc_to_sector(image,
                      bc,
                      base_ends,
                      center,
                      radius,
                      sector_start,
                      sector_angle,
                      ncopies,
                      radial_scaling,
                      distort_case,
                      ):
    ZERO = 1e-6
    sector_start += 3*pi/2 # From top
    theta = sector_angle / ncopies
    new_cp4list = []
    total_error = 0
    for i in range(ncopies):
        start_angle = sector_start + i*theta
        end_angle = start_angle + theta
        cp4list, bc_error = bc_to_sector(bc,             # Main call
                                              base_ends,
                                              center,
                                              theta,
                                              radius,
                                              start_angle,
                                              end_angle,
                                              radial_scaling,
                                              distort_case
                                              )
        new_cp4list += cp4list
        total_error = max(total_error, bc_error)
    if ((sector_angle % (2*pi) < ZERO) or (sector_angle % (2*pi) > 2*pi - ZERO)):
        # Whole circle: close the stroke:
        last_cp4 = new_cp4list[-1]
        new_balist = [BCurve.BezierArc(cp4=cp4) for cp4 in new_cp4list[:-1]]
        new_bc = BCurve.BezierCurve(bezier_arcs = new_balist,
                                    head_handle = last_cp4[2],
                                    tail_handle = last_cp4[1],
                                    closed = True
                                    )
    else:
        new_balist = [BCurve.BezierArc(cp4=cp4) for cp4 in new_cp4list]
        new_bc = BCurve.BezierCurve(bezier_arcs=new_balist)
    new_gs = new_bc.bc2gs()
    new_gv = BCurve.GimpVectors(stroke_list = [new_gs])
    return new_gv.gv2vectors_object(image), total_error

# ------------------------
# pattern_bclist_to_sector
# ------------------------
# Copy the bclist of the pattern path onto the circle sector ncopies times
# suitably scaled and rotated and possibly distorted.
# Distorted means that straight edges of the pattern are arched according
# to circle curvature.
# Args:
# - image
# - bclist:         [BCurve.BezierCurve]
# - base_ends:      [complex,complex]
# - center:         complex
# - radius:         float
# - sector_start:   float (start the sector, radians)
# - sector_angle:   float (central angle of the sector, radians)
# - ncopies:        integer (number of copies of pattern on the sector)
# - radial_scaling: float
# - distort_case:   string (distortion method)
# Returns: 
# - gimp.Vectors
# - float (an error measure)
def pattern_bclist_to_sector(image,
                             bclist,
                             base_ends,
                             center,
                             radius,
                             sector_start,
                             sector_angle,
                             ncopies,
                             radial_scaling,
                             distort_case
                             ):
    ZERO = 1e-6
    sector_start += 3*pi/2 # From top
    theta = sector_angle / ncopies
    bclist = preprocess_bclist(bclist) # Currently subdivides at inflection points
    new_gslist = []
    total_error = 0
    for i in range(ncopies):
        start_angle = sector_start + i*theta
        end_angle = start_angle + theta
        for bc in bclist:
            new_cp4list, bc_error = bc_to_sector(bc,             # Main call
                                                 base_ends,
                                                 center,
                                                 theta,
                                                 radius,
                                                 start_angle,
                                                 end_angle,
                                                 radial_scaling,
                                                 distort_case
                                                 )
            new_balist = [BCurve.BezierArc(cp4=cp4) for cp4 in new_cp4list]
            if bc.closed:
                gap_ba = new_balist[-1]
                new_balist = new_balist[:-1]
                new_bc = BCurve.BezierCurve(bezier_arcs=new_balist,
                                            head_handle=gap_ba.cp4[2],
                                            tail_handle=gap_ba.cp4[1],
                                            closed=True
                                            )
            else:
                new_bc = BCurve.BezierCurve(bezier_arcs=new_balist)
            new_gs = new_bc.bc2gs()
            new_gslist.append(new_gs)
            total_error = max(total_error, bc_error)
    new_gv = BCurve.GimpVectors(stroke_list = new_gslist)
    return new_gv.gv2vectors_object(image), total_error


distort_options = [ # (description, identifier)
        ('polar coordinates',                                'polar'),
        ('exponential (conformal when radial scaling is 1)', 'exponential'),
        ('no distortion',                                    'no distortion'),
        ]


#==============================================================
#             Main procedure
#==============================================================
def replicate_along_circle_main(image,          # gimp.Image
                                pattern_path,   # gimp.Vectors
                                base_path,      # gimp.Vectors
                                center_x,       # float
                                center_y,       # float
                                radius,         # float
                                sector_start,   # float
                                sector_angle,   # float
                                ncopies,        # integer
                                radial_scaling, # float
                                distort_option, # boolean
                                #draw_pattern_output, # boolean: always True
                                #draw_base_output,    # boolean: always False (*)
                                draw_circle     # boolean
                                ):
    ZERO = 1e-10
    draw_pattern_output = True # No option for the user
    # To include the option 'draw_base_output' into the plugin,
    # uncomment line (*) above and line (*) in the registration;
    # otherwise the option does not exist for the user the default False is used.
    try:
        draw_base_output # If ok, the option exists: use the input value...
    except NameError: # ...otherwise use this default:
        draw_base_output = False
    distort_case = distort_options[distort_option][1]
    ncopies = max(1,ncopies)
    sector_angle = max(1, min(360, sector_angle))
    if (distort_case == 'no distortion') and (ncopies == 1) and (sector_angle > 180):
        m = ("When there is no distortion and Number of copies is 1,"
             +"\nthe largest allowed Sector angle is 180.")
        raise Exception(m)
    if abs(radial_scaling) < ZERO:
        raise Exception("Radial scaling is too close to 0")
    if len(base_path.strokes) != 1:
        raise Exception("The base path must have one stroke exactly")
    if len(base_path.strokes[0].points[0]) <= 6:
        raise Exception(("The base path must have at least two anchors in its only stroke."
                         +"\nThe end anchors will be used as the reference points."
                          ))
    if base_path.strokes[0].points[1]: # closed
        raise Exception("The base path must have an open stroke as its only stroke")
    gv_base = vectors_object2gv(base_path)
    base_ends = [gv_base.stroke_list[0].cp_list[1], # First and last anchors
                 gv_base.stroke_list[0].cp_list[-2]]
    if abs(base_ends[0]-base_ends[1]) < ZERO:
        raise Exception("The ends of the base path are too close to each other.")
    if abs(base_ends[0].real - base_ends[1].real) < ZERO: # Base upright
        if base_ends[0].imag < base_ends[1].imag: # Make the base from bottom to up
            base_ends.reverse()
            gv_base.stroke_list[0].cp_list.reverse()
    elif base_ends[0].real > base_ends[1].real: # Make the base from left to right
        base_ends.reverse()
        gv_base.stroke_list[0].cp_list.reverse()
    bc_base = gv_base.stroke_list[0].gs2bc()
    gv_pattern = vectors_object2gv(pattern_path)
    bclist_pattern = [gs.gs2bc() for gs in gv_pattern.stroke_list] # One bc for each stroke
    center = complex(center_x,center_y)
    sector_start *= pi/180
    sector_angle *= pi/180
    # Main calls
    if draw_base_output: # Currently never
        base_output, base_error = base_bc_to_sector(image,
                                                    bc_base,
                                                    base_ends,
                                                    center,
                                                    radius,
                                                    sector_start,
                                                    sector_angle,
                                                    ncopies,
                                                    radial_scaling,
                                                    distort_case
                                                    )
        base_output.name = base_path.name+'|along circle'
    if draw_pattern_output: # Currently always
        pattern_output, pattern_error = pattern_bclist_to_sector(image,
                                                    bclist_pattern,
                                                    base_ends,
                                                    center,
                                                    radius,
                                                    sector_start,
                                                    sector_angle,
                                                    ncopies,
                                                    radial_scaling,
                                                    distort_case
                                                    )
        pattern_output.name = pattern_path.name+'|along circle'
    # Draw
    pdb.gimp_image_undo_group_start(image)
    if draw_circle:
        center = complex(center_x,center_y)
        gimp_draw_circle(image, center, radius, name='circle')
    if draw_base_output:
        gimp_draw_vectors_object(image, base_output, visible=True)
        #print("Base error    = "+str(base_error))
    if draw_pattern_output: # Currently always
        new_path = gimp_draw_vectors_object(image, pattern_output, visible=True)
        #print("Pattern error = "+str(pattern_error))
    pdb.gimp_image_undo_group_end(image)
    return new_path

#======================================================
#                    Registration
#======================================================

versionnumber = "0.3"
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2021"
image_types = "*"
menupath = '<Vectors>/Tools/Replicate'


#####################  Round path corners #####################

procedure_name  = "replicate_along_circle"
procedure_blurb = ("Replicate a path along a circle (sector),"
                   +"\noptionally distorted to follow the circle arc."
                   #+"\nTakes as input the PATH to replicate"
                   #+"\nand a BASE PATH (one open stroke)."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = (""
                   )

procedure_label = "Replicate path along circle"

procedure_function = replicate_along_circle_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "pattern_path", "The path to be replicated along the circle", None),
      (PF_VECTORS, "base_path", 
                         ("The base path: one open stroke"
                         +"\n  The end points will serve as the reference"
                         +"\n  points to be positioned on the circle."
                         ), None),
      (PF_FLOAT, "center_x", "Circle center x", 500),
      (PF_FLOAT, "center_y", "Circle center y", 500),
      (PF_FLOAT, "radius", "Circle radius", 200),
      (PF_FLOAT, "sector_start", "Sector start (degrees, clockwise from top)", 0.),
      (PF_FLOAT, "sector_angle", 
                 ("Sector angle (degrees, 1..360)"
                 ),
                 360.),
      (PF_INT,   "ncopies", "Number of copies ", 12),
      (PF_FLOAT, "radial_scaling", "Radial scaling", 1),
      (PF_OPTION, 'distortion_option','Distortion method',
                   0,
                  [case[0] for case in distort_options]), # Descriptions of cases
      #(PF_BOOL, "draw_base_output", "Apply to the base path too?", False), # (*)
      (PF_BOOL, "draw_circle", "Draw the circle?", False),
    ],
    [
        (PF_VECTORS, "new_path", "New path"),
    ],
    procedure_function,
    menu=menupath)

main()
