#!/usr/bin/env python

# Gimp plugins to draw tangents or normals to paths
#
# History:
# v0.1: 2020-10-04: Developing
# v0.2: 2020-10-20: Developing
# v0.3: 2020-10-24: Developing
# v0.4: 2020-10-26: Four plugins:
#                   Bounding box
#                   Parallel tangents or normals
#                   Tangents or normals from a point
#                   Tangents or normals from a stroke end of another path
# v0.5: 2020-12-17: First published version.
# v0.6: 2020-12-18: Developing
# v0.7: 2020-12-30: Developing
# v0.8: 2021-01-10: New plugin:
#                   Common tangent or normal between the path and another path

# (c) Markku Koppinen 2020
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'

from __future__ import division, print_function
from gimpfu import *
from math import *
from copy import deepcopy

#======================================================
#                       BCurve
#======================================================

# Class: BCurve. Designed to facilitate manipulating paths arc-by-arc.
# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

RowVector = complex

class BCurve(object):
    """Data structures to handle Gimp vectors objects by means of individual
    Bezier arcs.
    By a "Bezier arc" we mean here a Bezier curve determined by 4 control points
    p0,p1,p2,p3.
    This class contains data structures needed to do computations on
    a Bezier curve which are done arc by arc.
    To this end, Gimp data structures
    - gimp.Vectors
    - gimp.VectorsBezierStroke
    are here represented as data structures (classes)
    - GimpVectors
    - GimpStroke
    - BezierCurve
    - BezierArc
    - ControlPoint = complex
    with the aim that
    - each control point [x,y] is represented as an instance of ControlPoint;
    - each Bezier arc (four control points, see above) is represented as an instance
      of BezierArc.
    """
    
    ControlPoint = RowVector # =complex
    
    class GimpVectors(object):
        """Essentially same data as in a gimp.Vectors object except that
        strokes are instances of GimpStroke rather than
        gimp.VectorsBezierStroke.
        Attributes:
        - stroke_list: [GimpStroke]
        - name:        string
        """
        def __init__(self,
                     stroke_list,            # [GimpStroke]
                     name='GimpVectors'      # string
                     ):
            self.stroke_list = stroke_list
            self.name = name
        def __str__(self):
            s = 'GimpVectors '+self.name
            count = 0
            for stroke in self.stroke_list:
                s += '\n  Stroke '+str(count)+': GimpStroke'
                s += '\n    '+str(stroke)
                count += 1
            return s
        def gv2vectors_object(self, image, name=None):
            """Conversion GimpVectors -> gimp.Vectors
            (The converse: vectors_object2gv)
            """
            def cp_list2xy_list(cp_list):
                xy = []
                for cp in cp_list:
                    xy += [cp.real, cp.imag]
                return xy
            if name is None:
                name = self.name
            vectors = pdb.gimp_vectors_new(image, name)
            for stroke in self.stroke_list:
                closed = stroke.closed
                xy_list = cp_list2xy_list(stroke.cp_list)
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                             vectors, 0, len(xy_list), xy_list, closed)
            return vectors
    
    class GimpStroke(object):
        """Essentially same data as in a gimp.VectorsBezierStroke.points,
        except that control points list [x,y,x,y,...] is arranged as
        a list [ControlPoint] by joining successive x,y.
        Attributes:
        - cp_list:     [ControlPoint]
        - closed:      boolean
        - stroke_name: string
        """
        def __init__(self,
                     cp_list,                   # [ControlPoint]
                     closed,                    # boolean
                     stroke_name = 'GimpStroke' # string
                     ):
            self.stroke_name = stroke_name
            self.cp_list = cp_list
            self.closed = closed
        def __str__(self):
            s = self.stroke_name + '; Control points:'
            for cp in self.cp_list:
                s += '\n    '+str(cp)
            s += '\n    closed: '+str(self.closed)
            return s
        def gs2bc(self):
            """Conversion GimpStroke -> BezierCurve
            """
            head = self.cp_list[0]
            tail = self.cp_list[-1]
            inner = self.cp_list[1:-1]
            if len(inner) > 1: # Must be divisible by 4
                ba_list = []
                count = 0
                for i in range(0,len(inner)-1,3):
                    ba_list.append(BCurve.BezierArc(inner[i:i+4], 'arc '+str(count)))
                    count += 1
            elif len(inner) == 1: # Stroke has only one anchor
                ba_list = [BCurve.BezierArc([inner[0]], 'arc 0')]
            else:
                raise Exception("BCurve.GimpStroke.gs2bc: No anchors in stroke?")
            return BCurve.BezierCurve(
                               bezier_arcs = ba_list,
                               head_handle = head,
                               tail_handle = tail,
                               closed = self.closed,
                               curve_name = self.stroke_name
                               )
    
    class BezierCurve(object):
        """BezierCurve is a list of butting Bezier arcs with some extra data.
        Attributes:
        - curve_name:  string
        - bezier_arcs: [BezierArc];
        - head_handle: ControlPoint;
        - tail_handle: ControlPoint;
        - closed:      boolean.
        Note: "Butting" means that for any pair of successive arcs, the last
              control point of the former equals the first control point of
              the latter.
              In initialization no checks are done about this condition.
        """
        def __init__(self,
                     bezier_arcs=[],           # [BezierArc]
                     head_handle=None,         # ControlPoint
                     tail_handle=None,         # ControlPoint
                     closed=False,             # boolean
                     curve_name='BezierCurve', # string
                     ):
            self.curve_name = curve_name
            self.bezier_arcs = bezier_arcs
            if head_handle is None:
                self.head_handle = bezier_arcs[0].cp4[0]
            else:
                self.head_handle = head_handle
            if tail_handle is None:
                self.tail_handle = bezier_arcs[-1].cp4[-1]
            else:
                self.tail_handle = tail_handle
            self.closed = closed
        def __str__(self):
            s = self.curve_name + ': Bezier curve' + '; Bezier arcs:'
            count = 0
            for arc in self.bezier_arcs:
                s += '\n  arc '+ str(count)+ ': ' +str(arc)
                count += 1
            s += '\n  Head handle:    '+str(self.head_handle)
            s += '\n  Tail handle:    '+str(self.tail_handle)
            s += '\n  Closed:'    +str(self.closed)
            return s
        def bc2gs(self):
            """Conversion BezierCurve -> GimpStroke
            """
            cp_list = [self.head_handle]
            if len(self.bezier_arcs[0].cp4) > 1:
                for ba in self.bezier_arcs:
                    cp_list += ba.cp4[:3]
                cp_list.append(self.bezier_arcs[-1].cp4[-1])
            else:
                # Only one BezierArc and it has cp4 = [p0]
                # (may rise from Gimp stroke with only one anchor).
                cp_list.append(self.bezier_arcs[0].cp4[0])
            cp_list.append(self.tail_handle)
            return BCurve.GimpStroke(cp_list, self.closed, self.curve_name)
        def bc2vectors_object(self, image, name=None):
            """Conversion BezierCurve -> gimp.Vectors
            Makes vectors_object with 1 stroke.
            """
            if name is None:
                name = self.curve_name
            gs = self.bc2gs()             # BCurve.GimpStroke
            gv = BCurve.GimpVectors([gs]) # BCurve.GimpVectors
            return gv.gv2vectors_object(image, name)
    
    class BezierArc(object):
        """Data structure for one Bezier arc: arc determined by 4 control
           points.
        Attributes:
        - cp4:      [p0,p1,p2,p3] where each pi:ControlPoint, or
                    [p0] if only one control point (may rise from
                    a stroke with only one anchor);
        - arc_name: string
        """
        def __init__(self,
                     cp4,                 # cp4=[p0,p1,p2,p3] with pi:ControlPoint
                                          # or [p0]
                     arc_name='BezierArc' # string
                     ):
            self.cp4 = cp4
            self.arc_name = arc_name
        def reverse(self):
            self.cp4.reverse()
        def __str__(self):
            s = 'BezierArc, control points:'
            for cp in self.cp4:
                s += '\n  '+ str(cp)
            return s

# -----------------
# vectors_object2gv
# -----------------
# Conversion gimp.Vectors -> GimpVectors
# Args:
# - vectors_object: gimp.Vectors (Gimp vectors object)
# Returns:
# - GimpVectors
#
# Note: The inverse conversion is GimpVectors.gv2vectors_object(...).
#       See also BezierCurve.bc2vectors_object(...).
def vectors_object2gv(vectors_object):
    def xy_list2cp_list(xy_list): # xy_list = [x,y,x,y,...,]
        cp = []
        for i in range(0,len(xy_list),2):
            cp.append(BCurve.ControlPoint(xy_list[i], xy_list[i+1]))
        return cp
    stroke_list = []
    count = 0
    for stroke in vectors_object.strokes:
        xy_list = stroke.points[0]
        closed = stroke.points[1]
        cp_list = xy_list2cp_list(xy_list)
        stroke_name = vectors_object.name + ' stroke '+str(count)
        stroke_list.append(BCurve.GimpStroke(cp_list, closed, stroke_name))
        count += 1
    return BCurve.GimpVectors(stroke_list, vectors_object.name)


#======================================================
#                   Polynomial zeroes
#======================================================
# My own algorithm, written for fun.
# Certainly could find faster elsewhere.
# --------------------
# zeroes_of_polynomial
# --------------------
# Given a real polynomial, find the real zeroes, the x values of extrema,
# and the x values of the inflection points.
# The polynomial c0 + c1*x + ... + cn*(x**n) is inputted as list [c0,c1,...,cn]
# of the coefficients.
# Results are returned:
# 1. None in the case of a zero polynomial.
# 2. Otherwise:
#    - the list of real zeroes of the polynomial, in increasing order,
#      possibly [], with multiple roots repeated;
#    - the extrema (the x values);
#    - the inflection points (the x values).
# Args:
# - coefficients: [float or integer]
# Returns: Either None (in case of zero input polynomial), or
# - [float] (the list of real zeroes of the polynomial, in increasing order,
#            multiple roots repeated);
# - [float] (the extrema (the x values));
# - [float] (the inflection points (the x values)).
def zeroes_of_polynomial(coefficients):
    from math import sqrt
    from functools import partial
    def f(x, coeff): # Horner
        s = 0.
        for ci in coeff[::-1]:
            s = ci + x*s
        return s
    coefficients = [float(ci) for ci in coefficients]
    # Find largest n with coefficients[n] non-zero and
    # discard all higher terms:
    n = -1
    for i in range(len(coefficients)):
        if coefficients[i] != 0.:
            n = i
    if n < 0: # Polynomial is identically zero.
        return None
    c = coefficients[:n+1] 
    # Make c[n] positive:
    if c[n] < 0:
        c = [-ci for ci in c]
    if n == 0: # Polynomial is a non-zero constant.
        return [],[],[]
    if n == 1: # Polynomial is c[0] + c[1]*x.
        return [-c[0] / c[1]], [], []
    if n == 2: # Polynomial is c[0] + c[1]*x + c[2]*(x**2).
        discr = c[1]**2 - 4*c[0]*c[2]
        try:
            root = sqrt(discr)
            x1 = (-c[1] - root) / (2*c[2])
            x2 = (-c[1] + root) / (2*c[2])
            return [x1,x2], [-c[1]/(2*c[2])], []
        except ValueError:
            return [], [-c[1]/(2*c[2])], []
    # Build subdivision such that in each subinterval
    # the polynomial is monotonous:
    derivative = [e*i for e,i in enumerate(c)][1:]
    extrema, inflection, _ = zeroes_of_polynomial(derivative) # Recursive call
    x_far = sum([abs(ci/c[-1]) for ci in c])
    subdiv = [-x_far] + extrema + [x_far]
    # Solve and collect the zeroes of each monotonous segment:
    fc = partial(f, coeff=c)
    zeroes = []
    for i in range(len(subdiv)-1):
        lo, hi = subdiv[i], subdiv[i+1]
        x = zero_of_monotonous_function(fc, lo, hi,
                                    accuracy=1e-10, tolerance=1e-10)
        if not(x is None):
            zeroes.append(x)
    return zeroes, extrema, inflection


# ---------------------------
# zero_of_monotonous_function
# ---------------------------
# Find the zero, if any, of a monotonous function f(x)
# in the interval [lo,hi]. If none is found, None is returned.
# Args:
# - f:         callable (real function, assumed to be monotonous in [lo,hi];
# - lo, hi:    float; (the interval)
# - accuracy:  accuracy for x.
# - tolerance: float>=0; if no zeroes exist in the interval, then
#              if f(lo) or f(hi) is less than 'tolerance' from 0,
#              then lo or hi is returned, respectively.
# Returns: None or the unique zero.
def zero_of_monotonous_function(f, lo, hi, accuracy=1e-10, tolerance=1e-10):
    MAX_ROUNDS = 100
    lo, hi = float(lo), float(hi)
    flo, fhi = float(f(lo)), float(f(hi))
    if (flo > 0.) == (fhi > 0.):
        if abs(flo) <= tolerance:
            return lo
        elif abs(fhi) <= tolerance:
            return hi
        else:
            return None
    if flo == 0.:
        return lo
    if fhi == 0.:
        return hi
    count = 0
    while hi-lo > accuracy:
        mid = (lo+hi)/2.
        fmid = float(f(mid))
        if fmid == 0.:
            return mid
        if (flo > 0.) == (fmid > 0.):
            lo, flo = mid, fmid
        else:
            hi, fhi = mid, fmid
        count += 1
        if count > MAX_ROUNDS:
            break
    x = (fhi*lo - flo*hi) / (fhi-flo)
    return x


#======================================================
#                 Closest stroke end
#======================================================

class StrokeEnd(object):
    """
    Record end of stroke:
    - start:         boolean (start or end anchor?)
    - anchor:        ControlPoint (=complex)
    - stroke_number: integer
    """
    def __init__(self, start, anchor, num):
        self.start = start
        self.anchor = anchor
        self.stroke_number = num
    def __str__(self):
        if self.start:
            case = "start:"
        else:
            case = "end  :"
        return(case + " anchor "+str(self.anchor)+ ", stroke number "+str(self.stroke_number))

class NoStrokeEndError(Exception):
    def __init__(self,m):
        self.message = m
    def __str__(self):
        return str(self.message)

# ---------------
# get_stroke_ends
# ---------------
# Args:
# - gs: GimpVectors
# Returns:
# - [StrokeEnd]
def get_stroke_ends(gv):
    result = []
    num = 0
    for gs in gv.stroke_list:
        if gs.closed:
            num += 1
            continue
        try:
            anchor_start = gs.cp_list[1] # first anchor
            anchor_end = gs.cp_list[-2]  # last anchor
        except IndexError:
            return result
        result.append(StrokeEnd(True, anchor_start, num))
        result.append(StrokeEnd(False, anchor_end, num))
        num += 1
    if len(result) == 0:
        raise NoStrokeEndError("All strokes are closed.") 
    return result

# ----------------------
# get_closest_stroke_end
# ----------------------
# Args:
# - point: RowVector (=complex)
# - gs:    GimpVectors
# Returns:
# - StrokeEnd
def get_closest_stroke_end(point, gv):
    ends = get_stroke_ends(gv)
    closest = min(ends, key=(lambda x: abs(x.anchor - point)))
    return closest


#======================================================
#           Miscellaneous auxiliary routines
#======================================================

def gimp_message(txt, h=2):
    # 0 = message box; 1 = console; 2 = error console
    pdb.gimp_message_set_handler(h)
    pdb.gimp_message(txt)

# ---------
# bezier_rv
# ---------
# The usual Bezier curve B(t) and derivatives from control points
# Version where plane points are complex numbers.
# Args:
# - t:float
# - control_points = [P0,P1,P2,P3]:[complex]
# Returns:
# - complex
def bezier_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    if t > 0.5:
        u = (1-t)/t
        t3 = t**3
        return t3*(u*(u*(u*P0+3*P1)+3*P2)+P3) # complex
    else:
        u = t/(1-t)
        t3 = (1-t)**3
        return t3*(u*(u*(u*P3+3*P2)+3*P1)+P0) # complex

# B'(t): Returns complex.
def bezier_dot_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    P01 = -P0+P1
    P12 = -P1+P2
    P23 = -P2+P3
    if t > 0.5:
        u = (1-t)/t
        t2 = t**2
        return 3*t2*(u*(u*P01+2*P12)+P23) # complex
    else:
        u = t/(1-t)
        t2 = (1-t)**2
        return 3*t2*(u*(u*P23+2*P12)+P01) # complex

# B''(t): Returns complex.
def bezier_dot_dot_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    P01 = -P0+P1
    P12 = -P1+P2
    P23 = -P2+P3
    return 6 * (-(1-t)*P01 + (1-2*t)*P12 +  + t*P23)

# B'''(t): Returns complex.
def bezier_dot_dot_dot_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    P01 = -P0+P1
    P12 = -P1+P2
    P23 = -P2+P3
    return 6 * (P01 - 2*P12 + P23)

# -------------------------
# bezier_new_control_points
# -------------------------
# Given a Bezier arc B(t) and two distinct parameter values a and b,
# find a new parametrization C(t) of the same curve such that
# B(a) = C(0) and B(b) = C(1).
# Args:
# - a: float;
# - b: float;
# - control_points: the list [p0,p1,p2,p3] of control points.
# Returns:
# - new control points [q0,q1,q2,q3]:[RowVector] such that the corresponding
#   Bezier curve C(t) is precisely the same as B as a curve
#   but B(a) = C(0) and B(b) = C(1).
# Notes:
# - This enables one to extend or to shorten the given Bezier arc.
# - There is an exception: If a=b, the returned control points
#   are equal, so the new curve is a single point, not the original
#   curve. The function gives no signal about such occurence.
def bezier_new_control_points(a, b, control_points):
    def bezier_2_rv(t, r0, r1, r2): # Bezier curve of degree two (complex)
        b20 = (1-t)**2
        b21 = 2*(1-t)*t
        b22 = t**2
        return b20*r0 + b21*r1 + b22*r2
    p0,p1,p2,p3 = control_points
    q0 = bezier_rv(a, control_points)
    q1 = (1-b)*bezier_2_rv(a, p0,p1,p2) + b*bezier_2_rv(a, p1,p2,p3)
    q2 = (1-a)*bezier_2_rv(b, p0,p1,p2) + a*bezier_2_rv(b, p1,p2,p3)
    q3 = bezier_rv(b, control_points)
    return [q0,q1,q2,q3]

# -------------------------
# bezier_arc_end_directions
# -------------------------
# Given a Bezier arc, find, at each end, the direction vector of the arc.
# Note: this does not mean the derivatives of the parametric representation,
# but non-zero vectors which are tangential to the arc and point in the
# direction where the arc is located when viewed from that end point.
# Return the result as two RowVectors (=complex),
# or as None if the arc is one point.
# Args:
# - ba: BCurve.BezierArc
# Returns: either None (when the arc is one point), or
# - RowVector (direction vector at starting point)
# - RowVector (direction vector at ending point)
def bezier_arc_end_directions(ba):
    ZERO = 1e-12
    if len(ba.cp4) == 1:
        return None
    p0,p1,p2,p3 = ba.cp4
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    if vdot(p01,p01) > ZERO:
        head_dir = p01
    elif vdot(p12,p12) > ZERO:
        head_dir = p12
    elif vdot(p23,p23) > ZERO:
        head_dir = p23
    else:
        return None
    if vdot(p23,p23) > ZERO:
        tail_dir = -p23
    elif vdot(p12,p12) > ZERO:
        tail_dir = -p12
    elif vdot(p01,p01) > ZERO:
        tail_dir = -p01
    else:
        raise Exception("bezier_arc_end_directions: Should never happen!")
    return head_dir, tail_dir

# ------------------------
# gimp_draw_vectors_object
# ------------------------
# The Gimp version of the routine 'drawing_routine_for_bezier':
# Insert into Gimp the input vectors object, and make it visible if
# visible=True.
# Args:
# - image: Gimp's image (obtained usually from GUI);
# - vectors_object: a vectors object in Gimp;
# - visible: Boolean.
# Returns the vectors object in Gimp, and makes it visible if visible=True.
def gimp_draw_vectors_object(image, vectors_object, visible=True):
    # Insert a Gimp vectors object in Gimp.
    # Adapted from http://registry.gimp.org/files/lqr_wpset-0.14.py.txt
    # by Mike Kazantsev.
    def gimp_insert_vectors(image, vectors_object, position=0):
        if gimp.version >= (2, 8, 0):
            # 2.7.0 has gimp-image-insert-vectors, but fails with
            # parent=None
            parent = None
            return pdb.gimp_image_insert_vectors(
                     image, vectors_object, parent, position)
        else:
            return pdb.gimp_image_add_vectors(
                     image, vectors_object, position)# Deprecated!
    gimp_insert_vectors(image, vectors_object)
    vectors_object.visible = visible
    return vectors_object

# ------------
# draw_polygon
# ------------
# Given a list of corner points p = [[x0,y0],[x1,y1], ... ]
# draw the polygon.
def draw_polygon(image,p, name='polygon', closed=True):
    polygon = pdb.gimp_vectors_new(image, name)
    points = []
    for xy in p:
        points += 3*xy
    pdb.gimp_image_undo_group_start(image) ### Gimp
    stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                        polygon, 0, len(points), points, closed)
    gimp_draw_vectors_object(image, polygon, visible=True)
    pdb.gimp_image_undo_group_end(image) ### Gimp
    return polygon

# point1 = [x1,y1], point2 = [x2,y2]
def draw_point_point(image, p1, p2, name='line segment'):
    return draw_polygon(image, [p1,p2], name=name, closed=False)


# ----------
# copy_chain
# ----------
# From Gimp's vectors object (gimp.Vectors) make a copy.
# Ignore all strokes except first (so make a 1-stroke path).
# Make the copy non-closed. (Assume that the original is.)
# If collapse=True, make the last control point equal the preceding control
# point (collaps the forward handle to zero).
# NOT inserted in Gimp.
# Args:
# - chain_vectors: gimp.Vectors
# - collaps:       boolean
# Returns:
# - gimp.Vectors
def copy_chain(chain_vectors, collaps):
    points = work_path.strokes[0].points[0]
    closed = False
    if collaps:
        points[-1] = points[-3]
        points[-2] = points[-4]
    name=chain_vectors.name+'|work_path copy'
    new = pdb.gimp_vectors_new(image, name)
    new.visible = False
    closed=False
    pdb.gimp_vectors_stroke_new_from_points(new, 0, len(points), points, closed)
    return new


# -----------
# get_anchors
# -----------
# Get anchors of the path (optionally the first stroke only).
# Arg:
# - vectors_object:    gimp.Vectors
# - only_first_stroke: boolean
# Returns:
# - [BCurve.ControlPoint]
def get_anchors(vectors_object, only_first_stroke=False):
    gv = vectors_object2gv(vectors_object) # BCurve.GimpVectors
    if len(gv.stroke_list) == 0:
        raise Exception("Path without any strokes: "+vectors_object.name)
    anchors = []
    for gs in gv.stroke_list:
        bc = gs.gs2bc()                        # BCurve.BezierCurve
        bas = bc.bezier_arcs                   # [BCurve.BezierArc]
        #anchors = [ba.cp4[0] for ba in bas]
        anchors += [ba.cp4[0] for ba in bas]
        if len(bas[0].cp4) > 1: # Case len=1 may rise from stroke with one anchor.
            anchors.append(bas[-1].cp4[-1])
        if only_first_stroke:
            break
    return anchors

# ----------------
# get_line_segment
# ----------------
# Read and check line segment from a 2-anchors path.
# Args:
# - line_segment: gimp.Vectors
# Returns:
# - [complex,complex]
def get_line_segment(line_segment, name = 'tentative segment'):
    gv_pointer = vectors_object2gv(line_segment)  # BCurve.GimpVectors
    if len(gv_pointer.stroke_list) != 1:
        m = "The "+ name + " should have 1 stroke, got "
        m += str(len(gv_pointer.stroke_list)) + "."
        raise Exception(m)
    gs_pointer = gv_pointer.stroke_list[0]
    bc_pointer = gs_pointer.gs2bc()
    if len(bc_pointer.bezier_arcs) != 1:
        m = "The "+ name + " should have 2 anchors, got "
        m += str(len(bc_pointer.bezier_arcs)+1) + "."
        raise Exception(m)
    ba_pointer = bc_pointer.bezier_arcs[0]
    try:
        cp4_pointer = ba_pointer.cp4
        return [cp4_pointer[0],cp4_pointer[3]]
    except IndexError:
        m = "The "+ name + " should have 2 anchors."
        raise Exception(m)

# ---------
# is_convex
# ---------
# Is a polygon convex? The polygon is inputted as
# a list of corner points (complex).
# Args:
# - corners: [complex]
# Returns:
# - boolean
# Notes:
# 1. Returns True if all corner points are collinear.
def is_convex(corners):
    if len(corners) <= 3:
        return True
    if collinear(corners): # TEE
        return True
    c = corners + corners[:2]
    edges = [c[i]-c[i+1] for i in range(len(c)-1)]
    edges = [e for e in edges if e != 0] # remove zero edges
    pairs = [[edges[i],edges[i+1]] for i in range(len(edges)-1)]
    # Cross and dot products:
    cr_dots = [[-a.real*b.imag + a.imag*b.real,
                 a.real*b.real + a.imag*b.imag] for a,b in pairs]
    crs = []
    for crd in cr_dots:
        if (crd[0]==0):
            if (crd[1]<0):
                return False # spike of angle 0
        else:
            crs.append(crd[0]) # collect non-zero cross products
    try:
        pos = (crs[0] > 0)
    except IndexError: # All points collinear
        return True
    for cr in crs:
        if pos != (cr > 0):
            return False
    return True

## v dot (w perpendicular)
def cross(v,w):
    return (v*w.conjugate()).imag

def vdot(v,w):
    return v.real*w.real + v.imag*w.imag

def collinear(points): # points: [complex] (The plane R2 is the complex plane.)
    ZERO = 1e-14
    n = len(points)
    if n < 3:
        return True
    for i in range(n):
        for j in range(i+1,n):
            for k in range(j+1,n):
                a,b,c = points[i], points[j], points[k]
                ab = -a+b
                ac = -a+c
                bc = -b+c
                if abs(cross(ab,ac)) > ZERO:
                    return False
                if abs(cross(ab,bc)) > ZERO:
                    return False
                if abs(cross(ac,bc)) > ZERO:
                    return False
    return True

# -------
# get_box
# -------
# Get box in the form [SW, SE, NE, NW] where items:complex and the directions
# are as on the screen (so, top is N, bottom is S).
# If Gimp=True, y coordinate runs downwards.
# Four cases: see crop_box_options.
# Args:
# - path_vectors: gimp.Vectors
# - base_vectors: gimp.Vectors
# - box_case:     string, one of the identifiers in crop_box_options
# - Gimp: boolean
# Returns:
# - [complex,complex,complex,complex]
def get_box(image, box_case, Gimp=True):
    if box_case == 'bb_selection':
        bounds = pdb.gimp_selection_bounds(image)
        if bounds[0] == 0:
            raise Exception("No active selection")
        W,N,E,S = bounds[1:]
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        return [sw,se,ne,nw]
    elif box_case == 'guides':
        hor,ver = get_guide_positions(image)
        if (len(hor) > 2) or (len(ver) > 2):
            raise Exception("Too many guides: must have two horizontal and two vertical.")
        if (len(hor) < 2) or (len(ver) < 2):
            raise Exception("Not enough guides: must have two horizontal and two vertical.")
        W,E = sorted(ver)
        if Gimp:
            N,S = sorted(hor)
        else:
            S,N = sorted(hor)
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        return [sw,se,ne,nw]
    elif box_case == 'image':
        h,w = image.height, image.width
        return [complex(0,h), complex(w,h), complex(w,0), complex(0,0)]
    elif box_case == 'layer':
        layer = pdb.gimp_image_get_active_layer(image) # Active layer
        h,w = pdb.gimp_drawable_height(layer), pdb.gimp_drawable_width(layer)
        offx, offy = pdb.gimp_drawable_offsets(layer)
        return [complex(offx,offy+h), complex(offx+w,offy+h),
                complex(offx+w,offy), complex(offx,offy)]
    else:
        raise Exception("get_box: invalid choice: "+str(box_case))

# -------------------
# get_guide_positions
# -------------------
# Guide positions in image, horizontal and vertical in separate lists.
# Not sorted.
def get_guide_positions(image):
    horizontal = []
    vertical = []
    guide = 0
    while True:
        guide = pdb.gimp_image_find_next_guide(image,guide)
        if guide == 0:
            break
        position = pdb.gimp_image_get_guide_position(image, guide)
        orientation = pdb.gimp_image_get_guide_orientation(image, guide)
        if orientation == ORIENTATION_HORIZONTAL:
            horizontal.append(position)
        elif orientation == ORIENTATION_VERTICAL:
            vertical.append(position)
        else:
            raise Exception("get_guide_positions: ???")
    return horizontal, vertical

# -------------------------
# solve_bernstein2_equation
# -------------------------
# Find real zeroes of the equation c0*b0(t) + c1*b1(t) + c2*b2(t) = 0 where
# b0,b1,b2 are the Bernstein polynomials of degree 2 and
# c0,c1,c2 are real coefficients.
# Return the zeroes as a list (0, 1, or 2 items).
# Exception: If the function is constant 0, return None.
# Args:
# - c0,c1,c2: float
# Returns:
# - None or list of floats (possibly empty)
def solve_bernstein2_equation(c0,c1,c2):
    from math import sqrt
    ZERO = 1e-14
    if abs(c0 - 2.*c1 + c2) < ZERO: # equation of degree <= 1
        try:
            return [c0 / (2.*(c0 - c1))]
        except ZeroDivisionError: # equation of degree 0
            if abs(c0) < ZERO: # constant 0
                return None
            else: # constant not 0
                return []
    try:
        root = sqrt(c1**2 - c0*c2)
    except ValueError: # No real roots
        return []
    return [(c0 - c1 + root) / (c0 - 2.*c1 + c2),
            (c0 - c1 - root) / (c0 - 2.*c1 + c2)]

# -------------------------
# solve_matrix_equation_2x2
# -------------------------
# Equation
# [a b][x] = [e]
# [c d][y]   [f]
# Args: matrix=[a,b,c,d],  vector=[e,f], all float
# Returns: [x,y]: [float,float]
# Exception if determinant is 0.
def solve_matrix_equation_2x2(matrix, vector):
    a,b,c,d = matrix
    e,f = vector
    try:
        inv_det = 1. / (a*d - b*c)
    except ZeroDivisionError:
        raise Exception("solve_matrix_equation_2x2: zero determinant")
    return [(d*e - b*f)*inv_det, (-c*e + a*f)*inv_det]


#======================================================
#                   Bounding box
#======================================================
# Bounding Box (BB) of a path.

# ---------------------------
# bezier_curve_tangent_points
# ---------------------------
# Given an infinite Bezier curve B(t) (4 control points)
# and a non-zero direction vector d,
# find the points on B with B'(t) parallel to d.
# If restrict01=True, accept only those with 0<=t<=1.
# Return as a list of pairs [t,p] where t is the parameter value (float)
# and p is the touching point B(t) (complex).
# Exception: If d=0, or if the curve is contained on a line parallel to d,
# or if len)control_points) is not 4 (1-anchor stroke!),
# return None.
# Args:
# - control_points: [complex,complex,complex,complex]
# - direction:      complex (assumed non-zero)
# - restrict01:     boolean
# Returns: None or
# - list [[t,p]]: [[float,complex]]   (=[[t,B(t)],...])
def bezier_curve_tangent_points(control_points, direction, restrict01=True):
    if len(control_points) < 4:
        return None
    p0,p1,p2,p3 = control_points
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    # pij dot (direction perpendicular):
    p01d = -p01.real * direction.imag + p01.imag * direction.real
    p12d = -p12.real * direction.imag + p12.imag * direction.real
    p23d = -p23.real * direction.imag + p23.imag * direction.real
    #
    ts = solve_bernstein2_equation(p01d,p12d,p23d)
    if ts is None:
        return None
    if restrict01:
        ts = [t for t in ts if 0<=t<=1]
    tBs = [[t, bezier_rv(t, control_points)] for t in ts]
    return tBs

# --------------------------
# BezierCurve_tangent_points
# --------------------------
# Given a Bezier curve:Bcurve.BezierCurve (essentially list of butting
# Bezier arcs) and a non-zero direction vector d,
# find the points on B where the curve is parallel to d.
# Includes cusps and isolated anchors.
# For arcs which are all the way parallel to d, the point
# with t=0.5 is returned
# Args:
# - bcurve: BCurve.BezierCurve
# - direction: complex (assumed non-zero)
# Returns:
# - [complex]
def BezierCurve_tangent_points(bcurve, direction):
    if not bcurve.closed:
        balist = bcurve.bezier_arcs
    else: # closed
        # In Gimp a closed stroke is only marked to be closed.
        # We must append one more arc to close the gap:
        gap_ba = BCurve.BezierArc(cp4 = [bcurve.bezier_arcs[-1].cp4[-1],
                                         bcurve.tail_handle,
                                         bcurve.head_handle,
                                         bcurve.bezier_arcs[0].cp4[0]])
        balist = bcurve.bezier_arcs + [gap_ba]
    cp4s = [ba.cp4 for ba in balist]
    points = []
    for cp4 in cp4s:
        if len(cp4) == 1: # A one-anchore stroke
            points.append(cp4[0])
            continue
        tBs = bezier_curve_tangent_points(cp4, direction, restrict01=True)
        if tBs is None: # arc is straight line segment parallel to direction
            t = 0.5
            p = bezier_rv(t, cp4)
            points.append(p)
            #t = 1.
            #p = bezier_rv(t, cp4)
            #points.append(p)
        else:
            for t,p in tBs:
                points.append(p)
    return points

# -------------------
# path_tangent_points
# -------------------
# Given Gimp's path and a non-zero direction vector d,
# find the points where some arc is parallel to d.
# Include cusps.
# Exception: For arcs which are all the way parallel, the point t=0.5 is chosen.
# Args:
# - path_vectors: gimp.Vectors
# - direction: complex (assumed non-zero)
# Returns:
# - [complex] (touching points)
def path_tangent_points(path_vectors, direction):
    gv = vectors_object2gv(path_vectors) # BCurve.GimpVectors
    points = []
    for gs in gv.stroke_list:  # BCurve.GimpStroke
        bc = gs.gs2bc()         # BCurve.BezierCurve
        points += BezierCurve_tangent_points(bc, direction)
    return points

# -------
# path_BB
# -------
# Bounding box of Gimp's path. Return as list [NW, SE] of two corner points,
# where NW is top left, SE is bottom right. (The directions are indeed as
# on the screen: N is up, S is down, provided that the argument 'Gimp' is correct!)
# Args:
# - path_vectors: gimp.Vectors
# - Gimp: boolean
# Returns:
# - [NW, SE]: [complex,complex]
def path_BB(path_vectors, Gimp=True):
    hor = path_tangent_points(path_vectors, 1+0j)
    ver = path_tangent_points(path_vectors, 0+1j)
    anchors = get_anchors(path_vectors, only_first_stroke=False)
    xmax = max([v.real for v in ver+anchors])
    xmin = min([v.real for v in ver+anchors])
    ymax = max([v.imag for v in hor+anchors])
    ymin = min([v.imag for v in hor+anchors])
    if Gimp:
        return [complex(xmin,ymin), complex(xmax,ymax)]
    else:
        return [complex(xmin,ymax), complex(xmax,ymin)]

# ------------
# path_BB_main
# ------------
# Draw the bounding box of a path.
# Args:
# - path_vectors:    gimp.Vectors
# - padding: float
# Returns:
# - gimp.Vectors
def path_BB_main(image, path_vectors, padding=0):
    nw, se = path_BB(path_vectors, Gimp=True)
    points = [[nw.real - padding, se.imag + padding],
              [se.real + padding, se.imag + padding],
              [se.real + padding, nw.imag - padding],
              [nw.real - padding, nw.imag - padding]]
    bb = draw_polygon(image,points, name=path_vectors.name + '|bb', closed=True)
    return bb


#======================================================
#       Tangents or normals from point or work_path
#======================================================
# Tangents or normals (TN) to path from a given point (= the last or
# second last anchor of a 1-stroke path).

TN_options = [ # (description, identifier)
        ('Draw tangents',  'tangent'),
        ('Draw normals',   'normal'),
        ]

common_TN_options = [ # (description, identifier)
        ('Draw tangent',  'tangent'),
        ('Draw normal',   'normal'),
        ]

draw_options = [ # (description, identifier)
        ('Draw only the closest',       'closest'),
        ('Draw all for the closest stroke',  'stroke'),
        ('Draw all',                    'all'),
        ]

class PathPointData(object):
    """Record from point on a path:
    - one_anchor_stroke: boolean (True if the stroke consist of one anchor only)
    - stroke_num (the number of the stroke in the path)
    - arc_num (the number of the Bezier arc in the stroke)
    - t (parameter value of the point on the infinite Bezier curve B(t))
    - Bt (the point B(t))
    - Bdot (the derivative B'(t))
    - Bdotdot (the second derivative B''(t))
    """
    def __init__(self,
                 one_anchor_stroke = False,
                 stroke_num = 0,
                 arc_num    = 0,
                 parameter  = 0,
                 Bt         = None,
                 Bdot       = None,
                 Bdotdot    = None
                 ):
        self.one_anchor_stroke = one_anchor_stroke
        self.stroke_number = stroke_num
        self.arc_number = arc_num
        self.parameter = parameter
        self.Bt = Bt
        self.Bdot = Bdot
        self.Bdotdot = Bdotdot
    def __str__(self):
        s  = 'stroke_number = '+str(self.stroke_number)
        s += '\narc_number    = '+str(self.arc_number)
        s += '\nparameter     = '+str(self.parameter)
        s += '\nB(t)          = '+str(self.Bt)
        s += '\nB\'(t)         = '+str(self.Bdot)
        s += '\nB\'\'(t)        = '+str(self.Bdotdot)
        return s


# ---------------
# point_TN_bcurve
# ---------------
# Given an infinite cubic Bezier curve B(t) (4 control points) and a point,
# find the parameter values t=t1,... such that the tangents or normals
# (depending on TN_case) of B(t) at t1,... pass through the given point.
# If restrict01=True, accept only those with 0<=t<=1.
# Return list of [t, B(t), B'(t), B''(t)] where t=t1,...
# Exception: If the curve is contained on a straight line and contains the point,
# return None.
# Args:
# - control_points: [complex,complex,complex,complex]
# - point: complex
# - restrict01:     boolean
# - TN_case:        string ('tangents' or 'normals'; see TN_options)
# Returns: None or
# - [[float, complex, complex, complex]]   (=[[t1, B(t1), B'(t1), B''(t1)],...])
def point_TN_bcurve(control_points,
             point,
             restrict01=True,
             TN_case='tangent'
             ):
    # Compute the coefficients a,b,c,d,e of a quartic or quintic polynomial
    # p(t) = a + bt + ct^2 + dt^3 + et^4 + ft^5 whose real zeroes are t1,...:
    ZERO = 1e-9
    p0,p1,p2,p3 = control_points
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    s0 = p0 - point
    s1 = 3*p01
    s2 = 6*(-p01 + p12)
    s3 = 6*(p01 -2*p12 + p23)
    if TN_case == 'tangent':
        coefficients = [cross(s1,s0),
                        cross(s2,s0),
                        (cross(s3,s0) + cross(s2,s1)) / 2,
                        cross(s3,s1) / 3,
                        cross(s3,s2) / 12
                        ]
    elif TN_case == 'normal':
        coefficients =  [vdot(s1,s0),
                         vdot(s2,s0) + vdot(s1,s1),
                         (vdot(s3,s0) + 3*vdot(s2,s1)) / 2,
                         (4*vdot(s3,s1) + 3*vdot(s2,s2)) / 6,
                         5 * vdot(s3,s2) / 12,
                         vdot(s3,s3) / 12
                         ]
    else:
        raise Exception("Unknown case: "+str(TN_case))
    # Solve the polynomial:
    zeroes = zeroes_of_polynomial(coefficients) # None or [float,float,float]
    if zeroes is None:
        return None
    else:
        ts = zeroes[0]
    if restrict01:
        ts = [t for t in ts if 0<=t<=1]
    # The solutions include cusps: points where B'(t)=0.
    # Generally these are discarded.
    # Check if the real tangent or normal passes through the point nevertheless;
    # if so, accept:
    checked = []
    for t in ts:
        Bd = bezier_dot_rv(t, control_points)
        if abs(Bd) > ZERO: # B'(t) not 0. Solution ok
            checked.append(t)
            continue
        else: # B'(t)=0. Must check B''(t) and perhaps B'''(t)
            Bdd = bezier_dot_dot_rv(t, control_points)
            if abs(Bdd) > ZERO: # B''(t) not 0
                direction = Bdd # direction of the curve
            else: # B'(t)=B''(t)=0. Must check B'''(t)
                Bddd = bezier_dot_dot_dot_rv(t, control_points)
                if abs(Bddd) > ZERO: # B'''(t) not 0
                    direction = Bddd # direction of the curve
                else: # B'(t)=B''(t)=B'''(t)=0. The curve is constant
                    return []
            Bt = bezier_rv(t, control_points)
            if TN_case == 'tangent':
                if abs(cross(Bt - point, direction)) < ZERO: # tangent goes through point
                    checked.append(t)
            elif TN_case == 'normal':
                if abs(vdot(Bt - point, direction)) < ZERO: # normal goes through point
                    checked.append(t)
            else:
                raise Exception("Unknown case: "+str(TN_case))
    return [[t,
             bezier_rv(t, control_points),
             bezier_dot_rv(t, control_points),
             bezier_dot_dot_rv(t, control_points)
             ]
            for t in checked]

# ------------------
# TN_data_from_point
# ------------------
# Given gv: BCurve.GimpVectors and a point, find all tangential points
# (or base points of normals, depending on TN_option) on gv when viewed from
# tentative[0].
# If draw_case = 'closest', tentative[1] is the reference point for
# choosing the closest one of all tangents (or normals).
# Return the list of the tangential points (or base points of normals, depending
# on TN_option) on gv as list of objects:PathPointData.
# Args:
# - gv_path:   BCurve.GimpVectors
# - tentative: [complex,complex]
# - TN_case:   string ('tangents' or 'normals'; see TN_options)
# - draw_case: string ('closest' or 'all'; see draw_options)
# Returns: None or
# - [PathPointData]
def TN_data_from_point(gv_path,
                       tentative,
                       TN_case, 
                       draw_case,
                       include_isolated_points=False):
    # Collect tangential points (or base points of normals) together with
    # the stroke_numbers:
    path_point_datas = [] # [PathPointData]
    gs_num = 0
    # Run strokes of gv_path
    for gs in gv_path.stroke_list:              # gs: BCurve.GimpStroke
        bc = gs.gs2bc()                         # BCurve.BezierCurve
        bas = bc.bezier_arcs                    # [BCurve.BezierArcs]
        if bc.closed: # To close the gap
            bas = bas + [(BCurve.BezierArc(cp4 = [bc.bezier_arcs[-1].cp4[-1],
                                                  bc.tail_handle,
                                                  bc.head_handle,
                                                  bc.bezier_arcs[0].cp4[0]
                                                  ]
                         ))]
        # Run Bezier arcs of gs
        arc_num = 0
        for ba in bas:                          # BCurve.BezierArc
            cp = ba.cp4                         # [complex] (control points )
            if len(cp) < 4: # Stroke with only one anchor
                if not include_isolated_points: 
                    break # ignore these
                path_point_datas.append(
                     PathPointData(one_anchor_stroke = True,
                                   stroke_num        = gs_num,
                                   arc_num           = 0,
                                   Bt                = cp[0],
                                   ))
                gs_num += 1
                break    # Go to next stroke
            else:
                t_Bt_Bd_Bdds = point_TN_bcurve(cp,             # Main call
                                 tentative[0],
                                 restrict01=True,
                                 TN_case=TN_case
                                 )
                if t_Bt_Bd_Bdds is None:
                    arc_num += 1
                    continue
                for t,Bt,Bdot,Bdotdot in t_Bt_Bd_Bdds:
                    path_point_datas.append(
                         PathPointData(one_anchor_stroke = False,
                                       stroke_num        = gs_num,
                                       arc_num           = arc_num,
                                       parameter         = t,
                                       Bt                = Bt,
                                       Bdot              = Bdot,
                                       Bdotdot           = Bdotdot
                                       ))
                arc_num += 1
        gs_num += 1
    #print("------------------  path_point_datas  ----------------")
    #for x in path_point_datas:
    #    print()
    #    print(str(x))
    if len(path_point_datas) == 0:
        return None
    else:
        closest_path_point = min(path_point_datas,
                              key=(lambda pp:abs(pp.Bt-tentative[1])))
        if draw_case == 'closest':
            selected_path_points = [closest_path_point] # Assuming only one!
        elif draw_case == 'stroke':
            selected_path_points = [pp for pp in path_point_datas if
                         pp.stroke_number == closest_path_point.stroke_number]
        else: # draw_case == 'all'
            selected_path_points = path_point_datas
        #print("\n------------------  selected_path_points  ----------------")
        #for x in selected_path_points:
        #    print()
        #    print(str(x))
        return selected_path_points



# -----------------------
# tangents_from_path_main
# -----------------------
# Given a target_path and a work_path (as gimp.Vectors),
# draw tangents to target_path (or normals, depending on TN_option)
# from a stroke end of work_path. The tangent (or normal) to be drawn is
# determined by tentative_segment which is supposed to be a user-provided guess
# where the tangent (or normal) should reside.
# Depending on draw_option, either only one tangent (or normal) is drawn,
# or all for one stroke is drawn,
# or all of for the the target_path are drawn.
# If extend>0, extend each drawn tangent (normal) with another similar line
# segment, in the same direction but with length multiplied by 'extend'.
# Note: Possibly no drawings will appear if the job is impossible
# (at least if the target_path is on a straight line).
# Returns the last drawn vectors object.
#
# Args:
# - target_path:    gimp.Vectors
# - work_path:      gimp.Vectors
# - tentative:      gimp.Vectors
# - TN_option:      integer (see TN_options)
# - draw_option: integer (see draw_options)
# - extend:         float
# Returns: None or
# - gimp.Vectors
def tangents_from_path_main(image,
                            target_path,
                            work_path,
                            tentative_segment,
                            TN_option,
                            draw_option,
                            extend=0.):
    ZERO = 1e-3
    TN_case   = TN_options[TN_option][1]
    draw_case = draw_options[draw_option][1]
    gv_path = vectors_object2gv(target_path)                # BCurve.GimpVectors
    gv_work = vectors_object2gv(work_path)              # BCurve.GimpVectors
    tentative = get_line_segment(tentative_segment)
    # Change tentative so that:
    # tentative[0] = the anchor of work_path stroke closest to either end of the
    #                      the original tentative
    # tentative[1] = the other end of the original tentative.
    try:
        closest_point0 = get_closest_stroke_end(tentative[0], gv_work)
        closest_point1 = get_closest_stroke_end(tentative[1], gv_work)
    except NoStrokeEndError as m:
        raise Exception(str(m) + "\nMust have at least one end of a non-closed stroke.")
    distance0 = abs(closest_point0.anchor - tentative[0])
    distance1 = abs(closest_point1.anchor - tentative[1])
    if distance0 > distance1:
        tentative.reverse()
        closest_point = closest_point1
    else:
        closest_point = closest_point0
    tentative[0] = closest_point.anchor
    #
    path_points = TN_data_from_point(gv_path, tentative, TN_case, draw_case) # [PathPointData]
    if path_points is None:
        return None
    tangents = [[tentative[0], pp.Bt] for pp in path_points]
    if len(tangents) == 0:
        return None
    # For each tangent, build a copy of gv_work with the stroke
    # (closest_point.stroke_number) extended by the tangent and possible
    # further extension:
    new_work_paths = []
    for tangent in tangents:
        new_work = deepcopy(gv_work)
        gs = new_work.stroke_list[closest_point.stroke_number]
        new_cp_list = gs.cp_list          # Copy
        if closest_point.start: # add extension at start
            new_cp_list.reverse() # so that additions will be to the end, not start
            reverse = True
        else:
            reverse = False
        last_anchor = new_cp_list[-2]
        TN_point = tangent[1] # The tangential point or the base point of normal
        new_cp_list[-1] = (2*last_anchor + TN_point) / 3
        new_cp_list += [(last_anchor + 2*TN_point) / 3,
                        TN_point,
                        (-last_anchor + 4*TN_point) / 3]
        if abs(extend) > ZERO: # Further extension
            ext_end = -extend*last_anchor + (1+extend)*TN_point
            new_cp_list[-1] = (2*TN_point + ext_end) / 3
            new_cp_list += [(TN_point + 2*ext_end) / 3,
                            ext_end,
                            ext_end]
        if reverse:
            new_cp_list.reverse()
        # Replace stroke:
        new_work.stroke_list[closest_point.stroke_number].cp_list = new_cp_list
        new_work_paths.append(new_work)
    pdb.gimp_image_undo_group_start(image)
    for new_work in new_work_paths:
        if TN_case == 'tangent':
            name = 'tangent'
        else:
            name = 'normal'
        new_vectors = new_work.gv2vectors_object(image, name=name)
        last_drawn = gimp_draw_vectors_object(image, new_vectors, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return last_drawn


# ------------------------
# tangents_from_point_main
# ------------------------
# Given a target_path, draw tangents to target_path from a point
# (or normals, depending on TN_option). The tangent (or normal) to be drawn is
# determined by tentative_segment which is supposed to be a user-provided guess
# where the tangent (or normal) should reside.
# One of the anchors of tentative_segment will be the starting point of the tangent
# (or normal), and the tangential point (or the base point of the normal)
# will be the other anchor, whichever choice rsults in a tangent closer to
# tentative_segment.
# Depending on draw_option, either only one tangent (or normal) is drawn,
# or all for one stroke is drawn,
# or all of for the the target_path are drawn from one end of tentative_segment.
# If extend>0, extend each drawn tangent (normal) with another similar line
# segment, in the same direction but with length multiplied by 'extend'.
# Note: Possibly no drawings will appear if the job is impossible
# (at least if the target_path is on a straight line).
# Returns the last drawn vectors object.
#
# Args:
# - target_path: gimp.Vectors
# - tentative:   gimp.Vectors
# - TN_option:   integer (see TN_options)
# - draw_option: integer (see draw_options)
# - extend:      float
# Returns: None or
# - gimp.Vectors
def tangents_from_point_main(image,
                             target_path,
                             tentative_segment,
                             TN_option,
                             draw_option,
                             extend=0.):
    ZERO = 1e-3
    TN_case   = TN_options[TN_option][1]
    draw_case = draw_options[draw_option][1]
    gv_path = vectors_object2gv(target_path)                # BCurve.GimpVectors
    tentative = get_line_segment(tentative_segment)
    # Find out which end of tentative is which:
    # Out of its two orientations choose the one which gives a tangent
    # closer to tentative.
    t0,t1 = tentative
    #
    t01s = TN_data_from_point(gv_path, [t0,t1], TN_case, 'closest')
    t10s = TN_data_from_point(gv_path, [t1,t0], TN_case, 'closest')
    if (t01s is None) or (t10s is None):
        m = "Something wrong: couldn't get t01s, t10s."
        m += "\nTry to change the tentative tangent a little."
        raise Exception(m)
    try:
        T01 = [t0,t01s[0].Bt]
        T10 = [t1,t10s[0].Bt]
    except IndexError:
        m = "Something wrong: couldn't get T01, T10."
        m += "\nTry to change the tentative tangent a little."
        raise Exception(m)
    
    dist01 = abs(T01[1] - tentative[1])
    dist10 = abs(T10[1] - tentative[0])
    if dist10 < dist01: # Change orientation: turn tentative around
        tentative.reverse()
    # tangents or normals:
    TN_points = TN_data_from_point(gv_path, tentative, TN_case, draw_case)
    if TN_points is None:
        return None
    if len(TN_points) == 0:
        return None
    tangents = [[tentative[0], pp.Bt] for pp in TN_points]
    pdb.gimp_image_undo_group_start(image)
    for tangent in tangents:
        if abs(extend) < ZERO:
            extended = tangent
        else:
            t0,t1 = tangent
            extended = [t0, t1, (-extend)*t0 + (1+extend)*t1]
        p = [[t.real, t.imag] for t in extended]
        if TN_case == 'tangent':
            name = gv_path.name+'|tangent'
        elif TN_case == 'normal':
            name = gv_path.name+'|normal'
        else: # never
            name = gv_path.name+'|'
        last_drawn = draw_polygon(image, p, name=name, closed=False)
    pdb.gimp_image_undo_group_end(image)
    return last_drawn


#======================================================
#              Parallel tangents or normals
#======================================================
# Tangents or normals (TN) to path, parallel to a given line segment.

# ------------------
# parallel_TN_bcurve
# ------------------
# Given an infinite Bezier curve B(t) (4 control points) and a line
# segment (2 points), find the parameter values t=t1,... such that
# the tangents or normals (depending on TN_case)
# of B(t) at t1,... are parallel to the line segment.
# If restrict01=True, accept only those with 0<=t<=1.
# Exception: If the curve is contained on a straight line parallel to the direction,
# return None.
# Args:
# - control_points: [complex,complex,complex,complex]
# - points: [complex,complex]
# - restrict01:     boolean
# - TN_case: string ('tangent' or 'normal'; see TN_options)
# Returns: None or
# - [[float,complex]]   (=[[t,B(t)],...])
def parallel_TN_bcurve(control_points,
                       points,
                       restrict01=True,
                       TN_case='tangent'
                       ):
    ZERO = 1e-16
    direction = -points[0] + points[1] # case 'tangent'
    if abs(direction) == 0:
        raise Exception("parallel_TN_bcurve: got zero line segment as input")
    if TN_case == 'tangent':
        pass
    elif TN_case == 'normal':
        direction = complex(-direction.imag, direction.real) # perpendicular
    else:
        raise Exception("Unknown case: "+str(TN_case))
    t_Bts = bezier_curve_tangent_points(control_points, direction, restrict01=False)
    if t_Bts is None:
        return None
    if restrict01:
        t_Bts = [t_Bt for t_Bt in t_Bts if 0 <= t_Bt[0] <= 1]
    # The solutions include points where B'(t)=0.
    # Generally these are discarded.
    # Check if the real tangent or normal passes through the point nevertheless;
    # if so, accept:
    checked = []
    for t,Bt in t_Bts:
        Bd = bezier_dot_rv(t, control_points)
        if abs(Bd) > ZERO: # B'(t) not 0. Solution ok
            checked.append(t)
            continue
        else: # B'(t)=0. Must check B''(t) and perhaps B'''(t)
            Bdd = bezier_dot_dot_rv(t, control_points)
            if abs(Bdd) > ZERO: # B''(t) not 0
                curve_direction = Bdd # direction of the curve
            else: # B'(t)=B''(t)=0. Must check B'''(t)
                Bddd = bezier_dot_dot_dot_rv(t, control_points)
                if abs(Bddd) > ZERO: # B'''(t) not 0
                    curve_direction = Bddd # direction of the curve
                else: # B'(t)=B''(t)=B'''(t)=0. The curve is constant
                    return []
            if TN_case == 'tangent':
                if abs(cross(direction, curve_direction)) < ZERO: # parallel
                    checked.append(t)
            elif TN_case == 'normal':
                if abs(vdot(direction, curve_direction)) < ZERO: # perpendicular
                    checked.append(t)
            else:
                raise Exception("Unknown case: "+str(TN_case))
    return [[t, bezier_rv(t, control_points)] for t in checked]


# ------------------------
# parallel_TN_translations
# ------------------------
# Given a gv_path:BCurve.GimpVectors and points = [p0,p1]:[complex,complex],
# find the translation vectors (complex numbers) that move the line segment
# 'points' to a tangent of target_path (or to a normal) such that the
# middle point of 'points' goes to the tangential point (or the base point
# of the normal).
# Whether these are tangents or normals, depends on TN_case.
# Which ones of the possible tangents (or normals) are accepted,
# depends on draw_case.
# Args:
# - target_path: gimp.Vectors
# - points:      [complex,complex]
# - TN_case:     string ('tangent' or 'normal'; see TN_options)
# - draw_case:   string ('closest' or 'stroke' or 'all'; see draw_options)
# Returns:
# - [float]

def parallel_TN_translations(gv_path,
                             points,
                             TN_case,
                             draw_case):
    class Translation(object): # Records translation of tentative
        def __init__(self, translation, gs_num):
            self.translation = translation
            self.stroke_number = gs_num
    middle = (points[0] + points[1]) / 2       # complex
    translations = []
    gs_num = 0
    for gs in gv_path.stroke_list:             # gs: BCurve.GimpStroke
        bc = gs.gs2bc()                        # BCurve.BezierCurve
        bas = bc.bezier_arcs                   # [BCurve.BezierArcs]
        if bc.closed: # To close the gap
            bas = bas + [(BCurve.BezierArc(cp4 = [bc.bezier_arcs[-1].cp4[-1],
                                                  bc.tail_handle,
                                                  bc.head_handle,
                                                  bc.bezier_arcs[0].cp4[0]
                                                  ]
                         ))]
        for ba in bas:                          # BCurve.BezierArc
            cp = ba.cp4                         # [complex] (control points )
            t_Bts = parallel_TN_bcurve(cp,
                                       points,
                                       restrict01=True,
                                       TN_case=TN_case)
            if t_Bts is None:
                continue
            for pp in [-middle + t_Bt[1] for t_Bt in t_Bts]:
                translations.append(Translation(pp, gs_num))
        gs_num += 1
    if len(translations) == 0:
        return []
    shortest_translation = min(translations,
                             key = (lambda pp: abs(pp.translation)))
    if draw_case == 'closest':
        sel_translations = [shortest_translation] # Assumes only one!
    elif draw_case == 'stroke':
        sel_translations = [pp for pp in translations
                            if pp.stroke_number == shortest_translation.stroke_number]
    elif draw_case == 'all':
        sel_translations = translations
    else:
        raise Exception("Unknown: "+str(draw_case))
    return [pp.translation for pp in sel_translations]


# ---------------------
# parallel_tangent_main
# ---------------------
# Given a target_path and tentative_segment (as gimp.Vectors), consider all
# Bezier arcs the target_path consists of.
# Depending on TN_option, either:
#     Find the points on each Bezier arc where the tangents are parallel to
#     a given line segment.
#     Draw the tangents as translates of tentative_segment with the center points
#     as the tangent points (marked).
# or:
#     Find the points on each Bezier arc where the normals are parallel to
#     a given line segment.
#     Draw the normals as translates of the tentative_segment with the center
#     points as the base points (marked).
# Depending on draw_option, either only one tangent (or normal) is drawn,
# or all for one stroke is drawn,
# or all of for the the target_path are drawn from one end of tentative_segment.
# Returns the last drawn vectors object.
# Args:
# - image
# - target_path:       gimp.Vectors
# - tentative_segment: gimp.Vectors (line segment: 2-anchors path)
# - TN_option:         integer (see TN_options)
# - draw_option:       integer (see draw_options)
# Returns: None or
# - gimp.Vectors
# Note: May draw nothing if the job is impossible
# (at least if the path is on a straight line).
def parallel_tangent_main(image,
                          target_path,
                          tentative_segment,
                          TN_option,
                          draw_option):
    TN_case   = TN_options[TN_option][1] # 'tangent' or 'normal'
    draw_case = draw_options[draw_option][1]
    gv_path = vectors_object2gv(target_path)   # BCurve.GimpVectors
    points = get_line_segment(tentative_segment)
    middle = (points[0] + points[1]) / 2       # complex
    translations = parallel_TN_translations(gv_path,
                                  points,
                                  TN_case,
                                  draw_case)
    if len(translations) == 0:
        return None
    pdb.gimp_image_undo_group_start(image)
    for translate in translations:
        tangent_anchors = [points[0] + translate,
                           middle + translate,
                           points[1] + translate]
        tangent_anchors_R2 = [[p.real, p.imag] for p in tangent_anchors]
        if TN_case == 'tangent':
            name = gv_path.name+'|tangent'
        elif TN_case == 'normal':
            name = gv_path.name+'|normal'
        else: # never
            name = gv_path.name+'|'
        last_drawn = draw_polygon(image, tangent_anchors_R2, name=name, closed=False)
    pdb.gimp_image_undo_group_end(image)
    return last_drawn

#======================================================
#         Common tangent or normal of two paths
#======================================================

# ---------
# common_TN
# ---------
# Called by common_tangent_main.
# Compute the tangent (or normal) for common_tangent_main.
# Args:
# - gv_path:   BCurve.GimpVectors
# - gv_other:  BCurve.GimpVectors
# - tentative: [complex,complex]
# - TN_case:       string ('tangent' or 'normal' for path; see TN_options)
# - TN_case_other: string ('tangent' or 'normal' for other path; see TN_options)
# - include_isolated_points: boolean (include 1-anchor strokes?)
# Returns:
# - [complex,complex] or [complex,complex,complex]
def common_TN(gv_path,
              gv_other,
              tentative,
              TN_case,
              TN_case_other,
              include_isolated_points
              ):
    # Check if this happens to be a common point of the curves with
    # correct positions of tangents.
    # Return True or False.
    # Raise Exception if cusp.
    def check_meeting(B,C,Bd,Cd, case, scale): 
        ZERO_B_C = 1e-4
        ZERO_bc  = 1e-4
        try:
            b = Bd/abs(Bd)
            c = Cd/abs(Cd)
        except ZeroDivisionError: # cusp
            raise Exception('cusp')
        if case in ('tangent:tangent', 'normal:normal'):
            return (abs(B-C)/scale < ZERO_B_C) and (abs(cross(b,c)) < ZERO_bc)
        elif case in ('tangent:normal', 'normal:tangent'):
            return (abs(B-C)/scale < ZERO_B_C) and (abs(vdot(b,c)) < ZERO_bc)
        else:
            raise Exception("Unknown case: "+case)
    def compute_error(cp4x, cp4y, x, y, case, scale):
        B = bezier_rv(x, cp4x)       # B(x)
        C = bezier_rv(y, cp4y)       # C(y)
        Bd = bezier_dot_rv(x, cp4x)  # B'(x)
        Cd = bezier_dot_rv(y, cp4y)  # C'(y)
        if case == 'tangent:tangent':
            error_coeff = [1,1]
        elif case == 'normal:normal':
            error_coeff = [1j,1j]
        elif case == 'tangent:normal':
            error_coeff = [1,1j]
        elif case == 'normal:tangent':
            error_coeff = [1j,1]
        u,v = error_coeff
        return non_parallel((B-C)/scale, u*Bd, v*Cd) # May raise error 'cusp'
    def non_parallel(a,b,c): # a,b,c:complex, how parallel?
        try:
            b = b/abs(b)
            c = c/abs(c)
        except ZeroDivisionError: # cusp
            raise Exception('cusp')
        return abs(cross(a,b)) + abs(cross(a,c)) + abs(cross(b,c))
    error_message = "No acceptable result found."
    error_message += "\nPerhaps none exists close to your tentative "+TN_case+","
    error_message += "\nor perhaps this is due to a shortcoming in the algorithm,"
    error_message += "\nbut you could try to set the tentative "+TN_case+" more precisely."
    #
    det_0_error_message = "Encountered a case that the plugin cannot handle."
    det_0_error_message += "\nSorry. (A determinant became 0)"
    #
    ZERO = 1e-10
    ZERO_DET = 1e-10
    MAX_ALLOWED_ERROR = 1e-8
    MIN_ALLOWED_CHORD = 1e-3 # When the returned chord is deemed to be one point
    #
    tentative_length = abs(tentative[0]-tentative[1])
    gv0,gv1 = gv_path, gv_other
    t0,t1 = tentative
    # From t0,t1 we find ppd0,ppd1:PathPointData so that ppd0 is data of point
    # on gv0 and ppd1 is data of point on gv1:
    # We shall also order t0,t1 so that t0 is close to gv0 and t1 is close to gv1
    # rather than vice versa.
    # Then  t0 ~ ppd0 in gv0  and  t1 ~ ppd1 in gv1.
    # First try, use orientation t0,t1
    p101s = TN_data_from_point(gv1,
                               [t0,t1],
                               TN_case_other,
                               'closest',
                               include_isolated_points) # [PathPointData]
    p010s = TN_data_from_point(gv0,
                              [t1,t0],
                              TN_case,
                              'closest',
                              include_isolated_points)  # [PathPointData]
    # Second try: revert orientation of t0,t1
    p110s = TN_data_from_point(gv1,
                               [t1,t0],
                               TN_case_other,
                               'closest',
                               include_isolated_points) # [PathPointData]
    p001s = TN_data_from_point(gv0,
                               [t0,t1],
                               TN_case,
                               'closest',
                               include_isolated_points) # [PathPointData]
    # Choose from the results:
    if None in (p101s,p010s) and None in (p110s,p001s): # Neither try ok: fail
        raise Exception(error_message)
    elif None in (p101s,p010s): # First try not ok. Try the second, hoping best.
        p110 = min(p110s, key=(lambda pp:abs(pp.Bt-t0)))
        p001 = min(p001s, key=(lambda pp:abs(pp.Bt-t1)))
        ppd0 = p001      # PathPointData of point on gv0
        ppd1 = p110      # PathPointData of point on gv1
        t0,t1 = t1,t0    # Turn tentative ends around
    elif None in (p110s,p001s): # Second try not ok. Try the first, hoping best.
        p101 = min(p101s, key=(lambda pp:abs(pp.Bt-t1)))
        p010 = min(p010s, key=(lambda pp:abs(pp.Bt-t0)))
        ppd0 = p010      # PathPointData of point on gv0
        ppd1 = p101      # PathPointData of point on gv1
    else: # Both tries ok: now choose which try gives the closest case
        p110 = min(p110s, key=(lambda pp:abs(pp.Bt-t0)))
        p001 = min(p001s, key=(lambda pp:abs(pp.Bt-t1)))
        p101 = min(p101s, key=(lambda pp:abs(pp.Bt-t1)))
        p010 = min(p010s, key=(lambda pp:abs(pp.Bt-t0)))
        dist01 = abs(t0-p010.Bt) + abs(t1-p101.Bt)
        dist10 = abs(t1-p001.Bt) + abs(t0-p110.Bt)
        if dist10 > dist01:
            ppd0 = p010   # PathPointData of point on gv0
            ppd1 = p101   # PathPointData of point on gv1
            t0,t1 = t1,t0 # Turn tentative ends around
        else:
            ppd0 = p001   # PathPointData of point on gv0
            ppd1 = p110   # PathPointData of point on gv1
    # After this we work only one Bezier curve from either path:
    gs0 = gv0.stroke_list[ppd0.stroke_number] # BCurve.GimpStroke
    gs1 = gv1.stroke_list[ppd1.stroke_number] # BCurve.GimpStroke
    bc0 = gs0.gs2bc()       # BCurve.BezierCurve
    bc1 = gs1.gs2bc()       # BCurve.BezierCurve
    # Now we restrict the work to one Bezier curve B(t) and C(t)
    # given by 4 control points from either path:
    try:
        Bba = bc0.bezier_arcs[ppd0.arc_number] # BCurve.BezierArc (B(t))
    except IndexError: # Closed stroke: must close the gap with one more arc
        Bba = BCurve.BezierArc(cp4 = [bc0.bezier_arcs[-1].cp4[-1],
                                      bc0.tail_handle,
                                      bc0.head_handle,
                                      bc0.bezier_arcs[0].cp4[0]])
        if not gs0.closed:
            raise Exception("Something weird: IndexError from arcs but stroke not closed!")
    try:
        Cba = bc1.bezier_arcs[ppd1.arc_number] # BCurve.BezierArc (C(t))
    except IndexError: # Closed stroke: must close the gap with one more arc
        Cba = BCurve.BezierArc(cp4 = [bc1.bezier_arcs[-1].cp4[-1],
                                      bc1.tail_handle,
                                      bc1.head_handle,
                                      bc1.bezier_arcs[0].cp4[0]])
        if not gs1.closed:
            raise Exception("Something weird: IndexError from arcs but stroke not closed!")
    # Control points:
    Bcp4 = Bba.cp4 # [p0,p1,p2,p3] or [p0] (control points of B(t), pi:complex)
    Ccp4 = Cba.cp4 # [q0,q1,q2,q3] or [q0] (control points of C(t), qi:complex))
    if TN_case == 'tangent' and TN_case_other == 'tangent':
        case = 'tangent:tangent'
    elif TN_case == 'normal' and TN_case_other == 'normal':
        case = 'normal:normal'
    elif TN_case == 'tangent' and TN_case_other == 'normal':
        case = 'tangent:normal'
    elif TN_case == 'normal' and TN_case_other == 'tangent':
        case = 'normal:tangent'
    # Special cases: one-anchor strokes (control points: [p0] and/or [q0]):
    if ppd0.one_anchor_stroke and ppd1.one_anchor_stroke:
        return [ppd0.Bt, ppd1.Bt]
    elif ppd0.one_anchor_stroke:
        t_B_Bd_Bdds = point_TN_bcurve(Ccp4,
                                      ppd0.Bt,
                                      restrict01=True,
                                      TN_case=TN_case_other
                                      )
        if t_B_Bd_Bdds is None:
            raise Exception(error_message)
        t_B_Bd_Bdd = min(t_B_Bd_Bdds, key=(lambda xxx:abs(xxx[1]-t1)))
        return [ppd0.Bt, t_B_Bd_Bdd[1]]
    elif ppd1.one_anchor_stroke:
        t_B_Bd_Bdds = point_TN_bcurve(Bcp4,
                                      ppd1.Bt,
                                      restrict01=True,
                                      TN_case=TN_case
                                      )
        if t_B_Bd_Bdds is None:
            raise Exception(error_message)
        t_B_Bd_Bdd = min(t_B_Bd_Bdds, key=(lambda xxx:abs(xxx[1]-t0)))
        return [ppd1.Bt, t_B_Bd_Bdd[1]]
    # General case: neither is a one-anchor stroke.
    # Parameter values of the points on gv0 and gv1 found above,
    # viewed as points on the Bezier curves B(t):
    x = ppd0.parameter # parameter value for the point on gv0
    y = ppd1.parameter # parameter value for the point on gv1
    # Start iteration using linear approximation at each step:
    error = compute_error(Bcp4, Ccp4, x, y, case, tentative_length)
    #print("Initial: error = "+str(error))
    for i in range(50):
        #print("i="+str(i)+": at start of loop: error = "+str(error))
        B = bezier_rv(x, Bcp4) # B(x)
        C = bezier_rv(y, Ccp4) # C(y)
        Bd = bezier_dot_rv(x, Bcp4)  # B'(x)
        Cd = bezier_dot_rv(y, Ccp4)  # C'(y)
        Bdd = bezier_dot_dot_rv(x, Bcp4)  # B''(x)
        Cdd = bezier_dot_dot_rv(y, Ccp4)  # C''(y)
        
        #Bvector = [B.real,B.imag]
        #Cvector = [C.real,C.imag]
        #draw_polygon(image, [Bvector,Cvector], name=str(i), closed=False)
        
        if case == 'tangent:tangent':
            matrix = [cross(B-C, Bdd),  cross(Bd, Cd),
                      cross(Bd, Cd),    cross(B-C, Cdd)]
            vector = [-cross(B-C, Bd),
                      -cross(B-C, Cd)]
            M = max(abs(m) for m in matrix)
            matrix = [m/M for m in matrix]
            vector = [m/M for m in vector]
        elif case == 'normal:normal':
            matrix = [vdot(B-C, Bdd) + vdot(Bd,Bd),  -vdot(Bd, Cd),
                      vdot(Bd, Cd),                  vdot(B-C, Cdd) - vdot(Cd,Cd)]
            vector = [-vdot(B-C, Bd),
                      -vdot(B-C, Cd)]
            M = max(abs(m) for m in matrix)
            matrix = [m/M for m in matrix]
            vector = [m/M for m in vector]
        elif case == 'tangent:normal':
            matrix = [cross(B-C, Bdd),  cross(Bd, Cd),
                      vdot(Bd, Cd),     vdot(B-C, Cdd) - vdot(Cd,Cd)]
            vector = [-cross(B-C, Bd),
                      -vdot(B-C, Cd)]
            M = max(abs(m) for m in matrix)
            matrix = [m/M for m in matrix]
            vector = [m/M for m in vector]
        elif case == 'normal:tangent':
            matrix = [vdot(B-C, Bdd) + vdot(Bd,Bd),  -vdot(Bd, Cd),
                      cross(Bd, Cd),                  cross(B-C, Cdd)]
            vector = [-vdot(B-C, Bd),
                      -cross(B-C, Cd)]
            M = max(abs(m) for m in matrix)
            matrix = [m/M for m in matrix]
            vector = [m/M for m in vector]
        else:
            raise Exception("Unknown case: "+case)
        if abs(matrix[0]*matrix[3] - matrix[1]*matrix[2]) < ZERO_DET:
            #print("det = "+str(matrix[0]*matrix[3] - matrix[1]*matrix[2]))
            #raise Exception(det_0_error_message)
            break
        dx,dy = solve_matrix_equation_2x2(matrix, vector) # Result from linear appr.
        # First try using full dx, dy
        if (0 <= x+dx <= 1) and (0 <= y+dy <= 1): # Better: min,max?
            new_x, new_y = x+dx, y+dy
            try:
                new_error = compute_error(Bcp4, Ccp4, new_x, new_y, case, tentative_length)
                #print("new_error = "+str(new_error))
                if new_error < MAX_ALLOWED_ERROR:
                    x, y, error = new_x, new_y, new_error # Finish
                    #print("break at round "+str(i)+", error = "+str(error))
                    break    # Found. Break i-loop
                if new_error < error:
                    x, y, error = new_x, new_y, new_error
                    continue # Ok. Go to next round in i-loop
                else:
                    #print(" "*20+"new_error = "+str(new_error)+". Going to try-loop")
                    pass     # Not ok. Go to try-loop
            except Exception as e: # 'cusp'?
                raise Exception(e)
        else:
            #print("Not in [0,1]. Going to try-loop")
            pass # Go to try-loop
        # Try-loop: loop over smaller dx,dy (full dx,dy was not acceptable)
        N = 8
        try_x_y_errors = []
        run_list = list(range(-1,-N-1,-1)) + list(range(N+1))
        for xj in run_list:
            try_x = min(1., max(x + xj*dx/N, 0.))
            for yj in run_list:
                try_y = min(1., max(y + yj*dy/N, 0.))
                try:
                    try_error = compute_error(Bcp4, Ccp4, try_x, try_y, case, tentative_length)
                    #print("    xj,yj = "+str([xj,yj])+": try_error = "+str(try_error))
                except Exception as e:
                    raise Exception(e)
                try_x_y_errors.append([try_x, try_y, xj, yj, try_error])
                if try_error < MAX_ALLOWED_ERROR:
                    #print("break try-loop at round "+str([xj,yj])+", try_error = "+str(try_error))
                    #print("try_x, try_y = "+str(try_x) +",  "+str(try_y))
                    break
        if len(try_x_y_errors) == 0:
            #print("Even try-loop gave nothing: break")
            break
        best_try = min(try_x_y_errors, key=(lambda e:e[-1]))
        _,_,xj,yj,_ = best_try
        #print("best_try:    xj, yj = "+str([xj,yj])+ ";    best error = " +str(best_try[-1]))
        if best_try[-1] < error:
            x,y,_,_,error = best_try
        else:
            #print("Nothing better from try-loop: break")
            break
        #print("i="+str(i)+":   at end of loop: error = "+str(error))
        #print("x, y = "+str(x) +",  "+str(y))
        if error < MAX_ALLOWED_ERROR:
            #print("break at round "+str(i)+", error = "+str(error))
            break
        #print()
    else:
        #print("No break, error = "+str(error))
        pass
    B = bezier_rv(x, Bcp4) # B(x)
    C = bezier_rv(y, Ccp4) # C(y)
    Bd = bezier_dot_rv(x, Bcp4)  # B'(x)
    Cd = bezier_dot_rv(y, Ccp4)  # C'(y)
    
    if (error > MAX_ALLOWED_ERROR) or (abs(B-C) < MIN_ALLOWED_CHORD):
        #if (error > MAX_ALLOWED_ERROR):
        #    print("Failure with error = "+str(error))
        #else: # (abs(B-C) < MIN_ALLOWED_CHORD)
        #    print("abs(B-C) = "+str(abs(B-C)) +" < MIN_ALLOWED_CHORD = "+str(MIN_ALLOWED_CHORD))
        # In the case of algorithm failure, check if this happens to be a common
        # point of the curves with correct positions of tangents:
        if check_meeting(B,C,Bd,Cd, case, tentative_length): # Ok
            #print("check_meeting = True")
            # Return a 3-anchors array:
            middle = (B+C)/2
            if case in ('tangent:tangent', 'tangent:normal'):
                direction = Bd / abs(Bd)
            else:
                direction = 1j * Bd / abs(Bd)
            B0 = middle + tentative_length * direction / 2
            B1 = middle - tentative_length * direction / 2
            return [B0,middle,B1]
        else:
            #print("check_meeting = False")
            pass
        raise Exception(error_message)
    return [B,C]

# -------------------
# common_tangent_main
# -------------------
# Given a path and a other_path (as gimp.Vectors),
# draw common tangents (or normals, depending on TN_option).
# The tangent (or normal) to be drawn is determined by tentative_segment which
# is supposed to be a user-provided guess where the tangent (or normal) should
# reside.
# If extend>0, extend the tangent (normal) with similar line segment
# at each end, with length multiplied by 'extend'.
# Notes: 
# 1. Possibly no drawings will appear if the job is impossible
#    (at least if the target_path is on a straight line).
# 2. Special case: If it happens that the paths meet and at the meeting point
#    they have a common (or normal...), that tangent (or normal) is drawn
#    at the length of the tentative_segment, possibly extended.
# Args:
# - gv_path:       BCurve.GimpVectors
# - gv_other:      BCurve.GimpVectors
# - tentative:     [complex,complex]
# - TN_case:       string ('tangent' or 'normal' for path; see TN_options)
# - TN_case_other: string ('tangent' or 'normal' for other path; see TN_options)
# - extend:        float
# Returns:
# - gimp.Vectors
def common_tangent_main(image,
                        path,
                        other_path,
                        tentative_segment,
                        TN_option,
                        TN_option_other,
                        extend=0.):
    # 1-anchor strokes: If you change this to True, it should work!
    include_isolated_points = False # No option for the user!
    ZERO = 1e-3
    TN_case   = TN_options[TN_option][1]
    TN_case_other = TN_options[TN_option_other][1]
    if path == other_path:
        m = "Both paths are the same."
        m += "\nAre you are trying to find a tangent or a normal of the path?"
        m += "\nIf so, please use some other plugin."
        raise Exception(m)
    gv_path = vectors_object2gv(path)                # BCurve.GimpVectors
    gv_other = vectors_object2gv(other_path)              # BCurve.GimpVectors
    tentative = get_line_segment(tentative_segment)
    tangent = common_TN(gv_path,                     # len(tangent) = 2 or 3
                       gv_other,
                       tentative,
                       TN_case,
                       TN_case_other,
                       include_isolated_points
                       )
    if abs(extend) < ZERO:
        extended = tangent
    elif len(tangent) == 2:
        t0,t1 = tangent
        extended = [(1+extend)*t0 - extend*t1, t0, t1, (-extend)*t0 + (1+extend)*t1]
    elif len(tangent) == 3:
        t0,middle,t1 = tangent
        extended = [(1+extend)*t0 - extend*t1, t0, middle, t1, (-extend)*t0 + (1+extend)*t1]
    if (TN_case == 'tangent') and (TN_case_other == 'tangent'):
        name = 'tangent'
    elif (TN_case == 'normal') and (TN_case_other == 'normal'):
        name = 'normal'
    elif (TN_case == 'tangent') and (TN_case_other == 'normal'):
        name = 'tangent,normal'
    elif (TN_case == 'normal') and (TN_case_other == 'tangent'):
        name = 'normal,tangent'
    else: # never
        name = '' 
    name = path.name +","+other_path.name + "|"+name
    pdb.gimp_image_undo_group_start(image)
    p = [[t.real, t.imag] for t in extended]
    result = draw_polygon(image, p, name=name, closed=False)
    pdb.gimp_image_undo_group_end(image)
    return result


#======================================================
#                     Registrations
#======================================================

versionnumber = "0.8"
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
image_types = "*"
menupath = '<Vectors>/Tools/Tangents and normals'


#####################  Bounding box  #####################

procedure_name  = "bounding_box_of_path"
procedure_blurb = ("Draw the bounding box of a path."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Draw the bounding box of a path"
procedure_label = "Bounding box"

procedure_function = path_BB_main


register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path_vectors", "The path", None),
      (PF_FLOAT, "padding", "Padding", 0.)
    ],
    [
        (PF_VECTORS, "bb", "The bounding box"),
    ],
    procedure_function,
    menu=menupath)


#################  Tangents or normals from a stroke end of a path #################

procedure_name  = "tangents_of_path_from_stroke_ends"
procedure_blurb = ("Draw tangents (or normals) to a path"
                   +"\nfrom a stroke end of another path,"
                   +"\ngiving a 2-anchors path as a tentative tangent (or normal)."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Draw tangents or normals to a path from stroke end of another path."
procedure_label = "Tangents or normals from a stroke end of another path"

procedure_function = tangents_from_path_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "target_path", "The path to receive the tangents (or normals)", None),
      (PF_VECTORS, "work_path", "The path from whose stroke end the tangents (or normals) are drawn", None),
      (PF_VECTORS, "tentative_segment", "Tentative tangent or normal (a 2-anchors path)", None),
      (PF_OPTION, 'TN_option','Tangents or normals?',
                   0,
                  [case[0] for case in TN_options]), # Descriptions of cases
      (PF_OPTION, 'draw_option','Action',
                   0,
                  [case[0] for case in draw_options]), # Descriptions of cases
      (PF_FLOAT, "extend", ("Extend the result by factor"
                            +"\n(0 => no extension)"), 0.)
    ],
    [
    ],
    procedure_function,
    menu=menupath)

#################  Tangents or normals from a point  #################

procedure_name  = "tangents_of_path_from_point"
procedure_blurb = ("Draw tangents (or normals) to a path from a point,"
                   +"\ngiving a 2-anchors path as a tentative tangent (or normal)."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Draw tangents or normals to a path from a point."
procedure_label = "Tangents or normals from a point"

procedure_function = tangents_from_point_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "target_path", "The path", None),
      (PF_VECTORS, "tentative_segment", "Tentative tangent or normal (a 2-anchors path)", None),
      #(PF_BOOL, "use_tentative_reversed", "Use the tentative tangent reversed?", False),
      (PF_OPTION, 'TN_option','Tangents or normals?',
                   0,
                  [case[0] for case in TN_options]), # Descriptions of cases
      (PF_OPTION, 'draw_option','Action',
                   0,
                  [case[0] for case in draw_options]), # Descriptions of cases
      (PF_FLOAT, "extend", ("Extend the result by factor"
                            +"\n(0 => no extension)"), 0.)
    ],
    [
    ],
    procedure_function,
    menu=menupath)


##################  Parallel tangents or normals  #################
#
procedure_name  = "parallel_tangents_of_path"
procedure_blurb = ("Draw tangents (or normals) to a path,"
                   "\nparallel to a given tentative tangent (or normal)."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Draw parallel tangents or normals to a path."
procedure_label = "Parallel tangents or normals"

procedure_function = parallel_tangent_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "target_path", "The path", None),
      (PF_VECTORS, "tentative_segment",
                   "Tentative tangent or normal (a 2-anchors path)",
                   None),
      (PF_OPTION, 'TN_option','Tangents or normals?',
                   0,
                  [case[0] for case in TN_options]), # Descriptions of cases
      (PF_OPTION, 'draw_option','Action',
                   0,
                  [case[0] for case in draw_options]), # Descriptions of cases
    ],
    [
    ],
    procedure_function,
    menu=menupath)

#################  Common tangents or normals of two paths #################

procedure_name  = "common_tangent"
procedure_blurb = ("Draw a common tangent (or normal) between a path"
                   +" and another path,"
                   +"\ngiving a 2-anchors path as a tentative tangent (or normal)."
                   #+"\nNote: Points where the two paths meet are excluded."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Draw a common tangent or normal between the path and another path."
procedure_label = "Common tangent or normal between two paths"

procedure_function = common_tangent_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path", None),
      (PF_VECTORS, "other_path", "Another path", None),
      (PF_VECTORS, "tentative_segment", "Tentative tangent or normal (a 2-anchors path)", None),
      (PF_OPTION, 'TN_option','Tangent or normal of the path?',
                   0,
                  [case[0] for case in common_TN_options]), # Descriptions of cases
      (PF_OPTION, 'TN_option_other','Tangent or normal of the other path?',
                   0,
                  [case[0] for case in common_TN_options]), # Descriptions of cases
      #(PF_OPTION, 'draw_option','Action',
      #             0,
      #            [case[0] for case in draw_options]), # Descriptions of cases
      (PF_FLOAT, "extend", ("Extend the result by factor at each end"
                            +"\n(0 => no extension)"), 0.)
    ],
    [
    ],
    procedure_function,
    menu=menupath)

################################################

main()


