#!/usr/bin/env python

# Gimp plugins to transform vectors objects by some transformations.
#
# History:
# v0.5: 2020-04-22: First published version.
# v0.6: 2020-04-24: Added some properties.
# v0.9: 2020-05-04: Added transformation of general paths by a Bezier arc.
#                   Dropped plugin Curl a polyline by a Bezier arc.
# v0.12: 2020-06-08: Added Moebius transformation.
# v0.15: 2020-07-12: Added transformation by a Bezier arc quadrilateral.
# v0.16: 2020-07-14: Added transformation by a Bezier arc quadrilateral (easy).
# v0.17: 2020-07-16: Added options in the transformation by a Bezier arc quadrilateral (easy).
# v0.18: 2020-07-25: Changed options in the transformation by a Bezier arc quadrilateral (easy).
# v0.19: 2020-08-05: Renovated transformation by a Bezier arc quadrilateral (advanced).
#                    Added plugins to fit a path in a triangle or in a quadrangle.
# v0.20: 2020-08-11: Developed the plugin "fit a path in a quadrangle" further.
# v0.22: 2020-09-02: Added plugin to convert a path to polar coordinates.
# v0.24: 2020-09-17: Added transformation by complex exponential map.


# (c) Markku Koppinen 2020
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'


from __future__ import division, print_function 
from gimpfu import *                            
from math import *                              
from functools import partial                   


#==============================================================
#             class BCurve       
#==============================================================
# Class: BCurve.

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

RowVector = complex

class BCurve(object):
    """Data structures to handle Gimp vectors objects by means of individual
    Bezier arcs.
    By a "Bezier arc" we mean here a Bezier curve determined by 4 control points
    p0,p1,p2,p3.
    This class contains data structures needed to do computations on
    a Bezier curve which are done arc by arc.
    To this end, Gimp data structures
    - gimp.Vectors
    - gimp.VectorsBezierStroke
    are here represented as data structures (classes)
    - GimpVectors
    - GimpStroke
    - BezierCurve
    - BezierArc
    - ControlPoint = complex
    with the aim that
    - each control point [x,y] is represented as an instance of ControlPoint;
    - each Bezier arc (four control points, see above) is represented as an instance
      of BezierArc.
    """
    
    ControlPoint = RowVector # =complex
    
    class GimpVectors(object):
        """Essentially same data as in a gimp.Vectors object except that
        strokes are instances of GimpStroke rather than
        gimp.VectorsBezierStroke.
        Attributes:
        - stroke_list: [GimpStroke]
        - name:        string
        """
        def __init__(self,
                     stroke_list,            # [GimpStroke]
                     name='GimpVectors'      # string
                     ):
            self.stroke_list = stroke_list
            self.name = name
        def __str__(self):
            s = 'GimpVectors '+self.name
            count = 0
            for stroke in self.stroke_list:
                s += '\n  Stroke '+str(count)+': GimpStroke'
                s += '\n    '+str(stroke)
                count += 1
            return s
        def gv2vectors_object(self, image, name=None):
            """Conversion GimpVectors -> gimp.Vectors
            (The converse: vectors_object2gv)
            """
            def cp_list2xy_list(cp_list):
                xy = []
                for cp in cp_list:
                    xy += [cp.real, cp.imag]
                return xy
            if name is None:
                name = self.name
            vectors = pdb.gimp_vectors_new(image, name)
            for stroke in self.stroke_list:
                closed = stroke.closed
                xy_list = cp_list2xy_list(stroke.cp_list)
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                             vectors, 0, len(xy_list), xy_list, closed)
            return vectors
    
    class GimpStroke(object):
        """Essentially same data as in a gimp.VectorsBezierStroke.points,
        except that control points list [x,y,x,y,...] is arranged as
        a list [ControlPoint] by joining successive x,y.
        Attributes:
        - cp_list:     [ControlPoint]
        - closed:      boolean
        - stroke_name: string
        """
        def __init__(self,
                     cp_list,                   # [ControlPoint]
                     closed,                    # boolean
                     stroke_name = 'GimpStroke' # string
                     ):
            self.stroke_name = stroke_name
            self.cp_list = cp_list
            self.closed = closed
        def __str__(self):
            s = self.stroke_name + '; Control points:'
            for cp in self.cp_list:
                s += '\n    '+str(cp)
            s += '\n    closed: '+str(self.closed)
            return s
        def gs2bc(self):
            """Conversion GimpStroke -> BezierCurve
            """
            head = self.cp_list[0]
            tail = self.cp_list[-1]
            inner = self.cp_list[1:-1]
            if len(inner) > 1: # Must be divisible by 4
                ba_list = []
                count = 0
                for i in range(0,len(inner)-1,3):
                    ba_list.append(BCurve.BezierArc(inner[i:i+4], 'arc '+str(count)))
                    count += 1
            elif len(inner) == 1: # Stroke has only one anchor
                ba_list = [BCurve.BezierArc([inner[0]], 'arc 0')]
            else:
                raise Exception("BCurve.GimpStroke.gs2bc: No anchors in stroke?")
            return BCurve.BezierCurve(
                               bezier_arcs = ba_list,
                               head_handle = head,
                               tail_handle = tail,
                               closed = self.closed,
                               curve_name = self.stroke_name
                               )
    
    class BezierCurve(object):
        """BezierCurve is a list of butting Bezier arcs with some extra data.
        Attributes:
        - curve_name:  string
        - bezier_arcs: [BezierArc];
        - head_handle: ControlPoint;
        - tail_handle: ControlPoint;
        - closed:      boolean.
        Note: "Butting" means that for any pair of successive arcs, the last
              control point of the former equals the first control point of
              the latter.
              In initialization no checks are done about this condition.
        """
        def __init__(self,
                     bezier_arcs=[],           # [BezierArc]
                     head_handle=None,         # ControlPoint
                     tail_handle=None,         # ControlPoint
                     closed=False,             # boolean
                     curve_name='BezierCurve', # string
                     ):
            self.curve_name = curve_name
            self.bezier_arcs = bezier_arcs
            if head_handle is None:
                self.head_handle = bezier_arcs[0].cp4[0]
            else:
                self.head_handle = head_handle
            if tail_handle is None:
                self.tail_handle = bezier_arcs[-1].cp4[-1]
            else:
                self.tail_handle = tail_handle
            self.closed = closed
        def __str__(self):
            s = self.curve_name + ': Bezier curve' + '; Bezier arcs:'
            count = 0
            for arc in self.bezier_arcs:
                s += '\n  arc '+ str(count)+ ': ' +str(arc)
                count += 1
            s += '\n  Head handle:    '+str(self.head_handle)
            s += '\n  Tail handle:    '+str(self.tail_handle)
            s += '\n  Closed:'    +str(self.closed)
            return s
        def bc2gs(self):
            """Conversion BezierCurve -> GimpStroke
            """
            cp_list = [self.head_handle]
            if len(self.bezier_arcs[0].cp4) > 1:
                for ba in self.bezier_arcs:
                    cp_list += ba.cp4[:3]
                cp_list.append(self.bezier_arcs[-1].cp4[-1])
            else:
                # Only one BezierArc and it has cp4 = [p0]
                # (may rise from Gimp stroke with only one anchor).
                cp_list.append(self.bezier_arcs[0].cp4[0])
            cp_list.append(self.tail_handle)
            return BCurve.GimpStroke(cp_list, self.closed, self.curve_name)
        def bc2vectors_object(self, image, name=None):
            """Conversion BezierCurve -> gimp.Vectors
            Makes vectors_object with 1 stroke.
            """
            if name is None:
                name = self.curve_name
            gs = self.bc2gs()             # BCurve.GimpStroke
            gv = BCurve.GimpVectors([gs]) # BCurve.GimpVectors
            return gv.gv2vectors_object(image, name)
    
    class BezierArc(object):
        """Data structure for one Bezier arc: arc determined by 4 control
           points.
        Attributes:
        - cp4:      [p0,p1,p2,p3] where each pi:ControlPoint, or
                    [p0] if only one control point (may rise from
                    a stroke with only one anchor);
        - arc_name: string
        """
        def __init__(self,
                     cp4,                 # cp4=[p0,p1,p2,p3] with pi:ControlPoint
                                          # or [p0]
                     arc_name='BezierArc' # string
                     ):
            self.cp4 = cp4
            self.arc_name = arc_name
        def reverse(self):
            self.cp4.reverse()
        def __str__(self):
            s = 'BezierArc, control points:'
            for cp in self.cp4:
                s += '\n  '+ str(cp)
            return s

# -----------------
# vectors_object2gv
# -----------------
# Conversion gimp.Vectors -> GimpVectors
# Args:
# - vectors_object: gimp.Vectors (Gimp vectors object)
# Returns:
# - GimpVectors
#
# Note: The inverse conversion is GimpVectors.gv2vectors_object(...).
#       See also BezierCurve.bc2vectors_object(...).
def vectors_object2gv(vectors_object):
    def xy_list2cp_list(xy_list): # xy_list = [x,y,x,y,...,]
        cp = []
        for i in range(0,len(xy_list),2):
            cp.append(BCurve.ControlPoint(xy_list[i], xy_list[i+1]))
        return cp
    stroke_list = []
    count = 0
    for stroke in vectors_object.strokes:
        xy_list = stroke.points[0]
        closed = stroke.points[1]
        cp_list = xy_list2cp_list(xy_list)
        stroke_name = vectors_object.name + ' stroke '+str(count)
        stroke_list.append(BCurve.GimpStroke(cp_list, closed, stroke_name))
        count += 1
    return BCurve.GimpVectors(stroke_list, vectors_object.name)



#==============================================================
#             classes Support, SupportedFamily etc
#==============================================================
# Classes: Support, ComplexValuedFunction, SupportedFamily, ClosedInterval,
# BezierFunction.
# The algorithm to find an approximate Bezier arc by means of one-parameter
# families of Bezier arcs.

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

class NoSupportError(BaseException):
    """When initialization of Support fails, the points causing
    the failure are intermitted in NoSupportError as
    - self.msg:   string
    - self.error: integer
    - self.points: [complex]
    """
    def __init__(self, msg, error, points):
        self.msg    = msg
        self.error  = error
        self.points = points
    def __str__(self):
        return str(self.msg)

class Support(object):
    """Data structure to store "support" for certain Bezier arcs.
    By a "support" we mean non-collinear points S0,S1,S2,S3 where it is
    assumed that S0 != S1 and S3 != S2.
    This "supports" a Bezier arc with control points p0,p1,p2,p3 if
        - p0 = S0
        - p1 is on the line through S0,S1
        - p2 is on the line through S3,S2
        - p3 = S3
    Attributes:
    - support_points = [S0,S1,S2,S3] where Si:[complex]
    - case:string: one of
      'N':  S01,S32 not parallel; (where S01 = -S0+S1, S32 = -S3+S2)
      'P':  S01,S32     parallel.
      'sloppy': the support is used just to store points S0,S1,S2,S3
    - bezier_area: float (signed area between Bezier arc and chord where
                          the support points S0,S1,S2,S3 are taken as
                          control points)
    Methods:
    - __init__
    - coefficients2bezier_arc
    - sum_dist2_to_targets   (measure: sum of squared distances from middle 
                              point of chord S0S3 to target points)
    """
    def __init__(self, support_points, sloppy=False): # If sloppy 
        def get_case(S0,S1,S2,S3):
            #ZERO = 1e-10
            ZERO = 1e-16
            S01 = -S0+S1
            S32 = -S3+S2
            S03 = -S0+S3
            # CKOE
            #if (vdot(S01,S01) < ZERO) or (vdot(S32,S32) < ZERO):
            if (vdot(S01,S01) < ZERO) or (vdot(S32,S32) < ZERO):
                m1 = "SupportedFamily: "
                m2 = "The points to determine tangents are coincident (or too close)"
                #raise Exception(m1+m2)
                raise NoSupportError(msg = m1+m2,
                                     error = 1,
                                     points = [S0,S1,S2,S3])
            if collinear([S1,S0,S3,S2]):
                m1 = "SupportedFamily: "
                m2 = "The points to determine tangents are collinear (or too close)"
                #raise Exception(m1+m2)
                raise NoSupportError(msg = m1+m2,
                                     error = 2,
                                     points = [S0,S1,S2,S3])
            S01xS32 = cross(S01,S32)
            #S03xS01  = cross(S03,S01)
            #S03xS32  = cross(S03,S32)
            if abs(S01xS32) > ZERO: # Handles not parallel
                case = 'N'
            else:                   # Handles parallel
                case = 'P'
            return case
        if len(support_points) == 4:
            S0,S1,S2,S3 = support_points
        else:
            m1 = "SupportedFamily: "
            m2 = "Bad number of support points:"+len(support_points)
            raise NoSupportError(msg = m1+m2,
                                 error = 3,
                                 points = support_points)
        if not sloppy:
            case = get_case(S0,S1,S2,S3) # May raise NoSupportError
        else: 
            case = 'sloppy'
        self.support_points = [S0,S1,S2,S3]
        self.case = case
        self.bezier_area = BezierFunction([S0,S1,S2,S3]).signed_area()
    def sum_dist2_to_targets(self, targets): # targets = [[x,y],[x,y],...]
        def distance2(x,y):
            # CKOE
            #xy = points2vector(x,y)
            #return xy[0]**2 + xy[1]**2
            xy = -x+y
            return xy.real**2 + xy.imag**2
        S0,S1,S2,S3 = self.support_points
        # CKOE
        #middle = [(S0[0]+S3[0])/2, (S0[1]+S3[1])/2] # Middle point of chord
        middle = (S0+S3)/2 # Middle point of chord
        return sum([distance2(middle, target) for target in targets])
    # ---------------------------
    # coefficients2bezier_arc
    # ---------------------------
    # Given xi,eta, compute control points p0,p1,p2,p3 of a supported Bezier
    # arc such that |-p0+p1| = xi*|-S0+S1| and |-p3+p2| = eta*|-S3+S2|.
    # Return BezierFunction.
    def coefficients2bezier_arc(self,xi,eta):
        S0,S1,S2,S3 = self.support_points
        # CKOE
        #S01 = points2vector(S0,S1)
        #S32 = points2vector(S3,S2)
        #p01 = [xi*S01[0], xi*S01[1]]
        #p32 = [eta*S32[0], eta*S32[1]]
        S01 = -S0+S1
        S32 = -S3+S2
        p01 = xi*S01
        p32 = eta*S32
        #
        p0 = S0
        p3 = S3
        # CKOE
        #p1 = [p0[0] + p01[0], p0[1] + p01[1]]
        #p2 = [p3[0] + p32[0], p3[1] + p32[1]]
        p1 = p0 + p01
        p2 = p3 + p32
        return BezierFunction([p0,p1,p2,p3])
    def __str__(self):
        s = 'Support, case '+self.case
        s += '\n  support_points: '+str(self.support_points)
        return s


class ComplexValuedFunction(object):
    """Function R -> C.
    Initialization arguments:
    - f:    R->C:         the function;
    - fdot: None or R->C: None or the derivative of f
    - use_ad_hoc_fdot: boolean: If fdot=None then
                       - if use_ad_hoc_fdot=True, insert in place of fdot
                         a simple ad-hoc substitute algorithm;
                       - if use_ad_hoc_fdot=False, leave fdot=None;
                         this is likely to cause an error somewhere?
    - interval:       [a,b]: a,b are float or None; None denotes +-infinity
                      (default is [None,None], meaning whole R);
    Attributes:
    - function:   R->C
    - derivative: None or R->C
    - interval:   ClosedInterval
    Methods:
    - __init__
    - minimize
    - closest_point
    """
    def __init__(self,
                 f,
                 fdot=None,
                 use_ad_hoc_fdot=True, # applies if fdot=None
                 interval=[None,None]
                 ):
        self.function   = f
        lo, hi = interval
        if not ((lo is None) or (hi is None)):
            if lo > hi:
                lo,hi = hi,lo
        self.interval = ClosedInterval(lo,hi)
        if fdot is not None:
            self.derivative = fdot
        elif use_ad_hoc_fdot:
            def der(x):
                delta = 0.001
                try:
                    x1 = self.interval.closest(x-delta)
                    x2 = self.interval.closest(x+delta)
                    a1,b1 = f(x1)
                    a2,b2 = f(x2)
                except Exception as e:
                    raise Exception("ComplexValuedFunction: "+str(e))
                return [(a2-a1) / (x2-x1), (b2-b1) / (x2-x1)]
            self.derivative = der
        else:
            self.derivative = None
    # --------
    # minimize
    # --------
    # Minimize key(x) where x is the parameter for the curve.
    # Args:
    # - self:            ComplexValuedFunction;
    # - key:             callable: float -> float;
    # - search_interval: [float,float];
    # - guess:           float;
    # - samples:         integer;
    # - rec_depth:       integer;
    # - sloppy: Boolean.
    # Returns:
    # - x_min:   float
    # - key_min: float
    def minimize(self,
                 key,
                 search_interval=None,
                 guess=None,
                 samples=10,
                 rec_depth=0,
                 sloppy=False
                 ):
        if search_interval is None:
            lo,hi = ComplexValuedFunction.get_search_interval(self, key, guess, sloppy)
        else:
            lo, hi = min(search_interval), max(search_interval)
        lo, hi = self.intersect_interval(lo, hi)
        accuracy = 1e-3
        MIN_INTERVAL = max(1e-4, accuracy * (hi-lo)) # ?
        MAX_REC_DEPTH = 10
        # Create list of x's
        xs = sorted(ClosedInterval(lo,hi).samples(samples, include_endpoints=True))
        xlo, xhi = xs[0], xs[-1]
        if (rec_depth > MAX_REC_DEPTH) or (xhi-xlo < MIN_INTERVAL):
            x = (xlo+xhi)/2
            key_value = key(x)
            return x, key_value
        # Gather indices where local minima seem to occur:
        min_indices = []
        prev_key = key(xs[0])
        curr_key  = key(xs[1])
        for i in range(1,len(xs)-1): # i is index of current x
            next_key = key(xs[i+1]) # next x
            if prev_key >= curr_key <= next_key: # Catches minima not at ends of interval
                min_indices.append(i)
            elif i==1 and prev_key <= curr_key:         # Start of interval
                min_indices.append(0)
            elif i==len(xs)-2 and curr_key >= next_key: # End of interval
                min_indices.append(len(xs)-1)
            prev_key, curr_key = curr_key, next_key
        # Recursion: Call the routine for subintervals containing
        # the tentative minima:
        sub_x_keys = []
        for j in min_indices:
            e = 1
            low, high = max(j-e,0), min(j+e, len(xs)-1)
            subinterval = [xs[low], xs[high]] # Subinterval for a tentative minimum
            # Recursion:
            sub_x, key_value = self.minimize(key, subinterval,
                                             rec_depth=rec_depth+1)
            sub_x_keys.append([sub_x, key_value])
        # Choose from the list the one with smallest key_value:
        x_min, key_min = min(sub_x_keys, key=lambda d:d[1])
        return x_min, key_min
    # -------------
    # closest_point
    # -------------
    # Find locally closest point to v on the ComplexValuedFunction arc F(t)
    # (self).
    #
    # Args:
    # - self:     ComplexValuedFunction;
    # - v:        point (complex);
    # - search_interval: interval [start,end] for the parameter t;
    # - guess:    float in interval, an initial guess for the parameter value
    #             of the closest point (default=0.5, which is also used if
    #             the guess is not in the interval);
    # - samples: integer;
    # - sloppy: Boolean.
    # Returns:
    # - t0: float:   the parameter of a locally closest point;
    # - b0: complex: the point on the curve with parameter value t0;
    # - d0: float:   the squared distance between v and b0.
    #
    # Notes:
    # 1. If the distance has several minima, the routine may not find the
    #    globally best but only a locally best in the vicinity of the guess.
    #    Also, the formula is derived by means of linear approximation
    #    and it assumes that the changes in the parameter value are small.
    #    Thus, it is crucial that the initial guess is good!
    # 2. Possibly an end point of the interval is returned.
    # 3. The search interval is intersected by the defining interval of self.
    def closest_point(self,
                      v,
                      search_interval=None,
                      guess=None,
                      samples=10,
                      sloppy=False
                      ):
        def distance2_to_v(t):
            w = self.function(t)
            return ((v-w)*(v-w).conjugate()).real
        if search_interval is None:
            lo,hi = ComplexValuedFunction.get_search_interval(self,
                                        distance2_to_v,
                                        guess,
                                        sloppy)
        else:
            lo, hi = search_interval
        lo, hi = self.intersect_interval(lo, hi)
        #accuracy=1e-8
        accuracy=1e-3
        #MAX_ROUNDS = 20     # Used to avoid possibility of infinite loop.
        MAX_ROUNDS = 10     # Used to avoid possibility of infinite loop.
        MAX_T_DIFF = accuracy * (hi-lo) # To stop work when nothing
        #                                 much seems to be happening.
        if guess is None:
            guess = (lo + hi)/2.
        if not (lo <= guess <= hi):
            guess = (lo + hi)/2.
        f, fdot = self.function, self.derivative
        if fdot is None:
            raise Exception("ComplexValuedFunction.closest_point: fdot is None")
        guesses = ClosedInterval(lo,hi).samples(samples)
        if not (guess is None):
           guesses.append(self.interval.closest(guess))
        td = [[t, distance2_to_v(t)] for t in guesses]
        smallest = min(td, key=lambda x:x[1])
        t0, d2 = smallest # Use this t0 as guess.
        b0 = f(t0)
        d0 = distance2_to_v(t0)
        step = 1.
        for rnd in range(MAX_ROUNDS):
            # From t0 compute new tentative guess for closest point
            # by linear approximation.
            b0v = b0-v
            bd = fdot(t0)
            numer = vdot(b0v,bd)            # (B(t0)-v).B'(t0)
            denom1 = vdot(bd,bd)            # B'(t0).B'(t0)
            try:
                t1 = t0 - step * numer/denom1
            except ZeroDivisionError:
                # t0 is a cusp, therefore an end point: try to get some reasonable t1
                t1 = t0 + step * (hi-lo)
            t1 = max(lo, min(hi, t1))
            t12 = [t1]
            t_diff = max([t0]+t12) - min([t0]+t12)
            tbd = []
            for tx in t12:
                bx = f(tx)
                dx = distance2_to_v(tx)
                tbd.append([tx,bx,dx])
            tbd_smallest = min(tbd, key=lambda x:x[2]) # smallest wrt d
            if d0 < tbd_smallest[2]: # d0 smallest
                step /= 2
            else:
                t0,b0,d0 = tbd_smallest        
            if t_diff < MAX_T_DIFF:
                break
        return t0,b0,d0
    # -------------------
    # get_search_interval
    # -------------------
    # For a vector valued function self, find a suitable interval for searching for
    # minimal value of key.
    # Args:
    # - pf:    ComplexValuedFunction;
    # - key:   callable: float -> float;
    # - guess: float;
    # - sloppy: Boolean.
    # Returns:
    # - [lo value, hi value]: [float,float]
    def get_search_interval(self, key, guess=None, sloppy=False):
        pf = self
        if sloppy:
            pf_lo, pf_hi = [self.interval.lo, self.interval.hi]
            if not (None in (pf_lo, pf_hi)):
                return [pf_lo, pf_hi]
        MAX_ROUNDS = 100
        point = self.interval.typical_point
        step  = self.interval.typical_step
        #lo, hi = self.interval.lo, self.interval.hi
        if guess is None:
            cur_lo = point
        else:
            cur_lo = min(point, guess)
        cur_key = key(cur_lo)
        for r in range(MAX_ROUNDS):
            next_lo = self.interval.closest(cur_lo - step)
            next_key = key(next_lo)
            found_lo = next_lo
            found_key = next_key
            if next_key > cur_key:
                break
            else:
                cur_lo = next_lo
                cur_key = next_key
        else: # No break occured
            if sloppy:
                pass
                #found_lo = next_lo
            else:
                raise Exception("get_search_interval: No lower end found")
        if guess is None:
            cur_hi = point
        else:
            cur_hi = max(point, guess)
        cur_key = key(cur_hi)
        for r in range(MAX_ROUNDS):
            next_hi = self.interval.closest(cur_hi + step)
            next_key = key(next_hi)
            found_hi = next_hi
            found_key = next_key
            if next_key > cur_key:
                break
            else:
                cur_hi = next_hi
                cur_key = next_key
        else: # No break occured
            if sloppy:
                pass
            else:
                raise Exception("get_search_interval: No upper end found")
        return [found_lo, found_hi]
    # ------------------
    # intersect_interval
    # ------------------
    # Restrict interval [lo,hi] by intersecting with defining interval of self.
    # Args:
    # - lo: float
    # - hi: float
    # Returns:
    # - float
    # - float
    def intersect_interval(self, lo, hi):
        new_closed_interval = ClosedInterval(lo,hi).intersect(self.interval)
        return new_closed_interval.lo, new_closed_interval.hi


class SupportedFamily(Support, ComplexValuedFunction):
    """One-parameter family of Bezier arcs in a given support.
    Attributes:
    - all attributes of Support
    - all attributes of ComplexValuedFunction
    - type: string: one of
            - 'fixed point N';
            - 'fixed point P';
            (- 'fixed area'  does not appear in current version)
    - data: float
            - the fixed point ([float,float]) for type 'fixed point N';
            - the parameter for the fixed point (float) for type
              'fixed point P';
            (- the fixed area_ratio (float) for type 'fixed area')
    Methods:
    - __init__
    - parameter2bezier_arc
    """
    # Internally, the family is a vector valued function (plus some data).
    # From any parameter value x, a Bezier arc is produced by
    # parameter2bezier_arc.
    def __init__(self,
                 support_points,
                 f,
                 fdot=None,
                 use_ad_hoc_fdot=True,
                 interval=[None,None],
                 type=None,
                 data=None):
        Support.__init__(self, support_points)
        try:
            ComplexValuedFunction.__init__(self,
                                           f=f,
                                           fdot=fdot,
                                           use_ad_hoc_fdot=True,
                                           interval=interval
                                           )
        except Exception as e:
            raise Exception("SupportedFamily: "+str(e))
        self.type = type
        self.data = data
    # --------------------
    # parameter2bezier_arc
    # --------------------
    # For a SupportedFamily (self) and a parameter value x, construct a BezierFunction
    # (actually a Bezier arc, four control points):
    #     x -> xi,eta -> BezierFunction
    def parameter2bezier_arc(self,x):
        xi,eta = self.function(x)
        bezier_arc = self.coefficients2bezier_arc(xi,eta)
        return bezier_arc


class ClosedInterval(object):
    """Closed real number interval. Four cases: finite, infinite at one end,
    infinite at both ends.
    Initialization arguments:
    - a: float or None (None denotes -infinity);
    - b: float or None (None denotes +infinity).
    Attributes:
    - type: string: one of  'lo,hi',  '-inf,hi',  'lo,inf',  '-inf,inf'
    - lo:   float or None (None denotes -infinity);
    - hi:   float or None (None denotes +infinity);
    - typical_point: float: a good point to start coarse searching;
    - typical_step:  float: a good starting step in a coarse searching.
    Methods:
    - __init__
    - closest
    - samples
    - intersect
    """
    def __init__(self, a=None, b=None, typical_point=None, typical_step=None):
        STEP = 1.
        NSTEPS = 10
        self.lo, self.hi = a, b
        if not (a is None):
            if not (b is None):
                if a > b:
                    self.lo, self.hi = b, a
                self.type = 'lo,hi'
            else: # b = None meaning infinity
                self.type = 'lo,inf'
        else: # a = None meaning -infinity
            if not (b is None):
                self.type = '-inf,hi'
            else: # b = None meaning infinity
                self.type = '-inf,inf'
        if not(typical_point is None):
            self.typical_point = typical_point
        else:
            if self.type == 'lo,hi':
                self.typical_point = (self.lo + self.hi)/2
            elif self.type == 'lo,inf':
                self.typical_point = self.lo + NSTEPS*STEP
            elif self.type == '-inf,hi':
                self.typical_point = self.hi - NSTEPS*STEP
            elif self.type == '-inf,inf':
                self.typical_point = 0.
        if not(typical_step is None):
            self.typical_step = max(typical_step,0.)
        else:
            if self.type == 'lo,hi':
                self.typical_step = (self.hi - self.lo)/NSTEPS
            else:
                self.typical_step = STEP
    def closest(self, x): # The closest point to x in interval
        if self.type == 'lo,hi':
            return max(self.lo, min(self.hi, x))
        elif self.type == '-inf,hi':
            return min(self.hi, x)
        elif self.type == 'lo,inf':
            return max(self.lo, x)
        else:
            return x
    def samples(self, N=8, include_endpoints=False): # Some points in the interval
        N = max(0,N)
        if self.type == 'lo,hi':
            a,b = self.lo,self.hi
        elif self.type == '-inf,hi':
            a,b = self.hi - N, self.hi
        elif self.type == 'lo,inf':
            a,b = self.lo, self.lo + N
        else:
            a,b = -N/2, N/2
        if N <= 0:
            s = []
        elif N == 1:
            s = [(a+b)/2]
        else:
            n = N+1
            s = [((n-i)*a + i*b)/n for i in range(1,n)] # (a,b) open
        if include_endpoints:
            if self.type == 'lo,hi':
                s = [a]+s+[b]
            elif self.type == '-inf,hi':
                s = s+[b]
            elif self.type == 'lo,inf':
                s = [a]+s
        return s
    def intersect(self,other):
        lo1,hi1 = self.lo, self.hi
        lo2,hi2 = other.lo,other.hi
        if   lo1 is None: lo = lo2
        elif lo2 is None: lo = lo1
        else: lo = max(lo1,lo2)
        if   hi1 is None: hi = hi2
        elif hi2 is None: hi = hi1
        else: hi = min(hi1,hi2)
        typical_step = min(self.typical_step, other.typical_step)
        return ClosedInterval(lo,hi,
                              typical_point=None,
                              typical_step=typical_step)

class BezierFunction(ComplexValuedFunction):
    """Bezier arc as ComplexValuedFunction,
    Attributes:
    - all attributes of ComplexValuedFunction
    - control_points
    Methods:
    - __init__
    - signed_area (between arc and chord)
    """
    def __init__(self, control_points):
        ComplexValuedFunction.__init__(self,
                f       =partial(bezier_rv, control_points=control_points),
                fdot    =partial(bezier_dot_rv, control_points=control_points),
                use_ad_hoc_fdot=False,
                interval=[0.,1.])
        self.control_points = control_points
    # -----------
    # signed_area
    # -----------
    # Signed area between arc and chord.
    # Sign of area is + if the loop is run counterclockwise *in Gimp's window*
    # when
    # - first the arc is run from start to end,
    # - followed by running the chord from end to start.
    # Notes:
    # 1. Since the y coordinate in Gimp's window runs downwards, the sign
    #    in the formula is opposite to what it should be in general!
    # 2. Currently this feature is used nowhere (but it is nice to keep
    #    the formula around just in case).
    def signed_area(self):
        p0,p1,p2,p3 = self.control_points # complex
        p31 = -p3+p1
        p02 = -p0+p2
        p03 = -p0+p3
        return (3./20) * (cross(p31,p02) + cross(p31,p03) + cross(p02,p03))

###### The usual Bezier curve and derivatives from control points  #######
## Args:
## - t:float
## - control_points = [P0,P1,P2,P3]:[[float,float]]
## Returns:
## - [float,float]
#def bezier(t, control_points):
#    P0,P1,P2,P3 = control_points
#    if t > 0.5:
#        u = (1-t)/t
#        t3 = t**3
#        return [t3*(u*(u*(u*P0[0]+3*P1[0])+3*P2[0])+P3[0]),
#                t3*(u*(u*(u*P0[1]+3*P1[1])+3*P2[1])+P3[1])]
#    else:
#        u = t/(1-t)
#        t3 = (1-t)**3
#        return [t3*(u*(u*(u*P3[0]+3*P2[0])+3*P1[0])+P0[0]),
#                t3*(u*(u*(u*P3[1]+3*P2[1])+3*P1[1])+P0[1])]

# Version which returns RowVector = complex
# Args:
# - t:float
# - control_points = [P0,P1,P2,P3]:[complex]
# Returns:
# - RowVector = complex
def bezier_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    if t > 0.5:
        u = (1-t)/t
        t3 = t**3
        return t3*(u*(u*(u*P0+3*P1)+3*P2)+P3) # complex
    else:
        u = t/(1-t)
        t3 = (1-t)**3
        return t3*(u*(u*(u*P3+3*P2)+3*P1)+P0) # complex

# Derivative: Returns complex.
def bezier_dot_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    P01 = -P0+P1
    P12 = -P1+P2
    P23 = -P2+P3
    if t > 0.5:
        u = (1-t)/t
        t2 = t**2
        return 3*t2*(u*(u*P01+2*P12)+P23) # complex
    else:
        u = t/(1-t)
        t2 = (1-t)**2
        return 3*t2*(u*(u*P23+2*P12)+P01) # complex

## Used nowhere! Returns [float,float] (rather than complex)!
#def bezier_dot_dot(t, control_points): # Could be made faster as above.
#    p0,p1,p2,p3 = control_points
#    p01 = [-p0[0]+p1[0], -p0[1]+p1[1]]
#    p12 = [-p1[0]+p2[0], -p1[1]+p2[1]]
#    p23 = [-p2[0]+p3[0], -p2[1]+p3[1]]
#    return [-6*(1-t)*p01[0] + 6*(1-2*t)*p12[0] + 6*t*p23[0],
#            -6*(1-t)*p01[1] + 6*(1-2*t)*p12[1] + 6*t*p23[1]]


################################################################
#             Central functions f,g,h in the algorithm
################################################################

def f_function(x): return 3.*x*x - 2.*x*x*x # bernstein2 + bernstein3

def g_function(x): return 3.*(1-x)*(1-x)*x  # bernstein1

def h_function(x): return 3.*(1-x)*x*x      #  bernstein2

# -------
# solve_f
# -------
# Let 0<a<1. Solve f_function(t)=a.
# Return roots in ascending order.
# Return None if not 0<a<1.
# Note: Of the three roots the important is t2 one since it belongs to (0,1).
def solve_f(a):
    from math import cos, acos, pi
    try:
        theta = acos(-2*a+1)
    except ValueError:
        raise Exception("solve_f: Got invalid argument a =",a)
    t0 = .5 + cos(theta/3)
    t1 = .5 + cos(theta/3 + 2*pi/3)
    t2 = .5 + cos(theta/3 + 4*pi/3)
    return t1,t2,t0


################################################################
#                 Plane vectors (=complex numbers)
################################################################

# v dot (w perpendicular)
def cross(v,w):
    return (v*w.conjugate()).imag

def vdot(v,w):
    return v.real*w.real + v.imag*w.imag

def collinear(points): # points: [complex] (The plane R2 is the complex plane.)
    ZERO = 1e-14
    n = len(points)
    if n < 3:
        return True
    for i in range(n):
        for j in range(i+1,n):
            for k in range(j+1,n):
                a,b,c = points[i], points[j], points[k]
                ab = -a+b
                ac = -a+c
                bc = -b+c
                if abs(cross(ab,ac)) > ZERO:
                    return False
                if abs(cross(ab,bc)) > ZERO:
                    return False
                if abs(cross(ac,bc)) > ZERO:
                    return False
    return True


################################################################
#                    Squared distances
################################################################

# --------------------
# sum_dist2_arc_points
# --------------------
# Given a parametric arc and a list of points,
# compute the sum of squared distances of the points from the arc.
# Args:
# - pf:     VectorValuedFunction
# - points: [X1,X2,...,Xn] with all Xi:complex
# Returns
# - float
def sum_dist2_arc_points(pf, points, sloppy=False):
    sum_d2 = 0.
    for point in points:
        t, b, d2 = pf.closest_point(point, sloppy=sloppy)
        #print("sum_dist2_arc_points: d2 ="+str(d2))
        sum_d2 += d2
    return sum_d2

# -----------------------
# sum_dist2_family_points
# -----------------------
# Given a family of Bezier arcs and one parameter value x for the family
# (giving rise to one arc in the family),
# and a list of points,
# compute the sum of squared distances of the points from the arc.
# Args:
# - family:  SupportedFamily
# - x:       float
# - points:  [X1,X2,...,Xn] with all Xi:complex
# Returns
# - float
def sum_dist2_family_points(family, x, points, sloppy=False):
    bezier_arc = family.parameter2bezier_arc(x)
    #print("sum_dist2_family_points: len(points) ="+str(len(points)))
    return sum_dist2_arc_points(bezier_arc, points, sloppy=sloppy)


################################################################
#                       Make families
################################################################
# Make one-parameter families of Bezier arcs with common support S
# and some chosen other condition.
# (In this version the only appearing condition is that the arcs pass
# through a fixed point.)

# ------------------
# family_fixed_point
# ------------------
# Given support S:Support and a point X:[float,float],
# create the family of Bezier arcs having the support S and passing
# through X.
# If respect_directions is True, include in the family only those Bezier
# arcs for which the handles point in the same directins as the handles
# of the support.
# Args:
# - support:            Support
# - point:              complex
# - respect_directions: boolean
# Returns: SupportedFamily
def family_fixed_point(support, point, respect_directions=False):
    def check_condition(xi, eta, t, X): # Condition B(t) = X
        ba = support.coefficients2bezier_arc(xi,eta)
        cp = ba.control_points
        Bt = bezier(t, cp)
        diff = -Bt+X
    PADDING = 1e-8 # To avoid interval ends 0 and 1
    ZERO = 1e-10
    S0,S1,S2,S3 = support.support_points
    X = point
    case = support.case
    S0X = -S0+X
    S32 = -S3+S2
    S01 = -S0+S1
    S03 = -S0+S3
    S0XxS01 = cross(S0X,S01)
    S0XxS03 = cross(S0X,S03)
    S03xS32 = cross(S03,S32)
    S03xS01 = cross(S03,S01)
    S01xS32 = cross(S01,S32)
    S01xS03 = -S03xS01
    S32xS01 = -S01xS32
    S32xS03 = -S03xS32
    if case == 'N':
        def F(t):
            ft, gt, ht = f_function(t), g_function(t), h_function(t)
            diff = S0X - ft*S03
            try:
                xi  = cross(diff, S32) / (gt * S01xS32)
                eta = cross(diff, S01) / (ht * S32xS01)
            except ZeroDivisionError:
                raise Exception("parametric function: t = 0 or 1.")
            #check_condition(xi, eta, t, X)
            return [xi,eta]
        def Fdot(t):
            xi, eta = F(t)
            xidot  = -((1-3*t)*xi/t      + 2*S03xS32/S01xS32) / (1-t)
            etadot = -((2-3*t)*eta/(1-t) + 2*S03xS01/S32xS01) / t
            return [xidot, etadot]
        if respect_directions:
            interval, None_case = first_quadrant_subinterval(support, point)
            if interval is None:
                raise Exception("family_fixed_point: Got subinterval=None:"+None_case)
            if interval[0] < ZERO:
                interval[0] = PADDING
            if interval[1] > 1-ZERO:
                interval[1] = 1-PADDING
        else:
            interval=[PADDING,1-PADDING]
        type = 'fixed point N'
        data = point
    else: # Case P
        #print("P:")
        #print("  support:")
        #for cpp in support.support_points: print("    "+str(cpp))
        #print("  point: "+str(point))
        #print("  respect_directions: "+str(respect_directions))
        if not (0 + ZERO <= S0XxS01 / S03xS01 <= 1 - ZERO):
            raise Exception("family_fixed_point: The target point in illegal area: not between the tangents lines")
        try:
            t0 = solve_f(S0XxS01 / S03xS01)[1]
        except ZeroDivisionError:
            raise Exception("parametric function: Support points collinear.")
        ft, gt, ht = f_function(t0), g_function(t0), h_function(t0)
        a = gt * S01xS03
        b = ht * S32xS03
        c = S0XxS03
        try:
            invb = 1./b
        except ZeroDivisionError:
            raise Exception("parametric function: Support points collinear.")
        def F(xi):
            eta = (c-a*xi) * invb
            #check_condition(xi, eta, t0, X)
            return [xi,eta]
        def Fdot(xi):
            return [1., -a*invb]
        if respect_directions:
            interval, None_case = first_quadrant_subinterval(support, point)
            if interval is None:
                raise Exception("family_fixed_point: Got subinterval=None:"+None_case)
        else:
            interval=[None,None]
        #print("=> interval: "+str(interval))
        type = 'fixed point P'
        data = t0
    try:
        supported_family = SupportedFamily(support_points=[S0,S1,S2,S3],
                               f=F,
                               fdot=Fdot,
                               use_ad_hoc_fdot=False,
                               interval=interval,
                               type=type,
                               data=data)
    except Exception as e:
        raise Exception('family_fixed_point: ' +str(e))
    return supported_family

# --------------------------
# first_quadrant_subinterval
# --------------------------
# In the algorithm it may be required to find only those solutions
# (Bezier arcs) whose tangents obey the directions S01 and S32 in
# the support.
# This routine finds the interval [a,b] where the parameter x needs
# to be restricted to this end. The family in question is the one produced
# by family_fixed_point, and x is its parameter. Let F be the vector-valued
# function corresponding to the family. The condition to be enforced is that
# if x is in [a,b] and F(x) = (xi,eta), then xi >= 0 and eta >= 0.
# Args:
# - support:Support
# - point:  complex
# Returns:
# - [a,b] or None (may be b=None, meaning infinity)
# - None_case: string (only for debugging)
# Note: points 0,1 included: caller must handle!
def first_quadrant_subinterval(support, point):
    ZERO = 1e-10
    f = f_function
    S0,S1,S2,S3 = support.support_points
    X = point
    case = support.case
    S0X = -S0+X
    S32 = -S3+S2
    S01 = -S0+S1
    S03 = -S0+S3
    S0XxS01 = cross(S0X,S01)
    S0XxS03 = cross(S0X,S03)
    S03xS32 = cross(S03,S32)
    S03xS01 = cross(S03,S01)
    S01xS32 = cross(S01,S32)
    S01xS03 = -S03xS01
    S32xS01 = -S01xS32
    S32xS03 = -S03xS32
    S0XxS32 = cross(S0X,S32)
    None_case = None
    if case == 'N':
        a = S0XxS32 / S01xS32
        b = S03xS32 / S01xS32
        c = S0XxS01 / S32xS01
        d = S03xS01 / S32xS01
        cd_ab = []
        try:
            if 0 < c/d < 1:
                cd_ab.append(c/d)
        except ZeroDivisionError:
            pass
        try:
            if 0 < a/b < 1:
                cd_ab.append(a/b)
        except ZeroDivisionError:
            pass
        tx_ty = [solve_f(q)[1] for q in cd_ab]
        t1_t2 = sorted(tx_ty)
        if len(t1_t2) == 2:
            if -ZERO < t1_t2[0] - t1_t2[1] < ZERO:
                t1_t2 = [t1_t2[0]]
        if len(t1_t2) == 2:
            t1,t2 = t1_t2
            if a - b*f((t1+t2)/2) > 0 and c - d*f((t1+t2)/2) > 0:
                subinterval = [t1,t2]
            elif a - b*f(t1/2) > 0 and c - d*f(t1/2) > 0:
                subinterval = [0.,t1]
            elif a - b*f((t2+1)/2) > 0 and c - d*f((t2+1)/2) > 0:
                subinterval = [t2,1.]
            else:
                subinterval = None
                None_case = '2'
        elif len(t1_t2) == 1:
            t1 = t1_t2[0]
            if a - b*f(t1/2) > 0 and c - d*f(t1/2) > 0:
                subinterval = [0.,t1]
            elif a - b*f((t1+1)/2) > 0 and c - d*f((t1+1)/2) > 0:
                subinterval = [t1,1.]
            else:
                subinterval = None
                None_case = '1'
        elif len(t1_t2) == 0:
            if a - b*f(1/2) > 0 and c - d*f(1/2) > 0:
                subinterval = [0.,1.]
            else:
                subinterval = None
                None_case = '0'
    else: # Case 'P'
        try:
            t0 = solve_f(S0XxS01 / S03xS01)[1]
        except ZeroDivisionError:
            raise Exception("first_quadrant_subinterval: Support points collinear.")
        ft, gt, ht = f_function(t0), g_function(t0), h_function(t0)
        a = gt * S01xS03
        b = ht * S32xS03
        c = S0XxS03
        if a*b > 0 and c*a > 0:
            subinterval = [0, c/a]
        elif a*b < 0:
            subinterval = [max(0, c/a), None]
        else:
            subinterval = None
            None_case = 'P'
    return subinterval, None_case


################################################################
#                   Create special Bezier arcs
################################################################

# ------------
# check_target
# ------------
# Check that target is far enough from S0 and S3
def check_target(support,trg):
    MIN_DIST2 = 1e-8
    S0 = support.support_points[0]
    S3 = support.support_points[3]
    dist0 = (trg.real - S0.real)**2 + (trg.imag - S0.imag)**2
    dist3 = (trg.real - S3.real)**2 + (trg.imag - S3.imag)**2
    return (dist0 > MIN_DIST2) and (dist3 > MIN_DIST2)

# -----------------------------
# arc_fixed_point_approx_points
# -----------------------------
# Given a support S and a list of points X0,...,Xn
# find a Bezier arc with support S, passing exactly through X0,
# and as closely to X1,...,Xn as possible.
# Args:
# - support: Support;
# - target0: complex;
# - other_targets: [complex];
# - respect_directions: boolean: if True, search among Bezier arcs whose
#                       handle directions obey those of support;
# - sloppy: boolean: if True, include not-so-exact results in calculations.
# Returns:
# - ba_approx_cp (list of 4 control points: [p0,p1,p2,p3]);
# - abs_error: float (an error measure);
# - rel_error: float (another error measure).
def arc_fixed_point_approx_points(support,
                                  target0,
                                  other_targets,
                                  respect_directions=False,
                                  sloppy=False
                                  ):
    from math import sqrt
    # Discard targets too close to the corner(s) of the support
    # and that sufficiently many remain:
    if not check_target(support,target0):
        message = 'arc_fixed_point_approx_points: first target too close to corners'
        raise Exception(message)
    other_targets = [trg for trg in other_targets if check_target(support,trg)]
    if len(other_targets) < 1:
        message = 'arc_fixed_point_approx_points: not enough targets not too close to corners'
        raise Exception(message)
    #
    X0 = target0
    Xpoints = other_targets
    try:
        F0 = family_fixed_point(support, X0, respect_directions)
    except Exception as e:
        raise Exception("arc_fixed_point_approx_points: "+str(e))
    def pf_key(x):
        return sum_dist2_family_points(F0, x, Xpoints, sloppy)
    xmin, keymin = F0.minimize(key=pf_key, search_interval=None,
                                sloppy=sloppy)
    xi,eta = F0.function(xmin)
    ba_approx = support.coefficients2bezier_arc(xi,eta)
    abs_error = sum_dist2_arc_points(ba_approx, other_targets,sloppy)
    yard_stick = support.sum_dist2_to_targets(other_targets)
    rel_error = sqrt(abs_error / yard_stick)
    return ba_approx.control_points, abs_error, rel_error

# -----------------
# arc_approx_points
# -----------------
# Given a support S and a list of target points X0,X1,..., find a Bezier arc
# with support S and passing close(?) to the points.
# Args:
# - support: Support;
# - targets:  [complex];
# - respect_directions: boolean: if True, search among Bezier arcs whose
#                       handle directions obey those of support;
# - sloppy: boolean: if True, include not-so-exact results in calculations.
# - error_targets: None or [complex]: if not None, compute error
#                  of the resulting arc from the targets in this list,
#                  otherwise use the list targets
# Returns:
# - list of 4 control points: [p0,p1,p2,p3];
# - abs_error: float (an error measure);
# - rel_error: float (another error measure).
# Note: This is rather ad-hoc, especially the usage of weighted average.
def arc_approx_points(support,
                      targets,
                      respect_directions=False,
                      sloppy=False,
                      error_targets=None
                      ):
    # Given list [[cp,error],...], where cp = [p0,p1,p2,p3] (control points)
    # compute weighted average of the control points,
    # using as weights the inverses of the errors mapped with some function
    # tweak: float -> float
    def weighted_average_cp(cp_err_list, tweak):
        ave_p0 = sum([cp[0]/tweak(err) for cp,err in cp_err_list])
        ave_p1 = sum([cp[1]/tweak(err) for cp,err in cp_err_list])
        ave_p2 = sum([cp[2]/tweak(err) for cp,err in cp_err_list])
        ave_p3 = sum([cp[3]/tweak(err) for cp,err in cp_err_list])
        m = sum([1./tweak(err) for cp,err in cp_err_list])
        ave_p0 = ave_p0 / m
        ave_p1 = ave_p1 / m
        ave_p2 = ave_p2 / m
        ave_p3 = ave_p3 / m
        return [ave_p0, ave_p1, ave_p2, ave_p3]
    # Discard targets too close to the corner(s) of the support
    # and that sufficiently many remain:
    targets = [trg for trg in targets if check_target(support,trg)]
    if len(targets) < 1:
        message = 'arc_fixed_point_approx_points: not enough targets not too close to corners'
        raise Exception(message)
    # Compute a list of tentative approximate Bezier arcs (control points):
    cp_error_list = []
    for i in range(len(targets)):
        X0 = targets[i]
        rest = [targets[j] for j in range(len(targets)) if j != i]
        try:
            # Bezier arc exactly through X0 and trying
            # to run close to the rest of the targets:
            approx_result = arc_fixed_point_approx_points(support,
                                                      X0,
                                                      rest,
                                                      respect_directions,
                                                      sloppy=sloppy)
        except Exception as e:
            raise Exception("arc_approx_points: "+str(e))
        approx_cp, abs_error, rel_error = approx_result
        # Save control points and abs_error in list:
        cp_error_list.append([approx_cp, abs_error])
    # Sort with respect to abs_error:
    cp_error_list = sorted(cp_error_list, key=lambda x:x[1])
    # Compute weighted averages from the starting sections of the the list:
    ave_cp_error_list = []
    for i in range(1, 1+len(cp_error_list)):
        ave_cp = weighted_average_cp(cp_error_list[:i], tweak=sqrt)
        #ave_cp = weighted_average_cp(cp_error_list[:i], tweak=lambda x:x)
        ave_error = sum_dist2_arc_points(BezierFunction(ave_cp), targets, sloppy=True)
        ave_cp_error_list.append([ave_cp, ave_error])
    # Find the best; this one will be returned:
    best_cp, best_error = min(ave_cp_error_list, key=lambda x:x[1])
    # Compute anew the absolute and relative errors of the best:
    best_bf = BezierFunction(best_cp)
    if error_targets is None:
        error_targets = targets
    abs_error = sum_dist2_arc_points(best_bf, error_targets, sloppy=True)
    yard_stick = support.sum_dist2_to_targets(targets) # For relative error
    rel_error = sqrt(abs_error / yard_stick)
    return best_cp, abs_error, rel_error



#==============================================================
#             Common part of the plugins  
#==============================================================
# The common part in path transformation plugins

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

import copy

#shaping_output = dict() # Global variable

#---------------------------------------------------------
#                        Classes
#---------------------------------------------------------

class InfinityError(Exception):
    """To handle errors (ZeroDivisionErrors) occurring in some
    plane transformations
    (hit the pole in Moebius map, or
    hit the pre-image of the line at infinity in projective map).
    """
    def __init__(self, text):
        self.text = text
    def __str__(self):
        return str(self.text)

class ExitError(Exception):
    def __init__(x):
        return x
#---------------------------------------------------------
#             Make similarity transformations UNTESTED
#---------------------------------------------------------

# r: RowVector (=complex) (translation vector)
def make_identity_map():
    def pm(v): # v: RowVector = complex
        return v
    def jac(v):
        return Matrix2x2(1,0,0,1)
    return PlaneTransformation('affine', pm, jac)

# r: RowVector (=complex) (translation vector)
def make_translation(r):
    def pm(v): # v: RowVector = complex
        return r+v
    def jac(v):
        return Matrix2x2(1,0,0,1)
    return PlaneTransformation('affine', pm, jac)

# z0,z1:complex
def make_reflection(z0,z1): # UNTESTED
    try:
        w = (z1-z0) / (z1-z0).conjugate()
        u,v = w.real, w.imag
    except ZeroDivisionError:
        raise Exception("make_reflection: equal arguments")
    def pm(z): # z:complex
        return z0 + w*(z - z0).conjugate()
    def jac(z): # z:complex
        return Matrix2x2(u,v,v,-u) ####### Need Gimp trick?
    return PlaneTransformation('affine', pm, jac)

# center: complex
# angle: float
def make_rotation(center, angle): # UNTESTED
    from math import sin,cos
    c,s = cos(angle),sin(angle)
    w = complex(c,s)
    def pm(z): # z:complex
        return center + w*(z-center)
    def jac(v):
        return Matrix2x2(c,-s,s,c) ####### Need Gimp trick?
    return PlaneTransformation('affine', pm, jac)

# center: complex
# scale: float
def make_homothety(center, scale): # UNTESTED
    def pm(z): # z:complex
        return center + scale*(z-center)
    def jac(v):
        return Matrix2x2(scale,0,0,scale)
    return PlaneTransformation('affine', pm, jac)

class MakeSimilitudeError(Exception):
    def __init__(self, source_error=False, target_error=False):
        self.source_error = source_error
        self.target_error = target_error

# ----------------------
# make_direct_similitude
# ----------------------
# Direct similitude sending z0->w0, z1->w1
# Args:
# - z0,z1,w0,w1:complex
# Returns:
# - PlaneTransformation
def make_direct_similitude(z0,z1,w0,w1):
    try:
        a = (w0-w1) / (z0-z1)
    except ZeroDivisionError:
        raise MakeSimilitudeError(source_error=True)
    try:
        ia = 1./a
    except ZeroDivisionError:
        raise MakeSimilitudeError(target_error=True)
    a1,a2 = a.real, a.imag
    b = w0 - a*z0
    def pm(z): # z:complex
        return a*z+b
    def jac(z): # z:complex
        return Matrix2x2(a1,-a2,a2,a1)
    return PlaneTransformation('affine', pm, jac)


# -------------
# make_dilation
# -------------
# Dilation sending z0->w0 and having the given scale factor.
# Args:
# - z0, w0:complex
# scale_factor: float
# Returns:
# - PlaneTransformation
def make_dilation(z0, w0, scale_factor):
    def pm(z): # z:complex
        return scale_factor*(z-z0) + w0
    def jac(z): # z:complex
        return Matrix2x2(scale_factor,0,0,scale_factor)
    return PlaneTransformation('dilation', pm, jac)

#---------------------------------------------------------
#               Routines for PlaneTransformations
#---------------------------------------------------------

# ----------------
# combine_AB_pt_PQ
# ----------------
# Given pt:PlaneTransformation mapping with a "base" (0,0),(1,0) (in somehow
# connected with the definition of pt)
# and four points A,B,P,Q, combine pt from the left and from the right with
# a direct similitude so that the combined map will send
# - first A->(0,0) and B->(0,1),
# - then apply pt,
# - then finally send the images of A,B to P,Q.
# With tweak_a and tweak_b the effect can be changed.
# Args:
# - pt:      PlaneTransformation
# - base:    [ControlPoint,ControlPoint] = [complex,complex] (A,B)
# - target:  [ControlPoint,ControlPoint] = [complex,complex] (P,Q)
# - tweak_a: float
# - tweak_b: float
# - type:    None or string
# Returns:
# - PlaneTransformation
def combine_AB_pt_PQ(pt,
                     base,
                     target,
                     tweak_a=0.,
                     tweak_b=1.,
                     type=None
                     ):
    A,B = base   # RowVector (=complex)
    P,Q = target # RowVector (=complex)
    #
    rva = RowVector(tweak_a,0)
    rvb = RowVector(tweak_b,0)
    try:
        hABab = make_direct_similitude(A,B,rva,rvb)
    except MakeSimilitudeError as e:
        if e.source_error:
            raise Exception("Coincident end points in Base")
        elif e.target_rel_error: # tweak_a = tweak_b
            raise Exception("Coincident tweak A and tweak B")
        else:
            raise Exception("combine_AB_pt_PQ: 1: Should never happen!")
    hABab_map = hABab.point_map
    #
    raw_point_map = pt.point_map
    raw_jacobian = pt.jacobian
    X = raw_point_map(hABab_map(A))
    Y = raw_point_map(hABab_map(B))
    try:
        hXYCD = make_direct_similitude(X,Y,P,Q)
    except MakeSimilitudeError as e:
        if e.source_error:
            raise Exception("combine_AB_pt_PQ: Hit a loop. Cannot handle this!")
        elif e.target_rel_error:
            raise Exception("Coincident end points in Target")
        else:
            raise Exception("combine_AB_pt_PQ: 2: Should never happen!")
    #
    def pm(rv): # rv:RowVector
        return hXYCD.point_map(raw_point_map(hABab_map(rv)))
    def jac(rv): # rv:RowVector
        h_rv = hABab_map(rv)
        Bh_rv = raw_point_map(h_rv)
        return hXYCD.jacobian(Bh_rv)*raw_jacobian(h_rv)*hABab.jacobian(rv)
    if type is None:
        type = pt.type
    return PlaneTransformation(type      = type,
                               point_map = pm,
                               jacobian  = jac)


# -------------
# combine_AB_pt
# -------------
# Given pt:PlaneTransformation mapping with a "base" (0,0),(1,0) (in somehow
# connected with the definition of pt)
# and two points A,B, combine pt from the left with
# a direct similitude so that the combined map will send
# - first A->(0,0) and B->(0,1),
# - then apply pt.
# Return also the images X,Y of A,B.
# Args:
# - pt:      PlaneTransformation
# - base:    [ControlPoint,ControlPoint] = [complex,complex] (A,B)
# - type:    None or string
# Returns:
# - PlaneTransformation
# - X: RowVector (=complex)
# - Y: RowVector (=complex)
def combine_AB_pt(pt,
                  base,
                  type=None
                  ):
    A,B = base   # RowVector (=complex)
    #P,Q = target # RowVector (=complex)
    #
    rva = RowVector(0,0)
    rvb = RowVector(1,0)
    try:
        hABab = make_direct_similitude(A,B,rva,rvb)
    except MakeSimilitudeError as e:
        if e.source_error:
            raise Exception("Coincident end points in Base")
        elif e.target_rel_error: # SHould never happen
            raise Exception("combine_AB_pt_PQ: 0:Should never happen")
        else:
            raise Exception("combine_AB_pt_PQ: 1: Should never happen!")
    hABab_map = hABab.point_map
    #
    raw_point_map = pt.point_map
    raw_jacobian = pt.jacobian
    X = raw_point_map(rva)
    Y = raw_point_map(rvb)
    ##
    def pm(rv): # rv:RowVector
        return raw_point_map(hABab_map(rv))
    def jac(rv): # rv:RowVector
        h_rv = hABab_map(rv)
        return raw_jacobian(h_rv)*hABab.jacobian(rv)
    if type is None:
        type = pt.type
    return (PlaneTransformation(type      = type,
                               point_map = pm,
                               jacobian  = jac),
            X,
            Y)



class combine_ABCD_pt_PQRS_Error(Exception):
   def __init__(self, case, cause):
       self.case = case
       self.cause = cause

# --------------------
# combine_ABCD_pt_PQRS
# --------------------
# Given pt:PlaneTransformation with a "base" (0,0),(1,0),(1,1),(0,1)
# (somehow connected with the definition of pt)
# and four points A,B,C,D, and another four points P,Q,R,S,
# combine pt from the left and from the right with 
# a projective transformation so that the combined map will send
# - first A->(0,0), B->(0,1), C->(1,1), D->(1,0),
# - then apply pt,
# - then finally send the images of A,B,C,D to P,Q,R,S.
# Note: If Gimp=True, use unit square (0,0),(1,0),(1,-1),(0,-1) instead.
# Args:
# - pt:     PlaneTransformation
# - base:   [complex,complex,complex,complex] (A,B,C,D)
# - target: [complex,complex,complex,complex] (P,Q,R,S)
# - Gimp:   boolean
# - type:   None or string
# Returns:
# - PlaneTransformation
def combine_ABCD_pt_PQRS(pt,
                         base,
                         target,
                         Gimp=True,
                         type=None
                         ):
    A,B,C,D = base   # RowVector (=complex)
    P,Q,R,S = target # RowVector (=complex)
    #
    if Gimp:
        rva = -0j
        rvb = 1-0j
        rvc = 1-1j
        rvd = 0-1j
    else:
        rva = 0j
        rvb = 1+0j
        rvc = 1+1j
        rvd = 0+1j
    try:
        hABCD01 = make_projective_map(base,[rva,rvb,rvc,rvd])[0]
    except MakeProjectiveMapError as e:
        if e.case == 1:
            raise combine_ABCD_pt_PQRS_Error(case=1,
                  # Problem: some three of A,B,C,D are collinear
                  cause="some three base points are collinear")
        if e.case == 2:
            raise Exception("Can never happen!") # points 0,1,1+j,j
        else:
            raise Exception("combine_ABCD_pt_PQRS: Something weird happend at hABCD01")
    hABCD01_map = hABCD01.point_map
    #
    raw_point_map = pt.point_map
    raw_jacobian = pt.jacobian
    X = raw_point_map(hABCD01_map(A))
    Y = raw_point_map(hABCD01_map(B))
    Z = raw_point_map(hABCD01_map(C))
    U = raw_point_map(hABCD01_map(D))
    try:
        hXYZU_PQRS = make_projective_map([X,Y,Z,U],target)[0]
    except MakeProjectiveMapError as e:
        if e.case == 1:
            raise combine_ABCD_pt_PQRS_Error(case=2, 
                  # Problem: some three of X,Y,Z,Y are collinear
                  cause="some three intermediate points are collinear")
        if e.case == 2:
            raise combine_ABCD_pt_PQRS_Error(case=3, 
                  # Problem: some three of P,Q,R,S are collinear
                  cause="some three target points are collinear")
        else:
            raise Exception("combine_ABCD_pt_PQRS: Something weird happend at hXYZU_PQRS")
    def pm(rv): # rv:RowVector
        return hXYZU_PQRS.point_map(raw_point_map(hABCD01_map(rv)))
    def jac(rv): # rv:RowVector
        h_rv = hABCD01_map(rv)
        Bh_rv = raw_point_map(h_rv)
        return hXYZU_PQRS.jacobian(Bh_rv)*raw_jacobian(h_rv)*hABCD01.jacobian(rv)
    if type is None:
        type = pt.type
    return PlaneTransformation(type      = type,
                               point_map = pm,
                               jacobian  = jac)

class combine_ABCD_pt_rot_Error(Exception):
   def __init__(self, case, cause):
       self.case = case
       self.cause = cause

# -------------------
# combine_ABCD_pt_rot
# -------------------
# Given pt:PlaneTransformation with a "base" (0,0),(1,0),(1,1),(0,1)
# (somehow connected with the definition of pt)
# and four points A,B,C,D, and horizon = [c0,c1],
# combine pt from the left with a projective transformation and from
# the right with a rotation so that
# the combined map will send
# - first A->(0,0), B->(0,1), C->(1,1), D->(1,0),
# - then apply pt,
# - then rotate around the given rotation_center so that the line segment c0,c1
#   becomes horizontal pointing to the right.
# If rotation_center=None, (c0+c1)/2 is used as the rotation center.
# Note: If Gimp=True, use unit square (0,0),(1,0),(1,-1),(0,-1) instead.
# Args:
# - pt:      PlaneTransformation
# - base:    [complex,complex,complex,complex] (A,B,C,D)
# - horizon: [complex]
# - rotation_center: None or complex
# - Gimp:    boolean
# - type:    None or string
# Returns:
# - PlaneTransformation
def combine_ABCD_pt_rot(pt,
                    base,
                    horizon,
                    rotation_center=None,
                    Gimp=True,
                    type=None
                    ):
    A,B,C,D = base   # RowVector (=complex)
    #
    if Gimp:
        rva = -0j
        rvb = 1-0j
        rvc = 1-1j
        rvd = 0-1j
    else:
        rva = 0j
        rvb = 1+0j
        rvc = 1+1j
        rvd = 0+1j
    try:
        hABCD01 = make_projective_map(base,[rva,rvb,rvc,rvd])[0]
    except MakeProjectiveMapError as e:
        if e.case == 1:
            raise combine_ABCD_pt_rot_Error(case=1,
                  # Problem: some three of A,B,C,D are collinear
                  cause="some three base points are collinear")
        if e.case == 2:
            raise Exception("Can never happen!") # points 0,1,1+j,j
        else:
            raise Exception("combine_ABCD_pt_PQRS: Something weird happend at hABCD01")
    hABCD01_map = hABCD01.point_map
    #
    raw_point_map = pt.point_map
    raw_jacobian = pt.jacobian
    try:
        c0,c1 = horizon[0], horizon[1]
    except IndexError:
        raise Exception("combine_ABCD_pt_rot: horizon has not not enough points")
    if rotation_center is None:
        center = (c0+c1)/2 # rotation center
    else:
        center = rotation_center
    try:
        z = complex(abs(c1-c0), 0.) / (c1-c0) # Performs rotation
    except ZeroDivisionError:
        raise Exception("combine_ABCD_pt_rot: zero horizon")
    if Gimp:
        rot_matrix = Matrix2x2(z.real, -z.imag,
                               z.imag,  z.real)
    else: # Untested!
        rot_matrix = Matrix2x2( z.real, z.imag,
                               -z.imag, z.real)
    def pm(rv): # rv:RowVector
        #return raw_point_map(hABCD01_map(rv))
        return center + z * (raw_point_map(hABCD01_map(rv)) - center)
    def jac(rv): # rv:RowVector
        h_rv = hABCD01_map(rv)
        #return raw_jacobian(h_rv)*hABCD01.jacobian(rv)
        return rot_matrix * raw_jacobian(h_rv)*hABCD01.jacobian(rv)
    if type is None:
        type = pt.type
    return PlaneTransformation(type      = type,
                               point_map = pm,
                               jacobian  = jac)

# ------------------
# combine_AB_pt_PQRS
# ------------------
# Given pt:PlaneTransformation with a "base" (0,0),(1,0)
# (somehow connected with the definition of pt)
# and two points A,B, and another four points P,Q,R,S,
# combine pt from the left with a direct similitude
# and from the right with a projective transformation so that
# the combined map will send
# - first A->(0,0), B->(0,1)
# - then apply pt,
# - then finally send the images of (0,0),(1,0),(1,1),(0,1) to P,Q,R,S.
# Args:
# - pt:     PlaneTransformation
# - base:   [complex,complex] (A,B)
# - target: [complex,complex,complex,complex] (P,Q,R,S)
# - Gimp:   boolean
# - type:   None or string
# Returns:
# - PlaneTransformation
def combine_AB_pt_PQRS(pt,
                     base,
                     target,
                     Gimp=True,
                     type=None
                     ):
    A,B = base       # RowVector (=complex)
    P,Q,R,S = target # RowVector (=complex)
    # Complete A,B to a square
    if Gimp:
        ABperp = complex((-A+B).imag, -(-A+B).real) # Gimp!
    else:
        ABperp = complex(-(-A+B).imag, (-A+B).real)
    C = B+ABperp
    D = C+A-B
    #
    rva = 0j
    rvb = 1+0j
    try:
        hAB01 = make_direct_similitude(A,B,rva,rvb)
    except:
        raise
    hAB01_map = hAB01.point_map
    #
    raw_point_map = pt.point_map
    raw_jacobian = pt.jacobian
    X = raw_point_map(hAB01_map(A))
    Y = raw_point_map(hAB01_map(B))
    Z = raw_point_map(hAB01_map(C))
    U = raw_point_map(hAB01_map(D))
    try:
        hXYZU_PQRS = make_projective_map([X,Y,Z,U],target)[0]
    except:
        raise
    def pm(rv): # rv:RowVector
        return hXYZU_PQRS.point_map(raw_point_map(hAB01_map(rv)))
    def jac(rv): # rv:RowVector
        h_rv = hAB01_map(rv)
        Bh_rv = raw_point_map(h_rv)
        return hXYZU_PQRS.jacobian(Bh_rv)*raw_jacobian(h_rv)*hAB01.jacobian(rv)
    if type is None:
        type = pt.type
    return PlaneTransformation(type      = type,
                               point_map = pm,
                               jacobian  = jac)



# ------------------
# combine_ABCD_pt_PQ
# ------------------
# Given pt:PlaneTransformation with a "base" (0,0),(1,0),(1,1),(0,1)
# (somehow connected with the definition of pt)
# and four points A,B,C,D, and another two points P,Q,
# combine pt from the left and with a projective transformation
# and from the right with a projective transformation
# so that the combined map will send
# - first A->(0,0), B->(0,1), C->(1,1), D->(1,0),
# - then apply pt,
# - then finally send the images of A,B,C,D to P,Q,R,S.
# Note: If Gimp=True, use unit square (0,0),(1,0),(1,-1),(0,-1) instead.
# Args:
# - pt:     PlaneTransformation
# - base:   [complex,complex,complex,complex] (A,B,C,D)
# - target: [complex,complex] (P,Q)
# - Gimp:   boolean
# - type:   None or string
# Returns:
# - PlaneTransformation
def combine_ABCD_pt_PQ(pt,
                     base,
                     target,
                     Gimp=True,
                     type=None
                     ):
    A,B,C,D = base   # RowVector (=complex)
    P,Q = target # RowVector (=complex)
    #
    if Gimp:
        rva = -0j
        rvb = 1-0j
        rvc = 1-1j
        rvd = 0-1j
    else:
        rva = 0j
        rvb = 1+0j
        rvc = 1+1j
        rvd = 0+1j
    try:
        hABCD01 = make_projective_map(base,[rva,rvb,rvc,rvd])[0]
    except:
        raise
    hABCD01_map = hABCD01.point_map
    #
    raw_point_map = pt.point_map
    raw_jacobian = pt.jacobian
    X = raw_point_map(hABCD01_map(A))
    Y = raw_point_map(hABCD01_map(B))
    try:
        hXYZU_PQ = make_direct_similitude(X,Y,P,Q)
    except:
        raise
    def pm(rv): # rv:RowVector
        return hXYZU_PQ.point_map(raw_point_map(hABCD01_map(rv)))
    def jac(rv): # rv:RowVector
        h_rv = hABCD01_map(rv)
        Bh_rv = raw_point_map(h_rv)
        return hXYZU_PQ.jacobian(Bh_rv)*raw_jacobian(h_rv)*hABCD01.jacobian(rv)
    if type is None:
        type = pt.type
    return PlaneTransformation(type      = type,
                               point_map = pm,
                               jacobian  = jac)


#---------------------------------------------------------
#                   Auxiliary routines
#---------------------------------------------------------

# From input string make list.
# Arg s (string) should evaluate to
# - float or int
# - list of floats or ints
# - tuple of floats or ints
# Returns: [float]
def string_to_list(s): # s:string
    if s.strip() == '': # blank
        return []
    try:
        list_or_tuple = eval(s)
        if type(list_or_tuple) in (int,float):
            list_or_tuple = [float(list_or_tuple)]
        return [float(p) for p in list_or_tuple]
    except:
        m = ("Something wrong in input: "
            +"\n    "+s
            +"\nPlease check syntax and so on.")
        raise Exception(m)

def is_loop(cp4):
    ZERO = 1e-12
    p0, p3 = cp4[0], cp4[-1]
    p03 = -p0+p3
    return (vdot(p03,p03) < ZERO)

def gimp_message(txt, h=2):
    # 0 = message box; 1 = console; 2 = error console
    pdb.gimp_message_set_handler(h)
    pdb.gimp_message(txt)

# --------------
# dist_from_line
# --------------
# Find (signed) distance of x from line through a,b when a!=b.
# If a=b, return distance of x from a=b.
# Args:
# - a,b,x: RowVector # complex
# Returns:
# - float
def dist_from_line(a,b,x):
    ZERO = 1e-12
    x0 = x-a
    b0 = b-a
    b0len = abs(b0)
    if b0len < ZERO:
        return abs(x0)
    return (x0*b0.conjugate()).imag / b0len

# -----------------------
# line_through_two_points
# -----------------------
# Given two points, find coefficients g,h,k of the line gx+hy+k=0
# which passes through p and q.
# Args:
# - p,q:complex (the plane is treated as the complex plane)
# Returns:
# - [float,float,float] (these are the g,h,k)
def line_through_two_points(p,q):
    p1,p2 = p.real,p.imag
    q1,q2 = q.real,q.imag
    g = -p2+q2
    h =  p1-q1
    k = -p1*q2+p2*q1
    return [g,h,k]


# --------------------
# parallelogram_vertex
# --------------------
# Given three vertices for a parallelogram, find the third.
# Args:
# - a,b,c: complex
# Returns_
#- complex
def parallelogram_vertex(a,b,c):
    return a-b+c

# -----------------------------
# complete_quadrilateral_vertex
# -----------------------------
# Given three points and a line L:gx+hy+k=0 (all in a general position
# in some sense), consider the complete quadrilateral having a,b,c as
# vertices and two vertices e,f on L, find the sixth vertex d.
# The points b,a,e are collinear, as are b,c,f, and c,d,e, and a,d,f.
# Args:
# - a,b,c: complex;
# - line_coefficients: [float,float,float] (these are the g,h,k).
def complete_quadrilateral_vertex(a,b,c,line_coefficients):
    e,_ = line_intersect_segment(line_coefficients, a, b)
    f,_ = line_intersect_segment(line_coefficients, c, b)
    ec = line_through_two_points(e,c) # some g,h,k
    d,_ = line_intersect_segment(ec, a, f)
    return d

# ----------------------
# line_intersect_segment
# ----------------------
# Given a line L:gx+hy+k=0 and two points P,Q, find the intersection
# (if any) of L with the line through P,Q and whether it recides at:
# - exactly at P
# - exactly at Q
# - in the open interval (P,Q).
# Args:
# - line_coefficients: [float,float,float] (these are the g,h,k);
# - p: complex;
# - q: complex.
# Returns:
# - complex or None (intersection point; or None if lines parallel);
# - [boolean,boolean,boolean] (exactly at p, exactly at q, in open interval).
# Note: Special case:
# 1. When P,Q both on L, returned values are
#    - None
#    - [True,True,False]
def line_intersect_segment(line_coefficients, p, q):
    g,h,k = line_coefficients
    p1,p2 = p.real, p.imag
    q1,q2 = q.real, q.imag
    try:
        c = 1./((g*p1 + h*p2) - (g*q1 + h*q2))
    except ZeroDivisionError: # lines parallel or p=q
        if g*p1+h*p2+k == 0: # p and q on L
            return None, [True,True,False]
        else:
            return None, [False,False,False]
    pcoeff = -(g*q1 + h*q2 + k) * c
    qcoeff =  (g*p1 + h*p2 + k) * c
    p_exact = (qcoeff == 0)
    q_exact = (pcoeff == 0)
    in_open = (0 < pcoeff < 1)
    X = pcoeff*p + qcoeff*q
    #check = g*X[0] + h*X[1] + k
    return X, [p_exact, q_exact, in_open]

# -----------------------
# line_intersect_polyline
# -----------------------
# Given a line L:gx+hy+k=0 and a polyline, find intersection points.
# Args:
# - line_coefficients: [float,float,float] (these are the g,h,k);
# - polyline: [complex].
# Returns:
# - [complex] (possibly empty list)
# Notes:
# 1. If some line segment of the polyline is contained in L,
#    of the line segment only the end points will be in the returned list.
# 2. Repeated vertices will be repeated in the result too if on L.
def line_intersect_polyline(line_coefficients, polyline):
    vertices = polyline
    intersections = []
    for i in range(-1+len(vertices)):
        point,loc = line_intersect_segment(line_coefficients,
                                           vertices[i],
                                           vertices[i+1])
        if point is None: # Segment parallel to L or vertices coincide
            if loc[0]: # Both vertices on L
                intersections.append(vertices[i])
                if i == -2+len(vertices):
                    intersections.append(vertices[i+1])
        elif loc[0] or loc[2]: # L intersects either at start or in the open interval
            intersections.append(point)
        elif loc[1] and (i == -2+len(vertices)): # Last vertex is on L
            intersections.append(point)
    return intersections


def some_three_collinear(four_points): # four_points = [x,y,z,u]: [complex]
    x,y,z,u = four_points
    return (collinear([x,y,z]) or collinear([x,y,u])
                or collinear([x,z,u]) or collinear([y,z,u]))

# -----------------
# line_intersection
# -----------------
# Find intersection of two lines given by one point on each (p0,p1)
# and direction vectors of each.
# The plane is used as the complex plane.
# Args:
# - p0,d0,p1,d1: complex
# Returns:
# - complex
# Exception is raised if the lines are parallel (includes cases where
# a direction vector is zero).
def line_intersection(p0,d0,p1,d1):
    try:
        return (-cross(p0,d0)*d1 + cross(p1,d1)*d0) / cross(d0,d1)
    except ZeroDivisionError:
        raise Exception("line_intersection: parallel lines or zero direction vectors")


# ---------
# is_convex
# ---------
# Is a polygon convex? The polygon is inputted as
# a list of corner points (complex).
# Args:
# - corners: [complex]
# Returns:
# - boolean
# Notes:
# 1. Returns True if all corner points are collinear.
def is_convex(corners):
    if len(corners) <= 3:
        return True
    if collinear(corners): # TEE
        return True
    c = corners + corners[:2]
    edges = [c[i]-c[i+1] for i in range(len(c)-1)]
    edges = [e for e in edges if e != 0] # remove zero edges
    pairs = [[edges[i],edges[i+1]] for i in range(len(edges)-1)]
    # Cross and dot products:
    cr_dots = [[-a.real*b.imag + a.imag*b.real,
                 a.real*b.real + a.imag*b.imag] for a,b in pairs]
    crs = []
    for crd in cr_dots:
        if (crd[0]==0):
            if (crd[1]<0):
                return False # spike of angle 0
        else:
            crs.append(crd[0]) # collect non-zero cross products
    try:
        pos = (crs[0] > 0)
    except IndexError: # All points collinear
        return True
    for cr in crs:
        if pos != (cr > 0):
            return False
    return True


# -------------------
# get_guide_positions
# -------------------
# Guide positions in image, horizontal and vertical in separate lists.
# Not sorted.
def get_guide_positions(image):
    horizontal = []
    vertical = []
    guide = 0
    while True:
        guide = pdb.gimp_image_find_next_guide(image,guide)
        if guide == 0:
            break
        position = pdb.gimp_image_get_guide_position(image, guide)
        orientation = pdb.gimp_image_get_guide_orientation(image, guide)
        if orientation == ORIENTATION_HORIZONTAL:
            horizontal.append(position)
        elif orientation == ORIENTATION_VERTICAL:
            vertical.append(position)
        else:
            raise Exception("get_guide_positions: ???")
    return horizontal, vertical

def str_float(x): return '{:1.3f}'.format(x)

# -------------------------
# solve_bernstein2_equation
# -------------------------
# Find real zeroes of the equation c0*b0(t) + c1*b1(t) + c2*b2(t) = 0 where
# b0,b1,b2 are the Bernstein polynomials of degree 2 and
# c0,c1,c2 are real coefficients.
# Return the zeroes as a list (0, 1, or 2 items).
# Exception: If the function is constant 0, return None.
# Args:
# - c0,c1,c2: float
# Returns:
# - None or list of floats (possibly empty)
def solve_bernstein2_equation(c0,c1,c2):
    from math import sqrt
    ZERO = 1e-14
    if abs(c0 - 2.*c1 + c2) < ZERO: # equation of degree <= 1
        try:
            return [c0 / (2.*(c0 - c1))]
        except ZeroDivisionError: # equation of degree 0
            if abs(c0) < ZERO: # constant 0
                return None
            else: # constant not 0
                return []
    try:
        root = sqrt(c1**2 - c0*c2)
    except ValueError: # No real roots
        return []
    return [(c0 - c1 + root) / (c0 - 2.*c1 + c2),
            (c0 - c1 - root) / (c0 - 2.*c1 + c2)]

# -----------
# arrange_cps
# -----------
# Given list of control points ([complex]),
# arrange it cyclically and possibly reversing so that
# the resulting list  a  has as the segment  a[0],a[1]  the lowest
# on the screen (in some sense) and left to right.
# Args:
# - cps:  [complex]
# - Gimp: boolean
# Returns:
# - [complex]
def arrange_cps(cps, Gimp=True):
    if Gimp: c = -1
    else: c = 1
    n = len(cps)
    if n <= 1:
        return cps
    # Find lowest anchor:
    lowi = min(range(n), key=(lambda j: c*cps[j].imag))
    # Check the neighbouring anchors:
    upwards = (c*cps[(lowi-1)%n].imag > c*cps[(lowi+1)%n].imag)
    # Arrange item with index lowi to place 0
    # and the lower neighbour to place 1:
    if upwards:
        a = [cps[(i+lowi)%n] for i in range(n)]  # cyclic
    else:
        a = [cps[(-i+lowi)%n] for i in range(n)] # cyclic + reverse
    # Check if left to right:
    if a[0].real <= a[1].real:
        return a # already ok
    else:
        return [a[(1-i)%n] for i in range(n)] # reverse keeping the low segment

# --------------------
# rotate_to_horizontal
# --------------------
# Given a list c of complex numbers, rotate the whole contents in the
# complex plane around the center point by a constant angle so that the line segment
# c[0],c[1] becomes horizontal.
# Args:
# - complex_list:[complex]
# Returns:
# - [complex]
def rotate_to_horizontal(complex_list):
    try:
        c0,c1 = complex_list[0], complex_list[1]
    except IndexError:
        raise Exception("rotate_to_horizontal: not enough points")
    #center = c0 # Change if you want some other rotation center.
    center = sum(complex_list) / len(complex_list) # rotation center = center point
    try:
        z = complex(abs(c1-c0), 0.) / (c1-c0)
    except ZeroDivisionError:
        raise Exception("rotate_to_horizontal: zero first line segment")
    return [center + z*(c-center) for c in complex_list]


# -------------------
# rotate_complex_list
# -------------------
# Rotate a list of complex numbers in the complex plane around center by angle
# Positive angle: rotate ccw. If Gimp=True, y coordinate runs downwards.
# Args:
# - clist:  [complex]
# - center: complex
# - angle:  float
# - Gimp:   boolean
def rotate_complex_list(clist, center, angle, Gimp=True):
    from math import cos,sin
    ca, sa = cos(angle), sin(angle)
    if Gimp:
        za = complex(ca, -sa)
    else:
        za = complex(ca, sa)
    return [center + (c-center)*za for c in clist]


#---------------------------------------------------------
#                Routines for Bezier curves
#---------------------------------------------------------

# ------------
# polygon_path
# ------------
# From list of vertices (complex) construct gimp_vectors.
# Args:
# - image
# - vertices: [complex]
# - name:     string
# - closed:   boolean
# Returns:
#- gimp.Vectors
def polygon_path(image, vertices, name='polygon', closed=True):
    polygon = pdb.gimp_vectors_new(image, name)
    points = []
    for vertex in vertices:
        points += 3*[vertex.real, vertex.imag]
    #pdb.gimp_image_undo_group_start(image) ### Gimp
    stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                        polygon, 0, len(points), points, closed)
    #gimp_draw_vectors_object(image, polygon, visible=True)
    #pdb.gimp_image_undo_group_end(image) ### Gimp
    return polygon


# ------------
# handles2zero
# ------------
# From a list of Bezier arcs, make a new one with the same anchors
# but with zero length handles.
# Args:
# - ba_list: [BCurve.BezierArc]
# Returns:
# - [BCurve.BezierArc]
def handles2zero(ba_list):
    new_ba_list = []
    for ba in ba_list:
        cp4 = ba.cp4
        if len(cp4) == 1:
            new_ba_list.append(BCurve.BezierArc(cp4=cp4))
        else:
            new_ba_list.append(BCurve.BezierArc(cp4=[cp4[0],cp4[0],cp4[3],cp4[3]]))
    return new_ba_list

# -----------
# get_anchors
# -----------
# Get anchors of the path (optionally the first stroke only).
# Arg:
# - vectors_object:    gimp.Vectors
# - only_first_stroke: boolean
# Returns:
# - [BCurve.ControlPoint]
def get_anchors(vectors_object, only_first_stroke=False): # MUUTOS: True->False
    gv = vectors_object2gv(vectors_object) # BCurve.GimpVectors
    if len(gv.stroke_list) == 0:
        raise Exception("Path without any strokes: "+vectors_object.name)
    anchors = []
    for gs in gv.stroke_list:
        bc = gs.gs2bc()                        # BCurve.BezierCurve
        bas = bc.bezier_arcs                   # [BCurve.BezierArc]
        #anchors = [ba.cp4[0] for ba in bas]
        anchors += [ba.cp4[0] for ba in bas]
        if len(bas[0].cp4) > 1: # Case len=1 may rise from stroke with one anchor.
            anchors.append(bas[-1].cp4[-1])
        if only_first_stroke:
            break
    return anchors

# -------------------------
# bezier_new_control_points
# -------------------------
# Given a Bezier arc B(t) and two distinct parameter values a and b,
# find a new parametrization C(t) of the same curve such that
# B(a) = C(0) and B(b) = C(1).
# Args:
# - a: float;
# - b: float;
# - control_points: the list [p0,p1,p2,p3] of control points (pi:{float,float]).
# Returns:
# - new control points [q0,q1,q2,q3]:[RowVector] such that the corresponding
#   Bezier curve C(t) is precisely the same as B as a curve
#   but B(a) = C(0) and B(b) = C(1).
# Notes:
# - This enables one to extend or to shorten the given Bezier arc.
# - There is an exception: If a=b, the returned control points
#   are equal, so the new curve is a single point, not the original
#   curve. The function gives no signal about such occurence.
def bezier_new_control_points(a, b, control_points):
    def bezier_2_rv(t, r0, r1, r2): # Bezier curve of degree two (complex)
        b20 = (1-t)**2
        b21 = 2*(1-t)*t
        b22 = t**2
        return b20*r0 + b21*r1 + b22*r2
    p0,p1,p2,p3 = control_points
    q0 = bezier_rv(a, control_points)
    q1 = (1-b)*bezier_2_rv(a, p0,p1,p2) + b*bezier_2_rv(a, p1,p2,p3)
    q2 = (1-a)*bezier_2_rv(b, p0,p1,p2) + a*bezier_2_rv(b, p1,p2,p3)
    q3 = bezier_rv(b, control_points)
    return [q0,q1,q2,q3]

# ---------------------
# bezier_special_points
# ---------------------
# Find the special points of a cubic Bezier curve:
# inflection points, cusp points, self-intersection points.
#
# Args:
# - control_points: [p0,p1,p2,p3], pi:complex
#
# Returns: (case, param_list), which is one of:
# (-1, [t1,t2]) when there is a loop or a cusp point, with
#               t1,t2 the parameter values with B(t1)=B(t2)
#               (ideally, t1=t2 in the case of a cusp);
# (0, [])       when the curve is degenerate: a parabola or
#               contained on a straight line;
# (1, [t])      when there is a single inflection point, with
#               t the parameter of the inflection point.
# (2, [t1,t2])  when there are two inflection points, with
#               t1,t2 the parameters for the inflection points.
# Note: The algorithm works as if infinite computation precision were
#   possible. The caller had better check the following:
# - In the case of a loop or of two inflection points (cases -1 or 2):
#   If the returned parameter values t1,t2 are very close, it may be
#   a sign of a cusp point, instead. (Or maybe the caller should take
#   it as a cusp point.)
# - In the case of inflection points (cases 1 or 2):
#   If one or both of the returned parameter values has very large
#   absolute value, it may mean that the inflection point is at infinity
#   (so, non-existent).
def bezier_special_points(control_points):
    from math import sqrt
    p0,p1,p2,p3 = control_points # complex
    w1 = -p0 - p1 + p2 + p3
    w2 =  p0 - p1 - p2 + p3
    w3 = -p0 + 3*p1 - 3*p2 + p3
    #
    C1 = 18.0 * (-w2.real*w3.imag + w2.imag*w3.real)
    C2 = 4.50 * (-w3.real*w1.imag + w3.imag*w1.real)
    C3 = 2.25 * (-w1.real*w2.imag + w1.imag*w2.real)
    try:
        X = C2/C1
        Y = C3/C1
        Z = -X**2 + 2*Y
        try:
            root = sqrt(3*Z)
            t1 = .5 + X + root
            t2 = .5 + X - root
            return -1, [t1,t2] # loop or cusp
        except ValueError: # Z < 0
            root = sqrt(-Z)
            t1 = .5 + X + root
            t2 = .5 + X - root
            return 2, [t1,t2] # two inflection points
    except ZeroDivisionError: # C1 = 0
        try:
            t = .5 + C3/C2
            return 1, [t] # one inflection point
        except ZeroDivisionError: # C1 = C2 = 0
            return 0,[] # parabola or everything on straight line.

# ------------------------
# bezier_arc_protect_cusps
# ------------------------
# Given a Bezier curve bc:BCurve.BezierCurve, check if any of its Bezier arcs
# has a cusp or two very close inflection points in the interval 0<t<1.
# If so, replace such arc with two or three subarcs (generally three, the middle
# one very small), to have the cusp as one anchor or to have the arc between the
# inflection points isolated.
# Return new Bezier curve.
# Args:
# - bc:BCurve.BezierCurve
# Returns:
 # - BCurve.BezierCurve
def bezier_arc_protect_cusps(bc):
    ZERO = 1e-3
    new_bc = copy.deepcopy(bc) # Just in case bc is still used somewhere.
    new_ba_list = []           # Construct new list of BezierArcs
    for ba in bc.bezier_arcs:
        cp4xy = ba.cp4 # [ControlPoint] = [complex]
        if len(cp4xy) == 1:
            new_ba_list.append(ba)
            continue
        special = bezier_special_points(cp4xy)
        if special[0] not in (-1,2): # No loops and at most one inflection point
            new_ba_list.append(ba)
            continue
        # Found either a double point or two inflection points
        # with parameter values t1,t2.
        # If they are very close, we take it to be a sign of a cusp.
        t1,t2 = extra_ts = sorted(special[1])
        if t2-t1 > ZERO: # Not so very close
            new_ba_list.append(ba)
            continue
        if t1 == t2: # cusp
            extra_ts = [t1]
        #extra_ts_in_01 = [t for t in extra_ts if 0 < t < 1]
        extra_ts_in_01 = [t for t in extra_ts if ZERO < t < 1-ZERO]
        if len(extra_ts_in_01) == 0:
            new_ba_list.append(ba)
        else: # subdivide ba into subarcs
            ts = [0] + extra_ts_in_01 + [1]
            for i in range(len(ts)-1):
                left_t, right_t = ts[i], ts[i+1]
                sub_cp4 = bezier_new_control_points(left_t, right_t, cp4xy)
                sub_ba = BCurve.BezierArc(cp4 = sub_cp4)
                new_ba_list.append(sub_ba)
    new_bc.bezier_arcs = new_ba_list
    return new_bc

# -------------------------
# bezier_arc_end_directions
# -------------------------
# Given a Bezier arc, find, in each end, the direction vector of the arc.
# Note: this does not mean the derivatives of the parametric representation,
# but non-zero vectors which are tangential to the arc and point in the
# direction where the arc is located when viewed from that end point.
# Return the result as two RowVectors, or as None if the arc is one point.
# Args:
# - ba: BCurve.BezierArc
# Returns: either None (when the arc is one point), or
# - RowVector (direction vector at starting point)
# - RowVector (direction vector at ending point)
def bezier_arc_end_directions(ba):
    ZERO = 1e-12
    if len(ba.cp4) == 1:
        return None
    p0,p1,p2,p3 = ba.cp4
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    if vdot(p01,p01) > ZERO:
        head_dir = p01
    elif vdot(p12,p12) > ZERO:
        head_dir = p12
        #print("1:head_dir = p12 ="+str(p12))
    elif vdot(p23,p23) > ZERO:
        head_dir = p23
        #print("2:head_dir = p23 ="+str(p23))
    else:
        return None
    if vdot(p23,p23) > ZERO:
        tail_dir = -p23
        #print("3:head_dir = p23 ="+str(-p23))
    elif vdot(p12,p12) > ZERO:
        tail_dir = -p12
        #print("4:head_dir = -p12 ="+str(-p12))
    elif vdot(p01,p01) > ZERO:
        tail_dir = -p01
    else:
        raise Exception("bezier_arc_end_directions: Should never happen!")
    return head_dir, tail_dir

# ------------------------
# bezier_arc_map_subdivide
# ------------------------
# Given a Bezier arc B(t):[0,1]->C and a plane transformation F:C->C,
# subdivide the interval [0,1] in 2**n parts, 0 < t1 < t2 < ... < 1,
# so that the chords between successive points F(B(ti)) are very roughly of
# equal length.
# The Bezier arc is given as a list of 4 control points ([complex]),
# and the map is F = pt.point_map where pt:PlaneTransformation.
# Return (None or) two lists:
# - [[t,F(B(t))],...] where F = pt.point_map
# - [[t,J(B(t))],...] where J = pt.jacobian
# Args:
# - cp: [complex,...] (four control points)
# - pt: PlaneTransformation
# - n: integer
# Returns:
# - None or 
#  ([[t0,F0],[t1,F1], ...], [[t0,J0],[t1,J1], ...])
#  where ti:float, Fi:complex, Ji:complex
# (sorted relative parameter values for the subdivision, 0 < t0 < t1 ... < 1)
# Note: None is returned if the input arc cp is of zero length.
def bezier_arc_map_subdivide(cp, pt, n=1):
    ZERO_LENGTH = 1e-10
    F = pt.point_map
    def length2(a,b): # a,b:RowVector = complex
        return ((a-b)*(a-b).conjugate()).real
    # List of subarcs: [[[lo,F(B(lo)), [hi,F(B(hi))], length2], ...],
    # kept constantly sorted according to length2, longest first.
    F0 = F(bezier_rv(0,cp))
    F1 = F(bezier_rv(1,cp))
    subarcs = [[[0,F0], [1,F1], length2(F0,F1)]]
    if length2(F0,F1) < ZERO_LENGTH:
        Fmeans = [F(bezier_rv(t,cp)) for t in [.25,.5,.75]]
        maxdist20 = max([length2(F0,Fm) for Fm in Fmeans])
        maxdist21 = max([length2(Fm,F1) for Fm in Fmeans])
        if max(maxdist20,maxdist21) < ZERO_LENGTH:
            return None
    max_i_rounds = 0
    #no_break = 0
    for k in range(2**n - 1):
        longest = subarcs[0] # Zeroth is the longest
        loF, hiF = longest[0], longest[1]
        lo, hi = loF[0], hiF[0]
        t_try = (lo + hi)/2. # Ad hoc: split first at middle t
        # Try to find better t_try:
        try_results = []
        for i in range(20):
            F_try = F(bezier_rv(t_try, cp))
            d0 = sqrt(length2(loF[1], F_try)) # chord length
            d1 = sqrt(length2(F_try, hiF[1])) # chord length
            try:
                dd_try = d1/(d0+d1) # Trying to get d0/d1=1 => dd_try=1/2
                try_results.append([t_try, dd_try])
                if .35 < dd_try < .65: # Ad hoc
                    break # Found ok: break
                dd_try = (1-dd_try)*lo + dd_try*hi # Ad hoc
                #t_try = (t_try + dd_try) / 2 # Ad hoc
                t_try = (t_try + (i+1)*dd_try) / (i+2) # Ad hoc  KOE
            except ZeroDivisionError:
                break
        else:
            pass
        if len(try_results)>0:
            best_try = min(try_results, key=(lambda x:abs(x[1]-.5)))
            mi = best_try[0]
            miF = [mi, F(bezier_rv(mi,cp))]
            left  = [loF, miF, length2(loF[1],miF[1])]
            right = [miF, hiF, length2(miF[1],hiF[1])]
            # Replace longest with left,right and sort:
            subarcs = sorted([left,right] + subarcs[1:],
                              key = (lambda x:-x[2]))
        else: # Zero length arc: did not find anything
            raise Exception("bezier_arc_map_subdivide: Called with zero length arc?")
    loFs = [x[0] for x in subarcs]           # Drop hi's and lengths
    loFs = sorted(loFs, key=(lambda x:x[0])) # Sort according to t
    t_F_list = loFs[1:] # Drop zeroth
    t_jac_list = [[t, pt.jacobian(bezier_rv(t,cp))] for t,Ft in t_F_list]
    return t_F_list, t_jac_list

# ---------------------------
# bezier_curve_tangent_points
# ---------------------------
# Given a Bezier curve B(t) and a non-zero direction vector d,
# find the points on B with B'(t) parallel to d.
# If restrict01=True, accept only those with 0<=t<=1.
# Return as a list of pairs [t,p] where t is the parameter value (float)
# and p is the touching point B(t) (complex).
# Exception: If d=0 or if the the curve is contained on a line parallel to d,
# return None.
# Args:
# - control_points: [complex,complex,complex,complex]
# - direction:      complex (assumed non-zero)
# - restrict01:     boolean
# Returns:
# - list [[t,p]]: [[float,complex]]
def bezier_curve_tangent_points(control_points, direction, restrict01=True):
    p0,p1,p2,p3 = control_points
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    # pij dot (direction perpendicular):
    p01d = -p01.real * direction.imag + p01.imag * direction.real
    p12d = -p12.real * direction.imag + p12.imag * direction.real
    p23d = -p23.real * direction.imag + p23.imag * direction.real
    #
    ts = solve_bernstein2_equation(p01d,p12d,p23d)
    if ts is None:
        return None
    if restrict01:
        ts = [t for t in ts if 0<=t<=1]
    tBs = [[t, bezier_rv(t, control_points)] for t in ts]
    return tBs

# --------------------------
# BezierCurve_tangent_points
# --------------------------
# Given a Bezier curve:Bcurve.BezierCurve (essentially list of butting
# Bezier arcs) and a non-zero direction vector d,
# find the points on B where the curve is parallel to d.
# Includes cusps and isolated anchors.
# For arcs which are all the way parallel to d, the point
# with t=0.5 is returned
# Args:
# - bcurve: BCurve.BezierCurve
# - direction: complex (assumed non-zero)
# Returns:
# - [complex]
def BezierCurve_tangent_points(bcurve, direction):
    balist = bcurve.bezier_arcs
    # In Gimp a closed stroke is only marked to be closed.
    # If so, we must append one more arc to close the gap:
    if bcurve.closed:
        balist.append(BCurve.BezierArc(cp4 = [balist[-1].cp4[-1],
                                      bcurve.tail_handle,
                                      bcurve.head_handle,
                                      balist[0].cp4[0]]))
    cp4s = [ba.cp4 for ba in balist]
    points = []
    for cp4 in cp4s:
        if len(cp4) == 1: # A one-anchore stroke
            points.append(cp4[0])
            continue
        tBs = bezier_curve_tangent_points(cp4, direction, restrict01=True)
        if tBs is None: # arc is straight line segment parallel to direction
            t = 0.5
            p = bezier_rv(t, cp4)
            points.append(p)
            #t = 1.
            #p = bezier_rv(t, cp4)
            #points.append(p)
        else:
            for t,p in tBs:
                points.append(p)
    return points

# -------------------
# path_tangent_points
# -------------------
# Given Gimp's path and a non-zero direction vector d,
# find the points where some stroke is parallel to d.
# Include cusps.
# Exception: For arcs which are all the way parallel the point t=0.5 is chosen.
# Args:
# - path_vectors: gimp.Vectors
# - direction: complex (assumed non-zero)
# Returns:
# - [complex] (touching points)
def path_tangent_points(path_vectors, direction):
    gv = vectors_object2gv(path_vectors) # BCurve.GimpVectors
    points = []
    for gs in gv.stroke_list:  # BCurve.GimpStroke
        bc = gs.gs2bc()         # BCurve.BezierCurve
        points += BezierCurve_tangent_points(bc, direction)
        #try: # Check first if this is a one-anchor stroke
        #    if len(bc.bezier_arcs[0].cp4)==1:
        #        continue
        #except IndexError: # No arcs???
        #    raise Exception("path_tangent_points: Something weird: IndexError")
    return points

# -------
# path_bb
# -------
# Bounding box of Gimp's path. Return as list [NW, SE] of two corner points,
# where NW is top left, SE is bottom right. (The directions are indeed as
# on the screen: N is up, S is down, provided that the argument 'Gimp' is correct!)
# Args:
# - path_vectors: gimp.Vectors
# - Gimp: boolean
# Returns:
# - [NW, SE]: [complex,complex]
def path_bb(path_vectors, Gimp=True):
    hor = path_tangent_points(path_vectors, 1+0j)
    ver = path_tangent_points(path_vectors, 0+1j)
    anchors = get_anchors(path_vectors, only_first_stroke=False)
    xmax = max([v.real for v in ver+anchors])
    xmin = min([v.real for v in ver+anchors])
    ymax = max([v.imag for v in hor+anchors])
    ymin = min([v.imag for v in hor+anchors])
    if Gimp:
        return [complex(xmin,ymin), complex(xmax,ymax)]
    else:
        return [complex(xmin,ymax), complex(xmax,ymin)]


#---------------------------------------------------------
#                Some routines for Support
#---------------------------------------------------------

# ----------------
# map_make_support
# ----------------
# Given pt:PlaneTransformation and ba:BCurve.BezierArc (assumed to have 4
# control points), imagine that pt is applied to the arc; compute a support
# for the image.
# Args:
# - pt: PlaneTransformation
# - ba: BCurve.BezierArc
# Returns:
# - Support
# Note:
# 1. The end points (anchors) are mapped by pt.point_map.
# 2. The direction vectors derived from the inner control points (handles)
#    are mapped by pt.jacobian. Then they will be tangential to the (imagined)
#    mapped arc. The called routine 'bezier_arc_end_directions' takes care of
#    the cases of zero handles but returns None if the arc is one point
#    (coincident control points).
def map_make_support(pt, ba):
    try:
        rv0 = ba.cp4[0] # RowVector = complex
        rv3 = ba.cp4[3] # RowVector = complex
    except IndexError:
        raise Exception("map_make_support: Arc does not have 4 control points.")
    try:
        direction_vectors = bezier_arc_end_directions(ba)
    except:
        raise ExitError("map_make_support: bezier_arc_end_directions")
    if direction_vectors is None:
        raise Exception("map_make_support: Arc has 4 equal control points.")
    rv_01, rv_32 = direction_vectors # RowVectors = complex
    #
    pt_rv0 = pt.point_map(rv0) # RowVector = complex
    pt_rv3 = pt.point_map(rv3) # RowVector = complex
    pt_rv01 = rv_times_M2x2_transpose(rv_01, pt.jacobian(rv0)) # RowVector = complex
    pt_rv32 = rv_times_M2x2_transpose(rv_32, pt.jacobian(rv3)) # RowVector = complex
    pt_rv1 = pt_rv0 + pt_rv01
    pt_rv2 = pt_rv3 + pt_rv32
    #
    S = [pt_rv0, pt_rv1, pt_rv2, pt_rv3]
    return Support(S) # May raise NoSupportError


# ----------------------------------------------------------------
#   Map and approximate a bezier arc: various simple algorithms
# ----------------------------------------------------------------

# --------------------
# line_segment2ba_list
# --------------------
# Given
# - a plane transformation F
# - a line segment in the plane,
# compute the control points of the image Bezier arc F(line_segment).
# (Implemented formulas (1.25), (4.28)).
# Args:
# - pt:           PlaneTransformation
# - line_segment: [ControlPoint,ControlPoint] = [complex,complex]
# Returns:
# - [BCurve.BezierArc] (butting Bezier arcs)
# Note: This is exact! Possible because make_bezier_map_C_to_C
# returns a map which sends line segments to Bezier arcs.
def line_segment2ba_list(pt, line_segment):
    f = pt.point_map
    jac = pt.jacobian
    a,b = line_segment
    #
    Q01 =  rv_times_M2x2_transpose(-a+b, jac(a)) / 3.
    Q32 = -rv_times_M2x2_transpose(-a+b, jac(b)) / 3.
    #
    Q0 = f(a)
    Q1 = f(a) + Q01
    Q2 = f(b) + Q32
    Q3 = f(b)
    rv_cp4 = [Q0,Q1,Q2,Q3] # [RowVectors]
    return [BCurve.BezierArc(rv_cp4)]

# ----------------------
# simple_support2ba_list
# ----------------------
# From support:Support create a Bezier arc with the simple rule:
# Handle lengths = one third of the chord length.
# Arg:
# - support:Support
# Returns:
# - [BCurve.BezierArc]
def simple_support2ba_list(support):
    S0rv,S1rv,S2rv,S3rv = support.support_points # complex
    S01rv = -S0rv + S1rv
    S32rv = -S3rv + S2rv
    try:
        try:
            inv01rv = 1./ abs(S01rv)
        except ZeroDivisionError:
            S01rv = -S0rv + S2rv
            try:
                inv01rv = 1./ abs(S01rv)
            except ZeroDivisionError:
                S01rv = -S0rv + S3rv
                inv01rv = 1 / abs(S01rv)
        try:
            inv32rv = 1./ abs(S32rv)
        except ZeroDivisionError:
            S32rv = -S3rv + S1rv
            try:
                inv32rv = 1./ abs(S32rv)
            except ZeroDivisionError:
                S32rv = -S3rv + S0rv
                inv32rv = 1./ abs(S32rv)
    except ZeroDivisionError: # zero chord
        return [BCurve.BezierArc(cp4=[S0rv,S0rv,S3rv,S3rv])]
    chord_len3 = abs(S0rv-S3rv) / 3.
    #S01rvlen = abs(S01rv)
    #S32rvlen = abs(S32rv)
    #p01rv = S01rv*chord_len3/S01rvlen
    #p32rv = S32rv*chord_len3/S32rvlen
    p01rv = S01rv*chord_len3*inv01rv
    p32rv = S32rv*chord_len3*inv32rv
    p0rv = S0rv
    p3rv = S3rv
    p1rv = p0rv + p01rv
    p2rv = p3rv + p32rv
    cp4 = [p0rv,p1rv,p2rv,p3rv]
    return [BCurve.BezierArc(cp4=cp4)]


# ----------------
# ba2balist_simple
# ----------------
# Transform a Bezier arc ba:BCurve.BezierArc (assumed to have 4 control points)
# by pt:PlaneTransformation and make an approximate Bezier arc using a simple
# algorithm.
# Return the result as a list of butting Bezier arcs.
# Simple algorithm:
# 1. The arc is (if log_nsubarcs>0) subdivided;
# 2. the subdivision points are mapped exactly by pt;
# 3. the tangents (and directions) at the subdivision points will be exact,
#    but the lengths of the new handles in each subarc are set simply
#    to 1/3 of the chord length.
# The number of subarcs in the subdivision is 2**log_nsubarcs (>= 2**0 = 1).
# If log_nsubarcs=0, or if the arc is deemed to be a straight line segment,
# there will be no subdivision.
# The argument bezier_case:boolean tells if we are doing transformation
# constructed from a Bezier arc; in that case straight line segments are
# treated by a special exact algorithm.
# Args:
# - pt:           PlaneTransformation
# - ba:           BCurve.BezierArc
# - log_nsubarcs: integer (>=0)
# - shaping_input: dict
# - messages:     boolean (messages in Gimp's error console?)
# Returns:
# - [BCurve.BezierArc] (butting Bezier arcs)
#
def ba2balist_simple(pt, ba, shaping_input, log_nsubarcs=2, messages=False):
    log_nsubarcs = max(0,log_nsubarcs)
    cp4xy = ba.cp4 # [ControlPoint]
    if collinear(cp4xy): # Straight line segment
        if shaping_input['case'] == 'path by bezier':
            # Use the exact algorithm and return
            line_segment = [ba.cp4[0],ba.cp4[-1]]
            return line_segment2ba_list(pt, line_segment)
        if shaping_input['case'] == 'path by projective': # collineation
            # Collineation: return straight line segment
            cp4_pt = [pt.point_map(cp) for cp in cp4xy]
            return [BCurve.BezierArc(cp4=cp4_pt)]
        # Make new handles with the 1/3-rule to easify calculations.
        s0,s1,s2,s3 = cp4xy # complex
        s1 = (2*s0 + s3) / 3.
        s2 = (s0 + 2*s3) / 3.
        cp4xy = [s0,s1,s2,s3]
    # Subdivide arc in 2**log_nsubarcs subarcs:
    t_subdiv_jac = bezier_arc_map_subdivide(cp4xy, pt, n=log_nsubarcs) # [[t,F(B(t))],...]
    if t_subdiv_jac is None: # ba is of zero length
        return [BCurve.BezierArc(cp4 = [pt.point_map(cp) for cp in ba.cp4])]
    else:
        t_subdiv = t_subdiv_jac[0] # Drop the jacobians
    points = [tF[0] for tF in t_subdiv] # t-values
    # Create support for each subarc:
    skipped = 0
    if log_nsubarcs == 0:
        try:
            support_list = [map_make_support(pt, ba)] # Takes jacobian into account
        except Exception as e:
            # Two ways this can happen:
            # 1. map_make_support raises this: the arc has only one control point
            #    (possible if comes from a stroke with only one anchor)
            # 2. bezier_arc_end_directions raises this: The arc has 4 control points
            #    but they are equal (or very close).
            # In either case, treat the arc as a very tiny line segment.
            if messages:
                m = "ba2balist_simple: map_make_support raised exception: "+str(e)
                m += "\nTrying to treat the arc as straight line segment!"
                gimp_message(m)
            line_segment = [ba.cp4[0],ba.cp4[-1]]
            return line_segment2ba_list(pt, line_segment)
        except NoSupportError as e:
            # Comes from map_make_support when Support raises NoSupportError:
            # 1. one of the would-be handles is zero (or close to) (e.error=1), or
            # 2. all 4 would-be support_points are collinear (or close to) (e.error=2).
            # In either case, take the would-be support_points as control points for
            # the new arc. (Ad-hoc, in this simple algorithm!)
            if messages:
                m = "Met Exception NoSupportError: "+str(e)
                m += "\ne.error = "+str(e.error)
                m += "\nUsing ad-hoc solution!"
                gimp_message(m)
            # e.points: [complex,...] = the would-be support_points
            #cp4 = [points[0],points[0],points[3],points[3]]  ######  KOE
            #cp4 = points
            #print("simple: return 5:") ###################  HUOM kunnossa nyt?
            #print(str(BCurve.BezierArc(cp4=cp4)))
            #return [BCurve.BezierArc(cp4=cp4)]
            return [BCurve.BezierArc(cp4=e.points)] # ok now?
    else:
        points01 = [0]+points+[1]
        support_list = []
        for i in range(len(points01)-1):
            sub_cp = bezier_new_control_points(points01[i], points01[i+1], cp4xy)
            sub_ba = BCurve.BezierArc(cp4=sub_cp)
            try:
                support = map_make_support(pt, sub_ba) # Takes jacobian into account
            except Exception as e:
                # Two ways this can happen:
                # 1. map_make_support raises this: the arc has only one control point
                #    (possible if comes from a stroke with only one anchor)
                # 2. bezier_arc_end_directions raises this: The arc has 4 control points
                #    but they are equal (or very close).
                # In either case, treat the arc as a very tiny line segment.
                if messages:
                    m = "ba2balist_simple: map_make_support raised exception: "+str(e)
                    m += "\nTrying to treat the arc as straight line segment!"
                    gimp_message(m)
                line_segment = [ba.cp4[0],ba.cp4[-1]]
                return line_segment2ba_list(pt, line_segment)
            except NoSupportError as e:  ########  TO DO
                #print("ba2balist_simple got NoSupportError:")
                #print(str(e))
                # Ad-hoc as above
                if messages:
                    m = "Met Exception NoSupportError: "+str(e)
                    m += "\nUsing ad-hoc solution!"
                    gimp_message(m)
                #e.points: [[float,float],...] = the would-be support_points
                # jacobian already taken into account
                support = Support(e.points, sloppy=True) # Ad-hoc: ignore error
            support_list.append(support)
    ba_list = []
    for support in support_list: # jacobian already taken into account
        ba_list += simple_support2ba_list(support) # Create simple
    if skipped > 0 and messages:
        m = "Skipped " +str(skipped) + " very short(?) arcs."
        m += "\nThe result may contain oddities."
        gimp_message(m)
    return ba_list

# ----------------------------------------------------------------
#    Map and approximate a bezier arc: full recursive algorithm
# ----------------------------------------------------------------

# ----------------
# ba2balist_approx
# ----------------
# Transform a Bezier arc ba:BCurve.BezierArc (assumed to have 4 control points)
# by pt:PlaneTransformation using an approximating algorithm.
# Return the result as a list of butting Bezier arcs.
# The argument bezier_case:boolean tells if we are doing transformation
# constructed from a Bezier arc; in that case straight line segments are
# treated by a special exact algorithm.
# Args:
# - pt:      PlaneTransformation
# - ba:      BCurve.BezierArc
# - rec_depth:   integer (>=0)
# - shaping_input: dict
# - print_info:  boolean
# - messages:    boolean (messages in Gimp's error console?)
# Returns:
# - [BCurve.BezierArc] (butting Bezier arcs)
# - float (relative error)
# - integer (recursion depth)
#
def ba2balist_approx(pt,
                     ba,
                     shaping_input,
                     rec_depth=0,
                     print_info=False,
                     messages=False,
                     ):
    try:
        MAX_REC_DEPTH = shaping_input['max_rec_depth']
    except KeyError:
        #MAX_REC_DEPTH = 4
        MAX_REC_DEPTH = 10
        #MAX_REC_DEPTH = 12
        #MAX_REC_DEPTH = 16
    MAX_ALLOWED_ERROR = 0.002 # Sufficient?
    #MAX_ALLOWED_ERROR = 0.1 # ok?
    #MAX_ALLOWED_ERROR = 0.5 # ok?
    #MAX_ALLOWED_ERROR = 0.0005 # Unnecessary?
    LOG_SPOINTS = 5
    SPOINTS = 2**LOG_SPOINTS  # Number of subdivision points
    NPOINTS = 4 # Number of points used to find the approximations = SPOINTS/NPOINTS
    EPOINTS = 2 # Number of points used in error calculations = SPOINTS/EPOINTS
    def get_size(cp, targets): # cp, targets: [complex,...]
        def distance2(x,y):
            return ((x-y)*(x-y).conjugate()).real
        p0,p1,p2,p3 = cp # [float,float]
        middle = (p0+p3)/2 # Middle point of chord
        return sum([distance2(middle, target) for target in targets])
    def get_rel_error(ba_list_try, yard_stick):
        total_rel_error = 0
        for ba_try in ba_list_try:
            # Error target points (for error calculations):
            cp4xy = ba_try.cp4 # [[float,float],...]
            t_subdiv_jac = bezier_arc_map_subdivide(cp4xy, pt, n=LOG_SPOINTS) # [[t,F(B(t))],...]
            if t_subdiv_jac is None:
                return 0.
            else:
                t_subdiv = t_subdiv_jac[0] # Drop the jacobians
            t_etargets = t_subdiv[EPOINTS-1: -EPOINTS+1: EPOINTS] # [[t,RowVector],...]
            etargets = [tt[1] for tt in t_etargets]
            #
            bf = BezierFunction(cp4xy)
            abs_error = sum_dist2_arc_points(bf, etargets,sloppy=True)
            rel_error = sqrt(abs_error / yard_stick)
            total_rel_error = max(total_rel_error, rel_error)
        return total_rel_error
    # directions_ok: Check that the arc does not turn more than 90 degrees.
    # To that end, compute dot products of the direction vector at the start
    # with the following vectors:
    # - the direction vector at the end;
    # - the direction vectors at the subdivision points.
    # If there is a negative dot product, return False, otherwise True.
    def directions_ok(ba, t_jac):
        direction_vectors = bezier_arc_end_directions(ba)
        if direction_vectors is None:
            raise Exception("directions_ok: Arc has 4 equal control points.")
        head_dir, tail_dir = direction_vectors
        tail_dir = -tail_dir # To point in the direction of the derivative!
        pt_head_dir = rv_times_M2x2_transpose(head_dir, pt.jacobian(ba.cp4[0]))
        pt_tail_dir = rv_times_M2x2_transpose(tail_dir, pt.jacobian(ba.cp4[3]))
        if vdot(pt_head_dir, pt_tail_dir) < 0:
            return False
        for t, jac in t_jac:
            B_dot = bezier_dot_rv(t, ba.cp4)
            pt_B_dot = rv_times_M2x2_transpose(B_dot, jac)
            vd = vdot(pt_head_dir, pt_B_dot)
            if vd < 0:
                #rv = bezier_rv(t, ba.cp4)
                #pt_rv = pt.point_map(rv)
                return False
        return True
    # handle_MAX_REC_DEPTH:
    # Reaching MAX_REC_DEPTH but having no acceptable approximation.
    def handle_MAX_REC_DEPTH(pt, ba, targets, message):
        m = message
        m += "  Reached MAX_REC_DEPTH: no recursion"
        m += "\n  Using the simple algorithm."
        if print_info:
            print(m)
        bas = ba2balist_simple(pt, ba, shaping_input, log_nsubarcs=2) # Do 1 splitting
        yard_stick = get_size(ba.cp4, targets)
        rel_err = get_rel_error(bas, yard_stick)
        if messages:
            gm = "The plugin met a failure."
            gm += "\nCannot use recursion (at maximum allowed depth already)."
            gm += "\nResorting to a simple algorithm."
            gm += "\nThe result is probably not too accurate."
            gimp_message(gm)
        return bas, rel_err
    if rec_depth == 0:
        if print_info:
            print("\nMAX_ALLOWED_ERROR = "+str(MAX_ALLOWED_ERROR))
            print("MAX_REC_DEPTH = "+str(MAX_REC_DEPTH))
            print()
    error_message = ''     # Used in case of error
    cp4xy = ba.cp4 # [complex,...]
    if shaping_input['case'] == 'path by bezier':
        # If straight line segment, use the exact algorithm:
        if collinear(cp4xy): # Straight line segment
            line_segment = [ba.cp4[0],ba.cp4[-1]]
            bas = line_segment2ba_list(pt, line_segment)
            rel_err = 0
            return bas, rel_err, rec_depth 
    if shaping_input['case'] == 'path by projective': # collineation
        # If straight line segment, return straight line segment
        if collinear(cp4xy):
            #print("collinear")
            cp4_pt = [pt.point_map(cp) for cp in cp4xy]
            return [BCurve.BezierArc(cp4=cp4_pt)], 0., rec_depth
    # Create subdivision:
    t_subdiv_jac = bezier_arc_map_subdivide(cp4xy, pt, n=LOG_SPOINTS)
    if t_subdiv_jac is None:
        pt_ba = BCurve.BezierArc(cp4 = [pt.point_map(cp) for cp in ba.cp4])
        return [pt_ba], 0., rec_depth
    else:
        t_subdiv, t_jac = t_subdiv_jac # [[t,F(B(t))],...], [t,J(B(t))],...]
    t_targets  = t_subdiv[NPOINTS-1: -NPOINTS+1: NPOINTS] # [[t,RowVector],...]
    t_etargets = t_subdiv[EPOINTS-1: -EPOINTS+1: EPOINTS] # [[t,RowVector],...]
    targets = [tt[1] for tt in t_targets]   # Targets for approximation
    etargets = [tt[1] for tt in t_etargets] # Targets for error calculation
    # Potentially initial subdivision:
    # Check directions: if the curve turns > 90 degrees,
    # by-pass all tries of approximation.
    # Instead, go to the recursion part (split in two and so on)
    rel_err = 'unknown'
    go_to_recursion = not directions_ok(ba, t_jac)
    if go_to_recursion:
        rel_err = 'unknown'
        if print_info:
            pt_start = pt.point_map(ba.cp4[0])
            pt_end = pt.point_map(ba.cp4[-1])
            print()
            print("Start: "+str(pt_start))
            print("End:   "+str(pt_end))
            print("Directions not ok.")
        pass
    else: # Try to do approximation
        # Target points:
        if print_info:
            t_rvB01 = [(t, bezier_rv(t,cp4xy)) for t in [0,1]] # [(float,RowVector=complex)]
            t_targets01 = [(trvb[0],pt.point_map(trvb[1])) for trvb in t_rvB01]
            targets01 = [tt[1] for tt in t_targets01]
            print()
            print("Start: "+str(targets01[0]))
            print("End:   "+str(targets01[1]))
            print("targets:")
            for ttt in targets: print(str(ttt))
        try:
            pt_support = map_make_support(pt, ba)
            ok_support = True
            if print_info:
                if ok_support:
                    print("Support ok.")
                    print("    Support points:")
                    for sp in pt_support.support_points:  print("        "+str(sp))
                else:
                    print("No support.")
        except Exception as e:
            ok_support = False
            if print_info:
                print("No support. Exception from map_make_support:")
                print(str(e))
        except NoSupportError as nse:
            if nse.error == 2:
                # Collinear would-be support points: return straight line segment
                #print("nse.error == 2")
                xy = nse.points
                xy0 = xy[0]
                xy1 = xy[0] # Take zero length handle
                xy2 = xy[3] # Take zero length handle
                xy3 = xy[3]
                cp4_straight = [xy0,xy1,xy2,xy3]
                ba_straight = [BCurve.BezierArc(cp4_straight)]
                if print_info:
                    print("No support, arc straight.")
                    print("    would-be support points:")
                    for sp in nse.points:  print("        "+str(sp))
                    print("    Returning straight line segment, control points:")
                    for p in cp4_straight: print("        "+str(p))
                return ba_straight, 0., rec_depth
            else: # Go to recursion
                rel_err = 'None'
                ok_support = False
        if not ok_support: # No support
            error_message = 'No support'
        else: # Support ok
            try:
                cpxy, abs_err, rel_err = arc_approx_points(pt_support,
                                  targets,
                                  respect_directions=True,
                                  sloppy=True,
                                  error_targets=etargets
                                  )
                #failure = False
                cp4_result = cpxy
                ba_result = BCurve.BezierArc(cp4 = cp4_result)
                if rel_err < MAX_ALLOWED_ERROR: # OK result: return
                    ba_list = [ba_result]
                    if print_info:
                        print("  rec_depth = "+str(rec_depth)+", rel_err = "+str(rel_err)+"  OK")
                        print("    Returning Bezier arc with control points:")
                        for cp in ba_result.cp4:
                            print("        "+str(cp))
                    return ba_list, rel_err, rec_depth
                elif rec_depth >= MAX_REC_DEPTH: # No failure: got ba_result; use it!
                    ba_list = [ba_result]
                    if print_info:
                        print(">>>>                  Reached MAX_REC_DEPTH: no recursion")
                        print(">>>>  rec_depth = "+str(rec_depth)+", rel_err = "+str(rel_err)+"  QUIT")
                        #draw_polyline(image, targets, name='targets', closed=False)
                        #start, end = ba_result.cp4[0], ba_result.cp4[-1]
                        #draw_polyline(image, [start,end], name='chord', closed=False)
                    return ba_list, rel_err, rec_depth
                else: # Go to recursion
                    error_message = 'rel_err too large'
            except Exception as e: # Failure: arc_approx_points?
                # Go to recursion
                #failure = True
                m = "ba2balist_approx: Failure:"
                m += "\n"+str(e)
                error_message = m
    # Recursion:
    if print_info:
        print("  rec_depth = "+str(rec_depth)+", rel_err = "+str(rel_err)+"  GOING TO RECURSION")
    if rec_depth > MAX_REC_DEPTH:
        ba_list, rel_err = handle_MAX_REC_DEPTH(pt, ba, targets, error_message)
        return ba_list, rel_err, rec_depth
    # Split the Bezier arc in two:
    # Idea: split_t will be that parameter value t for which the target point
    # is the most distant from the chord.
    rvp0, rvp3 = pt.point_map(cp4xy[0]), pt.point_map(cp4xy[3]) # Chord
    far_t_target = max(t_targets,
                       key=(lambda tt: abs(dist_from_line(rvp0, rvp3, tt[1]))))
    split_t = far_t_target[0]
    if print_info:
        split_B = bezier_rv(split_t, cp4xy)
        split_pt = pt.point_map(split_B)
        print("    RECURSION")
        print("    Splitting at t = "+str(split_t)+ ", B(t) = "+str(split_B))
        print("    Adding point "+str(split_pt))
    #
    cp4_left  = bezier_new_control_points(0, split_t,  cp4xy)
    cp4_right = bezier_new_control_points(split_t, 1., cp4xy)
    ba_left = BCurve.BezierArc(cp4=cp4_left)
    ba_right = BCurve.BezierArc(cp4=cp4_right)
    # Recursive calls:
    left  = ba2balist_approx(pt,
                             ba_left,
                             shaping_input,
                             rec_depth=rec_depth+1,
                             print_info=print_info,
                             messages=messages
                             )
    right = ba2balist_approx(pt,
                             ba_right,
                             shaping_input,
                             rec_depth=rec_depth+1,
                             print_info=print_info,
                             messages=messages
                             )
    ba_list_left, rel_err_left, rec_depth_left = left
    ba_list_right, rel_err_right, rec_depth_right = right
    #
    ba_list = ba_list_left + ba_list_right
    rel_err = max(rel_err_left, rel_err_right)
    rec_depth = max(rec_depth_left, rec_depth_right)
    return ba_list, rel_err, rec_depth


# ------------------
# shape_bezier_curve
# ------------------
# Given a Bezier curve (one stroke), compute its image (approximately as
# a new Bezier curve) using the appropriate transformation specified in
# 'shaping_factors'.
#
# Args:
# - bezier_curve:       BCurve.BezierCurve
# - shaping_input: dict
# - 
# Returns:
# - BCurve.BezierCurve
def shape_bezier_curve(
               bezier_curve,
               shaping_input,
               ):
    def arc_measure(ba): # ba:BCurve.BezierArc (assume four control points)
        from math import sqrt
        p0,p1,p2,p3 = ba.cp4 # RowVector = complex
        diff = -p0+p1, -p1+p2, -p2+p3, -p3+p0
        # Vaihdettu kompleksilukuihin; hiukan toinen logiikka:
        #return sqrt(sum([vdot(d,d) for d in diff]))
        return sum([abs(d) for d in diff])
    try: # Set SHORT_ARC_SIZE: Shorter arcs are done by the simple algorithm.
        typical_arc_size = shaping_input['typical arc size']
        SHORT_ARC_SIZE = typical_arc_size / 50
        #print("Setting SHORT_ARC_SIZE to "+str(SHORT_ARC_SIZE))
    except KeyError:
        SHORT_ARC_SIZE = 10
    # Main work: Different cases treated differently:
    bezier_case_linefigure = False
    ba_list_ori = bezier_curve.bezier_arcs # [BCurve.BezierArc]
    if shaping_input['case'] == 'path by bezier':           # Bezier
        if shaping_input['linefigure']:
            ba_list_ori = handles2zero(ba_list_ori)
            bezier_case_linefigure = True
    pt = shaping_input['plane transformation']
    if len(ba_list_ori[0].cp4) == 1:
        # The stroke has just one anchor.
        # Just map that anchor by pt.point_map
        rv = ba_list_ori[0].cp4[0] # RowVector
        mapped_rv = pt.point_map(rv)
        ba = BCurve.BezierArc([mapped_rv])
        ba_list = [ba] # One Bezier arc where cp4=[p0] (one control point)
    else:
        if bezier_curve.closed:
            # Add the closing segment as one Bezier arc,
            # so that it will be transformed properly.
            start = ba_list_ori[0].cp4[0]
            end = ba_list_ori[-1].cp4[-1]
            head = bezier_curve.head_handle
            tail = bezier_curve.tail_handle
            ba_list_ori.append(BCurve.BezierArc([end, tail, head, start]))
        ba_list = []
        for ba_ori in ba_list_ori: # ba_ori:BCurve.BezierArc
            arc_size = arc_measure(ba_ori)
            # Main calls:
            if bezier_case_linefigure: # Handled exactly
                #print("shape_bezier_curve: main call: 1")
                line_segment = [ba_ori.cp4[0], ba_ori.cp4[-1]]
                bas = line_segment2ba_list(pt, line_segment)
            elif arc_size < SHORT_ARC_SIZE:
                #print("shape_bezier_curve: main call: 2")
                #print("arc_size = "+str(arc_size))
                #print("SHORT_ARC_SIZE = "+str(SHORT_ARC_SIZE))
                bas = ba2balist_simple(          # Simple algorithm
                        pt,
                        ba_ori,
                        shaping_input=shaping_input,
                        log_nsubarcs= 0,
                        )
            elif shaping_input['use_simple'] >= 1: # Simple algorithm.
                #print("shape_bezier_curve: main call: 3")
                bas = ba2balist_simple(
                        pt,
                        ba_ori,
                        shaping_input=shaping_input,
                        log_nsubarcs= -1 + shaping_input['use_simple'],
                        )
            else:                                 # Regular approx algorithm.
                #print("shape_bezier_curve: main call: 4")
                bas,_,_ = ba2balist_approx(
                        pt,
                        ba_ori,
                        shaping_input=shaping_input
                        )
            ba_list += bas
    # Map head_handle and tail_handle:
    if not bezier_curve.closed:
        # Made a global pt and the curve not closed.
        # Map the old head_handle and tail_handle.
        ## (Is this a correct idea? No!                       TO DO
        ## If not closed curve, should add an imagined closing arc, apply
        ## ba2balist_approx to it and then:
        ## -If the approximation is just one arc, take new head and tail from there.
        ## -If there are several arcs, then we have a problem! No satisfactory way?
        ##  Perhaps use ba2balist_simple with log_nsubarcs=0?)
        old_ba_list = bezier_curve.bezier_arcs
        old_start_rv = old_ba_list[0].cp4[0] # Old start: RowVector
        old_end_rv = old_ba_list[-1].cp4[-1] # Old end:   RowVector
        old_head_rv = bezier_curve.head_handle # Old head absolute: RowVector
        old_tail_rv = bezier_curve.tail_handle # Old tail absolute: RowVector
        old_head_rel_rv = -old_start_rv + old_head_rv # Old head relative: RowVector
        old_tail_rel_rv = -old_end_rv + old_tail_rv   # Old tail relative: RowVector
        # Map relative vectors:
        new_head_rel_rv = rv_times_M2x2_transpose(old_head_rel_rv, pt.jacobian(old_start_rv))
        new_tail_rel_rv = rv_times_M2x2_transpose(old_tail_rel_rv, pt.jacobian(old_end_rv))
        # Make absolute:
        new_start_rv = ba_list[0].cp4[0] # New start: RowVector
        new_end_rv = ba_list[-1].cp4[-1] # New end:   RowVector
        new_head_rv = new_start_rv + new_head_rel_rv # New head absolute: RowVector
        new_tail_rv = new_end_rv + new_tail_rel_rv   # New tail absolute: RowVector
        # To ControlPoints:
        new_head = new_head_rv # New head absolute: ControlPoint
        new_tail = new_tail_rv # New tail absolute: ControlPoint
    else:    
        # Made a global pt and the curve is closed.
        # The original head and tail handles were used already when
        # adding the closing line segment as one Bezier arc.
        # Now just tell Gimp that the curve is closed, hoping it
        # will close it properly without making a tiny new edge.
        new_head = None
        new_tail = None
    return BCurve.BezierCurve(
                     bezier_arcs = ba_list,
                     head_handle = new_head,
                     tail_handle = new_tail,
                     closed      = bezier_curve.closed,
                     curve_name = 'mapped '+bezier_curve.curve_name,
                     )

# ----------
# shape_path
# ----------
# Given a path (Gimp's vectors object), compute its image (approximately as
# a new path) in the transformation constructed from info in 'shaping_input'.
#
# Args:
# - image
# - path_vectors:  gimp.Vectors
# - shaping_input: dict
# Returns:
# - gimp.Vectors
def shape_path(
               image,
               path_vectors,
               shaping_input,
               ):
    gv_path = vectors_object2gv(path_vectors) # BCurve.GimpVectors
    # For each stroke (polyline) in the path, construct the shaped stroke:
    list_shaped_gs = []
    for gs in gv_path.stroke_list: # gs: BCurve.GimpStroke
        bc = gs.gs2bc()      # BCurve.BezierCurve
        new_bc = bezier_arc_protect_cusps(bc)
        shaped_bc = shape_bezier_curve(                        # Main call
                   new_bc,
                   shaping_input,
                   )
        shaped_bc.closed = gs.closed # Needed?
        shaped_gs = shaped_bc.bc2gs() # BCurve.GimpStroke
        list_shaped_gs.append(shaped_gs)
    name = path_vectors.name
    if shaping_input['case'] in ('path by bezier', 'path by bezier4'):
        name += ' shaped by '+shaping_input['shaper_name']
    mapped_gv = BCurve.GimpVectors(stroke_list=list_shaped_gs,
                                   name=name
                                   )
    mapped_vectors = mapped_gv.gv2vectors_object(image, mapped_gv.name)
    return mapped_vectors


#---------------------------------------------------------
#                   Gimp draw
#---------------------------------------------------------

# ------------------------
# gimp_draw_vectors_object
# ------------------------
# The Gimp version of the routine 'drawing_routine_for_bezier':
# Insert into Gimp the input vectors object, and make it visible if
# visible=True.
#
# Args:
# - image: Gimp's image (obtained usually from GUI);
# - vectors_object: a vectors object in Gimp;
# - visible: Boolean.
#
# Returns the vectors object in Gimp, and makes it visible if visible=True.
def gimp_draw_vectors_object(image, vectors_object, visible=True):
    # Insert a Gimp vectors object in Gimp.
    # Adapted from http://registry.gimp.org/files/lqr_wpset-0.14.py.txt
    # by Mike Kazantsev.
    def gimp_insert_vectors(image, vectors_object, position=0):
        if gimp.version >= (2, 8, 0):
            # 2.7.0 has gimp-image-insert-vectors, but fails with
            # parent=None
            parent = None
            return pdb.gimp_image_insert_vectors(
                     image, vectors_object, parent, position)
        else:
            return pdb.gimp_image_add_vectors(
                     image, vectors_object, position)# Deprecated!
    gimp_insert_vectors(image, vectors_object)
    vectors_object.visible = visible
    return vectors_object

# polyline = [complex]
def draw_polyline(image, polyline, name='polyline', closed=False):
    polyline_object = pdb.gimp_vectors_new(image, name)
    points = []
    for z in polyline:
        xy = [z.real,z.imag]
        points += 3*xy
    stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                        polyline_object, 0, len(points), points, closed)
    gimp_draw_vectors_object(image, polyline_object, visible=True)
    return polyline



#==============================================================
#             class PlaneTransformation       
#==============================================================
# Classes: Matrix2x2, PlaneTransformation

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

#---------------------------------------------------------
#       classes: Matrix2x2, PlaneTransformation
#---------------------------------------------------------

class Matrix2x2(object):
    """2x2-matrix
    Initialize m = Matrix2x2(1,2,3,4)
    Get elements as m[i,j]
    Set elements as m[i,j]=value
    Note: Works with real or complex matrices.
    """
    def __init__(self,a00,a01,a10,a11):
        #self._elements = [float(a00),float(a01),float(a10),float(a11)]
        self._elements = [a00,a01,a10,a11]
    def __getitem__(self,indexes): # indexes = [i,j]
        i,j = indexes
        if not (0<=i<=1 and 0<=j<=1):
            raise Exception("Matrix2x2 get: Index out of range")
        return self._elements[2*i+j]
    def __setitem__(self, indexes, value):
        i,j = indexes
        if not (0<=i<=1 and 0<=j<=1):
            raise Exception("Matrix2x2 set: Index out of range")
        self._elements[2*i+j] = value
    def transpose(self):
        a,b,c,d = self._elements
        return Matrix2x2(a,c,b,d)
    def scale(self, coeff):
        a,b,c,d = [x*coeff for x in self._elements]
        return Matrix2x2(a,b,c,d)
    def inverse(self):
        a,b,c,d = self._elements
        try:
            idet = 1./(a*d-b*c)
        except ZeroDivisionError:
            raise Exception("Matrix2x2 inverse: Zero determinant")
        A,B,C,D = [x*idet for x in self._elements]
        return Matrix2x2(D,-B,-C,A)
    def __add__(self,other):
        a,b,c,d = self._elements
        A,B,C,D = other._elements
        return Matrix2x2(a+A,b+B,c+C,d+D)
    def __mul__(self,other):
        return Matrix2x2(self[0,0]*other[0,0] + self[0,1]*other[1,0],
                         self[0,0]*other[0,1] + self[0,1]*other[1,1],
                         self[1,0]*other[0,0] + self[1,1]*other[1,0],
                         self[1,0]*other[0,1] + self[1,1]*other[1,1])
    def __str__(self):
        def str_float(x): return '{:1.3f}'.format(x)
        a00 = str_float(self[0,0])
        a01 = str_float(self[0,1])
        a10 = str_float(self[1,0])
        a11 = str_float(self[1,1])
        return '['+a00+','+a01+']' +'\n['+a10+','+a11+']\n'


class PlaneTransformation(object):
    """Plane map  RowVector -> RowVector
    with jacobian RowVector -> Matrix2x2
    Attributes:
    - type:      string
    - point_map: callable, maps RowVector -> RowVector
    - jacobian:  callable, maps RowVector -> Matrix2x2
    """
    def __init__(self, type, point_map, jacobian=None):
        self.type = type
        self.point_map = point_map
        self.jacobian = jacobian


# -------------
# rv_times_M2x2
# -------------
# RowVector times Matrix2x2.
# Args:
# - rv:     RowVector (=complex)
# - matrix: Matrix2x2
# Returns:
# - RowVector (=complex)
def rv_times_M2x2(rv,matrix):
    return RowVector(rv.real*matrix[0,0] + rv.imag*matrix[1,0],
                     rv.real*matrix[0,1] + rv.imag*matrix[1,1])

# -----------------------
# rv_times_M2x2_transpose
# -----------------------
# RowVector times Matrix2x2 transposed.
# Args:
# - rv:     RowVector (=complex)
# - matrix: Matrix2x2
# Returns:
# - RowVector (=complex)
def rv_times_M2x2_transpose(rv,matrix):
    return RowVector(rv.real*matrix[0,0] + rv.imag*matrix[0,1],
                     rv.real*matrix[1,0] + rv.imag*matrix[1,1])


#==============================================================
#===================  affine plugin proper  ===================
#==============================================================
# Transform a path by a general affine map
# (special cases: direct similitude, translation).

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

#---------------------------------------------------------
#                 Make affine transformation
#---------------------------------------------------------

# Affine map sending three given points u0,u1,u2 to another
# three given points v0,v1,v2. 
# Args:
# - base: [RowVector,RowVector,RowVector] (at least 3 points)
# - target: [RowVector,RowVector,RowVector] (at least 3 points)
# Returns:
# - PlaneTransformation
def make_affine_map(base, target):
    u0,u1,u2 = base[:3] # The first three anchors: RowVector
    v0,v1,v2 = target[:3] # The first three anchors: RowVector
    if collinear([u0,u1,u2]):
        raise Exception("make_affine_map: Anchors of Base are collinear.")
    u10 = u1-u0 # RowVector (=complex)
    u20 = u2-u0 # RowVector (=complex)
    v10 = v1-v0 # RowVector (=complex)
    v20 = v2-v0 # RowVector (=complex)
    A = Matrix2x2(u10.real, u10.imag, 
                  u20.real, u20.imag)
    B = Matrix2x2(v10.real, v10.imag, 
                  v20.real, v20.imag)
    try:
        AB = A.inverse() * B
    except Exception as e:
        raise Exception("make_affine_map: "+str(e))
    def pm(x):
        return v0 + rv_times_M2x2((x-u0),AB)
    def jac(x):
        return AB.transpose()
    return PlaneTransformation('affine', pm, jac)


#---------------------------------------------------------
#           Affine transformation:  main procedure
#---------------------------------------------------------

# ----------------
# transform_stroke
# ----------------
#  Args:
# - pt: PlaneTransformation
# - gs: BCurve.GimpStroke
# Returns:
# - BCurve.GimpStroke
def transform_stroke(point_map, gs):
    def transform_cp_list(cp_list):
        return [point_map(cp) for cp in cp_list]
    return BCurve.GimpStroke(cp_list = transform_cp_list(gs.cp_list),
                             closed = gs.closed,
                             stroke_name = gs.stroke_name + ' transformed'
                             )

# --------------------------
# affine_transformation_main
# --------------------------
# Given base triangle and target triangle, construct an affine map
# sending the base onto the target, and map the input vectors_object.
# However, if not all u0,u1,u2 given:
# - If only u0 given, the map is translation, sending u0 to v0;
# - If only u0,u1 given, the map is a direct similitude,
#   sending u0 to v0 and u1 to v1.
# If more than three u's given, the rest are ignored.
# Args:
# - vectors_object: gimp.Vectors;
# - base:           gimp.Vectors (need at least three anchors in first stroke);
# - target:         gimp.Vectors (need at least three anchors in first stroke).
# - reversed_base:  boolean
# Returns:
# - gimp.Vectors
def affine_transformation_main(image,
                               vectors_object, # Path to be transformed
                               base_vectors,
                               target_vectors,
                               reversed_base=False
                               ):
    def get_all_anchors(vectors_object): # Returns [[x,y],[x,y],...]
        y=[]
        for str in vectors_object.strokes:
            points = str.points[0]
            n = len(points)
            for i in range(0,n,6):
                y.append(points[i+2:i+4])
            break # Take only the zeroth stroke
        return y
    if len(base_vectors.strokes) == 0:
        raise Exception("Empty base: no strokes")
    if len(target_vectors.strokes) == 0:
        raise Exception("Empty target: no strokes")
    xy_base = get_all_anchors(base_vectors)           # [[x,y],[x,y],...]
    xy_target = get_all_anchors(target_vectors)       # [[x,y],[x,y],...]
    rv_base = [RowVector(x,y) for x,y in xy_base]     # [RowVector,...]
    rv_target = [RowVector(x,y) for x,y in xy_target] # [RowVector,...]
    #
    pdb.gimp_image_undo_group_start(image)
    vectors = map_path_by_affine(image,          # Main call: Map the path
                       vectors_object,
                       rv_base,
                       rv_target,
                       reversed_base=False
                       )
    gimp_draw_vectors_object(image, vectors, visible=True) # Draw
    pdb.gimp_image_undo_group_end(image)
    return vectors


# ------------------
# map_path_by_affine
# ------------------
# Do the main work of affine_transformation_main except for drawing.
# Args:
# - vectors_object: gimp.Vectors;
# - rv_base:        [RowVector] (=[complex])
# - rv_target:      [RowVector] (=[complex])
# - reversed_base:  boolean
# Returns:
# - gimp.Vectors
def map_path_by_affine(image,
                       vectors_object, # Path to be transformed
                       rv_base,
                       rv_target,
                       reversed_base=False
                       ):
    gv = vectors_object2gv(vectors_object)          # GimpVectors
    if len(rv_target) < len(rv_base):
        if len(rv_target) == 0:
            raise Exception("Empty target")
        else:
            raise Exception("The target must have "+str(len(rv_base))+" anchors since base has.")
    if reversed_base:
        rv_base = rv_base[:3][::-1] # The first three (or less) elements reversed
    #
    if len(rv_base) >= 3:
        pt = make_affine_map(rv_base, rv_target)
    elif len(rv_base) == 2:
        rv1, rv2 = rv_base
        rw1, rw2 = rv_target[0], rv_target[1]
        pt = make_direct_similitude(rv1, rv2, rw1, rw2)
    elif len(rv_base) == 1:
        rv = rv_base[0]
        rw = rv_target[0]
        pt = make_translation(rw-rv)
    #
    new_stroke_list = [transform_stroke(pt.point_map, gs)
                                              for gs in gv.stroke_list]
    new_gv = BCurve.GimpVectors(stroke_list = new_stroke_list,
                                name = vectors_object.name + ' transformed'
                                )
    pdb.gimp_image_undo_group_start(image)
    vectors = new_gv.gv2vectors_object(image, name=new_gv.name)
    pdb.gimp_image_undo_group_end(image)
    return vectors


#==============================================================
#=============  Bezier path plugin proper  ================
#==============================================================
# Transform a path by bezier arc (Shaper)

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

#---------------------------------------------------------
#        Make PlaneTransformation from a Bezier arc
#---------------------------------------------------------

# ----------------------
# make_bezier_map_C_to_C
# ----------------------
# A Bezier curve is normally given by the map B(t):[0,1]->R2, defined by
# control points P0,P1,P2,P3 and Bernstein polynomials b0,b1,b2,b3 of
# degree 3:
#
#     B(t) = b0(t)P0 + b1(t)P1 + b2(t)P2 + b3(t)P3    (t in[0,1]).
#
# This extends to a complex function B_C(z):C->C in the obvious way:
#
#     B_C(z) = b0(z)p0 + b1(z)p1 + b2(z)p2 + b3(z)p3    (z in C)
#
# where p0,p1,p2,p3 are complex numbers (the vectors P0,P1,P2,P3 when R2 is
# identified with the complex plane by [x,y] -> x+iy)
# and b0,b1,b2,b3 are the same Bernstein polynomials as before
# but now viewed C->C.
# When C is again viewed as R2, we obtain a new map B_R2: R2->R2.
# This map is built by make_bezier_map_C_to_C from the control points
# P0,P1,P2,P3, with the TWIST that the map is continued by a direct similitude
# causing  'base' to be mapped to 'target'.
#
# Args:
# - cp4:     [P0,P1,P2,P3]: [ControlPoint]
# - base:    [A,B]: [ControlPoint,ControlPoint]
# - target:  [C,D]: [ControlPoint,ControlPoint]
# - tweak_a: float
# - tweak_b: float
# - Gimp:    boolean: Set True if the y coordinate runs downwards
# Returns:
# - PlaneTransformation (point_map maps RowVector to RowVector).
def make_bezier_map_C_to_C(cp4, # Shaper
                             base,
                             target,
                             tweak_a=0.,
                             tweak_b=1.,
                             Gimp=True
                             ):
    raw_bezier = make_raw_bezier_map(cp4, Gimp=Gimp)
    return combine_AB_pt_PQ(raw_bezier,
                            base,
                            target,
                            tweak_a,
                            tweak_b,
                            type = 'bezier'
                            )

# -------------------
# make_raw_bezier_map
# -------------------
# Given cp4:[ControlPoint], control points of a Bezier arc B,
# make a PlaneTransformation such that it maps the line segment
# ((0,0), (1,0)) to a Bezier arc similar to B, such that the arc
# starts at (0,0) and ends at (1,0).
# Args:
# - cp4:    [P0,P1,P2,P3]: [ControlPoint]
# - Gimp:   boolean: Set True if the y coordinate runs downwards
# Returns:
# - PlaneTransformation (point_map maps RowVector to RowVector).
def make_raw_bezier_map(cp4, Gimp=True):
    bernstein3 = [lambda t: (1-t)**3, # Bernstein polynomials of degree 3
                  lambda t: 3*(1-t)*(1-t)*t,
                  lambda t: 3*(1-t)*t*t,
                  lambda t: t**3
                  ]
    bernstein3_dot = [lambda t: -3*(1-t)*(1-t), # Derivatives
                      lambda t: 3*(1-t)*(1-3*t),
                      lambda t: 3*t*(2-3*t),
                      lambda t: 3*t*t
                      ]
    zp = cp4
    def raw_bezier_pm(rv):
        z = rv
        Bz = sum([bernstein3[k](z)*zp[k] for k in range(4)])
        return Bz
    def raw_bezier_jac(rv): # Assume all normalized
        z = rv
        Bz_dot = sum([bernstein3_dot[k](z)*zp[k] for k in range(4)])
        ux,uy = Bz_dot.real, Bz_dot.imag
        if Gimp:
            return Matrix2x2(ux, -uy, uy, ux)
            # Possible explanation why this seems to work right in Gimp:
            # If the y coordinate runs downwards, derivating with respect
            # to y introduces an extra minus sign.
        else:
            return Matrix2x2(ux, uy, -uy, ux) # Theoretically correct
    return PlaneTransformation('raw bezier',
                                point_map = raw_bezier_pm,
                                jacobian = raw_bezier_jac)


#-----------------------------------------------------------
#   Transform a path by a Bezier arc:  main procedure  
#-----------------------------------------------------------

# ----------------
# bezier_path_main
# ----------------
# The procedure_function: called directly from GUI.
# Read inputs and call map_path_by_bezier.
#
# Args:
# - image
# - path:                gimp.Vectors (path to be mapped)
# - treat_as_linefigure: boolean      (ignore all curviness in path)
# - base:                gimp.Vectors (base line segment)
# - target:              gimp.Vectors (line segment, image of base)
# - shaper:              gimp.Vectors
# - use_reversed_base:   boolean
# - use_reversed_shaper: boolean
# - shape_string:        string       (shaping strength(s) as string)
# - tweak_a:             float
# - tweak_b:             float
# - use_simple:          integer (>=0) 
# Returns:
# - gimp.Vectors
# 
# Note: The string argument 'shape_string' shoud evaluate (eval) to one of:
# - int or float;
# - tuple: either (int) or (float);
# - list:  either [int] or [float].
def bezier_path_main(image,
                     path,                # gimp.Vectors
                     treat_as_linefigure, # boolean
                     base,                # gimp.Vectors
                     target,              # gimp.Vectors
                     shaper,              # gimp.Vectors
                     use_reversed_base,   # boolean
                     use_reversed_shaper, # boolean
                     shape_string,        # string
                     tweak_a,             # float
                     tweak_b,             # float
                     use_simple           # integer
                     ):
    shape_list = string_to_list(shape_string)
    merged_vectors = map_path_by_bezier(
               image,
               shaper_vectors      = shaper,
               path_vectors        = path,
               treat_as_linefigure = treat_as_linefigure,
               base_vectors        = base,
               target_vectors      = target,
               reverse_base        = use_reversed_base,
               reverse_shaper      = use_reversed_shaper,
               shape_strength_list = shape_list,
               tweak_a             = tweak_a,
               tweak_b             = tweak_b,
               use_simple          = use_simple,
               merge_down          = False # No option of merging down in this plugin
               )
    return merged_vectors

# ------------------
# map_path_by_bezier
# ------------------
# The true main procedure.
# Given
# - a Gimp's vectors object (only the first arc of the first stroke is used), and
# - a path (as Gimp's vectors object)
# - two line segments, base and target (as Gimp's vectors objects),
# shape the path by the map constructed from the Bezier arc,
# sending base to target.
# If treat_as_linefigure=True, the input path is treated as consisting of
# straight line segments: only anchors are taken into account and all curviness
# is ignored.
# If use_simple>0, use the simple algorithm throughout: inaccurate but quick;
# the higher the value, the more accurate the outcome but with more
# numerous control points.
# If merge_down=True, the created vectors objects are merged down before drawing
# (relevant if 'shape_strength_list' is a list of several items).
# Draw the result.
# 
# Args:
# - image
# - shaper_vectors:      gimp.Vectors
# - path_vectors:        gimp.Vectors
# - treat_as_linefigure: boolean
# - base_vectors:        gimp.Vectors
# - target_vectors:      gimp.Vectors
# - reverse_base:        boolean
# - reverse_shaper:      boolean
# - shape_strength:      [float]
# - tweak_a:             float
# - tweak_b:             float
# - use_simple:          integer
# - merge_down:          boolean
# Returns:
# - gimp.Vectors
def map_path_by_bezier(
               image,
               shaper_vectors,
               path_vectors,
               treat_as_linefigure,
               base_vectors,
               target_vectors,
               reverse_base=False,
               reverse_shaper=False,
               shape_strength_list=[1.],
               tweak_a=0.,
               tweak_b=1.,
               use_simple=0,
               merge_down=False
               ):
    gv_shaper = vectors_object2gv(shaper_vectors) # BCurve.GimpVectors
    try:
        gs_shaper = gv_shaper.stroke_list[0]      # BCurve.GimpStroke: only stroke 0
        bc_shaper = gs_shaper.gs2bc()             # BCurve.BezierCurve
        ba_shaper = bc_shaper.bezier_arcs[0]      # BezierArc: only first arc
    except IndexError:
        raise Exception("To define the shape, a path with at least one stroke\n and two anchors is needed.")
    cp4_shaper = ba_shaper.cp4
    
    gv_base = vectors_object2gv(base_vectors) # BCurve.GimpVectors
    try:
        gs_base = gv_base.stroke_list[0]      # BCurve.GimpStroke: only stroke 0
        bc_base = gs_base.gs2bc()             # BCurve.BezierCurve
        ba_base = bc_base.bezier_arcs[0]      # BezierArc: only first arc
    except IndexError:
        raise Exception("To define the base, a path with at least one stroke\n and two anchors is needed.")
    cp4_base = ba_base.cp4
    
    gv_target = vectors_object2gv(target_vectors) # BCurve.GimpVectors
    try:
        gs_target = gv_target.stroke_list[0]  # BCurve.GimpStroke: only stroke 0
        bc_target = gs_target.gs2bc()         # BCurve.BezierCurve
        ba_target = bc_target.bezier_arcs[0]  # BezierArc: only first arc
    except IndexError:
        raise Exception("To define the target, a path with at least one stroke\n and two anchors is needed.")
    cp4_target = ba_target.cp4
    #
    if reverse_base:
        cp4_base = cp4_base[::-1]
    if reverse_shaper:
        cp4_shaper = cp4_shaper[::-1]
    #
    if is_loop(cp4_shaper) or gs_shaper.closed:
        raise Exception("Shaper must not be a loop (or too close)")
    if is_loop(cp4_base) or gs_base.closed:
        raise Exception("Base must not be a loop (or too close)")
    if is_loop(cp4_target) or gs_target.closed:
        raise Exception("Target must not be a loop (or too close)")
    base_anchors   = [cp4_base[0],cp4_base[-1]]
    target_anchors = [cp4_target[0],cp4_target[-1]]
    shaping_input = dict()
    shaping_input['case']        = 'path by bezier'
    shaping_input['linefigure']  = treat_as_linefigure
    shaping_input['use_simple']  = use_simple
    shaping_input['shaper_name'] = shaper_vectors.name
    list_vectors_objects = []
    for shape_strength in shape_strength_list:
        p0,p1,p2,p3 = cp4_shaper # RowVector =complex
        p_13 = (2*p0+p3) / 3. # These would give
        p_23 = (p0+2*p3) / 3. # linear Bezier function
        new_p1 = p1*shape_strength + p_13*(1-shape_strength)
        new_p2 = p2*shape_strength + p_23*(1-shape_strength)
        #
        new_cp4 = [p0, new_p1, new_p2, p3]
        # PlaneTransformation:
        pt = make_bezier_map_C_to_C(new_cp4,
                                      base_anchors,
                                      target_anchors,
                                      tweak_a,
                                      tweak_b,
                                      )
        shaping_input['plane transformation'] = pt
        mapped_vectors = shape_path(                               # Main call
               image,
               path_vectors,
               shaping_input,
               )
        #mapped_vectors.name = 'mapped '+str(shape_strength)
        mapped_vectors.name = path_vectors.name+'|mapped '+str(shape_strength)
        list_vectors_objects.append(mapped_vectors)
    pdb.gimp_image_undo_group_start(image)
    # Create merged vectors object:
    merged_name = path_vectors.name+'|mapped by '+shaper_vectors.name
    merged_vectors = pdb.gimp_vectors_new(image, merged_name)
    for vec in list_vectors_objects:
        for stroke in vec.strokes:
            xy, closed = stroke.points
            stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                    merged_vectors, 0, len(xy), xy, closed)
    # Draw and return:
    if merge_down:
        mapped = gimp_draw_vectors_object(image, merged_vectors, visible=True)
    else:
        for vec in list_vectors_objects:
            mapped = gimp_draw_vectors_object(image, vec, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return merged_vectors






#==============================================================
#=============  Moebius path plugin proper  ================
#==============================================================
# Transform a path by a Moebius map

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #


#---------------------------------------------------------
#             Make Moebius transformations
#---------------------------------------------------------

# ----------------
# make_moebius_map
# ----------------
# Given Base and Target: [ControlPoint] (2 or 3 items) (ControlPoint=complex),
# create pt:PlaneTransformation which implements a Moebius transformation
# basically sending points in Base onto points in Target.
# Because these points are finite and also the infinity should be handled,
# the work is divided into 4 cases MAP_CASE 1,...,MAP_CASE 4,
# distingishued according to if Base and Target contain full 3 points or
# only 2 points, with "missing" points indicating infinity.
# MAP_CASE 1: len(Base) = len(Target) = 3
#     Base[0] -> Target[0]
#     Base[1] -> Target[1]
#     Base[2] -> Target[2]
# MAP_CASE 2: len(Base) = 3, len(Target) = 2
#     Base[0] -> Target[0]
#     Base[1] -> Target[1]
#     Base[2] -> infinity
# MAP_CASE 3: len(Base) = 2, len(Target) = 3
#     Base[0] -> Target[0]
#     Base[1] -> Target[1]
#     infinity -> Target[2]
# MAP_CASE 4: len(Base) = len(Target) = 2
#     Base[0] -> Target[0]
#     Base[1] -> infinity
#     infinity -> Target[1]
# Return also
# - pole or None
# - inverse pole or None.

# Args:
# - base:   [ControlPoint] (2 or three items)
# - target: [ControlPoint] (2 or three items)
# - Gimp:    boolean: Set True if the y coordinate runs downwards
# Returns:
# - PlaneTransformation (point_map maps RowVector to RowVector).
# - ControlPoint or None (pole or None)
# - ControlPoint or None (inverse pole or None)
# Note: The case infinity->infinity is not included (though it would be easily
# added) since in that case the Moebius transformation is a direct similitude.
def make_moebius_map(base,
                    target,
                    Gimp=True
                   ):
    if   (len(base)==3) and len(target)==3: map_case = 1
    elif (len(base)==3) and len(target)==2: map_case = 2
    elif (len(base)==2) and len(target)==3: map_case = 3
    elif (len(base)==2) and len(target)==2: map_case = 4
    else:
        raise Exception("make_moebius_map: wrong number of input points")
    if map_case == 1:
        z0,z1,z2 = base   # ControlPoint = RowVector = complex
        w0,w1,w2 = target # ControlPoint = RowVector = complex
        Hf1 = Matrix2x2(z1-z2, -z0*(z1-z2), z1-z0, -z2*(z1-z0))
        Hf2 = Matrix2x2(w1-w2, -w0*(w1-w2), w1-w0, -w2*(w1-w0))
    elif map_case == 2:
        z0,z1,z2 = base
        w0,w1 = target
        Hf1 = Matrix2x2(z1-z2, -z0*(z1-z2), z1-z0, -z2*(z1-z0))
        Hf2 = Matrix2x2(-1, w0, 0, -(w1-w0))
    elif map_case == 3:
        z0,z1 = base
        w0,w1,w2 = target
        Hf1 = Matrix2x2(-1, z0, 0, -(z1-z0))
        Hf2 = Matrix2x2(w1-w2, -w0*(w1-w2), w1-w0, -w2*(w1-w0))
    else: # map_case == 4
        z0,z1 = base
        w0,w2 = target
        Hf1 = Matrix2x2(-1, z0, 0, -(z1-z0))
        Hf2 = Matrix2x2(1, -w0, 1, -w2)
    #
    Hf = Hf2.inverse() * Hf1 # The matrix of the Moebius function
    a,b,c,d = Hf[0,0], Hf[0,1], Hf[1,0], Hf[1,1]
    det = a*d-b*c
    #
    def pm(z): # z:complex
        try:
            return (a*z + b) / (c*z + d)
        except ZeroDivisionError:
            raise InfinityError("Moebius map:Hit the pole.")
    def jac(z):
        try:
            fdot = det / ((c*z+d)**2)
        except ZeroDivisionError:
            raise InfinityError("Moebius map:Hit the pole.")
        ux,uy = fdot.real, fdot.imag
        if Gimp:
            return Matrix2x2(ux, -uy, uy, ux)
            # Possible explanation why this seems to work right in Gimp:
            # If the y coordinate runs downwards, derivating with respect
            # to y introduces an extra minus sign.
        else:
            return Matrix2x2(ux, uy, -uy, ux) # Theoretically correct
    try:
        pole, invpole = -d/c, a/c
    except ZeroDivisionError:
        pole = None
        invpole = None
    return (PlaneTransformation('Moebius',
                                point_map = pm,
                                jacobian = jac),
           pole,
           invpole
           )


#-----------------------------------------------------------
#     Transform a path by Moebius:  main procedure
#-----------------------------------------------------------

# ------------
# moebius_main
# ------------
# Moebius transformation, general (three base points, three target points, and
# no explicit pole or inverse pole).
# Read inputs and call map_path_by_moebius.
#
# Args:
# - image
# - path:                gimp.Vectors (path to be mapped)
# - base:                gimp.Vectors
# - target:              gimp.Vectors
# - base_permutation     string: '012', 102',...
# - use_simple:          integer (>=0) 
# Returns:
# - gimp.Vectors
def moebius_main(
                image,
                path,                   # gimp.Vectors
                #
                base,                   # gimp.Vectors
                target,                 # gimp.Vectors
                base_permutation='012', # string: '012', 102',...
                #
                mark_poles=False,             # boolean
                #
                use_simple=0            # integer
                ):
    bp = base_permutation
    bp_list = [int(bp[0]), int(bp[1]), int(bp[2])]
    tp_list = [0,1,2]
    vectors = map_path_by_moebius(
               image,
               path_vectors         = path,
               #
               base_vectors         = base,
               target_vectors       = target,
               base_permutation     = bp_list,
               #
               pole_case            = 0,  # Case: no explicit pole, inverse pole
               pole_vectors         = None,
               inverse_pole_vectors = None,
               #
               mark_pole            = mark_poles,
               mark_inverse_pole    = mark_poles,
               #
               use_simple           = use_simple,
               )
    return vectors

#-----------------------------------------------------------
#  Transform a path by Moebius with control of poles:  mains
#-----------------------------------------------------------

# Case 1: Pole given
# Case 2: Inverse pole given
# Case 3: Pole and inverse pole given

# --------------
# moebius_1_main
# --------------
# Moebius transformation with control of poles.
# Case 1: The pole is given explicitly but not the inverse pole.
# Read inputs and call map_path_by_moebius.
#
# Args:
# - image
# - path:                gimp.Vectors (path to be mapped)
# - base:                gimp.Vectors
# - target:              gimp.Vectors
# - use_reversed_base:   boolean
# - pole:                gimp.Vectors
# - use_simple:          integer (>=0) 
# Returns:
# - gimp.Vectors
def moebius_1_main(
            image,
            path,              # gimp.Vectors
            #                  
            base,              # gimp.Vectors (2 anchors)
            target,            # gimp.Vectors (2 anchors)
            use_reversed_base, # boolean
            #                  
            pole,              # gimp.Vectors (1 anchor)
            #
            mark_inverse_pole, # boolean
            #                  
            use_simple         # integer
            ):
    if not use_reversed_base: base_permutation = [1,0]
    else:                     base_permutation = [0,1]
    vectors = map_path_by_moebius(
            image,
            path_vectors     = path,
            #                
            base_vectors     = base,
            target_vectors   = target,
            base_permutation = base_permutation,
            #                
            pole_case        = 1,       # Pole explicit
            pole_vectors     = pole,
            #
            mark_pole            = False,
            mark_inverse_pole    = mark_inverse_pole,
            #
            use_simple       = use_simple,
            )
    return vectors

# --------------
# moebius_2_main
# --------------
# Moebius transformation with control of poles.
# Case 2: The inverse pole is given explicitly but not the pole.
# The procedure_function, called directly from GUI.
# Read inputs and call map_path_by_moebius.
#
# Args:
# - image
# - path:                gimp.Vectors (path to be mapped)
# - base:                gimp.Vectors
# - target:              gimp.Vectors
# - use_reversed_base:   boolean
# - use_simple:          integer (>=0) 
# Returns:
# - gimp.Vectors
def moebius_2_main(
            image,           
            path,              # gimp.Vectors
            #                  
            base,              # gimp.Vectors (2 anchors)
            target,            # gimp.Vectors (2 anchors)
            use_reversed_base, # boolean
            #                
            inverse_pole,     # gimp.Vectors (1 anchor)
            #                
            mark_pole,         # boolean
            #
            use_simple,        # integer
            ):
    if not use_reversed_base: base_permutation = [1,0]
    else:                     base_permutation = [0,1]
    vectors = map_path_by_moebius(
            image,
            path_vectors         = path,
            #
            base_vectors         = base,
            target_vectors       = target,
            base_permutation     = base_permutation,
            #
            pole_case            = 2,       # Inverse pole explicit
            inverse_pole_vectors = inverse_pole,
            #
            mark_pole            = mark_pole,
            mark_inverse_pole    = False,
            #
            use_simple           = use_simple,
            )
    return vectors

# --------------
# moebius_3_main
# --------------
# Moebius transformation with control of poles.
# Case 3: The pole and the inverse pole are given explicitly.
# The procedure_function, called directly from GUI.
# Read inputs and call map_path_by_moebius.
#
# Args:
# - image
# - path:                gimp.Vectors (path to be mapped)
# - base:                gimp.Vectors
# - target:              gimp.Vectors
# - use_reversed_base:   boolean
# - use_simple:          integer (>=0) 
# Returns:
# - gimp.Vectors
def moebius_3_main(
            image,
            path,                # gimp.Vectors
            #
            base,                # gimp.Vectors (1 anchor)
            target,              # gimp.Vectors (1 anchor)
            #
            pole,                # gimp.Vectors (1 anchor)
            inverse_pole,        # gimp.Vectors (1 anchor)
            #
            use_simple           # integer
            ):
    vectors = map_path_by_moebius(
            image,
            path_vectors         = path,
            #                 
            base_vectors         = base,
            target_vectors       = target,
            base_permutation     = [0], # No effect
            #                 
            pole_case            = 3,   # Both pole and inverse pole explicit
            pole_vectors         = pole,
            inverse_pole_vectors = inverse_pole,
            #
            mark_pole            = False,
            mark_inverse_pole    = False,
            #
            use_simple           = use_simple,
            )
    return vectors

#-----------------------------------------------------------
#     Transform a path by Moebius: The true main procedure
#-----------------------------------------------------------

# ------------------
# map_path_by_moebius
# ------------------
# Given
# - a path (as Gimp's vectors object),
# - base and target (as Gimp's vectors objects),
# transform the path by the Moebius transformation.
# Input cases:
# pole_case 0: No explicit pole or inverse pole. The three anchors of Base are
#              mapped onto the three anchors of Target.
# pole_case 1: Explicit pole but no explicit inverse pole.
#              Input base and target each have 2 anchors.
# pole_case 2: Explicit inverse pole but no explicit pole.
#              Input base and target each have 2 anchors.
# pole_case 3: Explicit pole and inverse pole.
#              Input base and target each have 1 anchor.
# For the call of shape_path (eventually make_moebius_map), two lists of
# anchor points are created (base_anchors and target_anchors), and the
# different pole_cases are indicated by the lengths (2 or 3) of the lists.
# The anchor lists are created by this rule:
# pole_case 0: base_anchors   = anchors of base_vectors,             (3 anchors)
#              target_anchors = anchors of target_vectors.           (3)
# pole_case 1: base_anchors   = anchors of base_vectors + the pole,  (3)
#              target_anchors = anchors of target_vectors.           (2)
# pole_case 2: base_anchors   = anchors of base_vectors,                      (2)
#              target_anchors = anchors of target_vectors + the inverse pole. (3)
# pole_case 3: base_anchors   = anchor of base_vectors + the pole,            (2)
#              target_anchors = anchor of target_vectors + the inverse pole.  (2)
# If use_simple>0, use the simple algorithm throughout: inaccurate but quick;
# the higher the value, the more accurate the outcome but with more
# numerous control points.
# Draw the result.
# 
# Args:
# - image
# - path_vectors:         gimp.Vectors
# - base_vectors:         gimp.Vectors
# - target_vectors:       gimp.Vectors
# - base_permutation:     [integer]
# - pole_case:            integer
# - pole_vectors:         gimp.Vectors or None
# - inverse_pole_vectors: gimp.Vectors or None
# - use_simple:           integer
# Returns:
# - gimp.Vectors
def map_path_by_moebius(
               image,
               path_vectors,
               #
               base_vectors,
               target_vectors,
               base_permutation     = [0,1,2],
               #
               pole_case            = 0,
               pole_vectors         = None,
               inverse_pole_vectors = None,
               #
               mark_pole            = False,
               mark_inverse_pole    = False,
               #
               use_simple           = 0,
               ):
    def too_close(rv_list): # [RowVector]
        LIMIT = 1e-3
        for i in range(len(rv_list)):
            for j in range(i+1,len(rv_list)):
                v,w = rv_list[i], rv_list[j]
                if abs(v-w) < LIMIT:
                    return True
        return False
    base_anchors = get_anchors(base_vectors, only_first_stroke=True)
    target_anchors = get_anchors(target_vectors, only_first_stroke=True)
    if pole_case == 0:
        base_anchors = base_anchors[:3]     # Only first three anchors
        target_anchors = target_anchors[:3] # Only first three anchors
        if len(base_anchors) < 3:
            raise Exception("Must have three anchors in Base.")
        if len(target_anchors) < 3:
            raise Exception("Must have three anchors in Target.")
        P = base_permutation
        base_anchors = [base_anchors[P[0]],base_anchors[P[1]],base_anchors[P[2]]]
    elif pole_case == 1:
        base_anchors = base_anchors[:2]     # Only first two anchors
        target_anchors = target_anchors[:2] # Only first two anchors
        if len(base_anchors) < 2:
            raise Exception("Must have at least two anchors in Base.")
        if len(target_anchors) < 2:
            raise Exception("Must have at least two anchors in Target.")
        # Base permutation must be done before adding pole:
        P = base_permutation
        base_anchors = [base_anchors[P[0]],base_anchors[P[1]]]
        pole = get_anchors(pole_vectors, only_first_stroke=True)[0] # Only first anchor
        base_anchors.append(pole)
    elif pole_case == 2:
        base_anchors = base_anchors[:2]     # Only first two anchors
        target_anchors = target_anchors[:2] # Only first two anchors
        if len(base_anchors) < 2:
            raise Exception("Must have two anchors in Base.")
        if len(target_anchors) < 2:
            raise Exception("Must have two anchors in Target.")
        P = base_permutation
        base_anchors = [base_anchors[P[0]],base_anchors[P[1]]]
        inv_pole = get_anchors(inverse_pole_vectors, only_first_stroke=True)[0] # Only first anchor
        target_anchors.append(inv_pole)
    elif pole_case == 3:
        base_anchors = base_anchors[:1]     # Only first anchor
        target_anchors = target_anchors[:1] # Only first anchor
        if len(base_anchors) < 1:
            raise Exception("Must have one anchor in Base.")
        if len(target_anchors) < 1:
            raise Exception("Must have one anchor in Target.")
        pole = get_anchors(pole_vectors, only_first_stroke=True)[0]             # Only first anchor
        inv_pole = get_anchors(inverse_pole_vectors, only_first_stroke=True)[0] # Only first anchor
        base_anchors.append(pole)
        target_anchors.append(inv_pole)
    else:
        raise Exception("Unknown case in map_path_by_moebius: "+str(pole_case))
    #
    if too_close(base_anchors):
        raise Exception("Base anchors must be distinct")
    if too_close(target_anchors):
        raise Exception("Target anchors must be distinct")
    shaping_input = dict()
    shaping_input['case']       = 'path by moebius'
    shaping_input['use_simple'] = use_simple
    #
    pt, pole, inverse_pole = make_moebius_map(base_anchors, target_anchors)
    shaping_input['plane transformation'] = pt
    pdb.gimp_image_undo_group_start(image)
    try:
        mapped_vectors = shape_path(                               # Main call
               image,
               path_vectors,
               shaping_input,
               )
        mapped_vectors.name = path_vectors.name+'|mapped by moebius'
        # Create merged vectors object:
        vectors_name = path_vectors.name+'|mapped by moebius'
        vectors = pdb.gimp_vectors_new(image, vectors_name)
        for stroke in mapped_vectors.strokes:
                xy, closed = stroke.points
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                        vectors, 0, len(xy), xy, closed)
        # Draw :
        mapped = gimp_draw_vectors_object(image, vectors, visible=True)
    except InfinityError:
        gimp_message("Hit the pole. No curve produced.")
        vectors = path_vectors # Just return the original to return something
    # Marking poles. Do even if no curve was drawn because of InfinityError.
    if mark_pole:
        if not (pole is None):
            if (0 <= pole.real <= image.width) and (0 <= pole.imag <= image.height):
                pole_ba = BCurve.BezierArc(cp4=[pole]) # Only one anchor
                pole_bc = BCurve.BezierCurve(bezier_arcs=[pole_ba],
                                      curve_name = 'pole'
                                      )
                pole_vectors = pole_bc.bc2vectors_object(image, name='pole')
                pole_path = gimp_draw_vectors_object(image, pole_vectors, visible=True)
            else:
                p = '['+str_float(pole.real) +', '+ str_float(pole.imag) +']'
                gimp_message("Pole not in the window (not marked):\n"+ p)
    if mark_inverse_pole:
        if not (inverse_pole is None):
            if (0 <= inverse_pole.real <= image.width) and (0 <= inverse_pole.imag <= image.height):
                ipole_ba = BCurve.BezierArc(cp4=[inverse_pole]) # Only one anchor
                ipole_bc = BCurve.BezierCurve(bezier_arcs=[ipole_ba],
                                      curve_name = 'inverse pole'
                                      )
                ipole_vectors = ipole_bc.bc2vectors_object(image, name='inverse pole')
                ipole_path = gimp_draw_vectors_object(image, ipole_vectors, visible=True)
            else:
                p = '['+str_float(inverse_pole.real) +', '+ str_float(inverse_pole.imag) +']'
                gimp_message("Inverse pole not in the window (not marked):\n"+p)
                pass
    pdb.gimp_image_undo_group_end(image)
    return vectors


#==============================================================
#=============  Projective map plugin proper  ================
#==============================================================
# Transform a path by a projective map

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

#---------------------------------------------------------
#             Make projective transformation
#---------------------------------------------------------

class Matrix3x3(object):
    """3x3-matrix
    Simple implementation, mainly just to perform matrix multiplication.
    """
    def __init__(self,a00,a01,a02, # Elements row by row
                      a10,a11,a12,
                      a20,a21,a22,
                      ):
        self._elements = [a00,a01,a02,a10,a11,a12,a20,a21,a22]
    def elements(self):
        # Returns list of matrix elements row by row.
        return self._elements
    def __mul__(self,other):
        s = self._elements
        o = other._elements
        return Matrix3x3(s[0]*o[0] + s[1]*o[3] + s[2]*o[6], # 00
                         s[0]*o[1] + s[1]*o[4] + s[2]*o[7], # 01
                         s[0]*o[2] + s[1]*o[5] + s[2]*o[8], # 02
                         s[3]*o[0] + s[4]*o[3] + s[5]*o[6], # 10
                         s[3]*o[1] + s[4]*o[4] + s[5]*o[7], # 11
                         s[3]*o[2] + s[4]*o[5] + s[5]*o[8], # 12
                         s[6]*o[0] + s[7]*o[3] + s[8]*o[6], # 20
                         s[6]*o[1] + s[7]*o[4] + s[8]*o[7], # 21
                         s[6]*o[2] + s[7]*o[5] + s[8]*o[8], # 22
                         )
    def __str__(self):
        s = self._elements
        return str(s[:3]) + '\n'+str(s[3:6])+'\n'+str(s[6:])


class MakeProjectiveMapError(Exception):
    def __init__(self, case, cause, points):
        self.case = case
        self.cause = cause
        self.points = points

# -------------------
# make_projective_map
# -------------------
# Given four points A,B,C,D (base) in a general position (no three collinear),
# another four points P,Q,R,S (target) in a general position,
# make plane transformation which sends
#    A -> P
#    B -> Q
#    C -> R
#    D -> S
# Return also
# - coefficients [g,h,k] such that the pre-image of the ideal line has
#   the equation gx+hy+k=0; or None if the pre-image equals the ideal line;
# - the same info for the image of the ideal line.
# 
# Args:
# - base:   [A,B,C,D]: [ControlPoint,ControlPoint,ControlPoint,ControlPoint]
# - target: [P,Q,R,S]: [ControlPoint,ControlPoint,ControlPoint,ControlPoint]
# Returns:
# - PlaneTransformation
# - [float,float,float] or None (coefficients g,h,k for the pre-image of ideal line)
# - [float,float,float] or None (coefficients g,h,k for the image of ideal line)
# Exceptions:
# - MakeProjectiveMapError
def make_projective_map(base,target):
    # Check base
    A,B,C,D = base # complex (=ControlPoint=RowVector)
    if some_three_collinear(base):
        #raise Exception("make_projective_map: Some three Base points are collinear,")
        raise MakeProjectiveMapError(case=1,
                                     cause="some three base points collinear",
                                     points=base)
    # Check target
    P,Q,R,S = target # complex (=ControlPoint=RowVector)
    if some_three_collinear(target):
         #raise Exception("make_projective_map: Some three Target points are collinear,")
         raise MakeProjectiveMapError(case=2,
                                      cause="some three target points collinear",
                                      points=target)
    #### Step 1 ###
    BB = B-A # B'
    CC = C-A # C'
    DD = D-A # D'
    try:
        M2 = Matrix2x2(BB.real,BB.imag,DD.real,DD.imag).inverse()
    except Exception:
        #print(str(Matrix2x2(BB.real,BB.imag,DD.real,DD.imag)))
        #raise Exception("make_projective_map: det=0, should never happen")
        raise MakeProjectiveMapError(case=3,
                                     cause="det=0, should never happen",
                                     points=base)
    CCC = rv_times_M2x2(CC,M2)
    # g3,h3:
    g3 = (CCC.imag-1)/CCC.real
    h3 = (CCC.real-1)/CCC.imag
    #### Step 2 ###
    KF1 = Matrix3x3( 1, 0, -A.real,
                     0, 1, -A.imag,
                     0, 0, 1)
    #### Step 3 ###
    KF2 = Matrix3x3( DD.imag, -DD.real, 0,
                    -BB.imag,  BB.real, 0,
                     0,      0,    BB.real*DD.imag - BB.imag*DD.real)
    #### Step 4 ###
    KF3 = Matrix3x3( g3+1, 0,   0,
                     0,   h3+1, 0,
                     g3,   h3,   1)
    #### Step 5 ###
    c0,f0 = P.real,P.imag
    try:
        PQ = Q-P
        SR = R-S
        QR = R-Q
        PS = S-P
        denom = cross(QR,SR)
        g0 =  cross(PQ,SR) / denom
        h0 = -cross(PS,QR) / denom
    except ZeroDivisionError:
        #raise Exception("make_projective_map: Should never happen.")
        raise MakeProjectiveMapError(case=4,
                                     cause="denom=0, should never happen",
                                     points=target)
    ad0 = (g0+1)*Q - P # complex
    be0 = (h0+1)*S - P # complex
    a0,d0 = ad0.real,ad0.imag
    b0,e0 = be0.real,be0.imag
    KG = Matrix3x3( a0, b0, c0,
                    d0, e0, f0,
                    g0, h0, 1 )
    #### Step 6 ###
    KF = KG * KF3 * KF2 * KF1 # The 3x3-matrix of the projective transformation F
    KF_elements = KF.elements()
    # To avoid large numbers normalize KF by condition k=1,
    # or if k=0, by condition g**2+h**2=1:
    try:
        inv = 1./KF_elements[-1]
    except ZeroDivisionError:
        try:
            inv = 1./sqrt(KF_elements[-3]**2 + KF_elements[-2]**2)
        except ZeroDivisionError:
            #raise Exception("make_projective_map: Should never happen: zero bottom row!")
            raise MakeProjectiveMapError(case=5,
                                     cause="zero bottom row, should never happen",
                                     points=target)
    KF_elements = [KFe * inv for KFe in KF_elements]
    # matrix elements:
    a,b,c = KF_elements[:3]
    d,e,f = KF_elements[3:6]
    g,h,k = KF_elements[6:]
    #print("KF_elements:")
    #print(str(KF_elements))
    #### Step 7 ###
    def pm(rv):
        x,y = rv.real,rv.imag
        denom = g*x + h*y + k
        try:
            return RowVector((a*x + b*y + c) / denom,
                             (d*x + e*y + f) / denom
                             )
        except ZeroDivisionError:
            raise InfinityError("Projective map:Hit the infinity.")
    #### Step 8 ###
    def jac(rv):
        x,y = rv.real,rv.imag
        try:
            invdenom = 1./((g*x + h*y + k)**2)
        except ZeroDivisionError:
            raise InfinityError("Projective map:Hit the infinity.")
        J00,J01 = (a*h-b*g)*y + (a*k-c*g), (b*g-a*h)*x + (b*k-c*h)
        J10,J11 = (d*h-e*g)*y + (d*k-f*g), (e*g-d*h)*x + (e*k-f*h)
        return Matrix2x2(J00, J01, J10, J11).scale(invdenom)
    pre_image_ideal = [g,h,k]
    image_ideal = [d*h-e*g, -a*h+b*g, a*e-b*d]
    return (PlaneTransformation('projective',
                                point_map = pm,
                                jacobian = jac),
            pre_image_ideal,
            image_ideal
            )


#-----------------------------------------------------------
#     Transform a path by projective map:  main procedure
#-----------------------------------------------------------

# ---------------
# projective_main
# ---------------
# Projective transformation, general (four base points, four target points)
# Read inputs and call map_path_by_projective.
#
# Args:
# - image
# - path:                gimp.Vectors (path to be mapped)
# - base:                gimp.Vectors
# - target:              gimp.Vectors
# - use_reversed_base:   boolean
# - mark_ideals:         boolean
# - use_simple:          integer (>=0) 
# Returns:
# - gimp.Vectors
def projective_main(
                image,
                path,                # gimp.Vectors
                #
                base,                # gimp.Vectors
                target,              # gimp.Vectors
                use_reversed_base,   # boolean
                #
                mark_ideals,          # boolean
                #
                use_simple=0         # integer
                ):
    vectors = map_path_by_projective(
               image,
               path_vectors   = path,
               #              
               base_vectors   = base,
               target_vectors = target,
               reverse_base   = use_reversed_base,
               #
               ideal_case = 0,  # Case: no explicit image or pre-image of ideal line
               image_ideal_vectors         = None,
               pre_image_ideal_vectors     = None,
               #
               mark_ideal     = mark_ideals,
               mark_pre_ideal = mark_ideals, # Mark both if any
               #              
               use_simple     = use_simple,
               )
    return vectors

#-----------------------------------------------------------
#          Transform a path by a projective map
#           with control of infinities:  mains
#-----------------------------------------------------------

# Case 1: Image of the ideal line given (the line where infinite line will be sent)
# Case 2: Pre-image of the ideal line given (the line to be sent to infinity)
# Case 3: Both the pre-image and the image of the ideal line given.

# Note: Case 3 not implemented, and it is unclear if it ever will be.

# -----------------
# projective_1_main
# -----------------
# Projective transformation with control of infinities.
# Case 1: The image of the ideal line is given explicitly but not the pre-image.
# Read inputs and call map_path_by_projective.
#
# Args:
# - image
# - path:                gimp.Vectors (path to be mapped)
# - base:                gimp.Vectors
# - target:              gimp.Vectors
# - use_reversed_base:   boolean
# - image_ideal:         gimp.Vectors
# - mark_pre_ideal:      boolean
# - use_simple:          integer (>=0) 
# Returns:
# - gimp.Vectors
def projective_1_main(
            image,
            path,              # gimp.Vectors
            #                  
            base,              # gimp.Vectors (2 anchors)
            target,            # gimp.Vectors (2 anchors)
            use_reversed_base, # boolean
            #                  
            image_ideal,       # gimp.Vectors (2 anchors)
            #
            mark_pre_ideal,    # boolean
            #                  
            use_simple         # integer
            ):
    vectors = map_path_by_projective(
               image,
               path_vectors   = path,
               #              
               base_vectors   = base,
               target_vectors = target,
               reverse_base   = use_reversed_base,
               #
               ideal_case     = 1,       # pre_image_ideal explicit
               image_ideal_vectors      = image_ideal,
               pre_image_ideal_vectors  = None,
               #
               mark_ideal     = False,
               mark_pre_ideal = mark_pre_ideal,
               #              
               use_simple     = use_simple,
               )
    return vectors



# -----------------
# projective_2_main
# -----------------
# Projective transformation with control of infinities.
# Case 2: The pre-image of the ideal line is given explicitly but not the image.
# Read inputs and call map_path_by_projective.
#
# Args:
# - image
# - path:                gimp.Vectors (path to be mapped)
# - base:                gimp.Vectors
# - target:              gimp.Vectors
# - use_reversed_base:   boolean
# - pre_image_ideal:     gimp.Vectors
# - mark_ideal:          boolean
# - use_simple:          integer (>=0) 
# Returns:
# - gimp.Vectors
def projective_2_main(
            image,
            path,              # gimp.Vectors
            #                  
            base,              # gimp.Vectors (3 anchors)
            target,            # gimp.Vectors (3 anchors)
            use_reversed_base, # boolean
            #                  
            pre_image_ideal,   # gimp.Vectors (2 anchors)
            #
            mark_ideal,        # boolean
            #                  
            use_simple         # integer
            ):
    vectors = map_path_by_projective(
               image,
               path_vectors   = path,
               #              
               base_vectors   = base,
               target_vectors = target,
               reverse_base   = use_reversed_base,
               #
               ideal_case     = 2,       # pre_image_ideal explicit
               image_ideal_vectors     = None,
               pre_image_ideal_vectors = pre_image_ideal,
               #
               mark_ideal     = mark_ideal,
               mark_pre_ideal = False,
               #              
               use_simple     = use_simple,
               )
    return vectors


#-----------------------------------------------------------
#  Transform a path by projective: The true main procedure
#-----------------------------------------------------------

# ----------------------
# map_path_by_projective
# ----------------------
# Given
# - a path (as Gimp's vectors object),
# - base and target (as Gimp's vectors objects),
# transform the path by the projective transformation.
# If use_simple>0, use the simple algorithm throughout: inaccurate but quick;
# the higher the value, the more accurate the outcome but with more
# numerous control points.
# Draw the result.
# 
# Args:
# - image
# - path_vectors:         gimp.Vectors
# - base_vectors:         gimp.Vectors
# - target_vectors:       gimp.Vectors
# - reverse_base:         boolean
# - ideal_case:           integer
# - image_ideal_vectors:     gimp.Vectors
# - pre_image_ideal_vectors: gimp.Vectors
# - use_simple:           integer
# Returns:
# - gimp.Vectors
def map_path_by_projective(
               image,
               path_vectors,
               #
               base_vectors,
               target_vectors,
               reverse_base,
               #
               ideal_case = 0,
               image_ideal_vectors     = None,
               pre_image_ideal_vectors = None,
               #
               mark_ideal     = False,
               mark_pre_ideal = False,
               #
               use_simple     = 0,
               ):
    def too_close(rv_list): # [RowVector]
        LIMIT = 1e-3
        for i in range(len(rv_list)):
            for j in range(i+1,len(rv_list)):
                v,w = rv_list[i], rv_list[j]
                #print("i,j = ",i,j)
                #print("v,w = "+str(v)+','+str(w))
                # CKOE
                #if (v-w).length() < LIMIT:
                if abs(v-w) < LIMIT:
                    return True
        return False
    base_anchors = get_anchors(base_vectors, only_first_stroke=True)
    target_anchors = get_anchors(target_vectors, only_first_stroke=True)
    if ideal_case == 0:
        if len(base_anchors) < 4:
            raise Exception("Must have four anchors in Base.")
        if len(target_anchors) < 4:
            raise Exception("Must have four anchors in Target.")
        #
        if too_close(base_anchors):
            raise Exception("Base anchors must be distinct")
        if too_close(target_anchors):
            raise Exception("Target anchors must be distinct")
        if reverse_base:
            base_anchors = base_anchors[::-1]
        base_anchors = base_anchors[:4]     # First four
        target_anchors = target_anchors[:4] # First four
    elif ideal_case == 1:
        if len(base_anchors) < 3:
            raise Exception("Must have three anchors in Base.")
        if len(target_anchors) < 3:
            raise Exception("Must have three anchors in Target.")
        #
        if too_close(base_anchors):
            raise Exception("Base anchors must be distinct")
        if too_close(target_anchors):
            raise Exception("Target anchors must be distinct")
        if reverse_base:
            base_anchors = base_anchors[::-1]
        # Find a fourth base point and target point:
        ideal = get_anchors(image_ideal_vectors, only_first_stroke=True)
        if len(ideal) != 2:
            raise Exception("Must have two anchors in the image of the line at infinity.")
        X,Y = ideal
        L_coefficients = line_through_two_points(X,Y)
        A,B,C = base_anchors[:3]   # First three
        D = parallelogram_vertex(A,B,C)
        P,Q,R = target_anchors[:3] # First three
        S = complete_quadrilateral_vertex(P,Q,R,L_coefficients)
        # CKOE
        #base_anchors = [BCurve.ControlPoint(x,y) for x,y in [A,B,C,D]]
        #target_anchors = [BCurve.ControlPoint(x,y) for x,y in [P,Q,R,S]]
        base_anchors = [A,B,C,D]
        target_anchors = [P,Q,R,S]
    elif ideal_case == 2:
        if len(base_anchors) < 3:
            raise Exception("Must have three anchors in Base.")
        if len(target_anchors) < 3:
            raise Exception("Must have three anchors in Target.")
        #
        if too_close(base_anchors):
            raise Exception("Base anchors must be distinct")
        if too_close(target_anchors):
            raise Exception("Target anchors must be distinct")
        if reverse_base:
            base_anchors = base_anchors[::-1]
        # Find a fourth base point and target point:
        preideal = get_anchors(pre_image_ideal_vectors, only_first_stroke=True)
        if len(preideal) != 2:
            raise Exception("Must have two anchors in the pre-image of the line at infinity.")
        X,Y = preideal
        L_coefficients = line_through_two_points(X,Y)
        A,B,C = base_anchors[:3]   # First three
        D = complete_quadrilateral_vertex(A,B,C,L_coefficients)
        P,Q,R = target_anchors[:3] # First three
        S = parallelogram_vertex(P,Q,R)
        base_anchors = [BCurve.ControlPoint(x,y) for x,y in [A,B,C,D]]
        target_anchors = [BCurve.ControlPoint(x,y) for x,y in [P,Q,R,S]]
    shaping_input = dict()
    shaping_input['case']       = 'path by projective'
    shaping_input['use_simple'] = use_simple
    #
    pt, pre_image_ideal, image_ideal = make_projective_map(base_anchors, target_anchors)
    shaping_input['plane transformation'] = pt
    pdb.gimp_image_undo_group_start(image)
    try:
        mapped_vectors = shape_path(                               # Main call
                   image,
                   path_vectors,
                   shaping_input,
                   )
        mapped_vectors.name = path_vectors.name+'|mapped by projective'
        # Create merged vectors object:
        vectors_name = path_vectors.name+'|mapped by projective'
        vectors = pdb.gimp_vectors_new(image, vectors_name)
        for stroke in mapped_vectors.strokes:
                xy, closed = stroke.points
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                        vectors, 0, len(xy), xy, closed)
        # Draw and return:
        #if merge_down:
        mapped = gimp_draw_vectors_object(image, vectors, visible=True)
    except InfinityError:
        gimp_message("Hit the infinity. No curve produced.")
        vectors = path_vectors # Just return the original to return something
    # Mark infinities. Done even when no curve was drawn because of InfinityError.
    # CKOE
    #border = [[0,0],
    #          [0,image.height],
    #          [image.width,image.height],
    #          [image.width,0],
    #          [0,0]
    #          ]
    border = [0j,
              complex(0,image.height),
              complex(image.width,image.height),
              complex(image.width,0),
              0j
              ]
    #mark_ideal = True # KOE
    #mark_pre_ideal = True # KOE
    if mark_ideal:
        if not (image_ideal is None): # image_ideal = [g,h,k]
            points_on_border = line_intersect_polyline(image_ideal, border) 
            if len(points_on_border) > 0:
                draw_polyline(image, points_on_border, name='image of line at infinity (vanishing line)', closed=False)
            else:
                m = "Image of line at infinity not in the window (not marked):"
                gimp_message(m)
    if mark_pre_ideal:
        if not (pre_image_ideal is None): # pre_image_ideal = [g,h,k]
            points_on_border = line_intersect_polyline(pre_image_ideal, border) 
            if len(points_on_border) > 0:
                draw_polyline(image, points_on_border, name='pre-image of line at infinity', closed=False)
            else:
                m = "Pre-image of line at infinity not in the window (not marked):"
                gimp_message(m)
    pdb.gimp_image_undo_group_end(image)
    return vectors


#==============================================================
#=============  Bezier16 path plugin proper  ================
#==============================================================
# Test file. Contains Bezier16 map with 2- or 4-anchor Base and target.
# Transform a path by 16 control points

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

#---------------------------------------------------------
#     Make PlaneTransformation from 16 control points
#            PlaneTransformation constructions
#---------------------------------------------------------

# -----------------
# make_bezier16_map
# -----------------
# Given a 4x4 table of control points, a Base, and a Target,
# make a PlaneTransformation based on the 16 control points such that
# it maps the anchors of the Base to the anchors of the Target.
# Two cases (currently):
# - The Base and the Target have 2 anchors each.
# - The Base and the Target have 4 anchors each.
# Args:
# - cp4x4:  [[ControlPoint]]: 4x4 table
# - base:   [A,B] or [A,B,C,D]: [[ControlPoint]]
# - target: [P,Q] or [P,Q,R,S]: [[ControlPoint]]
# - Gimp:   boolean
# Returns:
# - PlaneTransformation.
# Note: If Gimp=True the code takes into account that the y coordinate runs downwards.
def make_bezier16_map(cp4x4,
                     base,
                     target,
                     Gimp=True
                     ):
    raw_bezier16 = make_raw_bezier16_map(cp4x4, Gimp=Gimp)
    if (len(base) == 2) and (len(target) == 2):
        return combine_AB_pt_PQ(raw_bezier16,
                            base,
                            target,
                            type = 'bezier16'
                            )
    elif (len(base) == 4) and (len(target) == 4):
        return combine_ABCD_pt_PQRS(raw_bezier16,
                            base,
                            target,
                            Gimp=Gimp,
                            type = 'bezier16'
                            )
    else: # should never happen
        raise Exception("make_bezier16_map: invalid Base or Target length")

# ----------------------
# make_bezier16_map_easy
# ----------------------
# Easy version: The target is taken form the Shaper, so no target as input.
#
# Given a 4x4 table of control points and a Base
# make a PlaneTransformation based on the 16 control points such that
# it maps the anchors of the Base to the anchors of the Shaper.
# Args:
# - cp4x4:  [[ControlPoint]]: 4x4 table
# - base:   [A,B,C,D]: [[ControlPoint]] (4 anchors)
# - horizon: [z0,z1]: [complex] (to be rotated around z0 to horizontal)
# - Gimp:   boolean
# Returns:
# - PlaneTransformation.
# Note: If Gimp=True the code takes into account that the y coordinate runs downwards.
def make_bezier16_map_easy(cp4x4,
                           base,
                           horizon,
                           rotation_center=None,
                           Gimp=True
                           ):
    raw_bezier16 = make_raw_bezier16_map(cp4x4, Gimp=Gimp)
    return combine_ABCD_pt_rot(raw_bezier16,
                               base,
                               horizon,
                               rotation_center,
                               Gimp=Gimp,
                               type = 'bezier16'
                            )

# ---------------------
# make_raw_bezier16_map
# ---------------------
# Given a 4x4 table of control points,
# make a PlaneTransformation based on the 16 control points such that
# it maps the sides of the unit square (0,0),1,0),(1,1),(0,1) to
# the outer curvy quadrilateral formed by the Bezier arcs B0,C3,B3,C0
# read from the table (see the theory!).
# Args:
# - cp4x4: [[ControlPoint]]: 4x4 table
# - Gimp:  boolean
# Returns:
# - PlaneTransformation (point_map maps RowVector to RowVector).
def make_raw_bezier16_map(cp4x4, Gimp=True):
    b3 = [lambda t: (1-t)**3, # Bernstein polynomials of degree 3
          lambda t: 3*(1-t)*(1-t)*t,
          lambda t: 3*(1-t)*t*t,
          lambda t: t**3
          ]
    b3_dot = [lambda t: -3*(1-t)*(1-t), # Derivatives
              lambda t: 3*(1-t)*(1-3*t),
              lambda t: 3*t*(2-3*t),
              lambda t: 3*t*t
              ]
    # Control points cp4 for each Bezier arc B0,B1,B2,B3:
    cp = copy.copy(cp4x4)
    if Gimp:
        def raw_pm(z):
            u,v = z.real, -z.imag     # Why this works right?
            Fuv = 0
            for r in range(4):
                for s in range(4):
                    Fuv += b3[s](u)*b3[r](v)* cp[r][s]
            return Fuv
        def raw_jac(z):
            u,v = z.real, -z.imag     # Why this works right?
            fu_gu = 0
            for r in range(4):
                for s in range(4):
                    fu_gu += b3_dot[s](u)*b3[r](v)* cp[r][s]
            fv_gv = 0
            for r in range(4):
                for s in range(4):
                    fv_gv += b3[s](u)*b3_dot[r](v)* cp[r][s]
            fu, gu = fu_gu.real, fu_gu.imag
            #fv, gv = fv_gv.real, fv_gv.imag
            fv, gv = -fv_gv.real, -fv_gv.imag     # Why this works right?
            return Matrix2x2(fu, fv, gu, gv)
    else: # UNTESTED
        def raw_pm(z):
            u,v = z.real, z.imag
            Fuv = 0
            for r in range(4):
                for s in range(4):
                    Fuv += b3[s](u)*b3[r](v)* cp[r][s]
            return Fuv
        def raw_jac(z):
            u,v = z.real, z.imag
            fu_gu = 0
            for r in range(4):
                for s in range(4):
                    fu_gu += b3_dot[s](u)*b3[r](v)* cp[r][s]
            fv_gv = 0
            for r in range(4):
                for s in range(4):
                    fv_gv += b3[s](u)*b3_dot[r](v)* cp[r][s]
            fu, gu = fu_gu.real, fu_gu.imag
            fv, gv = fv_gv.real, fv_gv.imag
            return Matrix2x2(fu, fv, gu, gv)
    return PlaneTransformation('raw bezier16',
                                point_map = raw_pm,
                                jacobian = raw_jac)

#------------------------------
#     Auxiliary routines
#------------------------------

# -----------------------
# bezier16_to_4x4table
# -----------------------
# From a Bezier arc quadrilateral construct a 4x4 table of control points.
# The table will be:
#     row 0 = control points of B0 left to right
#     row 1 = control points of B1 left to right
#     row 2 = control points of B2 left to right
#     row 3 = control points of B3 left to right
# (see the theory!).
# Args:
# - shaper:    gimp.Vectors
# - permute:   integer (modulo 8)
# - linearize: boolean
# - adjust:    float
# - strength:  float
# Returns:
# - table4x4:      [[complex]] (a 4x4 table of control points)
def bezier16_to_4x4table(shaper,
                            permute,
                            linearize,
                            adjust=0.,
                            strength=1.
                            ):
    ZEROARC = 1e-8 # To decide when a stroke is closed with a negligible gap
    # arrange_bas: Given list of Bezier arcs,
    # arrange it cyclically and possibly reversing everything so that
    # in the resulting list  a  the Bezier arc  a[0]  is
    # the lowest on the screen (in some sense) and left to right.
    def arrange_bas(bas, Gimp=True):
        if Gimp: c = -1
        else: c = 1
        middle = [bezier_rv(.5, ba.cp4) for ba in bas] # Middle points of arcs
        lowi = min([0,1,2,3], key=lambda i: c*middle[i].imag) # bas[lowi] is lowest
        bas0 = [bas[(i+lowi)%4] for i in range(4)]      # bas0[0] is lowest
        if bas0[0].cp4[0].real <= bas0[0].cp4[-1].real: # bas0[0] is left to right
            return bas0
        return [BCurve.BezierArc(cp4 = list(reversed(bas0[(-i)%4].cp4)))
                                  for i in range(4)]
    def cyclic(table,n):
        b = [[None]*4,[None]*4,[None]*4,[None]*4]
        if n%4 == 0:
            for i in range(4):
                for j in range(4):
                    b[i][j] = table[i][j]
        elif n%4 == 1:
            for i in range(4):
                for j in range(4):
                    b[i][j] = table[3-j][i]
        elif n%4 == 2:
            for i in range(4):
                for j in range(4):
                    b[i][j] = table[3-i][3-j]
        elif n%4 == 3:
            for i in range(4):
                for j in range(4):
                    b[i][j] = table[j][3-i]
        return b
    def transpose(table):
        b = [[None]*4,[None]*4,[None]*4,[None]*4]
        for i in range(4):
            for j in range(4):
                b[i][j] = table[j][i]
        return b
    def p_table(balist): # Create that table p[i][j] (see theory)
        def pij(near_corner, side_corners, far_corner, near_handles, far_handles):
            return (-4 * near_corner
                    -2*side_corners
                    -far_corner
                    +6*near_handles
                    +3*far_handles) / 9
        # Read the input pij:
        p00,p01,p02,p03 = balist[0].cp4
        p13, p23        = balist[1].cp4[1], balist[1].cp4[2]
        p10, p20        = balist[3].cp4[2], balist[3].cp4[1]
        p30,p31,p32,p33 = balist[2].cp4[::-1]
        # Compute p11,p12,p21,p22:
        p11 = pij(near_corner  = p00,
                  side_corners = p03+p30,
                  far_corner   = p33,
                  near_handles = p10+p01,
                  far_handles  = p13+p31
                  )
        p12 = pij(near_corner  = p03,
                  side_corners = p00+p33,
                  far_corner   = p30,
                  near_handles = p13+p02,
                  far_handles  = p10+p32
                  )
        p21 = pij(near_corner  = p30,
                  side_corners = p00+p33,
                  far_corner   = p03,
                  near_handles = p20+p31,
                  far_handles  = p23+p01
                  )
        p22 = pij(near_corner  = p33,
                  side_corners = p03+p30,
                  far_corner   = p00,
                  near_handles = p23+p32,
                  far_handles  = p02+p20
                  )
        p11 = (1-adjust)*p11 + adjust*p00
        p12 = (1-adjust)*p12 + adjust*p03
        p21 = (1-adjust)*p21 + adjust*p30
        p22 = (1-adjust)*p22 + adjust*p33
        return [
                [p00,p01,p02,p03],
                [p10,p11,p12,p13],
                [p20,p21,p22,p23],
                [p30,p31,p32,p33]
                ]
    gv_shaper = vectors_object2gv(shaper) # BCurve.GimpVectors
    gs_shaper = gv_shaper.stroke_list[0]  # BCurve.GimpStroke: only first stroke
    bc_shaper = gs_shaper.gs2bc()         # BCurve.BezierCurve
    balist = bc_shaper.bezier_arcs
    if len(balist) == 3:
        # In Gimp a closed stroke is only marked to be closed.
        # Whether marked or not, we must append one more arc to close the gap:
        balist.append(BCurve.BezierArc(cp4 = [balist[-1].cp4[-1],
                                          bc_shaper.tail_handle,
                                          bc_shaper.head_handle,
                                          balist[0].cp4[0]]))
    elif len(balist) == 4:
        # It may happen that the stroke is marked closed with a negligible gap
        # (for example coming from the Parametric curves plugin).
        # If the gap is tiny enough, we view the start and end anchors equal.
        start_anchor = balist[0].cp4[0]
        end_anchor = balist[-1].cp4[-1]
        length = abs(start_anchor - end_anchor)
        if length < ZEROARC: # Correct the end anchor
            balist[-1].cp4[-1] = start_anchor
        else:
            raise Exception("Please give closed Shaper with four anchors")
    else:
        raise Exception("Please give closed Shaper with four anchors")
    # Apply strength
    new_balist = []
    for ba in balist:
        q0,q1,q2,q3 = ba.cp4
        qq1 = (2*q0 + q3)/3.
        qq2 = (q0 + 2*q3)/3.
        new_q1 = (1-strength)*qq1 + strength*q1
        new_q2 = (1-strength)*qq2 + strength*q2
        new_balist.append(BCurve.BezierArc(cp4=[q0, new_q1, new_q2, q3]))
    balist = new_balist
    # Apply linearize
    if linearize:
        new_balist = []
        for ba in balist:
            if collinear(ba.cp4):
                p0,p3 = ba.cp4[0], ba.cp4[3]
                p1 = (2*p0 + p3)/3
                p2 = (p0 + 2*p3)/3
                new_balist.append(BCurve.BezierArc(cp4=[p0,p1,p2,p3]))
            else:
                new_balist.append(ba)
        balist = new_balist
    # Arrange
    balist = arrange_bas(balist, Gimp=True)
    # Create 4x4-table
    table4x4 = p_table(balist)
    if permute%8 != 0:
        arranged = cyclic(table4x4, permute%4)
        if permute%8 >= 4:
            arranged = transpose(arranged)
        table4x4 = arranged
    return table4x4



#---------------------------------------------
#     Reference and target boxes or lines
#---------------------------------------------

reference_box_options_easy = [ # (description, identifier)
        ('1: Use the bounding box of the source path',            'bb_path'),
        ('2: Use the bounding box of anchors of the source path', 'bb_anchors'),
        ('3: Use the bounding box of selection',                  'bb_selection'),
        ('4: Take reference box from guides',                     'guides'),
        ]

reference_box_options_advanced = (reference_box_options_easy +
        [
        ('5: Use a path as the reference box (choose the path below)', 'path_base')
        ]
        )

target_box_options = [ # (description, identifier)
        ('1: Use the bounding box of the Shaper',                   'bb_shaper'),
        ('2: Use the anchors of the Shaper',                        'anchors_shaper'),
        ('3: Use a path as the target box (choose the path below)', 'path_target'),
        ]


target_usage_options = [ # (description, identifier)
        ('1: Let it serve as an enclosing box for the transformed path','enclosing'),
        ('2: Let its anchors receive the anchors of the reference box', 'anchors'),
        ('3: Use only the bottom edge', 'bottom edge'),
        ]

# -----------
# get_ref_box
# -----------
# From path return bounding box or something similar in the form
# [SW, SE, NE, NW] where items:complex and the directions are as on the screen
# (so, top is N, bottom is S).
# If Gimp=True, the y coordinate runs downwards.
# Args:
# - path_vectors: gimp.Vectors
# - base_vectors: gimp.Vectors
# - ref_box_case: string, one of the identifiers in reference_box_cases
#                 (easy or asvanced)
# - Gimp: boolean
# Returns:
# - [complex,complex,complex,complex]
def get_ref_box(image, path_vectors, base_vectors, ref_box_case, Gimp=True):
    if ref_box_case == 'bb_path': # bb of Path
        nw, se = path_bb(path_vectors, Gimp=Gimp)
        sw = complex(nw.real, se.imag)
        ne = complex(se.real, nw.imag)
        return [sw,se,ne,nw]
    elif ref_box_case == 'bb_anchors': # bb of anchors of Path
        anchors = get_anchors(path_vectors, only_first_stroke=False)
        W = min([a.real for a in anchors])
        E = max([a.real for a in anchors])
        if Gimp:
            S = max([a.imag for a in anchors])
            N = min([a.imag for a in anchors])
        else:
            S = min([a.imag for a in anchors])
            N = max([a.imag for a in anchors])
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        return [sw,se,ne,nw]
    elif ref_box_case == 'bb_selection':
        bounds = pdb.gimp_selection_bounds(image)
        if bounds[0] == 0:
            raise Exception("No active selection")
        W,N,E,S = bounds[1:]
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        return [sw,se,ne,nw]
    elif ref_box_case == 'guides':
        hor,ver = get_guide_positions(image)
        if (len(hor) > 2) or (len(ver) > 2):
            raise Exception("Too many guides: must have two horizontal and two vertical.")
        if (len(hor) < 2) or (len(ver) < 2):
            raise Exception("Not enough guides: must have two horizontal and two vertical.")
        W,E = sorted(ver)
        if Gimp:
            N,S = sorted(hor)
        else:
            S,N = sorted(hor)
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        return [sw,se,ne,nw]
    elif ref_box_case == 'path_base': # base from a path
        base_anchors = get_anchors(base_vectors, only_first_stroke=True)
        if len(base_anchors) != 4:
            raise Exception("The reference box (path) must have 4 anchors")
        if some_three_collinear(base_anchors):
            raise Exception("Collinear anchors in the reference box not allowed.")
        base_anchors = arrange_cps(base_anchors, Gimp=Gimp)
        return base_anchors
    else:
        raise Exception("get_ref_box: invalid choice: "+str(ref_box_case))

# --------------
# get_target_box
# --------------
# From path return bounding box or something similar in the form
# [SW, SE, NE, NW] where items:complex and the directions are as on the screen
# (so, top is N, bottom is S).
# If Gimp=True, the y coordinate runs downwards.
# Args:
# - image
# - shaper_vectors:  gimp.Vectors
# - table4x4:        [[complex]]
# - target_vectors:  gimp.Vectors
# - target_box_case: string, one of the identifiers in target_box_options
# - Gimp:            boolean
# Returns:
# - [complex,complex,complex,complex]
def get_target_box(image, shaper_vectors, table4x4, target_vectors, target_box_case, Gimp=True):
    if target_box_case == 'anchors_shaper': # anchors of Shaper
        target_anchors = [table4x4[0][0], table4x4[0][-1], table4x4[-1][-1], table4x4[-1][0]]
        if some_three_collinear(target_anchors):
            raise Exception("Collinear anchors in the target box not allowed.")
        target_anchors = arrange_cps(target_anchors, Gimp=Gimp)
        return target_anchors
    elif target_box_case == 'bb_shaper': # bounding box of Shaper
        nw, se = path_bb(shaper_vectors, Gimp=Gimp)
        sw = complex(nw.real, se.imag)
        ne = complex(se.real, nw.imag)
        return [sw,se,ne,nw]
    elif target_box_case == 'path_target': # target from a path
        target_anchors = get_anchors(target_vectors, only_first_stroke=True)
        if len(target_anchors) != 4:
            raise Exception("The target box (path) must have 4 anchors")
        if some_three_collinear(target_anchors):
            raise Exception("Collinear anchors in the target box not allowed.")
        target_anchors = arrange_cps(target_anchors, Gimp=Gimp)
        return target_anchors
    else:
        raise Exception("get_target_box: invalid choice: "+str(target_box_case))


# ------------
# get_ref_line
# ------------
# From 2-anchor path return anchors in the order left-to-right as
# [left,right] where items:complex.
# Args:
# - base_vectors: gimp.Vectors
# Returns:
# - [complex,complex]
def get_ref_line(base_vectors):
    a = get_anchors(base_vectors, only_first_stroke=True)
    if len(a) != 2:
        raise Exception("get_ref_line: Wrong number of anchors: "+str(len(a)))
    a0,a1 = a
    if a0.real <= a1.real:
        return a
    else:
        return [a1,a0]

# ---------------
# get_target_line
# ---------------
# From 2-anchor path return anchors in the order left-to-right as
# [left,right] where items:complex.
# Args:
# - target_vectors: gimp.Vectors
# Returns:
# - [complex,complex]
def get_target_line(target_vectors):
    a = get_anchors(target_vectors, only_first_stroke=True)
    if len(a) != 2:
        raise Exception("get_ref_line: Wrong number of anchors: "+str(len(a)))
    a0,a1 = a
    if a0.real <= a1.real:
        return a
    else:
        return [a1,a0]



#-----------------------------------------------------------
#          Transform a path by a transformation
#          based on a Bezier arc quadrilateral:
#                true main procedures
#-----------------------------------------------------------

# --------------------
# map_path_by_bezier16_advanced
# --------------------
# The true main procedure for bezier16_path_advanced_main
# Given
# - Path (as Gimp's vectors object)
# - Shaper in the form of a 4x4 table (16 control points)
# - Base and Target (as Gimp's vectors objects) (two or four anchors),
# shape the Path by the map constructed from the Shaper sending Base to Target.
# If use_simple>0, use the simple algorithm throughout: inaccurate but quick;
# the higher the value, the more accurate the outcome but with more
# numerous control points.
# Draw the result.
# 
# Args:
# - image
# - table4x4:          4x4 table of control points
# - shaper_name:       string
# - path_vectors:      gimp.Vectors
# - base_anchors:      [complex]
# - target_anchors:    [complex]
# - target_name:       string
# - target_usage_option: integer (index in target_usage_options)
# - automatic_arrange: boolean
# - permute_base:      integer
# - permute_target:    integer
# - use_simple:        integer
# Returns:
# - gimp.Vectors
def map_path_by_bezier16_advanced(
               image,
               table4x4,
               shaper_name,
               path_vectors,
               base_anchors,
               target_anchors,
               target_name,
               target_usage_option,
               automatic_arrange=True,
               permute_base=0,
               permute_target=0,
               use_simple=0,
               ):
    def check_edges(base_anchors, target_anchors):
        # Never reached because of non-collinearity checking in main/get_target_box
        ZERO = 1e-3
        a,b = base_anchors
        if abs(a-b) < ZERO:
            raise Exception("The reference box has bottom edge too short (zero?)")
        a,b = target_anchors
        if abs(a-b) < ZERO:
            raise Exception("The target box has bottom edge too short (zero?)")
    def check_convexity(shaper_corners, base_anchors, target_anchors, target_usage_option):
        target_usage_case = target_usage_options[target_usage_option][1]
        if not is_convex(base_anchors):
            raise Exception("Reference box must be convex")
        if not is_convex(target_anchors):
            raise Exception("Target box must be convex")
        if not (is_convex(shaper_corners) or (target_usage_case == 'bottom edge')):
            for x in target_usage_options:
                if x[1] == 'bottom edge':
                    be = x[0]
            m = "The anchors of the Shaper do not form a convex quadrilateral."
            m += "\nIn that case only the option "
            m += '\"'+ be +"\" is allowed."
            raise Exception(m)
    def check_collinearity(shaper_corners, base_anchors, target_anchors, target_usage_option):
        if some_three_collinear(shaper_corners):
            for x in target_usage_options:
                if x[1] == 'bottom edge':
                    be = x[0]
            m = "Collinear anchors in the Shaper are not allowed unless option\n"
            m += '\"'+ be +"\" is chosen."
            raise Exception(m)
        # Never reached because of non-collinearity checking in main/get_target_box
        if some_three_collinear(base_anchors):
            raise Exception("Collinear Base anchors not allowed.")
        if some_three_collinear(target_anchors):
            raise Exception("Collinear Target anchors not allowed.")
    if len(base_anchors) != 4:
        raise Exception("The reference box must have 4 anchors.")
    if len(target_anchors) != 4:
        raise Exception("The target box must have 4 anchors.")
    # Automatic arranging:
    if automatic_arrange: # Currently always
        base_anchors = arrange_cps(base_anchors, Gimp=True)
        target_anchors = arrange_cps(target_anchors, Gimp=True)
    ## User overriding of automatic arranging: Currently never
    #if len(base_anchors) == 2:
    #    if permute_base%2 != 0:
    #        base_anchors = base_anchors[::-1]
    #    if permute_target%2 != 0:
    #        target_anchors = target_anchors[::-1]
    #else: # len = 4:
    #    if permute_base%8 != 0:
    #        # rotate cyclically
    #        arranged = [base_anchors[(i+permute_base)%4] for i in range(4)]
    #        # reverse
    #        if permute_base%8 >= 4:
    #            arranged = arranged[::-1]
    #        base_anchors = arranged
    #    if permute_target%8 != 0:
    #        # rotate cyclically
    #        arranged = [target_anchors[(i+permute_target)%4] for i in range(4)]
    #        # reverse
    #        if permute_target%8 >= 4:
    #            arranged = arranged[::-1]
    #        target_anchors = arranged
    shaper_corners = table4x4[0][0],table4x4[0][3],table4x4[3][3],table4x4[3][0]
    if target_usage_options[target_usage_option][1] == 'bottom edge':
        base_anchors = base_anchors[:2]
        target_anchors = target_anchors[:2]
        check_edges(base_anchors, target_anchors)
    else:
        check_convexity(shaper_corners, base_anchors, target_anchors, target_usage_option)
        check_collinearity(shaper_corners, base_anchors, target_anchors, target_usage_option)
    shaping_input = dict()
    shaping_input['case']        = 'path by bezier16'
    shaping_input['use_simple']  = use_simple
    shaping_input['shaper_name'] = shaper_name
    # PlaneTransformation:
    try:
        if target_usage_options[target_usage_option][1] == 'enclosing':
            pt = make_bezier16_map(table4x4,
                           base_anchors,
                           #target_anchors,
                           shaper_corners, ########## ????????????
                           )
        elif target_usage_options[target_usage_option][1] in ('anchors', 'bottom edge'):
            pt = make_bezier16_map(table4x4,
                           base_anchors,
                           target_anchors,
                           )
        else:
            raise Exception("Unknown target_usage_option: "+str(target_usage_option))
    except combine_ABCD_pt_PQRS_Error as e:
        m = "Cannot construct the transformation."
        shaper_collinear = some_three_collinear(shaper_corners)
        if e.case == 1: # easy: base = reference box
            m += "\nSome three of reference points are collinear."
        elif e.case == 2:
            if shaper_collinear:
                m += "\nSome three of shaper anchors are collinear."
            else:
                m += "\nSome three of the intermediate points during the construction are collinear."
        elif e.case == 3: # easy: target points = shaper points
            if shaper_collinear:
                m += "\nSome three of shaper anchors are collinear."
            else:
                m += "\nUnclear error (e,case=3)"
        else:
            m += "\n"+e.cause
        raise Exception(m)
    shaping_input['plane transformation'] = pt
    mapped_vectors = shape_path(                               # Main call
           image,
           path_vectors,
           shaping_input
           )
    if target_usage_options[target_usage_option][1] == 'enclosing':
        # Map 'use_simple' to 'new_use_simple' to be used in the
        # approximation algorithm in fit_path_to_quadrangle:
        # Input use_simple: [0,1,2,3,4,5,6,7,8]
        new_use_simple =    [0,2,2,2,2,1,1,1,1][use_simple]
        mapped = fit_path_to_quadrangle(image, # Does drawing
                         path_vectors  = mapped_vectors,
                         #base_vertices = shaper_corners,
                         base_vertices = shaper_corners,
                         quad_vertices = target_anchors,
                         quad_name     = target_name,
                         use_simple    = new_use_simple
                         )
        mapped.name = path_vectors.name+'|'+shaping_input['shaper_name']
        mapped.name += '|' + target_name
    elif target_usage_options[target_usage_option][1] in ('anchors', 'bottom edge'):
        mapped = mapped_vectors
        mapped.name = path_vectors.name+'|'+shaping_input['shaper_name']
    else:
        raise Exception("Unknown target_usage_option: "+str(target_usage_option))
    return mapped

# -------------------------
# map_path_by_bezier16_easy
# -------------------------
# The true main procedure for bezier16_path_easy_main
# 
# Args:
# - image
# - table4x4:     4x4 table of control points
# - shaper_name:  string
# - path_vectors: gimp.Vectors
# - base_anchors: [complex,complex,complex,complex]
# - horizon:      [complex,complex] (to be rotated to horizontal)
# - rotation_center: None or complex
# - use_simple:   integer
# Returns:
# - gimp.Vectors
def map_path_by_bezier16_easy(
               image,
               table4x4,
               shaper_name,
               path_vectors,
               base_anchors,
               horizon,
               rotation_center=None,
               use_simple=0,
               ):
    shaping_input = dict()
    shaping_input['case']        = 'path by bezier16'
    shaping_input['use_simple']  = use_simple
    shaping_input['shaper_name'] = shaper_name
    # PlaneTransformation:
    try: # The easy version of the plugin: The target is taken from the Shaper:
        pt = make_bezier16_map_easy(table4x4,
                           base_anchors,
                           horizon,
                           rotation_center
                           )
    except combine_ABCD_pt_rot_Error as e:
        m = "Cannot construct the transformation."
        shaper_corners = [table4x4[0][0], table4x4[0][-1], table4x4[-1][-1], table4x4[-1][0]]
        shaper_collinear = some_three_collinear(shaper_corners)
        if e.case == 1: # easy: base = reference box
            m += "\nSome three of reference points are collinear."
        elif e.case == 2:
            if shaper_collinear:
                m += "\nSome three of shaper anchors are collinear."
            else:
                m += "\nSome three of the intermediate points during the construction are collinear."
        elif e.case == 3: # easy: target points = shaper points
            if shaper_collinear:
                m += "\nSome three of shaper anchors are collinear."
            else:
                m += "\nUnclear error (e,case=3)"
        else:
            m += "\n"+e.cause
        raise Exception(m)
    shaping_input['plane transformation'] = pt
    mapped_vectors = shape_path(                               # Main call
           image,
           path_vectors,
           shaping_input
           )
    mapped_vectors.name = path_vectors.name+'|'+shaping_input['shaper_name']
    pdb.gimp_image_undo_group_start(image)
    mapped = gimp_draw_vectors_object(image, mapped_vectors, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return mapped


#-----------------------------------------------------------
#          Transform a path by a transformation
#          based on a Bezier arc quadrilateral:
#                   main procedures
#-----------------------------------------------------------

# ------------------------------
# bezier16_path_advanced_main
# ------------------------------
# Shape a path by a transformation constructed from
# a Bezier arc quadrilateral (Shaper). It is assumed that the sides of
# the Shaper run roughly two horizontally and two vertically.
# Currently only 4-anchor Base and Target are supported.
#
# Args:
# - image
# - path:             gimp.Vectors (path to be mapped)
# - shaper:           gimp.Vectors (closed, four arcs in one stroke)
# - ref_box_option:   integer
# - base:             gimp.Vectors (two or four anchors in one stroke)
# - target_box_option:   integer
# - target:           gimp.Vectors (two or four anchors in one stroke)
# - target_usage_option: integer
# - linearize_shaper: boolean (for straight edges of shape, use (1/3,2/3)-rule
# - adjust_shaping:   float (perhaps around 0..1)
# - strength_shaping: float
# - use_simple:       integer (>=0) 
# Returns:
# - gimp.Vectors
def bezier16_path_advanced_main(image,
                          path,
                          shaper,
                          ref_box_option,
                          base, # >>>
                          target_box_option,
                          target, # >>>
                          target_usage_option,
                          #horizontal_bottom, # Not currently
                          linearize_shaper,
                          adjust_shaping,
                          strength_shaping,
                          #permute_base,
                          #permute_target,
                          #permute_shaper,
                          use_simple
                          ):
    automatic_arrange = True # No option for user!
    horizontal_bottom = False # No option for user!
    permute_base = permute_target = permute_shaper = 0 # No option for user!
    try:
        ref_box_case = reference_box_options_advanced[ref_box_option][1]
    except IndexError:
        raise Exception("Invalid choice for Path box: "+str(ref_box_option))
    try:
        target_box_case = target_box_options[target_box_option][1]
    except IndexError:
        raise Exception("Invalid choice for Path box: "+str(ref_box_option))
    # Create 4x4 table of control points from shapers
    table4x4 = bezier16_to_4x4table(shaper,
                                       permute   = permute_shaper,
                                       linearize = linearize_shaper,
                                       adjust    = adjust_shaping,
                                       strength  = strength_shaping
                                       )
    base_anchors = get_ref_box(image, path, base, ref_box_case)
    target_anchors = get_target_box(image, shaper, table4x4, target, target_box_case)
    if horizontal_bottom: # rotation around the center point of target - Not currently
        target_anchors = rotate_to_horizontal(target_anchors)
    # Main call
    mapped_vectors = map_path_by_bezier16_advanced(
               image,
               table4x4          = table4x4,
               shaper_name       = shaper.name,
               path_vectors      = path,
               base_anchors      = base_anchors,
               target_anchors    = target_anchors,
               target_name       = target.name,
               target_usage_option = target_usage_option,
               automatic_arrange = automatic_arrange,
               permute_base      = permute_base,
               permute_target    = permute_target,
               use_simple        = use_simple,
               )
    pdb.gimp_image_undo_group_start(image)
    vectors = gimp_draw_vectors_object(image, mapped_vectors, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return mapped_vectors

# --------------------------
# bezier16_path_easy_main
# --------------------------
# Easy version.
# Args:
# - image
# - path:             gimp.Vectors (path to be mapped)
# - shaper:           gimp.Vectors (closed, four arcs in one stroke)
# - ref_box_option:   integer
# - horizontal_bottom: boolean (the target
# - linearize_shaper: boolean (for straight edges of shape, use (1/3,2/3)-rule
# - adjust_shaping:   float (perhaps around 0..1)
# - strength_shaping: float
# - use_simple:       integer (>=0) 
# Returns:
# - gimp.Vectors
def bezier16_path_easy_main(image,
                          path,
                          shaper,
                          ref_box_option,
                          horizontal_bottom,
                          linearize_shaper,
                          adjust_shaping,
                          strength_shaping,
                          use_simple
                          ):
    automatic_arrange=True # No option for user!
    permute_shaper = 0
    try:
        ref_box_case = reference_box_options_easy[ref_box_option][1]
    except IndexError:
        raise Exception("Invalid choice for Path box: "+str(ref_box_option))
    # Create 4x4 table of control points from shapers
    table4x4 = bezier16_to_4x4table(shaper,
                                       permute   = permute_shaper,
                                       linearize = linearize_shaper,
                                       adjust    = adjust_shaping,
                                       strength  = strength_shaping
                                       )
    # easy: no base is used
    base = None
    base_anchors = get_ref_box(image, path, base, ref_box_case)
    # easy: target from shaper:
    if horizontal_bottom:
        horizon = [table4x4[0][0], table4x4[0][-1]] # Must be rotated to horizontal
        rotation_center = (table4x4[0][0] + table4x4[0][-1]
                         + table4x4[-1][-1] + table4x4[-1][0] ) / 4
    else:
        horizon = [table4x4[0][0], table4x4[0][0] + complex(1,0)] # Already horizontal
        rotation_center = None
    # Main call
    mapped_vectors = map_path_by_bezier16_easy(
               image,
               table4x4     = table4x4,
               shaper_name  = shaper.name,
               path_vectors = path,
               base_anchors = base_anchors,
               horizon      = horizon,
               rotation_center = rotation_center,
               use_simple   = use_simple,
               )
    return mapped_vectors


#==============================================================
#=============  Fit path plugin proper  ================
#==============================================================
#------ Fit path into a triangle or quadrangle. ----------

#-------------------  case: triangle  ------------------

# ----------------------
# path_tangent_triangle
# ----------------------
# Given a triangle (three points A,B,C (complex)),
# find the bounding triangle of Gimp's path
# with sides parallel to the sides of the triangle.
# Return as a list [a,b,c] of vertices (in the order corresponding to original).
# Return also the touching points [bc,ca,ab].
# Points are handled as complex numbers.
# Args:
# - path_vectors: gimp.Vectors
# - vertices: [complex,complex,complex]
# Returns:
# - [a,b,c]: [complex,complex,complex]
# - [bc,ca,ab]: [complex,complex,complex]
def path_tangent_triangle(path_vectors, vertices):
    v0,v1,v2 = vertices
    center = (v0+v1+v2)/3
    ccw = (cross(v0-center, -v0+v1) > 0)
    # Triangle side vectors:
    v01 = -v0+v1
    v12 = -v1+v2
    v20 = -v2+v0
    # Path: tangents parallel to v01,v12,v20 (or other candidate touch points):
    B01s = path_tangent_points(path_vectors, v01)
    B12s = path_tangent_points(path_vectors, v12)
    B20s = path_tangent_points(path_vectors, v20)
    #
    anchors = get_anchors(path_vectors, only_first_stroke=False)
    B01s += anchors
    B12s += anchors
    B20s += anchors
    # Choose the right triangle: The right touch points:
    if ccw:
        B01 = max(B01s, key=(lambda x: cross(x-center, v01)))
        B12 = max(B12s, key=(lambda x: cross(x-center, v12)))
        B20 = max(B20s, key=(lambda x: cross(x-center, v20)))
    else:
        B01 = min(B01s, key=(lambda x: cross(x-center, v01)))
        B12 = min(B12s, key=(lambda x: cross(x-center, v12)))
        B20 = min(B20s, key=(lambda x: cross(x-center, v20)))
    # Vertices of the triangle through touch points: New vertices:
    w0 = line_intersection(B20,v20,B01,v01)
    w1 = line_intersection(B01,v01,B12,v12)
    w2 = line_intersection(B12,v12,B20,v20)
    #draw_polyline(image, [w0,w1,w2], name='enclosing triangle', closed=True)
    return [w0,w1,w2], [B12,B20,B01]

# --------------------
# fit_path_to_triangle
# --------------------
# Do the work of fit_path_to_triangle_main, except for the drawing.
# The only difference is that instead of
# argument triangle_vectors:gimp.Vectors (a path for a triangle) we have
# arguments for vertices: [complex] (the vertices of the triangle) and the
# name of the triangle.
# Args:
# - image
# - path_vectors:        gimp.Vectors
# - vertices:            [complex,complex,complex]
# - triangle_name:       string
# Returns:
# - gimp.Vectors
def fit_path_to_triangle(image,
                         path_vectors,
                         vertices,
                         triangle_name,
                         ):
    def transform_stroke(point_map, gs): # just map control points (affine map!)
        def transform_cp_list(cp_list):
            return [point_map(cp) for cp in cp_list]
        return BCurve.GimpStroke(cp_list = transform_cp_list(gs.cp_list),
                                 closed = gs.closed,
                                 stroke_name = gs.stroke_name + ' transformed'
                                 )
    if len(vertices) != 3:
        raise Exception("Please give a triangle with 3 anchors, got "+str(len(vertices)))
    new_vertices, touch_vertices = path_tangent_triangle(path_vectors, vertices)
    v0,v1,v2 = vertices # complex
    w0,w1,w2 = new_vertices # complex
    scale_factor = abs(v0-v1) / abs(w0-w1)
    pt = make_dilation(w0, v0, scale_factor)
    gv = vectors_object2gv(path_vectors)          # GimpVectors
    new_stroke_list = [transform_stroke(pt.point_map, gs)
                                              for gs in gv.stroke_list]
    new_gv = BCurve.GimpVectors(stroke_list = new_stroke_list,
                                name = path_vectors.name + ' fitted in ' + triangle_name
                                )
    vectors = new_gv.gv2vectors_object(image, name=new_gv.name)
    return vectors

#-------------------  case: quadrangle  ------------------

reference_box_options = [ # (description, identifier)
        ('1: Use the bounding box of the source path',            'bb_path'),
        ('2: Use the quadrangle',                                 'quadrangle'),
        ('3: Use another quadrangle, drawn roughly around the source path (choose the path below)', 'path_base')
        ]

# -----------------------
# get_ref_box_for_fitting
# -----------------------
# Get the reference box.
# If Gimp=True, the y coordinate runs downwards.
# Args:
# - path_vectors:       gimp.Vectors
# - quadrangle_vectors: gimp.Vectors
# - ref_vectors:       gimp.Vectors
# - ref_box_case:       string, one of the identifiers in reference_box_cases
# - Gimp:               boolean
# Returns:
# - [complex,complex,complex,complex]
def get_ref_box_for_fitting(image,
                            path_vectors,
                            quadrangle_vectors,
                            ref_vectors,
                            ref_box_case,
                            Gimp=True):
    if ref_box_case == 'bb_path':      # bb of the source path
        nw, se = path_bb(path_vectors, Gimp=Gimp)
        sw = complex(nw.real, se.imag)
        ne = complex(se.real, nw.imag)
        return [sw,se,ne,nw]
    elif ref_box_case == 'quadrangle': # the quadrangle
        base_anchors = get_anchors(quadrangle_vectors, only_first_stroke=True)
        base_anchors = arrange_cps(base_anchors, Gimp=Gimp)
        return base_anchors
    elif ref_box_case == 'path_base':  # reference box from a path
        base_anchors = get_anchors(ref_vectors, only_first_stroke=True)
        if len(base_anchors) != 4:
            raise Exception("The reference box (path) must have 4 anchors")
        if some_three_collinear(base_anchors):
            raise Exception("Collinear anchors in the reference box not allowed.")
        base_anchors = arrange_cps(base_anchors, Gimp=Gimp)
        return base_anchors
    else:
        raise Exception("get_ref_box: invalid choice: "+str(ref_box_case))


# -----------------------
# path_tangent_quadrangle
# -----------------------
# Given a convex(!) quadrangle (four points A,B,C,D (complex)),
# find the bounding quadrangle of Gimp's path
# with sides parallel to the sides of the given quadrangle.
# Return as a list [a,b,c,d] of vertices (in the order corresponding to original).
# Return also the touching points [bc,ca,ab].
# Points are handled as complex numbers.
# Args:
# - path_vectors: gimp.Vectors
# - vertices: [complex,complex,complex,complex]
# Returns:
# - [a,b,c,d]: [complex,complex,complex,complex]
# - [ab,bc,cd,da]: [complex,complex,complex,complex]
def path_tangent_quadrangle(path_vectors, vertices):
    v0,v1,v2,v3 = vertices
    center = (v0+v1+v2+v3)/4
    ccw = (cross(v0-center, -v0+v1) > 0)
    # Quadrangle side vectors:
    v01 = -v0+v1
    v12 = -v1+v2
    v23 = -v2+v3
    v30 = -v3+v0
    # Path: tangents parallel to v01,v12,v20 (or other candidate touch points):
    B01s = path_tangent_points(path_vectors, v01)
    B12s = path_tangent_points(path_vectors, v12)
    B23s = path_tangent_points(path_vectors, v23)
    B30s = path_tangent_points(path_vectors, v30)
    #
    anchors = get_anchors(path_vectors, only_first_stroke=False)
    B01s += anchors
    B12s += anchors
    B23s += anchors
    B30s += anchors
    # Choose the right quadrangle: the right touch points:
    if ccw:
        B01 = max(B01s, key=(lambda x: cross(x-center, v01)))
        B12 = max(B12s, key=(lambda x: cross(x-center, v12)))
        B23 = max(B23s, key=(lambda x: cross(x-center, v23)))
        B30 = max(B30s, key=(lambda x: cross(x-center, v30)))
    else:
        B01 = min(B01s, key=(lambda x: cross(x-center, v01)))
        B12 = min(B12s, key=(lambda x: cross(x-center, v12)))
        B23 = min(B23s, key=(lambda x: cross(x-center, v23)))
        B30 = min(B30s, key=(lambda x: cross(x-center, v30)))
    # Vertices of the quadrangle through touch points:
    # New vertices = the corner vertices:
    w0 = line_intersection(B30,v30,B01,v01)
    w1 = line_intersection(B01,v01,B12,v12)
    w2 = line_intersection(B12,v12,B23,v23)
    w3 = line_intersection(B23,v23,B30,v30)
    #draw_polyline(image, [w0,w1,w2,w3], name='enclosing quadrangle', closed=True)
    return [w0,w1,w2,w3], [B01,B12,B23,B30]

# ----------------------
# fit_path_to_quadrangle
# ----------------------
# Do the work of fit_path_to_quadrangle_main, except for the drawing.
# The vertices of the base and target quadrangle are assumed to be arranged by
# the caller to corresponding orders; no arranging is done here.
# Args:
# - image
# - path_vectors:  gimp.Vectors
# - base_vertices: [complex,complex,complex,complex]
# - quad_vertices: [complex,complex,complex,complex]
# - quad_name:     string
# - use_simple:    integer
# Returns:
# - gimp.Vectors
def fit_path_to_quadrangle(image,
                           path_vectors,
                           base_vertices,
                           quad_vertices,
                           quad_name,
                           use_simple=0,
                           ):
    if len(base_vertices) != 4:
        raise Exception("Please give a reference box with 4 anchors, got "+str(len(base_vertices)))
    if not is_convex(base_vertices):
        raise Exception("The reference box must be convex.")
    if len(quad_vertices) != 4:
        raise Exception("Please give a quadrangle with 4 anchors, got "+str(len(quad_vertices)))
    if not is_convex(quad_vertices):
        raise Exception("The quadrangle must be convex.")
    new_vertices, touch_vertices = path_tangent_quadrangle(path_vectors, base_vertices)
    #
    pt = make_projective_map(new_vertices, quad_vertices)[0]
    #
    shaping_input = dict()
    shaping_input['case']        = 'path by bezier16'
    shaping_input['use_simple']  = use_simple
    shaping_input['shaper_name'] = 'shaper'
    #
    shaping_input['plane transformation'] = pt
    mapped_vectors = shape_path(                               # Main call
           image,
           path_vectors,
           shaping_input
           )
    mapped_vectors.name = path_vectors.name+' fitted in ' + quad_name
    return mapped_vectors


# -----------------------------------------------------------
#                 Fit path: Main procedures
# -----------------------------------------------------------

# -------------------------
# fit_path_to_triangle_main
# -------------------------
# Fit Gimp's path into a triangle by means of translation and scaling
# (that is, dilation).
# Args:
# - image
# - path_vectors:        gimp.Vectors
# - triangle_vectors:    gimp.Vectors
# Returns:
# - gimp.Vectors
def fit_path_to_triangle_main(image,
                              path_vectors,
                              triangle_vectors,
                              ):
    vertices = get_anchors(triangle_vectors, only_first_stroke=True)
    triangle_name = triangle_vectors.name
    fitted_vectors = fit_path_to_triangle(image, # Main call
                         path_vectors,
                         vertices,
                         triangle_name,
                         )
    pdb.gimp_image_undo_group_start(image)       # Draw
    vectors = gimp_draw_vectors_object(image, fitted_vectors, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return vectors

# ---------------------------
# fit_path_to_quadrangle_main
# ---------------------------
# Fit Gimp's path into a quadrangle by means of a projective transformation.
# Option to choose the algorithm for the projective transformation (good or simple).
# Args:
# - image
# - path_vectors:       gimp.Vectors
# - quadrangle_vectors: gimp.Vectors
# - ref_box_option:     integer
# - ref_vectors:       gimp.Vectors
# - use_simple:         integer (>=0) 
# Returns:
# - gimp.Vectors
def fit_path_to_quadrangle_main(image,
                                path_vectors,
                                quadrangle_vectors,
                                ref_box_option,
                                ref_vectors, # >>>
                                use_simple=0,
                                ):
    try:
        ref_box_case = reference_box_options[ref_box_option][1]
    except IndexError:
        raise Exception("Invalid choice for reference box: "+str(ref_box_option))
    base_anchors = get_ref_box_for_fitting(image,    # Get reference box
                                           path_vectors,
                                           quadrangle_vectors,
                                           ref_vectors,
                                           ref_box_case)
    quadrangle_vertices = get_anchors(quadrangle_vectors, only_first_stroke=True)
    quadrangle_vertices = arrange_cps(quadrangle_vertices, Gimp=True)
    quadrangle_name = quadrangle_vectors.name
    fitted_vectors = fit_path_to_quadrangle(image,   # Main call
                         path_vectors,
                         base_vertices = base_anchors,
                         quad_vertices = quadrangle_vertices,
                         quad_name = quadrangle_name,
                         use_simple = use_simple
                         )
    pdb.gimp_image_undo_group_start(image)           # Draw
    vectors = gimp_draw_vectors_object(image, fitted_vectors, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return vectors


#==============================================================
#=============  Exp path plugin proper  ================
#==============================================================
# Transform a path by complex exponential map.

#---------------------------------------------------------
#             Make exp transformations
#---------------------------------------------------------

# ----------------
# make_raw_exp_map
# ----------------
# Implemented is  exp(u*z)  where u is a fixed complex number,
# conjugated by rotation through angle 'rotate' (radians) to get a
# rotated version of exp.
# Return also the center (the point that is never reached).
#
# Args:
# - u:      complex
# - rotate: float (angle in radians)
# - Gimp:   boolean: Set True if the y coordinate runs downwards
# Returns:
# - PlaneTransformation
# - complex

###############  KOE
def make_raw_exp_map(u, rotate, Gimp=True):
    from cmath import exp as cexp
    rot = cexp(1j*rotate)
    def pm(z): # z:complex
        return cexp(rot*u*z) / rot
    if Gimp:
        def jac(z):
            euz = u*cexp(rot*u*z)
            cy, sy = euz.real, euz.imag
            return Matrix2x2(cy, -sy, sy, cy)
    else: # Untested
        def jac(z):
            euz = u*cexp(rot*u*z)
            cy, sy = euz.real, euz.imag
            return Matrix2x2(cy, sy, -sy, cy)
    center = complex(0,0)
    return (PlaneTransformation('exp',
                                point_map = pm,
                                jacobian = jac),
            center)


#---------------------------------------------------------
#             Options
#---------------------------------------------------------


exp_center_options = [ # (description, identifier, rotate_value)
        ('Center at the top',     'top',      -pi/2),
        ('Center at the bottom',  'bottom',   pi/2),
        ('Center on the left',    'left',     0),
        ('Center on the right',   'right',    pi),
        ]


exp_trg_box_options = [ # (description, identifier)
        ('Bounding box of the source path',            'bb_path'),
        ('Bounding box of selection',                  'bb_selection'),
        ('Guides - two horizontal, two vertical',      'guides'),
        ('Window',                                     'window')
        ]


#-----------------------------------------------------------
#     Transform a path by exponential map:  main procedure
#-----------------------------------------------------------

# --------------------
# map_path_by_exp_main
# --------------------
# Map a path by complex exponential function exp(u*z) for some fixed complex u,
# combined with some direct similitudes.
# Options:
# - center_option: The center of the transformation will be at
#   top, bottom, left, or right;
# - target_option: a quadrangle where the transformed path will be fit.
# The center_option will cause some rotated version of the exponential map
# to be used.
# Other inputs:
# - two angle parameters: alpha and gamma;
# - choice of algorithm.
# (The coefficient u is computed from alpha,gamma.)
# The map is combined from the left and from the right with direct
# similitudes to achieve that
# - when center_option is 'left', a line segment from the middle of the bounding
#   box of the source path to the right will take the role of the line segment [0,1]
#   in the complex plane for the exponential map;
# - the transformed path will be fitted in the target box as tightly as possible.
# The meaning of the two angle parameters alpha and gamma is:
# When center_option is 'left', the images of horizontal line segments
# will be logarithmic spiral arcs, where
# - alpha is the central angle of the arc;
# - gamma is the polar slope angle of the arc (the constant angle between the arc
#   and any circle centered at the center of the logarithmic spiral).
# Note: Instead of the four choices for the center_option, the routine
# map_path_by_exp accepts any rotation angle. This possibility is not offered to
# the user.
#
# Args:
# - image
# - path:          gimp.Vectors (path to be mapped)
# - center_option: integer (see exp_center_options)
# - target_option: integer (see exp_trg_box_options)
# - alpha:         float (central angle in degrees)
# - gamma:         float (polar slope angle in degrees)
# - use_simple:    integer (>=0) 
# Returns:
# - gimp.Vectors
def map_path_by_exp_main(
                image,
                path,
                center_option,
                target_option,
                alpha,
                gamma,
                mark_center,
                use_simple=4
                ):
    from cmath import exp as cexp
    ZERO = 1e-8
    center_case = exp_center_options[center_option][1]
    path_NW, path_SE = path_bb(path, Gimp=True)
    path_middle = (path_NW + path_SE) / 2
    if center_case in ('left','right'):
        path_height = abs(path_NW.imag - path_SE.imag)
        base_points = [path_middle, path_middle + complex(path_height/2, 0)]
    else:
        path_width = abs(path_NW.real - path_SE.real)
        base_points = [path_middle, path_middle + complex(path_width/2, 0)]
    try:
        rotate_value = exp_center_options[center_option][2]
    except IndexError:
        raise Exception("Invalid choice for center option: "+str(center_option))
    try:
        trg_box_case = exp_trg_box_options[target_option][1]
    except IndexError:
        raise Exception("Invalid choice for Target box: "+str(target_option))
    target_corners = get_box(image, path, trg_box_case)
    A,B = base_points # complex
    if abs(A-B) < ZERO:
        raise Exception("Path is one point?.")
    central_angle = alpha * pi / 180 # radians
    if not (-85 <= gamma <= 85):
        raise Exception("gamma must be between -85 and +85 degrees.")
    slope_angle = gamma * pi / 180 # radians
    exp_coeff = cexp(1j*slope_angle) * central_angle / cos(slope_angle)
    exp_coeff /= 2
    mapped_vectors, center = map_path_by_exp(
                                      image, # Main call
                                      path_vectors  = path,
                                      base_points   = base_points,
                                      target_corners = target_corners,
                                      exp_coeff     = exp_coeff,
                                      rotate = rotate_value,
                                      use_simple    = use_simple,
                                      )
    mapped_vectors.name = path.name+'|exp('+str(alpha)+','+str(gamma) +','+center_case+')'
    pdb.gimp_image_undo_group_start(image) # Draw
    mapped = gimp_draw_vectors_object(image, mapped_vectors, visible=True)
    if mark_center:
        if (0 <= center.real <= image.width) and (0 <= center.imag <= image.height):
            center_ba = BCurve.BezierArc(cp4=[center]) # Only one anchor
            center_bc = BCurve.BezierCurve(bezier_arcs=[center_ba],
                                  curve_name = 'center'
                                  )
            center_vectors = center_bc.bc2vectors_object(image, name='center of the exp plugin')
            center_path = gimp_draw_vectors_object(image, center_vectors, visible=True)
        else:
            p = '['+str_float(center.real) +', '+ str_float(center.imag) +']'
            gimp_message("Center not in the window (not marked):\n"+ p)
    pdb.gimp_image_undo_group_end(image)
    return mapped_vectors

# ---------------
# map_path_by_exp
# ---------------
# Do the main work of map_path_by_exp_main except the drawing.
# Return
# - gimp.Vectors
# - center: the center of the logarithmic spirals (the point never reached).
#
# Args:
# - image
# - path_vectors:   gimp.Vectors (path to be mapped)
# - base_points:    [complex,complex]
# - target_corners: [complex,complex,complex,complex] ([SW, SE, NE, NW])
# - exp_coeff:      complex
# - rotate:         float (angle in radians: rotated version of exp)
# - use_simple:     integer (>=0) 
# Returns:
# - gimp.Vectors,
# - complex
def map_path_by_exp(image,
                    path_vectors,
                    base_points,
                    target_corners,
                    exp_coeff,
                    rotate,
                    use_simple,
                    ):
    # fit_box_in_box: Given two boxes corner by corner [A,B,C,D], [a,b,c,d],
    # find suitable line segments [X,Y], [x,y], such that
    # the direct similitude sending X->x and Y->y maps the box [A,B,C,D]
    # into the box [a,b,c,d] as tightly as possible.
    from cmath import exp as cexp
    def fit_box_in_box(ABCD, abcd):
        A,B,C,D = ABCD # Base box
        a,b,c,d = abcd # Target box
        Middle = (A+B+C+D)/4
        middle = (a+b+c+d)/4
        try: # Target box: check if flat or one point
            ratio = abs(a-b) / abs(b-c)
            invratio = abs(b-c) / abs(a-b)
        except ZeroDivisionError:
            raise Exception("Flat target box not allowed.")
        try: # Base box: is horizontal line (flat)?
            Ratio = abs(A-B) / abs(B-C)
        except ZeroDivisionError: # Is horizontal line (flat)!
            try: # Base box: is also vertical line (flat)?
                invRatio = abs(B-C) / abs(A-B)
                base_segment = [Middle, (B+C)/2]
                target_segment = [middle, (b+c)/2]
                return [base_segment, target_segment]
            except ZeroDivisionError: # flat flat => point
                raise Exception("Transformed path becomes one point! Is alpha 0 or is source path one point?")
        # Now know that not horizontal line (flat).
        try: # Base box: is vertical line (flat)?
            invRatio = abs(B-C) / abs(A-B)
        except ZeroDivisionError: # Is vertical line (flat)!
            base_segment = [Middle, (C+D)/2]
            target_segment = [middle, (c+d)/2]
            return [base_segment, target_segment]
        # Now know that neither horizontal nor vertical line (flat).
        if Ratio >= ratio:
            base_segment = [Middle, (B+C)/2]
            target_segment = [middle, (b+c)/2]
        else:
            base_segment = [Middle, (C+D)/2]
            target_segment = [middle, (c+d)/2]
        return [base_segment, target_segment]
    # Map path by direct similitude where base_points -> 0,1:
    h1_path = map_path_by_affine(image,    #  h1(path)
                                 path_vectors,
                                 base_points,
                                 [0j, 1+0j],
                                 reversed_base=False
                                 )
    h1_path.name = 'h1'
    # Map h1_path with the complex exponetial map:
    shaping_input = dict()
    shaping_input['case']       = 'path by exp'
    shaping_input['use_simple'] = use_simple
    shaping_input['typical arc size'] = 1 # Moving in complex plane around origo
    #
    raw_pt, raw_center = make_raw_exp_map(exp_coeff, rotate, Gimp=True)
    shaping_input['plane transformation'] = raw_pt
    F_h1_path = shape_path(image,              # F(h1(path))
                           h1_path,
                           shaping_input
                           )
    F_h1_path.name = 'F_h1'
    #gimp_draw_vectors_object(image, h1_path, visible=True)
    #gimp_draw_vectors_object(image, F_h1_path, visible=True)
    # Get the bb of F_h1_path:
    F_h1_NW, F_h1_SE = path_bb(F_h1_path, Gimp=True)
    # Target cornes and corners of the bb of F_h1:
    trg_corners = target_corners
    Fh1_corners = [complex(F_h1_NW.real, F_h1_SE.imag),
                   F_h1_SE,
                   complex(F_h1_SE.real, F_h1_NW.imag),
                   F_h1_NW]
    fit_base, fit_target = fit_box_in_box(Fh1_corners, trg_corners)
    shaped_final = map_path_by_affine(         # h2(F(h1(path)))
                       image,
                       F_h1_path,
                       fit_base,
                       fit_target,
                       reversed_base=False   
                       ) 
    # Find the center:
    ds = make_direct_similitude(fit_base[0], fit_base[1], fit_target[0], fit_target[1])
    center = ds.point_map(raw_center)
    return shaped_final, center

#==============================================================
#==========  Path to polar coords plugin proper  ==============
#==============================================================
# Convert a path to polar coordinates.
# Transform a path by (x,y) -> x(cos(y),sin(y))
# inside the bounding box of the path.

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #


#---------------------------------------------------------
#             Make polar transformations
#---------------------------------------------------------

# ------------------
# make_raw_polar_map
# ------------------
# Implemented is x+iy -> x*exp(i*theta*y).
# Args:
# - theta: float
# - Gimp:  boolean
# Returns:
# - PlaneTransformation
def make_raw_polar_map(theta, Gimp):
    from math import exp,cos,sin
    if Gimp:
        theta = -theta
    def pm(z): # z:complex
        x, y = z.real, z.imag
        return complex(x*cos(theta*y), x*sin(theta*y))
    def jac(z):
        x, y = z.real, z.imag
        c, s = cos(theta*y), sin(theta*y)
        return Matrix2x2(c, -theta*x*s, s, +theta*x*c)
    return PlaneTransformation('polar',
                                point_map = pm,
                                jacobian = jac)

# --------------------------
# make_raw_inverse_polar_map
# --------------------------
# Implemented is the inverse of x+iy -> x*exp(i*theta*y).
# Args:
# - theta: float
# - Gimp:  boolean
# Returns:
# - PlaneTransformation
def make_raw_inverse_polar_map(theta, Gimp):
    from cmath import polar
    if Gimp:
        theta = -theta
    try:
        inv_theta = 1./theta
    except ZeroDivisionError:
        raise Exception("make_raw_inverse_polar_map: theta too close to 0: "+str(theta))
    def pm(z): # z:complex
        rho,psi = polar(z)
        return complex(rho, psi*inv_theta)
    def jac(z):
        xi, eta = z.real, z.imag
        rho,psi = polar(z)
        try:
            inv_rho = 1./rho
            ith = inv_theta*inv_rho*inv_rho
        except ZeroDivisionError:
            #raise Exception("inverse_polar_map, jac: rho too close to 0: "+str(rho))
            #return Matrix2x2(1,0,0,1)  ##  ???
            return Matrix2x2(0,0,0,0)  ##  ???
        return Matrix2x2(xi*inv_rho, eta*inv_rho, -eta*ith, xi*ith)
    return PlaneTransformation('polar',
                                point_map = pm,
                                jacobian = jac)


#---------------------------------------------------------
#             Options
#---------------------------------------------------------


edge_options = [ # (description, identifier)
        ('Top to middle','top'),
        ('Bottom to middle','bottom'),
        ('Left side to middle','left'),
        ('Right side to middle','right'),
        ]


polar_box_options = [ # (description, identifier)
        ('Bounding box of the source path',            'bb_path'),
        #('Bounding box of anchors of the source path', 'bb_anchors'),
        ('Bounding box of selection',                  'bb_selection'),
        ('Guides',                     'guides'),
        ('Window',      'window')
        ]

# -------
# get_box
# -------
# From path return bounding box or something similar in the form
# [SW, SE, NE, NW] where items:complex and the directions are as on the screen
# (so, top is N, bottom is S).
# If Gimp=True, y coordinate runs downwards.
# Args:
# - path_vectors: gimp.Vectors
# - base_vectors: gimp.Vectors
# - ref_box_case: string, one of the identifiers in reference_box_cases
#                 (easy or asvanced)
# - Gimp: boolean
# Returns:
# - [complex,complex,complex,complex]
def get_box(image, path_vectors, box_case, Gimp=True):
    #print(box_case)
    if box_case == 'bb_path': # bb of Path
        nw, se = path_bb(path_vectors, Gimp=Gimp)
        sw = complex(nw.real, se.imag)
        ne = complex(se.real, nw.imag)
        return [sw,se,ne,nw]
    elif box_case == 'bb_anchors': # bb of anchors of Path
        anchors = get_anchors(path_vectors, only_first_stroke=False)
        W = min([a.real for a in anchors])
        E = max([a.real for a in anchors])
        if Gimp:
            S = max([a.imag for a in anchors])
            N = min([a.imag for a in anchors])
        else:
            S = min([a.imag for a in anchors])
            N = max([a.imag for a in anchors])
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        return [sw,se,ne,nw]
    elif box_case == 'bb_selection':
        bounds = pdb.gimp_selection_bounds(image)
        if bounds[0] == 0:
            raise Exception("No active selection")
        W,N,E,S = bounds[1:]
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        return [sw,se,ne,nw]
    elif box_case == 'guides':
        hor,ver = get_guide_positions(image)
        if (len(hor) > 2) or (len(ver) > 2):
            raise Exception("Too many guides: must have two horizontal and two vertical.")
        if (len(hor) < 2) or (len(ver) < 2):
            raise Exception("Not enough guides: must have two horizontal and two vertical.")
        W,E = sorted(ver)
        if Gimp:
            N,S = sorted(hor)
        else:
            S,N = sorted(hor)
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        return [sw,se,ne,nw]
    if box_case == 'window': # window
        w,h = image.width, image.height
        #print("w,h = "+str(w)+','+str(h))
        sw = complex(0,h)
        se = complex(w,h)
        ne = complex(w,0)
        nw = complex(0,0)
        return [sw,se,ne,nw]
    #elif box_case == 'path_base': # base from a path
    #    base_anchors = get_anchors(base_vectors)
    #    if len(base_anchors) != 4:
    #        raise Exception("The reference box (path) must have 4 anchors")
    #    if some_three_collinear(base_anchors):
    #        raise Exception("Collinear anchors in the reference box not allowed.")
    #    base_anchors = arrange_cps(base_anchors, Gimp=Gimp)
    #    return base_anchors
    else:
        raise Exception("get_box: invalid choice: "+str(box_case))


#-----------------------------------------------------------
#     Convert a path to polar coordinates:  main procedure
#-----------------------------------------------------------

# --------------------
# path_to_polar_main
# --------------------
# Convert a path to polar coordinates (the basic map is (x,y) -> x(cos(y),sin(y))),
# for example in the space of its bounding box (or in the space determined
# by selection or guides).
# Combine with some stretching so that one edge of the bounding box will be
# sent to a circle. The opposite edge will be sent to the middle point.
# Allow choosing which edge is to be sent to the middle point.
# In addition to a whole circle, allow a circle sector with chosen central angle.
# Args:
# - image
# - path:           gimp.Vectors (path to be mapped)
# - edge_to_middle: integer (see edge_options)
# - central_angle:  float (degrees)
# - ref_box_option: integer (see polar_box_options)
# - use_simple:     integer (>=0) 
# Returns:
# - gimp.Vectors
def path_to_polar_main(
                image,
                path,         # gimp.Vectors
                edge_to_middle, # integer
                central_angle,  # float, central angle in degrees
                ref_box_option,
                #inverse_mapping, # boolean
                use_simple=4  # integer
                ):
    # check_bb: Check that the bounding box is not flat.
    # If it is, use padding to make it square. Return new nw,se.
    # But if nw=se (or too close) return None. The caller must handle.
    def check_bb(nw,se):
        ZERO = 1e-8
        N,W = nw.imag, nw.real
        S,E = se.imag, se.real
        hor_ok = (abs(W-E) > ZERO)
        vert_ok = (abs(N-S) > ZERO)
        if hor_ok and vert_ok:
            return nw,se
        if not (hor_ok or vert_ok):
            return None
        if not hor_ok:
            height = S-N # Use height/2 as horizontal padding
            new_nw = complex(W - height/2, N)
            new_se = complex(E + height/2, S)
            return [new_nw,new_se]
        if not vert_ok:
            width = E-W # Use width/2 as vertical padding
            new_nw = complex(W, N - width/2)
            new_se = complex(E, N + width/2)
            return [new_nw,new_se]
    inverse_mapping = False # Inverse mapping implemented but not offered.
    #from cmath import exp as cexp
    try:
        ref_box_case = polar_box_options[ref_box_option][1]
    except IndexError:
        raise Exception("Invalid choice for Path box: "+str(ref_box_option))
    box_sw, box_se, box_ne, box_nw = get_box(image, path, ref_box_case)
    # Make sure that box is not flat.
    new_nw_se = check_bb(box_nw, box_se)
    if new_nw_se is None:
        raise Exception("Box is one point or too close.")
    box_nw, box_se = new_nw_se
    #
    box_sw = complex(box_nw.real, box_se.imag)
    box_ne = complex(box_se.real, box_nw.imag)
    #
    middle = (box_nw + box_se)/2
    # Smallest square that fits in the box:
    square_size = min(abs(box_sw.imag-box_nw.imag), abs(box_se.real-box_sw.real))
    # Fill:
    # - box_corners = [a,b,c,d], corners of the bb of the path in ccw order
    #   where [d,a] is that edge which will be mapped to the middle point;
    # - target_segment = [middle, v] where v is the point where the
    #   middle point of [b,c] will be mapped.
    usage = edge_options[edge_to_middle][1]
    central_angle_radians = central_angle * pi / 180 # radians
    #
    if not inverse_mapping:
        if usage == 'top':
            box_corners = [box_nw, box_sw, box_se, box_ne]       # a,b,c,d
            target_segment = [middle, middle + complex(0, square_size/2)]
        elif usage == 'bottom':
            box_corners = [box_se, box_ne, box_nw, box_sw]       # a,b,c,d
            target_segment = [middle, middle + complex(0, -square_size/2)]
        elif usage == 'left':
            box_corners = [box_sw, box_se, box_ne, box_nw]       # a,b,c,d
            target_segment = [middle, middle + complex(square_size/2, 0)]
        elif usage == 'right':
            box_corners = [box_ne, box_nw, box_sw, box_se]       # a,b,c,d
            target_segment = [middle, middle + complex(-square_size/2, 0)]
        else:
            raise Exception("OPTION: "+usage)
        #
        mapped_vectors = path_to_polar(image,        # Main call
                                          path_vectors  = path,
                                          base_points   = box_corners,
                                          target_points = target_segment,
                                          central_angle = central_angle_radians,
                                          use_simple    = use_simple,
                                          )
        mapped_vectors.name = path.name+'|polar('+str(central_angle)+ ', '+usage+')'
    else: # inverse_mapping: never
        if usage == 'top':
            base_segment = [middle, middle + complex(0, square_size/2)]
            box_corners = [box_nw, box_sw, box_se, box_ne]       # a,b,c,d
        elif usage == 'bottom':
            base_segment = [middle, middle + complex(0, -square_size/2)]
            box_corners = [box_se, box_ne, box_nw, box_sw]       # a,b,c,d
        elif usage == 'left':
            base_segment = [middle, middle + complex(square_size/2, 0)]
            box_corners = [box_sw, box_se, box_ne, box_nw]       # a,b,c,d
        elif usage == 'right':
            base_segment = [middle, middle + complex(-square_size/2, 0)]
            box_corners = [box_ne, box_nw, box_sw, box_se]       # a,b,c,d
        else:
            raise Exception("OPTION: "+usage)
        #
        mapped_vectors = path_polar_inverse(image,        # Main call
                                          path_vectors  = path,
                                          base_points   = base_segment,
                                          target_points = box_corners,
                                          central_angle = central_angle_radians,
                                          use_simple    = use_simple,
                                          )
        mapped_vectors.name = path.name+'|polar('+str(central_angle)+ ', '+usage+')'
    pdb.gimp_image_undo_group_start(image) # Draw
    mapped = gimp_draw_vectors_object(image, mapped_vectors, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return mapped_vectors

# -------------
# path_to_polar
# -------------
# Do the main work of path_to_polar_main except the drawing.
# Args:
# - image
# - path_vectors:  gimp.Vectors (path to be mapped)
# - base_points:   [complex,complex,complex,complex] ([a,b,c,d], ad->middle)
# - target_points: [complex,complex] ([middle, image point of (b+c)/2)
# - central_angle: float (radians)
# - use_simple:    integer (>=0) 
# Returns:
# - gimp.Vectors
def path_to_polar(image,
                    path_vectors,
                    base_points,
                    target_points,
                    central_angle,
                    use_simple,
                    ):
    # Map path by affine map where p,q,c -> 0,1,1+i/2:
    z0, z1, z2 = 0j, 1+0j, 1+1j/2
    a,b,c,d = base_points # complex
    p,q = (a+d)/2, (b+c)/2
    h1_path = map_path_by_affine(image,    #  h1(path)
                                 path_vectors,
                                 [p,q,c],
                                 [z0,z1,z2],
                                 reversed_base=False
                                 )
    h1_path.name = 'h1'
    #gimp_draw_vectors_object(image, h1_path, visible=True)
    # Map h1_path with the complex map F:x+iy -> x*exp(i*theta*y):
    shaping_input = dict()
    shaping_input['case']       = 'path by polar'
    shaping_input['use_simple'] = use_simple
    shaping_input['typical arc size'] = 1 # Moving in complex plane around origo
    #
    raw_pt = make_raw_polar_map(central_angle, Gimp=True)
    shaping_input['plane transformation'] = raw_pt
    F_h1_path = shape_path(image,              # F(h1(path))
                           h1_path,
                           shaping_input
                           )
    F_h1_path.name = 'F_h1'
    #gimp_draw_vectors_object(image, F_h1_path, visible=True)
    # Map path F(h1(path)) with the affine map (direct similitude)
    # where 0,1 -> target_points
    shaped_final = map_path_by_affine(         # h2(F(h1(path)))
                       image,
                       F_h1_path,
                       [z0,z1],
                       target_points,
                       reversed_base=False   
                       ) 
    return shaped_final


# ------------------
# path_polar_inverse
# ------------------
# Do the main work of path_to_polar_main except the drawing.
# Args:
# - image
# - path_vectors:  gimp.Vectors (path to be mapped)
# - base_points:   [complex,complex,complex,complex] ([a,b,c,d], ad->middle)
# - target_points: [complex,complex] ([middle, image point of (b+c)/2)
# - central_angle: float (radians)
# - use_simple:    integer (>=0) 
# Returns:
# - gimp.Vectors
def path_polar_inverse(image,
                       path_vectors,
                       base_points,
                       target_points,
                       central_angle,
                       use_simple,
                       ):
    a,b,c,d = target_points # complex
    p,q = (a+d)/2, (b+c)/2
    # Map path by affine map where base_points -> 0,1:
    h1_path = map_path_by_affine(         # h1(path)
                       image,
                       path_vectors,
                       base_points,
                       [0j, 1+0j],
                       reversed_base=False   
                       ) 
    h1_path.name = 'h1'
    #gimp_draw_vectors_object(image, h1_path, visible=True)
    # Map h1_path with the inverse of the complex map x+iy -> x*exp(i*theta*y):
    shaping_input = dict()
    shaping_input['case']       = 'path by polar inverse'
    shaping_input['use_simple'] = use_simple
    shaping_input['typical arc size'] = 1 # Moving in complex plane around origo
    #
    raw_pt = make_raw_inverse_polar_map(central_angle, Gimp=True)
    shaping_input['plane transformation'] = raw_pt
    F_h1_path = shape_path(image,              # F(h1(path))
                           h1_path,
                           shaping_input
                           )
    F_h1_path.name = 'F_h1'
    #gimp_draw_vectors_object(image, F_h1_path, visible=True)
    # Map path by affine map where 0,1,1+i/2 -> p,q,c:
    z0, z1, z2 = 0j, 1+0j, 1+1j/2
    shaped_final = map_path_by_affine(image,    #  h1(path)
                                 F_h1_path,
                                 [z0,z1,z2],
                                 [p,q,c],
                                 reversed_base=False
                                 )
    return shaped_final



#==============================================================
#================  Affine map registration  ====================
#==============================================================
versionnumber = "0.24"

procedure_name  = "affine_transformation"
procedure_blurb = ("Transform a path by the affine map which sends"
                   +"\nthe Base to the Target."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Transform a path by an affine map."
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
procedure_label = "Affine map"
image_types = "*"

procedure_function = affine_transformation_main

menupath = '<Vectors>/Tools/Transformations'

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "vectors_object", "Path to transform", None),
      (PF_VECTORS, "base_vectors",
                   ("Base (path with one to three anchors in one stroke)"
                    +"\n* One anchor => a translation"
                    +"\n* Two anchors => a direct similitude"
                    +"\n* Three anchors => a general affine map"
                    ),
                   None),
      (PF_VECTORS, "target_vectors",
                   ("Target (path with at least as many anchors in one stroke as Base)"
                   +"\n* Target is the image of Base in the affine map"
                   ),
                   None),
        (PF_BOOL, "reversed_base", "Use reversed Base?", False),
    ],
    [
        (PF_VECTORS, "vectors", "Mapped by affine map"),
    ],
    procedure_function,
    menu=menupath)



#==============================================================
#=============  Bezier path registration  =================
#==============================================================
versionnumber = "0.24"

procedure_name  = "bezier_path_transformation"
procedure_blurb = ("Transform a path by a conformal map"
                   +"\nconstructed from a Bezier arc."
                   #+"\n(May be slow; please have patience.)"
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Transform a path by a Bezier arc."
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
#procedure_label = "Transformation constructed from a Bezier arc"
procedure_label = "Map shaped by a Bezier arc"
image_types = "*"

procedure_function = bezier_path_main

menupath = '<Vectors>/Tools/Transformations'

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path to be transformed", None),
      (PF_BOOL, "treat_as_linefigure", "Treat the path as consisting of straight line segments (a line figure)", False),
      (PF_VECTORS, "base",
                       ("Base (path with two anchors in one stroke)"
                       +"\n* The end points will be sent to the Target."
                       ),
                       None),
      (PF_VECTORS, "target",
                       ("Target (path with two anchors in one stroke)"
                       +"\n* This is where the end points of the Base will be sent."
                       ),
                       None),
      (PF_VECTORS, "shaper",
                      ("Shaper (path with two anchors in one stroke)"
                       +"\n* The Bezier arc to define the shape of the transformation."
                       ),
                       None),
      (PF_BOOL, "use_reversed_base", "Use reversed Base?", False),
      (PF_BOOL, "use_reversed_shaper", "Use reversed Shaper?", False),
      (PF_STRING, "shape_string", ("Strength of shaping (recommended 0..1)."
                                   +"\n* Format: a float in Python syntax;"
                                   +"\n      or floats listed, separated by commas;"
                                   +"\n      or valid Python tuple or list of floats."
                                  ),
                      "1."),
      (PF_FLOAT, "tweak_a", "A: Tweak the shape of the result", 0.),
      (PF_FLOAT, "tweak_b", "B: Another tweak (must differ from A)", 1.),
      (PF_INT, "use_simple", ("Choose algorithm (integer 0..8)."
                               +"\n*   0 => Best but slow."
                               +"\n*   1 => Simple algorithm: quick but inaccurate."
                               +"\n*   2..8 => Simple algorithm with ever more control points."
                               ),
                        0),
    ],
    [
        (PF_VECTORS, "merged_vectors", "Shaped line figure"),
    ],
    procedure_function,
    menu=menupath)


#==============================================================
#=============  Moebius map registration  =================
#==============================================================
versionnumber = "0.24"

#############  Register moebius_main: no control of poles  ############
procedure_name  = "moebius_path_transformation"
procedure_blurb = ("Transform a path by a Moebius map."
                   #+"\n(May be slow; please have patience.)"
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Transform a path by a Moebius map."
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
procedure_label = "Moebius map"
image_types = "*"

procedure_function = moebius_main

menupath = '<Vectors>/Tools/Transformations'

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path to be transformed", None),
      (PF_VECTORS, "base",
                       ("Base (path with three anchors in one stroke)"
                       +"\n* The anchors will be sent to the anchors of the Target."
                       ),
                       None),
      (PF_VECTORS, "target",
                       ("Target (path with three anchors in one stroke)"
                       +"\n* This is where the anchors of the Base will be sent."
                       ),
                       None),
      (PF_RADIO, "base_permutation", "Permutation of Base anchors:",
                       '012',
                       (("012", '012'),
                        ("120",'120'),
                        ("201",'201'),
                        ("210",'210'),
                        ("102",'102'),
                        ("021",'021')
                       )),
      (PF_BOOL, "mark_poles", "Mark the pole and the inverse pole if located inside the window?", False),
      (PF_INT, "use_simple", ("Choose algorithm (integer 0..8)."
                               +"\n*   0 => Best but slow."
                               +"\n*   1 => Simple algorithm: quick but inaccurate."
                               +"\n*   2..8 => Simple algorithm with ever more control points."
                               ),
                        0),
    ],
    [
        (PF_VECTORS, "vectors", "Shaped path"),
    ],
    procedure_function,
    menu=menupath)


#############  Register moebius_1_main: control of pole  ############

procedure_name  = "moebius_path_transformation_pole"
procedure_blurb = ("Transform a path by a Moebius map with control of the pole."
                   #+"\n(May be slow; please have patience.)"
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Transform a path by a Moebius map with control of the pole."
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
procedure_label = "Moebius: Set the pole"
image_types = "*"

procedure_function = moebius_1_main

menupath = '<Vectors>/Tools/Transformations/Moebius map with control of poles'

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path to be transformed", None),
      (PF_VECTORS, "base",
                       ("Base (path with two anchors in one stroke)"
                       +"\n* The end points will be sent to the Target."
                       ),
                       None),
      (PF_VECTORS, "target",
                       ("Target (path with two anchors in one stroke)"
                       +"\n* This is where the end points of the Base will be sent."
                       ),
                       None),
      (PF_BOOL, "use_reversed_base", "Use reversed Base?", False),
      (PF_VECTORS, "pole",
                       ("Pole (path with one anchor)"
                       +"\n* The point to be sent to infinity."
                       ),
                       None),
      (PF_BOOL, "mark_inverse_pole", "Mark the inverse pole if located inside the window?", False),
      (PF_INT, "use_simple", ("Choose algorithm (integer 0..8)."
                               +"\n*   0 => Best but slow."
                               +"\n*   1 => Simple algorithm: quick but inaccurate."
                               +"\n*   2..8 => Simple algorithm with ever more control points."
                               ),
                        0),
    ],
    [
        (PF_VECTORS, "vectors", "Shaped path"),
    ],
    procedure_function,
    menu=menupath)


#############  Register moebius_2_main: control of inverse pole  ############

procedure_name  = "moebius_path_transformation_inverse_pole"
procedure_blurb = ("Transform a path by a Moebius map with control of the inverse pole."
                   #+"\n(May be slow; please have patience.)"
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Transform a path by a Moebius map with control of the inverse pole."
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
procedure_label = "Moebius: Set the inverse pole"
image_types = "*"

procedure_function = moebius_2_main

menupath = '<Vectors>/Tools/Transformations/Moebius map with control of poles'

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path to be transformed", None),
      (PF_VECTORS, "base",
                       ("Base (path with two anchors in one stroke)"
                       +"\n* The end points will be sent to the Target."
                       ),
                       None),
      (PF_VECTORS, "target",
                       ("Target (path with two anchors in one stroke)"
                       +"\n* This is where the end points of the Base will be sent."
                       ),
                       None),
      (PF_BOOL, "use_reversed_base", "Use reversed Base?", False),
      (PF_VECTORS, "inverse_pole",
                       ("Inverse pole (path with one anchor)"
                       +"\n* The point where the infinity will be mapped."
                       ),
                       None),
      (PF_BOOL, "mark_pole", "Mark the pole if located inside the window?", False),
      (PF_INT, "use_simple", ("Choose algorithm (integer >= 0)."
                               +"\n*   0 => Best but slow."
                               +"\n*   1 => Simple algorithm: quick but inaccurate."
                               +"\n* >1 => Simple algorithm with more control points."
                               ),
                        0),
    ],
    [
        (PF_VECTORS, "vectors", "Shaped path"),
    ],
    procedure_function,
    menu=menupath)


#############  Register moebius_3_main: control of pole and inverse pole  ############

procedure_name  = "moebius_path_transformation_both_poles"
procedure_blurb = ("Transform a path by a Moebius map with control of both poles."
                   #+"\n(May be slow; please have patience.)"
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Transform a path by a Moebius map with control of both poles."
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
procedure_label = "Moebius: Set both the pole and the inverse pole"
image_types = "*"

procedure_function = moebius_3_main

menupath = '<Vectors>/Tools/Transformations/Moebius map with control of poles'

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path to be transformed", None),
      (PF_VECTORS, "base",
                       ("Base (path with one anchor)"
                       +"\n* This point will be sent to the Target."
                       ),
                       None),
      (PF_VECTORS, "target",
                       ("Target (path with one anchor)"
                       +"\n* This is where the Base will be sent."
                       ),
                       None),
      (PF_VECTORS, "pole",
                       ("Pole (path with one anchor)"
                       +"\n* The point to be sent to infinity."
                       ),
                       None),
      (PF_VECTORS, "inverse_pole",
                       ("Inverse pole (path with one anchor)"
                       +"\n* The point where the infinity will be mapped."
                       ),
                       None),
      (PF_INT, "use_simple", ("Choose algorithm (integer >= 0)."
                               +"\n*   0 => Best but slow."
                               +"\n*   1 => Simple algorithm: quick but inaccurate."
                               +"\n* >1 => Simple algorithm with more control points."
                               ),
                        0),
    ],
    [
        (PF_VECTORS, "vectors", "Shaped path"),
    ],
    procedure_function,
    menu=menupath)



#==============================================================
#=============  Bezier16 path registration  ================
#==============================================================
versionnumber = "0.24"


##############  bezier16_path advanced registration  ###############
procedure_name  = "bezier16_path_transformation"
procedure_blurb = ("Transform a path by a map"
                   +"\nconstructed from a Bezier arc quadrilateral."
                   +"\nAdvanced version."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Transform a path by a Bezier arc quadrilateral."
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
procedure_label = "Map shaped by a Bezier arc quadrilateral (advanced)"
image_types = "*"

procedure_function = bezier16_path_advanced_main

menupath = '<Vectors>/Tools/Transformations'

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The source path to be transformed", None),
      (PF_VECTORS, "shaper",
                      ("Shaper (closed path with four anchors)"
                       +"\n* The Bezier arc quadrilateral to define the shape of the transformation"
                       ),
                       None),
      (PF_OPTION,  'ref_box_option','Reference box (perhaps enclosing the source path).  Options:',
                   0,
                  [case[0] for case in reference_box_options_advanced]), # Descriptions of cases
      (PF_VECTORS, "base",
                      ">>> If you took option 5, choose the path here (four anchors):",
                       None),
      (PF_OPTION,  'target_box_option','Target box (destination of the transformed path).  Options:',
                   0,
                  [case[0] for case in target_box_options]), # Descriptions of cases
      (PF_VECTORS, "target",
                      ">>> If you took option 3, choose the path here (four anchors):",
                       None),
      (PF_OPTION,  'target_usage_option',
                    'How do you want to apply the target box?  Options:',
                    0,
                  [case[0] for case in target_usage_options]), # Descriptions of cases
      #(PF_BOOL, "horizontal_bottom", "Rotate the transformed path to have horizontal bottom?", False),
      (PF_BOOL, "linearize_shaper", "Linearize straight edges in Shaper?", True),
      (PF_FLOAT, "adjust_shaping", "Adjust the form of shaping (float, perhaps around 0..1)", 0.),
      (PF_FLOAT, "strength_shaping", "Strength of shaping (float, usually 0..1)", 1.),
      #(PF_INT, "permute_base", "Permute the Base anchors (integer 0..1 or 0..7)", 0),
      #(PF_INT, "permute_target", "Permute the Target anchors (integer 0..1 or 0..7)", 0),
      #(PF_INT, "permute_shaper", "Permute the Shaper anchors (integer 0..7)", 0),
      (PF_INT, "use_simple", ("Choose algorithm (integer 0..8)."
                               +"\n*   0 => Best but slow."
                               +"\n*   1 => Simple algorithm: quick but inaccurate."
                               +"\n*   2..8 => Simple algorithm with ever more control points."
                               ),
                        4),
    ],
    [
        (PF_VECTORS, "mapped_vectors", "Shaped path"),
    ],
    procedure_function,
    menu=menupath)


##############  bezier16_path easy registration  ###############
procedure_name  = "bezier16_path_easy_transformation"
procedure_blurb = ("Transform a path by a map"
                   +"\nconstructed from a Bezier arc quadrilateral."
                   +"\nEasy version."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Transform a path by a Bezier arc quadrilateral (easy)."
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
procedure_label = "Map shaped by a Bezier arc quadrilateral (easy)"
image_types = "*"

procedure_function = bezier16_path_easy_main

menupath = '<Vectors>/Tools/Transformations'

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The source path to be transformed", None),
      (PF_VECTORS, "shaper",
                      ("Shaper (closed path with four anchors)"
                       +"\n* The Bezier arc quadrilateral to define the transformation"
                       ),
                       None),
       (PF_OPTION,  'ref_box_option','Reference box (perhaps enclosing the source path).  Options:',
                  0,
                  [case[0] for case in reference_box_options_easy]), # Descriptions of cases
      (PF_BOOL, "horizontal_bottom", "Rotate the transformed path to have horizontal bottom?", True),
      (PF_BOOL, "linearize_shaper", "Linearize straight edges in Shaper?", True),
      (PF_FLOAT, "adjust_shaping", "Adjust the form of shaping (float, perhaps around 0..1)", 0.),
      (PF_FLOAT, "strength_shaping", "Strength of shaping (float, usually 0..1)", 1.),
      (PF_INT, "use_simple", ("Choose algorithm (integer 0..8)."
                               +"\n*   0 => Best but slow."
                               +"\n*   1 => Simple algorithm: quick but inaccurate."
                               +"\n*   2..8 => Simple algorithm with ever more control points."
                               ),
                        4),
    ],
    [
        (PF_VECTORS, "mapped_vectors", "Shaped path"),
    ],
    procedure_function,
    menu=menupath)





#==============================================================
#=============  Fit path registration  ================
#==============================================================
versionnumber = "0.24"

###  Fit path to triangle ###
procedure_name  = "fit_path_to_triangle"
procedure_blurb = ("Fit a path in a triangle"
                   #+"\nor the triangle around the path."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Fit a path in a triangle."
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
procedure_label = "Fit a path in a triangle"
image_types = "*"

procedure_function = fit_path_to_triangle_main

menupath = '<Vectors>/Tools/Transformations'

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path_vectors", "The path", None),
      (PF_VECTORS, "triangle_vectors", "A triangle (path with three anchors)", None),
    ],
    [
        (PF_VECTORS, "vectors", "Fitted path"),
    ],
    procedure_function,
    menu=menupath)


###  Fit path to quadrangle ###
procedure_name  = "fit_path_to_quadrangle"
procedure_blurb = ("Fit a path in a convex quadrangle (distorted)"
                   #+"\nor the quadrangle around the path (maintaining directions of sides)."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Fit a path (distorted) in a convex quadrangle."
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
procedure_label = "Fit a path in a convex quadrangle"
image_types = "*"

procedure_function = fit_path_to_quadrangle_main

menupath = '<Vectors>/Tools/Transformations'

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path_vectors", "The path", None),
      (PF_VECTORS, "quadrangle_vectors", "A convex quadrangle (path with four anchors)", None),
      (PF_OPTION,  'ref_box_option','Reference box.  Options:',
                   0,
                  [case[0] for case in reference_box_options]), # Descriptions of cases
      (PF_VECTORS, "ref_vectors",
                      ">>> If you took option 3, choose the path here (four anchors):",
                       None),
      (PF_INT, "use_simple", ("Choose algorithm (integer 0..8)."
                               +"\n*   0 => Best but slow."
                               +"\n*   1 => Simple algorithm: quick but inaccurate."
                               +"\n*   2..8 => Simple algorithm with ever more control points."
                               ),
                        4),
    ],
    [
        (PF_VECTORS, "vectors", "Fitted path"),
    ],
    procedure_function,
    menu=menupath)



#==============================================================
#=============  Exp path registration  =================
#==============================================================
versionnumber = "0.24"

#  Register map_path_by_exp_main
procedure_name  = "exp_path_transformation"
procedure_blurb = ("Transform a path by complex exponential map."
                   #+"\n(May be slow; please have patience.)"
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Transform a path by complex exponential map."
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
procedure_label = "Exponential map"
image_types = "*"

procedure_function = map_path_by_exp_main

menupath = '<Vectors>/Tools/Transformations'

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path to be transformed", None),
      (PF_OPTION,  'center_option',
                    'Center - the center of the transformation',
                    0,
                  [case[0] for case in exp_center_options]), # Descriptions of cases
      (PF_OPTION,  'target_option',
                    'Target box - where the transformed path will be fitted',
                    0,
                  [case[0] for case in exp_trg_box_options]), # Descriptions of cases
      (PF_FLOAT, "alpha", "alpha - central angle in degrees", 90),
      (PF_FLOAT, "gamma", "gamma - polar slope angle in degrees, -85..85", 0),
      (PF_BOOL, "mark_center", "Mark the center? (Makes a one-anchor path.)", False),
      (PF_INT, "use_simple", ("Choose algorithm (integer 0..8)."
                               +"\n*   0 => Best but slow."
                               +"\n*   1 => Simple algorithm: quick but inaccurate."
                               +"\n*   2..8 => Simple algorithm with ever more control points."
                               ),
                        4),
    ],
    [
        (PF_VECTORS, "vectors", "Shaped path"),
    ],
    procedure_function,
    menu=menupath)



#==============================================================
#==========  Path to polar coords registration  ===============
#==============================================================
versionnumber = "0.24"

#  Register map_path_by_exp_main
procedure_name  = "path_polar_coords"
procedure_blurb = ("Convert a path to polar coordinates."
                   #+"\n(May be slow; please have patience.)"
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Convert a path to polar coordinates."
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
procedure_label = "To polar coordinates"
image_types = "*"

procedure_function = path_to_polar_main

menupath = '<Vectors>/Tools/Transformations'

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path to be transformed", None),
      (PF_OPTION,  'edge_to_middle',
                    'Which edge to middle? The opposite edge will go to outside.',
                    0,
                  [case[0] for case in edge_options]), # Descriptions of cases
      (PF_FLOAT, "central_angle", "Central angle in degrees", 360),
      (PF_OPTION, 'ref_box_option','Reference box.  Options:',
                   0,
                  [case[0] for case in polar_box_options]), # Descriptions of cases
      #(PF_FLOAT, "offset_x", "Offset x", 0),
      #(PF_FLOAT, "offset_y", "Offset y", 0),
      (PF_INT, "use_simple", ("Choose algorithm (integer 0..8)."
                               +"\n*   0 => Best but slow."
                               +"\n*   1 => Simple algorithm: quick but inaccurate."
                               +"\n*   2..8 => Simple algorithm with ever more control points."
                               ),
                        4),
    ],
    [
        (PF_VECTORS, "vectors", "Shaped path"),
    ],
    procedure_function,
    menu=menupath)


main()
