#!/usr/bin/env python
#
# Gimp plugin to warp a path locally
#
# History:
# v0.1: 2021-01-18: Experimental plugin: Grow path locally
# v0.2: 2021-03-22: New option: Shrink path locally
# v0.3: 2021-03-22: New option: Swirl path locally
# v0.4: 2021-03-24: New option: Delete outside
#                   First published version

# (c) Markku Koppinen 2020
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'


from __future__ import division, print_function 
from gimpfu import *                            
from math import *                              
from functools import partial         
from copy import deepcopy


#==============================================================
#             class BCurve       
#==============================================================
# Class: BCurve.

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

RowVector = complex

class BCurve(object):
    """Data structures to handle Gimp vectors objects by means of individual
    Bezier arcs.
    By a "Bezier arc" we mean here a Bezier curve determined by 4 control points
    p0,p1,p2,p3.
    This class contains data structures needed to do computations on
    a Bezier curve which are done arc by arc.
    To this end, Gimp data structures
    - gimp.Vectors
    - gimp.VectorsBezierStroke
    are here represented as data structures (classes)
    - GimpVectors
    - GimpStroke
    - BezierCurve
    - BezierArc
    - ControlPoint = complex
    with the aim that
    - each control point [x,y] is represented as an instance of ControlPoint;
    - each Bezier arc (four control points, see above) is represented as an instance
      of BezierArc.
    """
    
    ControlPoint = RowVector # =complex
    
    class GimpVectors(object):
        """Essentially same data as in a gimp.Vectors object except that
        strokes are instances of GimpStroke rather than
        gimp.VectorsBezierStroke.
        Attributes:
        - stroke_list: [GimpStroke]
        - name:        string
        """
        def __init__(self,
                     stroke_list,            # [GimpStroke]
                     name='GimpVectors'      # string
                     ):
            self.stroke_list = stroke_list
            self.name = name
        def __str__(self):
            s = 'GimpVectors '+self.name
            count = 0
            for stroke in self.stroke_list:
                s += '\n  Stroke '+str(count)+': GimpStroke'
                s += '\n    '+str(stroke)
                count += 1
            return s
        def gv2vectors_object(self, image, name=None):
            """Conversion GimpVectors -> gimp.Vectors
            (The converse: vectors_object2gv)
            """
            def cp_list2xy_list(cp_list):
                xy = []
                for cp in cp_list:
                    xy += [cp.real, cp.imag]
                return xy
            if name is None:
                name = self.name
            vectors = pdb.gimp_vectors_new(image, name)
            for stroke in self.stroke_list:
                closed = stroke.closed
                xy_list = cp_list2xy_list(stroke.cp_list)
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                             vectors, 0, len(xy_list), xy_list, closed)
            return vectors
    
    class GimpStroke(object):
        """Essentially same data as in a gimp.VectorsBezierStroke.points,
        except that control points list [x,y,x,y,...] is arranged as
        a list [ControlPoint] by joining successive x,y.
        Attributes:
        - cp_list:     [ControlPoint]
        - closed:      boolean
        - stroke_name: string
        """
        def __init__(self,
                     cp_list,                   # [ControlPoint]
                     closed,                    # boolean
                     stroke_name = 'GimpStroke' # string
                     ):
            self.stroke_name = stroke_name
            self.cp_list = cp_list
            self.closed = closed
        def __str__(self):
            s = self.stroke_name + '; Control points:'
            for cp in self.cp_list:
                s += '\n    '+str(cp)
            s += '\n    closed: '+str(self.closed)
            return s
        def gs2bc(self):
            """Conversion GimpStroke -> BezierCurve
            """
            head = self.cp_list[0]
            tail = self.cp_list[-1]
            inner = self.cp_list[1:-1]
            if len(inner) > 1: # Must be divisible by 4
                ba_list = []
                count = 0
                for i in range(0,len(inner)-1,3):
                    ba_list.append(BCurve.BezierArc(inner[i:i+4], 'arc '+str(count)))
                    count += 1
            elif len(inner) == 1: # Stroke has only one anchor
                ba_list = [BCurve.BezierArc([inner[0]], 'arc 0')]
            else:
                raise Exception("BCurve.GimpStroke.gs2bc: No anchors in stroke?")
            return BCurve.BezierCurve(
                               bezier_arcs = ba_list,
                               head_handle = head,
                               tail_handle = tail,
                               closed = self.closed,
                               curve_name = self.stroke_name
                               )
    
    class BezierCurve(object):
        """BezierCurve is a list of butting Bezier arcs with some extra data.
        Attributes:
        - curve_name:  string
        - bezier_arcs: [BezierArc];
        - head_handle: ControlPoint;
        - tail_handle: ControlPoint;
        - closed:      boolean.
        Note: "Butting" means that for any pair of successive arcs, the last
              control point of the former equals the first control point of
              the latter.
              In initialization no checks are done about this condition.
        """
        def __init__(self,
                     bezier_arcs=[],           # [BezierArc]
                     head_handle=None,         # ControlPoint
                     tail_handle=None,         # ControlPoint
                     closed=False,             # boolean
                     curve_name='BezierCurve', # string
                     ):
            self.curve_name = curve_name
            self.bezier_arcs = bezier_arcs
            if head_handle is None:
                self.head_handle = bezier_arcs[0].cp4[0]
            else:
                self.head_handle = head_handle
            if tail_handle is None:
                self.tail_handle = bezier_arcs[-1].cp4[-1]
            else:
                self.tail_handle = tail_handle
            self.closed = closed
        def __str__(self):
            s = self.curve_name + ': Bezier curve' + '; Bezier arcs:'
            count = 0
            for arc in self.bezier_arcs:
                s += '\n  arc '+ str(count)+ ': ' +str(arc)
                count += 1
            s += '\n  Head handle:    '+str(self.head_handle)
            s += '\n  Tail handle:    '+str(self.tail_handle)
            s += '\n  Closed:'    +str(self.closed)
            return s
        def bc2gs(self):
            """Conversion BezierCurve -> GimpStroke
            """
            cp_list = [self.head_handle]
            if len(self.bezier_arcs[0].cp4) > 1:
                for ba in self.bezier_arcs:
                    cp_list += ba.cp4[:3]
                cp_list.append(self.bezier_arcs[-1].cp4[-1])
            else:
                # Only one BezierArc and it has cp4 = [p0]
                # (may rise from Gimp stroke with only one anchor).
                cp_list.append(self.bezier_arcs[0].cp4[0])
            cp_list.append(self.tail_handle)
            return BCurve.GimpStroke(cp_list, self.closed, self.curve_name)
        def bc2vectors_object(self, image, name=None):
            """Conversion BezierCurve -> gimp.Vectors
            Makes vectors_object with 1 stroke.
            """
            if name is None:
                name = self.curve_name
            gs = self.bc2gs()             # BCurve.GimpStroke
            gv = BCurve.GimpVectors([gs]) # BCurve.GimpVectors
            return gv.gv2vectors_object(image, name)
    
    class BezierArc(object):
        """Data structure for one Bezier arc: arc determined by 4 control
           points.
        Attributes:
        - cp4:      [p0,p1,p2,p3] where each pi:ControlPoint, or
                    [p0] if only one control point (may rise from
                    a stroke with only one anchor);
        - arc_name: string
        """
        def __init__(self,
                     cp4,                 # cp4=[p0,p1,p2,p3] with pi:ControlPoint
                                          # or [p0]
                     arc_name='BezierArc' # string
                     ):
            self.cp4 = cp4
            self.arc_name = arc_name
        def reverse(self):
            self.cp4.reverse()
        def __str__(self):
            s = 'BezierArc, control points:'
            for cp in self.cp4:
                s += '\n  '+ str(cp)
            return s

# -----------------
# vectors_object2gv
# -----------------
# Conversion gimp.Vectors -> GimpVectors
# Args:
# - vectors_object: gimp.Vectors (Gimp vectors object)
# Returns:
# - GimpVectors
#
# Note: The inverse conversion is GimpVectors.gv2vectors_object(...).
#       See also BezierCurve.bc2vectors_object(...).
def vectors_object2gv(vectors_object):
    def xy_list2cp_list(xy_list): # xy_list = [x,y,x,y,...,]
        cp = []
        for i in range(0,len(xy_list),2):
            cp.append(BCurve.ControlPoint(xy_list[i], xy_list[i+1]))
        return cp
    stroke_list = []
    count = 0
    for stroke in vectors_object.strokes:
        xy_list = stroke.points[0]
        closed = stroke.points[1]
        cp_list = xy_list2cp_list(xy_list)
        stroke_name = vectors_object.name + ' stroke '+str(count)
        stroke_list.append(BCurve.GimpStroke(cp_list, closed, stroke_name))
        count += 1
    return BCurve.GimpVectors(stroke_list, vectors_object.name)


#==============================================================
#             classes Support, SupportedFamily etc
#==============================================================
# Classes: Support, ComplexValuedFunction, SupportedFamily, ClosedInterval,
# BezierFunction.
# The algorithm to find an approximate Bezier arc by means of one-parameter
# families of Bezier arcs.

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

class NoSupportError(BaseException):
    """When initialization of Support fails, the points causing
    the failure are intermitted in NoSupportError as
    - self.msg:   string
    - self.error: integer
    - self.points: [complex]
    """
    def __init__(self, msg, error, points):
        self.msg    = msg
        self.error  = error
        self.points = points
    def __str__(self):
        return str(self.msg)

class Support(object):
    """Data structure to store "support" for certain Bezier arcs.
    By a "support" we mean non-collinear points S0,S1,S2,S3 where it is
    assumed that S0 != S1 and S3 != S2.
    This "supports" a Bezier arc with control points p0,p1,p2,p3 if
        - p0 = S0
        - p1 is on the line through S0,S1
        - p2 is on the line through S3,S2
        - p3 = S3
    Attributes:
    - support_points = [S0,S1,S2,S3] where Si:[complex]
    - case:string: one of
      'N':  S01,S32 not parallel; (where S01 = -S0+S1, S32 = -S3+S2)
      'P':  S01,S32     parallel.
      'sloppy': the support is used just to store points S0,S1,S2,S3
    - bezier_area: float (signed area between Bezier arc and chord where
                          the support points S0,S1,S2,S3 are taken as
                          control points)
    Methods:
    - __init__
    - coefficients2bezier_arc
    - sum_dist2_to_targets   (measure: sum of squared distances from middle 
                              point of chord S0S3 to target points)
    """
    def __init__(self, support_points, sloppy=False): # If sloppy 
        def get_case(S0,S1,S2,S3):
            #ZERO = 1e-10
            ZERO = 1e-16
            S01 = -S0+S1
            S32 = -S3+S2
            S03 = -S0+S3
            # CKOE
            #if (vdot(S01,S01) < ZERO) or (vdot(S32,S32) < ZERO):
            if (vdot(S01,S01) < ZERO) or (vdot(S32,S32) < ZERO):
                m1 = "SupportedFamily: "
                m2 = "The points to determine tangents are coincident (or too close)"
                #raise Exception(m1+m2)
                raise NoSupportError(msg = m1+m2,
                                     error = 1,
                                     points = [S0,S1,S2,S3])
            if collinear([S1,S0,S3,S2]):
                m1 = "SupportedFamily: "
                m2 = "The points to determine tangents are collinear (or too close)"
                #raise Exception(m1+m2)
                raise NoSupportError(msg = m1+m2,
                                     error = 2,
                                     points = [S0,S1,S2,S3])
            S01xS32 = cross(S01,S32)
            #S03xS01  = cross(S03,S01)
            #S03xS32  = cross(S03,S32)
            if abs(S01xS32) > ZERO: # Handles not parallel
                case = 'N'
            else:                   # Handles parallel
                case = 'P'
            return case
        if len(support_points) == 4:
            S0,S1,S2,S3 = support_points
        else:
            m1 = "SupportedFamily: "
            m2 = "Bad number of support points:"+len(support_points)
            raise NoSupportError(msg = m1+m2,
                                 error = 3,
                                 points = support_points)
        if not sloppy:
            case = get_case(S0,S1,S2,S3) # May raise NoSupportError
        else: 
            case = 'sloppy'
        self.support_points = [S0,S1,S2,S3]
        self.case = case
        self.bezier_area = BezierFunction([S0,S1,S2,S3]).signed_area()
    def sum_dist2_to_targets(self, targets): # targets = [[x,y],[x,y],...]
        def distance2(x,y):
            # CKOE
            #xy = points2vector(x,y)
            #return xy[0]**2 + xy[1]**2
            xy = -x+y
            return xy.real**2 + xy.imag**2
        S0,S1,S2,S3 = self.support_points
        # CKOE
        #middle = [(S0[0]+S3[0])/2, (S0[1]+S3[1])/2] # Middle point of chord
        middle = (S0+S3)/2 # Middle point of chord
        return sum([distance2(middle, target) for target in targets])
    # ---------------------------
    # coefficients2bezier_arc
    # ---------------------------
    # Given xi,eta, compute control points p0,p1,p2,p3 of a supported Bezier
    # arc such that |-p0+p1| = xi*|-S0+S1| and |-p3+p2| = eta*|-S3+S2|.
    # Return BezierFunction.
    def coefficients2bezier_arc(self,xi,eta):
        S0,S1,S2,S3 = self.support_points
        S01 = -S0+S1
        S32 = -S3+S2
        p01 = xi*S01
        p32 = eta*S32
        #
        p0 = S0
        p3 = S3
        p1 = p0 + p01
        p2 = p3 + p32
        return BezierFunction([p0,p1,p2,p3])
    def __str__(self):
        s = 'Support, case '+self.case
        s += '\n  support_points: '+str(self.support_points)
        return s


class ComplexValuedFunction(object):
    """Function R -> C.
    Initialization arguments:
    - f:    R->C:         the function;
    - fdot: None or R->C: None or the derivative of f
    - use_ad_hoc_fdot: boolean: If fdot=None then
                       - if use_ad_hoc_fdot=True, insert in place of fdot
                         a simple ad-hoc substitute algorithm;
                       - if use_ad_hoc_fdot=False, leave fdot=None;
                         this is likely to cause an error somewhere?
    - interval:       [a,b]: a,b are float or None; None denotes +-infinity
                      (default is [None,None], meaning whole R);
    Attributes:
    - function:   R->C
    - derivative: None or R->C
    - interval:   ClosedInterval
    Methods:
    - __init__
    - minimize
    - closest_point
    """
    def __init__(self,
                 f,
                 fdot=None,
                 use_ad_hoc_fdot=True, # applies if fdot=None
                 interval=[None,None]
                 ):
        self.function   = f
        lo, hi = interval
        if not ((lo is None) or (hi is None)):
            if lo > hi:
                lo,hi = hi,lo
        self.interval = ClosedInterval(lo,hi)
        if fdot is not None:
            self.derivative = fdot
        elif use_ad_hoc_fdot:
            def der(x):
                delta = 0.001
                try:
                    x1 = self.interval.closest(x-delta)
                    x2 = self.interval.closest(x+delta)
                    a1,b1 = f(x1)
                    a2,b2 = f(x2)
                except Exception as e:
                    raise Exception("ComplexValuedFunction: "+str(e))
                return [(a2-a1) / (x2-x1), (b2-b1) / (x2-x1)]
            self.derivative = der
        else:
            self.derivative = None
    # --------
    # minimize
    # --------
    # Minimize key(x) where x is the parameter for the curve.
    # Args:
    # - self:            ComplexValuedFunction;
    # - key:             callable: float -> float;
    # - search_interval: [float,float];
    # - guess:           float;
    # - samples:         integer;
    # - rec_depth:       integer;
    # - sloppy: Boolean.
    # Returns:
    # - x_min:   float
    # - key_min: float
    def minimize(self,
                 key,
                 search_interval=None,
                 guess=None,
                 samples=10,
                 rec_depth=0,
                 sloppy=False
                 ):
        if search_interval is None:
            lo,hi = ComplexValuedFunction.get_search_interval(self, key, guess, sloppy)
        else:
            lo, hi = min(search_interval), max(search_interval)
        lo, hi = self.intersect_interval(lo, hi)
        accuracy = 1e-3
        MIN_INTERVAL = max(1e-4, accuracy * (hi-lo)) # ?
        MAX_REC_DEPTH = 10
        # Create list of x's
        xs = sorted(ClosedInterval(lo,hi).samples(samples, include_endpoints=True))
        xlo, xhi = xs[0], xs[-1]
        if (rec_depth > MAX_REC_DEPTH) or (xhi-xlo < MIN_INTERVAL):
            x = (xlo+xhi)/2
            key_value = key(x)
            return x, key_value
        # Gather indices where local minima seem to occur:
        min_indices = []
        prev_key = key(xs[0])
        curr_key  = key(xs[1])
        for i in range(1,len(xs)-1): # i is index of current x
            next_key = key(xs[i+1]) # next x
            if prev_key >= curr_key <= next_key: # Catches minima not at ends of interval
                min_indices.append(i)
            elif i==1 and prev_key <= curr_key:         # Start of interval
                min_indices.append(0)
            elif i==len(xs)-2 and curr_key >= next_key: # End of interval
                min_indices.append(len(xs)-1)
            prev_key, curr_key = curr_key, next_key
        # Recursion: Call the routine for subintervals containing
        # the tentative minima:
        sub_x_keys = []
        for j in min_indices:
            e = 1
            low, high = max(j-e,0), min(j+e, len(xs)-1)
            subinterval = [xs[low], xs[high]] # Subinterval for a tentative minimum
            # Recursion:
            sub_x, key_value = self.minimize(key, subinterval,
                                             rec_depth=rec_depth+1)
            sub_x_keys.append([sub_x, key_value])
        # Choose from the list the one with smallest key_value:
        x_min, key_min = min(sub_x_keys, key=lambda d:d[1])
        return x_min, key_min
    # -------------
    # closest_point
    # -------------
    # Find locally closest point to v on the ComplexValuedFunction arc F(t)
    # (self).
    #
    # Args:
    # - self:     ComplexValuedFunction;
    # - v:        point (complex);
    # - search_interval: interval [start,end] for the parameter t;
    # - guess:    float in interval, an initial guess for the parameter value
    #             of the closest point (default=0.5, which is also used if
    #             the guess is not in the interval);
    # - samples: integer;
    # - sloppy: Boolean.
    # Returns:
    # - t0: float:   the parameter of a locally closest point;
    # - b0: complex: the point on the curve with parameter value t0;
    # - d0: float:   the squared distance between v and b0.
    #
    # Notes:
    # 1. If the distance has several minima, the routine may not find the
    #    globally best but only a locally best in the vicinity of the guess.
    #    Also, the formula is derived by means of linear approximation
    #    and it assumes that the changes in the parameter value are small.
    #    Thus, it is crucial that the initial guess is good!
    # 2. Possibly an end point of the interval is returned.
    # 3. The search interval is intersected by the defining interval of self.
    def closest_point(self,
                      v,
                      search_interval=None,
                      guess=None,
                      samples=10,
                      sloppy=False
                      ):
        def distance2_to_v(t):
            w = self.function(t)
            return ((v-w)*(v-w).conjugate()).real
        if search_interval is None:
            lo,hi = ComplexValuedFunction.get_search_interval(self,
                                        distance2_to_v,
                                        guess,
                                        sloppy)
        else:
            lo, hi = search_interval
        lo, hi = self.intersect_interval(lo, hi)
        #accuracy=1e-8
        accuracy=1e-3
        #MAX_ROUNDS = 20     # Used to avoid possibility of infinite loop.
        MAX_ROUNDS = 10     # Used to avoid possibility of infinite loop.
        MAX_T_DIFF = accuracy * (hi-lo) # To stop work when nothing
        #                                 much seems to be happening.
        if guess is None:
            guess = (lo + hi)/2.
        if not (lo <= guess <= hi):
            guess = (lo + hi)/2.
        f, fdot = self.function, self.derivative
        if fdot is None:
            raise Exception("ComplexValuedFunction.closest_point: fdot is None")
        guesses = ClosedInterval(lo,hi).samples(samples)
        if not (guess is None):
           guesses.append(self.interval.closest(guess))
        td = [[t, distance2_to_v(t)] for t in guesses]
        smallest = min(td, key=lambda x:x[1])
        t0, d2 = smallest # Use this t0 as guess.
        b0 = f(t0)
        d0 = distance2_to_v(t0)
        step = 1.
        for rnd in range(MAX_ROUNDS):
            # From t0 compute new tentative guess for closest point
            # by linear approximation.
            b0v = b0-v
            bd = fdot(t0)
            numer = vdot(b0v,bd)            # (B(t0)-v).B'(t0)
            denom1 = vdot(bd,bd)            # B'(t0).B'(t0)
            try:
                t1 = t0 - step * numer/denom1
            except ZeroDivisionError:
                # t0 is a cusp, therefore an end point: try to get some reasonable t1
                t1 = t0 + step * (hi-lo)
            t1 = max(lo, min(hi, t1))
            t12 = [t1]
            t_diff = max([t0]+t12) - min([t0]+t12)
            tbd = []
            for tx in t12:
                bx = f(tx)
                dx = distance2_to_v(tx)
                tbd.append([tx,bx,dx])
            tbd_smallest = min(tbd, key=lambda x:x[2]) # smallest wrt d
            if d0 < tbd_smallest[2]: # d0 smallest
                step /= 2
            else:
                t0,b0,d0 = tbd_smallest        
            if t_diff < MAX_T_DIFF:
                break
        return t0,b0,d0
    # -------------------
    # get_search_interval
    # -------------------
    # For a vector valued function self, find a suitable interval for searching for
    # minimal value of key.
    # Args:
    # - pf:    ComplexValuedFunction;
    # - key:   callable: float -> float;
    # - guess: float;
    # - sloppy: Boolean.
    # Returns:
    # - [lo value, hi value]: [float,float]
    def get_search_interval(self, key, guess=None, sloppy=False):
        pf = self
        if sloppy:
            pf_lo, pf_hi = [self.interval.lo, self.interval.hi]
            if not (None in (pf_lo, pf_hi)):
                return [pf_lo, pf_hi]
        MAX_ROUNDS = 100
        point = self.interval.typical_point
        step  = self.interval.typical_step
        #lo, hi = self.interval.lo, self.interval.hi
        if guess is None:
            cur_lo = point
        else:
            cur_lo = min(point, guess)
        cur_key = key(cur_lo)
        for r in range(MAX_ROUNDS):
            next_lo = self.interval.closest(cur_lo - step)
            next_key = key(next_lo)
            found_lo = next_lo
            found_key = next_key
            if next_key > cur_key:
                break
            else:
                cur_lo = next_lo
                cur_key = next_key
        else: # No break occured
            if sloppy:
                pass
                #found_lo = next_lo
            else:
                raise Exception("get_search_interval: No lower end found")
        if guess is None:
            cur_hi = point
        else:
            cur_hi = max(point, guess)
        cur_key = key(cur_hi)
        for r in range(MAX_ROUNDS):
            next_hi = self.interval.closest(cur_hi + step)
            next_key = key(next_hi)
            found_hi = next_hi
            found_key = next_key
            if next_key > cur_key:
                break
            else:
                cur_hi = next_hi
                cur_key = next_key
        else: # No break occured
            if sloppy:
                pass
            else:
                raise Exception("get_search_interval: No upper end found")
        return [found_lo, found_hi]
    # ------------------
    # intersect_interval
    # ------------------
    # Restrict interval [lo,hi] by intersecting with defining interval of self.
    # Args:
    # - lo: float
    # - hi: float
    # Returns:
    # - float
    # - float
    def intersect_interval(self, lo, hi):
        new_closed_interval = ClosedInterval(lo,hi).intersect(self.interval)
        return new_closed_interval.lo, new_closed_interval.hi


class SupportedFamily(Support, ComplexValuedFunction):
    """One-parameter family of Bezier arcs in a given support.
    Attributes:
    - all attributes of Support
    - all attributes of ComplexValuedFunction
    - type: string: one of
            - 'fixed point N';
            - 'fixed point P';
            (- 'fixed area'  does not appear in current version)
    - data: float
            - the fixed point ([float,float]) for type 'fixed point N';
            - the parameter for the fixed point (float) for type
              'fixed point P';
            (- the fixed area_ratio (float) for type 'fixed area')
    Methods:
    - __init__
    - parameter2bezier_arc
    """
    # Internally, the family is a vector valued function (plus some data).
    # From any parameter value x, a Bezier arc is produced by
    # parameter2bezier_arc.
    def __init__(self,
                 support_points,
                 f,
                 fdot=None,
                 use_ad_hoc_fdot=True,
                 interval=[None,None],
                 type=None,
                 data=None):
        Support.__init__(self, support_points)
        try:
            ComplexValuedFunction.__init__(self,
                                           f=f,
                                           fdot=fdot,
                                           use_ad_hoc_fdot=True,
                                           interval=interval
                                           )
        except Exception as e:
            raise Exception("SupportedFamily: "+str(e))
        self.type = type
        self.data = data
    # --------------------
    # parameter2bezier_arc
    # --------------------
    # For a SupportedFamily (self) and a parameter value x, construct a BezierFunction
    # (actually a Bezier arc, four control points):
    #     x -> xi,eta -> BezierFunction
    def parameter2bezier_arc(self,x):
        xi,eta = self.function(x)
        bezier_arc = self.coefficients2bezier_arc(xi,eta)
        return bezier_arc


class ClosedInterval(object):
    """Closed real number interval. Four cases: finite, infinite at one end,
    infinite at both ends.
    Initialization arguments:
    - a: float or None (None denotes -infinity);
    - b: float or None (None denotes +infinity).
    Attributes:
    - type: string: one of  'lo,hi',  '-inf,hi',  'lo,inf',  '-inf,inf'
    - lo:   float or None (None denotes -infinity);
    - hi:   float or None (None denotes +infinity);
    - typical_point: float: a good point to start coarse searching;
    - typical_step:  float: a good starting step in a coarse searching.
    Methods:
    - __init__
    - closest
    - samples
    - intersect
    """
    def __init__(self, a=None, b=None, typical_point=None, typical_step=None):
        STEP = 1.
        NSTEPS = 10
        self.lo, self.hi = a, b
        if not (a is None):
            if not (b is None):
                if a > b:
                    self.lo, self.hi = b, a
                self.type = 'lo,hi'
            else: # b = None meaning infinity
                self.type = 'lo,inf'
        else: # a = None meaning -infinity
            if not (b is None):
                self.type = '-inf,hi'
            else: # b = None meaning infinity
                self.type = '-inf,inf'
        if not(typical_point is None):
            self.typical_point = typical_point
        else:
            if self.type == 'lo,hi':
                self.typical_point = (self.lo + self.hi)/2
            elif self.type == 'lo,inf':
                self.typical_point = self.lo + NSTEPS*STEP
            elif self.type == '-inf,hi':
                self.typical_point = self.hi - NSTEPS*STEP
            elif self.type == '-inf,inf':
                self.typical_point = 0.
        if not(typical_step is None):
            self.typical_step = max(typical_step,0.)
        else:
            if self.type == 'lo,hi':
                self.typical_step = (self.hi - self.lo)/NSTEPS
            else:
                self.typical_step = STEP
    def closest(self, x): # The closest point to x in interval
        if self.type == 'lo,hi':
            return max(self.lo, min(self.hi, x))
        elif self.type == '-inf,hi':
            return min(self.hi, x)
        elif self.type == 'lo,inf':
            return max(self.lo, x)
        else:
            return x
    def samples(self, N=8, include_endpoints=False): # Some points in the interval
        N = max(0,N)
        if self.type == 'lo,hi':
            a,b = self.lo,self.hi
        elif self.type == '-inf,hi':
            a,b = self.hi - N, self.hi
        elif self.type == 'lo,inf':
            a,b = self.lo, self.lo + N
        else:
            a,b = -N/2, N/2
        if N <= 0:
            s = []
        elif N == 1:
            s = [(a+b)/2]
        else:
            n = N+1
            s = [((n-i)*a + i*b)/n for i in range(1,n)] # (a,b) open
        if include_endpoints:
            if self.type == 'lo,hi':
                s = [a]+s+[b]
            elif self.type == '-inf,hi':
                s = s+[b]
            elif self.type == 'lo,inf':
                s = [a]+s
        return s
    def intersect(self,other):
        lo1,hi1 = self.lo, self.hi
        lo2,hi2 = other.lo,other.hi
        if   lo1 is None: lo = lo2
        elif lo2 is None: lo = lo1
        else: lo = max(lo1,lo2)
        if   hi1 is None: hi = hi2
        elif hi2 is None: hi = hi1
        else: hi = min(hi1,hi2)
        typical_step = min(self.typical_step, other.typical_step)
        return ClosedInterval(lo,hi,
                              typical_point=None,
                              typical_step=typical_step)

class BezierFunction(ComplexValuedFunction):
    """Bezier arc as ComplexValuedFunction,
    Attributes:
    - all attributes of ComplexValuedFunction
    - control_points
    Methods:
    - __init__
    - signed_area (between arc and chord)
    """
    def __init__(self, control_points):
        ComplexValuedFunction.__init__(self,
                f       =partial(bezier_rv, control_points=control_points),
                fdot    =partial(bezier_dot_rv, control_points=control_points),
                use_ad_hoc_fdot=False,
                interval=[0.,1.])
        self.control_points = control_points
    # -----------
    # signed_area
    # -----------
    # Signed area between arc and chord.
    # Sign of area is + if the loop is run counterclockwise *in Gimp's window*
    # when
    # - first the arc is run from start to end,
    # - followed by running the chord from end to start.
    # Notes:
    # 1. Since the y coordinate in Gimp's window runs downwards, the sign
    #    in the formula is opposite to what it should be in general!
    # 2. Currently this feature is used nowhere (but it is nice to keep
    #    the formula around just in case).
    def signed_area(self):
        p0,p1,p2,p3 = self.control_points # complex
        p31 = -p3+p1
        p02 = -p0+p2
        p03 = -p0+p3
        return (3./20) * (cross(p31,p02) + cross(p31,p03) + cross(p02,p03))


################################################################
#             Central functions f,g,h in the algorithm
################################################################

def f_function(x): return 3.*x*x - 2.*x*x*x # bernstein2 + bernstein3

def g_function(x): return 3.*(1-x)*(1-x)*x  # bernstein1

def h_function(x): return 3.*(1-x)*x*x      #  bernstein2

# -------
# solve_f
# -------
# Let 0<a<1. Solve f_function(t)=a.
# Return roots in ascending order.
# Return None if not 0<a<1.
# Note: Of the three roots the important is t2 one since it belongs to (0,1).
def solve_f(a):
    from math import cos, acos, pi
    try:
        theta = acos(-2*a+1)
    except ValueError:
        raise Exception("solve_f: Got invalid argument a =",a)
    t0 = .5 + cos(theta/3)
    t1 = .5 + cos(theta/3 + 2*pi/3)
    t2 = .5 + cos(theta/3 + 4*pi/3)
    return t1,t2,t0

################################################################
#                       Make families
################################################################
# Make one-parameter families of Bezier arcs with common support S
# and some chosen other condition.
# (In this version the only appearing condition is that the arcs pass
# through a fixed point.)

# ------------------
# family_fixed_point
# ------------------
# Given support S:Support and a point X:[float,float],
# create the family of Bezier arcs having the support S and passing
# through X.
# If respect_directions is True, include in the family only those Bezier
# arcs for which the handles point in the same directins as the handles
# of the support.
# Args:
# - support:            Support
# - point:              complex
# - respect_directions: boolean
# Returns: SupportedFamily
def family_fixed_point(support, point, respect_directions=False):
    def check_condition(xi, eta, t, X): # Condition B(t) = X
        ba = support.coefficients2bezier_arc(xi,eta)
        cp = ba.control_points
        Bt = bezier(t, cp)
        diff = -Bt+X
    PADDING = 1e-8 # To avoid interval ends 0 and 1
    ZERO = 1e-10
    S0,S1,S2,S3 = support.support_points
    X = point
    case = support.case
    S0X = -S0+X
    S32 = -S3+S2
    S01 = -S0+S1
    S03 = -S0+S3
    S0XxS01 = cross(S0X,S01)
    S0XxS03 = cross(S0X,S03)
    S03xS32 = cross(S03,S32)
    S03xS01 = cross(S03,S01)
    S01xS32 = cross(S01,S32)
    S01xS03 = -S03xS01
    S32xS01 = -S01xS32
    S32xS03 = -S03xS32
    if case == 'N':
        def F(t):
            ft, gt, ht = f_function(t), g_function(t), h_function(t)
            diff = S0X - ft*S03
            try:
                xi  = cross(diff, S32) / (gt * S01xS32)
                eta = cross(diff, S01) / (ht * S32xS01)
            except ZeroDivisionError:
                raise Exception("parametric function: t = 0 or 1.")
            #check_condition(xi, eta, t, X)
            return [xi,eta]
        def Fdot(t):
            xi, eta = F(t)
            xidot  = -((1-3*t)*xi/t      + 2*S03xS32/S01xS32) / (1-t)
            etadot = -((2-3*t)*eta/(1-t) + 2*S03xS01/S32xS01) / t
            return [xidot, etadot]
        if respect_directions:
            interval, None_case = first_quadrant_subinterval(support, point)
            if interval is None:
                raise Exception("family_fixed_point: Got subinterval=None:"+None_case)
            if interval[0] < ZERO:
                interval[0] = PADDING
            if interval[1] > 1-ZERO:
                interval[1] = 1-PADDING
        else:
            interval=[PADDING,1-PADDING]
        type = 'fixed point N'
        data = point
    else: # Case P
        if not (0 + ZERO <= S0XxS01 / S03xS01 <= 1 - ZERO):
            raise Exception("family_fixed_point: The target point in illegal area: not between the tangents lines")
        try:
            t0 = solve_f(S0XxS01 / S03xS01)[1]
        except ZeroDivisionError:
            raise Exception("parametric function: Support points collinear.")
        ft, gt, ht = f_function(t0), g_function(t0), h_function(t0)
        a = gt * S01xS03
        b = ht * S32xS03
        c = S0XxS03
        try:
            invb = 1./b
        except ZeroDivisionError:
            raise Exception("parametric function: Support points collinear.")
        def F(xi):
            eta = (c-a*xi) * invb
            #check_condition(xi, eta, t0, X)
            return [xi,eta]
        def Fdot(xi):
            return [1., -a*invb]
        if respect_directions:
            interval, None_case = first_quadrant_subinterval(support, point)
            if interval is None:
                raise Exception("family_fixed_point: Got subinterval=None:"+None_case)
        else:
            interval=[None,None]
        type = 'fixed point P'
        data = t0
    try:
        supported_family = SupportedFamily(support_points=[S0,S1,S2,S3],
                               f=F,
                               fdot=Fdot,
                               use_ad_hoc_fdot=False,
                               interval=interval,
                               type=type,
                               data=data)
    except Exception as e:
        raise Exception('family_fixed_point: ' +str(e))
    return supported_family

# --------------------------
# first_quadrant_subinterval
# --------------------------
# In the algorithm it may be required to find only those solutions
# (Bezier arcs) whose tangents obey the directions S01 and S32 in
# the support.
# This routine finds the interval [a,b] where the parameter x needs
# to be restricted to this end. The family in question is the one produced
# by family_fixed_point, and x is its parameter. Let F be the vector-valued
# function corresponding to the family. The condition to be enforced is that
# if x is in [a,b] and F(x) = (xi,eta), then xi >= 0 and eta >= 0.
# Args:
# - support:Support
# - point:  complex
# Returns:
# - [a,b] or None (may be b=None, meaning infinity)
# - None_case: string (only for debugging)
# Note: points 0,1 included: caller must handle!
def first_quadrant_subinterval(support, point):
    ZERO = 1e-10
    f = f_function
    S0,S1,S2,S3 = support.support_points
    X = point
    case = support.case
    S0X = -S0+X
    S32 = -S3+S2
    S01 = -S0+S1
    S03 = -S0+S3
    S0XxS01 = cross(S0X,S01)
    S0XxS03 = cross(S0X,S03)
    S03xS32 = cross(S03,S32)
    S03xS01 = cross(S03,S01)
    S01xS32 = cross(S01,S32)
    S01xS03 = -S03xS01
    S32xS01 = -S01xS32
    S32xS03 = -S03xS32
    S0XxS32 = cross(S0X,S32)
    None_case = None
    if case == 'N':
        a = S0XxS32 / S01xS32
        b = S03xS32 / S01xS32
        c = S0XxS01 / S32xS01
        d = S03xS01 / S32xS01
        cd_ab = []
        try:
            if 0 < c/d < 1:
                cd_ab.append(c/d)
        except ZeroDivisionError:
            pass
        try:
            if 0 < a/b < 1:
                cd_ab.append(a/b)
        except ZeroDivisionError:
            pass
        tx_ty = [solve_f(q)[1] for q in cd_ab]
        t1_t2 = sorted(tx_ty)
        if len(t1_t2) == 2:
            if -ZERO < t1_t2[0] - t1_t2[1] < ZERO:
                t1_t2 = [t1_t2[0]]
        if len(t1_t2) == 2:
            t1,t2 = t1_t2
            if a - b*f((t1+t2)/2) > 0 and c - d*f((t1+t2)/2) > 0:
                subinterval = [t1,t2]
            elif a - b*f(t1/2) > 0 and c - d*f(t1/2) > 0:
                subinterval = [0.,t1]
            elif a - b*f((t2+1)/2) > 0 and c - d*f((t2+1)/2) > 0:
                subinterval = [t2,1.]
            else:
                subinterval = None
                None_case = '2'
        elif len(t1_t2) == 1:
            t1 = t1_t2[0]
            if a - b*f(t1/2) > 0 and c - d*f(t1/2) > 0:
                subinterval = [0.,t1]
            elif a - b*f((t1+1)/2) > 0 and c - d*f((t1+1)/2) > 0:
                subinterval = [t1,1.]
            else:
                subinterval = None
                None_case = '1'
        elif len(t1_t2) == 0:
            if a - b*f(1/2) > 0 and c - d*f(1/2) > 0:
                subinterval = [0.,1.]
            else:
                subinterval = None
                None_case = '0'
    else: # Case 'P'
        try:
            t0 = solve_f(S0XxS01 / S03xS01)[1]
        except ZeroDivisionError:
            raise Exception("first_quadrant_subinterval: Support points collinear.")
        ft, gt, ht = f_function(t0), g_function(t0), h_function(t0)
        a = gt * S01xS03
        b = ht * S32xS03
        c = S0XxS03
        if a*b > 0 and c*a > 0:
            subinterval = [0, c/a]
        elif a*b < 0:
            subinterval = [max(0, c/a), None]
        else:
            subinterval = None
            None_case = 'P'
    return subinterval, None_case


################################################################
#                   Create special Bezier arcs
################################################################

# ------------
# check_target
# ------------
# Check that target is far enough from S0 and S3
def check_target(support,trg):
    MIN_DIST2 = 1e-8
    S0 = support.support_points[0]
    S3 = support.support_points[3]
    dist0 = (trg.real - S0.real)**2 + (trg.imag - S0.imag)**2
    dist3 = (trg.real - S3.real)**2 + (trg.imag - S3.imag)**2
    return (dist0 > MIN_DIST2) and (dist3 > MIN_DIST2)

# -----------------------------
# arc_fixed_point_approx_points
# -----------------------------
# Given a support S and a list of points X0,...,Xn
# find a Bezier arc with support S, passing exactly through X0,
# and as closely to X1,...,Xn as possible.
# Args:
# - support: Support;
# - target0: complex;
# - other_targets: [complex];
# - respect_directions: boolean: if True, search among Bezier arcs whose
#                       handle directions obey those of support;
# - sloppy: boolean: if True, include not-so-exact results in calculations.
# Returns:
# - ba_approx_cp (list of 4 control points: [p0,p1,p2,p3]);
# - abs_error: float (an error measure);
# - rel_error: float (another error measure).
def arc_fixed_point_approx_points(support,
                                  target0,
                                  other_targets,
                                  respect_directions=False,
                                  sloppy=False
                                  ):
    from math import sqrt
    # Discard targets too close to the corner(s) of the support
    # and that sufficiently many remain:
    if not check_target(support,target0):
        message = 'arc_fixed_point_approx_points: first target too close to corners'
        raise Exception(message)
    other_targets = [trg for trg in other_targets if check_target(support,trg)]
    if len(other_targets) < 1:
        message = 'arc_fixed_point_approx_points: not enough targets not too close to corners'
        raise Exception(message)
    #
    X0 = target0
    Xpoints = other_targets
    try:
        F0 = family_fixed_point(support, X0, respect_directions)
    except Exception as e:
        raise Exception("arc_fixed_point_approx_points: "+str(e))
    def pf_key(x):
        return sum_dist2_family_points(F0, x, Xpoints, sloppy)
    xmin, keymin = F0.minimize(key=pf_key, search_interval=None,
                                sloppy=sloppy)
    xi,eta = F0.function(xmin)
    ba_approx = support.coefficients2bezier_arc(xi,eta)
    abs_error = sum_dist2_arc_points(ba_approx, other_targets,sloppy)
    yard_stick = support.sum_dist2_to_targets(other_targets)
    rel_error = sqrt(abs_error / yard_stick)
    return ba_approx.control_points, abs_error, rel_error

# -----------------
# arc_approx_points
# -----------------
# Given a support S and a list of target points X0,X1,..., find a Bezier arc
# with support S and passing close(?) to the points.
# Args:
# - support: Support;
# - targets:  [complex];
# - respect_directions: boolean: if True, search among Bezier arcs whose
#                       handle directions obey those of support;
# - sloppy: boolean: if True, include not-so-exact results in calculations.
# - error_targets: None or [complex]: if not None, compute error
#                  of the resulting arc from the targets in this list,
#                  otherwise use the list targets
# Returns:
# - list of 4 control points: [p0,p1,p2,p3];
# - abs_error: float (an error measure);
# - rel_error: float (another error measure).
# Note: This is rather ad-hoc, especially the usage of weighted average.
def arc_approx_points(support,
                      targets,
                      respect_directions=False,
                      sloppy=False,
                      error_targets=None
                      ):
    # Given list [[cp,error],...], where cp = [p0,p1,p2,p3] (control points)
    # compute weighted average of the control points,
    # using as weights the inverses of the errors mapped with some function
    # tweak: float -> float
    def weighted_average_cp(cp_err_list, tweak):
        ave_p0 = sum([cp[0]/tweak(err) for cp,err in cp_err_list])
        ave_p1 = sum([cp[1]/tweak(err) for cp,err in cp_err_list])
        ave_p2 = sum([cp[2]/tweak(err) for cp,err in cp_err_list])
        ave_p3 = sum([cp[3]/tweak(err) for cp,err in cp_err_list])
        m = sum([1./tweak(err) for cp,err in cp_err_list])
        ave_p0 = ave_p0 / m
        ave_p1 = ave_p1 / m
        ave_p2 = ave_p2 / m
        ave_p3 = ave_p3 / m
        return [ave_p0, ave_p1, ave_p2, ave_p3]
    # Discard targets too close to the corner(s) of the support
    # and that sufficiently many remain:
    targets = [trg for trg in targets if check_target(support,trg)]
    if len(targets) < 1:
        message = 'arc_fixed_point_approx_points: not enough targets not too close to corners'
        raise Exception(message)
    # Compute a list of tentative approximate Bezier arcs (control points):
    cp_error_list = []
    for i in range(len(targets)):
        X0 = targets[i]
        rest = [targets[j] for j in range(len(targets)) if j != i]
        try:
            # Bezier arc exactly through X0 and trying
            # to run close to the rest of the targets:
            approx_result = arc_fixed_point_approx_points(support,
                                                      X0,
                                                      rest,
                                                      respect_directions,
                                                      sloppy=sloppy)
        except Exception as e:
            raise Exception("arc_approx_points: "+str(e))
        approx_cp, abs_error, rel_error = approx_result
        # Save control points and abs_error in list:
        cp_error_list.append([approx_cp, abs_error])
    # Sort with respect to abs_error:
    cp_error_list = sorted(cp_error_list, key=lambda x:x[1])
    # Compute weighted averages from the starting sections of the the list:
    ave_cp_error_list = []
    for i in range(1, 1+len(cp_error_list)):
        ave_cp = weighted_average_cp(cp_error_list[:i], tweak=sqrt)
        #ave_cp = weighted_average_cp(cp_error_list[:i], tweak=lambda x:x)
        ave_error = sum_dist2_arc_points(BezierFunction(ave_cp), targets, sloppy=True)
        ave_cp_error_list.append([ave_cp, ave_error])
    # Find the best; this one will be returned:
    best_cp, best_error = min(ave_cp_error_list, key=lambda x:x[1])
    # Compute anew the absolute and relative errors of the best:
    best_bf = BezierFunction(best_cp)
    if error_targets is None:
        error_targets = targets
    abs_error = sum_dist2_arc_points(best_bf, error_targets, sloppy=True)
    yard_stick = support.sum_dist2_to_targets(targets) # For relative error
    rel_error = sqrt(abs_error / yard_stick)
    return best_cp, abs_error, rel_error

#==============================================================
#                       Bezier curve
#==============================================================

###### The usual Bezier curve and derivatives from control points  #######
# Version which returns RowVector = complex
# Args:
# - t:float
# - control_points = [P0,P1,P2,P3]:[complex]
# Returns:
# - RowVector = complex
def bezier_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    if t > 0.5:
        u = (1-t)/t
        t3 = t**3
        return t3*(u*(u*(u*P0+3*P1)+3*P2)+P3) # complex
    else:
        u = t/(1-t)
        t3 = (1-t)**3
        return t3*(u*(u*(u*P3+3*P2)+3*P1)+P0) # complex

# Derivative: Returns complex.
def bezier_dot_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    P01 = -P0+P1
    P12 = -P1+P2
    P23 = -P2+P3
    if t > 0.5:
        u = (1-t)/t
        t2 = t**2
        return 3*t2*(u*(u*P01+2*P12)+P23) # complex
    else:
        u = t/(1-t)
        t2 = (1-t)**2
        return 3*t2*(u*(u*P23+2*P12)+P01) # complex

## Used nowhere! Returns [float,float] (rather than complex)!
#def bezier_dot_dot(t, control_points): # Could be made faster as above.
#    p0,p1,p2,p3 = control_points
#    p01 = [-p0[0]+p1[0], -p0[1]+p1[1]]
#    p12 = [-p1[0]+p2[0], -p1[1]+p2[1]]
#    p23 = [-p2[0]+p3[0], -p2[1]+p3[1]]
#    return [-6*(1-t)*p01[0] + 6*(1-2*t)*p12[0] + 6*t*p23[0],
#            -6*(1-t)*p01[1] + 6*(1-2*t)*p12[1] + 6*t*p23[1]]


#==============================================================
#                   Miscellaneous routines
#==============================================================

################################################################
#                 Plane vectors (=complex numbers)
################################################################

# v dot (w perpendicular)
def cross(v,w):
    return (v*w.conjugate()).imag

def vdot(v,w):
    return v.real*w.real + v.imag*w.imag

def collinear(points): # points: [complex] (The plane R2 is the complex plane.)
    ZERO = 1e-14
    n = len(points)
    if n < 3:
        return True
    for i in range(n):
        for j in range(i+1,n):
            for k in range(j+1,n):
                a,b,c = points[i], points[j], points[k]
                ab = -a+b
                ac = -a+c
                bc = -b+c
                if abs(cross(ab,ac)) > ZERO:
                    return False
                if abs(cross(ab,bc)) > ZERO:
                    return False
                if abs(cross(ac,bc)) > ZERO:
                    return False
    return True


################################################################
#                    Squared distances
################################################################

# --------------------
# sum_dist2_arc_points
# --------------------
# Given a parametric arc and a list of points,
# compute the sum of squared distances of the points from the arc.
# Args:
# - pf:     VectorValuedFunction
# - points: [X1,X2,...,Xn] with all Xi:complex
# Returns
# - float
def sum_dist2_arc_points(pf, points, sloppy=False):
    sum_d2 = 0.
    for point in points:
        t, b, d2 = pf.closest_point(point, sloppy=sloppy)
        sum_d2 += d2
    return sum_d2

# -----------------------
# sum_dist2_family_points
# -----------------------
# Given a family of Bezier arcs and one parameter value x for the family
# (giving rise to one arc in the family),
# and a list of points,
# compute the sum of squared distances of the points from the arc.
# Args:
# - family:  SupportedFamily
# - x:       float
# - points:  [X1,X2,...,Xn] with all Xi:complex
# Returns
# - float
def sum_dist2_family_points(family, x, points, sloppy=False):
    bezier_arc = family.parameter2bezier_arc(x)
    return sum_dist2_arc_points(bezier_arc, points, sloppy=sloppy)



#---------------------------------------------------------
#                        Classes
#---------------------------------------------------------


class ExitError(Exception):
    def __init__(x):
        return x
#---------------------------------------------------------
#             Make similarity transformations UNTESTED
#---------------------------------------------------------

# r: RowVector (=complex) (translation vector)
def make_translation(r):
    def pm(v): # v: RowVector = complex
        return r+v
    def jac(v):
        return Matrix2x2(1,0,0,1)
    return PlaneTransformation('affine', pm, jac)

class MakeSimilitudeError(Exception):
    def __init__(self, source_error=False, target_error=False):
        self.source_error = source_error
        self.target_error = target_error

# ----------------------
# make_direct_similitude
# ----------------------
# Direct similitude sending z0->w0, z1->w1
# Args:
# - z0,z1,w0,w1:complex
# Returns:
# - PlaneTransformation
def make_direct_similitude(z0,z1,w0,w1):
    try:
        a = (w0-w1) / (z0-z1)
    except ZeroDivisionError:
        raise MakeSimilitudeError(source_error=True)
    try:
        ia = 1./a
    except ZeroDivisionError:
        raise MakeSimilitudeError(target_error=True)
    a1,a2 = a.real, a.imag
    b = w0 - a*z0
    def pm(z): # z:complex
        return a*z+b
    def jac(z): # z:complex
        return Matrix2x2(a1,-a2,a2,a1)
    return PlaneTransformation('affine', pm, jac)


# -------------
# make_dilation
# -------------
# Dilation sending z0->w0 and having the given scale factor.
# Args:
# - z0, w0:complex
# scale_factor: float
# Returns:
# - PlaneTransformation
def make_dilation(z0, w0, scale_factor):
    def pm(z): # z:complex
        return scale_factor*(z-z0) + w0
    def jac(z): # z:complex
        return Matrix2x2(scale_factor,0,0,scale_factor)
    return PlaneTransformation('dilation', pm, jac)


#---------------------------------------------------------
#                   Auxiliary routines
#---------------------------------------------------------

# From input string make list.
# Arg s (string) should evaluate to
# - float or int
# - list of floats or ints
# - tuple of floats or ints
# Returns: [float]
def string_to_list(s): # s:string
    if s.strip() == '': # blank
        return []
    try:
        list_or_tuple = eval(s)
        if type(list_or_tuple) in (int,float):
            list_or_tuple = [float(list_or_tuple)]
        return [float(p) for p in list_or_tuple]
    except:
        m = ("Something wrong in input: "
            +"\n    "+s
            +"\nPlease check syntax and so on.")
        raise Exception(m)

def is_loop(cp4):
    ZERO = 1e-12
    p0, p3 = cp4[0], cp4[-1]
    p03 = -p0+p3
    return (vdot(p03,p03) < ZERO)

def gimp_message(txt, h=2):
    # 0 = message box; 1 = console; 2 = error console
    pdb.gimp_message_set_handler(h)
    pdb.gimp_message(txt)

# --------------
# dist_from_line
# --------------
# Find (signed) distance of x from line through a,b when a!=b.
# If a=b, return distance of x from a=b.
# Args:
# - a,b,x: RowVector # complex
# Returns:
# - float
def dist_from_line(a,b,x):
    ZERO = 1e-12
    x0 = x-a
    b0 = b-a
    b0len = abs(b0)
    if b0len < ZERO:
        return abs(x0)
    return (x0*b0.conjugate()).imag / b0len

# -----------------------
# line_through_two_points
# -----------------------
# Given two points, find coefficients g,h,k of the line gx+hy+k=0
# which passes through p and q.
# Args:
# - p,q:complex (the plane is treated as the complex plane)
# Returns:
# - [float,float,float] (these are the g,h,k)
def line_through_two_points(p,q):
    p1,p2 = p.real,p.imag
    q1,q2 = q.real,q.imag
    g = -p2+q2
    h =  p1-q1
    k = -p1*q2+p2*q1
    return [g,h,k]


# ----------------------
# line_intersect_segment
# ----------------------
# Given a line L:gx+hy+k=0 and two points P,Q, find the intersection
# (if any) of L with the line through P,Q and whether it recides at:
# - exactly at P
# - exactly at Q
# - in the open interval (P,Q).
# Args:
# - line_coefficients: [float,float,float] (these are the g,h,k);
# - p: complex;
# - q: complex.
# Returns:
# - complex or None (intersection point; or None if lines parallel);
# - [boolean,boolean,boolean] (exactly at p, exactly at q, in open interval).
# Note: Special case:
# 1. When P,Q both on L, returned values are
#    - None
#    - [True,True,False]
def line_intersect_segment(line_coefficients, p, q):
    g,h,k = line_coefficients
    p1,p2 = p.real, p.imag
    q1,q2 = q.real, q.imag
    try:
        c = 1./((g*p1 + h*p2) - (g*q1 + h*q2))
    except ZeroDivisionError: # lines parallel or p=q
        if g*p1+h*p2+k == 0: # p and q on L
            return None, [True,True,False]
        else:
            return None, [False,False,False]
    pcoeff = -(g*q1 + h*q2 + k) * c
    qcoeff =  (g*p1 + h*p2 + k) * c
    p_exact = (qcoeff == 0)
    q_exact = (pcoeff == 0)
    in_open = (0 < pcoeff < 1)
    X = pcoeff*p + qcoeff*q
    #check = g*X[0] + h*X[1] + k
    return X, [p_exact, q_exact, in_open]

# -----------------------
# line_intersect_polyline
# -----------------------
# Given a line L:gx+hy+k=0 and a polyline, find intersection points.
# Args:
# - line_coefficients: [float,float,float] (these are the g,h,k);
# - polyline: [complex].
# Returns:
# - [complex] (possibly empty list)
# Notes:
# 1. If some line segment of the polyline is contained in L,
#    of the line segment only the end points will be in the returned list.
# 2. Repeated vertices will be repeated in the result too if on L.
def line_intersect_polyline(line_coefficients, polyline):
    vertices = polyline
    intersections = []
    for i in range(-1+len(vertices)):
        point,loc = line_intersect_segment(line_coefficients,
                                           vertices[i],
                                           vertices[i+1])
        if point is None: # Segment parallel to L or vertices coincide
            if loc[0]: # Both vertices on L
                intersections.append(vertices[i])
                if i == -2+len(vertices):
                    intersections.append(vertices[i+1])
        elif loc[0] or loc[2]: # L intersects either at start or in the open interval
            intersections.append(point)
        elif loc[1] and (i == -2+len(vertices)): # Last vertex is on L
            intersections.append(point)
    return intersections


def some_three_collinear(four_points): # four_points = [x,y,z,u]: [complex]
    x,y,z,u = four_points
    return (collinear([x,y,z]) or collinear([x,y,u])
                or collinear([x,z,u]) or collinear([y,z,u]))

# -----------------
# line_intersection
# -----------------
# Find intersection of two lines given by one point on each (p0,p1)
# and direction vectors of each.
# The plane is used as the complex plane.
# Args:
# - p0,d0,p1,d1: complex
# Returns:
# - complex
# Exception is raised if the lines are parallel (includes cases where
# a direction vector is zero).
def line_intersection(p0,d0,p1,d1):
    try:
        return (-cross(p0,d0)*d1 + cross(p1,d1)*d0) / cross(d0,d1)
    except ZeroDivisionError:
        raise Exception("line_intersection: parallel lines or zero direction vectors")


# ---------
# is_convex
# ---------
# Is a polygon convex? The polygon is inputted as
# a list of corner points (complex).
# Args:
# - corners: [complex]
# Returns:
# - boolean
# Notes:
# 1. Returns True if all corner points are collinear.
def is_convex(corners):
    if len(corners) <= 3:
        return True
    if collinear(corners): # TEE
        return True
    c = corners + corners[:2]
    edges = [c[i]-c[i+1] for i in range(len(c)-1)]
    edges = [e for e in edges if e != 0] # remove zero edges
    pairs = [[edges[i],edges[i+1]] for i in range(len(edges)-1)]
    # Cross and dot products:
    cr_dots = [[-a.real*b.imag + a.imag*b.real,
                 a.real*b.real + a.imag*b.imag] for a,b in pairs]
    crs = []
    for crd in cr_dots:
        if (crd[0]==0):
            if (crd[1]<0):
                return False # spike of angle 0
        else:
            crs.append(crd[0]) # collect non-zero cross products
    try:
        pos = (crs[0] > 0)
    except IndexError: # All points collinear
        return True
    for cr in crs:
        if pos != (cr > 0):
            return False
    return True


# -------------------
# get_guide_positions
# -------------------
# Guide positions in image, horizontal and vertical in separate lists.
# Not sorted.
def get_guide_positions(image):
    horizontal = []
    vertical = []
    guide = 0
    while True:
        guide = pdb.gimp_image_find_next_guide(image,guide)
        if guide == 0:
            break
        position = pdb.gimp_image_get_guide_position(image, guide)
        orientation = pdb.gimp_image_get_guide_orientation(image, guide)
        if orientation == ORIENTATION_HORIZONTAL:
            horizontal.append(position)
        elif orientation == ORIENTATION_VERTICAL:
            vertical.append(position)
        else:
            raise Exception("get_guide_positions: ???")
    return horizontal, vertical

def str_float(x): return '{:1.3f}'.format(x)

# -------------------------
# solve_bernstein2_equation
# -------------------------
# Find real zeroes of the equation c0*b0(t) + c1*b1(t) + c2*b2(t) = 0 where
# b0,b1,b2 are the Bernstein polynomials of degree 2 and
# c0,c1,c2 are real coefficients.
# Return the zeroes as a list (0, 1, or 2 items).
# Exception: If the function is constant 0, return None.
# Args:
# - c0,c1,c2: float
# Returns:
# - None or list of floats (possibly empty)
def solve_bernstein2_equation(c0,c1,c2):
    from math import sqrt
    ZERO = 1e-14
    if abs(c0 - 2.*c1 + c2) < ZERO: # equation of degree <= 1
        try:
            return [c0 / (2.*(c0 - c1))]
        except ZeroDivisionError: # equation of degree 0
            if abs(c0) < ZERO: # constant 0
                return None
            else: # constant not 0
                return []
    try:
        root = sqrt(c1**2 - c0*c2)
    except ValueError: # No real roots
        return []
    return [(c0 - c1 + root) / (c0 - 2.*c1 + c2),
            (c0 - c1 - root) / (c0 - 2.*c1 + c2)]

# -----------
# arrange_cps
# -----------
# Given list of control points ([complex]),
# arrange it cyclically and possibly reversing so that
# the resulting list  a  has as the segment  a[0],a[1]  the lowest
# on the screen (in some sense) and left to right.
# Args:
# - cps:  [complex]
# - Gimp: boolean
# Returns:
# - [complex]
def arrange_cps(cps, Gimp=True):
    if Gimp: c = -1
    else: c = 1
    n = len(cps)
    if n <= 1:
        return cps
    # Find lowest anchor:
    lowi = min(range(n), key=(lambda j: c*cps[j].imag))
    # Check the neighbouring anchors:
    upwards = (c*cps[(lowi-1)%n].imag > c*cps[(lowi+1)%n].imag)
    # Arrange item with index lowi to place 0
    # and the lower neighbour to place 1:
    if upwards:
        a = [cps[(i+lowi)%n] for i in range(n)]  # cyclic
    else:
        a = [cps[(-i+lowi)%n] for i in range(n)] # cyclic + reverse
    # Check if left to right:
    if a[0].real <= a[1].real:
        return a # already ok
    else:
        return [a[(1-i)%n] for i in range(n)] # reverse keeping the low segment

# --------------------
# rotate_to_horizontal
# --------------------
# Given a list c of complex numbers, rotate the whole contents in the
# complex plane around the center point by a constant angle so that the line segment
# c[0],c[1] becomes horizontal.
# Args:
# - complex_list:[complex]
# Returns:
# - [complex]
def rotate_to_horizontal(complex_list):
    try:
        c0,c1 = complex_list[0], complex_list[1]
    except IndexError:
        raise Exception("rotate_to_horizontal: not enough points")
    #center = c0 # Change if you want some other rotation center.
    center = sum(complex_list) / len(complex_list) # rotation center = center point
    try:
        z = complex(abs(c1-c0), 0.) / (c1-c0)
    except ZeroDivisionError:
        raise Exception("rotate_to_horizontal: zero first line segment")
    return [center + z*(c-center) for c in complex_list]


# -------------------
# rotate_complex_list
# -------------------
# Rotate a list of complex numbers in the complex plane around center by angle
# Positive angle: rotate ccw. If Gimp=True, y coordinate runs downwards.
# Args:
# - clist:  [complex]
# - center: complex
# - angle:  float
# - Gimp:   boolean
def rotate_complex_list(clist, center, angle, Gimp=True):
    from math import cos,sin
    ca, sa = cos(angle), sin(angle)
    if Gimp:
        za = complex(ca, -sa)
    else:
        za = complex(ca, sa)
    return [center + (c-center)*za for c in clist]


#======================================================
#                   Polynomial zeroes
#======================================================
# My own algorithm, written for fun.
# Certainly could find faster elsewhere.
# --------------------
# zeroes_of_polynomial
# --------------------
# Given a real polynomial, find the real zeroes, the x values of extrema,
# and the x values of the inflection points. May restrict to an interval.
#
# Args:
# - coefficients: list [c0,c1,...,cn] of the coefficients of the
#                     polynomial c0 + c1*x + ... + cn*(x**n);
#                     the ci are integer or float.
# - interval:     None or [float,float]
# Returns: Either None (in case of zero input polynomial), or
# - [float] (the list of real zeroes of the polynomial, in increasing order,
#            multiple roots repeated);
# - [float] (the extrema (the x values));
# - [float] (the inflection points (the x values)).
# Notes:
# 1. Return None in the case of a zero polynomial.
# 2. Return None if interval is given (not None) with interval=[a,b] where a>b.
# 3. If interval=[a,a] is given, if a is a root then [a],[],[] is returned,
#    otherwise [],[],[]. So, no effort is made to give any extrema or inflection
#    points.
# 3. Otherwise:
#    - the list of real zeroes of the polynomial, in increasing order
#      (possibly []) with multiple roots repeated;
#    - the extrema (the x values);
#    - the inflection points (the x values).
# 4. Because of the chosen accuracy and tolerance in the code, choosing different
#    interval may change the returned values slightly.
def zeroes_of_polynomial(coefficients, interval=None):
    from math import sqrt
    from functools import partial
    def f(x, coeff): # Horner
        s = 0.
        for ci in coeff[::-1]:
            s = ci + x*s
        return s
    if not (interval is None):
        LO,HI = interval
        if LO > HI:
            return None
        if LO == HI:
            if f(LO) == 0:
                return [LO],[],[]
            else:
                return [],[],[]
    coefficients = [float(ci) for ci in coefficients]
    # Find largest n with coefficients[n] non-zero and
    # discard all higher terms:
    n = -1
    for i in range(len(coefficients)):
        if coefficients[i] != 0.:
            n = i
    if n < 0: # Polynomial is identically zero.
        return None
    c = coefficients[:n+1] 
    # Make c[n] positive:
    if c[n] < 0:
        c = [-ci for ci in c]
    if n <= 2:
        if n == 0: # Polynomial is a non-zero constant.
            #return [],[],[]
            results = [[],[],[]]
        if n == 1: # Polynomial is c[0] + c[1]*x
            #return [-c[0] / c[1]], [], []
            results = [[-c[0] / c[1]], [], []]
        if n == 2: # Polynomial is c[0] + c[1]*x + c[2]*(x**2).
            discr = c[1]**2 - 4*c[0]*c[2]
            try:
                root = sqrt(discr)
                x1 = (-c[1] - root) / (2*c[2])
                x2 = (-c[1] + root) / (2*c[2])
                #return [x1,x2], [-c[1]/(2*c[2])], []
                results = [[x1,x2], [-c[1]/(2*c[2])], []]
            except ValueError:
                #return [], [-c[1]/(2*c[2])], []
                results = [[], [-c[1]/(2*c[2])], []]
        if interval is None:
            return results
        else:
            results0 = [t for t in results[0] if LO <= t <= HI]
            results1 = [t for t in results[1] if LO <= t <= HI]
            results2 = [t for t in results[2] if LO <= t <= HI]
            return results0, results1, results2
    # Build subdivision such that in each subinterval
    # the polynomial is monotonous:
    derivative = [e*i for e,i in enumerate(c)][1:]
    extrema, inflection, _ = zeroes_of_polynomial(derivative, interval)
    if interval is None:
        x_far = sum([abs(ci/c[-1]) for ci in c])
        subdiv = [-x_far] + extrema + [x_far]
    else:
        subdiv = [LO] + [t for t in extrema if LO < t < HI] + [HI]
    # Solve and collect the zeroes of each monotonous segment:
    fc = partial(f, coeff=c)
    zeroes = []
    for i in range(len(subdiv)-1):
        lo, hi = subdiv[i], subdiv[i+1]
        x = zero_of_monotonous_function(fc, lo, hi,
                                    accuracy=1e-10, tolerance=1e-10)
        if not(x is None):
            zeroes.append(x)
    return zeroes, extrema, inflection

# ---------------------------
# zero_of_monotonous_function
# ---------------------------
# Find the zero, if any, of a monotonous function f(x)
# in the interval [lo,hi]. If none is found, None is returned.
# Args:
# - f:         callable (real function, assumed to be monotonous in [lo,hi];
# - lo, hi:    float; (the interval)
# - accuracy:  accuracy for x.
# - tolerance: float>=0; if no zeroes exist in the interval, then
#              if f(lo) or f(hi) is less than 'tolerance' from 0,
#              then lo or hi is returned, respectively.
# Returns: None or the unique zero.
def zero_of_monotonous_function(f, lo, hi, accuracy=1e-10, tolerance=1e-10):
    MAX_ROUNDS = 100
    lo, hi = float(lo), float(hi)
    flo, fhi = float(f(lo)), float(f(hi))
    if (flo > 0.) == (fhi > 0.):
        if abs(flo) <= tolerance:
            return lo
        elif abs(fhi) <= tolerance:
            return hi
        else:
            return None
    if flo == 0.:
        return lo
    if fhi == 0.:
        return hi
    count = 0
    while hi-lo > accuracy:
        mid = (lo+hi)/2.
        fmid = float(f(mid))
        if fmid == 0.:
            return mid
        if (flo > 0.) == (fmid > 0.):
            lo, flo = mid, fmid
        else:
            hi, fhi = mid, fmid
        count += 1
        if count > MAX_ROUNDS:
            break
    x = (fhi*lo - flo*hi) / (fhi-flo)
    return x


#---------------------------------------------------------
#                Routines for Bezier curves
#---------------------------------------------------------

# -----------
# get_anchors
# -----------
# Get anchors of the path (optionally the first stroke only).
# Arg:
# - vectors_object:    gimp.Vectors
# - only_first_stroke: boolean
# Returns:
# - [BCurve.ControlPoint]
def get_anchors(vectors_object, only_first_stroke=False): # MUUTOS: True->False
    gv = vectors_object2gv(vectors_object) # BCurve.GimpVectors
    if len(gv.stroke_list) == 0:
        raise Exception("Path without any strokes: "+vectors_object.name)
    anchors = []
    for gs in gv.stroke_list:
        bc = gs.gs2bc()                        # BCurve.BezierCurve
        bas = bc.bezier_arcs                   # [BCurve.BezierArc]
        #anchors = [ba.cp4[0] for ba in bas]
        anchors += [ba.cp4[0] for ba in bas]
        if len(bas[0].cp4) > 1: # Case len=1 may rise from stroke with one anchor.
            anchors.append(bas[-1].cp4[-1])
        if only_first_stroke:
            break
    return anchors

# -------------------------
# bezier_new_control_points
# -------------------------
# Given a Bezier arc B(t) and two distinct parameter values a and b,
# find a new parametrization C(t) of the same curve such that
# B(a) = C(0) and B(b) = C(1).
# Args:
# - a: float;
# - b: float;
# - control_points: the list [p0,p1,p2,p3] of control points (pi:complex).
# Returns:
# - new control points [q0,q1,q2,q3]:[RowVector] such that the corresponding
#   Bezier curve C(t) is precisely the same as B as a curve
#   but B(a) = C(0) and B(b) = C(1).
# Notes:
# - This enables one to extend or to shorten the given Bezier arc.
# - There is an exception: If a=b, the returned control points
#   are equal, so the new curve is a single point, not the original
#   curve. The function gives no signal about such occurence.
def bezier_new_control_points(a, b, control_points):
    def bezier_2_rv(t, r0, r1, r2): # Bezier curve of degree two (complex)
        b20 = (1-t)**2
        b21 = 2*(1-t)*t
        b22 = t**2
        return b20*r0 + b21*r1 + b22*r2
    p0,p1,p2,p3 = control_points
    q0 = bezier_rv(a, control_points)
    q1 = (1-b)*bezier_2_rv(a, p0,p1,p2) + b*bezier_2_rv(a, p1,p2,p3)
    q2 = (1-a)*bezier_2_rv(b, p0,p1,p2) + a*bezier_2_rv(b, p1,p2,p3)
    q3 = bezier_rv(b, control_points)
    return [q0,q1,q2,q3]

# ------------------------------
# bezier_subdivide_at_parameters
# ------------------------------
# Given a Bezier arc B(t) and a list of parameter values t1,t2,...,
# find control points for all subarcs when B(t), 0<=t<=1, is subdivided
# at parameter values t1,...
# The input Bezier arc is given as its 4 control points [p0,p1,p2,p3],
# and the output is a list of new sets of 4 control points [[q0,q1,q2,q3],...].
# The ordering is along the Bezier curve, so that successive [q0,q1,q2,q3]'s
# are butting.
# The plane is viewed as the complex number plane.
# Args:
# - parameter_list: [float]                           ([t1,t2,...])
# - control_points: [complex,complex,complex,complex] ([p0,p1,p2,p3])
# Returns:
# - [[complex,complex,complex,complex]]               ([[q0,q1,q2,q3],...])
def bezier_subdivide_at_parameters(parameter_list, control_points):
    ts01 = [t for t in parameter_list if 0 < t < 1] # Drop t's outside (0,1)
    ts01.sort()
    try:
        ts = [ts01[0]]
    except IndexError: # Empty parameter list
        #return [[p for p in control_points]] # copy
        return [control_points[:]] # copy
    ts = [0] + ts01 + [1] # Add t=0,1
    # Collect pairs [t(i),t(i+1)]
    abs = []
    for i in range(1,len(ts)):
        abs.append([ts[i-1],ts[i]])
    arc_cps = []
    for a,b in abs:
        arc_cps.append(bezier_new_control_points(a, b, control_points))
    return arc_cps


# ---------------------
# bezier_special_points
# ---------------------
# Find the special points of a cubic Bezier curve:
# inflection points, cusp points, self-intersection points.
#
# Args:
# - control_points: [p0,p1,p2,p3], pi:complex
#
# Returns: (case, param_list), which is one of:
# (-1, [t1,t2]) when there is a loop or a cusp point, with
#               t1,t2 the parameter values with B(t1)=B(t2)
#               (ideally, t1=t2 in the case of a cusp);
# (0, [])       when the curve is degenerate: a parabola or
#               contained on a straight line;
# (1, [t])      when there is a single inflection point, with
#               t the parameter of the inflection point.
# (2, [t1,t2])  when there are two inflection points, with
#               t1,t2 the parameters for the inflection points.
# Note: The algorithm works as if infinite computation precision were
#   possible. The caller had better check the following:
# - In the case of a loop or of two inflection points (cases -1 or 2):
#   If the returned parameter values t1,t2 are very close, it may be
#   a sign of a cusp point, instead. (Or maybe the caller should take
#   it as a cusp point.)
# - In the case of inflection points (cases 1 or 2):
#   If one or both of the returned parameter values has very large
#   absolute value, it may mean that the inflection point is at infinity
#   (so, non-existent).
def bezier_special_points(control_points):
    from math import sqrt
    p0,p1,p2,p3 = control_points # complex
    w1 = -p0 - p1 + p2 + p3
    w2 =  p0 - p1 - p2 + p3
    w3 = -p0 + 3*p1 - 3*p2 + p3
    #
    C1 = 18.0 * (-w2.real*w3.imag + w2.imag*w3.real)
    C2 = 4.50 * (-w3.real*w1.imag + w3.imag*w1.real)
    C3 = 2.25 * (-w1.real*w2.imag + w1.imag*w2.real)
    try:
        X = C2/C1
        Y = C3/C1
        Z = -X**2 + 2*Y
        try:
            root = sqrt(3*Z)
            t1 = .5 + X + root
            t2 = .5 + X - root
            return -1, [t1,t2] # loop or cusp
        except ValueError: # Z < 0
            root = sqrt(-Z)
            t1 = .5 + X + root
            t2 = .5 + X - root
            return 2, [t1,t2] # two inflection points
    except ZeroDivisionError: # C1 = 0
        try:
            t = .5 + C3/C2
            return 1, [t] # one inflection point
        except ZeroDivisionError: # C1 = C2 = 0
            return 0,[] # parabola or everything on straight line.

# ------------------------
# bezier_arc_protect_cusps
# ------------------------
# Given a Bezier curve bc:BCurve.BezierCurve, check if any of its Bezier arcs
# has a cusp or two very close inflection points in the interval 0<t<1.
# If so, replace such arc with two or three subarcs (generally three, the middle
# one very small), to have the cusp as one anchor or to have the arc between the
# inflection points isolated.
# Return new Bezier curve.
# Args:
# - bc:BCurve.BezierCurve
# Returns:
 # - BCurve.BezierCurve
def bezier_arc_protect_cusps(bc):
    ZERO = 1e-3
    new_bc = deepcopy(bc) # Just in case bc is still used somewhere.
    new_ba_list = []           # Construct new list of BezierArcs
    for ba in bc.bezier_arcs:
        cp4xy = ba.cp4 # [ControlPoint] = [complex]
        if len(cp4xy) == 1:
            new_ba_list.append(ba)
            continue
        special = bezier_special_points(cp4xy)
        if special[0] not in (-1,2): # No loops and at most one inflection point
            new_ba_list.append(ba)
            continue
        # Found either a double point or two inflection points
        # with parameter values t1,t2.
        # If they are very close, we take it to be a sign of a cusp.
        t1,t2 = extra_ts = sorted(special[1])
        if t2-t1 > ZERO: # Not so very close
            new_ba_list.append(ba)
            continue
        if t1 == t2: # cusp
            extra_ts = [t1]
        #extra_ts_in_01 = [t for t in extra_ts if 0 < t < 1]
        extra_ts_in_01 = [t for t in extra_ts if ZERO < t < 1-ZERO]
        if len(extra_ts_in_01) == 0:
            new_ba_list.append(ba)
        else: # subdivide ba into subarcs
            ts = [0] + extra_ts_in_01 + [1]
            for i in range(len(ts)-1):
                left_t, right_t = ts[i], ts[i+1]
                sub_cp4 = bezier_new_control_points(left_t, right_t, cp4xy)
                sub_ba = BCurve.BezierArc(cp4 = sub_cp4)
                new_ba_list.append(sub_ba)
    new_bc.bezier_arcs = new_ba_list
    return new_bc


#======================================================
#                 Split path by a circle
#======================================================

class PathPointData(object):
    """Record from point on a path:
    - one_anchor_stroke: boolean (True if the stroke consist of one anchor only)
    - stroke_number (the number of the stroke in the path)
    - arc_number (the number of the Bezier arc in the stroke)
    - t (parameter value of the point on the infinite Bezier curve B(t))
    - Bt (the point B(t))
    - Bdot (the derivative B'(t))
    - Bdotdot (the second derivative B''(t))
    Note: It may happen that arc_num is 1 larger than actually exists
    in the stroke. This means that the stroke is closed and this arc is the
    gap-closing (imaginary) arc.
    """
    def __init__(self,
                 one_anchor_stroke = False,
                 stroke_num = 0,
                 arc_num    = 0,
                 parameter  = 0,
                 Bt         = None,
                 Bdot       = None,
                 Bdotdot    = None
                 ):
        self.one_anchor_stroke = one_anchor_stroke
        self.stroke_number = stroke_num
        self.arc_number = arc_num
        self.parameter = parameter
        self.Bt = Bt
        self.Bdot = Bdot
        self.Bdotdot = Bdotdot
    def __str__(self):
        s  = 'stroke_number = '+str(self.stroke_number)
        s += '\narc_number    = '+str(self.arc_number)
        s += '\nparameter     = '+str(self.parameter)
        s += '\nB(t)          = '+str(self.Bt)
        s += '\nB\'(t)         = '+str(self.Bdot)
        s += '\nB\'\'(t)        = '+str(self.Bdotdot)
        return s


# -------------------
# gv_intersect_circle
# -------------------
# Find intersection points (as PathPointData) between a gv:BCurve.GimpVectors
# and a circle.
# Args:
# - gv:         BCurve.GimpVectors
# - center:     complex
# - radius:     float
# Returns:
# - [PathPointData] (parameters t)
# Notes:
# 1. Each closed stroke is extended with a gap-closing arc, and the field
#    arc_number in PathPointData shows it.
# 2. Multiple roots are reduced to one.
def gv_intersect_circle(gv, center, radius):
    ZERO = 1e-6
    EQUAL_ROOT = 1e-10
    results = []
    gs_num = 0
    for gs in gv.stroke_list:     # BCurve.GimpStroke
        bc = gs.gs2bc()           # BCurve.BezierCurve
        if not bc.closed:
            bas = bc.bezier_arcs
        else: # Closed stroke. To close the gap, add one arc
            bas = bc.bezier_arcs + [(BCurve.BezierArc(cp4 = [bc.bezier_arcs[-1].cp4[-1],
                                                  bc.tail_handle,
                                                  bc.head_handle,
                                                  bc.bezier_arcs[0].cp4[0]
                                                  ]
                         ))]
        ba_num = 0
        for ba in bas: # BCurve.BezierArc
            cp4 = ba.cp4
            if len(cp4) == 1: # stroke with one anchor
                if abs(cp4[0]-center) < ZERO:
                    results.append(PathPointData(one_anchor_stroke=True,
                                                 stroke_num = gs_num,
                                                 arc_num = ba_num,
                                                 Bt = cp4[0]))
                ba_num += 1
                continue
            ts = bezier_intersect_circle(cp4, center, radius, restrict01=True)
            # Reduce multiple roots:
            #ts = list(set(ts))
            ts.sort()
            try:
                reduced = [ts[0]]
            except IndexError: # ts = []
                ba_num += 1
                continue
            for t in ts[1:]:
                if t-reduced[-1] > EQUAL_ROOT:
                    reduced.append(t)
                #else:
                #    print("Reject t="+str(t)+" with t-reduced[-1]="+str(t-reduced[-1]))
            for t in reduced:
                results.append(PathPointData(one_anchor_stroke=False,
                                             stroke_num = gs_num,
                                             arc_num = ba_num,
                                             parameter = t,
                                             Bt = bezier_rv(t, cp4)
                                             ))
            ba_num += 1
        gs_num += 1
    return results

# ------------------
# gv_split_by_circle
# ------------------
# Given gv:BCurve.GimpVectors and a circle (center, radius),
# split the gv at the intersection points with the circle.
# Return two list of new Bezier curves, one for the inside curves, and the
# other for the outside curves.
# Args:
# - gv:     BCurve.GimpVectors
# - center: complex
# - radius: float
# Returns:
# - [BCurve.BezierCurve] (inside)
# - [BCurve.BezierCurve] (outside)
def gv_split_by_circle(gv, center, radius):
    ZERO = 1e-8
    gv_anchors = gv_split_by_circle_at_anchors(gv, center, radius)
    pps_all = gv_intersect_circle(gv_anchors, center, radius) # [PathPointData]
    bc_list = []
    gs_num = 0
    split_bc_list = []
    for gs in gv_anchors.stroke_list: # BCurve.GimpStroke
        pps = [pp for pp in pps_all if pp.stroke_number == gs_num]
        split_bc_list += gs_split_by_pps(gs, pps)
        gs_num += 1
    inside_bcs = []
    outside_bcs = []
    for bc in split_bc_list:
        if bcurve_inside_circle(bc, center, radius):
            inside_bcs.append(bc)
        else:
            outside_bcs.append(bc)
    return inside_bcs, outside_bcs

# -----------------------------
# gv_split_by_circle_at_anchors
# -----------------------------
# Preliminary step for splitting by circle: split at anchors.
# Split each stroke of gv where an anchor is on the circle.
# Return new gv (copy of the old one with some strokes split).
# Args:
# - gv:     BCurve.GimpVectors
# - center: complex
# - radius: float
# Returns:
# - BCurve.GimpVectors
def gv_split_by_circle_at_anchors(gv, center, radius):
    from copy import deepcopy
    ZERO = 1e-8
    gv = deepcopy(gv)
    new_gs_list = []
    for gs in gv.stroke_list:
        split_indexes = []
        for i in range(1,len(gs.cp_list),3): # anchors
            anchor = gs.cp_list[i]
            if radius-ZERO < abs(anchor-center) < radius+ZERO:
                split_indexes.append(i)
        if len(split_indexes) == 0:
            new_gs_list.append(gs)  # No splitting
            continue
        first_i = split_indexes[0]
        last_i = split_indexes[-1]
        split_at_first_anchor = (first_i == 1)
        split_at_last_anchor = (last_i == len(gs.cp_list)-2)
        if len(split_indexes) == 1 and (split_at_first_anchor or split_at_last_anchor):
            if not gs.closed:           # No splitting: save as such
                pass
            elif split_at_first_anchor: # Closed: must make open at first anchor
                gs.cp_list += [gs.cp_list[0], gs.cp_list[1], gs.cp_list[1]]
                gs.cp_list[0] = gs.cp_list[1]
                gs.closed = False
            elif split_at_last_anchor:  # Closed: must make open at last anchor
                add_at_start = [gs.cp_list[-2],gs.cp_list[-2],gs.cp_list[-1]]
                gs.cp_list[-1] = gs.cp_list[-2]
                gs.cp_list = add_at_start + gs.cp_list
                gs.closed = False
            new_gs_list.append(gs)      # Save the modified
            continue
        if not split_at_first_anchor:
            first_new_gs = BCurve.GimpStroke(cp_list=gs.cp_list[:first_i+1]+[gs.cp_list[first_i]],
                                         closed=False)
            new_gs_list.append(first_new_gs)
        for k in range(len(split_indexes)):
            try:
                i,j = split_indexes[k], split_indexes[k+1]
            except IndexError:
                #print("break")
                break
            new_cp_list = gs.cp_list[i:j+1]
            new_cp_list = [new_cp_list[0]] + new_cp_list + [new_cp_list[-1]]
            new_gs = BCurve.GimpStroke(cp_list=new_cp_list,
                                         closed=False)
            new_gs_list.append(new_gs)
        if not split_at_last_anchor:
            last_new_gs = BCurve.GimpStroke(cp_list=[gs.cp_list[last_i]]+gs.cp_list[last_i:],
                                         closed=False)
            new_gs_list.append(last_new_gs)
        if gs.closed:
            if split_at_first_anchor and split_at_last_anchor:
                new_cp_list = [gs.cp_list[-2],gs.cp_list[-2],gs.cp_list[-1],
                               gs.cp_list[0],gs.cp_list[1],gs.cp_list[1]]
                new_gs_list.append(BCurve.GimpStroke(cp_list=new_cp_list,
                                                     closed=False))
            elif split_at_first_anchor: # but not at last
                new_gs_list[-1].cp_list += [gs.cp_list[0],gs.cp_list[1],gs.cp_list[1]]
                new_gs_list[-1].closed = False
            elif split_at_last_anchor: # but not at first
                add_at_start = [gs.cp_list[-2],gs.cp_list[-2],gs.cp_list[-1]]
                new_gs_list[0].cp_list = add_at_start + new_gs_list[0].cp_list
            elif len(split_indexes) > 0: # Must join two strokes with gap-closing arc
                gap_cp_list = [gs.cp_list[-1], gs.cp_list[0]]
                first_new_gs.cp_list = (last_new_gs.cp_list[:-1] + # replace
                                        gap_cp_list +
                                        first_new_gs.cp_list[1:])
                new_gs_list = new_gs_list[:-1]                     # remove
    new_gv = BCurve.GimpVectors(stroke_list=new_gs_list)
    return new_gv

# ---------------
# gs_split_by_pps
# ---------------
# Given gs:BCurve.GimpStroke and a list pps:[PathPointData]
# split the gs at the path points.
# Reject possible splittings at anchors
# (handled already by gv_split_by_circle_at_anchors).
# Return list of Bezier curves [BCurve.BezierCurve].
# Args:
# - gv:  BCurve.GimpStroke
# - pps: [PathPointData]
# Returns:
# - [BCurve.BezierCurve]
def gs_split_by_pps(gs, pps):
    ZERO = 1e-8
    bc_list_for_gs = []   # List bezier curves resulting from splitting this gs
    bc = gs.gs2bc()       # BCurve.BezierCurve
    pps.sort(key=(lambda p: p.arc_number))
    cp4_list = []
    len_list = []
    current = 0
    bas = bc.bezier_arcs
    if bc.closed:
        #(*) In Gimp a closed stroke is just marked closed.
        #(*) We must add one Bezier arc to close the gap:
        bas.append(BCurve.BezierArc(cp4=[bc.bezier_arcs[-1].cp4[-1],
                                         bc.tail_handle,
                                         bc.head_handle,
                                         bc.bezier_arcs[0].cp4[0]]
                                    )
                   )
    ba_num = 0
    splits_in_bc = 0
    for ba in bas:
        # splitting t's for this arc:
        ts = [pp.parameter for pp in pps if pp.arc_number == ba_num]
        ts = [t for t in ts if ZERO < t < 1-ZERO] # Reject splittings at anchors
        splits_in_bc += len(ts)
        sub_arcs = bezier_subdivide_at_parameters(ts, ba.cp4)
        cp4_list += sub_arcs
        n = len(ts)
        if n == 0:
            current += 1
            ba_num += 1
            continue
        else:
            len_list.append(current+1)
            len_list += [1]*(n-1)
            current = 1
        ba_num += 1
    len_list.append(current)
    cp4_sublists = []
    cum = 0
    for slen in len_list:
        cp4_sublists.append(cp4_list[cum : cum+slen])
        cum += slen
    for sublist in cp4_sublists:
        new_ba_list = [BCurve.BezierArc(cp4=cp4) for cp4 in sublist]
        if len(new_ba_list) == 0: raise Exception("ULOS")
        bc_list_for_gs.append(BCurve.BezierCurve(bezier_arcs=new_ba_list))
    if bc.closed and (splits_in_bc > 0):
        last = bc_list_for_gs[-1] # Must combine these two: last+first
        first = bc_list_for_gs[0]
        combined_bc = BCurve.BezierCurve(
            bezier_arcs = last.bezier_arcs + first.bezier_arcs,
            head_handle = last.head_handle,
            tail_handle = first.tail_handle,
            )
        bc_list_for_gs = bc_list_for_gs[1:-1] + [combined_bc]
    if bc.closed and (splits_in_bc == 0):
        bc1 = bc_list_for_gs[-1] # Last appended: must close properly
        last_arc = bc1.bezier_arcs[-1]
        bc1.bezier_arcs = bc1.bezier_arcs[:-1] # Drop last arc
        bc1.head_handle = last_arc.cp4[2]
        bc1.tail_handle = last_arc.cp4[1]
        bc1.closed = True
    return bc_list_for_gs

# --------------------
# bcurve_inside_circle
# --------------------
# Given a Bezier curve and a circle, assuming that the curve is either inside
# or outside of the circle (or on the circle), tell if it is inside or outside.
# (If the curve lies on the circle sufficiently closely, it is taken to be inside.)
# Arg:
# - bc:     BCurve.BezierCurve
# - center: complex
# - radius: float
def bcurve_inside_circle(bc, center, radius):
    ZERO = 1e-10
    # First try anchors:
    anchors = [ba.cp4[0] for ba in bc.bezier_arcs] + [bc.bezier_arcs[-1].cp4[-1]]
    # Find the anchor farthest from the circle (inside or outside):
    farthest = max(anchors, key=(lambda a: abs(abs(a-center)-radius)))
    distance = abs(farthest-center)
    if distance > radius+ZERO:
        return False
    elif distance < radius-ZERO:
        return True
    # If not found, try middle points of arcs:
    for ba in bc.bezier_arcs:
        try:
            p0,p1,p2,p3 = ba.cp4
            middle = (p0 + 3*p1 + 3*p2 + p3) / 8
        except ValueError: # one-point stroke: handled above
            #middle = ba.cp4[0]
            continue
        r = abs(middle-center)
        if r < radius-ZERO:
            return True
        elif r > radius+ZERO:
            return False
    # If not found, try some other points:
    for ba in bc.bezier_arcs:
        if len(ba.cp4) == 1:
            continue
        for t in [.15, .3, .45, .55, .7, .85]:
            r = abs(bezier_rv(t, ba.cp4) - center)
        if r < radius-ZERO:
            return True
        elif r > radius+ZERO:
            return False
    # Nothing found: the curve is so exactly(?) on the circle
    # that return that it is inside:
    return True

## This is not used anywhere.
## It is here only to record how the coefficients array in
## bezier_intersect_circle was created.
#def initialize_coefficients_for_bezier_intersect_circle():
#    coeff = []
#    for i in range(7):
#        coeff.append([[None,None,None,None], # a[i][k][l]
#                      [None,None,None,None],
#                      [None,None,None,None],
#                      [None,None,None,None]
#                     ])
#    c = [[  1,   0,   0,   0,   0,   0,   0], # c0
#         [ -6,   1,   0,   0,   0,   0,   0], # c1
#         [ 15,  -5,   1,   0,   0,   0,   0], # c2
#         [-20,  10,  -4,   1,   0,   0,   0], # c3
#         [ 15, -10,   6,  -3,   1,   0,   0], # c4
#         [ -6,   5,  -4,   3,  -2,   1,   0], # c5
#         [  1,  -1,   1,  -1,   1,  -1,   1], # c6
#         ]
#    for i in range(7):
#        for k in range(4):
#            for l in range(4):
#                coeff[i][k][l] = (3**(min(k,3-k)+min(l,3-l))) * c[i][k+l]
#    return coeff

# -----------------------
# bezier_intersect_circle
# -----------------------
# Given a Bezier curve (4 control points) and a circle (center and radius),
# find the parameter values of all intersection points.
# The plane is viewed as the complex number plane.
# Args:
# - cp:         [complex,complex,complex,complex]
# - center:     complex
# - radius:     float
# - restrict01: boolean (accept only 0 <= t <= 1?)
# Returns:
# - [float] (parameters t)
def bezier_intersect_circle(cp, center, radius, restrict01=True):
    coefficients = [
    [[  1,   0,   0,  0], [  0,   0,   0,  0], [  0,   0,   0,  0], [ 0,  0,  0, 0]],
    [[ -6,   3,   0,  0], [  3,   0,   0,  0], [  0,   0,   0,  0], [ 0,  0,  0, 0]],
    [[ 15, -15,   3,  0], [-15,   9,   0,  0], [  3,   0,   0,  0], [ 0,  0,  0, 0]],
    [[-20,  30, -12,  1], [ 30, -36,   9,  0], [-12,   9,   0,  0], [ 1,  0,  0, 0]],
    [[ 15, -30,  18, -3], [-30,  54, -27,  3], [ 18, -27,   9,  0], [-3,  3,  0, 0]],
    [[ -6,  15, -12,  3], [ 15, -36,  27, -6], [-12,  27, -18,  3], [ 3, -6,  3, 0]],
    [[  1,  -3,   3, -1], [ -3,   9,  -9,  3], [  3,  -9,   9, -3], [-1,  3, -3, 1]]
    ]
    ## Test:
    #coeff = initialize_coefficients_for_bezier_intersect_circle()
    #print("Test: "+str(coeff == coefficients))
    A = [0,0,0,0,0,0,0] # This will be the coefficients of the polynomial (deg=6)
    for i in range(7):
        s = 0
        for k in range(4):
            for l in range(4):
                s += coefficients[i][k][l] * vdot(cp[k]-center, cp[l]-center)
        A[i] = s
    A[0] -= radius**2
    m = max([abs(x) for x in A])
    A = [x/m for x in A]
    if restrict01:
        zeroes = zeroes_of_polynomial(A, [0,1])[0] # Solve
    else:
        zeroes = zeroes_of_polynomial(A)[0] # Solve
    #if restrict01:
    #    zeroes = [t for t in zeroes if 0 <= t <= 1]
    return zeroes

# ------------------------------
# bezier_subdivide_at_parameters
# ------------------------------
# Given a Bezier arc B(t) and a list of parameter values t1,t2,...,
# find control points for all subarcs when B(t), 0<=t<=1, is subdivided
# at parameter values t1,...
# The input Bezier arc is given as its 4 control points [p0,p1,p2,p3],
# and the output is a list of new sets of 4 control points [[q0,q1,q2,q3],...].
# The plane is viewed as the complex number plane.
# Args:
# - parameter_list: [float]                           ([t1,t2,...])
# - control_points: [complex,complex,complex,complex] ([p0,p1,p2,p3])
# Returns:
# - [[complex,complex,complex,complex]]               ([[q0,q1,q2,q3],...])
def bezier_subdivide_at_parameters(parameter_list, control_points):
    ts01 = [t for t in parameter_list if 0 < t < 1] # Drop t's outside (0,1)
    ts01.sort()
    try:
        ts = [ts01[0]]
    except IndexError: # Empty parameter list
        return [[p for p in control_points]] # copy
    for t in ts01[1:]: # Drop duplicates.
        if t != ts[-1]:
            ts.append(t)
    ts = [0]+ts+[1] # Add t=0,1
    # Collect pairs [t(i),t(i+1)]
    abs = []
    for i in range(1,len(ts)):
        abs.append([ts[i-1],ts[i]])
    arc_cps = []
    for a,b in abs:
        arc_cps.append(bezier_new_control_points(a, b, control_points))
    return arc_cps

# ---------
# gv_stitch
# ---------
# Stitch two objects:BCurve.GimpVectors together by combining strokes with
# equal endpoints.
# Args:
# - gv1:BCurve.GimpVectors
# - gv2:BCurve.GimpVectors
# Returns:
# - BCurve.GimpVectors
def gv_stitch(gv1, gv2):
    ZERO = 1e-4 ##########  test: 1e-5 works, 1e-6 not
    gs_list = gv1.stroke_list + gv2.stroke_list
    # Identifiers of strokes at each end of the stitched stroke:
    for gs in gs_list:
        gs.ends = [gs.stroke_name, gs.stroke_name] # Start with same stroke at each end
    new_gs_list = [] # combined
    count = 0
    while len(gs_list) > 0:
        count += 1 # temporary
        if count > 1000:
            raise Exception("gv_stitch: reached limit of rounds: 1000")
        if len(gs_list) == 1:
            new_gs_list.append(gs_list[0]) # All done
            break
        gs = gs_list[0]
        g0 = gs.cp_list[1] # first anchor of gs
        g1 = gs.cp_list[-2] # last anchor of gs
        #for h_index in range(1,len(gs_list)):
        for h_index in range(len(gs_list)-1,0,-1): # Backwards all except 0
            hs = gs_list[h_index]
            h0 = hs.cp_list[1] # first anchor of hs
            h1 = hs.cp_list[-2] # last anchor of hs
            # From gs_list find the stroke to be stitched to gs:
            found_h = True
            if abs(g1-h0) < ZERO:
                left_cp_list  = gs.cp_list
                right_cp_list = hs.cp_list
                left_ends = gs.ends
                right_ends = hs.ends
            elif abs(g1-h1) < ZERO:
                left_cp_list  = gs.cp_list
                right_cp_list = hs.cp_list[::-1] # reversed
                left_ends = gs.ends
                right_ends = [hs.ends[1], hs.ends[0]] # reversed
            elif abs(g0-h1) < ZERO:
                left_cp_list  = gs.cp_list[::-1] # reversed
                right_cp_list = hs.cp_list[::-1] # reversed
                left_ends = [gs.ends[1], gs.ends[0]] # reversed
                right_ends = [hs.ends[1], hs.ends[0]] # reversed
            elif abs(g0-h0) < ZERO:
                left_cp_list  = gs.cp_list[::-1] # reversed
                right_cp_list = hs.cp_list
                left_ends = [gs.ends[1], gs.ends[0]] # reversed
                right_ends = hs.ends
            else:
                found_h = False
                continue
            if left_ends[1] != right_ends[0]: # Ends do not meet: reject
                continue
            # Stitch the two strokes:
            new_cp_list = left_cp_list[:-1] + right_cp_list[2:]
            # Substitute the combined stroke for hs:
            new_stroke = BCurve.GimpStroke(cp_list = new_cp_list,
                                           closed = False,)
                                           #stroke_name = gs.stroke_name)
            new_stroke.ends = [left_ends[0], right_ends[1]]
            gs_list[h_index] = new_stroke
            break
        if not found_h:
            new_gs_list.append(gs) # Not found: move gs to results
        gs_list = gs_list[1:]  # Remove gs from gs_list
    # Close any non-closed strokes that should be closed:
    for i in range(len(new_gs_list)):
        gs = new_gs_list[i]
        if len(gs.cp_list) <= 3: # 1-anchor stroke
            continue
        if gs.closed:
            continue
        g0 = gs.cp_list[1] # first anchor of gs
        g1 = gs.cp_list[-2] # last anchor of gs
        if abs(g0-g1) < ZERO:
            gs.cp_list = [gs.cp_list[-3]] + gs.cp_list[1:-3]
            gs.closed = True
    result_gv = BCurve.GimpVectors(stroke_list = new_gs_list)
    return remove_zero_arcs(result_gv, limit=1e-3)


# --------------
# gv_add_anchors
# --------------
# Do work of path_add_anchors
# Args:
# - gv: BCurve.GimpVectors
# - list_point_data: [PathPointData]
# Returns:
# - BCurve.GimpVectors
# Note: It may happen that some stroke is closed and that some intersection points
# occur on the gap-closing (imaginary) arc. In that case the stroke is extended
# to contain the intersection points.
def gv_add_anchors(gv, list_point_data):
    new_gv = deepcopy(gv)
    gs_num = 0
    for gs in new_gv.stroke_list: # BCurve.GimpVectors
        # Find the items:PathPointData for this stroke:
        gs_pps = [pp for pp in list_point_data if pp.stroke_number == gs_num]
        #
        bc = gs.gs2bc()           # BCurve.BezierCurve
        bas = bc.bezier_arcs      # [BCurve.BezierArc]
        if bc.closed: # Add one more arc to close the gap
            bas.append(BCurve.BezierArc(cp4 = [bc.bezier_arcs[-1].cp4[-1],
                                      bc.tail_handle,
                                      bc.head_handle,
                                      bc.bezier_arcs[0].cp4[0]]))
        new_bas = []
        ba_num = 0
        for ba in bas:            # BCurve.BezierArc
            # Find the subdividing parameter values for this Bezier arc:
            ba_ts = [pp.parameter for pp in gs_pps if pp.arc_number == ba_num]
            ba_ts = [t for t in ba_ts if 0 < t < 1]
            ba_ts.sort()
            
            new_cp4s = bezier_subdivide_at_parameters(ba_ts, ba.cp4)
            new_bas += [BCurve.BezierArc(cp4=new_cp4) for new_cp4 in new_cp4s]
            
            ba_num += 1
        if bc.closed: # Remove the last arc and take from it new head and tail:
            last_arc = new_bas[-1]
            new_bc = BCurve.BezierCurve(bezier_arcs = new_bas[:-1], # remove last
                                    head_handle = last_arc.cp4[2],
                                    tail_handle = last_arc.cp4[1],
                                    closed = bc.closed,
                                    curve_name = gs.stroke_name)
        else:
            new_bc = BCurve.BezierCurve(bezier_arcs = new_bas,
                                    head_handle = bc.head_handle,
                                    tail_handle = bc.tail_handle,
                                    closed = bc.closed,
                                    curve_name = gs.stroke_name)
        new_gs = new_bc.bc2gs()
        new_gv.stroke_list[gs_num] = new_gs
        gs_num += 1
    return new_gv

# ----------------
# remove_zero_arcs
# ----------------
# Remove zero bezier arcs, except for 1-anchor strokes.
# Args:
# - gv: BCurve.GimpVectors
# - limit: float
# Returns:
# - BCurve.GimpVectors
# Note: Not thoroughly tested.
def remove_zero_arcs(gv, limit=1e-3):
    new_gs_list = []
    for gs in gv.stroke_list:
        new_cp_list = [gs.cp_list[0]]
        for i in range(1, len(gs.cp_list)-4, 3):
            p0,p1,p2,p3 = gs.cp_list[i:i+4]
            if abs(p0-p3) > limit:
                new_cp_list +=[p0,p1,p2]
            elif (abs(p0-p1) > limit) or (abs(p2-p3) > limit):
                new_cp_list +=[p0,p1,p2]
            else: # Zero arc, reject
                pass
        new_cp_list += gs.cp_list[-2:]
        if gs.closed:
            p0,p1,p2,p3 = gs.cp_list[-2:] + gs.cp_list[:2]
            if abs(p0-p3) > limit:
                pass
            elif (abs(p0-p1) > limit) or (abs(p2-p3) > limit):
                pass
            else: # Zero arc, reject
                new_cp_list = new_cp_list[1:-2]
        new_gs_list.append(BCurve.GimpStroke(cp_list = new_cp_list,
                                                       closed = gs.closed,
                                                       stroke_name = gs.stroke_name))
    return BCurve.GimpVectors(stroke_list = new_gs_list,
                              name = gv.name)

# -------------------------
# bezier_arc_end_directions
# -------------------------
# Given a Bezier arc, find, at each end, the direction vector of the arc.
# Note: this does not mean the derivatives of the parametric representation,
# but non-zero vectors which are tangential to the arc and point in the
# direction where the arc is located when viewed from that end point.
# Return the result as two RowVectors, or as None if the arc is one point.
# Args:
# - ba: BCurve.BezierArc
# Returns: either None (when the arc is one point), or
# - RowVector (direction vector at starting point)
# - RowVector (direction vector at ending point)
def bezier_arc_end_directions(ba):
    ZERO = 1e-12
    if len(ba.cp4) == 1:
        return None
    p0,p1,p2,p3 = ba.cp4
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    if vdot(p01,p01) > ZERO:
        head_dir = p01
    elif vdot(p12,p12) > ZERO:
        head_dir = p12
    elif vdot(p23,p23) > ZERO:
        head_dir = p23
    else:
        return None
    if vdot(p23,p23) > ZERO:
        tail_dir = -p23
    elif vdot(p12,p12) > ZERO:
        tail_dir = -p12
    elif vdot(p01,p01) > ZERO:
        tail_dir = -p01
    else:
        raise Exception("bezier_arc_end_directions: Should never happen!")
    return head_dir, tail_dir

# ------------------------
# bezier_arc_map_subdivide
# ------------------------
# Given a Bezier arc B(t):[0,1]->C and a plane transformation F:C->C,
# subdivide the interval [0,1] in 2**n parts, 0 < t1 < t2 < ... < 1,
# so that the chords between successive points F(B(ti)) are very roughly of
# equal length.
# The Bezier arc is given as a list of 4 control points ([complex]),
# and the map is F = pt.point_map where pt:PlaneTransformation.
# Return (None or) two lists:
# - [[t,F(B(t))],...] where F = pt.point_map
# - [[t,J(B(t))],...] where J = pt.jacobian
# Args:
# - cp: [complex,...] (four control points)
# - pt: PlaneTransformation
# - n: integer
# Returns:
# - None or 
#  ([[t0,F0],[t1,F1], ...], [[t0,J0],[t1,J1], ...])
#  where ti:float, Fi:complex, Ji:complex
# (sorted relative parameter values for the subdivision, 0 < t0 < t1 ... < 1)
# Note: None is returned if the input arc cp is of zero length.
def bezier_arc_map_subdivide(cp, pt, n=1):
    ZERO_LENGTH = 1e-10
    F = pt.point_map
    def length2(a,b): # a,b:RowVector = complex
        return ((a-b)*(a-b).conjugate()).real
    # List of subarcs: [[[lo,F(B(lo)), [hi,F(B(hi))], length2], ...],
    # kept constantly sorted according to length2, longest first.
    F0 = F(bezier_rv(0,cp))
    F1 = F(bezier_rv(1,cp))
    subarcs = [[[0,F0], [1,F1], length2(F0,F1)]]
    if length2(F0,F1) < ZERO_LENGTH:
        Fmeans = [F(bezier_rv(t,cp)) for t in [.25,.5,.75]]
        maxdist20 = max([length2(F0,Fm) for Fm in Fmeans])
        maxdist21 = max([length2(Fm,F1) for Fm in Fmeans])
        if max(maxdist20,maxdist21) < ZERO_LENGTH:
            return None
    max_i_rounds = 0
    #no_break = 0
    for k in range(2**n - 1):
        longest = subarcs[0] # Zeroth is the longest
        loF, hiF = longest[0], longest[1]
        lo, hi = loF[0], hiF[0]
        t_try = (lo + hi)/2. # Ad hoc: split first at middle t
        # Try to find better t_try:
        try_results = []
        for i in range(20):
            F_try = F(bezier_rv(t_try, cp))
            d0 = sqrt(length2(loF[1], F_try)) # chord length
            d1 = sqrt(length2(F_try, hiF[1])) # chord length
            try:
                dd_try = d1/(d0+d1) # Trying to get d0/d1=1 => dd_try=1/2
                try_results.append([t_try, dd_try])
                if .35 < dd_try < .65: # Ad hoc
                    break # Found ok: break
                dd_try = (1-dd_try)*lo + dd_try*hi # Ad hoc
                #t_try = (t_try + dd_try) / 2 # Ad hoc
                t_try = (t_try + (i+1)*dd_try) / (i+2) # Ad hoc  KOE
            except ZeroDivisionError:
                break
        else:
            pass
        if len(try_results)>0:
            best_try = min(try_results, key=(lambda x:abs(x[1]-.5)))
            mi = best_try[0]
            miF = [mi, F(bezier_rv(mi,cp))]
            left  = [loF, miF, length2(loF[1],miF[1])]
            right = [miF, hiF, length2(miF[1],hiF[1])]
            # Replace longest with left,right and sort:
            subarcs = sorted([left,right] + subarcs[1:],
                              key = (lambda x:-x[2]))
        else: # Zero length arc: did not find anything
            raise Exception("bezier_arc_map_subdivide: Called with zero length arc?")
    loFs = [x[0] for x in subarcs]           # Drop hi's and lengths
    loFs = sorted(loFs, key=(lambda x:x[0])) # Sort according to t
    t_F_list = loFs[1:] # Drop zeroth
    t_jac_list = [[t, pt.jacobian(bezier_rv(t,cp))] for t,Ft in t_F_list]
    return t_F_list, t_jac_list


#---------------------------------------------------------
#                Some routines for Support
#---------------------------------------------------------

# ----------------
# map_make_support
# ----------------
# Given pt:PlaneTransformation and ba:BCurve.BezierArc (assumed to have 4
# control points), imagine that pt is applied to the arc; compute a support
# for the image.
# Args:
# - pt: PlaneTransformation
# - ba: BCurve.BezierArc
# Returns:
# - Support
# Note:
# 1. The end points (anchors) are mapped by pt.point_map.
# 2. The direction vectors derived from the inner control points (handles)
#    are mapped by pt.jacobian. Then they will be tangential to the (imagined)
#    mapped arc. The called routine 'bezier_arc_end_directions' takes care of
#    the cases of zero handles but returns None if the arc is one point
#    (coincident control points).
def map_make_support(pt, ba):
    try:
        rv0 = ba.cp4[0] # RowVector = complex
        rv3 = ba.cp4[3] # RowVector = complex
    except IndexError:
        raise Exception("map_make_support: Arc does not have 4 control points.")
    try:
        direction_vectors = bezier_arc_end_directions(ba)
    except:
        raise ExitError("map_make_support: bezier_arc_end_directions")
    if direction_vectors is None:
        raise Exception("map_make_support: Arc has 4 equal control points.")
    rv_01, rv_32 = direction_vectors # RowVectors = complex
    #
    pt_rv0 = pt.point_map(rv0) # RowVector = complex
    pt_rv3 = pt.point_map(rv3) # RowVector = complex
    pt_rv01 = rv_times_M2x2_transpose(rv_01, pt.jacobian(rv0)) # RowVector = complex
    pt_rv32 = rv_times_M2x2_transpose(rv_32, pt.jacobian(rv3)) # RowVector = complex
    pt_rv1 = pt_rv0 + pt_rv01
    pt_rv2 = pt_rv3 + pt_rv32
    #
    S = [pt_rv0, pt_rv1, pt_rv2, pt_rv3]
    return Support(S) # May raise NoSupportError


# ----------------------------------------------------------------
#   Map and approximate a bezier arc: various simple algorithms
# ----------------------------------------------------------------

# --------------------
# line_segment2ba_list
# --------------------
# Given
# - a plane transformation F
# - a line segment in the plane,
# compute the control points of the image Bezier arc F(line_segment).
# (Implemented formulas (1.25), (4.28)).
# Args:
# - pt:           PlaneTransformation
# - line_segment: [ControlPoint,ControlPoint] = [complex,complex]
# Returns:
# - [BCurve.BezierArc] (butting Bezier arcs)
# Note: This is exact! Possible because make_bezier_map_C_to_C
# returns a map which sends line segments to Bezier arcs.
def line_segment2ba_list(pt, line_segment):
    f = pt.point_map
    jac = pt.jacobian
    a,b = line_segment
    #
    Q01 =  rv_times_M2x2_transpose(-a+b, jac(a)) / 3.
    Q32 = -rv_times_M2x2_transpose(-a+b, jac(b)) / 3.
    #
    Q0 = f(a)
    Q1 = f(a) + Q01
    Q2 = f(b) + Q32
    Q3 = f(b)
    rv_cp4 = [Q0,Q1,Q2,Q3] # [RowVectors]
    return [BCurve.BezierArc(rv_cp4)]

# ----------------------
# simple_support2ba_list
# ----------------------
# From support:Support create a Bezier arc with the simple rule:
# Handle lengths = one third of the chord length.
# Arg:
# - support:Support
# Returns:
# - [BCurve.BezierArc]
def simple_support2ba_list(support):
    S0rv,S1rv,S2rv,S3rv = support.support_points # complex
    S01rv = -S0rv + S1rv
    S32rv = -S3rv + S2rv
    try:
        try:
            inv01rv = 1./ abs(S01rv)
        except ZeroDivisionError:
            S01rv = -S0rv + S2rv
            try:
                inv01rv = 1./ abs(S01rv)
            except ZeroDivisionError:
                S01rv = -S0rv + S3rv
                inv01rv = 1 / abs(S01rv)
        try:
            inv32rv = 1./ abs(S32rv)
        except ZeroDivisionError:
            S32rv = -S3rv + S1rv
            try:
                inv32rv = 1./ abs(S32rv)
            except ZeroDivisionError:
                S32rv = -S3rv + S0rv
                inv32rv = 1./ abs(S32rv)
    except ZeroDivisionError: # zero chord
        return [BCurve.BezierArc(cp4=[S0rv,S0rv,S3rv,S3rv])]
    chord_len3 = abs(S0rv-S3rv) / 3.
    #S01rvlen = abs(S01rv)
    #S32rvlen = abs(S32rv)
    #p01rv = S01rv*chord_len3/S01rvlen
    #p32rv = S32rv*chord_len3/S32rvlen
    p01rv = S01rv*chord_len3*inv01rv
    p32rv = S32rv*chord_len3*inv32rv
    p0rv = S0rv
    p3rv = S3rv
    p1rv = p0rv + p01rv
    p2rv = p3rv + p32rv
    cp4 = [p0rv,p1rv,p2rv,p3rv]
    return [BCurve.BezierArc(cp4=cp4)]


# ----------------
# ba2balist_simple
# ----------------
# Transform a Bezier arc ba:BCurve.BezierArc (assumed to have 4 control points)
# by pt:PlaneTransformation and make an approximate Bezier arc using a simple
# algorithm.
# Return the result as a list of butting Bezier arcs.
# Simple algorithm:
# 1. The arc is (if log_nsubarcs>0) subdivided;
# 2. the subdivision points are mapped exactly by pt;
# 3. the tangents (and directions) at the subdivision points will be exact,
#    but the lengths of the new handles in each subarc are set simply
#    to 1/3 of the chord length.
# The number of subarcs in the subdivision is 2**log_nsubarcs (>= 2**0 = 1).
# If log_nsubarcs=0, or if the arc is deemed to be a straight line segment,
# there will be no subdivision.
# The argument bezier_case:boolean tells if we are doing transformation
# constructed from a Bezier arc; in that case straight line segments are
# treated by a special exact algorithm.
# Args:
# - pt:           PlaneTransformation
# - ba:           BCurve.BezierArc
# - log_nsubarcs: integer (>=0)
# - shaping_input: dict
# - messages:     boolean (messages in Gimp's error console?)
# Returns:
# - [BCurve.BezierArc] (butting Bezier arcs)
#
def ba2balist_simple(pt, ba, shaping_input, log_nsubarcs=2, messages=False):
    log_nsubarcs = max(0,log_nsubarcs)
    cp4xy = ba.cp4 # [ControlPoint]
    if collinear(cp4xy): # Straight line segment
        if shaping_input['case'] == 'path by bezier':
            # Use the exact algorithm and return
            line_segment = [ba.cp4[0],ba.cp4[-1]]
            return line_segment2ba_list(pt, line_segment)
        if shaping_input['case'] == 'path by projective': # collineation
            # Collineation: return straight line segment
            cp4_pt = [pt.point_map(cp) for cp in cp4xy]
            return [BCurve.BezierArc(cp4=cp4_pt)]
        # Make new handles with the 1/3-rule to easify calculations.
        s0,s1,s2,s3 = cp4xy # complex
        s1 = (2*s0 + s3) / 3.
        s2 = (s0 + 2*s3) / 3.
        cp4xy = [s0,s1,s2,s3]
    # Subdivide arc in 2**log_nsubarcs subarcs:
    t_subdiv_jac = bezier_arc_map_subdivide(cp4xy, pt, n=log_nsubarcs) # [[t,F(B(t))],...]
    if t_subdiv_jac is None: # ba is of zero length
        return [BCurve.BezierArc(cp4 = [pt.point_map(cp) for cp in ba.cp4])]
    else:
        t_subdiv = t_subdiv_jac[0] # Drop the jacobians
    points = [tF[0] for tF in t_subdiv] # t-values
    # Create support for each subarc:
    skipped = 0
    if log_nsubarcs == 0:
        try:
            support_list = [map_make_support(pt, ba)] # Takes jacobian into account
        except Exception as e:
            # Two ways this can happen:
            # 1. map_make_support raises this: the arc has only one control point
            #    (possible if comes from a stroke with only one anchor)
            # 2. bezier_arc_end_directions raises this: The arc has 4 control points
            #    but they are equal (or very close).
            # In either case, treat the arc as a very tiny line segment.
            if messages:
                m = "ba2balist_simple: map_make_support raised exception: "+str(e)
                m += "\nTrying to treat the arc as straight line segment!"
                gimp_message(m)
            line_segment = [ba.cp4[0],ba.cp4[-1]]
            return line_segment2ba_list(pt, line_segment)
        except NoSupportError as e:
            # Comes from map_make_support when Support raises NoSupportError:
            # 1. one of the would-be handles is zero (or close to) (e.error=1), or
            # 2. all 4 would-be support_points are collinear (or close to) (e.error=2).
            # In either case, take the would-be support_points as control points for
            # the new arc. (Ad-hoc, in this simple algorithm!)
            if messages:
                m = "Met Exception NoSupportError: "+str(e)
                m += "\ne.error = "+str(e.error)
                m += "\nUsing ad-hoc solution!"
                gimp_message(m)
            # e.points: [complex,...] = the would-be support_points
            #cp4 = [points[0],points[0],points[3],points[3]]  ######  KOE
            #cp4 = points
            #print("simple: return 5:") ###################  HUOM kunnossa nyt?
            #print(str(BCurve.BezierArc(cp4=cp4)))
            #return [BCurve.BezierArc(cp4=cp4)]
            return [BCurve.BezierArc(cp4=e.points)] # ok now?
    else:
        points01 = [0]+points+[1]
        support_list = []
        for i in range(len(points01)-1):
            sub_cp = bezier_new_control_points(points01[i], points01[i+1], cp4xy)
            sub_ba = BCurve.BezierArc(cp4=sub_cp)
            try:
                support = map_make_support(pt, sub_ba) # Takes jacobian into account
            except Exception as e:
                # Two ways this can happen:
                # 1. map_make_support raises this: the arc has only one control point
                #    (possible if comes from a stroke with only one anchor)
                # 2. bezier_arc_end_directions raises this: The arc has 4 control points
                #    but they are equal (or very close).
                # In either case, treat the arc as a very tiny line segment.
                if messages:
                    m = "ba2balist_simple: map_make_support raised exception: "+str(e)
                    m += "\nTrying to treat the arc as straight line segment!"
                    gimp_message(m)
                line_segment = [ba.cp4[0],ba.cp4[-1]]
                return line_segment2ba_list(pt, line_segment)
            except NoSupportError as e:  ########  TO DO
                #print("ba2balist_simple got NoSupportError:")
                #print(str(e))
                # Ad-hoc as above
                if messages:
                    m = "Met Exception NoSupportError: "+str(e)
                    m += "\nUsing ad-hoc solution!"
                    gimp_message(m)
                #e.points: [[float,float],...] = the would-be support_points
                # jacobian already taken into account
                support = Support(e.points, sloppy=True) # Ad-hoc: ignore error
            support_list.append(support)
    ba_list = []
    for support in support_list: # jacobian already taken into account
        ba_list += simple_support2ba_list(support) # Create simple
    if skipped > 0 and messages:
        m = "Skipped " +str(skipped) + " very short(?) arcs."
        m += "\nThe result may contain oddities."
        gimp_message(m)
    return ba_list

# ----------------------------------------------------------------
#    Map and approximate a bezier arc: full recursive algorithm
# ----------------------------------------------------------------

# ----------------
# ba2balist_approx
# ----------------
# Transform a Bezier arc ba:BCurve.BezierArc (assumed to have 4 control points)
# by pt:PlaneTransformation using an approximating algorithm.
# Return the result as a list of butting Bezier arcs.
# The argument bezier_case:boolean tells if we are doing transformation
# constructed from a Bezier arc; in that case straight line segments are
# treated by a special exact algorithm.
# Args:
# - pt:      PlaneTransformation
# - ba:      BCurve.BezierArc
# - rec_depth:   integer (>=0)
# - shaping_input: dict
# - print_info:  boolean
# - messages:    boolean (messages in Gimp's error console?)
# Returns:
# - [BCurve.BezierArc] (butting Bezier arcs)
# - float (relative error)
# - integer (recursion depth)
#
def ba2balist_approx(pt,
                     ba,
                     shaping_input,
                     rec_depth=0,
                     print_info=False,
                     messages=False,
                     ):
    try:
        MAX_REC_DEPTH = shaping_input['max_rec_depth']
    except KeyError:
        #MAX_REC_DEPTH = 4
        MAX_REC_DEPTH = 10
        #MAX_REC_DEPTH = 12
        #MAX_REC_DEPTH = 16
    MAX_ALLOWED_ERROR = 0.002 # Sufficient?
    #MAX_ALLOWED_ERROR = 0.1 # ok?
    #MAX_ALLOWED_ERROR = 0.5 # ok?
    #MAX_ALLOWED_ERROR = 0.0005 # Unnecessary?
    LOG_SPOINTS = 5
    SPOINTS = 2**LOG_SPOINTS  # Number of subdivision points
    NPOINTS = 4 # Number of points used to find the approximations = SPOINTS/NPOINTS
    EPOINTS = 2 # Number of points used in error calculations = SPOINTS/EPOINTS
    def get_size(cp, targets): # cp, targets: [complex,...]
        def distance2(x,y):
            return ((x-y)*(x-y).conjugate()).real
        p0,p1,p2,p3 = cp # [float,float]
        middle = (p0+p3)/2 # Middle point of chord
        return sum([distance2(middle, target) for target in targets])
    def get_rel_error(ba_list_try, yard_stick):
        total_rel_error = 0
        for ba_try in ba_list_try:
            # Error target points (for error calculations):
            cp4xy = ba_try.cp4 # [[float,float],...]
            t_subdiv_jac = bezier_arc_map_subdivide(cp4xy, pt, n=LOG_SPOINTS) # [[t,F(B(t))],...]
            if t_subdiv_jac is None:
                return 0.
            else:
                t_subdiv = t_subdiv_jac[0] # Drop the jacobians
            t_etargets = t_subdiv[EPOINTS-1: -EPOINTS+1: EPOINTS] # [[t,RowVector],...]
            etargets = [tt[1] for tt in t_etargets]
            #
            bf = BezierFunction(cp4xy)
            abs_error = sum_dist2_arc_points(bf, etargets,sloppy=True)
            rel_error = sqrt(abs_error / yard_stick)
            total_rel_error = max(total_rel_error, rel_error)
        return total_rel_error
    # directions_ok: Check that the arc does not turn more than 90 degrees.
    # To that end, compute dot products of the direction vector at the start
    # with the following vectors:
    # - the direction vector at the end;
    # - the direction vectors at the subdivision points.
    # If there is a negative dot product, return False, otherwise True.
    def directions_ok(ba, t_jac):
        direction_vectors = bezier_arc_end_directions(ba)
        if direction_vectors is None:
            raise Exception("directions_ok: Arc has 4 equal control points.")
        head_dir, tail_dir = direction_vectors
        tail_dir = -tail_dir # To point in the direction of the derivative!
        pt_head_dir = rv_times_M2x2_transpose(head_dir, pt.jacobian(ba.cp4[0]))
        pt_tail_dir = rv_times_M2x2_transpose(tail_dir, pt.jacobian(ba.cp4[3]))
        if vdot(pt_head_dir, pt_tail_dir) < 0:
            return False
        for t, jac in t_jac:
            B_dot = bezier_dot_rv(t, ba.cp4)
            pt_B_dot = rv_times_M2x2_transpose(B_dot, jac)
            vd = vdot(pt_head_dir, pt_B_dot)
            if vd < 0:
                #rv = bezier_rv(t, ba.cp4)
                #pt_rv = pt.point_map(rv)
                return False
        return True
    # handle_MAX_REC_DEPTH:
    # Reaching MAX_REC_DEPTH but having no acceptable approximation.
    def handle_MAX_REC_DEPTH(pt, ba, targets, message):
        m = message
        m += "  Reached MAX_REC_DEPTH: no recursion"
        m += "\n  Using the simple algorithm."
        if print_info:
            print(m)
        bas = ba2balist_simple(pt, ba, shaping_input, log_nsubarcs=2) # Do 1 splitting
        yard_stick = get_size(ba.cp4, targets)
        rel_err = get_rel_error(bas, yard_stick)
        if messages:
            gm = "The plugin met a failure."
            gm += "\nCannot use recursion (at maximum allowed depth already)."
            gm += "\nResorting to a simple algorithm."
            gm += "\nThe result is probably not too accurate."
            gimp_message(gm)
        return bas, rel_err
    # If NoSupportError, nse==2, check if the mapped arc indeed is straight.
    # If not, can go to resursion.
    def check_linearity(pt, ba):
        p0,p1,p2,p3 = ba.cp4
        ptmap = pt.point_map
        test_points = [p0, (p0+p3)/2, p3]
        pt_test_points = [pt.point_map(p) for p in test_points]
        if collinear(pt_test_points):
            return True # linear arc
        else:
            return False
    if rec_depth == 0:
        if print_info:
            print("\nMAX_ALLOWED_ERROR = "+str(MAX_ALLOWED_ERROR))
            print("MAX_REC_DEPTH = "+str(MAX_REC_DEPTH))
            print()
    error_message = ''     # Used in case of error
    cp4xy = ba.cp4 # [complex,...]
    if shaping_input['case'] == 'path by bezier':
        # If straight line segment, use the exact algorithm:
        if collinear(cp4xy): # Straight line segment
            line_segment = [ba.cp4[0],ba.cp4[-1]]
            bas = line_segment2ba_list(pt, line_segment)
            rel_err = 0
            return bas, rel_err, rec_depth 
    if shaping_input['case'] == 'path by projective': # collineation
        # If straight line segment, return straight line segment
        if collinear(cp4xy):
            cp4_pt = [pt.point_map(cp) for cp in cp4xy]
            return [BCurve.BezierArc(cp4=cp4_pt)], 0., rec_depth
    # Create subdivision:
    t_subdiv_jac = bezier_arc_map_subdivide(cp4xy, pt, n=LOG_SPOINTS)
    if t_subdiv_jac is None:
        if print_info:
            print("ba2balist_approx: t_subdiv_jac is None")
        pt_ba = BCurve.BezierArc(cp4 = [pt.point_map(cp) for cp in ba.cp4])
        return [pt_ba], 0., rec_depth
    else:
        t_subdiv, t_jac = t_subdiv_jac # [[t,F(B(t))],...], [t,J(B(t))],...]
    t_targets  = t_subdiv[NPOINTS-1: -NPOINTS+1: NPOINTS] # [[t,RowVector],...]
    t_etargets = t_subdiv[EPOINTS-1: -EPOINTS+1: EPOINTS] # [[t,RowVector],...]
    targets = [tt[1] for tt in t_targets]   # Targets for approximation
    etargets = [tt[1] for tt in t_etargets] # Targets for error calculation
    # Potentially initial subdivision:
    # Check directions: if the curve turns > 90 degrees,
    # by-pass all tries of approximation.
    # Instead, go to the recursion part (split in two and so on)
    rel_err = 'unknown'
    go_to_recursion = not directions_ok(ba, t_jac)
    if go_to_recursion:
        rel_err = 'unknown'
        if print_info:
            pt_start = pt.point_map(ba.cp4[0])
            pt_end = pt.point_map(ba.cp4[-1])
            print()
            print("Start: "+str(pt_start))
            print("End:   "+str(pt_end))
            print("Directions not ok.")
        pass
    else: # Try to do approximation
        # Target points:
        if print_info:
            t_rvB01 = [(t, bezier_rv(t,cp4xy)) for t in [0,1]] # [(float,RowVector=complex)]
            t_targets01 = [(trvb[0],pt.point_map(trvb[1])) for trvb in t_rvB01]
            targets01 = [tt[1] for tt in t_targets01]
            print()
            print("Start: "+str(targets01[0]))
            print("End:   "+str(targets01[1]))
            print("targets:")
            for ttt in targets: print(str(ttt))
        try:
            pt_support = map_make_support(pt, ba)
            ok_support = True
            if print_info:
                if ok_support:
                    print("Support ok.")
                    print("    Support points:")
                    for sp in pt_support.support_points:  print("        "+str(sp))
                else:
                    print("No support.")
        except Exception as e:
            ok_support = False
            if print_info:
                print("No support. Exception from map_make_support:")
                print(str(e))
        except NoSupportError as nse:
            if nse.error == 2:
                if check_linearity(pt, ba):
                    # Collinear would-be support points: return straight line segment
                    xy = nse.points
                    xy0 = xy[0]
                    xy1 = xy[0] # Take zero length handle
                    xy2 = xy[3] # Take zero length handle
                    xy3 = xy[3]
                    cp4_straight = [xy0,xy1,xy2,xy3]
                    ba_straight = [BCurve.BezierArc(cp4_straight)]
                    if print_info:
                        print("nse.error == 2")
                        print(nse.msg)
                        print("No support, arc straight.")
                        print("    would-be support points:")
                        for sp in nse.points:  print("        "+str(sp))
                        print("    Returning straight line segment, control points:")
                        for p in cp4_straight: print("        "+str(p))
                    return ba_straight, 0., rec_depth
                else:
                    if print_info:
                        print("nse.error == 2 but check_linearity = False.")
                        print("Going to recursion")
                    rel_err = 'None'
                    ok_support = False
            else: # Go to recursion
                rel_err = 'None'
                ok_support = False
        if not ok_support: # No support
            error_message = 'No support'
        else: # Support ok
            try:
                cpxy, abs_err, rel_err = arc_approx_points(pt_support,
                                  targets,
                                  respect_directions=True,
                                  sloppy=True,
                                  error_targets=etargets
                                  )
                #failure = False
                cp4_result = cpxy
                ba_result = BCurve.BezierArc(cp4 = cp4_result)
                if rel_err < MAX_ALLOWED_ERROR: # OK result: return
                    ba_list = [ba_result]
                    if print_info:
                        print("  rec_depth = "+str(rec_depth)+", rel_err = "+str(rel_err)+"  OK")
                        print("    Returning Bezier arc with control points:")
                        for cp in ba_result.cp4:
                            print("        "+str(cp))
                    return ba_list, rel_err, rec_depth
                elif rec_depth >= MAX_REC_DEPTH: # No failure: got ba_result; use it!
                    ba_list = [ba_result]
                    if print_info:
                        print(">>>>                  Reached MAX_REC_DEPTH: no recursion")
                        print(">>>>  rec_depth = "+str(rec_depth)+", rel_err = "+str(rel_err)+"  QUIT")
                        #draw_polyline(image, targets, name='targets', closed=False)
                        #start, end = ba_result.cp4[0], ba_result.cp4[-1]
                        #draw_polyline(image, [start,end], name='chord', closed=False)
                    return ba_list, rel_err, rec_depth
                else: # Go to recursion
                    error_message = 'rel_err too large'
            except Exception as e: # Failure: arc_approx_points?
                # Go to recursion
                #failure = True
                m = "ba2balist_approx: Failure:"
                m += "\n"+str(e)
                error_message = m
    # Recursion:
    if print_info:
        print("  rec_depth = "+str(rec_depth)+", rel_err = "+str(rel_err)+"  GOING TO RECURSION")
    if rec_depth > MAX_REC_DEPTH:
        ba_list, rel_err = handle_MAX_REC_DEPTH(pt, ba, targets, error_message)
        return ba_list, rel_err, rec_depth
    # Split the Bezier arc in two:
    # Idea: split_t will be that parameter value t for which the target point
    # is the most distant from the chord.
    rvp0, rvp3 = pt.point_map(cp4xy[0]), pt.point_map(cp4xy[3]) # Chord
    far_t_target = max(t_targets,
                       key=(lambda tt: abs(dist_from_line(rvp0, rvp3, tt[1]))))
    split_t = far_t_target[0]
    if print_info:
        split_B = bezier_rv(split_t, cp4xy)
        split_pt = pt.point_map(split_B)
        print("    RECURSION")
        print("    Splitting at t = "+str(split_t)+ ", B(t) = "+str(split_B))
        print("    Adding point "+str(split_pt))
    #
    cp4_left  = bezier_new_control_points(0, split_t,  cp4xy)
    cp4_right = bezier_new_control_points(split_t, 1., cp4xy)
    ba_left = BCurve.BezierArc(cp4=cp4_left)
    ba_right = BCurve.BezierArc(cp4=cp4_right)
    # Recursive calls:
    left  = ba2balist_approx(pt,
                             ba_left,
                             shaping_input,
                             rec_depth=rec_depth+1,
                             print_info=print_info,
                             messages=messages
                             )
    right = ba2balist_approx(pt,
                             ba_right,
                             shaping_input,
                             rec_depth=rec_depth+1,
                             print_info=print_info,
                             messages=messages
                             )
    ba_list_left, rel_err_left, rec_depth_left = left
    ba_list_right, rel_err_right, rec_depth_right = right
    #
    ba_list = ba_list_left + ba_list_right
    rel_err = max(rel_err_left, rel_err_right)
    rec_depth = max(rec_depth_left, rec_depth_right)
    return ba_list, rel_err, rec_depth


# ------------------
# shape_bezier_curve
# ------------------
# Given a Bezier curve (one stroke), compute its image (approximately as
# a new Bezier curve) using the appropriate transformation specified in
# 'shaping_factors'.
#
# Args:
# - bezier_curve:       BCurve.BezierCurve
# - shaping_input: dict
# - 
# Returns:
# - BCurve.BezierCurve
def shape_bezier_curve(
               bezier_curve,
               shaping_input,
               ):
    def arc_measure(ba): # ba:BCurve.BezierArc (assume four control points)
        from math import sqrt
        p0,p1,p2,p3 = ba.cp4 # RowVector = complex
        diff = -p0+p1, -p1+p2, -p2+p3, -p3+p0
        # Vaihdettu kompleksilukuihin; hiukan toinen logiikka:
        #return sqrt(sum([vdot(d,d) for d in diff]))
        return sum([abs(d) for d in diff])
    try: # Set SHORT_ARC_SIZE: Shorter arcs are done by the simple algorithm.
        typical_arc_size = shaping_input['typical arc size']
        SHORT_ARC_SIZE = typical_arc_size / 50
    except KeyError:
        SHORT_ARC_SIZE = 10
    # Main work: Different cases treated differently:
    bezier_case_linefigure = False
    ba_list_ori = bezier_curve.bezier_arcs # [BCurve.BezierArc]
    pt = shaping_input['plane transformation']
    if len(ba_list_ori[0].cp4) == 1:
        # The stroke has just one anchor.
        # Just map that anchor by pt.point_map
        rv = ba_list_ori[0].cp4[0] # RowVector
        mapped_rv = pt.point_map(rv)
        ba = BCurve.BezierArc([mapped_rv])
        ba_list = [ba] # One Bezier arc where cp4=[p0] (one control point)
    else:
        if bezier_curve.closed:
            # Add the closing segment as one Bezier arc,
            # so that it will be transformed properly.
            start = ba_list_ori[0].cp4[0]
            end = ba_list_ori[-1].cp4[-1]
            head = bezier_curve.head_handle
            tail = bezier_curve.tail_handle
            ba_list_ori.append(BCurve.BezierArc([end, tail, head, start]))
        ba_list = []
        for ba_ori in ba_list_ori: # ba_ori:BCurve.BezierArc
            arc_size = arc_measure(ba_ori)
            # Main calls:
            if bezier_case_linefigure: # Handled exactly
                line_segment = [ba_ori.cp4[0], ba_ori.cp4[-1]]
                bas = line_segment2ba_list(pt, line_segment)
            elif arc_size < SHORT_ARC_SIZE:
                bas = ba2balist_simple(          # Simple algorithm
                        pt,
                        ba_ori,
                        shaping_input=shaping_input,
                        log_nsubarcs= 0,
                        )
            elif shaping_input['use_simple'] >= 1: # Simple algorithm.
                bas = ba2balist_simple(
                        pt,
                        ba_ori,
                        shaping_input=shaping_input,
                        log_nsubarcs= -1 + shaping_input['use_simple'],
                        )
            else:                                 # Regular approx algorithm.
                bas,_,_ = ba2balist_approx(
                        pt,
                        ba_ori,
                        shaping_input=shaping_input
                        )
            ba_list += bas
    # Map head_handle and tail_handle:
    if not bezier_curve.closed:
        # Made a global pt and the curve not closed.
        # Map the old head_handle and tail_handle.
        ## (Is this a correct idea? No!                       TO DO
        ## If not closed curve, should add an imagined closing arc, apply
        ## ba2balist_approx to it and then:
        ## -If the approximation is just one arc, take new head and tail from there.
        ## -If there are several arcs, then we have a problem! No satisfactory way?
        ##  Perhaps use ba2balist_simple with log_nsubarcs=0?)
        old_ba_list = bezier_curve.bezier_arcs
        old_start_rv = old_ba_list[0].cp4[0] # Old start: RowVector
        old_end_rv = old_ba_list[-1].cp4[-1] # Old end:   RowVector
        old_head_rv = bezier_curve.head_handle # Old head absolute: RowVector
        old_tail_rv = bezier_curve.tail_handle # Old tail absolute: RowVector
        old_head_rel_rv = -old_start_rv + old_head_rv # Old head relative: RowVector
        old_tail_rel_rv = -old_end_rv + old_tail_rv   # Old tail relative: RowVector
        # Map relative vectors:
        new_head_rel_rv = rv_times_M2x2_transpose(old_head_rel_rv, pt.jacobian(old_start_rv))
        new_tail_rel_rv = rv_times_M2x2_transpose(old_tail_rel_rv, pt.jacobian(old_end_rv))
        # Make absolute:
        new_start_rv = ba_list[0].cp4[0] # New start: RowVector
        new_end_rv = ba_list[-1].cp4[-1] # New end:   RowVector
        new_head_rv = new_start_rv + new_head_rel_rv # New head absolute: RowVector
        new_tail_rv = new_end_rv + new_tail_rel_rv   # New tail absolute: RowVector
        # To ControlPoints:
        new_head = new_head_rv # New head absolute: ControlPoint
        new_tail = new_tail_rv # New tail absolute: ControlPoint
    else:    
        # Made a global pt and the curve is closed.
        # The original head and tail handles were used already when
        # adding the closing line segment as one Bezier arc.
        # Now just tell Gimp that the curve is closed, hoping it
        # will close it properly without making a tiny new edge.
        new_head = None
        new_tail = None
    return BCurve.BezierCurve(
                     bezier_arcs = ba_list,
                     head_handle = new_head,
                     tail_handle = new_tail,
                     closed      = bezier_curve.closed,
                     curve_name = 'mapped '+bezier_curve.curve_name,
                     )

# ----------
# shape_path
# ----------
# Given a path (Gimp's vectors object), compute its image (approximately as
# a new path) in the transformation constructed from info in 'shaping_input'.
#
# Args:
# - image
# - path_vectors:  gimp.Vectors
# - shaping_input: dict
# Returns:
# - gimp.Vectors
def shape_path(
               image,
               path_vectors,
               shaping_input,
               ):
    gv_path = vectors_object2gv(path_vectors) # BCurve.GimpVectors
    # For each stroke (polyline) in the path, construct the shaped stroke:
    list_shaped_gs = []
    for gs in gv_path.stroke_list: # gs: BCurve.GimpStroke
        bc = gs.gs2bc()      # BCurve.BezierCurve
        new_bc = bezier_arc_protect_cusps(bc)
        shaped_bc = shape_bezier_curve(                        # Main call
                   new_bc,
                   shaping_input,
                   )
        shaped_bc.closed = gs.closed # Needed?
        shaped_gs = shaped_bc.bc2gs() # BCurve.GimpStroke
        list_shaped_gs.append(shaped_gs)
    name = path_vectors.name
    if shaping_input['case'] in ('path by bezier', 'path by bezier4'):
        name += ' shaped by '+shaping_input['shaper_name']
    mapped_gv = BCurve.GimpVectors(stroke_list=list_shaped_gs,
                                   name=name
                                   )
    mapped_vectors = mapped_gv.gv2vectors_object(image, mapped_gv.name)
    return mapped_vectors


#---------------------------------------------------------
#                   Gimp draw
#---------------------------------------------------------

# ------------------------
# gimp_draw_vectors_object
# ------------------------
# The Gimp version of the routine 'drawing_routine_for_bezier':
# Insert into Gimp the input vectors object, and make it visible if
# visible=True.
#
# Args:
# - image: Gimp's image (obtained usually from GUI);
# - vectors_object: a vectors object in Gimp;
# - visible: Boolean.
#
# Returns the vectors object in Gimp, and makes it visible if visible=True.
def gimp_draw_vectors_object(image, vectors_object, visible=True):
    # Insert a Gimp vectors object in Gimp.
    # Adapted from http://registry.gimp.org/files/lqr_wpset-0.14.py.txt
    # by Mike Kazantsev.
    def gimp_insert_vectors(image, vectors_object, position=0):
        if gimp.version >= (2, 8, 0):
            # 2.7.0 has gimp-image-insert-vectors, but fails with
            # parent=None
            parent = None
            return pdb.gimp_image_insert_vectors(
                     image, vectors_object, parent, position)
        else:
            return pdb.gimp_image_add_vectors(
                     image, vectors_object, position)# Deprecated!
    gimp_insert_vectors(image, vectors_object)
    vectors_object.visible = visible
    return vectors_object

# polyline = [complex]
def draw_polyline(image, polyline, name='polyline', closed=False):
    polyline_object = pdb.gimp_vectors_new(image, name)
    points = []
    for z in polyline:
        xy = [z.real,z.imag]
        points += 3*xy
    stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                        polyline_object, 0, len(points), points, closed)
    gimp_draw_vectors_object(image, polyline_object, visible=True)
    return polyline

# ----------------
# gimp_draw_circle
# ----------------
# Args:
# - center:[float,float]
# - radius: float
# Returns:
# - gimp.Vectors
def gimp_draw_circle(image, center, radius, name='circle'):
    kappa = 4.*(sqrt(2)-1)/3
    vcenter = [center.real, center.imag]
    xy = [1, -kappa, 1, 0, 1, kappa,
          kappa, 1, 0, 1, -kappa, 1,
          -1, kappa, -1, 0, -1, -kappa,
          -kappa, -1, 0, -1, kappa, -1]
    for i in range(len(xy)):
        if i%2 == 0:
            xy[i] = vcenter[0] + radius*xy[i]
        else:
            xy[i] = vcenter[1] + radius*xy[i]
    circle = pdb.gimp_vectors_new(image, name)
    pdb.gimp_vectors_stroke_new_from_points(
                        circle, 0, len(xy), xy, True)
    gimp_draw_vectors_object(image, circle, visible=True)
    return circle


#==============================================================
#             class PlaneTransformation       
#==============================================================
# Classes: Matrix2x2, PlaneTransformation

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

#---------------------------------------------------------
#       classes: Matrix2x2, PlaneTransformation
#---------------------------------------------------------

class Matrix2x2(object):
    """2x2-matrix
    Initialize m = Matrix2x2(1,2,3,4)
    Get elements as m[i,j]
    Set elements as m[i,j]=value
    Note: Works with real or complex matrices.
    """
    def __init__(self,a00,a01,a10,a11):
        #self._elements = [float(a00),float(a01),float(a10),float(a11)]
        self._elements = [a00,a01,a10,a11]
    def __getitem__(self,indexes): # indexes = [i,j]
        i,j = indexes
        if not (0<=i<=1 and 0<=j<=1):
            raise Exception("Matrix2x2 get: Index out of range")
        return self._elements[2*i+j]
    def __setitem__(self, indexes, value):
        i,j = indexes
        if not (0<=i<=1 and 0<=j<=1):
            raise Exception("Matrix2x2 set: Index out of range")
        self._elements[2*i+j] = value
    def transpose(self):
        a,b,c,d = self._elements
        return Matrix2x2(a,c,b,d)
    def scale(self, coeff):
        a,b,c,d = [x*coeff for x in self._elements]
        return Matrix2x2(a,b,c,d)
    def inverse(self):
        a,b,c,d = self._elements
        try:
            idet = 1./(a*d-b*c)
        except ZeroDivisionError:
            raise Exception("Matrix2x2 inverse: Zero determinant")
        A,B,C,D = [x*idet for x in self._elements]
        return Matrix2x2(D,-B,-C,A)
    def __add__(self,other):
        a,b,c,d = self._elements
        A,B,C,D = other._elements
        return Matrix2x2(a+A,b+B,c+C,d+D)
    def __mul__(self,other):
        return Matrix2x2(self[0,0]*other[0,0] + self[0,1]*other[1,0],
                         self[0,0]*other[0,1] + self[0,1]*other[1,1],
                         self[1,0]*other[0,0] + self[1,1]*other[1,0],
                         self[1,0]*other[0,1] + self[1,1]*other[1,1])
    def __str__(self):
        def str_float(x): return '{:1.3f}'.format(x)
        a00 = str_float(self[0,0])
        a01 = str_float(self[0,1])
        a10 = str_float(self[1,0])
        a11 = str_float(self[1,1])
        return '['+a00+','+a01+']' +'\n['+a10+','+a11+']\n'


class PlaneTransformation(object):
    """Plane map  RowVector -> RowVector
    with jacobian RowVector -> Matrix2x2
    Attributes:
    - type:      string
    - point_map: callable, maps RowVector -> RowVector
    - jacobian:  callable, maps RowVector -> Matrix2x2
    """
    def __init__(self, type, point_map, jacobian=None):
        self.type = type
        self.point_map = point_map
        self.jacobian = jacobian


# -------------
# rv_times_M2x2
# -------------
# RowVector times Matrix2x2.
# Args:
# - rv:     RowVector (=complex)
# - matrix: Matrix2x2
# Returns:
# - RowVector (=complex)
def rv_times_M2x2(rv,matrix):
    return RowVector(rv.real*matrix[0,0] + rv.imag*matrix[1,0],
                     rv.real*matrix[0,1] + rv.imag*matrix[1,1])

# -----------------------
# rv_times_M2x2_transpose
# -----------------------
# RowVector times Matrix2x2 transposed.
# Args:
# - rv:     RowVector (=complex)
# - matrix: Matrix2x2
# Returns:
# - RowVector (=complex)
def rv_times_M2x2_transpose(rv,matrix):
    return RowVector(rv.real*matrix[0,0] + rv.imag*matrix[0,1],
                     rv.real*matrix[1,0] + rv.imag*matrix[1,1])


#==============================================================
#===================  affine plugin proper  ===================
#==============================================================
# Transform a path by a general affine map
# (special cases: direct similitude, translation).

# Note: The plane is treated as the complex plane: #
#       BCurve.ControlPoint = RowVector = complex  #

#---------------------------------------------------------
#                 Make affine transformation
#---------------------------------------------------------

# Affine map sending three given points u0,u1,u2 to another
# three given points v0,v1,v2. 
# Args:
# - base: [RowVector,RowVector,RowVector] (at least 3 points)
# - target: [RowVector,RowVector,RowVector] (at least 3 points)
# Returns:
# - PlaneTransformation
def make_affine_map(base, target):
    u0,u1,u2 = base[:3] # The first three anchors: RowVector
    v0,v1,v2 = target[:3] # The first three anchors: RowVector
    if collinear([u0,u1,u2]):
        raise Exception("make_affine_map: Anchors of Base are collinear.")
    u10 = u1-u0 # RowVector (=complex)
    u20 = u2-u0 # RowVector (=complex)
    v10 = v1-v0 # RowVector (=complex)
    v20 = v2-v0 # RowVector (=complex)
    A = Matrix2x2(u10.real, u10.imag, 
                  u20.real, u20.imag)
    B = Matrix2x2(v10.real, v10.imag, 
                  v20.real, v20.imag)
    try:
        AB = A.inverse() * B
    except Exception as e:
        raise Exception("make_affine_map: "+str(e))
    def pm(x):
        return v0 + rv_times_M2x2((x-u0),AB)
    def jac(x):
        return AB.transpose()
    return PlaneTransformation('affine', pm, jac)


# ----------------
# transform_stroke
# ----------------
#  Args:
# - pt: PlaneTransformation
# - gs: BCurve.GimpStroke
# Returns:
# - BCurve.GimpStroke
def transform_stroke(point_map, gs):
    def transform_cp_list(cp_list):
        return [point_map(cp) for cp in cp_list]
    return BCurve.GimpStroke(cp_list = transform_cp_list(gs.cp_list),
                             closed = gs.closed,
                             stroke_name = gs.stroke_name + ' transformed'
                             )

# ------------------
# map_path_by_affine
# ------------------
# Do the main work of affine_transformation_main except for drawing.
# Args:
# - vectors_object: gimp.Vectors;
# - rv_base:        [RowVector] (=[complex])
# - rv_target:      [RowVector] (=[complex])
# - reversed_base:  boolean
# Returns:
# - gimp.Vectors
def map_path_by_affine(image,
                       vectors_object, # Path to be transformed
                       rv_base,
                       rv_target,
                       reversed_base=False
                       ):
    gv = vectors_object2gv(vectors_object)          # GimpVectors
    if len(rv_target) < len(rv_base):
        if len(rv_target) == 0:
            raise Exception("Empty target")
        else:
            raise Exception("The target must have "+str(len(rv_base))+" anchors since base has.")
    if reversed_base:
        rv_base = rv_base[:3][::-1] # The first three (or less) elements reversed
    #
    if len(rv_base) >= 3:
        pt = make_affine_map(rv_base, rv_target)
    elif len(rv_base) == 2:
        rv1, rv2 = rv_base
        rw1, rw2 = rv_target[0], rv_target[1]
        pt = make_direct_similitude(rv1, rv2, rw1, rw2)
    elif len(rv_base) == 1:
        rv = rv_base[0]
        rw = rv_target[0]
        pt = make_translation(rw-rv)
    #
    new_stroke_list = [transform_stroke(pt.point_map, gs)
                                              for gs in gv.stroke_list]
    new_gv = BCurve.GimpVectors(stroke_list = new_stroke_list,
                                name = vectors_object.name + ' transformed'
                                )
    pdb.gimp_image_undo_group_start(image)
    vectors = new_gv.gv2vectors_object(image, name=new_gv.name)
    pdb.gimp_image_undo_group_end(image)
    return vectors


#---------------------------------------------------------
#          Path warping: make transformations
#---------------------------------------------------------

# -----------------
# make_raw_grow_map
# -----------------
# Formula: z -> [ 1 + force * (1-|z|)^smooth * (h0(|z|) - 1) ] * z (when |z| <= 1)
# where h0(r) = sin((pi/2)*r)/r
# Args:
# - smooth: float (>= 0)
# - force: float (0..1)
# Returns:
# - PlaneTransformation
def make_raw_grow_map(smooth, force):
    ZERO = 1e-3
    PROTECT_1 = 1e-10
    if smooth < ZERO:
        smooth == 0
    def pm(z): # z:complex
        if abs(z) > 1+PROTECT_1:
            return z
        else:
            r = abs(z)
            try:
                coeff1 = force * ((1-r)**smooth)
            except ValueError:
                # Happens if r>1 and smooth not integer.
                # This should never be called with r>1 but nevertheless r>1
                # may occur because of computation precision when |z|=1.
                coeff1 = 0
            try:
                return z * (1 + coeff1 * (sin(r*pi/2) / r - 1))
            except ZeroDivisionError: # z=0
                return z
    def jac(z):
        if abs(z) > 1+PROTECT_1:
            return Matrix2x2(1,0,0,1)
        x,y = z.real,z.imag
        r = abs(z)
        try:
            ss,cc = sin(r*pi/2), cos(r*pi/2)
            try:
                coeff1 = force * ((1-r)**smooth)
                coeff2 = force * smooth * ((1-r)**(smooth-1))
            except (ValueError, ZeroDivisionError):
                # Happens if r>1 and smooth not integer
                # or r=1 and smooth-1<0.
                # This should never be called with r>1 but nevertheless r>1
                # may occur because of computation precision when |z|=1.
                coeff1 = 0
                coeff2 = 0
            h = 1 + coeff1 * (ss/r - 1)                 # h(r)
            hdr = (-coeff2 * r * (ss-r)
                    + coeff1 * (r*cc*pi/2 - ss)) / (r**3) # h'(r)/r
            return Matrix2x2(h + hdr*x*x,
                             hdr*x*y,
                             hdr*x*y,
                             h + hdr*y*y)
        except ZeroDivisionError: # z=0
            return Matrix2x2(pi/2, 0, 0, pi/2)
    return PlaneTransformation('grow',
                                point_map = pm,
                                jacobian = jac)

# -------------------
# make_raw_shrink_map
# -------------------
# Formula: z -> [ 1 + force * (1-|z|)^smooth * (h0(|z|) - 1) ] * z (when |z| <= 1)
# where h0(r) = (1-sqrt(1-r))/r.
# Args:
# - smooth: float (>= 0)
# - force: float (0..1)
# Returns:
# - PlaneTransformation
def make_raw_shrink_map(smooth, force):
    ZERO = 1e-3
    #PROTECT_1 = 1e-10
    #PROTECT_0 = 1e-6
    if smooth < ZERO:
        smooth == 0
    #smooth = (smooth/5)**2
    def pm(z): # z:complex
        r = abs(z)
        try:
            coeff1 = force * ((1-r)**smooth)
        except ValueError:
            # Happens if r>1 and smooth not integer.
            # This should never be called with r>1 but nevertheless r>1
            # may occur because of computation precision when |z|=1.
            coeff1 = 0
        try:
            h0 = (1 - sqrt(1-r))/r
        except ValueError: # r >= 1
            #print("pm: ValueError, r = "+str(r))
            h0 = 1
        except ZeroDivisionError: # r = 0
            #print("pm: ZeroDivisionError, r = "+str(r))
            h0 = 1/2
        return (1 + coeff1 * (h0 - 1)) * z
    def jac(z):
        r = abs(z)
        try:
            coeff1 = force * ((1-r)**smooth)
            coeff2 = force * smooth * ((1-r)**(smooth-1))
        except (ValueError, ZeroDivisionError):
            # Happens if r>1 and smooth not integer
            # or r=1 and smooth-1<0.
            # This should never be called with r>1 but nevertheless r>1
            # may occur because of computation precision when |z|=1.
            coeff1 = 0
            coeff2 = 0
        try:
            sr = sqrt(1-r)
            h0 = (1 - sr)/r                       # h0(r)
            h0dr = (1 - r/2 - sr) / (r*r*r*sr)    # h0'(r)/r
            h  = 1 + coeff1 * (h0-1)                              # h(r)
            hdr  = -coeff2 * (h0-1) / r + coeff1 * h0dr  # h'(r)/r
        except ValueError: # r >= 1
            #print("jac: ValueError, r = "+str(r))
            r = .9999999
            sr = sqrt(1-r)
            h0 = 1                          # h0(r)
            h0dr = .5 / sr                  # h0'(r)/r
            h  = 1 + coeff1 * (h0-1)                     # h(r)
            hdr  = -coeff2 * (h0-1) / r + coeff1 * h0dr  # h'(r)/r
        except ZeroDivisionError: # r = 0
            #print("jac: ZeroDivisionError, r = "+str(r))
            h0 = 1/2
            hdr = 0 # ?
        x,y = z.real, z.imag
        return Matrix2x2(h0 + hdr*x*x,
                              hdr*x*y,
                              hdr*x*y,
                         h0 + hdr*y*y)
    return PlaneTransformation('shrink',
                                point_map = pm,
                                jacobian = jac)

# ------------------
# make_raw_swirl_map
# ------------------
# Formula: z -> [ 1 + force * (1-|z|)^smooth * (h0(|z|) - 1) ] * z (when |z| <= 1)
# where h0(r) = sin((pi/2)*r)/r
# Args:
# - smooth: float (>= 0)
# - force:  float (0..1)
# - ccw:    boolean
# Returns:
# - PlaneTransformation
def make_raw_swirl_map(smooth, force, ccw=True):
    from cmath import exp as cexp
    ZERO = 1e-3
    PROTECT_1 = 1e-10
    if smooth < ZERO:
        smooth == 0
    
    #smooth = (smooth/5)**2 # Sama!
    
    if ccw:
        alpha = -force*pi/2
    else:
        alpha = force*pi/2
    def pm(z): # z:complex
        if abs(z) > 1+PROTECT_1:
            return z
        else:
            r = abs(z)
            try:
                xi = alpha * ((1-r)**(1+smooth))
                zrot = cexp(1j * xi)
                return z * zrot
            except ValueError:
                # Happens if r>1 and smooth not integer.
                # This should never be called with r>1 but nevertheless r>1
                # may occur because of computation precision when |z|=1.
                return z
    def jac(z):
        if abs(z) > 1+PROTECT_1:
            return Matrix2x2(1,0,0,1)
        x,y = z.real,z.imag
        r = abs(z)
        try:
            xi = alpha * ((1-r)**(1+smooth))
        except ValueError:
            xi = 0
        cxi, sxi = cos(xi), sin(xi)
        try:
            ar = alpha * ((1-r)**smooth) / r
            return Matrix2x2( cxi + ar * (sxi*x*x + cxi*x*y),
                             -sxi + ar * (sxi*x*y + cxi*y*y),
                              sxi + ar * (sxi*x*y - cxi*x*x),
                              cxi + ar * (sxi*y*y - cxi*x*y),
                            )
        except ZeroDivisionError: # z=0
            salpha, calpha = sin(alpha), cos(alpha)
            return Matrix2x2(calpha, -salpha, salpha, calpha)
        except ValueError:
            # Happens if r>1 and smooth not integer.
            # This should never be called with r>1 but nevertheless r>1
            # may occur because of computation precision when |z|=1.
            return Matrix2x2(1, 0, 0, 1)
    return PlaneTransformation('grow',
                                point_map = pm,
                                jacobian = jac)


#---------------------------------------------------------
#              Path warping: the central procedure
#---------------------------------------------------------

# -----------------
# warp_path_locally
# -----------------
# Do the main work of the experimental warp_path_locally except for the drawing.
# Args:
# - image
# - path_vectors: gimp.Vectors
# - center:       complex
# - radius:       float 
# - smooth:       float (typically 0..10)
# - force:        float (0..1)
# - delete_outside: boolean
# - use_simple:   integer
# Returns:
# - gimp.Vectors
def warp_path_locally(image,
                    path_vectors,
                    warp_case,
                    center,
                    radius,
                    smooth,
                    force,
                    delete_outside,
                    use_simple,
                    ):
    path_gv = vectors_object2gv(path_vectors)
    inside_bcs, outside_bcs = gv_split_by_circle(path_gv, center, radius)
    
    inside_gs_list = [bc.bc2gs() for bc in inside_bcs]
    outside_gs_list = [bc.bc2gs() for bc in outside_bcs]
    
    outside_gv = BCurve.GimpVectors(outside_gs_list)
    inside_gv = BCurve.GimpVectors(inside_gs_list)
    
    pps_half = gv_intersect_circle(inside_gv, center, radius/2) # [PathPointData]
    inside_gv_plus = gv_add_anchors(inside_gv, pps_half) # New, changed copy
    
    # Store the individual stroke names before converting to gimp.Vectors.
    # Hoping that the order is preserved in the transformations!
    inside_stroke_names = [gs.stroke_name for gs in inside_gv_plus.stroke_list]
    
    # Convert to gimp.Vectors:
    inside_path_plus = inside_gv_plus.gv2vectors_object(image, 'inside_path_plus')
    inside_path_plus.name = path_vectors.name + ' plus'
    
    # Transform to "standard positon"
    base_points = [center, center+radius]
    h1_base_points = [0j, 1+0j]
    h1_path = map_path_by_affine(image,    #  h1(path)
                                 inside_path_plus,
                                 base_points,
                                 h1_base_points,
                                 reversed_base=False
                                 )
    h1_path.name = 'h1'
    
    # Map h1_path with the raw warping map:
    if warp_case == 'grow':
        raw_pt = make_raw_grow_map(smooth, force)
    elif warp_case == 'shrink':
        raw_pt = make_raw_shrink_map(smooth, force)
    elif warp_case == 'swirl ccw':
        raw_pt = make_raw_swirl_map(smooth, force, True)
    elif warp_case == 'swirl cw':
        raw_pt = make_raw_swirl_map(smooth, force, False)
    else:
        raise Exception("Unknown case: "+warp_case)
    shaping_input = dict()
    shaping_input['case']       = 'warp koe'
    shaping_input['use_simple'] = use_simple
    shaping_input['typical arc size'] = 1 # Moving in complex plane around origo
    shaping_input['plane transformation'] = raw_pt
    F_h1_path = shape_path(image,              # F(h1(path))
                           h1_path,
                           shaping_input
                           )
    F_h1_path.name = 'F_h1'
    
    # Map back from the "standard position"
    pm = raw_pt.point_map
    F1_h1_base_points = [pm(z) for z in h1_base_points]
    shaped_inside_path = map_path_by_affine(         # h2(F(h1(path)))
                       image,
                       F_h1_path,
                       F1_h1_base_points,
                       base_points,     # target_points = base_points!
                       reversed_base=False   
                       )
    shaped_inside_gv = vectors_object2gv(shaped_inside_path)
    
    # Restore individual stroke names_
    for i in range(len(shaped_inside_gv.stroke_list)):
        shaped_inside_gv.stroke_list[i].stroke_name = inside_stroke_names[i]
    
    if not delete_outside:
        new_gv = gv_stitch(shaped_inside_gv, outside_gv)
    else:
        new_gv = shaped_inside_gv
    return new_gv.gv2vectors_object(image)


#---------------------------------------------------------
#           Path warping: main procedures
#---------------------------------------------------------

warp_options = [ # (description, identifier)
        ('Grow',  'grow'),
        ('Shrink',  'shrink'),
        ('Swirl ccw',  'swirl ccw'),
        ('Swirl cw',  'swirl cw'),
        ]

circle_options = [ # (description, identifier)
        ('a diameter of the circle of action',                 'diameter'),
        ('the center and one point on the circle of action',   'center_point'),
        ('the same as above but used reversed',    'center_point_reversed'),
        ]

# ----------------------
# warp_path_locally_main
# ----------------------
# Grow a path inside a circular region.
# Args:
# - image
# - path_vectors:   gimp.Vectors
# - warp_option:    int (see warp_options)
# - circle_path:    gimp.Vectors (a 2-anchors path)
# - circle_option:  int (see circle_options)
# - smoothing:      float (>=0, typically 1..10)
# - force:          float (0..1)
# - draw_circle:    boolean (the warping circle)
# - delete_outside: boolean
# - use_simple:    int (0..8)
# Returns:
# - gimp.Vectors
def warp_path_locally_main(image,
                           path_vectors,
                           warp_option,
                           circle_path,
                           circle_option,
                           smoothing,
                           strength,
                           draw_circle,
                           delete_outside,
                           use_simple=4
                           ):
    smoothing = max(0., smoothing)
    smooth = (smoothing/5)**2 # Some kind of scaling and deforming (experimental)
    force = max(0., min(1., strength))
    #
    warp_case = warp_options[warp_option][1]
    circle_case = circle_options[circle_option][1]
    #print("warp_case = "+warp_case)
    #
    circle_anchors = get_anchors(circle_path, only_first_stroke=True)
    if len(circle_anchors) != 2:
        raise Exception("Please give a 2-anchors path for the circle")
    a,b = circle_anchors
    if circle_case == 'diameter':
        center = (a+b)/2
        radius = abs(a-b)/2
    elif circle_case == 'center_point':
        center = a
        radius = abs(a-b)
    elif circle_case == 'center_point_reversed':
        center = b
        radius = abs(a-b)
    else:
        raise Exception("crop_path_by_circle_main: Unknown case: "+circle_case)
    #print("center = "+str(center))
    #print("radius = "+str(radius))
    mapped_vectors = warp_path_locally(image, # Main call
                                       path_vectors = path_vectors,
                                       warp_case = warp_case,
                                       center = center,
                                       radius = radius,
                                       smooth = smooth,
                                       force = force,
                                       delete_outside = delete_outside,
                                       use_simple = use_simple,
                                       )
    mapped_vectors.name = path_vectors.name+'|'+warp_case
    if smoothing > 0. or force < 1.:
        mapped_vectors.name += '('+str(smoothing) +', '+str(force)+')'
    pdb.gimp_image_undo_group_start(image) # Draw
    if draw_circle:
        gimp_draw_circle(image, center, radius, name='Warping circle')
    mapped = gimp_draw_vectors_object(image, mapped_vectors, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return mapped_vectors


#======================================================
#                    Registration
#======================================================

versionnumber = "0.4"
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2021"
image_types = "*"
menupath = '<Vectors>/Tools/Warping'


#####################  Warp path locally  #####################

procedure_name  = "warp_path"
procedure_blurb = ("Warp a path locally (inside a circle)."
                   +"\nThe circle is input as a line segment giving either"
                   +"\n(1)   a diameter of the circle, or"
                   +"\n(2)   the center and one point on the circle."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "Warp a path locally."
procedure_label = "Warp path locally"

procedure_function = warp_path_locally_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path_vectors", "The path to be warped locally", None),
      (PF_OPTION, 'warp_option','Action',
                   0,
                  [case[0] for case in warp_options]), # Descriptions of cases
      (PF_VECTORS, "circle_path", "A 2-anchors path to determine the circle of action", None),
      (PF_OPTION, 'circle_option','The 2-anchors path defines:',
                   0,
                  [case[0] for case in circle_options]), # Descriptions of cases
      (PF_FLOAT, "smoothing", "Smoothing (float 0..10..)", 0.),
      (PF_FLOAT, "strength", "Strength of action (float 0..1)", 1.),
      (PF_BOOL, "draw_circle", "Draw the circle of action?", False),
      (PF_BOOL, "delete_outside", "Delete the part outside of the circle of action?", False),
      (PF_INT, "use_simple", ("Choose algorithm (integer 0..8)."
                               +"\n*   0 => Best but slow."
                               +"\n*   1 => Simple algorithm: quick but inaccurate."
                               +"\n*   2..8 => Simple algorithm with ever more control points."
                               ),
                        4),
    ],
    [
        (PF_VECTORS, "warped", "warped path"),
    ],
    procedure_function,
    menu=menupath)


main()

