#!/usr/bin/env python
#
# Gimp plugin to transform the active layer by perspective transformation.
# Interfaces to procedure pdb.gimp_item_transform_matrix.
#
# History:
# v0.1: 2020-10-28: First published version.
#                   Plugin "Perspective transform by two paths"
# v0.2: 2020-11-01: Added plugin "Perspective transform by a path and a rectangle"
# v0.3: 2020-11-05: Dropped the option to do operation backwards.
#                   Added plugin "Perspective transform - 4 points to circle"
# v0.4: 2020-11-21: Unpublished.
#                   Experimental plugin "Perspective transform - 3+3 points to circle (diameter and center)"
# v0.5: 2020-11-25: Unpublished.
#                   Experimental plugin "Perspective transform - 5+1 points to circle (5 points and center)"
# v0.6: 2020-11-30: Added plugin "Perspective transform - 1+2+3 points to circle (center, direction, 3 points)"
#                   New wordings in older plugins.
# v0.7: 2020-12-04: Added plugin "Perspective transform - 2+2+2 points to circle (3 diameters)"

# (c) Markku Koppinen 2020
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'

from __future__ import division, print_function 
from gimpfu import *                            
from math import *                              

#---------------------------------------------------------
#                Auxiliary routines
#---------------------------------------------------------

# v dot (w perpendicular)
def cross(v,w):
    return (v*w.conjugate()).imag

def vdot(v,w):
    return v.real*w.real + v.imag*w.imag

def collinear(points): # points: [complex] (The plane R2 is the complex plane.)
    ZERO = 1e-14
    n = len(points)
    if n < 3:
        return True
    for i in range(n):
        for j in range(i+1,n):
            for k in range(j+1,n):
                a,b,c = points[i], points[j], points[k]
                ab = -a+b
                ac = -a+c
                bc = -b+c
                if abs(cross(ab,ac)) > ZERO:
                    return False
                if abs(cross(ab,bc)) > ZERO:
                    return False
                if abs(cross(ac,bc)) > ZERO:
                    return False
    return True

def some_three_collinear(points): # points: [complex]
    n = len(points)
    for i in range(n):
        for j in range(i+1,n):
            for k in range(j+1,n):
                if collinear([points[i],points[j],points[k]]):
                    return True
    return False

# -----------------
# line_intersection
# -----------------
# Given two point pairs a=[a0,a1] and b=[b0,b1], find the intersection of
# the line through a0,a1 with the line through b0,b1.
# Find also if the intersection is on the intervals [a0,a1] and [b0,b1].
# Return None if the lines are parallel or if a0=a1 or b0=b1.
# Plane is viewed as the complex number plane, so points are complex numbers.
# Args:
# - a:[complex,complex]
# - b:[complex,complex]
# Returns: Either None, or:
# - complex
# - boolean (True if the intersection is on the intervals)
def line_intersection(a,b):
    a0,a1 = a
    b0,b1 = b
    try:
       invdenom = 1. / cross(-a0+a1, -b0+b1)
    except ZeroDivisionError: # parallel or a0=a1 or b0=b1
        return None
    t = cross(-a0+b0, -b0+b1) * invdenom
    u = cross(-a0+b0, -a0+a1) * invdenom
    p = a0 + t*(-a0+a1)
    #q = b0 + u*(-b0+b1) # q = p
    return p, (0 <= t <= 1) and (0 <= u <= 1)

# -----------
# get_anchors
# -----------
# Get anchors of the path (optionally the first stroke only).
# Arg:
# - vectors_object:    gimp.Vectors
# - only_first_stroke: boolean
# Returns:
# - [complex]
def get_anchors(vectors_object, only_first_stroke=False):
    strokes = vectors_object.strokes
    if only_first_stroke:
        strokes = [strokes[0]]
    zs = [] # [complex]
    for stroke in strokes:
        xyxy = stroke.points[0] # [x,y,x,y,...]
        for i in range(0,len(xyxy),6):
            zs.append(complex(xyxy[i+2],xyxy[i+3]))
    return zs

# ---------------------------
# get_anchors_stroke_by_stroke
# ----------------------------
# Get anchors of the path stroke by stroke, return as a list lof lists.
# The plane is viewed as the complex number plane, so anchors are complex numbers.
# Arg:
# - vectors_object:    gimp.Vectors
# Returns:
# - [[complex]]
def get_anchors_stroke_by_stroke(vectors_object):
    strokes = vectors_object.strokes
    anchor_lists = [] # [[complex]], each item is the list of anchors of one stroke
    for stroke in strokes:
        zs = [] # [complex]
        xyxy = stroke.points[0] # [x,y,x,y,...]
        for i in range(0,len(xyxy),6):
            zs.append(complex(xyxy[i+2],xyxy[i+3]))
        anchor_lists.append(zs)
    return anchor_lists

# --------------
# box_meets_line
# --------------
# Given a box (height, width, offsetx, offsety) and 
# and a line ax+by+c=0 (line_coefficients=[a,b,c]),
# check if the line meets the box within safety.
# Args:
# - height, width, offsetx, offsety: float
# - line_coefficients: [float] ([a,b,c])
def box_meets_line(height, width, offsetx, offsety, line_coefficients, safety=0):
    # Corners:
    nw = [0     + offsetx, 0      + offsety]
    sw = [0     + offsetx, height + offsety]
    se = [width + offsetx, height + offsety]
    ne = [width + offsetx, 0      + offsety]
    # With safety (enlarge the box):
    nw = [nw[0] - safety, nw[1] - safety]
    sw = [sw[0] - safety, sw[1] + safety]
    se = [se[0] + safety, se[1] + safety]
    ne = [ne[0] + safety, ne[1] - safety]
    # Values of the line function at corners:
    a,b,c = line_coefficients
    lnw = a*nw[0] + b*nw[1] + c
    lsw = a*sw[0] + b*sw[1] + c
    lse = a*se[0] + b*se[1] + c
    lne = a*ne[0] + b*ne[1] + c
    # Line meets diagonals?
    return (lnw * lse <= 0) or (lne * lsw <= 0)

def gimp_draw_vectors_object(image, vectors_object, visible=True):
    # Insert a Gimp vectors object in Gimp.
    # Adapted from http://registry.gimp.org/files/lqr_wpset-0.14.py.txt
    # by Mike Kazantsev.
    def gimp_insert_vectors(image, vectors_object, position=0):
        if gimp.version >= (2, 8, 0):
            # 2.7.0 has gimp-image-insert-vectors, but fails with
            # parent=None
            parent = None
            return pdb.gimp_image_insert_vectors(
                     image, vectors_object, parent, position)
        else:
            return pdb.gimp_image_add_vectors(
                     image, vectors_object, position)# Deprecated!
    gimp_insert_vectors(image, vectors_object)
    vectors_object.visible = visible
    return vectors_object

# polyline = [complex]
def draw_polyline(image, polyline, name='polyline', closed=False):
    polyline_object = pdb.gimp_vectors_new(image, name)
    points = []
    for z in polyline:
        xy = [z.real,z.imag]
        points += 3*xy
    stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                        polyline_object, 0, len(points), points, closed)
    gimp_draw_vectors_object(image, polyline_object, visible=True)
    return polyline

def line_intersect_polyline(line_coefficients, polyline):
    vertices = polyline
    intersections = []
    for i in range(-1+len(vertices)):
        point,loc = line_intersect_segment(line_coefficients,
                                           vertices[i],
                                           vertices[i+1])
        if point is None: # Segment parallel to L or vertices coincide
            if loc[0]: # Both vertices on L
                intersections.append(vertices[i])
                if i == -2+len(vertices):
                    intersections.append(vertices[i+1])
        elif loc[0] or loc[2]: # L intersects either at start or in the open interval
            intersections.append(point)
        elif loc[1] and (i == -2+len(vertices)): # Last vertex is on L
            intersections.append(point)
    return intersections

def line_intersect_segment(line_coefficients, p, q):
    g,h,k = line_coefficients
    p1,p2 = p.real, p.imag
    q1,q2 = q.real, q.imag
    try:
        c = 1./((g*p1 + h*p2) - (g*q1 + h*q2))
    except ZeroDivisionError: # lines parallel or p=q
        if g*p1+h*p2+k == 0: # p and q on L
            return None, [True,True,False]
        else:
            return None, [False,False,False]
    pcoeff = -(g*q1 + h*q2 + k) * c
    qcoeff =  (g*p1 + h*p2 + k) * c
    p_exact = (qcoeff == 0)
    q_exact = (pcoeff == 0)
    in_open = (0 < pcoeff < 1)
    X = pcoeff*p + qcoeff*q
    #check = g*X[0] + h*X[1] + k
    return X, [p_exact, q_exact, in_open]

def gimp_message(txt, h=2):
    # 0 = message box; 1 = console; 2 = error console
    pdb.gimp_message_set_handler(h)
    pdb.gimp_message(txt)

# -----------
# arrange_cps
# -----------
# Given list of control points ([complex]),
# arrange it cyclically and possibly reversing so that
# the resulting list  a  has as the segment  a[0],a[1]  the lowest
# on the screen (in some sense) and left to right.
# Args:
# - cps:  [complex]
# - Gimp: boolean
# Returns:
# - [complex]
def arrange_cps(cps, Gimp=True):
    if Gimp: c = -1
    else: c = 1
    n = len(cps)
    if n <= 1:
        return cps
    # Find lowest anchor:
    lowi = min(range(n), key=(lambda j: c*cps[j].imag))
    # Check the neighbouring anchors:
    upwards = (c*cps[(lowi-1)%n].imag > c*cps[(lowi+1)%n].imag)
    # Arrange item with index lowi to place 0
    # and the lower neighbour to place 1:
    if upwards:
        a = [cps[(i+lowi)%n] for i in range(n)]  # cyclic
    else:
        a = [cps[(-i+lowi)%n] for i in range(n)] # cyclic + reverse
    # Check if left to right:
    if a[0].real <= a[1].real:
        return a # already ok
    else:
        return [a[(1-i)%n] for i in range(n)] # reverse keeping the low segment

# -------
# get_box
# -------
# From path return bounding box or something similar in the form
# [SW, SE, NE, NW] where items:complex and the directions are as on the screen
# (so, top is N, bottom is S).
# If Gimp=True, y coordinate runs downwards.
# Args:
# - path_vectors: gimp.Vectors
# - base_vectors: gimp.Vectors
# - box_case:     string (one of the identifiers in target_box_options)
# - rotate:       float
# - Gimp:         boolean
# Returns:
# - [complex,complex,complex,complex]
def get_box(image, path_vectors, box_case, rotate, Gimp=True):
    from cmath import exp as cexp
    from math import pi
    if box_case == 'bb_path': # bb of Path
        nw, se = path_bb(path_vectors, Gimp=Gimp)
        sw = complex(nw.real, se.imag)
        ne = complex(se.real, nw.imag)
        result =  [sw,se,ne,nw]
    elif box_case == 'bb_anchors': # bb of anchors of Path
        anchors = get_anchors(path_vectors, only_first_stroke=False)
        W = min([a.real for a in anchors])
        E = max([a.real for a in anchors])
        if Gimp:
            S = max([a.imag for a in anchors])
            N = min([a.imag for a in anchors])
        else:
            S = min([a.imag for a in anchors])
            N = max([a.imag for a in anchors])
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        result = [sw,se,ne,nw]
    elif box_case == 'bb_selection':
        bounds = pdb.gimp_selection_bounds(image)
        if bounds[0] == 0:
            raise Exception("No active selection")
        W,N,E,S = bounds[1:]
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        result = [sw,se,ne,nw]
    elif box_case == 'guides':
        hor,ver = get_guide_positions(image)
        if (len(hor) > 2) or (len(ver) > 2):
            raise Exception("Too many guides: must have two horizontal and two vertical.")
        if (len(hor) < 2) or (len(ver) < 2):
            raise Exception("Not enough guides: must have two horizontal and two vertical.")
        W,E = sorted(ver)
        if Gimp:
            N,S = sorted(hor)
        else:
            S,N = sorted(hor)
        sw = complex(W,S)
        se = complex(E,S)
        ne = complex(E,N)
        nw = complex(W,N)
        result = [sw,se,ne,nw]
    elif box_case == 'gen source': # rectangle generated from path
        source = get_anchors(path_vectors, only_first_stroke=True) # [complex]
        source_corners = arrange_cps(source, Gimp=True) # a[0],a[1] bottom edge, left to right.
        result = generate_box_from_points(source_corners)
        #draw_polyline(image, rect, name='rectangle', closed=False) # Testing only
    elif box_case == 'image': # canvas
        w,h = image.width, image.height
        sw = complex(0,h)
        se = complex(w,h)
        ne = complex(w,0)
        nw = complex(0,0)
        result = [sw,se,ne,nw]
    else:
        raise Exception("get_box: invalid choice: "+str(box_case))
    # Rotation
    center = sum(result) / 4
    rot_angle = -rotate * pi / 180 # Minus sign: positive angle means cw
    if Gimp:
        rot_angle = -rot_angle
    zrot = cexp(1j * rot_angle)
    result = [center + (x-center) * zrot for x in result]
    #draw_polyline(image, result, name='result', closed=True)
    return result

# ----------------
# get_clock_points
# ----------------
# Given source=[s0,s1,s2,s3] (si:complex) and hours=[h0,h1,h2,h3] (hi:float),
# find points (complex) on a clock face corresponding to those hours.
# The clock face is imagined to be drawn approximately where the source points are.
# The resulting arc may be rotated ariund the center of the circle.
# Args:
# - source: [complex,complex,complex,complex]
# - hours:  [float,float,float,float]
# - rotate: float
# - Gimp:   boolean
# Returns:
# -  [complex,complex,complex,complex]
def get_clock_points(source, hours, rotate, Gimp=True):
    def diameter(pts):
        p0,p1,p2,p3 = pts
        return max(abs(z) for z in [p0-p1,p0-p2,p0-p3,p1-p2,p1-p3,p2-p3])
    from cmath import exp as cexp
    from math import pi
    # Clock face:
    if Gimp:
        dial_points = [-1j * cexp(1j * h * pi / 6) for h in hours]
    else: # Untested
        dial_points = [1j * cexp(-1j * h * pi / 6) for h in hours]
    radius = diameter(source) / diameter(dial_points)
    try: # Find center of would-be clock face:
        matrix = make_projective_map_matrix(dial_points, source)[0] # Inverse map
        ch = matrix33_times_vector3(matrix, [0,0,1]) # center: homogeneous coordinates
        center = complex(ch[0]/ch[2], ch[1]/ch[2])   # center: complex
    except ZeroDivisionError:
        cg = sum(source) / 4 # center of gravity
        cg_clock = radius * sum(dial_points) / 4
        center = cg - cg_clock
    rot_angle = -rotate * pi / 180 # Minus sign: positive angle means cw
    if Gimp:
        rot_angle = -rot_angle
    zrot = cexp(1j * rot_angle)
    clock_points = [center + zrot*radius*dp for dp in dial_points]
    return clock_points

# -------------------
# get_guide_positions
# -------------------
# Guide positions in image, horizontal and vertical in separate lists.
# Not sorted.
def get_guide_positions(image):
    horizontal = []
    vertical = []
    guide = 0
    while True:
        guide = pdb.gimp_image_find_next_guide(image,guide)
        if guide == 0:
            break
        position = pdb.gimp_image_get_guide_position(image, guide)
        orientation = pdb.gimp_image_get_guide_orientation(image, guide)
        if orientation == ORIENTATION_HORIZONTAL:
            horizontal.append(position)
        elif orientation == ORIENTATION_VERTICAL:
            vertical.append(position)
        else:
            raise Exception("get_guide_positions: ???")
    return horizontal, vertical

# ------------------------
# generate_box_from_points
# ------------------------
# Let points [A0,A1,A2,A3] be corners of a convex quadrangle, arranged cyclically
# and that the edge A0,A1 is horizontal (and lowest on the screen?).
# Make of the quadrangle a rectangular approximation with sides horizontal and
# vertical. If horizontal=True, make the new box to have sides horizontal and
# vertical, otherwise try to maintain the inclination of the box.
# Return corner points in the order corresponding to that of the input quadrangle.
# Args:
# - points:     [complex,complex,complex,complex]
# - horizontal: boolean
# Returns:
# - [complex,complex,complex,complex]
def generate_box_from_points(points, horizontal=True):
    center = sum(points) / 4
    q = [p-center for p in points]
    sides = [q[1]-q[0],
            (q[2]-q[1])*(1j),
            (q[3]-q[2])*(-1),
            (q[0]-q[3])*(-1j)]
    try:
        reference_direction = sum(sides)
        ref1 = reference_direction / abs(reference_direction)
    except ZeroDivisionError: # Never?
        ref1 = 1
    # Rotate points to horizontal (reference_direction -> horizontal)
    q = [x / ref1 for x in q]
    r = [q[0],
         complex(-q[1].real,  q[1].imag),
         complex(-q[2].real, -q[2].imag), # -q[2]
         complex( q[3].real, -q[3].imag)]
    ave = sum(r) / 4
    x,y = ave.real, ave.imag
    s = [ave,
         complex(-x,y),
         complex(-x,-y),
         complex(x,-y)]
    if not horizontal: # Rotate back
         s = [x * ref1 for x in s]
    return [p+center for p in s]

#---------------------------------------------------------
#                  Routines for ellipses
#---------------------------------------------------------

# -----------
# determinant
# -----------
# Determinant of a square matrix.
# The matrix is given row by row (or column by column) as
# [[float,...,float], ..., [float,...,float]].
# Args:
# - M: [[float]] (or [[integer]])
# - check_square: boolean (if True, check that the matrix is square)
# Returns:
# - float (or integer)
# Notes:
# 1. The returned value is integer if M is an integer matrix.
def determinant(M, check_square=True):
    from copy import deepcopy
    if check_square:
        n = len(M)
        for row in M:
            if len(row) != n:
                raise Exception("determinant: input matrix not square")
    if len(M) <= 1:
        try:
            return M[0][0]
        except IndexError:
            raise Exception("determinant: input matrix empty?")
    if len(M) == 2:
        return M[0][0] * M[1][1] - M[1][0] * M[0][1]
    det = 0
    sign = +1
    for j in range(len(M)):
        Mj = deepcopy(M)[1:]     # copy and delete first row
        for i in range(len(Mj)): # delete column j
            Mj[i] = Mj[i][0:j] + Mj[i][j+1:]
        det += sign * M[0][j] * determinant(Mj, False) # recursion
        sign = -sign
    return det

# -------------------
# conic_from_5_points
# -------------------
# Given 5 points in the plane, find the equation of the conic section
# passing through the 5 points.
# The plane is viewed as the complex number plane, so that points are
# complex numbers.
# The equation
#    Ax^2 + Bxy + Cy^2 + Dx + Ey + F = 0
# is returned as the list [A,B,C,D,E], coefficients all real.
# Args:
# - points: [complex,complex,complex,complex,complex]
# Returns:
# - [float,float,float,float,float,float]
def conic_from_5_points(points):
    if some_three_collinear(points):
        raise Exception("conic_from_5_points: some three input points are collinear")
    v = [[p.real,p.imag] for p in points]
    #
    A =  determinant([[x*y, y*y, x, y, 1] for x,y in v])
    B = -determinant([[x*x, y*y, x, y, 1] for x,y in v])
    C =  determinant([[x*x, x*y, x, y, 1] for x,y in v])
    D = -determinant([[x*x, x*y, y*y, y, 1] for x,y in v])
    E =  determinant([[x*x, x*y, y*y, x, 1] for x,y in v])
    F = -determinant([[x*x, x*y, y*y, x, y] for x,y in v])
    m = max(abs(c) for c in [A,B,C,D,E,F]) # Normalize coefficients
    return [A/m, B/m, C/m, D/m, E/m, F/m]

# --------------
# ellipse_center
# --------------
# Given an ellipse (equation Ax^2 + Bxy + Cy^2 + Dx + Ey + F = 0),
# find its center.
# The plane is viewed as the complex number plane: points are complex numbers.
# Args:
# - coeff: [float,float,float,float,float,float] (coefficients A,...,F)
# Returns:
# - complex           (the center)
def ellipse_center(coeff):
    A,B,C,D,E,F = coeff
    det = B*B - 4*A*C
    if det > 0:
        raise Exception("ellipse_center: input is a hyperbola, not ellipse")
    elif det == 0:
        raise Exception("ellipse_center: input is a parabola, not ellipse")
    return complex((2*C*D - B*E)/det, (-B*D + 2*A*E)/det)

# ------------
# ellipse_axes
# ------------
# Given an ellipse (equation Ax^2 + Bxy + Cy^2 + Dx + Ey + F = 0),
# find the end points of the axes. Return the long and the short axes separately.
# The plane is viewed as the complex number plane: points are complex numbers.
# Args:
# - coeff: [float,float,float,float,float,float] (coefficients A,...,F)
# Returns:
# - [complex,complex] (end points of the long axis)
# - [complex,complex] (end points of the short axis)
def ellipse_axes(coeff):
    from math import sqrt
    A,B,C,D,E,F = coeff
    center = ellipse_center(coeff)
    x0,y0 = center.real, center.imag
    # Equation is ax^2 + bxy + cy^2 + f = 0 when the origo is translated to the
    # center; coefficients:
    a,b,c = A,B,C
    f = A*x0*x0 + B*x0*y0 + C*y0*y0 + D*x0 + E*y0 + F
    root = sqrt((a-c)**2 + b**2)
    try:
        tau1 = (-a + c + root)/b
        tau2 = (-a + c - root)/b
        tau_denom = [[tau1, (a + b*tau1 + c*tau1*tau1)],
                     [tau2, (a + b*tau2 + c*tau2*tau2)]]
    except ZeroDivisionError:
        try:
            fc, fa = sqrt(-f/c), sqrt(-f/a)
            if abs(c) <= abs(a):
                long = [complex(0,fc), complex(0,-fc)]
                short = [complex(fa,0), complex(-fa,0)]
            else:
                short = [complex(0,fc), complex(0,-fc)]
                long = [complex(fa,0), complex(-fa,0)]
            return [center + z for z in long], [center + z for z in short], center
        except ValueError:
            raise Exception("ellipse_axes: ValueError when b = 0")
    try:
        tau,denom = tau_denom[0]
        x = sqrt(-f / denom)
        first = [complex(x, tau*x), complex(-x, -tau*x)]
        #
        tau,denom = tau_denom[1]
        x = sqrt(-f / denom)
        second = [complex(x, tau*x), complex(-x, -tau*x)]
        #
        if abs(first[1]-first[0]) >= abs(second[1]-second[0]):
            long, short = first, second
        else:
            long, short = second, first
    except ValueError:
        raise Exception("ellipse_axes: ValueError when b != 0")
    return [center + z for z in long], [center + z for z in short]#, center

# ---------------------------
# ellipse_conjugate_diameters
# ---------------------------
# Given an ellipse (equation Ax^2 + Bxy + Cy^2 + Dx + Ey + F = 0),
# and a direction vector (complex), find the end points of the pair of
# conjugate diameters, one of them being parallel to the direction vector.
# The plane is viewed as the complex number plane: points are complex numbers.
# Args:
# - coeff: [float,float,float,float,float,float] (coefficients A,...,F)
# - direction: complex
# Returns:
# - [complex,complex] (end points of one diameter, parallel to the direction vector)
# - [complex,complex] (end points of the conjugate diameter)
def ellipse_conjugate_diameters(coeff, direction):
    from math import sqrt
    A,B,C,D,E,F = coeff
    center = ellipse_center(coeff)
    x0,y0 = center.real, center.imag
    # Equation is ax^2 + bxy + cy^2 + f = 0 when the origo is translated to the
    # center; coefficients:
    a,b,c = A,B,C
    f = A*x0*x0 + B*x0*y0 + C*y0*y0 + D*x0 + E*y0 + F
    # diameter1
    vx,vy = direction.real, direction.imag
    try:
        K1 = -f / (a*vx*vx + b*vx*vy + c*vy*vy)
    except ZeroDivisionError:
        raise Exception("ellipse_conjugate_diameters: ZeroDivisionError: K1")
    try:
        k1 = sqrt(K1)
    except ValueError:
        raise Exception("ellipse_conjugate_diameters: ValueError: k1")
    diameter1 = [center - complex(k1*vx, k1*vy), center + complex(k1*vx, k1*vy)]
    # diameter2
    wx,wy = -b*vx - 2*c*vy, 2*a*vx + b*vy
    try:
        K2 = -f / (a*wx*wx + b*wx*wy + c*wy*wy)
    except ZeroDivisionError:
        raise Exception("ellipse_conjugate_diameters: ZeroDivisionError: K21")
    try:
        k2 = sqrt(K2)
    except ValueError:
        raise Exception("ellipse_conjugate_diameters: ValueError: k2")
    diameter2 = [center - complex(k2*wx, k2*wy), center + complex(k2*wx, k2*wy)]
    return diameter1, diameter2

# -------------------------
# ellipse_line_intersection
# -------------------------
# Given an ellipse, a point K on the ellipse, and another points C (!=K),
# find the other intersection point of the line through K,C with the ellipse.
# The ellipse is Ax^2 + Bxy + Cy^2 + Dx + Ey + F = 0, input as
# coeff=[A,B,C,D,E,F].
# The plane is viewed as the complex number plane, so all points are
# complex numbers.
# Args:
# - coeff: [float,float,float,float,float] (=[A,B,C,D,E,F])
# - point_on_ellipse: complex
# - some_other_point: complex
# Returns:
# - complex
def ellipse_line_intersection(coeff, point_on_ellipse, some_other_point):
    ZERO = 1e-12
    def check(coeff, z): # point on ellipse?
        A,B,C,D,E,F = coeff
        x,y = z.real, z.imag
        return (abs(A*x*x + B*x*y + C*y*y + D*x + E*y + F) < ZERO)
    if not check(coeff,point_on_ellipse):
        raise Exception("ellipse_line_intersection: point not on ellipse")
    A,B,C,D,E,F = coeff
    kz = point_on_ellipse
    cz = some_other_point
    if abs(kz-cz) < ZERO:
        raise Exception("ellipse_line_intersection: input points too close:"+str(abs(kz-cz)))
    sz = cz-kz # the direction vector of the line
    kx,ky = kz.real, kz.imag
    sx,sy = sz.real, sz.imag
    try:
        idenom = 1. / (A*sx*sx + B*sx*sy + C*sy*sy)
    except ZeroDivisionError:
        raise Exception("ellipse_line_intersection: denom=0")
    numer = 2 * (A*sx*kx + B*(sy*kx + sx*ky)/2 + C*sy*ky) + D*sx + E*sy
    t = -numer*idenom
    return kz + t*sz


#---------------------------------------------------------
#       classes: Matrix2x2, Matrix3x3
#---------------------------------------------------------
class Matrix2x2(object):
    """2x2-matrix
    Initialize m = Matrix2x2(1,2,3,4)
    Get elements as m[i,j]
    Set elements as m[i,j]=value
    Note: Works with real or complex matrices.
    """
    def __init__(self,a00,a01,a10,a11):
        #self._elements = [float(a00),float(a01),float(a10),float(a11)]
        self._elements = [a00,a01,a10,a11]
    def __getitem__(self,indexes): # indexes = [i,j]
        i,j = indexes
        if not (0<=i<=1 and 0<=j<=1):
            raise Exception("Matrix2x2 get: Index out of range")
        return self._elements[2*i+j]
    def __setitem__(self, indexes, value):
        i,j = indexes
        if not (0<=i<=1 and 0<=j<=1):
            raise Exception("Matrix2x2 set: Index out of range")
        self._elements[2*i+j] = value
    def transpose(self):
        a,b,c,d = self._elements
        return Matrix2x2(a,c,b,d)
    def scale(self, coeff):
        a,b,c,d = [x*coeff for x in self._elements]
        return Matrix2x2(a,b,c,d)
    def inverse(self):
        a,b,c,d = self._elements
        try:
            idet = 1./(a*d-b*c)
        except ZeroDivisionError:
            raise Exception("Matrix2x2 inverse: Zero determinant")
        A,B,C,D = [x*idet for x in self._elements]
        return Matrix2x2(D,-B,-C,A)
    def __add__(self,other):
        a,b,c,d = self._elements
        A,B,C,D = other._elements
        return Matrix2x2(a+A,b+B,c+C,d+D)
    def __mul__(self,other):
        return Matrix2x2(self[0,0]*other[0,0] + self[0,1]*other[1,0],
                         self[0,0]*other[0,1] + self[0,1]*other[1,1],
                         self[1,0]*other[0,0] + self[1,1]*other[1,0],
                         self[1,0]*other[0,1] + self[1,1]*other[1,1])
    def __str__(self):
        def str_float(x): return '{:1.3f}'.format(x)
        a00 = str_float(self[0,0])
        a01 = str_float(self[0,1])
        a10 = str_float(self[1,0])
        a11 = str_float(self[1,1])
        return '['+a00+','+a01+']' +'\n['+a10+','+a11+']\n'

def rv_times_M2x2(rv,matrix):
    return complex(rv.real*matrix[0,0] + rv.imag*matrix[1,0],
                   rv.real*matrix[0,1] + rv.imag*matrix[1,1])

class Matrix3x3(object):
    """3x3-matrix
    Simple implementation, mainly just to perform matrix multiplication.
    """
    def __init__(self,a00,a01,a02, # Elements row by row
                      a10,a11,a12,
                      a20,a21,a22,
                      ):
        self._elements = [a00,a01,a02,a10,a11,a12,a20,a21,a22]
    def elements(self):
        # Returns list of matrix elements row by row.
        return self._elements
    def __mul__(self,other):
        s = self._elements
        o = other._elements
        return Matrix3x3(s[0]*o[0] + s[1]*o[3] + s[2]*o[6], # 00
                         s[0]*o[1] + s[1]*o[4] + s[2]*o[7], # 01
                         s[0]*o[2] + s[1]*o[5] + s[2]*o[8], # 02
                         s[3]*o[0] + s[4]*o[3] + s[5]*o[6], # 10
                         s[3]*o[1] + s[4]*o[4] + s[5]*o[7], # 11
                         s[3]*o[2] + s[4]*o[5] + s[5]*o[8], # 12
                         s[6]*o[0] + s[7]*o[3] + s[8]*o[6], # 20
                         s[6]*o[1] + s[7]*o[4] + s[8]*o[7], # 21
                         s[6]*o[2] + s[7]*o[5] + s[8]*o[8], # 22
                         )
    def __str__(self):
        s = self._elements
        return str(s[:3]) + '\n'+str(s[3:6])+'\n'+str(s[6:])

class MakeProjectiveMapError(Exception):
    def __init__(self, case, cause, points):
        self.case = case
        self.cause = cause
        self.points = points

# ----------------------
# matrix33_times_vector3
# ----------------------
# Matrix3x3 times 3-dimensional vector
# Args:
# - m: Matrix3x3
# - v: [complex,complex,complex] (coordinates, perhaps homogeneous)
# Returns:
# - [complex,complex,complex]
def matrix33_times_vector3(m,v):
    matrix_elements = m.elements()
    row0 = matrix_elements[:3]
    row1 = matrix_elements[3:6]
    row2 = matrix_elements[6:]
    return [sum([x*y for x,y in zip(row0,v)]),
            sum([x*y for x,y in zip(row1,v)]),
            sum([x*y for x,y in zip(row2,v)])]

# ----------------------
# matrix33_times_complex
# ----------------------
# Real Matrix3x3 times 2-dimensional vector represented by a complex number:
# 1. convert vector (complex) to homogeneous coordinates ([float,float,float=1]);
# 2. multiply by the matrix (=> [float,float,float]);
# 3. convert back to complex (raise Exception if infinite).
# Args:
# - m: Matrix3x3
# - z: complex
# Returns:
# - complex
def matrix33_times_complex(m,z):
    v = [z.real,z.imag,1]
    mv = matrix33_times_vector3(m,v)
    try:
        return complex(mv[0]/mv[2], mv[1]/mv[2])
    except ZeroDivisionError:
        raise Exception("matrix33_times_complex: ZeroDivisionError: hit infinity")


#---------------------------------------------------------
#             Make projective transformation
#---------------------------------------------------------

# --------------------------
# make_projective_map_matrix
# --------------------------
# Given four points A,B,C,D (base) in a general position (no three collinear),
# another four points P,Q,R,S (target) in a general position,
# make projective transformation which sends
#    A -> P
#    B -> Q
#    C -> R
#    D -> S
# Return the attached 3x3-matrix.
# Return also the coefficients g,h,k of the line g*x+h*y+k=0 for
# (1) the image of ideal line and
# (2) the pre-image of the ideal line.
# 
# Args:
# - base:   [A,B,C,D]: [complex,complex,complex,complex]
# - target: [P,Q,R,S]: [complex,complex,complex,complex]
# Returns:
# - Matrix3x3
# - [float,float,float] or None (coefficients g,h,k for the pre-image of ideal line)
# - [float,float,float] or None (coefficients g,h,k for the image of ideal line)
# Exceptions:
# - MakeProjectiveMapErrordef make_projective_map_matrix(base,target):
def make_projective_map_matrix(base,target):
    # Check base
    A,B,C,D = base # complex
    if some_three_collinear(base):
        #raise Exception("make_projective_map_matrix: Some three Base points are collinear,")
        raise MakeProjectiveMapError(case=1,
                                     cause="some three base points collinear",
                                     points=base)
    # Check target
    P,Q,R,S = target # complex
    if some_three_collinear(target):
         #raise Exception("make_projective_map_matrix: Some three Target points are collinear,")
         raise MakeProjectiveMapError(case=2,
                                      cause="some three target points collinear",
                                      points=target)
    #### Step 1 ###
    BB = B-A # B'
    CC = C-A # C'
    DD = D-A # D'
    try:
        M2 = Matrix2x2(BB.real,BB.imag,DD.real,DD.imag).inverse()
    except Exception:
        raise MakeProjectiveMapError(case=3,
                                     cause="det=0, should never happen",
                                     points=base)
    CCC = rv_times_M2x2(CC,M2)
    # g3,h3:
    g3 = (CCC.imag-1)/CCC.real
    h3 = (CCC.real-1)/CCC.imag
    #### Step 2 ###
    KF1 = Matrix3x3( 1, 0, -A.real,
                     0, 1, -A.imag,
                     0, 0, 1)
    #### Step 3 ###
    KF2 = Matrix3x3( DD.imag, -DD.real, 0,
                    -BB.imag,  BB.real, 0,
                     0,      0,    BB.real*DD.imag - BB.imag*DD.real)
    #### Step 4 ###
    KF3 = Matrix3x3( g3+1, 0,   0,
                     0,   h3+1, 0,
                     g3,   h3,   1)
    #### Step 5 ###
    c0,f0 = P.real,P.imag
    try:
        PQ = Q-P
        SR = R-S
        QR = R-Q
        PS = S-P
        denom = cross(QR,SR)
        g0 =  cross(PQ,SR) / denom
        h0 = -cross(PS,QR) / denom
    except ZeroDivisionError:
        #raise Exception("make_projective_map_matrix: Should never happen.")
        raise MakeProjectiveMapError(case=4,
                                     cause="denom=0, should never happen",
                                     points=target)
    ad0 = (g0+1)*Q - P # complex
    be0 = (h0+1)*S - P # complex
    a0,d0 = ad0.real,ad0.imag
    b0,e0 = be0.real,be0.imag
    KG = Matrix3x3( a0, b0, c0,
                    d0, e0, f0,
                    g0, h0, 1 )
    #### Step 6 ###
    KF = KG * KF3 * KF2 * KF1 # The 3x3-matrix of the projective transformation F
    KF_elements = KF.elements()
    # To avoid large numbers normalize KF by condition k=1,
    # or if k=0, by condition g**2+h**2=1:
    try:
        inv = 1./KF_elements[-1]
    except ZeroDivisionError:
        try:
            inv = 1./sqrt(KF_elements[-3]**2 + KF_elements[-2]**2)
        except ZeroDivisionError:
            #raise Exception("make_projective_map_matrix: Should never happen: zero bottom row!")
            raise MakeProjectiveMapError(case=5,
                                     cause="zero bottom row, should never happen",
                                     points=target)
    KF_elements = [KFe * inv for KFe in KF_elements]
    # matrix elements row by row:
    a,b,c = KF_elements[:3]
    d,e,f = KF_elements[3:6]
    g,h,k = KF_elements[6:]
    matrix = Matrix3x3(a,b,c,d,e,f,g,h,k)
    pre_image_ideal = [g,h,k]
    image_ideal = [d*h-e*g, -a*h+b*g, a*e-b*d]
    return matrix, pre_image_ideal, image_ideal


#---------------------------------------------------------
#     Apply projective transformation: main procedures
#---------------------------------------------------------

class PT_InfinityError(Exception):
    def __init__(self,message):
        self.message = message

# ------------------------------
# projective_transform_by_points
# ------------------------------
# Called by the main procedures of the plugins.
# Calls pdb.gimp_item_transform_matrix to do the real work.
#
# Given four points (source=[A,B,C,D]) and another four points (target=[P,Q,R,S]),
# transform the input layer by the projective transformation which sends the
# points A,B,C,D to the points P,Q,R,S.
# The plane is viewed as the complex number plane, so the points are complex numbers.
# Return also the coefficients g,h,k of the line g*x+h*y+k=0 for
# (1) the image of the ideal line and
# (2) the pre-image of the ideal line
# ("the ideal line" means the line at infinity).
# Args:
# - layer
# - source:        [complex,complex,complex,complex]
# - target:        [complex,complex,complex,complex]
# - mark_pre_ideal: boolean
# - mark_ideal:     boolean
# - keep_original:  boolean
# Returns:
# - the transformed layer
def projective_transform_by_points(image,
                                   layer,
                                   source,
                                   target,
                                   mark_pre_ideal,
                                   mark_ideal,
                                   keep_original,
                                   check_infinity=True):
    matrix, pre_image_ideal, image_ideal = make_projective_map_matrix(source,target)
    if check_infinity:
        if box_meets_line(layer.height,
                          layer.width,
                          layer.offsets[0],
                          layer.offsets[1],
                          pre_image_ideal,
                          safety=0):
            raise PT_InfinityError("Part of the layer would go to infinity.")
    border = [0j, # Image border
          complex(0,image.height),
          complex(image.width,image.height),
          complex(image.width,0),
          0j
          ]
    if mark_ideal:
        if not (image_ideal is None): # image_ideal = [g,h,k]
            points_on_border = line_intersect_polyline(image_ideal, border) 
            if len(points_on_border) > 0:
                draw_polyline(image, points_on_border, name='line come from infinity (vanishing line)', closed=False)
            else:
                m = "line coming from infinity is not in the window (not marked)."
                gimp_message(m)
    if mark_pre_ideal:
        if not (pre_image_ideal is None): # pre_image_ideal = [g,h,k]
            points_on_border = line_intersect_polyline(pre_image_ideal, border) 
            if len(points_on_border) > 0:
                draw_polyline(image, points_on_border, name='line that goes to infinity', closed=False)
            else:
                m = "line that goes to infinity is not in the window (not marked)."
                gimp_message(m)
    matrix_elements = matrix.elements()
    coeff_0_0, coeff_0_1, coeff_0_2 = matrix_elements[:3]  # first row
    coeff_1_0, coeff_1_1, coeff_1_2 = matrix_elements[3:6] # second row
    coeff_2_0, coeff_2_1, coeff_2_2 = matrix_elements[6:]  # third row
    pdb.gimp_image_undo_group_start(image)
    if keep_original:
        add_alpha=True
        work_layer = pdb.gimp_layer_copy(layer, add_alpha)
        work_layer.name = layer.name+'|perspective transform'
        parent = None
        position = 0
        pdb.gimp_image_insert_layer(image, work_layer, parent, position)
    else:
        work_layer = layer
    transformed = pdb.gimp_item_transform_matrix(work_layer,
                                                 coeff_0_0, coeff_0_1, coeff_0_2,
                                                 coeff_1_0, coeff_1_1, coeff_1_2,
                                                 coeff_2_0, coeff_2_1, coeff_2_2)
    pdb.gimp_image_undo_group_end(image)
    return transformed

# ---------------------------------
# perspective_transform_4_to_4_main
# ---------------------------------
# Given two vectors objects (source_path, target_path), 
# transform the active layer by the projective transformation which sends the
# anchors of the first stroke of 'source_path' to
# the anchors of the first stroke of 'target_path'.
# Args:
# - image
# - source_path:     gimp.Vectors
# - target_path:     gimp.Vectors
# - reversed_target: boolean (use the target path reversed?)
# - keep_original:   boolean (if True, work on a copy, otherwise overwrite original)
# Returns:
# - the transformed layer
def perspective_transform_4_to_4_main(image,
                                  source_path,
                                  target_path,
                                  reversed_target,
                                  #mark_pre_ideal,
                                  #mark_ideal,
                                  keep_original
                                  ):
    mark_pre_ideal = mark_ideal = False # No option for user!
    active_layer = pdb.gimp_image_get_active_layer(image)
    source = get_anchors(source_path, only_first_stroke=True) # [complex]
    target = get_anchors(target_path, only_first_stroke=True) # [complex]
    if some_three_collinear(source):
        raise Exception("The source path contains three collinear anchors. This is not allowed.")
    if some_three_collinear(target):
        raise Exception("The target path contains three collinear anchors. This is not allowed.")
    if reversed_target:
        target.reverse()
    if len(source) != 4:
        raise Exception("The source path must have 4 anchors in first stroke, got "+str(len(source)))
    if len(target) != 4:
        raise Exception("The target path must have 4 anchors in first stroke, got "+str(len(target)))
    try:
        new_layer = projective_transform_by_points(image,
                                          active_layer,
                                          source,
                                          target,
                                          mark_pre_ideal,
                                          mark_ideal,
                                          keep_original
                                          )
        return new_layer
    except PT_InfinityError as e:
        m = e.message
        m += "\nCannot handle this. Things to try:"
        m += "\n- check that the right layer is active;"
        m += "\n- use the target path reversed;"
        m += "\n- crop the active layer to its essential content;"
        m += "\n- check that the source and target paths are not too dissimilar."
        m += "\n- change the paths a little."
        raise Exception(m)


target_box_options = [ # (description, identifier)
        ('Box generated from the source path',          'gen source'),
        ('Rectangular selection',                  'bb_selection'),
        ('Guides - two horizontal, two vertical',                                      'guides'),
        #('Bounding box of the source path',            'bb_path'),
        #('Bounding box of anchors of the source path', 'bb_anchors'),
        #('Image (canvas)',      'image')
        ]

# -----------------------------------------
# perspective_transform_4_to_rectangle_main
# -----------------------------------------
# Given a vectors object and a box with sides horizontal and vertical,
# transform the active layer by the projective transformation which sends the
# anchors of the first stroke of 'source_path' to the corners of the target box.
# Args:
# - image
# - source_path:       gimp.Vectors
# - target_box_option: integer (see target_box_options)
# - rotate:            float (rotate box, in degrees)
# - keep_original:     boolean (if True, work on a copy, otherwise overwrite original)
# Returns:
# - the transformed layer
def perspective_transform_4_to_rectangle_main(image,
                                  source_path,
                                  target_box_option,
                                  rotate,
                                  #inverse_op,
                                  #mark_pre_ideal,
                                  #mark_ideal,
                                  keep_original
                                  ):
    mark_pre_ideal = mark_ideal = False # No option for user!
    inverse_op = False                  # No option for user!
    source = get_anchors(source_path, only_first_stroke=True) # [complex]
    source_corners = arrange_cps(source, Gimp=True) # a[0],a[1] bottom edge, left to right.
    box_case = target_box_options[target_box_option][1]
    box_corners = get_box(image, source_path, box_case, rotate, Gimp=True) # [SW, SE, NE, NW]
    #
    if some_three_collinear(source):
        raise Exception("The source path contains three collinear anchors. This is not allowed.")
    if len(source) != 4:
        raise Exception("The source path must have 4 anchors in first stroke, got "+str(len(source)))
    if inverse_op:
        source_corners, box_corners = box_corners, source_corners
    active_layer = pdb.gimp_image_get_active_layer(image)
    if box_case == 'bb_selection':
        # The following check, that the pre_ideal does not meet the box,
        # should be done in projective_transform_by_points (and is normally done),
        # but for some reason Gimp works strangely: In the case of Exception,
        # the plugin appears in the undo stack.
        # Therefore the check is done here.
        matrix, pre_ideal,_ = make_projective_map_matrix(source_corners, box_corners)
        infinity_occurs = box_meets_line(active_layer.height,
                                         active_layer.width,
                                         active_layer.offsets[0],
                                         active_layer.offsets[1],
                                         pre_ideal,
                                         safety=0)
        if not infinity_occurs: # Ok, do the transform
            # To get pdb.gimp_item_transform_matrix work on the whole layer,
            # we must kill the selection temporarily.
            pdb.gimp_image_undo_group_start(image)
            saved_to_channel = pdb.gimp_selection_save(image)
            pdb.gimp_selection_none(image)
            infinity_error = None
            try:
                new_layer = projective_transform_by_points(image,
                                              active_layer,
                                              source_corners,
                                              box_corners,
                                              mark_pre_ideal,
                                              mark_ideal,
                                              keep_original,
                                              check_infinity=False # not twice
                                              )
                # Restore the selection
                pdb.gimp_image_select_item(image, CHANNEL_OP_REPLACE, saved_to_channel)
                pdb.gimp_image_remove_channel(image, saved_to_channel)
                pdb.gimp_image_undo_group_end(image)
                return new_layer
            except PT_InfinityError as e:
                infinity_error = e.message
                gimp_message("Should not happen: "+infinity_error)
        else: # infinity_occurs: raise Exception
            m = "Part of the layer would go to infinity."
            m += "\nCannot handle this. Things to try:"
            m += "\n- check that the right layer is active;"
            m += "\n- crop the active layer to its essential content;"
            m += "\n- check the source path (should enclose a convex region)."
            raise Exception(m)
    else:
        try:
            new_layer = projective_transform_by_points(image,
                                          active_layer,
                                          source_corners,
                                          box_corners,
                                          mark_pre_ideal,
                                          mark_ideal,
                                          keep_original
                                          )
            return new_layer
        except PT_InfinityError as e:
            m = e.message
            m += "\nCannot handle this. Things to try:"
            m += "\n- check that the right layer is active;"
            m += "\n- crop the active layer to its essential content;"
            m += "\n- check the source path (should enclose a convex region)."
            raise Exception(m)

# --------------------------------------
# perspective_transform_4_to_circle_main
# --------------------------------------
# Given a source path with 4 anchors, transform the active layer by the projective
# transformation which sends the 4 anchors onto specified locations on a circle.
# Args:
# - image
# - source_path:     gimp.Vectors
# - hour0:           float (corresponding hour of the anchor on the clock face)
# - hour1:           float (corresponding hour of the anchor on the clock face)
# - hour2:           float (corresponding hour of the anchor on the clock face)
# - hour3:           float (corresponding hour of the anchor on the clock face)
# - reversed_source: boolean (use the reversed version of source_path?)
# - rotate:          float (rotate the resulting aarc around the circle center)
# - keep_original:   boolean (if True, work on a copy, otherwise overwrite original)
# Returns:
# - the transformed layer
def perspective_transform_4_to_circle_main(image,
                                  source_path,
                                  hour0,
                                  hour1,
                                  hour2,
                                  hour3,
                                  reversed_source,
                                  rotate,
                                  #mark_pre_ideal,
                                  #mark_ideal,
                                  keep_original
                                  ):
    mark_pre_ideal = mark_ideal = False # No option for user!
    active_layer = pdb.gimp_image_get_active_layer(image)
    source = get_anchors(source_path, only_first_stroke=True) # [complex]
    if some_three_collinear(source):
        raise Exception("The source path contains three collinear anchors. This is not allowed.")
    if len(source) != 4:
        raise Exception("The source path must have 4 anchors in first stroke, got "+str(len(source)))
    if reversed_source:
        source.reverse()
    clock_points = get_clock_points(source, [hour0,hour1,hour2,hour3], rotate)
    try:
        #pdb.gimp_image_undo_group_start(image)
        new_layer = projective_transform_by_points(image,
                                      active_layer,
                                      source,
                                      clock_points,
                                      mark_pre_ideal,
                                      mark_ideal,
                                      keep_original
                                      )
        #pdb.gimp_image_undo_group_end(image)  # KOE
        return new_layer
    except PT_InfinityError as e:
        m = e.message
        m += "\nCannot handle this. Things to try:"
        m += "\n- check that the right layer is active;"
        m += "\n- use the source path reversed;"
        m += "\n- check the source path and the hours."
        m += "\n- crop the active layer to its essential content;"
        raise Exception(m)


# -----------------
# matrix_from_1_2_5
# -----------------
# Make the 3x3 matrix for correcting a distorted circle (ellipse) to a circle.
# Inputs are:
# - would_be_center:  the point inside the distorted circle which should become
#                     the center of the corrected circle;
# - direction_points: 2 points giving the direction that 
#                     should be turned to goal_direction;
# - five_points:      5 points of the distorted circle;
# - goal_direction:   the goal direction of the direction_points (prior to rotations);
# - rotate180:        possible 180 degrees rotation after direction set;
# - rotate:           possible further rotation after everything else.
# Args:
# - image
# - would_be_center_path: complex           (one point in the plane)
# - five_points:          [complex,..,complex] (five points in the plane)
# - direction_points:     [complex,complex] (two points in the plane)
# - goal_direction:       complex           (unit vector in complex plane)
# - rotate180:            boolean 
# - rotate:               float (rotate the result, degrees, clock-wise)
# Returns:
# - Matrix3x3
def matrix_from_1_2_5(would_be_center,
                      five_points,
                      direction_points,
                      goal_direction,
                      rotate180,
                      rotate
                      ):
    from cmath import exp as cexp
    # Get the ellipse and its center from the 5 points ("ellipse a"):
    coeff_a = conic_from_5_points(five_points) # coefficients
    try:
        center_a = ellipse_center(coeff_a)     # center
    except Exception as e:
        m = "An error occured:"
        m += "\n\""+str(e)+"\""
        m += "\nPlease check the input 5-anchors path"
        raise Exception(m)
    Ca = would_be_center
    Ka, Na = five_points[:2]
    #
    # Take the two points Ka,Na of the direction (two points on the distorted
    # circle), and find corresponding points Ha,Sa on the opposite side of the
    # ellipse such that Ka,Ca,Ha are collinear and Na,Ca,Sa are collinear.
    # This gives an inscribed quadrangle Na,Ka,Sa,Ha such that Ca is the
    # intersection of the diagonals:
    Ha = ellipse_line_intersection(coeff_a, Ka, Ca) # opposite to Ka
    Sa = ellipse_line_intersection(coeff_a, Na, Ca) # opposite to Na
    if cross(-Sa+Na, -Ha+Ka) < 0:
        Ha,Ka = Ka,Ha # To make Na,Ka,Sa,Ha ccw (in Gimp!)
    #
    # The axes of the ellipse.
    # This will give an inscribed parallelogram Nb,Hb,Sb,Kb
    # (its diagonals intersect at the center of ellipse a):
    long, short = ellipse_axes(coeff_a)
    if long[0].imag < long[1].imag:
        long.reverse() # To points upwards.
    if cross(-long[0]+long[1], -short[0]+short[1]) < 0:
        short.reverse() # To make Nb,Hb,Sb,Kb ccw (in Gimp!).
    Sb,Nb = long
    Hb,Kb = short
    #
    # The 3x3 matrix (projective transformation!) which sends the inscribed
    # quadrangle Na,Ka,Sa,Ha to the inscribed parallelogram Nb,Hb,Sb,Kb:
    matrix_ab,_,_ = make_projective_map_matrix([Na, Sa, Ka, Ha],
                                               [Nb, Sb, Kb, Hb])
    # Transform ellipse a by the matrix. This gives a new ellipse b
    # with the same inscribed parallelogram Nb,Hb,Sb,Kb:
    ell5b = [matrix33_times_complex(matrix_ab, z) for z in five_points] # 5 points on ellipse b
    coeff_b = conic_from_5_points(ell5b)       # The coefficients of ellipse b
    try:
        center_b = ellipse_center(coeff_b)     # Find center just to check
    except Exception as e:                     # possible Exception
        m = "An error occured when computing an intermediate ellipse:"
        m += "\n\""+str(e)+"\""
        m += "\nPlease check the input paths (5 points, would-be center)"
        raise Exception(m)
    #
    # [Sb,Nb] is a diameter of ellipse b. Find its conjugate diameter:
    diam1, diam2 = ellipse_conjugate_diameters(coeff_b, Nb-Sb)
    if vdot(-diam1[0]+diam1[1], -Sb+Nb)<0: # Should be diam1=[Sb,Nb] anyway
        diam1.reverse()
    if cross(-diam1[0]+diam1[1], -diam2[0]+diam2[1]) < 0:
        diam2.reverse() # To make Nb,Eb,Sb,Wb ccw.
    Eb,Wb = diam2
    #
    # The pair of conjugate diameters can be mapped (even by an affine
    # transformation) to a pair of perpendicular diameters of a circle.
    # That transformation sends ellipse b onto that circle and the
    # center of ellipse b onto the center of the circle.
    # This transformation is done below (matrix_bc).
    #
    # Form circle c:
    # 1. the unit circle, with the default orientation:
    circle_points = [-1j, 1j, 1, -1] # N, S, E, W (Gimp!)
    # 2. Circle c: take center and size from original ellipse a:
    long_a, short_a = ellipse_axes(coeff_a)
    big_radius_a = abs(long_a[0]-long_a[1]) / 2
    Nc, Sc, Ec, Wc = [center_a + big_radius_a*z for z in circle_points] # circle c
    #
    # The 3x3 matrix which performs the transformation sending
    # ellipse b onto circle c:
    matrix_bc,_,_ = make_projective_map_matrix([Nb, Sb, Eb, Wb],
                                               [Nc, Sc, Ec, Wc])
    matrix_ac = matrix_bc * matrix_ab # the total transformation a->b->c
    #
    # Now we have the total transformation.
    # We must still correct it so that the direction will be as required.
    # Find the actual direction after the transformation:
    dp = [direction_points[0], direction_points[1]]
    if dp[0].imag < dp[1].imag:
        dp.reverse() # To point upwards in Gimp
    mapped_direction_points = [matrix33_times_complex(matrix_ac, z) for z in dp]
    mapped_direction = -mapped_direction_points[0]+mapped_direction_points[1]
    mapped_direction = mapped_direction / abs(mapped_direction)
    # Do correction to unit circle:
    zdir_corr = goal_direction / mapped_direction        # Corrective angle
    circle_points = [zdir_corr*z for z in circle_points] # Corrected unit circle
    #
    # Rotations from inputs:
    if rotate180:
        circle_points = [-z for z in circle_points]
    zrotate = cexp(1j*rotate*pi/180)
    circle_points = [zrotate*z for z in circle_points]
    # Final circle d:
    Nd, Sd, Ed, Wd = [center_a + big_radius_a*z for z in circle_points] # circle d
    # The 3x3 matrix which performs the transformation sending
    # ellipse b onto circle d:
    matrix_bd,_,_ = make_projective_map_matrix([Nb, Sb, Eb, Wb],
                                               [Nd, Sd, Ed, Wd])
    # New matrix of the total transformation a->b->d:
    matrix_ad = matrix_bd * matrix_ab
    return matrix_ad


orientation_options = [ # (description, identifier)
        ('Make it vertical',             'vertical'),
        ('Make it horizontal',           'horizontal'),
        ('Keep its direction as it is',  'keep'),
        ]

# ------------------------------------------
# perspective_transform_2_2_2_to_circle_main
# ------------------------------------------
# Transform the active layer by the Perspective Transform trying to
# correct a specific distorted circle to a true circle.
# Inputs are four paths:
# - diameter1_path: a would-be diameter of the distorted circle;
# - diameter2_path: a would-be diameter of the distorted circle;
# - diameter3_path: a would-be diameter of the distorted circle;
# - direction_path: a 2-anchor path used for setting the direction of the figure.
# The orientation_option tells what to do with direction_path: which way to turn it
# (see orientation_options)
# Args:
# - image
# - diameter1_path:     gimp.Vectors
# - diameter2_path:     gimp.Vectors
# - diameter3_path:     gimp.Vectors
# - direction_path:     gimp.Vectors
# - orientation_option: integer (see orientation_options)
# - rotate180:          boolean 
# - rotate:             float (rotate the result, degrees, clock-wise)
# - keep_original:      boolean (if True, work on a copy, otherwise overwrite original)
# Returns:
# - the transformed layer
def perspective_transform_2_2_2_to_circle_main(image,
                                               diameter1_path,
                                               diameter2_path,
                                               diameter3_path,
                                               direction_path,
                                               orientation_option,
                                               rotate180,
                                               rotate,
                                               #mark_pre_ideal,
                                               #mark_ideal,
                                               keep_original
                                               ):
    from cmath import exp as cexp
    mark_pre_ideal = mark_ideal = False # No option for user!
    active_layer = pdb.gimp_image_get_active_layer(image)
    diameter1 = get_anchors(diameter1_path, only_first_stroke=True) # [complex,complex]
    diameter2 = get_anchors(diameter2_path, only_first_stroke=True) # [complex,complex]
    diameter3 = get_anchors(diameter3_path, only_first_stroke=True) # [complex,complex]
    direction = get_anchors(direction_path, only_first_stroke=True) # [complex,complex]
    if len(diameter1) != 2:
        raise Exception("diameter1 should have 2 anchors")
    if len(diameter2) != 2:
        raise Exception("diameter2 should have 2 anchors")
    if len(diameter3) != 2:
        raise Exception("diameter3 should have 2 anchors")
    if len(direction) != 2:
        raise Exception("the direction path should have 2 anchors")
    d1d2 = line_intersection(diameter1, diameter2)
    d1d3 = line_intersection(diameter1, diameter3)
    d2d3 = line_intersection(diameter2, diameter3)
    if None in (d1d2, d1d3, d2d3):
        m = "Check the diameters."
        m += "\nThey should be be non-parallel and of non-zero length."
        raise Exception(m)
    c12, ok12 = d1d2
    c13, ok13 = d1d3
    c23, ok23 = d2d3
    if not(ok12 and ok13 and ok23):
        m = "Check the diameters."
        m += "\nThey should intersect one another."
        raise Exception(m)
    Ca = (c12 + c13 + c23) / 3 # Average
    five_points =  diameter1 + diameter2 + [diameter3[0]] # Ignore one: need only 5
    if some_three_collinear(five_points):
        raise Exception("No collinearities allowed among the end points of the diameters.")
    # Find the goal direction:
    orientation_case = orientation_options[orientation_option][1]
    if orientation_case == 'keep':
        dir_vector = -direction[0]+direction[-1]
        try:
            goal_direction = dir_vector / abs(dir_vector)
        except ZeroDivisionError:
            raise Exception("the direction path should have non-zero length")
        if goal_direction.imag > 0:
            goal_direction *= -1
    elif orientation_case == 'horizontal':
        goal_direction = 1
    else: # vertical
        goal_direction = -1j
    
    matrix = matrix_from_1_2_5(would_be_center  = Ca,
                               five_points      = five_points,
                               direction_points = direction,
                               goal_direction   = goal_direction,
                               rotate180        = rotate180,
                               rotate           = rotate
                               )
    
    # Two quadruples of points to feed to projective_transform_by_points:
    transform_source = diameter1 + diameter2 # Just some four points on the distorted circle
    transform_target = [matrix33_times_complex(matrix, z) for z in transform_source]
    #
    # Finally, transform the layer:
    try:
        new_layer = projective_transform_by_points(image,
                                                   active_layer,
                                                   transform_source,
                                                   transform_target,
                                                   mark_pre_ideal,
                                                   mark_ideal,
                                                   keep_original
                                                   )
        return new_layer
    except PT_InfinityError as e:
        m = e.message
        m += "\nCannot handle this. Things to try:"
        m += "\n- check that the right layer is active;"
        m += "\n- crop the active layer to its essential content;"
        m += "\n- check the input paths."
        raise Exception(m)


# ------------------------------------------
# perspective_transform_1_2_3_to_circle_main
# ------------------------------------------
# Transform the active layer by the Perspective Transform trying to
# correct a specific distorted circle to a true circle.
# Input is one path 'distorted_points_path' with 3 (or 2) strokes:
# - stroke 1, 1 anchor:  the would-be center of the corrected circle;
# - stroke 2, 2 anchors: 2 points of the distorted circle;
#                        this is used to set the direction (see orientation_options);
# - stroke 3, 3 anchors: other 3 points of the distorted circle.
# The first stroke is optional; if it is missing, the center of the distorted
# circle (ellipse) will be used.
# The orientation_option tells what to do with the direction taken from stroke 2:
# which way to turn (see orientation_options)
# Args:
# - image
# - distorted_points_path:   gimp.Vectors
# - orientation_option: integer (see orientation_options)
# - rotate180:          boolean 
# - rotate:             float (rotate the result, degrees, clock-wise)
# - keep_original:      boolean (if True, work on a copy, otherwise overwrite original)
# Returns:
# - the transformed layer
def perspective_transform_1_2_3_to_circle_main(image,
                                               distorted_points_path,
                                               orientation_option,
                                               rotate180,
                                               rotate,
                                               #mark_pre_ideal,
                                               #mark_ideal,
                                               keep_original
                                               ):
    from cmath import exp as cexp
    mark_pre_ideal = mark_ideal = False # No option for user!
    active_layer = pdb.gimp_image_get_active_layer(image)
    #
    stroke_anchor_lists = get_anchors_stroke_by_stroke(distorted_points_path)
    stroke_anchor_lists.sort(key=(lambda x:-len(x))) # descending order
    stroke_lengths = [len(s) for s in stroke_anchor_lists]
    if stroke_lengths == [3,2,1]:
        stroke3, stroke2, stroke1 = stroke_anchor_lists
    elif stroke_lengths == [3,2]:
        stroke3, stroke2 = stroke_anchor_lists
        stroke1 = None
    else:
        m = "The input path should have three strokes of lengths 1,2,3 (in any order),"
        m += "\nor two strokes of lengths 2,3 (in any order),"
        m += "\nbut got lengths "+str(stroke_lengths)
        raise Exception(m)
    five_points = stroke3 + stroke2
    if some_three_collinear(five_points):
        raise Exception("No collinearities allowed among the points on the distorted circle.")
    if stroke1 is None:
        Ca = center_a # No would-be center input
    else:
        Ca = stroke1[0]
    # Find the goal direction:
    orientation_case = orientation_options[orientation_option][1]
    if orientation_case == 'keep':
        direction = (-stroke2[0]+stroke2[1]) / abs(-stroke2[0]+stroke2[1])
        if direction.imag > 0:
            direction *= -1
    elif orientation_case == 'horizontal':
        direction = 1
    else: # vertical
        direction = -1j
    # The matrix of the transformation:
    matrix = matrix_from_1_2_5(would_be_center  = Ca,
                               five_points      = stroke2 + stroke3,
                               direction_points = stroke2,
                               goal_direction   = direction,
                               rotate180        = rotate180,
                               rotate           = rotate
                               )
    # Two quadruples of points to feed to projective_transform_by_points:
    transform_source = five_points[:-1]     # Just some four points on ellipse a
    transform_target = [matrix33_times_complex(matrix, z) for z in transform_source]
    # Finally, transform the layer:
    try:
        new_layer = projective_transform_by_points(image,
                                                   active_layer,
                                                   transform_source,
                                                   transform_target,
                                                   mark_pre_ideal,
                                                   mark_ideal,
                                                   keep_original
                                                   )
        return new_layer
    except PT_InfinityError as e:
        m = e.message
        m += "\nCannot handle this. Things to try:"
        m += "\n- check that the right layer is active;"
        m += "\n- crop the active layer to its essential content;"
        m += "\n- check the input paths."
        raise Exception(m)

#---------------------------------------------------------
#                      Registrations
#---------------------------------------------------------

versionnumber = "0.7"
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2020"
image_types = "*"
menupath = "<Image>/Filters/Distorts/Perspective transform"

###############  4 points to 4 points (path to path)  #################

procedure_name  = "perspective_transform_4_to_4"
procedure_blurb = ("Transform the active layer by perspective transform"
                   +"\ndetermined by two paths: the source and the target."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "A wrapper for pdb.gimp_item_transform_matrix"
procedure_label = "Perspective transform - 4 points to 4 points"

procedure_function = perspective_transform_4_to_4_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "source_path", "The source path (4 anchors)", None),
      (PF_VECTORS, "target_path", "The target path (4 anchors)", None),
      (PF_BOOL, "reversed_target", "Use the target reversed?", False),
      #(PF_BOOL, "mark_pre_ideal", "Mark the line that goes to infinity?", False),
      #(PF_BOOL, "mark_ideal", ("Mark the line that comes from infinity?"
      #                        +"\n(This is the vanishing line)"),
      #                        False),
      (PF_BOOL, "keep_original", "Keep the original layer?", False),
    ],
    [
        (PF_LAYER, "transformed", "the transformed layer"),
    ],
    procedure_function,
    menu=menupath)


###############  4 points to rectangle corners #################

procedure_name  = "perspective_transform_4_to_rectangle"
procedure_blurb = ("Transform the active layer by perspective transform"
                   +"\nsending given path anchors (4 points) to rectangle corners."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "A wrapper for pdb.gimp_item_transform_matrix"
procedure_label = "Perspective transform - 4 points to rectangle corners"

procedure_function = perspective_transform_4_to_rectangle_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "source_path", "The source path (4 anchors)", None),
      (PF_OPTION, 'target_box_option','Target rectangle',
                   0,
                  [case[0] for case in target_box_options]), # Descriptions of cases
      (PF_FLOAT, "rotate", "Further rotation (degrees)", 0.),
      #(PF_BOOL, "inverse_op", "Operation backwards?", False),
      (PF_BOOL, "keep_original", "Keep the original layer?", False),
    ],
    [
        (PF_LAYER, "transformed", "the transformed layer"),
    ],
    procedure_function,
    menu=menupath)


###############  4 points to circle (clock face) #################

procedure_name  = "perspective_transform_4_to_circle_with_clock_face"
procedure_blurb = ("Transform the active layer by perspective transform"
                   +"\nto correct a distorted circle by giving 4 points"
                   +"\non the distorted circle (a path)"
                   +"\nand corresponding hours on a clock face."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "A wrapper for pdb.gimp_item_transform_matrix"
procedure_label = "Perspective transform - 4 points to circle (clock face)"

procedure_function = perspective_transform_4_to_circle_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "source_path", "The source path (4 anchors)", None),
      (PF_FLOAT, "hour0", "The hour on a clock face corresponding to the first anchor", 12.),
      (PF_FLOAT, "hour1", "The hour on a clock face corresponding to the second anchor", 3.),
      (PF_FLOAT, "hour2", "The hour on a clock face corresponding to the third anchor", 6.),
      (PF_FLOAT, "hour3", "The hour on a clock face corresponding to the fourth anchor", 9.),
      (PF_BOOL, "reversed_source", "Use the source path reversed?", False),
      (PF_FLOAT, "rotate", "Further rotation (degrees)", 0.),
      (PF_BOOL, "keep_original", "Keep the original layer?", False),
    ],
    [
        (PF_LAYER, "transformed", "the transformed layer"),
    ],
    procedure_function,
    menu=menupath)

###############  2+2+2 points to circle (3 diameters) #################

procedure_name  = "perspective_transform_2_2_2_to_circle"
procedure_blurb = ("Transform the active layer by perspective transform"
                   +"\nto correct a distorted circle by giving"
                   +"\n3 would-be diameters of the circle."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "A wrapper for pdb.gimp_item_transform_matrix"
procedure_label = "Perspective transform - 2+2+2 points to circle (3 diameters)"

procedure_function = perspective_transform_2_2_2_to_circle_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "diameter1_path", 
          ("diameter 1: path with two anchors"
          + "\n    a line segment to become a diameter of the resulting circle"
          + "\n    (should pass through the would-be center of the circle)"
          #+ "\n    (this diameter is also used to set the direction)"
          ),
          None),
      (PF_VECTORS, "diameter2_path", 
          ("diameter 2: path with two anchors"
          + "\n    another such diameter"
          ),
          None),
      (PF_VECTORS, "diameter3_path", 
          ("diameter 3: path with two anchors"
          + "\n    yet another such diameter"
          ),
          None),
      (PF_VECTORS, "direction_path", 
          ("line segment for setting the direction: path with two anchors"
          + "\n    (one of the diameters can be used)"
          ),
          None),
      (PF_OPTION,  'orientation_option',
                    ('How would you like to set the direction of the above line segment?'
                    ),
                    0,
                  [case[0] for case in orientation_options]), # Descriptions of cases
      (PF_BOOL, "rotate180", "Rotate by 180 degrees?", False),
      (PF_FLOAT, "rotate", "Further rotation (degrees)", 0.),
      (PF_BOOL, "keep_original", "Keep the original layer?", False),
    ],
    [
        (PF_LAYER, "transformed", "the transformed layer"),
    ],
    procedure_function,
    menu=menupath)

###############  1+2+3 points to circle (diameter and center) #################

procedure_name  = "perspective_transform_1_2_3_to_circle"
procedure_blurb = ("Transform the active layer by perspective transform"
                   +"\nto correct a distorted circle by giving"
                   +"\n- 1 anchor for the would-be center of the circle"
                   +"\n- 2 points on the distorted circle (to set direction), and"
                   +"\n- 3 other points on the distorted circle,"
                   +"\nall as three strokes of one path."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = "A wrapper for pdb.gimp_item_transform_matrix"
procedure_label = "Perspective transform - 1+2+3 points to circle (center and 2+3 points)"

procedure_function = perspective_transform_1_2_3_to_circle_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "distorted_points_path", 
          ("1+2+3 points: path with three strokes of lengths 1, 2, and 3:"
          + "\n    stroke 1:  1 point - the would-be center of the circle"
          + "\n    stroke 2:  2 anchors - 2 points on the distorted circle to set the direction"
          + "\n    stroke 3:  3 anchors - 3 other points on the distorted circle"
          ),
          None),
      (PF_OPTION,  'orientation_option',
                    ('How would you like to set the direction of stroke 2 above?'
                    ),
                    0,
                  [case[0] for case in orientation_options]), # Descriptions of cases
      (PF_BOOL, "rotate180", "Rotate by 180 degrees?", False),
      (PF_FLOAT, "rotate", "Further rotation (degrees)", 0.),
      (PF_BOOL, "keep_original", "Keep the original layer?", False),
    ],
    [
        (PF_LAYER, "transformed", "the transformed layer"),
    ],
    procedure_function,
    menu=menupath)

##########################################

main()
